/*=============================================================================
 *     Copyright Texas Instruments 2001. All Rights Reserved.
 */

/**
 * This contains a hardware register class.  A register is further subdivided
 * into fields.
 * 
 * @param name         the name of this register
 * @param addr         the address of this register
 * @param initValue    the initial value of this register
 * @param readable     <code>true</code> if this register readable
 * @param writable     <code>true</code> if this register is writable
 * @param fields       an array of <code>RegisterField</code>s contained in this register
 */
public function Register( name, addr, initialValue, readable, writable, fields )
{
  /**
   * The RegisterBlock that contains this register.
   */
  const var parent;  // const so value can only be set once
  
  if( fields == null )
    throw new ProgrammingErrorException( "Error creating " + name + " Register, no fields specified" );
  
  //  for now the RegisterField doesn't track it's parent, but it might need
  //  to at some point:
  //    for( var i=0; i<fields.length(); i++ )
  //      fields[i].setParent(this);
  
  // these are null unless there are any registered listeners, to save memory:
  var readListenerList  = null;
  var writeListenerList = null;
  
  
  /**
   * Constant to indicate memory that is 8bit addressable.
   */
  public static const var MEMORY_TYPE_8  = 8;
  
  /**
   * Constant to indicate memory that is 16bit addressable.
   */
  public static const var MEMORY_TYPE_16 = 16;
  
  /**
   * Constant to indicate memory that is 32bit addressable.
   */
  public static const var MEMORY_TYPE_32 = 32;
  
  var value         = initialValue;
  var regBitWidth   = 0;
  
  /*
   * Figure out the register size from the fields
   */
  for( var i = 0;
       i < fields.length();
       i++ )
  {
    regBitWidth += fields[ i ].getBitWidth();
  }    
  
  if( ( regBitWidth != MEMORY_TYPE_32 ) &&
      ( regBitWidth != MEMORY_TYPE_16 ) &&
      ( regBitWidth != MEMORY_TYPE_8  ) )
  {
    throw new ProgrammingErrorException( "Error: invalid register width: " + regBitWidth );
  }
  
  /**
   * This should only be called by the RegisterBlock during construction...
   */
  public function setParent(parent)
  {
    this.parent = parent;
  }
  
  /**
   * Get the name of this register.
   * 
   * @return a string
   */
  public function getName()
  {
    return( name );
  }
  
  /**
   * Get the address of this register.
   * 
   * @return address, an integer
   */
  public function getAddr()
  {
    return( addr );
  }
  
  /**
   * Get the initial value of the register.
   * 
   * @return an integer
   */
  public function getInitialValue()
  {
    return( initialValue );
  }
  
  /**
   * Is this register readable?
   * 
   * @return <code>true</code> if this register is readable
   */
  public function isReadable()
  {
    return( readable );
  }
  
  /**
   * Is this register writable?
   * 
   * @return <code>true</code> if this register is writable
   */
  public function isWritable()
  {
    return( writable );
  }
  
  /**
   * Get the fields of this register.
   * 
   * @return an array of <code>RegisterField</code>s
   */
  public function getFields()
  {
    return( fields );
  }
  
  /**
   * Get the size of this register in bits.
   * 
   * @return <code>MEMORY_TYPE_8</code>, <code>MEMORY_TYPE_16</code>, or <code>MEMORY_TYPE_32</code>
   */
  public function getBitWidth()
  {
    return( regBitWidth );
  }
  
  /**
   * Get this size of this register in octets.
   * 
   * @return <code>1</code>, <code>2</code>, or <code>4</code>
   */
  public function getOctetWidth()
  {
    return getBitWidth() / 8;
  }
  
  /**
   * Reset this register to it's initial value.
   */
  public function reset()
  {
    value = initialValue;
    return;
  }
  
  /**
   * Get the value of this register.
   * 
   * @return the current value of this register
   */
  public function getValue()
  {
    return( value );
  }
  
  /**
   * Set the value of this register.
   * 
   * @param newValue     the new value of this register
   */
  public function setValue( newValue )
  {
    value = newValue;
    return( true );     // XXX why does this return a value?
  }
  
  /**
   * Get the named field within this register.
   * 
   * @param fieldName    the string name of the requested field
   * @return a <code>RegisterField</code>, or <code>null</code> if not found.
   */
  public function getField( fieldName )
  {
    for( var i = 0;
         i < fields.length();
         i++ )
    {
      if( fields[ i ].getName() == fieldName )
      {
        return( ( value & fields[ i ].getFieldMask() ) >> fields[ i ].getBitOffset() );
      }
    }
    
    /*
     * If we made it here then the name must not have matched.
     */
    pkg.output.writeln( "Error: " + fieldName + " not found in " + name + " register" );  // XXX if this is fatal error, probably should throw an exception
    
    return null;
  }
  
  /**
   * Set the value of the named field in this register.
   * 
   * @param fieldName    the string name of the field to set
   * @param newValue     the new value for the field.
   */
  public function setField( fieldName, newValue )
  {
    for( var i = 0;
         i < fields.length();
         i++ )
    {
      if( fields[ i ].getName() == fieldName )
      {
        value = ( (value & ~(fields[ i ].getFieldMask())) |
                  ( (newValue << fields[ i ].getBitOffset()) & fields[ i ].getFieldMask() ) );
        return;
      }
    }
    
    /*
     * If we made it here then the name must not have matched.
     */
    pkg.output.writeln( "Error: " + fieldName + " not found in " + name + " register" );  // XXX if this is fatal error, probably should throw an exception
    
    return;
  }
  
  /**
   * Add a read listener.  A listener is a function that will be called when
   * the register is read from the software under test.  The register whose
   * value is read is passed to the callback function.  For example:
   * <pre>
   *   regBlock.addReadListener( function() {
   *     
   *     pkg.output.writeln("the register has been read");
   *     
   *   } );
   * </pre>
   * The read-listener is called before the value of the register is read,
   * so it has the opportunity to modify the value of the register before
   * it is read.
   * 
   * @param fxn          the callback function
   * @see #removeReadListener
   * @see #addWriteListener
   */
  public function addReadListener(fxn)
  {
    // NOTE: we need to synchronize add/remove because of the way we
    // dynamically create the list on demand.  In RegisterBlock the list
    // isn't dynamically created, so there is no race-condition between
    // add/remove:
    synchronized(this)
    {
      if( readListenerList == null )
      {
        // order is important... if we create the list first, there are no
        // race conditions, and no need to have dispatchRead synchronized:
        readListenerList = new java.util.LinkedList();
        parent.addReadListener(dispatchRead);
      }
      
      readListenerList.add(fxn);
    }
  }
  
  /**
   * Remove a read listener.
   * 
   * @param fxn          the callback function
   * @see #addReadListener
   */
  public function removeReadListener(fxn)
  {
    synchronized(this)
    {
      if( readListenerList != null )
      {
        readListenerList.remove(fxn);
        
        if( readListenerList.size() == 0 )
        {
          // order is important here too, see addReadListener:
          parent.removeReadListener(dispatchRead);
          readListenerList = null;
        }
      }
    }
  }
  
  function dispatchRead(reg)
  {
    if( reg == this )
    {
      var arr = readListenerList.toArray();
      for( var i=0; i<arr.length(); i++ )
        arr[i]();
    }
  }
  
  
  /**
   * Add a write listener.  A listener is a function that will be called when
   * the register is write from the software under test.  The register whose
   * value is write is passed to the callback function, as well as the new
   * value that is written to the register.  For example:
   * <pre>
   *   regBlock.addWriteListener( function(val) {
   *     
   *     pkg.output.writeln("the register has been written");
   *     
   *   } );
   * </pre>
   * The write-listener is called before the memHandler has written the new
   * value to the register, so it is possible to act on the value about to
   * be written before it is actually written.
   * 
   * @param fxn          the callback function
   * @see #removeWriteListener
   * @see #addWriteListener
   */
  public function addWriteListener(fxn)
  {
    // NOTE: we need to synchronize add/remove because of the way we
    // dynamically create the list on demand.  In RegisterBlock the list
    // isn't dynamically created, so there is no race-condition between
    // add/remove:
    synchronized(this)
    {
      if( writeListenerList == null )
      {
        // order is important... if we create the list first, there are no
        // race conditions, and no need to have dispatchWrite synchronized:
        writeListenerList = new java.util.LinkedList();
        parent.addWriteListener(dispatchWrite);
      }
      
      writeListenerList.add(fxn);
    }
  }
  
  /**
   * Remove a write listener.
   * 
   * @param fxn          the callback function
   * @see #addWriteListener
   */
  public function removeWriteListener(fxn)
  {
    synchronized(this)
    {
      if( writeListenerList != null )
      {
        writeListenerList.remove(fxn);
        
        if( writeListenerList.size() == 0 )
        {
          // order is important here too, see addWriteListener:
          parent.removeWriteListener(dispatchWrite);
          writeListenerList = null;
        }
      }
    }
  }
  
  function dispatchWrite( reg, val )
  {
    if( reg == this )
    {
      var arr = writeListenerList.toArray();
      
      for( var i=0; i<arr.length(); i++ )
        arr[i](val);
    }
  }
  
  
  /**
   * Make a duplicate copy of this register.
   * 
   * @return a newly constructed Register object that is a duplicate of this
   */
  public function clone()
  {
    var reg = new Register( getName(), getAddr(), getInitialValue(), isReadable(), isWritable(), getFields() );
    reg.setParent(parent);
    return reg;
  }
  
  /**
   * For debugging.
   * 
   * @return a string
   */
  public function castToString()
  {
    return "[register: " + getName() + "]";
  }
}



/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
