/*=============================================================================
 *     Copyright Texas Instruments 2002. All Rights Reserved.
 */


/*=============================================================================
    Description:
    This test is derived from the TestCase base class.

    This test case verifies that edge cases concerning the size of entries
    and the sizes of primitives are handled well.


    Args:
    logWriter        -  A writer object that is used to log any messages.
                        User can either pass in a writer object or null.

    warningWriter    -  A writer object that is used to log any warning messages.
                        User can either pass in a writer object or null.

    errorWriter      -  A writer object that is used to log any error messages
                        User can either pass in a writer object or null.

    Returns:
    n/a

 */
public function TestBufferExhaustion( logWriter, warningWriter, errorWriter )
  extends pkg.tf.TestCase( "esf.age.TestBufferExhaustion", logWriter, warningWriter, errorWriter)
{
  /****************************************************************
   * Global variable initionalizations.
   ****************************************************************/
  const var scriptEhId = 0x101;

  const var rctx = services["esf routing"].getRoutingContext();


  //====================================================================================
  /**
   * Overrides the runTest() method in TestCase base class.
   *
   */
  public function runTest()
  {
    var prim;

    logInfo( "Beginning Age Event Handler Unit Test- TestBufferExhaustion" );
    const var p = pkg.esf.age.util;
    var phase;
    const var preallocSize = 20;
    const var preallocCount = 4;
    const var refreshPeriod = 0;


    // TODO make sure that nothing else is using this event handler



    // Make the system quiet

    phase = "Init:";
    if (!p.unitTestStart( rctx, scriptEhId, phase, preallocCount, preallocSize, 0 )) return;

    phase = "TestBufferExhaustion:";
    const var seed = 0x2345423;
    const var entryHeaderSize = 4;  /* Could change; unlikely */
    const var osEntry = [];
    var osEntryCount = 0;

    var tagIncrement = 0xf0;

    const var initialStats = p.getStats( rctx, scriptEhId, null );

    var countGoodEntries = 0;
    var countSkippedEntries = 0;

    function writeExpecting( size, errorCode )
    {
      const var anEntry = new p.OSEntry();
      anEntry.randomize( 0, 0, size - entryHeaderSize, seed );
      const var tag = 0xFA00 + tagIncrement++;
      anEntry.setTag( tag );

      if (!p.writeBasicEntry(
            rctx
          , scriptEhId
          , anEntry
          , errorCode
        ) ) logError( phase + "Entry write fail." );

      if (errorCode == ti.th.prim.Age_AgeEh_InsertEntryResult.AGE_AGEEH_INSERT_ENTRY_SUCCESS)
      {
        logInfo( phase + "ACCEPTED: #"
                 + osEntryCount + " "
                 + osEntry[ osEntryCount ] + " "
                 + errorCode );
        osEntry[ osEntryCount ] = anEntry;
        ++osEntryCount;
        ++countGoodEntries;
      }
      else
      {
        ++countSkippedEntries;
      }

      return;
    }

    /* Start listening */
    var trap = p.attachTrap( rctx, scriptEhId );

    var repeats = preallocCount;
    logInfo( phase + " Exhaust buffers part 1: repeats=" + repeats );
    for (var i = 0; i < repeats; ++i ) writeExpecting( preallocSize,
        ti.th.prim.Age_AgeEh_InsertEntryResult.AGE_AGEEH_INSERT_ENTRY_SUCCESS );

    repeats = preallocCount * 9;
    logInfo( phase + " Expect skipped buffers part 1: repeats=" + repeats );
    for (var i = 0; i < repeats; ++i ) writeExpecting( preallocSize,
        ti.th.prim.Age_AgeEh_InsertEntryResult.AGE_AGEEH_INSERT_ENTRY_FAIL_NO_BUFFERS );

    logInfo( phase + " Triggering buffer service part 1..." );
    if (!p.triggerBufferService( rctx, scriptEhId) ) logError( phase + "Buffer Service Fail." );

    var repeats = 4;
    logInfo( phase + " Exhaust buffers part 2: repeats=" + repeats );
    for (var i = 0; i < preallocCount; ++i ) writeExpecting( preallocSize,
        ti.th.prim.Age_AgeEh_InsertEntryResult.AGE_AGEEH_INSERT_ENTRY_SUCCESS );

    repeats = preallocCount * 3;
    logInfo( phase + " Expect skipped buffers part 2: repeats=" + repeats );
    for (var i = 0; i < repeats; ++i ) writeExpecting( preallocSize,
        ti.th.prim.Age_AgeEh_InsertEntryResult.AGE_AGEEH_INSERT_ENTRY_FAIL_NO_BUFFERS );

    logInfo( phase + " Shrinking config to 1 for part 3" );
    if (!p.setConfiguration( rctx, scriptEhId, 1, preallocSize, refreshPeriod ) ) logError( phase + " buf cfg fail." );

    logInfo( phase + " Exhaust buffers part 3: singleton" );
    writeExpecting( preallocSize, ti.th.prim.Age_AgeEh_InsertEntryResult.AGE_AGEEH_INSERT_ENTRY_SUCCESS );

    repeats = 3;
    logInfo( phase + " Expect skipped buffers part 3: repeats=" + repeats );
    for (var i = 0; i < repeats; ++i ) writeExpecting( preallocSize,
        ti.th.prim.Age_AgeEh_InsertEntryResult.AGE_AGEEH_INSERT_ENTRY_FAIL_NO_BUFFERS );

    logInfo( phase + " Shrinking config to 0 for part 4" );
    if (!p.setConfiguration( rctx, scriptEhId, 0, preallocSize, refreshPeriod ) ) logError( phase + " buf cfg fail." );

    repeats = 3;
    logInfo( phase + " Expect skipped buffers part 4: repeats=" + repeats );
    for (var i = 0; i < repeats; ++i ) writeExpecting( preallocSize,
        ti.th.prim.Age_AgeEh_InsertEntryResult.AGE_AGEEH_INSERT_ENTRY_FAIL_NO_BUFFERS );

    logInfo( phase + " Triggering buffer service part 5..." );
    if (!p.triggerBufferService( rctx, scriptEhId) ) logError( phase + "Buffer Service Fail." );

    repeats = 3;
    logInfo( phase + " Expect skipped buffers part 5: repeats=" + repeats );
    for (var i = 0; i < repeats; ++i ) writeExpecting( preallocSize,
        ti.th.prim.Age_AgeEh_InsertEntryResult.AGE_AGEEH_INSERT_ENTRY_FAIL_NO_BUFFERS );


    phase = "VerifyWrittenAreWritten:";
    if (!p.triggerBufferService( rctx, scriptEhId) ) logError( phase + "Buffer Service Fail." );

    var decoder = pkg.esf.age.util.acquireEntryDecoder( logInfo, logWarning, logError );
    p.detachDecodeTrap( rctx, trap, decoder );

    const var decodedEntryCount = decoder.entryCount();
    logInfo( "Decoded entry count was " + decodedEntryCount
             + " and osEntryCount is " + osEntryCount );
    if (decodedEntryCount != osEntryCount)
    {
      logError( phase + "Expected " + osEntryCount
                + " entries but found " + decoder.entryCount() );
    }
    for (var i = 0; i < decodedEntryCount; ++i) {
      logInfo( "Inspecting entry i= " + i + ": Expected is " + osEntry[ i ].toString() );
      const var decodedEntry = decoder.get();
      if (!p.verifyBasicEntry( decodedEntry, osEntry[ i ] ) )
      {
        logError( phase + "Entry verify fail." );
      }
    }
    while( decoder.entryCount() > 0) {
      const var unhandled = decoder.get();
      logError( phase + "Unhandled Entry!" + unhandled );
    }
    for (
          var lostEntryIndex = decodedEntryCount
        ; lostEntryIndex < osEntryCount
        ; ++lostEntryIndex
        )
    {
      logError(" Lost the entry " + osEntry[ lostEntryIndex ] );
    }

    logInfo(" Verifying stats: good:" + countGoodEntries
            + " and skipped: " + countSkippedEntries );

    const var finalStats = p.getStats( rctx, scriptEhId, initialStats );

    logInfo( finalStats.toString() );

    if (finalStats.getDiffWrittenEntries() != countGoodEntries) {logError(" Written Entry Mismatch ");}
    if (finalStats.getDiffMissedEntries() != countSkippedEntries) {logError(" Skipped Entry Mismatch ");}


    phase = "Cleanup:";
    p.unitTestEnd( rctx, scriptEhId, phase );

  }

}




