/*=============================================================================
 *     Copyright Texas Instruments, Inc., 2002.  All Rights Reserved.
 */


/**
 * Installs a prim-handler that checks sequence numbers of incoming prims
 * to verify that no sequence numbers are lost.  The check uses a table
 * of sequence numbers in order to tolerate out of order sequence numbers
 * withing a window (equal to 1/2 the size of the table).  When a primitive
 * is received, the seqn is stored in the <code>seqn % table.length()</code>
 * entry.  Then, the two table entries located 1/2 table size away are
 * checked to verify that they contain sequential sequence numbers.  This 
 * provides constant size and constant order performance, while still
 * tolerating local-out-of-order sequence numbers.
 * 
 * @param tableSize     the size of the table to use, which determines the
 *    size of the window within which out of order sequence numbers are
 *    tolerated
 */
public function seqnChecker(tableSize)
{
  var seqnTable = [];
  
  // initialize table entries
  for( var i=0; i<tableSize; i++ )
    seqnTable[i] = null;
  
  // get rctx:
  var esf = services["esf routing"];
  var rctx = esf.getRoutingContext( esf.RCTX_MODE_EVERYTHING );
  
  /**
   * This is called to log a message.  The log message is timestamped
   */
  function log(str)
  {
    pkg.output.writeln(java.lang.System.currentTimeMillis() + ": " + str);
  }
  
  // install prim-handler:
  rctx.addPrimHandler( function(prim) {
    
    var seqn = prim.getSequenceNumber();
    
    // ignore primitive if no valid sequence number:
    if( seqn == -1 )
      return;
    
    var curIdx = seqn % tableSize;
    
    // figure out the indexes of the two adjacent entries located
    // half table-size away, taking into account wrap-around:
    var a = (curIdx + (tableSize/2)) % tableSize;
    var b = (a + 1) % tableSize;
    
    // check the two adjacent entries to verify they contain successive values:
    if( (seqnTable[a] != null) && (seqnTable[b] != null) )
      if( (seqnTable[a] + 1) != seqnTable[b] )
        log("missing sequence number detected: " + seqnTable[a] + ", " + seqnTable[b] + " (a=" + a + ", b=" + b + ")");
    
    seqnTable[curIdx] = seqn;
    
  }, null );
}


/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
