/*
+-------------------------------------------------------------------+
| PROJECT: MMI-Framework (8445)         $Workfile:: mfw_bt_dm.h				$|
| $Author:: NDH					     $Revision::  1						$|
| CREATED: 17.04.04				     $Modtime:: 17.04.04 10:48			$|
| STATE  : code														 |
+-------------------------------------------------------------------+


   MODULE  : MFW_BT

   PURPOSE : This module contains the definitions for MFW Bluetooth Device Manager Profile

*/

#ifndef DEF_MFW_BT_DM_HEADER
#define DEF_MFW_BT_DM_HEADER

#include "bta_api.h"
#include "bd.h"
#include "mfw_bte.h"

#define MFW_BT_NAME_LENGTH   34
#define MFW_BT_DEFAULT_INQ_DURATION 4
#define MFW_BT_MAX_PIN_CODE_LEN 16
#define MFW_BT_MAX_CONSEC_CONNECTIONS MAX_L2CAP_LINKS

/* redefine the BD_ADDR type in a "MFW-like" way */
typedef BD_ADDR					T_MFW_BT_DM_DEV_ADDR;
#define MFW_BT_DM_DEV_ADDR_LEN 	BD_ADDR_LEN

/*
** Event Parameter Structure Definitions
*/
typedef tBTA_DM_ENABLE			T_MFW_BT_DM_ENABLE;
typedef 	tBTA_DM_LINK_UP			T_MFW_BT_DM_LINK_UP;
typedef tBTA_DM_LINK_DOWN 		T_MFW_BT_DM_LINK_DOWN;
typedef tBTA_DM_PIN_REQ 		T_MFW_BT_DM_PIN_REQ;

typedef tBTA_DM_AUTHORIZE 		T_MFW_BT_DM_AUTHORIZE_REQ;
typedef tBTA_DM_SIG_STRENGTH 	T_MFW_BT_DM_SIG_STRENGTH_IND;

/* Search structures */
typedef tBTA_DM_INQ_RES 		T_MFW_BT_DM_INQ_RES;
typedef tBTA_DM_INQ_CMPL 		T_MFW_BT_DM_INQ_CMPL;

/* other BT structures*/
typedef tBTA_SERVICE_MASK		T_MFW_BT_SERVICE_MASK;
typedef tBTA_SIG_STRENGTH_MASK	T_MFW_BT_DM_SIG_STRENGTH;

typedef tBTA_DM_DISC 			T_MFW_BT_VIS_MASK;


enum
{
	MFW_BT_AUTH_FAIL = 0,
	MFW_BT_AUTH_ALWAYS,
	MFW_BT_AUTH_ONCE
};

typedef  UINT8 T_MFW_BT_AUTHORIZE;

typedef struct
{
	BD_ADDR				bd_addr;
	UINT8					bd_name[MFW_BT_NAME_LENGTH];
	BOOL					is_success;
}
T_MFW_BT_DM_AUTH_CMPL;

/* remote device */
typedef struct
{
	/* important!The first parameters should be the same as T_MFW_BT_REM_DEVICE_INFO */
	BD_ADDR				bd_addr;
	char      					friendly_name[MFW_BT_NAME_LENGTH];
	DEV_CLASS				dev_class;
	BOOL	   				link_key_present;
	BOOL	   				is_trusted;
	T_MFW_BT_SERVICE_MASK 	trusted_services;	
	T_MFW_BT_SERVICE_MASK	services;
	BOOL					is_new;
	/* then these are "internal" parameters */
	BOOL	   				in_use;
	char      					name[MFW_BT_NAME_LENGTH];
	LINK_KEY  				link_key;
} 
T_MFW_BT_REM_DEVICE;


/* remote device. !!! Please keep this structure synchronized with the 
	previous one: T_MFW_BT_REM_DEVICE !!! */
typedef struct
{
	BD_ADDR				bd_addr;
	char      					name[MFW_BT_NAME_LENGTH];
	DEV_CLASS				dev_class;
	BOOLEAN   				link_key_present;
	BOOLEAN   				is_trusted;
	T_MFW_BT_SERVICE_MASK 	trusted_services;	
	T_MFW_BT_SERVICE_MASK	services;
	BOOL					is_new;	/* specify whether this device is already or not into our device DB */
} 
T_MFW_BT_REM_DEVICE_INFO;

typedef T_MFW_BT_REM_DEVICE_INFO 		T_MFW_BT_DM_DISC_RES;


/* MFW_BT main control block */
typedef struct
{
	T_MFW_BT_SERVICE_MASK 	ui_current_active_connection;  	/* active connection mask */
	T_MFW_BT_REM_DEVICE	*p_selected_rem_device;        	/* pointer to selected device */
	BD_ADDR				peer_bdaddr;                             	/* peer bdaddr stored for pin_reply etc*/
	UINT8    					peer_name[MFW_BT_NAME_LENGTH];        /* bluetooth name of peer device for pin reply etc */
	T_MFW_BT_SERVICE_MASK	peer_service;					/* peer services stored for auth etc*/
	T_MFW_BT_SERVICE_MASK 	search_services;               		/* services to search for */
	BOOL					is_discovery;					/* remember if it's an inquiry or a discovery */
	T_MFW_BT_SERVICE_MASK	established_linked;				/* which services have an established link */
	BOOL					is_name_requested;				/* remember if we asked for the local name */
	BOOL					is_started;						/* if BT is not started, we will not call BTA APIs */
} 
T_MFW_BT_CB;

extern T_MFW_BT_CB 			mfw_bt_cb;



#define MFW_BT_NUM_REM_DEVICE 10

/* Inquiry results database */
typedef struct
{
    T_MFW_BT_REM_DEVICE    	remote_device[MFW_BT_NUM_REM_DEVICE];
    UINT8               				rem_index;
} 
tMFW_BT_INQ_DB;



/*
** The following Macros define the Device Manager Signals (BT -> Mfw)
*/
#define T_MFW_BT_DM_SEC_EVT			tBTA_DM_SEC_EVT
#define T_MFW_BT_DM_SEC_SIG_DATA		tBTA_DM_SEC

#define T_MFW_BT_DM_SRCH_EVT			tBTA_DM_SEARCH_EVT
#define T_MFW_BT_DM_SRCH_SIG_DATA	tBTA_DM_SEARCH

/*
** Event Identifiers (Mfw -> BMI)
*/
#define E_BT_DM_LINK_UP			(T_MFW_EVENT)0x00000010
#define E_BT_DM_LINK_DOWN			(T_MFW_EVENT)0x00000020
#define E_BT_DM_INQ_RES			(T_MFW_EVENT)0x00000030
#define E_BT_DM_INQ_CMPL			(T_MFW_EVENT)0x00000040
#define E_BT_DM_DISC_RES			(T_MFW_EVENT)0x00000050
#define E_BT_DM_DISC_CMPL			(T_MFW_EVENT)0x00000060
#define E_BT_DM_PIN_REQ			(T_MFW_EVENT)0x00000070
#define E_BT_DM_AUTH_CMPL			(T_MFW_EVENT)0x00000080
#define E_BT_DM_AUTHORIZE_REQ		(T_MFW_EVENT)0x00000090
#define E_BT_DM_DEV_NAME_REQ		(T_MFW_EVENT)0x000000A0
#define E_BT_DM_ENABLE				(T_MFW_EVENT)0x00000100


#define E_BT_DM_SIG_STRENGTH_IND	(T_MFW_EVENT)0x000000E0
#define E_BT_ALL_DM_SERVICES		(T_MFW_EVENT)0x000000F0





/*******************************************************************************
 $Function:		mfw_bt_dm_set_visibility
 $Description:		set visibility of local BT device
 $Returns:		T_MFW_BT_STATUS. Success or fail.	
 $Arguments:		is_visible: new visibility setting. is_temp: define whether it is a temporary
				modification ( TRUE ) or a permanent one ( FALSE ).
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_set_visibility( BOOL is_visible, BOOL is_temp);

/*******************************************************************************
 $Function:		mfw_bt_dm_get_visibility
 $Description:		get visibility of local BT device
 $Returns:		TRUE if visible, FALSE if not.
 $Arguments:		none
*******************************************************************************/
BOOL mfw_bt_dm_get_visibility( void );

/*******************************************************************************
 $Function:		mfw_bt_dm_get_local_name
 $Description:		read  local BT device name
 $Returns:		a pointer to a string. NULL if no name available. One should make a copy of
				string if he wants to use it/modify it. 
 $Arguments:		none
*******************************************************************************/
UINT8 * mfw_bt_dm_get_local_name( void );

/*******************************************************************************
 $Function:		mfw_bt_dm_set_local_name
 $Description:		set local BT device name
 $Returns:		T_MFW_BT_STATUS	
 $Arguments:		new name. Pointer to a string. This string is copied locally. Buffer is not freed.
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_set_local_name( INT8 *name );

/*******************************************************************************
 $Function:		mfw_bt_dm_get_bt_status
 $Description:		get the status of the local Bluetooth system. Used at start-up to decide whether
 				or not we want BT to be started by default.
 $Returns:		MFW_BT_SUCCESS is BT is started. 0 if OFF	
 $Arguments:		
*******************************************************************************/
BOOL mfw_bt_dm_get_bt_status( void );

/*******************************************************************************
 $Function:		mfw_bt_dm_pin_code
 $Description:		used by BMI to send back a pin code
 $Returns:		T_MFW_BT_STATUS. 	MFW_BT_NOT_INITIALISED is BT is not started.
 $Arguments:		pin code and pin code length. This data is copied.
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_pin_code(UINT8* pin_code, UINT8 pin_len);

/*******************************************************************************
 $Function:		mfw_bt_dm_bond
 $Description:		used by BMI to send a pin code in order to establish a bonding with a 
 			remote device.
 $Returns:		T_MFW_BT_STATUS. MFW_BT_NOT_INITIALISED is BT is not started.
 			
 $Arguments:		BD_ADDR of the remote device, pin code and pin code length. Data is copied.
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_bond(BD_ADDR bd_addr, UINT8* pin_code, UINT8 pin_len );

/*******************************************************************************
 $Function:		mfw_bt_dm_authorize_resp
 $Description:		used to answer to an authorization request
 $Returns:		T_MFW_BT_STATUS. SUCCESS/FAIL, but also:
				MFW_BT_NOT_INITIALISED is BT is not started. 
				MFW_BT_INVALID_DATA if auth value is not correct.
 				MFW_BT_UNKNOWN_DEVICE if BD_addr is not recognized.
 				MFW_BT_DATA_BASE_FULL if there are already too many devices in the DB.	
 $Arguments:		MFW_BT_AUTH_FAIL to refuse, 
 				MFW_BT_AUTH_ONCE to grant access temporarily,
 				MFW_BT_AUTH_ALWAYS to grant permanent access
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_authorize_resp( T_MFW_BT_AUTHORIZE auth);

/*******************************************************************************
 $Function:		mfw_bt_dm_sig_strength
 $Description:		ask for the link quality
 $Returns:		T_MFW_BT_STATUS. MFW_BT_NOT_INITIALISED if BT is not started.
 $Arguments:		T_MFW_BT_DM_SIG_STRENGTH sig_strength,  UINT16 period, 	BOOLEAN is_on
     			See BTA documentation for more details.
			
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_sig_strength (	T_MFW_BT_DM_SIG_STRENGTH sig_strength,
													UINT16 period,
													BOOLEAN is_on );

/*******************************************************************************
 $Function:		mfw_bt_dm_get_known_devices
 $Description:		used by BMI to get the list of known devices for a specified condition: either a 
				service mask, either a bd address ( union ). 
 				MFW will fill in the variables: pp_device => table of known devices
 				number_of_devices => number of devices.
 				MFW will allocate memory for each device found. It's up to BMI to free it!
 $Returns:		T_MFW_BT_STATUS. 
 				If BD_ADDR is specified, might return MFW_BT_UNKNOWN_DEVICE.
 				 The answers will be returned using BMI signals:
 				- E_BT_DM_INQ_CMPL with the number of devices
 				- E_BT_DM_DISC_RES for each device
 				- then E_BT_DM_DISC_CMPL when it's done.
 $Arguments:		BD_ADDR if looking for a particular device. services if looking for a 
 				category of device. If bd_addr is specified, ignore services. If services set
 				to 0, will return all the known devices.
*******************************************************************************/
void mfw_bt_dm_get_known_devices ( 	BD_ADDR bd_addr,
											T_MFW_BT_SERVICE_MASK services);

/*******************************************************************************
 $Function:		mfw_bt_dm_delete_device
 $Description:		remove a device from the local DB
 $Returns:		T_MFW_BT_STATUS ( MFW_BT_SUCCESS, MFW_BT_FAIL, MFW_BT_UNKNOWN_DEVICE )	
 $Arguments:		BD address
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_delete_device(BD_ADDR bd_addr);

/*******************************************************************************
 $Function:		mfw_bt_dm_add_device
 $Description:		This will store permanently a device in Flash.
 $Returns:		T_MFW_BT_STATUS
				errors: MFW_BT_UNKNOWN_DEVICE, MFW_BT_DATA_BASE_FULL, 				
 $Arguments:		new BD address
*******************************************************************************/
 T_MFW_BT_STATUS mfw_bt_dm_add_device(BD_ADDR bd_addr);

/*******************************************************************************
 $Function:		mfw_bt_dm_rename_device
 $Description:		modify the friendly name of a known ( already stored ) remote device
 $Returns:		T_MFW_BT_STATUS.
				MFW_BT_UNKNOWN_DEVICE, DATA_BASE_FULL, MFW_BT_INVALID_DATA...
 $Arguments:		bd address of the remote device and the new name associated
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_rename_device(BD_ADDR bd_addr, UINT8* new_name);




/*
**  Functions used to interact with the BT "search engine"
*/

/*******************************************************************************
 $Function:		mfw_bt_dm_discover_device
 $Description:		Discovers services on a new device
 $Returns:		T_MFW_BT_STATUS. MFW_BT_NOT_INITIALISED if BT not started.	
 $Arguments:		bd address of the remote device to discover
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_discover_device(BD_ADDR bd_addr);


/*******************************************************************************
 $Function:		mfw_bt_dm_is_discover
 $Description:	Checks if we are in discovering services process
 $Returns:		
 $Arguments:	None
*******************************************************************************/
UINT8 mfw_bt_dm_is_discover(void);

/*******************************************************************************
 $Function:		mfw_bt_dm_cancel_search
 $Description:		cancel an ongoing search
 $Returns:		MFW_BT_NOT_INITIALISED if BT not started.	
 $Arguments:		
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_cancel_search( void );

/*******************************************************************************
 $Function:		mfw_bt_dm_search
 $Description:		Searches for devices supporting the services specified. If services = 0, will 
				return all the found devices regardless of their functionalities.
 $Returns:		MFW_BT_NOT_INITIALISED if BT not started.	
 $Arguments:		services. If
*******************************************************************************/
T_MFW_BT_STATUS mfw_bt_dm_search(T_MFW_BT_SERVICE_MASK services);






/*
** Bluetooth Device Manager Profile Prototypes
*/
void	mfw_bt_dm_security_cb(T_MFW_BT_DM_SEC_EVT event, T_MFW_BT_DM_SEC_SIG_DATA *data);
void	mfw_bt_dm_search_cb(T_MFW_BT_DM_SRCH_EVT event, T_MFW_BT_DM_SRCH_SIG_DATA *data);

T_MFW_BT_STATUS	mfw_bt_dm_security_hndlr (T_MFW_BT_DM_SEC_EVT event, T_MFW_BT_DM_SEC_SIG_DATA *data);
T_MFW_BT_STATUS	mfw_bt_dm_search_hndlr (T_MFW_BT_DM_SRCH_EVT event, T_MFW_BT_DM_SRCH_SIG_DATA *data);








#endif

