/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8445)         $Workfile:: mfw_bt.c        $|
| $Author:: Rm                          $Revision::  1              $|
| CREATED: 03.01.01                     $Modtime:: 03.01.01 17:21   $|
| STATE  : code                                                      |
+--------------------------------------------------------------------+


   MODULE  : MFW_BT

   PURPOSE : This modul contains the functions for MFW-BT management.


*/
#define ENTITY_MFW

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include "typedefs.h"
#include "vsi.h"
#include "pei.h"
#include "custom.h"
#include "gsm.h"

#include "p_btp.h"
#include "Bti.h"
#include "bti_btp.h"
#include "bti_cb.h"

#include "mfw_mfw.h"
#include "mfw_bt.h"
#include "mfw_bta.h"
#include "mfw_acie.h"
#include "mfw_win.h"

/* static used structures/variables */
EXTERN MfwHdr * current_mfw_elem;

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_init            |
+--------------------------------------------------------------------+

  PURPOSE : initialize BT manager

*/
int bt_init (void)
{
  TRACE_FUNCTION("mfw_bt:bt_init()");
  /*
   * install prim handler (primitives directly to MFW)
   */
  pSlotBt = aci_create(bta_response_cb,NULL);
  /*
  * initialize internal lists for found services
  */
  found_headset.service = MFW_BT_HEADSET;
  found_headset.device_id = NULL;
  found_dial_up.service = MFW_BT_DIAL_UP;
  found_dial_up.device_id = NULL;
  found_fax.service = MFW_BT_FAX_GW;
  found_fax.device_id = NULL;
  found_opp.service = MFW_BT_OPP;
  found_opp.device_id = NULL;
  found_sync.service = MFW_BT_SYNC;
  found_sync.device_id = NULL;
  found_sync_cmd.service = MFW_BT_SYNC_CMD;
  found_sync_cmd.device_id = NULL;
  return TRUE;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_exit            |
+--------------------------------------------------------------------+

  PURPOSE : finalize BT manager

*/
void bt_exit (void)
{
  TRACE_FUNCTION("bt_exit()");
  /*
   * remove prim handler
   */
  aci_delete(pSlotBt);
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_create          |
+--------------------------------------------------------------------+

  PURPOSE : create BT event handler and control block

*/
T_MFW_HND bt_create (MfwHnd hWin, MfwEvt event, MfwCb cbfunc)
{
  MfwHdr *hdr;
  T_MFW_BT *bt_para;
  MfwHdr *insert_status =0;

  TRACE_FUNCTION("bt_create()");

  hdr = (MfwHdr *) mfwAlloc(sizeof(MfwHdr));
  bt_para = (T_MFW_BT *) mfwAlloc(sizeof(T_MFW_BT));

  if (!hdr || !bt_para)
  	{
    	TRACE_ERROR("ERROR: bt_create() Mem Alloc Failed.");
			
	   	if(hdr)
   			mfwFree((U8*)hdr,sizeof(MfwHdr));

   		if(bt_para)
   			mfwFree((U8*)bt_para,sizeof(T_MFW_BT));
   		
	   	return 0;
  	}

  bt_para->emask = event;
  bt_para->handler = cbfunc;            /* event callback function */

  hdr->data = bt_para;
  hdr->type = MfwTypBt;

  insert_status= mfwInsert((MfwHdr *) hWin,hdr); /* handle is brought in chain */
  if(!insert_status)
	{
  		TRACE_ERROR("ERROR: bt_create() Failed to Install Handler. ");
   		mfwFree((U8*)hdr,sizeof(MfwHdr));
   		mfwFree((U8*)bt_para,sizeof(T_MFW_BT));
		return 0;
  	}
    return insert_status;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_delete          |
+--------------------------------------------------------------------+

  PURPOSE : delete BT event handler and control block, free Memory

*/
MfwRes bt_delete (MfwHnd h)
{
  TRACE_FUNCTION("bt_delete()");

  if (!h || !((MfwHdr *) h)->data)
    return MfwResIllHnd;

  if (!mfwRemove((MfwHdr *) h))
  return MfwResIllHnd;

  mfwFree((U8 *) ((MfwHdr *) h)->data,sizeof(T_MFW_BT));
  mfwFree((U8 *) h,sizeof(MfwHdr));

  return MfwResOk;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_signal          |
+--------------------------------------------------------------------+

  PURPOSE : Send a signal to MMI

*/
void bt_signal (MfwEvt event, void *para)
{	/*MC, SPR 1389, we have to enable the display whenever 
	we send an event up to the MMI*/
	UBYTE temp = dspl_Enable(0);

  	TRACE_FUNCTION ("bt_signal()");

	  if (mfwSignallingMethod EQ 0) /* default search method (old bmi design) */
	  {
	    if (mfwFocus)               /* look for element in focussed chain */
	      if (bt_sign_exec(mfwFocus,event,para))
	      { dspl_Enable(temp);/*MC, SPR 1389*/
	        return;
	      }
	    if (mfwRoot)                /* look for element in root */
	      bt_sign_exec(mfwRoot,event,para);
	  }
	  else /* new bmi design */
	  {
	    MfwHdr * h = 0;
	    /*
	     * Focus set, then start here
	     */
	    if (mfwFocus)
	      h = mfwFocus;
	      /*
	       * Focus not set, then start root
	       */
	    if (!h)
	      h = mfwRoot;
	    /*
	     * No elements available, return
	     */
	    while (h)
	    {
	      /*
	       * Signal consumed, then return
	       */
	      if (bt_sign_exec (h, event, para))
	      { dspl_Enable(temp);/*MC, SPR 1389*/
	        return;
	      }
	      /*
	       * All windows tried inclusive root
	       */
	      if (h == mfwRoot)
	      { dspl_Enable(temp);/*MC, SPR 1389*/
	        return;
	      }
	      /*
	       * get parent window
	       */
	      h = mfwParent(mfwParent(h));
	      if(h)
	        h = ((MfwWin * )(h->data))->elems;
	    }
	    bt_sign_exec (mfwRoot, event, para);
	  }
	  dspl_Enable(temp);/*MC, SPR 1389*/
	
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_sign_exec       |
+--------------------------------------------------------------------+

  PURPOSE : Send a signal if BT management handler

*/
int bt_sign_exec (MfwHdr *cur_elem, MfwEvt event, T_MFW_BT_PARA *para)
{
  T_MFW_BT_SERVICE_SEARCH_CNF * search_confirmation_ptr;
  T_MFW_BT_DEVICE_SEARCH_CNF * dev_search_conf_ptr;
  T_MFW_BT_DISCONNECT_CNF * discon_ptr;
  T_MFW_BT_DISCON_DUN_FAX_CNF * discon_dun_fax_ptr;
  T_MFW_BT_DISCON_DUN_FAX_IND * discon_dun_fax_ind_ptr;
  T_MFW_BT_DISCONNECT_IND * discon_i_ptr;
  T_MFW_BT_CONNECT_CNF * connect_ptr;
  T_MFW_BT_SRV_SYNC_CNF * srv_sync_complete;
  T_MFW_BT_SRV_SYNC_AUTH_IND * srv_sync_auth_ind;
  T_MFW_BT_SRV_SYNC_PULL_IND * srv_sync_pull_ind;
  T_MFW_BT_SRV_SYNC_PUSH_IND * srv_sync_push_ind;
  T_MFW_BT_SRV_SYNC_PUSH_CNF * srv_sync_push_cnf;
  T_MFW_BT_BD_ADDR dummy_address[MFW_BT_ADDR_MAX_LEN];

  memset(&dummy_address,0,sizeof(dummy_address));

  TRACE_FUNCTION("bt_sign_exec()");

  while (cur_elem)
  {
    if (cur_elem->type EQ MfwTypBt)
    {
      T_MFW_BT *bt_data;
      bt_data = (T_MFW_BT *) cur_elem->data;
      if (bt_data->emask & event)
      {
        bt_data->event = event;
        switch (event)
        {

          case BT_CREATE_PROFILE_CNF:
            memcpy(&bt_data->para.prof_create_cnf, para,sizeof(T_MFW_BT_PROFILE_CREATE_CNF));
            break;

          case BT_DELETE_PROFILE_CNF:
            memcpy(&bt_data->para.prof_delete_cnf, para,sizeof(T_MFW_BT_PROFILE_DELETE_CNF));
            break;

          case BT_INIT_PROFILE_CNF:   /* confirms initialize/reconfiguration of service profile  */
            memcpy(&bt_data->para.profile,
                   para,sizeof(T_MFW_BT_PROFILE_CNF));
            /* service,result_bd,subtype(,mfw_opp_mode,mfw_sync_mode,mfw_syn_aut_mode) with event BT_INIT_PROFILE_CNF to MMI */
            break;

          case BT_DEINIT_PROFILE_CNF:/* confirms deinitialize profile  */
            memcpy(&bt_data->para.profile,
                   para,sizeof(T_MFW_BT_PROFILE_CNF));
            /* service,result_bd,subtype(,mfw_opp_mode,mfw_sync_mode,mfw_syn_aut_mode) with event BT_DEINIT_PROFILE_CNF to MMI */
            break;
        }

        switch (event)
        {
          case BT_RESTORE_LIST_RESULT:/* result restoring default service list  */
            memcpy(&bt_data->para.restore_conf,
                   para,sizeof(T_MFW_BT_REST_CONFIG));
            /* service and cause with event BT_RESTORE_LIST_RESULT to MMI */
            break;

          case BT_RESTORE_CONF_RESULT:/* result restoring configuration parameter  */
            memcpy(&bt_data->para.restore_conf,
                   para,sizeof(T_MFW_BT_REST_CONFIG));
            /* service and cause with event BT_RESTORE_CONF_RESULT to MMI */
            break;

          case BT_SERVICE_SEARCH_CNF:
            search_confirmation_ptr = (T_MFW_BT_SERVICE_SEARCH_CNF *)para;
            if(search_confirmation_ptr->service EQ MFW_BT_HEADSET)
            {
              /* pointer of list of detected hs to mmi */
              bt_data->para.service_lst_p.list_ptr = &found_headset;
            }
            else if(search_confirmation_ptr->service EQ MFW_BT_DIAL_UP)
            {
              /* pointer of list of detected dial up  to mmi */
              bt_data->para.service_lst_p.list_ptr = &found_dial_up;
            }
            else if(search_confirmation_ptr->service EQ MFW_BT_FAX_GW)
            {
              /* pointer of list of detected fax  to mmi */
              bt_data->para.service_lst_p.list_ptr = &found_fax;
            }
            else if(search_confirmation_ptr->service EQ MFW_BT_OPP)
            {
              /* pointer of list of detected opp  to mmi */
              bt_data->para.service_lst_p.list_ptr = &found_opp;
            }
            else if(search_confirmation_ptr->service EQ MFW_BT_SYNC)
            {
              /* pointer of list of detected sync  server  to mmi */
              bt_data->para.service_lst_p.list_ptr = &found_sync;
            }
            else if(search_confirmation_ptr->service EQ MFW_BT_SYNC_CMD)
            {
              /* pointer of list of detected sync server with command support  to mmi */
              bt_data->para.service_lst_p.list_ptr = &found_sync_cmd;
            }
            break;

          case BT_DEVICE_SEARCH_CNF:
            dev_search_conf_ptr = (T_MFW_BT_DEVICE_SEARCH_CNF *)para;
            if(dev_search_conf_ptr->result EQ BTI_OK)
            {/* pointer of list of found services to mmi */
              bt_data->para.device_lst_p.list_ptr = &service_list;
            }
            break;

          case BT_DEVICE_FOUND_IND: /* new device was found */
            memcpy(&bt_data->para.device_ind, para, sizeof(T_MFW_BT_DEVICE_IND));
            break;

          case BT_SERVICE_FOUND_IND: /* new service was found */
            memcpy(&bt_data->para.service_ind, para, sizeof(T_MFW_BT_SERVICE_IND));
            break;

          case BT_CONNECT_DEVICE_CNF:
            connect_ptr = (T_MFW_BT_CONNECT_CNF *)para;
            if(connect_ptr->service EQ MFW_BT_HEADSET)
            {
              memcpy(&bt_data->para.connect_cnf,
                     para,sizeof(T_MFW_BT_CONNECT_CNF));
            }
            else
              return FALSE;
            break;

          case BT_SRV_SYNC_CNF:
            srv_sync_complete = (T_MFW_BT_SRV_SYNC_CNF *)para;
            if(srv_sync_complete->service EQ MFW_BT_SYNC)
            {
              memcpy(&bt_data->para.srv_sync_cnf,
                     para,sizeof(T_MFW_BT_SRV_SYNC_CNF));
            }
            else
              return FALSE;
            break;

          case BT_SRV_SYNC_AUTH_IND:
            srv_sync_auth_ind = (T_MFW_BT_SRV_SYNC_AUTH_IND *)para;
            if(srv_sync_auth_ind->service EQ MFW_BT_SYNC)
            {
              memcpy(&bt_data->para.srv_sync_auth,
                     para,sizeof(T_MFW_BT_SRV_SYNC_AUTH_IND));
            }
            else
              return FALSE;
            break;

          case BT_SRV_SYNC_PULL_IND:
            srv_sync_pull_ind = (T_MFW_BT_SRV_SYNC_PULL_IND *)para;
            if(srv_sync_pull_ind->service EQ MFW_BT_SYNC)
            {
              memcpy(&bt_data->para.sync_pull_ind,
                     para,sizeof(T_MFW_BT_SRV_SYNC_PULL_IND));
            }
            else
              return FALSE;
            break;

          case BT_SRV_SYNC_PUSH_IND:
            srv_sync_push_ind = (T_MFW_BT_SRV_SYNC_PUSH_IND *)para;
            if(srv_sync_push_ind->service EQ MFW_BT_SYNC)
            {
              memcpy(&bt_data->para.sync_push_ind,
                     para,sizeof(T_MFW_BT_SRV_SYNC_PUSH_IND));
            }
            else
              return FALSE;
            break;

          case BT_SRV_SYNC_PUSH_CNF:
            srv_sync_push_cnf = (T_MFW_BT_SRV_SYNC_PUSH_CNF *)para;
            if(srv_sync_push_cnf->service EQ MFW_BT_SYNC)
            {
              memcpy(&bt_data->para.sync_push_cnf,
                     para,sizeof(T_MFW_BT_SRV_SYNC_PUSH_CNF));
            }
            else
              return FALSE;
            break;
        }

        switch (event)
        {

          case BT_HSG_SPECIFIC_CMD_CFM:
            memcpy(&bt_data->para.hsg_specific_cmd_cfm, para,sizeof(T_MFW_BT_HSG_SPECIFIC_CMD_CFM));
            break;

          case BT_HSG_HEADSET_CONNECTION_IND:
            memcpy(&bt_data->para.hsg_headset_connection_ind, para,sizeof(T_MFW_BT_HSG_HEADSET_CONNECTION_IND));
            break;

          case BT_HSG_SAVE_LIST_CNF:
            memcpy(&bt_data->para.hsg_save_list_cnf, para,sizeof(T_MFW_BT_HSG_SAVE_LIST_CNF));
            break;
        }

        switch (event)
        {
          case BT_CONNECT_DEVICE_IND:
             memcpy(&bt_data->para.connect_indication,
                   para,sizeof(T_MFW_BT_CONNECT_IND));
            /* service with event BT_CONNECT_DEVICE_IND to MMI */
              break;

          case BT_CONNECT_DEVICE_INF:
             memcpy(&bt_data->para.connect_information,
                   para,sizeof(T_MFW_BT_CONNECT_INF));
            /* service with event BT_CONNECT_DEVICE_INF to MMI */
              break;


          case BT_DISCONNECT_DEVICE_CNF:
            discon_ptr = (T_MFW_BT_DISCONNECT_CNF *)para;
            if(discon_ptr->service EQ MFW_BT_HEADSET)
            {
              /* confirms disconnection */
              memcpy(&bt_data->para.disconnect_cnf,
                     para,sizeof(T_MFW_BT_DISCONNECT_CNF));
              /* service, addr with event BT_DISCONNECT_DEVICE_CNF to MMI */
              break;
            }
            else
              return FALSE;

          case BT_DISCON_DUN_FAX_CNF:
            discon_dun_fax_ptr = (T_MFW_BT_DISCON_DUN_FAX_CNF *)para;
            if((discon_dun_fax_ptr->service EQ MFW_BT_DIAL_UP) OR
               (discon_dun_fax_ptr->service EQ MFW_BT_FAX_GW))
            {
              /* confirms disconnection by remote dial up/fax */
              memcpy(&bt_data->para.disc_dun_fax_cnf,
                     para,sizeof(T_MFW_BT_DISCON_DUN_FAX_CNF));
              /* device, result with event BT_DISCON_DUN_FAX_CNF to MMI */
              break;
            }
            else
              return FALSE;

          case BT_DISCONNECT_DEVICE_IND:
            discon_i_ptr = (T_MFW_BT_DISCONNECT_IND *)para;
            if(discon_i_ptr->service EQ MFW_BT_HEADSET)
            {
              /* confirms indication */
              memcpy(&bt_data->para.disconnect_ind,
                     para,sizeof(T_MFW_BT_DISCONNECT_IND));
              /* service, addr, error_cause with event BT_DISCONNECT_DEVICE_IND to MMI */
              break;
            }
            else
              return FALSE;

          case BT_DISCON_DUN_FAX_IND:
            discon_dun_fax_ind_ptr = (T_MFW_BT_DISCON_DUN_FAX_IND *)para;
            if((discon_dun_fax_ind_ptr->service EQ MFW_BT_DIAL_UP) OR
              (discon_dun_fax_ind_ptr->service EQ MFW_BT_FAX_GW))
            {
              /* indicats disconnection by remote dial up */
              memcpy(&bt_data->para.con_dun_fax_ind,
                     para,sizeof(T_MFW_BT_DISCON_DUN_FAX_IND));
              /* device, result with event BT_DISCON_DUN_FAX_IND to MMI */
              break;
            }
            else
              return FALSE;
        }

        switch (event) /* BT CTRL */
        {
          case BT_CHNG_LOCAL_NAME:
            memcpy(&bt_data->para.chng_local_name, para, sizeof(T_MFW_BT_CHNG_LOCAL_NAME));
            break;

          case BT_READ_LOCAL_NAME:
            memcpy(&bt_data->para.read_local_name, para, sizeof(T_MFW_BT_READ_LOCAL_NAME));
            break;

          case BT_REMOTE_DEV_INFO_RES:
            memcpy(&bt_data->para.remote_dev_info, para, sizeof(T_MFW_BT_REMOTE_DEV_INFO_RES));
            break;

          case BT_CHNG_CONNECTABLE_MODE:
            memcpy(&bt_data->para.chng_conn_mode, para, sizeof(T_MFW_BT_CHNG_CONNECTABLE_MODE));
            break;

          case BT_CHNG_DISCOVERABLE_MODE:
            memcpy(&bt_data->para.chng_disc_mode, para, sizeof(T_MFW_BT_CHNG_DISCOVERABLE_MODE));
            break;

          case BT_READ_BD_ADDR:
            memcpy(&bt_data->para.read_bd_addr, para, sizeof(T_MFW_BT_READ_BD_ADDR));
            break;

          case BT_TRUSTED_DEV_LIST_FULL:
            break;

        }

        switch (event)
        {

          case BT_TRANSFER_AUDIO_IN_CNF:
            /* confirms transfer audio out */
            memcpy(&bt_data->para.audio_in_cnf,
                   para,sizeof(T_MFW_BT_TRANSFER_AUDIO_IN_CNF));
            /* service, addr with event BT_TRANSFER_AUDIO_IN_CNF to MMI */
            break;

          case BT_TRANSFER_AUDIO_OUT_CNF:
            /* confirms transfer audio in */
            memcpy(&bt_data->para.audio_out_cnf,
                   para,sizeof(T_MFW_BT_TRANSFER_AUDIO_OUT_CNF));
            /* service, addr with event BT_TRANSFER_AUDIO_OUT_CNF to MMI */
            break;

          case BT_AUTHORIZATION_IND:
            /* indicats remote device authorization request */
            memcpy(&bt_data->para.authoriz_ind,
                   para,sizeof(T_MFW_BT_AUTHORIZATION_IND));
            /* addr,remote name,service name, authorization_mask,connection_dir with event BT_AUTHORIZATION_IND to MMI */
            break;

          case BT_PIN_IND:
            /* indicats remote pin request */
            memcpy(&bt_data->para.pin_ind,
                   para,sizeof(T_MFW_BT_PIN_IND));
            /* addr,name with event BT_PIN_IND to MMI */
            break;

          case BT_RECONFIG_PROFILE_CNF:
            /* confirm reconfiguration profile  */
            memcpy(&bt_data->para.profile,
                   para,sizeof(T_MFW_BT_PROFILE_CNF));
            /* service, result_bd(,mfw_opp_mode,mfw_sync_mode,mfw_syn_aut_mode) with event BT_RECONFIG_PROFILE_CNF to MMI */
            break;

          case BT_DEVICE_PAIRED_IND:
            /* confirm pairing procedure  */
            memcpy(&bt_data->para.pair_ind,
                   para,sizeof(T_MFW_BT_DEV_PAIR_IND));
            /* remote addr, remote name and pair_res with event BT_DEVICE_PAIRED_IND to MMI */
            break;

          case BT_CALL_MONITORING_STATUS:
            /* indicats call monitoring events by dun/fax */
            memcpy(&bt_data->para.call_status,
                   para,sizeof(T_MFW_BT_CALL_STATUS_DUN_FAX));
            /* device and call status with event BT_CALL_MONITORING_STATUS to MMI */
            break;

          case BT_OPP_SERV_PUT_IND:
            /* indicats put request by opp server */
            memcpy(&bt_data->para.opp_s_obj,
                   para,sizeof(T_MFW_BT_OPP_PUT_IND));
            /* device,subtype,client_addr,object with event BT_OPP_SERV_PUT_IND to MMI */
            break;

          case BT_OPP_SERV_PUT_CNF:
            /* indicats put request by opp server (server side) */
            memcpy(&bt_data->para.opp_s_obj_cnf,
                   para,sizeof(T_MFW_BT_OPP_PUT_CNF));
            /* device,subtype,client_addr,object,error with event BT_OPP_SERV_PUT_CNF to MMI */
            break;

          case BT_OPP_OBJECT_PUSH_CNF:
            /* indicats push object to opp server (client side)*/
            memcpy(&bt_data->para.opp_cl_push_ob_cnf,
                   para,sizeof(T_MFW_BT_OPP_PUSH_CNF));
            /* device,subtype,server_addr,object,error with event BT_OPP_OBJECT_PUSH_CNF to MMI */
            break;

          case BT_OPP_OBJECT_PULL_CNF:
            /* indicats push object from opp server (client side)*/
            memcpy(&bt_data->para.opp_cl_pull_ob_cnf,
                   para,sizeof(T_MFW_BT_OPP_PULL_CNF));
            /* device,subtype,server_addr,object,error with event BT_OPP_OBJECT_PULL_CNF to MMI */
            break;
          }

/*#ifdef PCA_6350*/
        switch (event)
        {
          case BT_PCA_GW_STATUS_CFM:
            memcpy(&bt_data->para.pca_gw_status, para, sizeof(T_MFW_BT_PCA_GW_STATUS_CFM));
            break;
          case BT_PCA_GW_LINK_MONITORING:
            memcpy(&bt_data->para.pca_link_mon, para, sizeof(T_MFW_BT_PCA_GW_LINK_MONITORING));
            break;
          case BT_PCA_GW_CALL_MONITORING:
            memcpy(&bt_data->para.pca_call_mon, para, sizeof(T_MFW_BT_PCA_GW_CALL_MONITORING));
            break;
          case BT_PCA_GW_HANGUP_CFM:
            memcpy(&bt_data->para.pca_hangup, para, sizeof(T_MFW_BT_PCA_GW_HANGUP_CFM));
            break;
        }
/*#endif*/ /* PCA_6350 */
        if (bt_data->handler)
        {
          current_mfw_elem = cur_elem;
          /* to call back function in mmi  */
          if ((*(bt_data->handler))(bt_data->event,
              (void *) &bt_data->para))
          return TRUE;
        }
      }
    }/* next element for look for type of handle */
    cur_elem = cur_elem->next;
  }
  return FALSE;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE : bt_init_profile_hsg_cl|
+--------------------------------------------------------------------+

  PURPOSE : configure and initialize a BT headset profile in client mode

*/
T_MFW_BT_RESULT_BT bt_init_profile_hsg_cl (T_MFW_BT_SERVICE_TYPE service,
                                  T_MFW_BT_HSG_CLIENT_CONFIG mfw_config)
{
  T_BTI_HSG_CLIENT_CONF bt_config;

  TRACE_FUNCTION ("bt_init_profile_hsg_cl()");

  memset(&bt_config,0,sizeof(bt_config));

  if(service EQ MFW_BT_HEADSET)
  {
    /* type of connection:
    ***  MFW_BT_AUTO_OUTG_DEF_CONN_OFF or
         MFW_BT_AUTO_OUTG_DEF_CONN_ON
        (connection with default pre-defined headset as soon as it receives a
        RING command from GSM)
    ***
    */
    switch(mfw_config.config_mode)
    {
        case MFW_BT_AUTO_OUTG_DEF_CONN_OFF:
          bt_config.config_mode = BTI_AUTO_OUTG_DEF_CONN_OFF;
          break;
        case MFW_BT_AUTO_OUTG_DEF_CONN_ON:
          bt_config.config_mode = BTI_AUTO_OUTG_DEF_CONN_ON;
          break;
    }

    bt_config.security = mfw_config.security;

    switch(btibtp_init_profile_hsg_req_cl((T_BTI_DEVICE_TYPE)service,bt_config))
    {/* BTP_INIT_PROFILE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE : bt_init_profile_hsg_serv|
+----------------------------------------------------------------------+

  PURPOSE : configure and initialize a BT headset profile in server mode

*/
T_MFW_BT_RESULT_BT bt_init_profile_hsg_serv (T_MFW_BT_SERVICE_TYPE service,
                                  T_MFW_BT_HSG_SERVER_CONFIG mfw_config)
{
  T_BTI_HSG_SERVER_CONF bt_config;

  TRACE_FUNCTION ("bt_init_profile_hsg_serv()");

  memset(&bt_config,0,sizeof(bt_config));

  if(service EQ MFW_BT_HEADSET)
  {
    bt_config.conn_config = mfw_config.serv_con_conf;
    /* length scan of connection requests of remote headsets *
    /* default value: MFW_BT_CONN_SCAN_TIME_DEFAULT_VALUE */
    bt_config.conn_time = mfw_config.conn_time;
    /* period between 2 scans of connection of remote headsets */
    /* default value: MFW_BT_CONN_SCAN_BREAK_DEFAULT_VALUE */
    bt_config.conn_break= mfw_config.conn_break;
    /* number of phone number in phone list */
    bt_config.nb_phone = mfw_config.nb_phone;
    /* phone number list of remote headset */
    memcpy(bt_config.phon_list,mfw_config.mfw_phone_list,sizeof(mfw_config.mfw_phone_list));
    /* associatd key list of  remote headset */
    memcpy(bt_config.key_list,mfw_config.mfw_key_list,sizeof(mfw_config.mfw_key_list));

    bt_config.security = mfw_config.security;

    switch(btibtp_init_profile_hsg_req_serv((T_BTI_DEVICE_TYPE)service,bt_config))
    {/* BTP_INIT_PROFILE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_init_profile_dun|
+--------------------------------------------------------------------+

  PURPOSE : configure and initialize a BT dialup profile

*/
T_MFW_BT_RESULT_BT bt_init_profile_dun (T_MFW_BT_SERVICE_TYPE service,
                                     T_MFW_BT_DUN_CONFIG dun_filter)
{
  T_BTI_DUN_CONF dun_config;

  TRACE_FUNCTION ("bt_init_profile_dun()");

  memset(&dun_config,0,sizeof(dun_config));

  if(service EQ MFW_BT_DIAL_UP)
  {/* show status of BT link - informs if a serial port is opened between DUN-DT and DUN-GW */
    if((dun_filter.link_event EQ BTI_DUN_LINK_MONIT_ON) OR
       (dun_filter.link_event EQ BTI_DUN_LINK_MONIT_OFF))
    {
      dun_config.link_filter = dun_filter.link_event;
    }
    else
    {
        TRACE_EVENT ("Error:bt_init_profile_dun()");
        return MFW_BT_FAIL;
    }
    /* show status of data call */
    if((dun_filter.call_event EQ BTI_DUN_CALL_MONIT_ON) OR
       (dun_filter.call_event EQ BTI_DUN_CALL_MONIT_OFF))
    {
      dun_config.call_filter = dun_filter.call_event;
    }
    else
    {
        TRACE_EVENT ("Error:bt_init_profile_dun()");
        return MFW_BT_FAIL;
    }

    switch(btibtp_init_profile_dun_req((T_BTI_DEVICE_TYPE)service,dun_config))
    {/* BTP_INIT_PROFILE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_init_profile_fax|
+--------------------------------------------------------------------+

  PURPOSE : configure and initialize a BT fax gateway profile

*/
T_MFW_BT_RESULT_BT bt_init_profile_fax (T_MFW_BT_SERVICE_TYPE service,
                                     T_MFW_BT_FAX_CONFIG fax_filter)
{
  T_BTI_FAX_CONF fax_config;

  TRACE_FUNCTION ("bt_init_profile_fax()");

  memset(&fax_config,0,sizeof(fax_config));

  if(service EQ MFW_BT_FAX_GW)
  {/* show status of BT link - informs if a serial port is opened between FAX-DT and FAX-GW */
    if((fax_filter.link_event EQ BTI_FAX_LINK_MONIT_ON) OR
       (fax_filter.link_event EQ BTI_FAX_LINK_MONIT_OFF))
    {
      fax_config.link_filter = fax_filter.link_event;
    }
    else
    {
        TRACE_EVENT ("Error:bt_init_profile_fax()");
        return MFW_BT_FAIL;
    }
    /* show status of data call */
    if((fax_filter.call_event EQ BTI_FAX_CALL_MONIT_ON) OR
       (fax_filter.call_event EQ BTI_FAX_CALL_MONIT_OFF))
    {
      fax_config.call_filter = fax_filter.call_event;
    }
    else
    {
        TRACE_EVENT ("Error:bt_init_profile_fax()");
        return MFW_BT_FAIL;
    }

    switch(btibtp_init_profile_fax_req((T_BTI_DEVICE_TYPE)service,fax_config))
    {/* BTP_INIT_PROFILE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_init_profile_opp|
+--------------------------------------------------------------------+

  PURPOSE : configure and initialize a BT OPP client or server
*/
T_MFW_BT_RESULT_BT bt_init_profile_opp (T_MFW_BT_SERVICE_TYPE service,
                                        T_MFW_BT_SUBTYPE_DEV subtype,
                                        T_MFW_BT_OPP_SERVER_CONFIG serv_config)
{
  T_BTI_OPP_SERVER_CONFIG bt_opp_s_conf;

  TRACE_FUNCTION ("bt_init_profile_opp()");

  memset(&bt_opp_s_conf,0,sizeof(bt_opp_s_conf));

  if(service EQ MFW_BT_OPP)
  {
    if(subtype EQ MFW_BT_CLIENT)
    {
      switch(btibtp_init_profile_opp_cl())
      {/* BTP_INIT_PROFILE_REQ to BT */
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else if(subtype EQ MFW_BT_SERVER)
    {
      /* mode of OPP server:
         MFW_BT_OPP_NO_MODE: no OPP server mode
         MFW_BT_OPP_SILENT_MODE: OPP server stores objects as files without alerting the user
         MFW_BT_OPP_FILE_MODE: OPP server stores objects as files with alerting user
         MFW_BT_OPP_BUFFER_MODE: OPP server asks for a buffer to store the each object
      */

      switch(serv_config.mfw_opp_mode)
      {
        case MFW_BT_OPP_NO_MODE:
          bt_opp_s_conf.bt_opp_mode = BTI_NO_OPP_MODE;
          break;
        case MFW_BT_OPP_BUFFER_MODE:
          bt_opp_s_conf.bt_opp_mode = BTI_OPP_EVENT_BUFFER_MODE;
          break;
        case MFW_BT_OPP_FILE_MODE:
          bt_opp_s_conf.bt_opp_mode = BTI_OPP_EVENT_FILE_MODE;
          break;
        case MFW_BT_OPP_SILENT_MODE:
          bt_opp_s_conf.bt_opp_mode = BTI_OPP_SILENT_FILE_MODE;
          break;
        default:
          return MFW_BT_FAIL;
      }
      /* this function change the parameter of the existing OPP server */
      /* initalizing of an OBEX server and service, registration in SDP and enabling of OPP server took place
      after switching power on */
      switch(serv_config.mfw_opp_object.mfw_object_type)
      {
        case MFW_BT_BUFFER:/* object is stored in buffer */
          bt_opp_s_conf.bt_object.bt_object_type = BTI_TYPE_BUFFER;
          bt_opp_s_conf.bt_object.bt_buffer_start = serv_config.mfw_opp_object.mfw_buffer_start;
          bt_opp_s_conf.bt_object.bt_buffer_size = serv_config.mfw_opp_object.mfw_buffer_size;
          break;
        case MFW_BT_PATH:/* object is stored in file system */
          bt_opp_s_conf.bt_object.bt_object_type = BTI_TYPE_PATH;
          /* path of default_business_card of server */
          bt_opp_s_conf.bt_object.bt_path = serv_config.mfw_opp_object.mfw_path;
          /* path of inbox folder in file system of OPP server */
          bt_opp_s_conf.bt_inbox_path = serv_config.mfw_inbox_path;
          break;
        default:
          return MFW_BT_FAIL;

      }
      /* MIME type of object, null-terminated string */
      bt_opp_s_conf.bt_object.bt_object_mime_type = serv_config.mfw_opp_object.mfw_object_mime_type;
      /* name of default_business_card of server */
      bt_opp_s_conf.bt_object.bt_object_name = serv_config.mfw_opp_object.mfw_object_name;
      /* length of default_business_card of server */
      bt_opp_s_conf.bt_object.bt_object_length = serv_config.mfw_opp_object.mfw_object_length;

      switch(btibtp_init_profile_opp_s(bt_opp_s_conf))
      {/* BTP_INIT_PROFILE_REQ to BT */
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  return MFW_BT_FAIL;
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE : bt_init_profile_syn_s   |
+----------------------------------------------------------------------+

  PURPOSE : configure and initialize a BT SYNC profile in server mode

*/
T_MFW_BT_RESULT_BT bt_init_profile_syn_s (T_MFW_BT_SERVICE_TYPE service,
                                          T_MFW_BT_SUBTYPE_DEV subtype,
                                  T_MFW_BT_SYN_OBJECT_STORE_LIST list_availabe_objects,
                                  T_MFW_BT_SYNC_SERVER_CONFIG mfw_config)
{
  T_BTI_SYNC_SERVER_CONFIG bt_config;
  T_BTI_SYN_OBJECT_STORE_LIST bt_list_sync_objects;
  TRACE_FUNCTION ("bt_init_profile_syn_s()");

  memset(&bt_config,0,sizeof(bt_config));
  bt_list_sync_objects = (T_BTI_SYN_OBJECT_STORE_LIST)list_availabe_objects;

  if(service EQ MFW_BT_SYNC)
  {
    if(subtype EQ MFW_BT_SERVER)
    {
      switch(mfw_config.mfw_syn_srv_mode)
      {
        case MFW_BT_SYNC_GEN_MODE:
          /* general SYNC server mode, for devices
          which are made discoverable continuously or for no
          specific conditions, the server is in connectable mode; the Sync
          client connects the server and starts the synchronization */
          bt_config.bt_sync_srv_mode = mfw_config.mfw_syn_srv_mode;
          break;
        case MFW_BT_SYNC_INIT_MODE:
          /* limited inquiry scan for devices which
          are made discoverable only for a limited period of time or
          for specific conditions; the server is in connectable and pairable mode */
          bt_config.bt_sync_srv_mode = mfw_config.mfw_syn_srv_mode;
          break;
        default:
          return MFW_BT_FAIL;
      }
      switch(mfw_config.mfw_syn_srv_auth_mode)
      {
        case MFW_BT_SYNC_INIT_AUTH_MODE:
          /* server initiates authentication */
          bt_config.bt_sync_aut_mode = mfw_config.mfw_syn_srv_auth_mode;
          break;
        case MFW_BT_SYNC_NO_INIT_AUTH_MODE:
          /* server does not initiate authentication*/
          bt_config.bt_sync_aut_mode = mfw_config.mfw_syn_srv_auth_mode;
          break;
        default:
          return MFW_BT_FAIL;
      }

      switch(btibtp_init_profile_sync_req_serv(bt_list_sync_objects,bt_config))
      { /* BTP_INIT_PROFILE_REQ to BT */
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  return MFW_BT_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)      MODULE  : MFW_BT               |
| STATE   : code                      ROUTINE : bt_reconf_profile_opp|
+--------------------------------------------------------------------+

  PURPOSE : reconfigure a BT OPP in server mode
*/
T_MFW_BT_RESULT_BT bt_reconf_profile_opp (T_MFW_BT_SERVICE_TYPE service,
                                        T_MFW_BT_SUBTYPE_DEV subtype,
                                        T_MFW_BT_OPP_SERVER_CONFIG serv_config)
{
  T_BTI_OPP_SERVER_CONFIG bt_opp_s_conf;

  TRACE_FUNCTION ("bt_init_profile_opp()");

  memset(&bt_opp_s_conf,0,sizeof(bt_opp_s_conf));
  if(service EQ MFW_BT_OPP)
  {
    if(subtype EQ MFW_BT_SERVER)
    {
      /* mode of OPP server:
         MFW_BT_OPP_NO_MODE: no OPP server mode
         MFW_BT_OPP_SILENT_MODE: OPP server stores objects as fileswithout alerting the user
         MFW_BT_OPP_FILE_MODE: OPP server stores objects as fileswith alerting user
         MFW_BT_OPP_BUFFER_MODE: OPP server asks for a buffer to store the each object
      */
      switch(serv_config.mfw_opp_mode)
      {
        case MFW_BT_OPP_NO_MODE:
          bt_opp_s_conf.bt_opp_mode = BTI_NO_OPP_MODE;
          break;
        case MFW_BT_OPP_BUFFER_MODE:
          bt_opp_s_conf.bt_opp_mode = BTI_OPP_EVENT_BUFFER_MODE;
          break;
        case MFW_BT_OPP_FILE_MODE:
          bt_opp_s_conf.bt_opp_mode = BTI_OPP_EVENT_FILE_MODE;
          break;
        case MFW_BT_OPP_SILENT_MODE:
          bt_opp_s_conf.bt_opp_mode = BTI_OPP_SILENT_FILE_MODE;
          break;
        default:
          return MFW_BT_FAIL;
      }
      /* this function change the parameter of the existing OPP server */
      switch(serv_config.mfw_opp_object.mfw_object_type)
      {
        case MFW_BT_BUFFER:/* object is stored in buffer */
          bt_opp_s_conf.bt_object.bt_object_type = BTI_TYPE_BUFFER;
          bt_opp_s_conf.bt_object.bt_buffer_start = serv_config.mfw_opp_object.mfw_buffer_start;
          bt_opp_s_conf.bt_object.bt_buffer_size = serv_config.mfw_opp_object.mfw_buffer_size;
          break;
        case MFW_BT_PATH:/* object is stored in file system */
          bt_opp_s_conf.bt_object.bt_object_type = BTI_TYPE_PATH;
          /* path of default_business_card of server */
          bt_opp_s_conf.bt_object.bt_path = serv_config.mfw_opp_object.mfw_path;
          /* path of inbox folder in file system of OPP server */
          bt_opp_s_conf.bt_inbox_path = serv_config.mfw_inbox_path;
          break;
        default:
          return MFW_BT_FAIL;
      }
      /* MIME type of object, null-terminated string */
      bt_opp_s_conf.bt_object.bt_object_mime_type = serv_config.mfw_opp_object.mfw_object_mime_type;
      /* name of default_business_card of server */
      bt_opp_s_conf.bt_object.bt_object_name = serv_config.mfw_opp_object.mfw_object_name;
      /* length of default_business_card of server */
      bt_opp_s_conf.bt_object.bt_object_length = serv_config.mfw_opp_object.mfw_object_length;

      switch(btibtp_reconf_profile_opp_s(bt_opp_s_conf))
      {/* BTP_RECONFIG_PROFILE_REQ to BT */
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  return MFW_BT_FAIL;
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE : bt_reconf_profile_syn_s   |
+----------------------------------------------------------------------+

  PURPOSE : reconfigure a BT SYNC profile in server mode

*/
T_MFW_BT_RESULT_BT bt_reconf_profile_syn_s (T_MFW_BT_SERVICE_TYPE service,
                                          T_MFW_BT_SUBTYPE_DEV subtype,
                                  T_MFW_BT_SYN_OBJECT_STORE_LIST list_availabe_objects,
                                  T_MFW_BT_SYNC_SERVER_CONFIG mfw_config)
{
  T_BTI_SYNC_SERVER_CONFIG bt_config;
  T_BTI_SYN_OBJECT_STORE_LIST bt_list_sync_objects;
  TRACE_FUNCTION ("bt_reconf_profile_syn_s()");

  memset(&bt_config,0,sizeof(bt_config));
  bt_list_sync_objects = (T_BTI_SYN_OBJECT_STORE_LIST)list_availabe_objects;

  if(service EQ MFW_BT_SYNC)
  {
    switch(mfw_config.mfw_syn_srv_mode)
    {
      case MFW_BT_SYNC_GEN_MODE:
        /* general SYNC server mode, for devices
        which are made discoverable continuously or for no
        specific conditions */
        bt_config.bt_sync_srv_mode = mfw_config.mfw_syn_srv_mode;
        break;
      case MFW_BT_SYNC_INIT_MODE:
        /* limited inquiry scan for devices which
        are made discoverable only for a limited period of time or
        for specific conditions */
        bt_config.bt_sync_srv_mode = mfw_config.mfw_syn_srv_mode;
        break;
      default:
        return MFW_BT_FAIL;
    }
    switch(mfw_config.mfw_syn_srv_auth_mode)
    {
      case MFW_BT_SYNC_INIT_AUTH_MODE:
        /* server initiates authentication */
        bt_config.bt_sync_aut_mode = mfw_config.mfw_syn_srv_auth_mode;
        break;
      case MFW_BT_SYNC_NO_INIT_AUTH_MODE:
        /* server does not initiate authentication*/
        bt_config.bt_sync_aut_mode = mfw_config.mfw_syn_srv_auth_mode;
        break;
      default:
        return MFW_BT_FAIL;
    }

    switch(btibtp_reconf_profile_sync_req_serv(bt_list_sync_objects,bt_config))
    { /* BTP_RECONFIG_PROFILE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE : bt_deinit_profile  |
+--------------------------------------------------------------------+

  PURPOSE : deinitialize a BT profile, not stop of profile application

*/
T_MFW_BT_RESULT_BT bt_deinit_profile (T_MFW_BT_SERVICE_TYPE service,T_MFW_BT_SUBTYPE_DEV subtype)
{
  TRACE_FUNCTION ("bt_deinit_profile()");

  if((service EQ MFW_BT_HEADSET) OR
     (service EQ MFW_BT_DIAL_UP) OR
     (service EQ MFW_BT_FAX_GW)  OR
     (service EQ MFW_BT_OPP)  OR
     (service EQ MFW_BT_SYNC))
  {
    if((subtype EQ MFW_BT_CLIENT) OR
       (subtype EQ MFW_BT_SERVER) OR
       (subtype EQ MFW_BT_NO_SUBTYPE))
    {
      switch(btibtp_deinit_profile_req((T_BTI_DEVICE_TYPE)service,(T_BTI_DEVICE_SUBTYP)subtype))
      { /* BTP_DEINIT_PROFILE_REQ to BT */
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
  }
  return MFW_BT_FAIL;
}

/*
+-----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)    MODULE  : MFW_BT                    |
| STATE   : code                    ROUTINE : bt_reconfig_profile_hsg_cl|
+-----------------------------------------------------------------------+

  PURPOSE : reconfigure a BT profile headset in client mode

*/
T_MFW_BT_RESULT_BT bt_reconfig_profile_hsg_cl (T_MFW_BT_SERVICE_TYPE service,
                                     T_MFW_BT_HSG_CLIENT_CONFIG config)
{
  T_BTI_HSG_CLIENT_CONF bt_config;

  TRACE_FUNCTION ("bt_reconfig_profile_hsg_cl()");

  memset(&bt_config,0,sizeof(bt_config));

  if(service EQ MFW_BT_HEADSET)
  {/* reconfiguration is only accepted when no outgoing call is in processing */
    switch(config.config_mode)
    {
      case MFW_BT_AUTO_OUTG_DEF_CONN_OFF:
        bt_config.config_mode = BTI_AUTO_OUTG_DEF_CONN_OFF;
        break;
      case MFW_BT_AUTO_OUTG_DEF_CONN_ON:
        bt_config.config_mode = BTI_AUTO_OUTG_DEF_CONN_ON;
        break;
    }

    bt_config.security = config.security;

    switch(btibtp_reconfig_profile_req_cl((T_BTI_DEVICE_TYPE)service,bt_config))
    { /* BTP_RECONF_PROFILE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                      |
| STATE   : code                     ROUTINE : bt_reconfig_profile_hsg_serv|
+--------------------------------------------------------------------------+

  PURPOSE : reconfigure a BT headset profile in server mode

*/
T_MFW_BT_RESULT_BT bt_reconfig_profile_hsg_serv (T_MFW_BT_SERVICE_TYPE service,
                                  T_MFW_BT_HSG_SERVER_CONFIG mfw_config)
{
  T_BTI_HSG_SERVER_CONF bt_config;

  TRACE_FUNCTION ("bt_reconfig_profile_hsg_serv()");

  memset(&bt_config,0,sizeof(bt_config));

  if(service EQ MFW_BT_HEADSET)
  {
    bt_config.conn_config = mfw_config.serv_con_conf;
    /* length scan of connection requests of remote headsets *
    /* default value: MFW_BT_CONN_SCAN_TIME_DEFAULT_VALUE */
    bt_config.conn_time = mfw_config.conn_time;
    /* period between 2 scans of connection of remote headsets */
    /* default value: MFW_BT_CONN_SCAN_BREAK_DEFAULT_VALUE */
    bt_config.conn_break= mfw_config.conn_break;
    /* number of phone number in phone list */
    bt_config.nb_phone = mfw_config.nb_phone;
    /* phone number list of remote headset */
    memcpy(bt_config.phon_list,mfw_config.mfw_phone_list,sizeof(mfw_config.mfw_phone_list));
    /* associatd key list of  remote headset */
    memcpy(bt_config.key_list,mfw_config.mfw_key_list,sizeof(mfw_config.mfw_key_list));

    bt_config.security = mfw_config.security;

    switch(btibtp_reconfig_profile_req_serv((T_BTI_DEVICE_TYPE)service,bt_config))
    {/* BTP_INIT_PROFILE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)    MODULE  : MFW_BT                 |
| STATE   : code                    ROUTINE : bt_reconfig_profile_dun|
+--------------------------------------------------------------------+

  PURPOSE : reconfigure a BT profile

*/
T_MFW_BT_RESULT_BT bt_reconfig_profile_dun (T_MFW_BT_SERVICE_TYPE service,
                                     T_MFW_BT_DUN_CONFIG dun_filter)
{
  T_BTI_DUN_CONF dun_config;

  TRACE_FUNCTION ("bt_reconfig_profile_dun()");

  memset(&dun_config,0,sizeof(dun_config));

  if(service EQ MFW_BT_DIAL_UP )
  {/* show status of BT link - informs if a serial port is opened between DUN-DT and DUN-GW */
    if((dun_filter.link_event EQ BTI_DUN_LINK_MONIT_ON) OR
       (dun_filter.link_event EQ BTI_DUN_LINK_MONIT_OFF))
    {
      dun_config.link_filter = dun_filter.link_event;
    }
    else
    {
        TRACE_EVENT ("Error:bt_reconfig_profile_dun()");
        return MFW_BT_FAIL;
    }
    /* show status of data call */
    if((dun_filter.call_event EQ BTI_DUN_CALL_MONIT_ON) OR
       (dun_filter.call_event EQ BTI_DUN_CALL_MONIT_OFF))
    {
      dun_config.call_filter = dun_filter.call_event;
    }
    else
    {
        TRACE_EVENT ("Error:bt_reconfig_profile_dun()");
        return MFW_BT_FAIL;
    }

    switch(btibtp_reconf_profile_dun_req((T_BTI_DEVICE_TYPE)service,dun_config))
    { /* BTP_RECONF_PROFILE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)    MODULE  : MFW_BT                 |
| STATE   : code                    ROUTINE : bt_reconfig_profile_fax|
+--------------------------------------------------------------------+

  PURPOSE : reconfigure a BT profile

*/
T_MFW_BT_RESULT_BT bt_reconfig_profile_fax (T_MFW_BT_SERVICE_TYPE service,
                                     T_MFW_BT_FAX_CONFIG fax_filter)
{
  T_BTI_FAX_CONF fax_config;

  TRACE_FUNCTION ("bt_reconfig_profile_fax()");

  memset(&fax_config,0,sizeof(fax_config));

  if(service EQ MFW_BT_FAX_GW )
  {/* show status of BT link - informs if a serial port is opened between FAX-DT and FAX-GW */
    if((fax_filter.link_event EQ BTI_FAX_LINK_MONIT_ON) OR
       (fax_filter.link_event EQ BTI_FAX_LINK_MONIT_OFF))
    {
      fax_config.link_filter = fax_filter.link_event;
    }
    else
    {
        TRACE_EVENT ("Error:bt_reconfig_profile_fax()");
        return MFW_BT_FAIL;
    }
    /* show status of data call */
    if((fax_filter.call_event EQ BTI_FAX_CALL_MONIT_ON) OR
       (fax_filter.call_event EQ BTI_FAX_CALL_MONIT_OFF))
    {
      fax_config.call_filter = fax_filter.call_event;
    }
    else
    {
        TRACE_EVENT ("Error:bt_reconfig_profile_fax()");
        return MFW_BT_FAIL;
    }

    switch(btibtp_reconf_profile_fax_req((T_BTI_DEVICE_TYPE)service,fax_config))
    { /* BTP_RECONF_PROFILE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)          MODULE  : MFW_BT           |
| STATE   : code                          ROUTINE : bt_service_search|
+--------------------------------------------------------------------+

  PURPOSE : request search of services and their service names

*/
T_MFW_BT_RESULT_BT  bt_service_search(T_MFW_BT_SERVICE_TYPE service)
{
  TRACE_FUNCTION ("bt_service_search()");

  if((service EQ MFW_BT_HEADSET) OR
     (service EQ MFW_BT_DIAL_UP) OR
     (service EQ MFW_BT_FAX_GW) OR
     (service EQ MFW_BT_OPP) OR
     (service EQ MFW_BT_SYNC_CMD))
  {/* start browsing procedure */
    switch(btibtp_service_search_req((T_BTI_DEVICE_TYPE)service, SERVICE_SEARCH))
    {
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)          MODULE  : MFW_BT           |
| STATE   : code                          ROUTINE : bt_device_search |
+--------------------------------------------------------------------+

  PURPOSE : request search of devices and their service id's

*/
T_MFW_BT_RESULT_BT  bt_device_search(void)
{

  TRACE_FUNCTION ("bt_device_search()");

  /* start browsing procedure */
  switch(btibtp_device_search_req(DEVICE_SEARCH))
  {
    case BTP_OK:
      return MFW_BT_EXECUTE;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)          MODULE  : MFW_BT           |
| STATE   : code                          ROUTINE : bt_extended_search |
+--------------------------------------------------------------------+

  PURPOSE : request search of devices and their service id's

*/
T_MFW_BT_RESULT_BT  bt_extended_search(UINT8 inq_length,
                     UINT8                  max_num_of_responses,
                     T_MFW_DEVICE_CLASS     class_of_device[],
                     BOOLEAN                need_device_name,
                     BOOLEAN                need_services,
                     T_MFW_SERVICE_ID       service_id)
{

  TRACE_FUNCTION ("bt_extended_search()");

  /* start browsing procedure */
  switch(btibtp_extended_search_req(inq_length, max_num_of_responses,
                                    class_of_device, need_device_name,
                                    need_services, service_id))
  {
    case BTP_OK:
      return MFW_BT_EXECUTE;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)          MODULE  : MFW_BT           |
| STATE   : code                          ROUTINE : bt_search_abort  |
+--------------------------------------------------------------------+

  PURPOSE : start abort of browsing procedure

*/
T_MFW_BT_RESULT_BT bt_search_abort(void)
{
  TRACE_FUNCTION ("bt_search_abort()");

  switch(btibtp_search_abort())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE : bt_connect_hsg        |
+--------------------------------------------------------------------+

  PURPOSE : request connection with the service headset

*/
T_MFW_BT_RESULT_BT  bt_connect_hsg (T_MFW_BT_SERVICE_TYPE service,
                                    T_MFW_BT_REQ_ID req_id,
                                    T_MFW_BT_BD_ADDR bd_addr[], UINT8 mode)
{
  TRACE_FUNCTION ("bt_connect_hsg()");

  /*
  *** if req_id equal MFW_BT_DEFAULT_HEADSET_ID: headset gateway will
                      automatically select a headset from the default headset list
      if req_id equal MFW_BT_INVALID_HEADSET_ID: headset gateway will try to
                      connect to the headset whose BT headset address is in bd_addr
      otherwise headset gateway will call the headset related to headset_id
                      in the default headset list
  ***
  */
   /* BTP_CONNECT_DEVICE_REQ to BT */
  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_connect_hsg_req(req_id, bd_addr, mode))
    {
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)         MODULE  : MFW_BT            |
| STATE   : code                         ROUTINE :bt_get_auto_connect|
+--------------------------------------------------------------------+

  PURPOSE : get state automatic connection on/off for headset gateway
*/

T_MFW_BT_RESULT_BT bt_get_auto_connect(T_MFW_BT_SERVICE_TYPE service,
                                    T_MFW_BT_HSG_CLIENT_CONFIG *conf)
{
  T_MFW_BT_HSG_CLIENT_CONFIG client_conf;
  T_MFW_BT_HSG_SERVER_CONFIG server_conf;
  T_MFW_BT_RESULT_BT result;

  TRACE_FUNCTION ("bt_get_auto_connect()");

  memset(&client_conf,0,sizeof(client_conf));
  memset(&server_conf,0,sizeof(server_conf));

  result = bt_get_config((T_BTI_DEVICE_TYPE)service,&client_conf,&server_conf);

  if(result EQ MFW_BT_OK)
  {
    conf->config_mode = (T_MFW_BT_CONFIG_MODE)client_conf.config_mode;
  }
  return result;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)            MODULE  : MFW_BT         |
| STATE   : code                            ROUTINE :bt_auto_connect |
+--------------------------------------------------------------------+

  PURPOSE : set feature automatic connection on/off for headset gateway
*/

T_MFW_BT_RESULT_BT bt_auto_connect(T_MFW_BT_SERVICE_TYPE service,
                                T_MFW_BT_AUTO_CONNECT_STATE set_state)
{
  T_MFW_BT_HSG_CLIENT_CONFIG client_conf;
  T_MFW_BT_HSG_SERVER_CONFIG server_conf;
  T_MFW_BT_RESULT_BT result;

  TRACE_FUNCTION ("bt_auto_connect()");

  memset(&client_conf,0,sizeof(client_conf));
  memset(&server_conf,0,sizeof(server_conf));

  if(set_state EQ MFW_BT_AUTO_CONNECT_ON OR
     set_state EQ MFW_BT_AUTO_CONNECT_OFF)
  {/* read state of config_mode */
    result = bt_get_config((T_BTI_DEVICE_TYPE)service,&client_conf,&server_conf);
    if(result NEQ MFW_BT_OK)
    {
      return result;
    }
    else
    {
      if(set_state EQ MFW_BT_AUTO_CONNECT_ON)
      {
        if(client_conf.config_mode NEQ MFW_BT_AUTO_OUTG_DEF_CONN_ON)
        {/*  connection with default pre-defined headset as soon as it receives a
             RING command from GSM)*/
          client_conf.config_mode = MFW_BT_AUTO_OUTG_DEF_CONN_ON;
          return (bt_reconfig_profile_hsg_cl(service,client_conf));
        }
      }
      else
      {
        if(client_conf.config_mode EQ MFW_BT_AUTO_OUTG_DEF_CONN_ON)
        {
          client_conf.config_mode = MFW_BT_AUTO_OUTG_DEF_CONN_OFF;
          return (bt_reconfig_profile_hsg_cl(service,client_conf));
        }
      }
    }
  }
  else
    return MFW_BT_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)         MODULE  : MFW_BT            |
| STATE   : code                        ROUTINE :bt_set_pref_headset |
+--------------------------------------------------------------------+

  PURPOSE : add preferred headset in default headset list
*/

T_MFW_BT_RESULT_BT bt_set_pref_headset(T_MFW_BT_SERVICE_TYPE service,
                                       T_MFW_BT_BD_ADDR bd_addr[],
                                       T_MFW_BT_HSG_NAME hsg_name[],
                                       T_MFW_BT_PRIORITY priority,
                                       BOOL rem_audio_ctr_supp_hsg,
                                       T_MFW_BT_CNF_ID * headset_cnf_id)
{
  T_MFW_BT_CNF_ID cnf_id = 0;

  TRACE_FUNCTION ("bt_set_pref_service()");

  if(service EQ MFW_BT_HEADSET)
  {
    if(priority <= MFW_HSG_HEADSET_MAX_PRIORITY)
    {
      switch(btibtp_set_default_headset(bd_addr,hsg_name,priority,rem_audio_ctr_supp_hsg,&cnf_id))
      /* add headset in default headset list */
      {
        case BTP_OK:
          *headset_cnf_id = cnf_id;
          return MFW_BT_OK;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_HSG_ALREADY_STOR:
          return MFW_BT_HSG_ALREADY_STORED;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_INVALID_PARA;
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)         MODULE  : MFW_BT            |
| STATE   : code                         ROUTINE :bt_is_pref_headset |
+--------------------------------------------------------------------+

  PURPOSE : check if preferred headset is in default headset list
*/

T_MFW_BT_RESULT_BT bt_is_pref_headset(T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_BD_ADDR bd_addr[],
                                      T_MFW_BT_CNF_ID * headset_cnf_id)
{
  T_MFW_BT_CNF_ID cnf_id = 0;

  TRACE_FUNCTION ("bt_is_pref_headset()");

  if(service EQ MFW_BT_HEADSET)
  {
      switch(btibtp_query_default_headset(bd_addr,&cnf_id))
      /* add headset in default headset list */
      {
        case BTP_OK:
          *headset_cnf_id = cnf_id;
          return MFW_BT_OK;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_HSG_ALREADY_STOR:
          return MFW_BT_HSG_ALREADY_STORED;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
  }
  else
    return MFW_BT_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)          MODULE  : MFW_BT           |
| STATE   : code                          ROUTINE :bt_reset_pref_hsg |
+--------------------------------------------------------------------+

  PURPOSE : delete current entry of a preferred headset
*/

T_MFW_BT_RESULT_BT  bt_reset_pref_hsg(T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_DEV_ID dev_id)
{
  TRACE_FUNCTION ("bt_reset_pref_hsg()");
  if(service EQ MFW_BT_HEADSET)
  {
    /* parameter: dev_id equal headset_id in default headset list */
    switch(btibtp_delete_default_headset(dev_id))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)       MODULE  : MFW_BT              |
| STATE   : code                       ROUTINE :bt_disconnect_service|
+--------------------------------------------------------------------+

  PURPOSE : request disconnect of service

*/
T_MFW_BT_RESULT_BT bt_disconnect_service (T_MFW_BT_SERVICE_TYPE service)
{
  TRACE_FUNCTION ("bt_disconnect_service()");

  if((service EQ MFW_BT_HEADSET) OR
     (service EQ MFW_BT_DIAL_UP) OR
     (service EQ MFW_BT_FAX_GW))
  {
    switch(btibtp_disconnect_service_req((T_BTI_DEVICE_TYPE)service))
    { /* BTP_DISCONNECT_DEVICE_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)                MODULE  : MFW_BT     |
| STATE   : code                                ROUTINE :bt_send_pin |
+--------------------------------------------------------------------+

  PURPOSE : send the pin requesting by bt (for authorization)

*/
T_MFW_BT_RESULT_BT bt_send_pin (T_MFW_BT_BD_ADDR bd_addr[],
                             T_MFW_BT_PIN pin_code[],
                             T_MFW_BT_PIN_MODE pin_mode)
{
  UBYTE user_pin_code_length=4;

  TRACE_FUNCTION ("bt_send_pin()");

  if(pin_mode EQ PIN_PROVIDED)
  {     /* BTP_PIN_RES to BT */
    user_pin_code_length = strlen((const char *)pin_code);

    switch(btibtp_pin_res(user_pin_code_length, pin_code, bd_addr))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
  {/* if device isn't able to provide a PIN code */
    /* BTP_PIN_RES to BT */
    switch(btibtp_pin_res(0,0,bd_addr))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE :bt_authorization_reply |
+--------------------------------------------------------------------+

  PURPOSE : reply the authorization requesting by bt

*/
T_MFW_BT_RESULT_BT bt_authorization_reply (T_MFW_BT_BD_ADDR bd_addr[],
                                           T_MFW_BT_AUTHORIZATION_MASK mfw_auth_mask,
                                           T_MFW_BT_AUTHORIZATION_MODE auth_mode)
{
  T_BTI_AUTHORIZATION_MASK author_mask = 0;
  T_BTI_APPLI_NAME service_name[BTI_APPLI_NAME_LENGTH];
  U8 len;

  TRACE_FUNCTION ("bt_authorization_reply()");

  memset(service_name,0,BTI_APPLI_NAME_LENGTH);
  /*
  *** MFW_BT_AUTHORIZED_ONCE : remote device is granted access to the service once time
      MFW_BT_AUTHORIZED_ALWAYS:remote device is granted access to the service always
      MFW_BT_UNAUTHORIZED: local device does not give access to the service
  ***
  */
  if((auth_mode EQ MFW_BT_AUTHORIZED_ONCE) OR
     (auth_mode EQ MFW_BT_AUTHORIZED_ALWAYS) OR
     (auth_mode EQ MFW_BT_UNAUTHORIZED))
  {
    switch(mfw_auth_mask)
    {/* mask refer to the service (mask about BT_AUTHORIZATION_IND in MMI) */
      case MFW_SERVICE_HSG:
        memcpy(service_name,MFW_BT_SERV_HSG,sizeof(MFW_BT_SERV_HSG));
        len = sizeof(MFW_BT_SERV_HSG);
        break;
      case MFW_SERVICE_DUN:
        memcpy(service_name,MFW_BT_SERV_DUN,sizeof(MFW_BT_SERV_DUN));
        len = sizeof(MFW_BT_SERV_DUN);
        break;
      case MFW_SERVICE_FAX:
        memcpy(service_name,MFW_BT_SERV_FAX,sizeof(MFW_BT_SERV_FAX));
        len = sizeof(MFW_BT_SERV_FAX);
        break;
      case MFW_SERVICE_OPP:
        memcpy(service_name,MFW_BT_SERV_OPP,sizeof(MFW_BT_SERV_OPP));
        len = sizeof(MFW_BT_SERV_OPP);
        break;
      case MFW_SERVICE_SYNC_C:
        memcpy(service_name,MFW_BT_SERV_SYNC_C,sizeof(MFW_BT_SERV_SYNC_C));
        len = sizeof(MFW_BT_SERV_SYNC_C);
        break;
      default:
        return MFW_BT_NOK;
    }
    /* because authorization mask is SCM-depend MFW requests specify mask */
    if(btibtp_get_authorization_mask(service_name,len,&author_mask) NEQ BTP_OK)
    {
      TRACE_EVENT ("Error:bt_authorization_reply()");
      return MFW_BT_NOK;
    }
    else
    {
      switch(btibtp_authorization_reply(bd_addr,author_mask,auth_mode))
      {
        case BTP_OK:
          return MFW_BT_OK;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445) MODULE  : MFW_BT                    |
| STATE   : code                 ROUTINE :bt_get_authorization_device|
+--------------------------------------------------------------------+

  PURPOSE:get authorizations to access available services on local device
*/
T_MFW_BT_RESULT_BT bt_get_authorization_device (T_MFW_BT_BD_ADDR bd_addr[],
                                            T_MFW_BT_AUTHORIZATION_MASK * mask)
{
  T_BTI_AUTHORIZATION_MASK author_mask,mask_hsg,mask_dun,mask_fax,mask_sync_cmd,mask_opp,auth_mask;

  T_BTI_APPLI_NAME service_name[BTI_APPLI_NAME_LENGTH];

  TRACE_FUNCTION ("bt_get_authorization_device()");

  memset(service_name,0,BTI_APPLI_NAME_LENGTH);
  author_mask = mask_hsg = mask_dun = mask_fax = mask_sync_cmd = mask_opp = auth_mask =0;

  switch(btibtp_get_authorization(bd_addr,&auth_mask))
  {/* auth_mask contains bits for several services, bits are SCM dependent */
    case BTP_OK:
      break;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }

  memcpy(service_name,MFW_BT_SERV_DUN,sizeof(MFW_BT_SERV_DUN));
  switch(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_DUN),&author_mask))
  {/* get mask for service dial-up */
    case BTP_OK:
      mask_dun =  author_mask;
      break;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
  memcpy(service_name,MFW_BT_SERV_HSG,sizeof(MFW_BT_SERV_HSG));
  switch(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_HSG),&author_mask))
  {/* get mask for service headset gateway */
    case BTP_OK:
      mask_hsg =  author_mask;
      break;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
  memcpy(service_name,MFW_BT_SERV_FAX,sizeof(MFW_BT_SERV_FAX));
  switch(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_FAX),&author_mask))
  {/* get mask for service fax gateway */
    case BTP_OK:
      mask_fax =  author_mask;
      break;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }

  memcpy(service_name,MFW_BT_SERV_OPP,sizeof(MFW_BT_SERV_OPP));
  switch(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_OPP),&author_mask))
  {/* get mask for service opp object push profile */
    case BTP_OK:
      mask_opp =  author_mask;
      break;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }

  memcpy(service_name,MFW_BT_SERV_SYNC_C,sizeof(MFW_BT_SERV_SYNC_C));
  switch(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_SYNC_C),&author_mask))
  {/* get mask for service SYNC server with SYNC command support */
    case BTP_OK:
      mask_sync_cmd =  author_mask;
      break;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
  *mask = 0;
  /* assemble mask for MMI */
  if((auth_mask & mask_hsg) EQ mask_hsg)
  {
    *mask = *mask | MFW_SERVICE_HSG;
  }
  if((auth_mask & mask_dun) EQ mask_dun)
  {
    *mask = *mask | MFW_SERVICE_DUN;
  }
  if((auth_mask & mask_fax) EQ mask_fax)
  {
    *mask = *mask | MFW_SERVICE_FAX;
  }
  if((auth_mask & mask_opp) EQ mask_opp)
  {
    *mask = *mask | MFW_SERVICE_OPP;
  }
  if((auth_mask & mask_sync_cmd) EQ mask_sync_cmd)
  {
    *mask = *mask | MFW_SERVICE_SYNC_C;
  }
  return MFW_BT_OK;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445) MODULE  : MFW_BT                    |
| STATE   : code                 ROUTINE :bt_set_authorization_device|
+--------------------------------------------------------------------+

  PURPOSE:set authorization mask to access available service on local device
*/
T_MFW_BT_RESULT_BT bt_set_authorization_device (T_MFW_BT_BD_ADDR bd_addr[],
                                            T_MFW_BT_AUTHORIZATION_MASK mask)
{
  T_BTI_AUTHORIZATION_MASK author_mask = 0;
  T_BTI_APPLI_NAME service_name[BTI_APPLI_NAME_LENGTH];
  U8 len;

  TRACE_FUNCTION ("bt_set_authorization_device()");

  memset(service_name,0,BTI_APPLI_NAME_LENGTH);

  switch(mask)
  {
    case MFW_SERVICE_DUN:
      memcpy(service_name,MFW_BT_SERV_DUN,sizeof(MFW_BT_SERV_DUN));
      len = sizeof(MFW_BT_SERV_DUN);
      break;
    case MFW_SERVICE_HSG:
      memcpy(service_name,MFW_BT_SERV_HSG,sizeof(MFW_BT_SERV_HSG));
      len = sizeof(MFW_BT_SERV_HSG);
      break;
    case MFW_SERVICE_FAX:
      memcpy(service_name,MFW_BT_SERV_FAX,sizeof(MFW_BT_SERV_FAX));
      len = sizeof(MFW_BT_SERV_FAX);
      break;
    case MFW_SERVICE_OPP:
      memcpy(service_name,MFW_BT_SERV_OPP,sizeof(MFW_BT_SERV_OPP));
      len = sizeof(MFW_BT_SERV_OPP);
      break;
    case MFW_SERVICE_SYNC_C:
      memcpy(service_name,MFW_BT_SERV_SYNC_C,sizeof(MFW_BT_SERV_SYNC_C));
      len = sizeof(MFW_BT_SERV_SYNC_C);
      break;
    default:
      return MFW_BT_NOK;
  }

  if(btibtp_get_authorization_mask(service_name,len,&author_mask) NEQ BTP_OK)
  {/* get SCM dependent mask for specified service */
    TRACE_EVENT ("Error:bt_list_auth_dev()");
    return MFW_BT_RET_FAIL;
  }

  switch(btibtp_set_authorization(bd_addr,author_mask))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+---------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445) MODULE  : MFW_BT                     |
| STATE   : code                 ROUTINE :bt_del_authorizations_device|
+---------------------------------------------------------------------+

  PURPOSE:delete all authorizations to access available services on local device
*/
T_MFW_BT_RESULT_BT bt_del_authorizations_device (T_MFW_BT_BD_ADDR bd_addr[])
{
  TRACE_FUNCTION ("bt_del_authorizations_device()");

  switch(btibtp_set_authorization(bd_addr,0))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)       MODULE  : MFW_BT              |
| STATE   : code                       ROUTINE :bt_del_authorization |
+--------------------------------------------------------------------+

  PURPOSE:delete an authorization to access available service on local device
*/
T_MFW_BT_RESULT_BT bt_del_authorization (T_MFW_BT_BD_ADDR bd_addr[],T_MFW_BT_AUTHORIZATION_MASK service_mask)
{
  T_BTI_AUTHORIZATION_MASK auth_mask,author_mask = 0;
  T_BTI_APPLI_NAME service_name[BTI_APPLI_NAME_LENGTH];
  U8 len;

  TRACE_EVENT ("bt_del_authorization()");

  memset(service_name,0,BTI_APPLI_NAME_LENGTH);

  switch(btibtp_get_authorization(bd_addr,&auth_mask))
  {/* give authorization mask with all authorizations of remote device */
    case BTP_OK:
      break;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }

  switch(service_mask)
  {
    case MFW_SERVICE_DUN:
      memcpy(service_name,MFW_BT_SERV_DUN,sizeof(MFW_BT_SERV_DUN));
      len = sizeof(MFW_BT_SERV_DUN);
      break;
    case MFW_SERVICE_HSG:
      memcpy(service_name,MFW_BT_SERV_HSG,sizeof(MFW_BT_SERV_HSG));
      len = sizeof(MFW_BT_SERV_HSG);
      break;
    case MFW_SERVICE_FAX:
      memcpy(service_name,MFW_BT_SERV_FAX,sizeof(MFW_BT_SERV_FAX));
      len = sizeof(MFW_BT_SERV_FAX);
      break;
    case MFW_SERVICE_OPP:
      memcpy(service_name,MFW_BT_SERV_OPP,sizeof(MFW_BT_SERV_OPP));
      len = sizeof(MFW_BT_SERV_OPP);
      break;
    case MFW_SERVICE_SYNC_C:
      memcpy(service_name,MFW_BT_SERV_SYNC_C,sizeof(MFW_BT_SERV_SYNC_C));
      len = sizeof(MFW_BT_SERV_SYNC_C);
      break;
    default:
      return MFW_BT_NOK;
  }

  if(btibtp_get_authorization_mask(service_name,len,&author_mask) NEQ BTP_OK)
  {/* give SCM dependent authorization mask for specified service */
    TRACE_EVENT ("Error:bt_list_auth_dev()");
    return MFW_BT_RET_FAIL;
  }
  else
  {
    if((auth_mask & author_mask) EQ author_mask)
    {
      auth_mask= auth_mask & ~author_mask;/* delete the specified authorization in mask*/
    }
    else
      return MFW_BT_INVALID_PARA;
  }

  switch(btibtp_set_authorization(bd_addr,auth_mask))
  {/* set authorization mask for remote device again */
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)           MODULE  : MFW_BT          |
| STATE   : code                           ROUTINE :bt_start_pairing |
+--------------------------------------------------------------------+

  PURPOSE : start pairing procedure

*/
T_MFW_BT_RESULT_BT bt_start_pairing (T_MFW_BT_BD_ADDR bd_addr[],
                                  T_MFW_BT_PIN pin_code[],
                                  T_MFW_BT_PIN_MODE pin_mode)
{
  UBYTE user_pin_code_length=4;

  TRACE_FUNCTION ("bt_start_pairing()");

  if(pin_mode EQ PIN_PROVIDED)
  {/* pairing provides to a link key between the gateway and the remote device */

    user_pin_code_length = strlen((const char *)pin_code);

    switch(btibtp_start_pairing(user_pin_code_length, pin_code, bd_addr))
    {
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;/* wrong pin */
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
  {/* if device isn't able to provide a PIN code */
    switch(btibtp_start_pairing(0,0,bd_addr))
    {
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)    MODULE  : MFW_BT                 |
| STATE   : code                    ROUTINE :bt_delete_paired_device |
+--------------------------------------------------------------------+

  PURPOSE : delete paired device from the paired devices database

*/
T_MFW_BT_RESULT_BT bt_delete_paired_device (T_MFW_BT_BD_ADDR bd_addr[])
{
  TRACE_FUNCTION ("bt_delete_paired_device()");

  switch(btibtp_delete_paired_device(bd_addr))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)              MODULE  : MFW_BT       |
| STATE   : code                              ROUTINE :bt_check_pair |
+--------------------------------------------------------------------+

  PURPOSE : check whether the device in the data base exists

*/
T_MFW_BT_RESULT_BT bt_check_pair (T_MFW_BT_BD_ADDR bd_addr[])
{
  TRACE_FUNCTION ("bt_check_pair()");

  switch(btibtp_check_pairing_state(bd_addr))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}
#ifdef _SIMULATION_
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)       MODULE  : MFW_BT              |
| STATE   : code                       ROUTINE :bt_transfer_audio_in |
+--------------------------------------------------------------------+

  PURPOSE : request transfer audio from headset to phone(headset gateway)

*/
T_MFW_BT_RESULT_BT bt_transfer_audio_in (T_MFW_BT_SERVICE_TYPE service)
{
  TRACE_FUNCTION ("bt_transfer_audio_in()");

  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_transfer_audio_in_req((T_BTI_DEVICE_TYPE)service))
    {  /* BTP_TRANSFER_AUDIO_IN_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)       MODULE  : MFW_BT              |
| STATE   : code                       ROUTINE :bt_transfer_audio_out|
+--------------------------------------------------------------------+

  PURPOSE : request transfer audio from headset gateway(phone)to the
            remote headset
*/

T_MFW_BT_RESULT_BT bt_transfer_audio_out(T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_REQ_ID req_id,
                                      T_MFW_BT_BD_ADDR bd_addr[])
{
  TRACE_FUNCTION ("bt_transfer_audio_out()");

  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_transfer_audio_out_req((T_BTI_DEVICE_TYPE)service,req_id,bd_addr))
    { /* BTP_TRANSFER_AUDIO_OUT_REQ to BT */
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  return MFW_BT_FAIL;
}
#endif

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)       MODULE  : MFW_BT              |
| STATE   : code                       ROUTINE :bt_list_paired_dev   |
+--------------------------------------------------------------------+

  PURPOSE : list all paired devices

*/
T_MFW_BT_RETURN  bt_list_paired_dev(T_MFW_BT_DEV_PAIR_LIST * pair_list)
{
  T_BT_DEV_PAIR_LIST paired_dev_list;
  U8 i;

  TRACE_FUNCTION ("bt_list_paired_dev()");

  memset(&paired_dev_list,0,sizeof(paired_dev_list));
   /* get list of devices of paired database */
  if(btibtp_list_paired_dev(&paired_dev_list) NEQ BTP_OK)
  {
    TRACE_EVENT ("Error:bt_list_paired_dev()");
    return MFW_BT_RET_FAIL;
  }
  else
  {
    pair_list->count = paired_dev_list.pair_counter;/* number of devices */
    for(i = 0;i<paired_dev_list.pair_counter; i++)
      memcpy(pair_list->paired_bd_addr[i],paired_dev_list.paired_addr[i],MFW_BT_ADDR_MAX_LEN);

    return MFW_BT_RET_OK;
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)         MODULE  : MFW_BT            |
| STATE   : code                         ROUTINE :bt_list_auth_dev   |
+--------------------------------------------------------------------+

  PURPOSE : list all devices of authorization database with their
            masks of available services on the local device

*/
T_MFW_BT_RETURN  bt_list_auth_dev(T_MFW_BT_AUTHORIZATION_LIST * authorization_list)
{
  T_BT_DEV_AUTHORIZATION_LIST auth_dev_list;
  U8 i;
  T_BTI_AUTHORIZATION_MASK mask_hsg,mask_dun,mask_fax,author_mask = 0;
  T_BTI_AUTHORIZATION_MASK mask_opp,mask_sync_cmd = 0;
  T_BTI_APPLI_NAME service_name[BTI_APPLI_NAME_LENGTH];

  TRACE_FUNCTION ("bt_list_auth_dev()");

  memset(&auth_dev_list,0,sizeof(auth_dev_list));
  memset(service_name,0,BTI_APPLI_NAME_LENGTH);
   /* get list of devices of authorization database */
  if(btibtp_list_auth_dev(&auth_dev_list) NEQ BTP_OK)
  {
    TRACE_EVENT ("Error:bt_list_auth_dev()");
    return MFW_BT_RET_FAIL;
  }
  else
  {
    memcpy(service_name,MFW_BT_SERV_DUN,sizeof(MFW_BT_SERV_DUN));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_DUN),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service DUN */
      TRACE_EVENT ("Error:bt_list_auth_dev()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_dun = author_mask;
    }
    memcpy(service_name,MFW_BT_SERV_HSG,sizeof(MFW_BT_SERV_HSG));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_HSG),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service HSG */
      TRACE_EVENT ("Error:bt_list_auth_dev()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_hsg = author_mask;
    }
    memcpy(service_name,MFW_BT_SERV_FAX,sizeof(MFW_BT_SERV_FAX));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_FAX),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service FAX */
      TRACE_EVENT ("Error:bt_list_auth_dev()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_fax = author_mask;
    }
    memcpy(service_name,MFW_BT_SERV_OPP,sizeof(MFW_BT_SERV_OPP));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_OPP),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service OPP */
      TRACE_EVENT ("Error:bt_list_auth_dev()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_opp = author_mask;
    }
    memcpy(service_name,MFW_BT_SERV_SYNC_C,sizeof(MFW_BT_SERV_SYNC_C));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_SYNC_C),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service SYNC server with SYNC command support */
      TRACE_EVENT ("Error:bt_list_auth_dev()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_sync_cmd = author_mask;
    }

    authorization_list->count = auth_dev_list.auth_counter;
    for(i = 0;i<auth_dev_list.auth_counter; i++)
    {
      memcpy(authorization_list->auth_devices[i].remote_bd_addr,auth_dev_list.auth_el[i].remote_addr,MFW_BT_ADDR_MAX_LEN);
      /* assemble mask for each device */
      if((auth_dev_list.auth_el[i].mask_auth & mask_hsg) EQ mask_hsg)
      {
        authorization_list->auth_devices[i].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_HSG;
      }

      if((auth_dev_list.auth_el[i].mask_auth & mask_dun) EQ mask_dun)
      {
        authorization_list->auth_devices[i].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_DUN;
      }

      if((auth_dev_list.auth_el[i].mask_auth & mask_fax) EQ mask_fax)
      {
        authorization_list->auth_devices[i].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_FAX;
      }

      if((auth_dev_list.auth_el[i].mask_auth & mask_opp) EQ mask_opp)
      {
        authorization_list->auth_devices[i].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_OPP;
      }

      if((auth_dev_list.auth_el[i].mask_auth & mask_sync_cmd) EQ mask_sync_cmd)
      {
        authorization_list->auth_devices[i].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_SYNC_C;
      }
    }
    return MFW_BT_RET_OK;
  }
}

/*
+-------------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)         MODULE  : MFW_BT                 |
| STATE   : code                         ROUTINE :bt_authorized_devices   |
+-------------------------------------------------------------------------+

  PURPOSE : list all devices of authorization database with desired
            authorization (service_mask)
*/
T_MFW_BT_RETURN  bt_authorized_devices(T_MFW_BT_AUTHORIZATION_LIST * authorization_list,T_MFW_BT_AUTHORIZATION_MASK service_mask )
{
  T_BT_DEV_AUTHORIZATION_LIST auth_dev_list;
  U8 i,j;
  T_BTI_AUTHORIZATION_MASK author_mask,mask_hsg,mask_dun,mask_fax,mask_opp = 0;
  T_BTI_AUTHORIZATION_MASK mask_sync_cmd = 0;
  T_BTI_APPLI_NAME service_name[BTI_APPLI_NAME_LENGTH];

  TRACE_EVENT ("bt_authorized_devices()");

  memset(&auth_dev_list,0,sizeof(auth_dev_list));
  memset(service_name,0,BTI_APPLI_NAME_LENGTH);
   /* get list of devices of authorization database */
  if(btibtp_list_auth_dev(&auth_dev_list) NEQ BTP_OK)
  {
    TRACE_EVENT ("Error:bt_authorized_devices()");
    return MFW_BT_RET_FAIL;
  }
  else
  {
    memcpy(service_name,MFW_BT_SERV_DUN,sizeof(MFW_BT_SERV_DUN));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_DUN),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service DUN */
      TRACE_EVENT ("Error:bt_authorized_devices()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_dun = author_mask;
    }
    memcpy(service_name,MFW_BT_SERV_HSG,sizeof(MFW_BT_SERV_HSG));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_HSG),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service HSG */
      TRACE_EVENT ("Error:bt_authorized_devices()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_hsg = author_mask;
    }
    memcpy(service_name,MFW_BT_SERV_FAX,sizeof(MFW_BT_SERV_FAX));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_FAX),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service FAX */
      TRACE_EVENT ("Error:bt_authorized_devices()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_fax = author_mask;
    }
    memcpy(service_name,MFW_BT_SERV_OPP,sizeof(MFW_BT_SERV_OPP));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_OPP),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service OPP */
      TRACE_EVENT ("Error:bt_authorized_devices()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_opp = author_mask;
    }
    memcpy(service_name,MFW_BT_SERV_SYNC_C,sizeof(MFW_BT_SERV_SYNC_C));
    if(btibtp_get_authorization_mask(service_name,sizeof(MFW_BT_SERV_SYNC_C),&author_mask) NEQ BTP_OK)
    {/* get SCM-dependent authorization mask of service SYNC server with SYN command support */
      TRACE_EVENT ("Error:bt_authorized_devices()");
      return MFW_BT_RET_FAIL;
    }
    else
    {
      mask_sync_cmd = author_mask;
    }

    for(i=0,j=0;i<auth_dev_list.auth_counter; i++)
    {
      switch(service_mask)
      {
        case MFW_SERVICE_HSG:
          if((auth_dev_list.auth_el[i].mask_auth & mask_hsg) EQ mask_hsg)
          {
            authorization_list->auth_devices[j].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_HSG;
            memcpy(authorization_list->auth_devices[j].remote_bd_addr,auth_dev_list.auth_el[i].remote_addr,MFW_BT_ADDR_MAX_LEN);
            authorization_list->count++;
            j++;
          }
          break;
        case MFW_SERVICE_DUN:
          if((auth_dev_list.auth_el[i].mask_auth & mask_dun) EQ mask_dun)
          {
            authorization_list->auth_devices[j].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_DUN;
            memcpy(authorization_list->auth_devices[j].remote_bd_addr,auth_dev_list.auth_el[i].remote_addr,MFW_BT_ADDR_MAX_LEN);
            authorization_list->count++;
            j++;
          }
          break;
        case MFW_SERVICE_FAX:
          if((auth_dev_list.auth_el[i].mask_auth & mask_fax) EQ mask_fax)
          {
            authorization_list->auth_devices[j].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_FAX;
            memcpy(authorization_list->auth_devices[j].remote_bd_addr,auth_dev_list.auth_el[i].remote_addr,MFW_BT_ADDR_MAX_LEN);
            authorization_list->count++;
            j++;
          }
          break;
        case MFW_SERVICE_OPP:
          if((auth_dev_list.auth_el[i].mask_auth & mask_opp) EQ mask_opp)
          {
            authorization_list->auth_devices[j].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_OPP;
            memcpy(authorization_list->auth_devices[j].remote_bd_addr,auth_dev_list.auth_el[i].remote_addr,MFW_BT_ADDR_MAX_LEN);
            authorization_list->count++;
            j++;
          }
          break;
        case MFW_SERVICE_SYNC_C:
          if((auth_dev_list.auth_el[i].mask_auth & mask_sync_cmd) EQ mask_sync_cmd)
          {
            authorization_list->auth_devices[j].mask_authorization = authorization_list->auth_devices[i].mask_authorization | MFW_SERVICE_SYNC_C;
            memcpy(authorization_list->auth_devices[j].remote_bd_addr,auth_dev_list.auth_el[i].remote_addr,MFW_BT_ADDR_MAX_LEN);
            authorization_list->count++;
            j++;
          }
          break;
        default:
          return MFW_BT_RET_FAIL;
      }
    }
    return MFW_BT_RET_OK;
  }
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE :bt_clear_list_found_dev|
+--------------------------------------------------------------------+

  PURPOSE : clear list of found devices with theirs service id's

*/
T_MFW_BT_RETURN bt_clear_list_found_dev (void)
{
  T_MFW_BT_DEVICE_ID *help_ptr;
  T_MFW_BT_DEVICE_ID *help_h_ptr;

  TRACE_FUNCTION ("bt_clear_list_found_dev()");

  if(service_list.device_id) /* check:is list of found hs empty ?*/
  {
    help_ptr = service_list.device_id->next;
    MFREE(service_list.device_id);   /* free memory first element */
    service_list.device_id = help_ptr;
    while(help_ptr)
    {
      help_h_ptr = help_ptr->next;
      MFREE(help_ptr);               /* free next elements */
      help_ptr = help_h_ptr;
      service_list.device_id = help_ptr;
    }
  }
  return MFW_BT_RET_OK;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)MODULE  : MFW_BT                     |
| STATE   : code               ROUTINE :bt_clear_all_lists_found_serv|
+--------------------------------------------------------------------+

  PURPOSE : clear all list of found  services

*/
T_MFW_BT_RETURN bt_clear_all_lists_found_serv (void)
{
  T_MFW_BT_SERVICE_ID *help_ptr;
  T_MFW_BT_SERVICE_ID *help_h_ptr;

  TRACE_FUNCTION ("bt_clear_all_lists_found_serv()");

  if(found_headset.device_id)  /* check: is list of found hs empty ? */
  {
    help_ptr = found_headset.device_id->next;
    MFREE(found_headset.device_id->bd_name);/* free name of first element */
    MFREE(found_headset.device_id);         /* free memory first element */
    found_headset.device_id = help_ptr;
    while(help_ptr)
    {
      help_h_ptr = help_ptr->next;
      MFREE(help_ptr->bd_name);          /* free name of next elements */
      MFREE(help_ptr);                   /* free next elements */
      help_ptr = help_h_ptr;
      found_headset.device_id = help_ptr;
    }
  }
  if(found_dial_up.device_id)/* check: is list of found dun empty ? */
  {
    help_ptr = found_dial_up.device_id->next;
    MFREE(found_dial_up.device_id->bd_name);/* free name of first element */
    MFREE(found_dial_up.device_id);       /* free memory first element */
    found_dial_up.device_id = help_ptr;
    while(help_ptr)
    {
      help_h_ptr = help_ptr->next;
      MFREE(help_ptr->bd_name);          /* free name of next elements */
      MFREE(help_ptr);                   /* free next elements */
      help_ptr = help_h_ptr;
      found_dial_up.device_id = help_ptr;
    }
  }
  if(found_fax.device_id)/* check: is list of found fax empty ? */
  {
    help_ptr = found_fax.device_id->next;
    MFREE(found_fax.device_id->bd_name);/* free name of first element */
    MFREE(found_fax.device_id);       /* free memory first element */
    found_fax.device_id = help_ptr;
    while(help_ptr)
    {
      help_h_ptr = help_ptr->next;
      MFREE(help_ptr->bd_name);          /* free name of next elements */
      MFREE(help_ptr);                   /* free next elements */
      help_ptr = help_h_ptr;
      found_fax.device_id = help_ptr;
    }
  }
  if(found_opp.device_id)/* check: is list of found OPP  empty ? */
  {
    help_ptr = found_opp.device_id->next;
    MFREE(found_opp.device_id->bd_name);/* free name of first element */
    MFREE(found_opp.device_id);       /* free memory first element */
    found_opp.device_id = help_ptr;
    while(help_ptr)
    {
      help_h_ptr = help_ptr->next;
      MFREE(help_ptr->bd_name);          /* free name of next elements */
      MFREE(help_ptr);                   /* free next elements */
      help_ptr = help_h_ptr;
      found_opp.device_id = help_ptr;
    }
  }
  if(found_sync.device_id)/* check: is list of found SYNC server empty ? */
  {
    help_ptr = found_sync.device_id->next;
    MFREE(found_sync.device_id->bd_name);/* free name of first element */
    MFREE(found_sync.device_id);       /* free memory first element */
    found_sync.device_id = help_ptr;
    while(help_ptr)
    {
      help_h_ptr = help_ptr->next;
      MFREE(help_ptr->bd_name);          /* free name of next elements */
      MFREE(help_ptr);                   /* free next elements */
      help_ptr = help_h_ptr;
      found_sync.device_id = help_ptr;
    }
  }
  if(found_sync_cmd.device_id)/* check: is list of found SYNC_CMD server  empty ? */
  {
    help_ptr = found_sync_cmd.device_id->next;
    MFREE(found_sync_cmd.device_id->bd_name);/* free name of first element */
    MFREE(found_sync_cmd.device_id);       /* free memory first element */
    found_sync_cmd.device_id = help_ptr;
    while(help_ptr)
    {
      help_h_ptr = help_ptr->next;
      MFREE(help_ptr->bd_name);          /* free name of next elements */
      MFREE(help_ptr);                   /* free next elements */
      help_ptr = help_h_ptr;
      found_sync_cmd.device_id = help_ptr;
    }
  }
  return MFW_BT_RET_OK;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)    MODULE  : MFW_BT                 |
| STATE   : code                    ROUTINE :bt_clear_list_found_serv|
+--------------------------------------------------------------------+

  PURPOSE : clear list of specific found service

*/
T_MFW_BT_RETURN bt_clear_list_found_serv (T_MFW_BT_SERVICE_TYPE service)
{
  T_MFW_BT_SERVICE_ID *help_ptr;
  T_MFW_BT_SERVICE_ID *help_h_ptr;

  TRACE_FUNCTION ("bt_clear_list_found_serv()");

  if(service EQ MFW_BT_HEADSET)
  {
    if(found_headset.device_id)  /* check: is list of found hs  empty ? */
    {
      help_ptr = found_headset.device_id->next;
      MFREE(found_headset.device_id->bd_name);/* free name of first element */
      MFREE(found_headset.device_id);   /* free memory first element */
      found_headset.device_id = help_ptr;
      while(help_ptr)
      {
        help_h_ptr = help_ptr->next;
        MFREE(help_ptr->bd_name);         /* free name of next elements */
        MFREE(help_ptr);               /* free next elements */
        help_ptr = help_h_ptr;
        found_headset.device_id = help_ptr;
      }
    }
    return MFW_BT_RET_OK;
  }
  else if(service EQ MFW_BT_DIAL_UP)
  {/* check: is list of found dun   empty ? */
    if(found_dial_up.device_id)
    {
      help_ptr = found_dial_up.device_id->next;
      MFREE(found_dial_up.device_id->bd_name);/* free name of first element */
      MFREE(found_dial_up.device_id);   /* free memory first element */
      found_dial_up.device_id = help_ptr;
      while(help_ptr)
      {
        help_h_ptr = help_ptr->next;
        MFREE(help_ptr->bd_name);      /* free name of next elements */
        MFREE(help_ptr);               /* free next elements */
        help_ptr = help_h_ptr;
        found_dial_up.device_id = help_ptr;
      }
    }
    return MFW_BT_RET_OK;
  }
  else if(service EQ MFW_BT_FAX_GW)
  {/* check: is list of found fax empty ? */
    if(found_fax.device_id)
    {
      help_ptr = found_fax.device_id->next;
      MFREE(found_fax.device_id->bd_name);/* free name of first element */
      MFREE(found_fax.device_id);   /* free memory first element */
      found_fax.device_id = help_ptr;
      while(help_ptr)
      {
        help_h_ptr = help_ptr->next;
        MFREE(help_ptr->bd_name);      /* free name of next elements */
        MFREE(help_ptr);               /* free next elements */
        help_ptr = help_h_ptr;
        found_fax.device_id = help_ptr;
      }
    }
    return MFW_BT_RET_OK;
  }
  else if(service EQ MFW_BT_OPP)
  {/* check: is list of found OPP  empty ? */
    if(found_opp.device_id)
    {
      help_ptr = found_opp.device_id->next;
      MFREE(found_opp.device_id->bd_name);/* free name of first element */
      MFREE(found_opp.device_id);   /* free memory first element */
      found_opp.device_id = help_ptr;
      while(help_ptr)
      {
        help_h_ptr = help_ptr->next;
        MFREE(help_ptr->bd_name);      /* free name of next elements */
        MFREE(help_ptr);               /* free next elements */
        help_ptr = help_h_ptr;
        found_opp.device_id = help_ptr;
      }
    }
    return MFW_BT_RET_OK;
  }
  else if(service EQ MFW_BT_SYNC)
  {/* check: is list of found SYNC server  empty ? */
    if(found_sync.device_id)
    {
      help_ptr = found_sync.device_id->next;
      MFREE(found_sync.device_id->bd_name);/* free name of first element */
      MFREE(found_sync.device_id);   /* free memory first element */
      found_sync.device_id = help_ptr;
      while(help_ptr)
      {
        help_h_ptr = help_ptr->next;
        MFREE(help_ptr->bd_name);      /* free name of next elements */
        MFREE(help_ptr);               /* free next elements */
        help_ptr = help_h_ptr;
        found_sync.device_id = help_ptr;
      }
    }
    return MFW_BT_RET_OK;
  }
  else if(service EQ MFW_BT_SYNC_CMD)
  {/* check:is list of found SYNC server with support SYN commands  empty ? */
    if(found_sync_cmd.device_id)
    {
      help_ptr = found_sync_cmd.device_id->next;
      MFREE(found_sync_cmd.device_id->bd_name);/* free name of first element */
      MFREE(found_sync_cmd.device_id);   /* free memory first element */
      found_sync_cmd.device_id = help_ptr;
      while(help_ptr)
      {
        help_h_ptr = help_ptr->next;
        MFREE(help_ptr->bd_name);      /* free name of next elements */
        MFREE(help_ptr);               /* free next elements */
        help_ptr = help_h_ptr;
        found_sync_cmd.device_id = help_ptr;
      }
    }
    return MFW_BT_RET_OK;
  }
  else
    return MFW_BT_RET_FAIL;
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE :bt_set_param_headset|
+--------------------------------------------------------------------+

  PURPOSE:remote control of speaker or microphone gain (profile headset)
*/

T_MFW_BT_RESULT_BT bt_set_param_headset (T_MFW_BT_SERVICE_TYPE service,
                                         T_MFW_BT_PARAM_NUMBER nb,
                                         T_MFW_BT_CHANGE_PARA type,
                                         U16 new_value)
{
  TRACE_FUNCTION ("bt_set_param_headset()");
  if(service EQ MFW_BT_HEADSET)
  {
    /* update parameter speaker or microphone */
    if((nb EQ MFW_BT_MIC_GAIN) OR
       (nb EQ MFW_BT_SPEAKER_GAIN))
    {/* increase or decrease (+1/-1) or set new value */
      if((type EQ MFW_BT_PARA_INC) OR
         (type EQ MFW_BT_PARA_DEC) OR
         (type EQ MFW_BT_PARA_SET))
      {
        if(type EQ MFW_BT_PARA_SET)
        {/* for set value valid value is needed */
          if(!new_value)
          {
            return MFW_BT_NOK;
          }
        }

        switch(btibtp_set_param_headset(nb,type,new_value))
        {
          case BTP_OK:
            return MFW_BT_OK;
          case BTP_INVALID_PARAMETER:
            return MFW_BT_INVALID_PARA;
          case BTP_NOT_SUPP:
            return MFW_BT_NO_SUPP;
          case BTP_NOT_READY:
            return MFW_BT_NOT_READY;
          case BTP_INT_ERR:
            return MFW_BT_INT_ERR;
          case BTP_MEMORY_ERR:
            return MFW_BT_MEM_ERR;
          case BTP_NOK:
            return MFW_BT_NOK;
        }
      }
      else
        return MFW_BT_NOK;
    }
    else
       return MFW_BT_FAIL;
  }
  else
     return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445) MODULE  : MFW_BT                    |
| STATE   : code                 ROUTINE :bt_get_default_service_info|
+--------------------------------------------------------------------+

  PURPOSE:get device info from default headset list
*/
T_MFW_BT_RESULT_BT bt_get_default_service_info (T_MFW_BT_SERVICE_TYPE service,
                                            T_MFW_BT_DEV_ID dev_id,
                                            T_MFW_BT_HEADSET_INFO *hs_info)
{
  T_BTI_HSG_DEFAULT_INFO headset_info;

  TRACE_FUNCTION ("bt_get_default_service_info()");

  memset(&headset_info,0,sizeof(headset_info));

  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_get_headset_info(dev_id,&headset_info))
    {
      case BTP_OK:
        hs_info->priority = headset_info.priority;
        memcpy(hs_info->bd_addr,headset_info.bd_addr,sizeof(headset_info.bd_addr));
        memcpy(hs_info->hsg_name,headset_info.name,MFW_BT_HSG_NAME_MAX_LEN);
        hs_info->mfw_remote_audio_control_support = headset_info.bt_remote_audio_control_support;
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445) MODULE  : MFW_BT                    |
| STATE   : code                 ROUTINE :bt_set_prio_default_service|
+--------------------------------------------------------------------+

  PURPOSE:set priority in default headset list
*/
T_MFW_BT_RESULT_BT bt_set_prio_default_service (T_MFW_BT_SERVICE_TYPE service,
                                  T_MFW_BT_DEV_ID dev_id,
                                  T_MFW_BT_PRIORITY priority)
{
  TRACE_FUNCTION ("bt_set_prio_default_service()");

  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_set_prio_default_headset(dev_id,priority))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
  {
    TRACE_EVENT ("bt_set_prio_default_service:Error");
    return MFW_BT_FAIL;
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)            MODULE  : MFW_BT         |
| STATE   : code                            ROUTINE :bt_get_config   |
+--------------------------------------------------------------------+

  PURPOSE:get configuration parameters of bluetooth profile headset
*/
T_MFW_BT_RESULT_BT  bt_get_config (T_MFW_BT_SERVICE_TYPE service,
                                T_MFW_BT_HSG_CLIENT_CONFIG *client_conf,
                                T_MFW_BT_HSG_SERVER_CONFIG *server_conf)
{
  T_BTI_HSG_CLIENT_CONF bt_client_conf;
  T_BTI_HSG_SERVER_CONF bt_server_conf;

  TRACE_FUNCTION ("bt_get_config()");

  memset(&bt_client_conf,0,sizeof(bt_client_conf));
  memset(&bt_server_conf,0,sizeof(bt_server_conf));

  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_get_config((T_BTI_DEVICE_TYPE)service,&bt_client_conf,&bt_server_conf))
    {
      case BTP_OK:
        switch(bt_client_conf.config_mode)
        {
          case BTI_AUTO_OUTG_DEF_CONN_OFF:
            client_conf->config_mode = MFW_BT_AUTO_OUTG_DEF_CONN_OFF;
            break;
          case BTI_AUTO_OUTG_DEF_CONN_ON:
            client_conf->config_mode = MFW_BT_AUTO_OUTG_DEF_CONN_ON;
            break;
          default:
            return MFW_BT_INVALID_PARA;
        }
        client_conf->security = bt_client_conf.security;

        server_conf->security = bt_server_conf.security;
        server_conf->serv_con_conf = bt_server_conf.conn_config;
        server_conf->conn_break = bt_server_conf.conn_break;
        server_conf->conn_time = bt_server_conf.conn_time;
        server_conf->nb_phone = bt_server_conf.nb_phone;
        memcpy(server_conf->mfw_phone_list,bt_server_conf.phon_list,sizeof(bt_server_conf.phon_list));
        memcpy(server_conf->mfw_key_list,bt_server_conf.key_list,sizeof(bt_server_conf.key_list));
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}
#ifdef _SIMULATION_
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)            MODULE  : MFW_BT         |
| STATE   : code                            ROUTINE :bt_save_config  |
+--------------------------------------------------------------------+

  PURPOSE:save configuration parameters of bluetooth profile headset
*/
T_MFW_BT_RESULT_BT  bt_save_config (T_MFW_BT_SERVICE_TYPE service)
{
  TRACE_FUNCTION ("bt_save_config()");

  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_save_config((T_BTI_DEVICE_TYPE)service))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)           MODULE  : MFW_BT          |
| STATE   : code                           ROUTINE :bt_restore_config|
+--------------------------------------------------------------------+

  PURPOSE:restore configuration parameters of bluetooth profile headset
*/
T_MFW_BT_RESULT_BT  bt_restore_config (T_MFW_BT_SERVICE_TYPE service)
{
  TRACE_FUNCTION ("bt_restore_config()");

  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_restore_config((T_BTI_DEVICE_TYPE)service))
    {
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)        MODULE  : MFW_BT             |
| STATE   : code                        ROUTINE :bt_save_default_list|
+--------------------------------------------------------------------+

  PURPOSE: save default headset list in non-volatile memory
*/
T_MFW_BT_RESULT_BT  bt_save_default_list (T_MFW_BT_SERVICE_TYPE service)
{
  TRACE_FUNCTION ("bt_save_default_list()");

  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_save_def_list((T_BTI_DEVICE_TYPE)service))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE :bt_restore_default_list|
+--------------------------------------------------------------------+

  PURPOSE: restore default headset list from non-volatile memory
*/
T_MFW_BT_RESULT_BT bt_restore_default_list (T_MFW_BT_SERVICE_TYPE service)
{
  TRACE_FUNCTION ("bt_restore_default_list()");

  if(service EQ MFW_BT_HEADSET)
  {
    switch(btibtp_restore_def_list((T_BTI_DEVICE_TYPE)service))
    {
      case BTP_OK:
        return MFW_BT_EXECUTE;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}
#endif
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE : bt_set_security_mode  |
+--------------------------------------------------------------------+

  PURPOSE: set common security mode
*/
T_MFW_BT_RESULT_BT  bt_set_security_mode (T_MFW_BT_SECURITY_MODE sec_mode)
{
  TRACE_FUNCTION ("bt_set_security_mode()");

  /* security mode 1: local device will never initiate any security procedure,
                      only services which registered their security requirements
                      as no-security, will be usable
     security mode 2: local device shall initiate security procedure
                      according to the requirements of the requested service
     security mode 3: local device will initiate security procedures
                      before setting up the link and informing the host,
                      some services with no security requirements may be unusable */
  if((sec_mode EQ MFW_BT_SECURITY_MODE_1) OR
     (sec_mode EQ MFW_BT_SECURITY_MODE_2) OR
     (sec_mode EQ MFW_BT_SECURITY_MODE_3))
  {
    switch(btibtp_set_sec_mode((T_BTI_SM_SECURITY_MODE)sec_mode))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)         MODULE  : MFW_BT            |
| STATE   : code                         ROUTINE : bt_set_pair_mode  |
+--------------------------------------------------------------------+

  PURPOSE: set pairing mode
*/
T_MFW_BT_RESULT_BT  bt_set_pair_mode (T_MFW_BT_PAIRABLE_MODE pair_mode)
{
  TRACE_FUNCTION ("bt_set_pair_mode()");

  if((pair_mode EQ MFW_BT_PAIRABLE) OR
     (pair_mode EQ MFW_BT_NON_PAIRABLE))
  {
    switch(btibtp_set_pair_mode((T_BTI_SM_PAIRABLE_MODE)pair_mode))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;
      case BTP_NOT_SUPP:
        return MFW_BT_NO_SUPP;
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;
      case BTP_NOK:
        return MFW_BT_NOK;
    }
  }
  else
    return MFW_BT_FAIL;
}

/*
+------------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)      MODULE  : MFW_BT                   |
| STATE   : code                      ROUTINE : bt_set_default_security  |
+------------------------------------------------------------------------+

  PURPOSE: set default security requirements
*/
T_MFW_BT_RESULT_BT  bt_set_default_security (U8 security_level)
{
  TRACE_FUNCTION ("bt_set_default_security()");

  /* set security level of BT protocol stack for incoming and outgoing
     connections - this determines the security level needed to access
     any unregistered services */

  switch(btibtp_set_default_security(security_level))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445) MODULE : MFW_BT                     |
| STATE   : code                 ROUTINE:bt_start_profile_application|
+--------------------------------------------------------------------+

  PURPOSE : start bluetooth application

*/
T_MFW_BT_RESULT_BT  bt_start_profile_application (T_MFW_BT_SERVICE_TYPE service)
{
  /* start to activate bluetooth profiles, create resources and the profile
     instance */

  TRACE_FUNCTION ("bt_start_profile_application()");

  if((service EQ MFW_BT_HEADSET) OR
     (service EQ MFW_BT_DIAL_UP) OR
     (service EQ MFW_BT_FAX_GW) OR
     (service EQ MFW_BT_OPP) OR
     (service EQ MFW_BT_SYNC) OR
     (service EQ MFW_BT_PCA_GW))
  {
    switch(btibtp_start((T_BTI_DEVICE_TYPE) service))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;/* appli_name is not known by BTE */
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;/* appli_name is already running */
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;/* appli_name does not have a get_info function */
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;/* problem of memory in BTE */
    }
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445) MODULE : MFW_BT                     |
| STATE   : code                 ROUTINE:bt_stop_profile_application |
+--------------------------------------------------------------------+

  PURPOSE : start bluetooth application

*/
T_MFW_BT_RESULT_BT  bt_stop_profile_application (T_MFW_BT_SERVICE_TYPE service)
{
  /* start to activate bluetooth profiles, create resources and the profile
     instance */

  TRACE_FUNCTION ("bt_stop_profile_application()");

  if((service EQ MFW_BT_HEADSET) OR
     (service EQ MFW_BT_DIAL_UP) OR
     (service EQ MFW_BT_FAX_GW) OR
     (service EQ MFW_BT_OPP) OR
     (service EQ MFW_BT_SYNC) OR
     (service EQ MFW_BT_PCA_GW))
  {
    switch(btibtp_stop((T_BTI_DEVICE_TYPE) service))
    {
      case BTP_OK:
        return MFW_BT_OK;
      case BTP_INVALID_PARAMETER:
        return MFW_BT_INVALID_PARA;/* appli_name is not known by BTE */
      case BTP_NOT_READY:
        return MFW_BT_NOT_READY;/* appli_name is already running */
      case BTP_INT_ERR:
        return MFW_BT_INT_ERR;/* appli_name does not have a get_info function */
      case BTP_MEMORY_ERR:
        return MFW_BT_MEM_ERR;/* problem of memory in BTE */
    }
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)          MODULE  : MFW_BT           |
| STATE   : code                          ROUTINE : bt_register_sm   |
+--------------------------------------------------------------------+

  PURPOSE : start bluetooth security manager

*/
T_MFW_BT_RESULT_BT bt_register_sm (void)
{
  TRACE_FUNCTION ("bt_register_sm()");

  /* mandatory registration in order to handle requests from SCM -
     Security Manager - authentication and authorization */

  switch(btibtp_register_sm())/* set callback function */
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      TRACE_EVENT ("Error:bt_register_sm()");
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)          MODULE  : MFW_BT           |
| STATE   : code                          ROUTINE : bt_deregister_sm |
+--------------------------------------------------------------------+

  PURPOSE : stop bluetooth security manager

*/
T_MFW_BT_RESULT_BT bt_deregister_sm(void)
{
  TRACE_EVENT ("bt_deregister_sm()");

  /* mandatory deregistration  */

  switch(btibtp_deregister_sm())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      TRACE_EVENT ("Error:bt_deregister_sm()");
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}
/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE :bt_opp_put_reply       |
+--------------------------------------------------------------------+

  PURPOSE : reply the put requesting by opp client (about server)
*/
T_MFW_BT_RESULT_BT bt_opp_put_reply ( T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_SUBTYPE_DEV subtype,
                                      T_MFW_BT_OPP_PUT_RES opp_put_res,
                                      T_MFW_BT_OPP_OBJECT received_obj)
{
  T_BTI_OPP_OBJECT bti_opp_object;
  T_BTI_OPP_PUT_RES bti_opp_put_res;

  TRACE_EVENT ("bt_opp_put_reply()");

  memset(&bti_opp_object,0,sizeof(T_BTI_OPP_OBJECT));

  if(service EQ MFW_BT_OPP)
  {
    if(subtype EQ MFW_BT_SERVER)
    {
      /*
      *** MFW_BT_OPP_CONTINUE : positive response
          MFW_BT_OPP_R_ENTITY_TOO_L:negative response,the request is too big
          MFW_BT_OPP_FORBIDDEN: negative response, put is rejected
      ***
      */
      if(opp_put_res EQ MFW_BT_OPP_CONTINUE)
      {
        switch(received_obj.mfw_object_type)
        {
          case MFW_BT_BUFFER:/* object will be supposed to be stored in buffer */
            bti_opp_object.bt_object_type = BTI_TYPE_BUFFER;
            bti_opp_object.bt_buffer_start = received_obj.mfw_buffer_start;
            bti_opp_object.bt_buffer_size = received_obj.mfw_buffer_size;
            break;
          case MFW_BT_PATH:/* object will be supposed to be stored in file system */
            bti_opp_object.bt_object_type = BTI_TYPE_PATH;
            bti_opp_object.bt_path = received_obj.mfw_path;
            break;
          default:
            return MFW_BT_FAIL;
        }
        bti_opp_object.bt_object_name = received_obj.mfw_object_name;
        bti_opp_object.bt_object_length = received_obj.mfw_object_length;

        switch(btibtp_opp_put_resp((T_BTI_OPP_PUT_RES)opp_put_res,bti_opp_object))
        {
          case BTP_OK:
            return MFW_BT_OK;
          case BTP_INVALID_PARAMETER:
            return MFW_BT_INVALID_PARA;
          case BTP_NOT_SUPP:
            return MFW_BT_NO_SUPP;
          case BTP_NOT_READY:
            return MFW_BT_NOT_READY;
          case BTP_INT_ERR:
            return MFW_BT_INT_ERR;
          case BTP_MEMORY_ERR:
            return MFW_BT_MEM_ERR;
          case BTP_NOK:
            return MFW_BT_NOK;
        }
      }
      else if((opp_put_res EQ MFW_BT_OPP_R_ENTITY_TOO_L) OR
              (opp_put_res EQ MFW_BT_OPP_FORBIDDEN))
      {
        switch(opp_put_res)
        {
          case MFW_BT_OPP_R_ENTITY_TOO_L:
            bti_opp_put_res = BTI_OPP_TOO_LARGE;
            break;
          case MFW_BT_OPP_FORBIDDEN:
            bti_opp_put_res = BTI_OPP_FORBIDDEN;
            break;
          default:
            return MFW_BT_FAIL;
        }
        switch(btibtp_opp_put_resp(bti_opp_put_res,bti_opp_object))
        {
          case BTP_OK:
            return MFW_BT_OK;
          case BTP_INVALID_PARAMETER:
            return MFW_BT_INVALID_PARA;
          case BTP_NOT_SUPP:
            return MFW_BT_NO_SUPP;
          case BTP_NOT_READY:
            return MFW_BT_NOT_READY;
          case BTP_INT_ERR:
            return MFW_BT_INT_ERR;
          case BTP_MEMORY_ERR:
            return MFW_BT_MEM_ERR;
          case BTP_NOK:
            return MFW_BT_NOK;
        }
      }
      else
        return MFW_BT_FAIL;
    }
    else
      return MFW_BT_FAIL;
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE :bt_opp_push_object     |
+--------------------------------------------------------------------+

  PURPOSE : push an object to OPP server

*/
T_MFW_BT_RESULT_BT bt_opp_push_object (T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_SUBTYPE_DEV subtype,
                                      T_MFW_BT_BD_ADDR server_address[],
                                      BOOL mfw_keep_connection,
                                      T_MFW_BT_OPP_OBJECT mfw_obj_to_push)
{
  T_BTI_OPP_OBJECT bti_opp_object_to_push;

  TRACE_FUNCTION ("bt_opp_push_object()");

  memset(&bti_opp_object_to_push,0,sizeof(T_BTI_OPP_OBJECT));

  if(service EQ MFW_BT_OPP)
  {
    if(subtype EQ MFW_BT_CLIENT)
    {
      switch(mfw_obj_to_push.mfw_object_type)
      {
        case MFW_BT_BUFFER:/* object to be sent is stored in buffer */
          bti_opp_object_to_push.bt_object_type = BTI_TYPE_BUFFER;
          bti_opp_object_to_push.bt_buffer_start = mfw_obj_to_push.mfw_buffer_start;
          bti_opp_object_to_push.bt_buffer_size = mfw_obj_to_push.mfw_buffer_size;
          break;
        case MFW_BT_PATH:/* object to be sent is stored in file system */
          bti_opp_object_to_push.bt_object_type = BTI_TYPE_PATH;
          bti_opp_object_to_push.bt_path = mfw_obj_to_push.mfw_path;
          break;
        default:
          return MFW_BT_FAIL;
      }
      /* MIME type of object to be sent */
      bti_opp_object_to_push.bt_object_mime_type = mfw_obj_to_push.mfw_object_mime_type;
      /* name of object to be sent */
      bti_opp_object_to_push.bt_object_name = mfw_obj_to_push.mfw_object_name;
      /* length of object to be sent */
      bti_opp_object_to_push.bt_object_length = mfw_obj_to_push.mfw_object_length;

      switch(btibtp_opp_push_object(bti_opp_object_to_push,server_address,mfw_keep_connection))
      {
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE :bt_opp_pull_object     |
+--------------------------------------------------------------------+

  PURPOSE : pull an default server object from OPP server

*/
T_MFW_BT_RESULT_BT bt_opp_pull_object (T_MFW_BT_SERVICE_TYPE service,
                                       T_MFW_BT_SUBTYPE_DEV subtype,
                                       T_MFW_BT_BD_ADDR server_address[],
                                       BOOL mfw_keep_connection,
                                       T_MFW_BT_OPP_OBJECT mfw_obj_to_pull)
{
  T_BTI_OPP_OBJECT bti_opp_object_to_pull;

  TRACE_FUNCTION ("bt_opp_pull_object()");

  memset(&bti_opp_object_to_pull,0,sizeof(T_BTI_OPP_OBJECT));

  if(service EQ MFW_BT_OPP)
  {
    if(subtype EQ MFW_BT_CLIENT)
    {
      switch(mfw_obj_to_pull.mfw_object_type)
      {
        case MFW_BT_BUFFER:/* object to be pulled will be supposed to be stored in buffer */
          bti_opp_object_to_pull.bt_object_type = BTI_TYPE_BUFFER;
          bti_opp_object_to_pull.bt_buffer_start = mfw_obj_to_pull.mfw_buffer_start;
          bti_opp_object_to_pull.bt_buffer_size = mfw_obj_to_pull.mfw_buffer_size;
          break;
        case MFW_BT_PATH:/* object to be pulled will be supposed to be stored in file system */
          bti_opp_object_to_pull.bt_object_type = BTI_TYPE_PATH;
          bti_opp_object_to_pull.bt_path = mfw_obj_to_pull.mfw_path;
          break;
        default:
          return MFW_BT_FAIL;
      }
      /* MIME type of object to be pulled */
      bti_opp_object_to_pull.bt_object_mime_type = mfw_obj_to_pull.mfw_object_mime_type;
      /* name of object to be pulled */
      bti_opp_object_to_pull.bt_object_length = mfw_obj_to_pull.mfw_object_length;
      /* length of object to be pulled */
      bti_opp_object_to_pull.bt_object_name = mfw_obj_to_pull.mfw_object_name;
      switch(btibtp_opp_pull_object(bti_opp_object_to_pull,server_address,mfw_keep_connection))
      {
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  else
    return MFW_BT_FAIL;
}

/*
+--------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                |
| STATE   : code                     ROUTINE :bt_opp_exch_objects    |
+--------------------------------------------------------------------+

  PURPOSE : exchange business cards with OPP server

*/
T_MFW_BT_RESULT_BT bt_opp_exch_objects (T_MFW_BT_SERVICE_TYPE service,
                                        T_MFW_BT_SUBTYPE_DEV subtype,
                                        T_MFW_BT_OPP_OBJECT obj_to_push,
                                        T_MFW_BT_OPP_OBJECT obj_to_pull,
                                        T_MFW_BT_BD_ADDR server_address[],
                                        BOOL mfw_keep_connection)
{
  T_BTI_OPP_OBJECT bti_opp_object_to_pull;
  T_BTI_OPP_OBJECT bti_opp_object_to_push;

  TRACE_FUNCTION ("bt_opp_exch_objects()");

  memset(&bti_opp_object_to_push,0,sizeof(T_BTI_OPP_OBJECT));
  memset(&bti_opp_object_to_pull,0,sizeof(T_BTI_OPP_OBJECT));

  if(service EQ MFW_BT_OPP)
  {
    if(subtype EQ MFW_BT_CLIENT)
    {
      switch(obj_to_push.mfw_object_type)
      {
        case MFW_BT_BUFFER:/* object to be pushed is stored in buffer */
          bti_opp_object_to_push.bt_object_type = BTI_TYPE_BUFFER;
          bti_opp_object_to_push.bt_buffer_start = obj_to_push.mfw_buffer_start;
          bti_opp_object_to_push.bt_buffer_size = obj_to_push.mfw_buffer_size;
          break;
        case MFW_BT_PATH:/* object to be pushed is stored in file system */
          bti_opp_object_to_push.bt_object_type = BTI_TYPE_PATH;
          bti_opp_object_to_push.bt_path = obj_to_push.mfw_path;
          break;
        default:
          return MFW_BT_FAIL;
      }
      /* MIME type of object to be pushed */
      bti_opp_object_to_push.bt_object_mime_type = obj_to_push.mfw_object_mime_type;
      /* name of object to be pushed */
      bti_opp_object_to_push.bt_object_name = obj_to_push.mfw_object_name;
      /* length of object to be pushed */
      bti_opp_object_to_push.bt_object_length = obj_to_push.mfw_object_length;

      switch(obj_to_pull.mfw_object_type)
      {
        case MFW_BT_BUFFER:/* object to be pulled will be supposed to be stored in buffer */
          bti_opp_object_to_pull.bt_object_type = BTI_TYPE_BUFFER;
          bti_opp_object_to_pull.bt_buffer_start = obj_to_pull.mfw_buffer_start;
          bti_opp_object_to_pull.bt_buffer_size = obj_to_pull.mfw_buffer_size;
          break;
        case MFW_BT_PATH:/* object to be pulled will be supposed to be stored in file system */
          bti_opp_object_to_pull.bt_object_type = BTI_TYPE_PATH;
          bti_opp_object_to_pull.bt_path = obj_to_pull.mfw_path;
          break;
        default:
          return MFW_BT_FAIL;
      }
      /*  MIME type of object to be pulled */
      bti_opp_object_to_pull.bt_object_mime_type = obj_to_pull.mfw_object_mime_type;
      /*  length of object to be pulled */
      bti_opp_object_to_pull.bt_object_length = obj_to_pull.mfw_object_length;
      /*  name of object to be pulled */
      bti_opp_object_to_pull.bt_object_name = obj_to_pull.mfw_object_name;

      switch(btibtp_opp_exch_objects(bti_opp_object_to_push,bti_opp_object_to_pull,
                                     server_address,mfw_keep_connection))
      {
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  else
    return MFW_BT_FAIL;
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE : bt_syn_s_send_com       |
+----------------------------------------------------------------------+

  PURPOSE : SYNC server send SYN command to SYNC client with
            SYNC command support

*/
T_MFW_BT_RESULT_BT bt_syn_s_send_com (T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_SUBTYPE_DEV subtype,
                                      T_MFW_BT_SYNC_COMMAND sync_comm,
                                      T_MFW_BT_SYN_OBJECT_STORE sync_object,
                                      T_MFW_BT_BD_ADDR sync_client_address[])
{
  T_BTI_SYNC_COMMAND bt_sync_com;
  T_BTI_SYN_OBJECT_STORE bt_sync_object;

  TRACE_FUNCTION ("bt_syn_s_send_com()");

  bt_sync_object = (T_BTI_SYN_OBJECT_STORE)sync_object;

  if(service EQ MFW_BT_SYNC)
  {
    if(subtype EQ MFW_BT_SERVER)
    {
      switch(sync_comm)
      {
        case MFW_BT_SYNC_CO_SYNC:
          /* SYNC server push command: SYNC synchronization initiated by Server */
          bt_sync_com = sync_comm;
          break;
        case MFW_BT_SYNC_CO_PUT:
          /* SYNC server push command: PUT */
          bt_sync_com = sync_comm;
          break;
        default:
          return MFW_BT_FAIL;
      }

      switch(btibtp_sync_ser_send_com(bt_sync_com,bt_sync_object,sync_client_address))
      {
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  else
    return MFW_BT_FAIL;
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE : bt_syn_s_auth_res       |
+----------------------------------------------------------------------+

  PURPOSE : SYNC server sends authentication response

*/
T_MFW_BT_RESULT_BT bt_syn_s_auth_res (T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_SUBTYPE_DEV subtype,
                                      T_MFW_BT_SYN_PASSWD * mfw_sync_password)
{
  TRACE_FUNCTION ("bt_syn_s_auth_res()");

  if(service EQ MFW_BT_SYNC)
  {
    if(subtype EQ MFW_BT_SERVER)
    {
      if(mfw_sync_password)
      {
        switch(btibtp_sync_ser_auth_res(mfw_sync_password))
        {
          case BTP_OK:
            return MFW_BT_OK;
          case BTP_INVALID_PARAMETER:
            return MFW_BT_INVALID_PARA;
          case BTP_NOT_SUPP:
            return MFW_BT_NO_SUPP;
          case BTP_NOT_READY:
            return MFW_BT_NOT_READY;
          case BTP_INT_ERR:
            return MFW_BT_INT_ERR;
          case BTP_MEMORY_ERR:
            return MFW_BT_MEM_ERR;
          case BTP_NOK:
            return MFW_BT_NOK;
        }
      }
      else
        return MFW_BT_INVALID_PARA;
    }
    else
      return MFW_BT_FAIL;
  }
  else
    return MFW_BT_FAIL;
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE : bt_syn_s_sync_terminate |
+----------------------------------------------------------------------+

  PURPOSE : SYNC server terminates synchronization

*/
T_MFW_BT_RESULT_BT bt_syn_s_sync_terminate (T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_SUBTYPE_DEV subtype)
{
  TRACE_FUNCTION ("bt_syn_s_sync_terminate()");

  if(service EQ MFW_BT_SYNC)
  {
    if(subtype EQ MFW_BT_SERVER)
    {
      switch(btibtp_sync_syn_terminate())
      {
        case BTP_OK:
          return MFW_BT_OK;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  else
    return MFW_BT_FAIL;
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE : bt_syn_s_pull_resp      |
+----------------------------------------------------------------------+

  PURPOSE : SYNC server sends pull request response

*/
T_MFW_BT_RESULT_BT bt_syn_s_pull_resp (T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_SUBTYPE_DEV subtype,
                                      T_MFW_BT_SYNC_OBJECT mfw_syn_pull_obj)
{
  T_BTI_SYNC_OBJECT syn_pull_object;

  TRACE_FUNCTION ("bt_syn_s_pull_resp()");

  memset(&syn_pull_object,0,sizeof(syn_pull_object));

  if(service EQ MFW_BT_SYNC)
  {
    if(subtype EQ MFW_BT_SERVER)
    {
      switch(mfw_syn_pull_obj.mfw_object_type)
      {
        case MFW_BT_BUFFER:/* object to be pulled will be supposed to be stored in buffer */
          syn_pull_object.bt_object_type = BTI_TYPE_BUFFER;
          syn_pull_object.bt_buffer_start = mfw_syn_pull_obj.mfw_buffer_start;
          syn_pull_object.bt_buffer_size = mfw_syn_pull_obj.mfw_buffer_size;
          break;
        case MFW_BT_PATH:/* object to be pulled will be supposed to be stored in file system */
          syn_pull_object.bt_object_type = BTI_TYPE_PATH;
          syn_pull_object.bt_path = mfw_syn_pull_obj.mfw_path;
          break;
        default:
          return MFW_BT_FAIL;
      }
      /* MIME type of object to be pulled */
      syn_pull_object.bt_object_mime_type = mfw_syn_pull_obj.mfw_object_mime_type;
      /* name of object to be pulled */
      syn_pull_object.bt_object_length = mfw_syn_pull_obj.mfw_object_length;
      /* length of object to be pulled */
      syn_pull_object.bt_object_name = mfw_syn_pull_obj.mfw_object_name;

      switch(btibtp_sync_ser_pull_res(syn_pull_object))
      {
        case BTP_OK:
          return MFW_BT_OK;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  else
    return MFW_BT_FAIL;
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE : bt_syn_s_push_resp      |
+----------------------------------------------------------------------+

  PURPOSE : SYNC server sends push request response

*/
T_MFW_BT_RESULT_BT bt_syn_s_push_resp (T_MFW_BT_SERVICE_TYPE service,
                                      T_MFW_BT_SUBTYPE_DEV subtype,
                                      T_MFW_BT_SYNC_OBJECT mfw_syn_push_obj)
{
  T_BTI_SYNC_OBJECT syn_push_object;

  TRACE_FUNCTION ("bt_syn_s_push_resp()");

  memset(&syn_push_object,0,sizeof(syn_push_object));

  if(service EQ MFW_BT_SYNC)
  {
    if(subtype EQ MFW_BT_SERVER)
    {
      switch(mfw_syn_push_obj.mfw_object_type)
      {
        case MFW_BT_BUFFER:/* object to be pulled will be supposed to be stored in buffer */
          syn_push_object.bt_object_type = BTI_TYPE_BUFFER;
          syn_push_object.bt_buffer_start = mfw_syn_push_obj.mfw_buffer_start;
          syn_push_object.bt_buffer_size = mfw_syn_push_obj.mfw_buffer_size;
          break;
        case MFW_BT_PATH:/* object to be pulled will be supposed to be stored in file system */
          syn_push_object.bt_object_type = BTI_TYPE_PATH;
          syn_push_object.bt_path = mfw_syn_push_obj.mfw_path;
          break;
        default:
          return MFW_BT_FAIL;
      }
      /* MIME type of object to be pulled */
      syn_push_object.bt_object_mime_type = mfw_syn_push_obj.mfw_object_mime_type;
      /* name of object to be pulled */
      syn_push_object.bt_object_length = mfw_syn_push_obj.mfw_object_length;
      /* length of object to be pulled */
      syn_push_object.bt_object_name = mfw_syn_push_obj.mfw_object_name;

      switch(btibtp_sync_ser_push_res(syn_push_object))
      {
        case BTP_OK:
          return MFW_BT_EXECUTE;
        case BTP_INVALID_PARAMETER:
          return MFW_BT_INVALID_PARA;
        case BTP_NOT_SUPP:
          return MFW_BT_NO_SUPP;
        case BTP_NOT_READY:
          return MFW_BT_NOT_READY;
        case BTP_INT_ERR:
          return MFW_BT_INT_ERR;
        case BTP_MEMORY_ERR:
          return MFW_BT_MEM_ERR;
        case BTP_NOK:
          return MFW_BT_NOK;
      }
    }
    else
      return MFW_BT_FAIL;
  }
  else
    return MFW_BT_FAIL;
}

/* BT CTRL */

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_set_conn_mode(T_MFW_BT_CONNECTABLE_MODE cmode,
                           UINT16 scan_interval,
                           UINT16 scan_window)
{
  TRACE_FUNCTION ("bt_set_conn_mode()");

  switch(btibtp_set_conn_mode(cmode, scan_interval, scan_window))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }

}


/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_set_disc_mode(T_MFW_BT_DISCOVERABLE_MODE dmode,
                           UINT16 scan_interval,
                           UINT16 scan_window)
{
  TRACE_FUNCTION ("bt_set_disc_mode()");

  switch(btibtp_set_disc_mode(dmode, scan_interval, scan_window))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_get_local_name(void)
{
  TRACE_FUNCTION ("bt_get_local_name()");

  switch(btibtp_get_local_name())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_set_local_name(T_MFW_BT_BD_NAME name[])
{
  TRACE_FUNCTION ("bt_set_local_name()");

  switch(btibtp_set_local_name(name))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_get_bd_addr(void)
{
  TRACE_FUNCTION ("bt_get_bd_addr()");

  switch(btibtp_get_bd_addr())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_get_remote_dev_info(T_MFW_BT_BD_ADDR bd_addr[])
{
  TRACE_FUNCTION ("bt_get_remote_dev_info()");

  switch(btibtp_get_remote_dev_info(bd_addr))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_start_btctrl(void)
{
  TRACE_FUNCTION ("bt_start_btctrl()");

  switch(btibtp_start_btctrl())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_stop_btctrl(void)
{
  TRACE_FUNCTION ("bt_stop_btctrl()");

  switch(btibtp_stop_btctrl())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/* enable profiles */
/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_enable_hsg_profile(void)
{
  TRACE_FUNCTION ("bt_enable_hsg_profile()");
  switch(btibtp_enable_hsg_profile())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_enable_dun_profile(void)
{
  TRACE_FUNCTION ("bt_enable_dun_profile()");
  switch(btibtp_enable_dun_profile())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_enable_fax_profile(void)
{
  TRACE_FUNCTION ("bt_enable_fax_profile()");
  switch(btibtp_enable_fax_profile())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_hsg_client_disable(void)
{
  TRACE_FUNCTION ("bt_hsg_client_disable()");
  switch(btibtp_hsg_client_disable())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_hsg_server_disable(void)
{
  TRACE_FUNCTION ("bt_hsg_server_disable()");
  switch(btibtp_hsg_server_disable())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_dun_gw_disable(void)
{
  TRACE_FUNCTION ("bt_dun_gw_disable()");
  switch(btibtp_dun_gw_disable())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_fax_gw_disable(void)
{
  TRACE_FUNCTION ("bt_fax_gw_disable()");
  switch(btibtp_fax_gw_disable())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_opp_client_disable(void)
{
  TRACE_FUNCTION ("bt_opp_client_disable()");
  switch(btibtp_opp_client_disable())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_opp_server_disable(void)
{
  TRACE_FUNCTION ("bt_opp_server_disable()");
  switch(btibtp_opp_server_disable())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : MMI-Framework (8445)     MODULE  : MFW_BT                  |
| STATE   : code                     ROUTINE :                         |
+----------------------------------------------------------------------+

  PURPOSE :
*/
T_MFW_BT_RESULT_BT bt_syn_server_disable(void)
{
  TRACE_FUNCTION ("bt_syn_server_disable()");
  switch(btibtp_syn_server_disable())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

/*#ifdef PCA_6350*/
T_MFW_BT_RESULT_BT bt_pca_gw_enable (void)
{
  TRACE_FUNCTION ("bt_pca_gw_enable()");
  switch(btibtp_pca_gw_enable())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

T_MFW_BT_RESULT_BT bt_pca_gw_config (T_MFW_BT_PCA_CONFIG event_filters)
{
  T_BTI_PCA_CONF events;
  TRACE_FUNCTION ("bt_pca_gw_config()");

  events.link_filter = (T_BTI_PCA_LINK_EVENT)event_filters.link_event;
  events.call_filter = (T_BTI_PCA_CALL_EVENT)event_filters.call_event;

  switch(btibtp_pca_gw_config (events))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

T_MFW_BT_RESULT_BT bt_pca_gw_security (UINT8  security_level)
{
  TRACE_FUNCTION ("bt_pca_gw_security()");
  switch(btibtp_pca_gw_security (security_level))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

T_MFW_BT_RESULT_BT bt_pca_gw_hangup (void)
{
  TRACE_FUNCTION ("bt_pca_gw_hangup()");
  switch(btibtp_pca_gw_hangup())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}

T_MFW_BT_RESULT_BT bt_pca_gw_disable (void)
{
  TRACE_FUNCTION ("bt_pca_gw_disable()");
  switch(btibtp_pca_gw_disable())
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}


/*#endif*/ /* PCA_6350 */


/* Cartman added begin */

T_MFW_BT_RESULT_BT bt_hsg_connect_network (T_MFW_BT_HSG_CONNECT_NETWORK_ANSWER network_answer,
                                                  T_MFW_BT_HSG_CONNECT_NETWORK_TYPE network_type,
                                                  T_MFW_BT_HSG_PHONE_NUMBER phone_nb[])
{
  TRACE_FUNCTION ("bt_hsg_connect_network()");
  switch(btibtp_hsg_connect_network(network_answer,network_type,phone_nb))
  {
    case BTP_OK:
      return MFW_BT_EXECUTE;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}


T_MFW_BT_RESULT_BT bt_hsg_send_specific_cmd_to_hs (T_MFW_BT_HSG_SPECIFIC_CMD_TYPE command_type,
                                                  T_MFW_BT_HSG_CMD_TO_HS cmd[])
{
  TRACE_FUNCTION ("bt_hsg_send_specific_cmd_to_hs()");
  switch(btibtp_hsg_send_specific_cmd_to_hs(command_type,cmd))
  {
    case BTP_OK:
      return MFW_BT_EXECUTE;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}


T_MFW_BT_RESULT_BT bt_dun_gw_security (UINT8  security_level)
{
  TRACE_FUNCTION ("bt_dun_gw_security()");
  switch(btibtp_dun_gw_security (security_level))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}


T_MFW_BT_RESULT_BT bt_opp_client_security (UINT8  security_level)
{
  TRACE_FUNCTION ("bt_opp_client_security()");
  switch(btibtp_opp_client_security (security_level))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}


T_MFW_BT_RESULT_BT bt_opp_server_security (UINT8  security_level)
{
  TRACE_FUNCTION ("bt_opp_server_security()");
  switch(btibtp_opp_server_security (security_level))
  {
    case BTP_OK:
      return MFW_BT_OK;
    case BTP_INVALID_PARAMETER:
      return MFW_BT_INVALID_PARA;
    case BTP_NOT_SUPP:
      return MFW_BT_NO_SUPP;
    case BTP_NOT_READY:
      return MFW_BT_NOT_READY;
    case BTP_INT_ERR:
      return MFW_BT_INT_ERR;
    case BTP_MEMORY_ERR:
      return MFW_BT_MEM_ERR;
    case BTP_NOK:
      return MFW_BT_NOK;
  }
}


/* Cartman added end */
