/*
+--------------------------------------------------------------------+
| PROJECT:	MMI-Framework (8417)		$Workfile::	mfw_aud.c	    $|
| $Author::	NDH							$Revision::	1			    $|
| CREATED:	04.02.03		     		$Modtime::	10.04.00 14:58	$|
| STATE  :	code														 |
+--------------------------------------------------------------------+

   MODULE  : MFW_AUD

   PURPOSE : This module contains Audio Riveria Interface functions.

   HISTORY:

    Mar 03, 2005 REF: CRR MMI-ENH-28950 xnkulkar
    Description:  RE: Vocoder interface change
    Solution: Function call 'enable_tch_vocoder()' is replaced with new functions 'hl_drv_enable_vocoder()'
    		    and 'hl_drv_disable_vocoder()'

    Aug 25, 2004  REF: CRR 20655  xnkulkar
    Description: Voice Memo functionality not working
    Solution:	  The voice recording functionality was failing because 
    			 " mmi" folder is not present. As a solution, we create the 
    			 "mmi" folder and then proceed with recording.
*/


/*
** Include Files
*/

#define ENTITY_MFW

/* BEGIN ADD: Req ID: : Sumit : 14-Mar-05 */
#ifndef NEPTUNE_BOARD
/* END ADD: Req ID: : Sumit : 14-Mar-05 */

/* includes */
#include <string.h>

#include "typedefs.h"
#include "vsi.h"
#include "pei.h"
#include "custom.h"
#include "gsm.h"

#include "mfw_ffs.h"
#include "mfw_aud.h"

#include "Audio/audio_api.h"

// Mar 03, 2005 REF: CRR MMI-ENH-28950 xnkulkar
#include "hl_audio_drv.h"

/*
** Local Variable Definitions
*/

static T_RV_RETURN_PATH voice_memo_return_path;


/*
** Local Macro Definitions
*/

#define VM_FILE_NAME	"vmfile"	// FFS File Name
#define VM_FOLDER		"/mmi/vm"

#define VM_MICROPHONE_GAIN		(0x0100)	// Default Gain of 1
#define VM_NETWORK_GAIN			(0x0100)	// Default Gain of 1

#define VM_TONE_DURATION		(50)
#define VM_TONE_INTERVAL		(50)
#define VM_TONE_AMPLITUDE		(-24)		// Default Amplitude of -24dB


/*
** Local Function Prototypes
*/

static void mfw_aud_vm_create_file(const char *folder, const char *fname, UBYTE index);
static void configure_callback_fn(void (*callback_fn)(void *));
static void configure_vm_filename(char *vm_filename, const char *folder, const char *fname, UBYTE index);


/*
** Public function Definitions
*/



/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: mfw_aud_vm_delete_file	         |
+--------------------------------------------------------------------+


   PURPOSE :  Delete the file which held the Voice Memo

*/
SHORT mfw_aud_vm_delete_file(void)
{
	char tmpFile[AUDIO_PATH_NAME_MAX_SIZE];

	configure_vm_filename(tmpFile, VM_FOLDER, VM_FILE_NAME, 0);
	
	ffs_remove(tmpFile);

	return MFW_AUD_VM_OK;
}


/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: mfw_aud_vm_start_playback       |
+--------------------------------------------------------------------+


   PURPOSE :  Start playback of a previously recorded Voice Memo

*/
SHORT mfw_aud_vm_start_playback(void (*callback_fn)(void *))
{
	T_AUDIO_RET						audio_riv_retVal;
	T_AUDIO_VM_PLAY_PARAMETER		mfw_vm_play_param;

	if (FFS_flashData.voice_memo_position EQ 0)
	{
		return MFW_AUD_VM_MEM_EMPTY;
	}
	else
	{

		configure_callback_fn(callback_fn);

		//Set up the Voice Memo filename in FFS
		configure_vm_filename(mfw_vm_play_param.memo_name, VM_FOLDER, VM_FILE_NAME, 0);

		// Call Riviera function to start playback.
#ifndef WIN32	// only if not in Windows
		audio_riv_retVal = audio_vm_play_start(&mfw_vm_play_param,
							 				voice_memo_return_path);

		// If the Riviera call failed
		if (audio_riv_retVal != RV_OK)
			return MFW_AUD_VM_RIVIERA_FAILED;
#endif
	}

	return MFW_AUD_VM_OK;
}


/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: mfw_aud_vm_stop_playback        |
+--------------------------------------------------------------------+


   PURPOSE :  Stop playback of a previously recorded Voice Memo

*/
SHORT mfw_aud_vm_stop_playback(void (*callback_fn)(void *))
{
	T_AUDIO_RET						audio_riv_retVal;

	configure_callback_fn(callback_fn);

#ifndef WIN32	// only if not in Windows
	audio_riv_retVal = audio_vm_play_stop(voice_memo_return_path);

	// If the Riviera call failed
	if (audio_riv_retVal != RV_OK)
		return MFW_AUD_VM_RIVIERA_FAILED;

#endif

	return MFW_AUD_VM_OK;
}


/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: mfw_aud_vm_start_record         |
+--------------------------------------------------------------------+


   PURPOSE :  Configure Riviera and start recording a Voice Memo

*/
SHORT mfw_aud_vm_start_record(UBYTE max_duration, void (*callback_fn)(void *))
{
	T_AUDIO_RET						audio_riv_retVal;
	T_AUDIO_VM_RECORD_PARAMETER		mfw_vm_record_param;
	T_AUDIO_TONES_PARAMETER			mfw_vm_tones_param;
	char * mmiDir = "/mmi"; // Aug 25, 2004  REF: CRR 20655  xnkulkar

	configure_callback_fn(callback_fn);

	//Set up, and create, the Voice Memo filename in FFS
	configure_vm_filename(mfw_vm_record_param.memo_name, VM_FOLDER, VM_FILE_NAME, 0);
       //Aug 25, 2004  REF: CRR 20655  xnkulkar
       // We now create the "mmi" folder and then proceed with recording.	
       flash_makedir(mmiDir);  
	mfw_aud_vm_create_file(VM_FOLDER, VM_FILE_NAME, 0);

	// Setup the Voice Memo Tones
	mfw_vm_record_param.memo_duration = (UINT32)max_duration;
	mfw_vm_record_param.compression_mode = FALSE;	// No Compression
	mfw_vm_record_param.microphone_gain = VM_MICROPHONE_GAIN;
	mfw_vm_record_param.network_gain = VM_NETWORK_GAIN;

	mfw_vm_tones_param.tones[0].start_tone = 0;
	mfw_vm_tones_param.tones[0].stop_tone = VM_TONE_DURATION;
	mfw_vm_tones_param.tones[0].frequency_tone = 520;			// Tone 1 Frequecny in Hz
	mfw_vm_tones_param.tones[0].amplitude_tone = VM_TONE_AMPLITUDE;

	mfw_vm_tones_param.tones[1].start_tone = mfw_vm_tones_param.tones[0].stop_tone + VM_TONE_INTERVAL;
	mfw_vm_tones_param.tones[1].stop_tone = mfw_vm_tones_param.tones[1].start_tone + VM_TONE_DURATION;
	mfw_vm_tones_param.tones[1].frequency_tone = 643;			// Tone 2 Frequecny in Hz
	mfw_vm_tones_param.tones[1].amplitude_tone = VM_TONE_AMPLITUDE;

	mfw_vm_tones_param.tones[2].start_tone = mfw_vm_tones_param.tones[1].stop_tone + VM_TONE_INTERVAL;
	mfw_vm_tones_param.tones[2].stop_tone = mfw_vm_tones_param.tones[2].start_tone + VM_TONE_DURATION;
	mfw_vm_tones_param.tones[2].frequency_tone = 775;			// Tone 3 Frequecny in Hz
	mfw_vm_tones_param.tones[2].amplitude_tone = VM_TONE_AMPLITUDE;

	mfw_vm_tones_param.frame_duration = mfw_vm_tones_param.tones[2].stop_tone * 2;
	mfw_vm_tones_param.sequence_duration = mfw_vm_tones_param.frame_duration * 2;
	mfw_vm_tones_param.period_duration = mfw_vm_tones_param.sequence_duration;
	mfw_vm_tones_param.repetition = TONE_INFINITE;

	// Call Riviera function to start recording.
#ifndef WIN32	// only if not in Windows
	audio_riv_retVal = audio_vm_record_start(&mfw_vm_record_param,
										  &mfw_vm_tones_param,
						 				  voice_memo_return_path);

	// If the Riviera call failed
	if (audio_riv_retVal != RV_OK)
		return MFW_AUD_VM_RIVIERA_FAILED;
#endif

	return MFW_AUD_VM_OK;
}


/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: mfw_aud_vm_stop_record          |
+--------------------------------------------------------------------+


   PURPOSE :  Stop recording a Voice Memo

*/
SHORT mfw_aud_vm_stop_record(void (*callback_fn)(void *))
{
	T_AUDIO_RET						audio_riv_retVal;

	configure_callback_fn(callback_fn);

#ifndef WIN32	// only if not in Windows
	audio_riv_retVal = audio_vm_record_stop(voice_memo_return_path);

	// If the Riviera call failed
	if (audio_riv_retVal != RV_OK)
		return MFW_AUD_VM_RIVIERA_FAILED;
#endif

	return MFW_AUD_VM_OK;
}


/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: mfw_aud_vm_get_duration         |
+--------------------------------------------------------------------+


   PURPOSE :  Get the duration of the previously recorded Voice Memo

*/
UBYTE mfw_aud_vm_get_duration(void)
{
	return FFS_flashData.voice_memo_position;
}


/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: mfw_aud_vm_get_duration         |
+--------------------------------------------------------------------+


   PURPOSE :  Set the duration of the previously recorded Voice Memo

*/
void mfw_aud_vm_set_duration(UBYTE duration)
{
	FFS_flashData.voice_memo_position = duration;

	flash_write();
	return;
}


/*
** Local Function Definitions
*/

/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: mfw_aud_vm_create_file	         |
+--------------------------------------------------------------------+


   PURPOSE :  Create the FFS file required for the Riviera Voice Memo

*/
static void mfw_aud_vm_create_file(const char *folder, const char *fname, UBYTE index)
{
	UBYTE dummy = 0x00;
	
	flash_data_write(folder, fname, &dummy, sizeof(dummy));
	return;
}


/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: configure_callback_fn	         |
+--------------------------------------------------------------------+


   PURPOSE :  Configure the Riviera Return PAth

*/
static void configure_callback_fn(void (*callback_fn)(void *))
{
	voice_memo_return_path.addr_id = 0;
	voice_memo_return_path.callback_func = callback_fn;

	return;
}



/*
+--------------------------------------------------------------------+
| PROJECT: MMI-Framework (8417) 	MODULE: MFW_AUD		             |
| STATE  : code 			ROUTINE: configure_vm_filename	         |
+--------------------------------------------------------------------+


   PURPOSE :  Create the Voice memo filename from the incoming parameters

*/
static void configure_vm_filename(char *vm_filename, const char *folder, const char *fname, UBYTE index)
{
	memset(vm_filename, 0x00, AUDIO_PATH_NAME_MAX_SIZE);
	strcpy(vm_filename, folder);
	strcat(vm_filename, "/");
	strcat(vm_filename, fname);
}

// Mar 03, 2005 REF: CRR MMI-ENH-28950 xnkulkar
// Commented as we are no more using 'enable_tch_vocoder()' from MFW
/*
#ifndef  _SIMULATION_
void enable_tch_vocoder(BOOL enabled);
#endif
*/

/**********************************************************************
**
**          MFW Riviera Vocoder Interface functions
**
**********************************************************************/

void mfw_aud_l1_enable_vocoder ( void )
{
#ifndef  _SIMULATION_
// Mar 03, 2005 REF: CRR MMI-ENH-28950 xnkulkar
// Call 'hl_drv_enable_vocoder()' instead of 'enable_tch_vocoder(TRUE)'
//	enable_tch_vocoder(TRUE);
	T_HL_VOICE_DRV_RSLT vocoderState;
	vocoderState = hl_drv_enable_vocoder();
	if ( vocoderState == HL_VOICE_DRV_FAIL )
	{
		TRACE_EVENT("vocoder driver : FAILED ");
		return;
	}
#endif
	return;
}

void mfw_aud_l1_disable_vocoder ( void )
{
#ifndef  _SIMULATION_
// Mar 03, 2005 REF: CRR MMI-ENH-28950 xnkulkar
// Call 'hl_drv_disable_vocoder()' instead of 'enable_tch_vocoder(FALSE)'
//	enable_tch_vocoder(FALSE);
	hl_drv_disable_vocoder();	
#endif
	return;
}

/****************************************************************/
/* NEPTUNE DEFINITIONS START HERE */
/* BEGIN ADD: Req ID: : Sumit : 14-Mar-05 */
#else /* NEPTUNE_BOARD */
#include "typedefs.h"
#include "mfw_aud.h"

SHORT mfw_aud_vm_delete_file(void)
{
	return MFW_AUD_VM_OK;
}

SHORT mfw_aud_vm_start_playback(void (*callback_fn)(void *))
{
	return MFW_AUD_VM_OK;
}

SHORT mfw_aud_vm_stop_playback(void (*callback_fn)(void *))
{
	return MFW_AUD_VM_OK;
}

SHORT mfw_aud_vm_start_record(UBYTE max_duration, void (*callback_fn)(void *))
{
	return MFW_AUD_VM_OK;
}

SHORT mfw_aud_vm_stop_record(void (*callback_fn)(void *))
{
	return MFW_AUD_VM_OK;
}

UBYTE mfw_aud_vm_get_duration(void)
{
	return 0;
}

void mfw_aud_vm_set_duration(UBYTE duration)
{
	return;
}

/*
** Layer1 Audio interface functions
*/
void mfw_aud_l1_enable_vocoder ( void )
{
}

void mfw_aud_l1_disable_vocoder ( void )
{
}

#endif /* NEPTUNE_BOARD*/
/* END ADD: Req ID: : Sumit : 14-Mar-05 */

