/*******************************************************************************

					CONDAT (UK)

********************************************************************************                                                                              

 This software product is the property of Condat (UK) Ltd and may not be
 disclosed to any third party without the express permission of the owner.                                 
                                                                              
********************************************************************************

 $Project name:                                                   
 $Project code:	                                                           
 $Module:		
 $File:		    AUIWapPush.c
 $Revision:		                                                      
                                                                              
 $Author:		Condat(UK)                                                         
 $Date:		                                                          
                                                                               
********************************************************************************
                                                                              
 Description:
  	The user interface for WAP PUSH.
                        
********************************************************************************

 $History: AUIWapPush.c

	xrashmic 11 Jan, 2005 MMI-SPR-OMAPS00063638
	Ignoring the duplicate push message part received. (The same part 
	already exists in the push message store)
				
	xreddymn Mar-22-2005 MMI-SPR-29767
	Modified behaviour of SL push message handling in MMI

    xrashmic 9 May, 2005 MMI-SPR-30159
    Concat PUSH msg are stored and sent to wap client once all parts are received

    xreddymn Jan-10-2005 MMI-SPR-27618: Modified WAP push URL display screen
             to prevent user from editing the contents.

	21/07/03 - SH - Created
	   
 $End

*******************************************************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>


#if defined (NEW_FRAME)

#include "typedefs.h"
#include "vsi.h"
#include "pei.h"
#include "custom.h"
#include "gsm.h"

#else  /* NEW_FRAME */

#include "STDDEFS.H"
#include "custom.h"
#include "gsm.h"
#include "vsi.h"

#endif /* NEW_FRAME */

#include "mfw_sys.h"

#include "mfw_mfw.h"
#include "mfw_win.h"
#include "mfw_kbd.h"
#include "mfw_edt.h"
#include "mfw_tim.h"
#include "mfw_phb.h"
#include "mfw_sms.h"
#include "mfw_icn.h"
#include "mfw_mnu.h"
#include "mfw_lng.h"
#include "mfw_sat.h"
#include "mfw_kbd.h"
#include "mfw_nm.h"
#include "mfw_cm.h"

#include "dspl.h"

#include "wapmic_types.h"

#include "MmiMmi.h"
#include "MmiMain.h"
#include "MmiDummy.h"
#include "MmiDialogs.h"
#include "MmiLists.h"
#include "mmiCall.h"

#include "MmiMenu.h"
#include "MmiSoftKeys.h"
#include "MmiResources.h"

#include "MmiBlkLangDB.h"
#include "MmiBookUtils.h"
//#include "MmiEditor.h"
//#include "MmiEditor_i.h"
#include "MmiBookShared.h"
#include "ATBData.h"
#include "ATBWapAUI.h"
#include "mmiSmsMenu.h"
#include "mmismssend.h"
#include "AUIWapext.h"
#include "AUIWap.h"
#ifdef MMI_GPRS_ENABLED
#include "MmiGprs.h"
#endif

#include "cus_aci.h"
#include "prim.h"
#ifndef PCM_2_FFS
#include "pcm.h"
#endif


#include "mmiColours.h"
#include "font_bitmaps.h"

//xrashmic 9 May, 2005 MMI-SPR-30159
#include "conc_sms.h"
#include "ffs/ffs.h"
#include "mfw_td.h"

//xrashmic 9 May, 2005 MMI-SPR-30159
//Maximum number of parts per message
#define MAX_PUSH_MSG_PARTS 5
//Maximum PUSH message that can be stored in FFS
#define MAX_PUSH_MSG_STORAGE 5
//Maximum duration after which the PUSH message will be deleted from FFS
#define MAX_DURATION 2 //no of days

typedef struct
{
    U16 msg_ref;
    U8 msg_saved;
    U8 seq_num;
    U8 num_msgs;
    U8 year;
    U8 month;
    U8 day;
}AUI_PUSH_REPOSITORY_TABLE;
AUI_PUSH_REPOSITORY_TABLE  PUSH_Table[MAX_PUSH_MSG_STORAGE];

enum PUSH_STORAGE
{
    MSG_NONE=0,
    MSG_FFS,
    MSG_NEW    
};
 //xrashmic 11 Jan, 2005 MMI-SPR-OMAPS00063638
 typedef enum
{
    DUPLICATE=-1,
    NOT_FOUND=0,
    FOUND
}PUSHMSG_SEARCH_RESULT;
typedef struct
{
   U8 storage;
   U8 index;
}AUI_SORTED_PUSH_REPOSITORY;
AUI_SORTED_PUSH_REPOSITORY PUSH_Sorted_Parts[MAX_PUSH_MSG_PARTS];


static int AUI_wap_mfw_cb(MfwEvt event, MfwHnd hnd);
static void AUI_wap_push_start_load_SI(UBYTE result);

static void AUI_wap_push_start_cb(UBYTE result);
static void AUI_wap_push_show_cb(T_MFW_HND win, USHORT identifier, USHORT reason);
static void AUI_wap_push_list_cb(T_MFW_HND win, ListMenuData *EntryListData);


/*******************************************************************************

 $Function:    	AUI_wap_push_start

 $Description:	Start PUSH
 				SPR#2086 - SH - Added
 
 $Returns:		None

 $Arguments:	None
 
*******************************************************************************/

void AUI_wap_push_start()
{

#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_start()");
#endif

	/* Create SMS handler  */
		
	AUI_wap_start(WAP_PUSH, (T_WAP_CB)AUI_wap_push_start_cb);

	return;
}


static void AUI_wap_push_start_cb(UBYTE result)
{
	T_WAP_DATA *data;
	
	TRACE_FUNCTION("AUI_wap_push_start_cb");

	if (!result)
	{
		TRACE_EVENT("*** Failed to start up for WAP PUSH ***");
		return;
	}

	data = AUI_wap_data();
	
	data->PushData = (T_WAP_PUSH_DATA *)AUI_wap_memory_alloc(sizeof(T_WAP_PUSH_DATA));
	data->PushData->List = NULL;
	data->PushData->list_created = FALSE;
	data->PushData->service_indication = -1;
	memset((void *)data->PushData->list_ids, 0, sizeof(WAP_PUSH_MAX_MESSAGES));

	/* Create SMS handler */
	data->PushData->Sms_handler = sms_create(0, E_SMS_WAP_MT_RECEIVED, (T_MFW_CB)AUI_wap_mfw_cb);

	return;
}

/*******************************************************************************

 $Function:    	AUI_wap_push_destroy

 $Description:	Start PUSH
 				SPR#2086 - SH - Added
 
 $Returns:		None

 $Arguments:	None
 
*******************************************************************************/

void AUI_wap_push_destroy()
{
	T_WAP_DATA *data = AUI_wap_data();
	
#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_destroy()");
#endif

	if (!data)
	{
		return;
	}

	if (!data->PushData)
	{
		return;
	}

	if (data->PushData->List)
	{
		AUI_wap_push_list_destroy();
	}
	
	AUI_wap_memory_free((void *)data->PushData, sizeof(T_WAP_PUSH_DATA));

	return;
}


/*******************************************************************************

 $Function:    	AUI_read_PUSH_table

 $Description:	Reads the table from the FFS
 //xrashmic 9 May, 2005 MMI-SPR-30159
 
 $Returns:		

 $Arguments:	event - The event received
 
*******************************************************************************/
void AUI_read_PUSH_table(void)
{
    T_FFS_OPEN_FLAGS Ffsflags;
    fd_t FfsFd;

    Ffsflags = FFS_O_RDONLY;
    FfsFd = ffs_open ("/mmi/concatwappush/table", Ffsflags);
    if(FfsFd > 0)
    {
        ffs_read(FfsFd,(void*)PUSH_Table,sizeof(PUSH_Table)); 
        ffs_close(FfsFd);
    }
    else
        TRACE_EVENT("Table open failed");
}

/*******************************************************************************

 $Function:    	AUI_store_PUSH_table

  $Description: Stores the table into FFS
//xrashmic 9 May, 2005 MMI-SPR-30159
 
 $Returns:		

 $Arguments:	event - The event received
 
*******************************************************************************/
void AUI_store_PUSH_table(void)
{
    T_FFS_OPEN_FLAGS Ffsflags;
    fd_t FfsFd;

    Ffsflags = FFS_O_WRONLY|FFS_O_CREATE;
    FfsFd = ffs_open("/mmi/concatwappush/table",Ffsflags);
    if(FfsFd > 0)
    {
        ffs_write(FfsFd,(void*)PUSH_Table,sizeof(PUSH_Table));
        ffs_close(FfsFd);
    }
    else
        TRACE_ERROR("Table write failed");
}

/*******************************************************************************

 $Function:    	AUI_ffs_PUSH_init

 $Description:	Creates the concat push directory and initializes the push table 
                     and creates a new push table file
                     //xrashmic 9 May, 2005 MMI-SPR-30159
 
 $Returns:		

 $Arguments:	event - The event received
 
*******************************************************************************/
void AUI_ffs_PUSH_init(void)
{
    T_FFS_RET ffsResult;

    ffsResult = ffs_mkdir("/mmi/concatwappush");
    if(EFFS_OK == ffsResult)
    {
        //Initialize the table
        memset(PUSH_Table,0,sizeof(PUSH_Table));
        //write this to FFS
        AUI_store_PUSH_table();
    }
    else
        TRACE_EVENT("Create Dir Failed");    
}

/*******************************************************************************

 $Function:    	AUI_check_concat_msg

 $Description:	Checks for concat push msg
 //xrashmic 9 May, 2005 MMI-SPR-30159
 
 $Returns:  TRUE - if it is a concat push message (also the msg ref, seq num and number of msgs are populates in this case)
                 FALSE - if it is not a concat push message

 $Arguments:	UdhData - UDH data from the sms push message received
                     UdhLength - UDH length
                     msg_ref   - Msg ref of the rcvd sms push message
                     num_msgs - total number of msgs in the concat rcvd sms push message
                     seq_num - The sequence number of the push message
 
*******************************************************************************/
BOOL AUI_check_concat_msg( UBYTE* UdhData, UBYTE UdhLength,U16 *msg_ref,U8 *num_msgs,U8 *seq_num)
{
    S32 	offset = 0;
    U8 		IE_ID;
    U8 		IE_length;
    BOOL           concatSMS = FALSE;
    *msg_ref = 0xffff;
    *num_msgs = 0;
    *seq_num = 0;
    while(offset<UdhLength)
    {
        IE_ID = UdhData[offset++];
        IE_length = UdhData[offset++];
        switch(IE_ID)
        {
            case SMS_IEI_CONC_8BIT:	/* 8-bit message reference number */
                *msg_ref = UdhData[offset+0];
                *num_msgs = UdhData[offset+1];
                *seq_num = UdhData[offset+2];
                concatSMS = TRUE;
            break;

            case SMS_IEI_CONC_16BIT:	/* 16-bit message reference number */
                *msg_ref = UdhData[offset+0];
                *msg_ref|= (UdhData[offset+1] << 8);
                *num_msgs = UdhData[offset+2];
                *seq_num = UdhData[offset+3];
                concatSMS = TRUE;
            break;
        }
        if(TRUE == concatSMS)
            break;
        offset+=IE_length;
    }
    TRACE_EVENT_P3("Concat Push Msg Ref-%d, Seq-%d, Total-%d",*msg_ref,*seq_num,*num_msgs);
    return concatSMS;
}

/*******************************************************************************

 $Function:    	AUI_delete_PUSH_msg

 $Description:	Delets the push msg file from the FFS
 //xrashmic 9 May, 2005 MMI-SPR-30159
 
 $Returns:		

 $Arguments:	Index: index of the messgae to be deleted
 
*******************************************************************************/
void AUI_delete_PUSH_msg(U8 index)
{
    char filename[40]; 

    sprintf(filename,"/mmi/concatwappush/push%d",index); 
    ffs_remove(filename);

    //Removing the corresponding entry from the table
    memset(&PUSH_Table[index],0,sizeof(AUI_PUSH_REPOSITORY_TABLE));
    AUI_store_PUSH_table();
}

/*******************************************************************************

 $Function:    	AUI_cleanup_PUSH_repository

 $Description:	 Deletes the old push msg from the FFS
 //xrashmic 9 May, 2005 MMI-SPR-30159
 
 $Returns:		

 $Arguments:	msg_ref: Msg ref of the rcvd msgs
 
*******************************************************************************/
void AUI_cleanup_PUSH_repository(U16 msg_ref)
{
    U8 index;
    T_MFW_DATE *currDate;
    BOOL deletePUSHMsg;
   
    currDate = mfw_td_get_date();
    
    //Traverse throught the PUSH table and delete old push msg parts
    for(index = 0; index < MAX_PUSH_MSG_STORAGE ; index++)
    {
        deletePUSHMsg = FALSE;
        if((MSG_FFS == (int)PUSH_Table[index].msg_saved) &&( PUSH_Table[index].msg_ref != msg_ref))
        {
            //Need to take only the last two digits for the year
            if ( (currDate->year%100) > (USHORT)PUSH_Table[index].year)
            {
                //very old one delete it
                deletePUSHMsg = TRUE;
            }
             //Need to take only the last two digits for the year
            else  if ((currDate->year%100) == (USHORT)PUSH_Table[index].year)
            {
                //msg of same year check for the month
                if ((USHORT)currDate->month >(USHORT) PUSH_Table[index].month)                       
                {
                   //older than one month delete it
                    deletePUSHMsg = TRUE;
                }
                else  if ((USHORT)currDate->month == (USHORT) PUSH_Table[index].month)
                {
                    //older than MAX_DURATION, delete it
                    if ((USHORT)currDate->day - (USHORT)PUSH_Table[index].day>MAX_DURATION)
                        deletePUSHMsg = TRUE;                                
                }
            }
        }            
        if(TRUE == deletePUSHMsg)
        {
           AUI_delete_PUSH_msg(index);
        }
    }
}

/*******************************************************************************

 $Function:    	AUI_check_repository

 $Description:	Checks the repository for the complete PUSH msg
 //xrashmic 9 May, 2005 MMI-SPR-30159
 
 $Returns:		

 $Arguments:	msg_ref: Msg ref of the rcvd msg
                     num_msgs: total num of parts for the concat msgs
 
*******************************************************************************/
//xrashmic 11 Jan, 2005 MMI-SPR-OMAPS00063638
//Added the seq number in the parameter list of this function to check for duplicate push messages
PUSHMSG_SEARCH_RESULT AUI_check_repository(U16 msg_ref,U8 num_msgs,U8 seq_num)
{
    U8 index,parts = 1;
    for( index = 0; index < MAX_PUSH_MSG_STORAGE ; index++ )
    {
        if((MSG_FFS == (int)PUSH_Table[index].msg_saved ) &&( PUSH_Table[index].msg_ref == msg_ref) )
        {
		PUSH_Sorted_Parts[PUSH_Table[index].seq_num-1].storage = (U8)MSG_FFS;
		//Copy the FFS file index 
		PUSH_Sorted_Parts[PUSH_Table[index].seq_num-1].index = index;
		//xrashmic 11 Jan, 2005 MMI-SPR-OMAPS00063638
		//If the same seq number is already present in the repository, then this is a 
		//duplicate push message and needs to be discarded.
		if((PUSH_Table[index].seq_num ==seq_num))
		{
			return DUPLICATE;
		}
		else
			parts++;
        }            
        //All parts found
        if(parts == num_msgs)
            return FOUND;
    }
    return NOT_FOUND;    
}

/*******************************************************************************

 $Function:    	AUI_store_PUSH_msg

 $Description:	Stores the new PUSH msg into FFS
 //xrashmic 9 May, 2005 MMI-SPR-30159
 
 $Returns:		

 $Arguments:	message: Structure containing the udh and other msg details
                     msg_ref: Msg ref of the rcvd msg
                     num_msgs: total num of parts for the concat msgs
                     seq_num: Seq number of the rcvd msg
 
*******************************************************************************/
void AUI_store_PUSH_msg(T_MFW_SMS_MT *message, U16 msg_ref,U8 num_msgs,U8 seq_num)
{
    int index=0;
    char filename[40];
    T_FFS_OPEN_FLAGS Ffsflags;
    fd_t FfsFd;
       
    //Locate a free space
    for(index = 0 ; index < MAX_PUSH_MSG_STORAGE; index ++)
    {
        if(MSG_NONE == (int)PUSH_Table[index].msg_saved)
        {
            //Free space found 
            break;
        }
    }
    //If space available in the reposiroty store it.
    if( index < MAX_PUSH_MSG_STORAGE)
    {
        //Store the new msg part details into the table
        PUSH_Table[index].msg_saved = (U8)MSG_FFS;
        PUSH_Table[index].msg_ref = msg_ref;
        PUSH_Table[index].num_msgs = num_msgs;
        PUSH_Table[index].seq_num = seq_num;
        PUSH_Table[index].year = message->sctp.year[0]*10+message->sctp.year[1];
        PUSH_Table[index].month = message->sctp.month[0]*10+message->sctp.month[1];
        PUSH_Table[index].day = message->sctp.day[0]*10+message->sctp.day[1];
        AUI_store_PUSH_table();

        //Save the msg part into a new file
        sprintf(filename,"/mmi/concatwappush/push%d",index); 
        Ffsflags = FFS_O_WRONLY|FFS_O_CREATE;
        FfsFd = ffs_open (filename, Ffsflags);
        if(FfsFd > 0)
        {
            ffs_write(FfsFd,(void*)message,sizeof(T_MFW_SMS_MT));
            ffs_close(FfsFd);
        }
        else
            TRACE_EVENT("File open failed");
    }
}

/*******************************************************************************

 $Function:    	AUI_wap_mfw_cb

 $Description:	MFW callbacks
 				SPR#2086 - SH - Added
 
 $Returns:		

 $Arguments:	event - The event received
 
*******************************************************************************/
static int AUI_wap_mfw_cb(MfwEvt event, MfwHnd hnd)
{
	T_WAP_DATA *data = AUI_wap_data();
	T_MFW_SMS_MT *Message;
	BOOL concatSMS = FALSE;
	//xrashmic 11 Jan, 2005 MMI-SPR-OMAPS00063638
	PUSHMSG_SEARCH_RESULT completeMsg = NOT_FOUND;
       T_FFS_RET ffsResult;
       U16 msg_ref;
       U8 num_msgs, seq_num;
       T_FFS_OPEN_FLAGS Ffsflags;
       char filename[40];

	
#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_mfw_cb()");
#endif

	if (!data)
	{
		return;
	}
	
	switch(event)
	{
	    case E_SMS_WAP_MT_RECEIVED:
           {    
                Message = (T_MFW_SMS_MT *) hnd;
                concatSMS = AUI_check_concat_msg(Message->udh,Message->udh_len, &msg_ref, &num_msgs, &seq_num); 
                if(concatSMS)
                {  
                    T_FFS_DIR  dir;
                    ffsResult = ffs_opendir("/mmi/concatwappush",&dir);                    
                    if(EFFS_NOTFOUND == ffsResult)
                        //Directory does not exsists, create it and initialize the table and write it to FFS
                        AUI_ffs_PUSH_init();
                    else
                        //Read the table from ffs.
                        AUI_read_PUSH_table();

                    //Delete the old PUSH messages in FFS and update repository table
                    AUI_cleanup_PUSH_repository(msg_ref);

                    memset(PUSH_Sorted_Parts,0,sizeof(PUSH_Sorted_Parts));                    
                    //update the sorted Array with the new msg index
                    PUSH_Sorted_Parts[seq_num-1].storage = (U8)MSG_NEW;

                    //check for the complete message in repository and the new one. 
                    //If present sort and update the index in the PUSH_Sorted_Parts.
                    completeMsg=AUI_check_repository( msg_ref,num_msgs,seq_num);
                    
                    if(FOUND == completeMsg)
                    {
                        T_MFW_SMS_MT PUSH_Message;
                        U8 part = 0;
                        TRACE_EVENT("Complete Push msg found");
                        for( part = 0; part < num_msgs ; part++ )                    
                        {
                            //retrieve the complete message one by one in the correct order and send to wap client
                            switch( PUSH_Sorted_Parts[part].storage)                                
                            {
                                case MSG_NEW:
                                    //copy new msg from hnd into the comon buffer
                                    memcpy(&PUSH_Message,Message,sizeof(T_MFW_SMS_MT));
                                    break;
                                case MSG_FFS:
                                    {
                                        //copy the stored msg from the ffs into the common buffer
                                        fd_t FfsFd;
                                        sprintf(filename,"/mmi/concatwappush/push%d",PUSH_Sorted_Parts[part].index); 
                                        Ffsflags = FFS_O_RDWR;
                                        FfsFd = ffs_open (filename, Ffsflags);
                                        ffs_read(FfsFd,(void*)&PUSH_Message,sizeof(T_MFW_SMS_MT)); 
                                        ffs_close(FfsFd);
                                        AUI_delete_PUSH_msg(PUSH_Sorted_Parts[part].index);                                    
                                        break;
                                    }
                            }
                            //Send the msg part in the common buffer to the wap client
                            ATB_wap_push_SMS_received(
                                    (char *)PUSH_Message.sc_addr,
                                    (UBYTE)MFW_TON_INTERNATIONAL,
                                    (char *)PUSH_Message.orig_addr.number,
                                    (UBYTE)PUSH_Message.orig_addr.ton,
                                    (char *)PUSH_Message.udh,
                                    (ULONG)PUSH_Message.udh_len,
                                    (char *)PUSH_Message.sms_msg,
                                    (ULONG)PUSH_Message.msg_len);
                        
                        }
                    }
			//xrashmic 11 Jan, 2005 MMI-SPR-OMAPS00063638
			//Store the new part into the ffs
                    else if(completeMsg == NOT_FOUND)
                    {
                        TRACE_EVENT("Storing the Push msg part to FFS");
                        //store this in push repository and update repository table
                        AUI_store_PUSH_msg(Message, msg_ref, num_msgs, seq_num);
                    }
			else
			{
				//xrashmic 11 Jan, 2005 MMI-SPR-OMAPS00063638
				//Duplicate part received , ignore/discard it.
				TRACE_EVENT("Duplicate part received , ignore/discard it");
			}
                }
                else
                {
                    //Not a concat, so send it directly
			ATB_wap_push_SMS_received(
				(char *)Message->sc_addr,
				(UBYTE)MFW_TON_INTERNATIONAL,
				(char *)Message->orig_addr.number,
				(UBYTE)Message->orig_addr.ton,
				(char *)Message->udh,
				(ULONG)Message->udh_len,
				(char *)Message->sms_msg,
				(ULONG)Message->msg_len);
                 }
		break;
	    }
	}
	return;
}

/*******************************************************************************

 $Function:    	AUI_wap_push_show

 $Description:	Push SI viewing screen
 				SPR#2086 - SH - Added
 
 $Returns:		void

 $Arguments:	Text		- The text to be displayed
 				text_length	- The length of the text to be displayed in Unicode chars
 				identifier	- The ID of the SI
 				Url			- The URL of the service
 				url_length	- The length of the URL in characters
 
*******************************************************************************/
#if 0
void AUI_wap_push_show(USHORT *Text, USHORT text_length, SHORT identifier, char *Url, USHORT url_length)
{
	T_WAP_DATA 		*data	= AUI_wap_data();
	T_EDITOR_DATA editor_data;

#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_show");
#endif

	if (!data)
	{
		return;
	}

	if (!data->PushData)
	{
		return;
	}

	strncpy(data->Buffer, Url, url_length);	/* Store URL here for now */
	data->Buffer[url_length] = 0;

	data->PushData->service_indication = identifier;

	AUI_wap_string_alloc(&data->Message, TEXT_MAX_LEN);

	if (text_length>=TEXT_MAX_LEN)
	{
		text_length = TEXT_MAX_LEN-1;
	}
	
	data->Message.text[0] = 0x7F80;
	ATB_uc_text_copy(&data->Message.text[1], Text, text_length);
	ATB_uc_text_convert(&data->Message.text[1], text_length);
	
	AUI_standard_editor(&editor_data, (char *)data->Message.text, text_length+1, READ_ONLY_MODE, (T_EDIT_CB)AUI_wap_push_show_cb); // Setup standard editor
	
	editor_data.destroyEditor = TRUE;
	editor_data.TextId = TxtMessage;									// Prompt to display above editor	
	editor_data.LeftSoftKey = TxtGoTo;
	editor_data.RightSoftKey = TxtExit;
	data->edit_win = editor_start(data->parent_win,&editor_data); /* SPR#2086 */

	return;
}
#endif

void AUI_wap_push_show(USHORT *Text, USHORT text_length, SHORT identifier, char *Url, USHORT url_length)
{
	T_WAP_DATA 		*data	= AUI_wap_data();
#ifdef NEW_EDITOR
        T_AUI_EDITOR_DATA   editor_data;
#else
        T_EDITOR_DATA   editor_data;
#endif /* NEW_EDITOR */

#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_show");
#endif

	if (!data)
	{
		return;
	}

	if (!data->PushData)
	{
		return;
	}

	strncpy(data->Buffer, Url, url_length);	/* Store URL here for now */
	data->Buffer[url_length] = 0;

	data->PushData->service_indication = identifier;

	AUI_wap_string_alloc(&data->Message, TEXT_MAX_LEN);

	if (text_length>=TEXT_MAX_LEN)
	{
		text_length = TEXT_MAX_LEN-1;
	}
	
	data->Message.text[0] = 0x7F80;
	ATB_uc_text_copy(&data->Message.text[1], Text, text_length);
	ATB_uc_text_convert(&data->Message.text[1], text_length);
    #if 0
	AUI_standard_editor(&editor_data, (char *)data->Message.text, text_length+1, READ_ONLY_MODE, (T_EDIT_CB)AUI_wap_push_show_cb); // Setup standard editor
	
	editor_data.destroyEditor = TRUE;
	editor_data.TextId = TxtMessage;									// Prompt to display above editor	
	editor_data.LeftSoftKey = TxtGoTo;
	editor_data.RightSoftKey = TxtExit;
	data->edit_win = editor_start(data->parent_win,&editor_data); /* SPR#2086 */
    #endif
    
#ifdef NEW_EDITOR
        // xreddymn Jan-10-2005 MMI-SPR-27618: Modified editor to read-only
        AUI_standard_editor(&editor_data, WAP_EDIT_DOWNLOAD, ED_MODE_ALPHA | ED_MODE_READONLY, TxtPushMessages, (T_AUI_EDIT_CB)AUI_wap_push_show_cb);
        editor_data.RightSoftKey = TxtSoftBack;

       // if (unicode)
       /* xreddymn Jan-10-2005 MMI-SPR-27618:
        * data->Buffer holding the URL in this case is ASCII. Changed condition if(TRUE) to if(FALSE)
        */
       if (FALSE)
        {
            AUI_edit_SetBuffer(&editor_data, ATB_DCS_UNICODE, (UBYTE *)data->Buffer, strlen(data->Buffer)+1);
        }
        else
        {
            AUI_edit_SetBuffer(&editor_data, ATB_DCS_ASCII, (UBYTE *)data->Buffer, strlen(data->Buffer)+1);
        }
        
        //data->edit_win = AUI_edit_Start(data->win, &editor_data);
        data->edit_win = AUI_edit_Start(data->parent_win, &editor_data);
#else /* NEW_EDITOR */
        AUI_standard_editor(&editor_data, (char *)data->Message.text, text_length+1, READ_ONLY_MODE, (T_EDIT_CB)AUI_wap_push_show_cb); // Setup standard editor
        
        editor_data.destroyEditor = TRUE;
        editor_data.TextId = TxtMessage;                                    // Prompt to display above editor   
        editor_data.LeftSoftKey = TxtGoTo;
        editor_data.RightSoftKey = TxtExit;
        data->edit_win = editor_start(data->parent_win,&editor_data); /* SPR#2086 */
#endif /* NEW_EDITOR */
	return;
}

/*******************************************************************************

 $Function:    	AUI_wap_push_show_cb

 $Description:	Callback function for push viewing screen.
 				SPR#2086 - SH - Added
 
 $Returns:		void

 $Arguments:	win			- the WAP data window
 				identifier	- identifier
 				reason		- reason for callback
 
*******************************************************************************/

static void AUI_wap_push_show_cb(T_MFW_HND win, USHORT identifier, USHORT reason)
{
	T_WAP_DATA 		*data	= AUI_wap_data();
	T_WAP_VIEW		*View;
	
#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_show_cb");
#endif

	if (!data)
	{
		return;
	}

	/* Editor is destroyed automatically */
	
	// xreddymn Jan-10-2005 MMI-SPR-27618: Modified to add "back" function to this screen 
	// data->edit_win = NULL;

	/* Free message text */

	AUI_wap_string_free(&data->Message);
	
	switch (reason)
	{
		case INFO_KCD_LEFT:

			// xreddymn Jan-10-2005 MMI-SPR-27618: Leave editor screen when HangUp or Right Softkey is pressed
			AUI_destroy(EDIT_WIN);
		
			/* Check if browsing view exists.  If it doesn't, start WAP for
			 * browsing and load SI on callback */
			 
			View = ATB_wap_get_view(WAP_OBJECT_ID);

			if (!View)
			{
				AUI_wap_start(WAP_BROWSER, AUI_wap_push_start_load_SI);
			}
			else
			{
				AUI_wap_push_start_load_SI(TRUE);
			}
			break;

		// xreddymn Jan-10-2005 MMI-SPR-27618: Leave editor screen when HangUp or Right Softkey is pressed
		case INFO_KCD_HUP:
		case INFO_KCD_RIGHT:
			AUI_destroy(EDIT_WIN);
			break;
	}
	
	return;
}

/*******************************************************************************

 $Function:    	AUI_wap_push_load_SL

 $Description:  Loads SL message by launching WAP
 
 $Returns:      void

 $Arguments:    identifier - an integer used to identify the SL message
                Url - URL to which the WAP browser must be launched into
                url_length - length of the URL in characters

                xreddymn Mar-16-2005 MMI-SPR-29767

*******************************************************************************/

void AUI_wap_push_load_SL(SHORT identifier, char *Url, USHORT url_length)
{
	T_WAP_VIEW		*View;
	T_WAP_DATA 		*data = AUI_wap_data();

#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_load_SL");
#endif

	if (!data)
	{
		return;
	}

	if (!data->PushData)
	{
		return;
	}

	// The URL is stored in data->Buffer and accessed by 
	// AUI_wap_push_start_load_SI
	strncpy(data->Buffer, Url, url_length);
	data->Buffer[url_length] = 0;

	data->PushData->service_indication = identifier;

	// Check if browsing view exists.  If it doesn't, start WAP for
	// browsing and load SL on callback.

	View = ATB_wap_get_view(WAP_OBJECT_ID);

	if (!View)
	{
		AUI_wap_start(WAP_BROWSER, AUI_wap_push_start_load_SI);
	}
	else
	{
		AUI_wap_push_start_load_SI(TRUE);
	}
}

/*******************************************************************************

 $Function:    	AUI_wap_push_start_load_SI

 $Description:	Callback function for starting push and loading an SI
 				SPR#2086 - SH - Added
 
 $Returns:		void

 $Arguments:	result - TRUE if WAP started successfully.
 
*******************************************************************************/

static void AUI_wap_push_start_load_SI(UBYTE result)
{
	T_WAP_DATA 		*data	= AUI_wap_data();
	
#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_start_load_SI");
#endif

	if (!data || !result)
	{
		return;
	}

	/* Create menu */

	if (!data->menu_win)
	{
		AUI_menu_create(TRUE);
	}
	
	/* Configure for connection */
	
	AUI_prepare_to_connect();

	/* Copy URL into View storage */

	strcpy(data->View->URL, data->Buffer);

	/* Load Service Indication */
	
	ATB_wap_push_load_SI(data->View, (SHORT)data->PushData->service_indication);

	return;
}


/*******************************************************************************

 $Function:    	AUI_wap_push_list

 $Description:	Lists all push messages
 
 $Returns:		MFW_EVENT_CONSUMED

 $Arguments:	menu 	- pointer to current menu
 				item 	- pointer to current menu item
 
*******************************************************************************/

int AUI_wap_push_list(MfwMnu* menu, MfwMnuItem* item)
{
	T_WAP_DATA *data = AUI_wap_data();
	USHORT entryIndex;

#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_list");
#endif

	if (!data)
	{
		return;
	}

	if (!data->PushData)
	{
		TRACE_EVENT("***ERROR - Push not initialised ***");
		return;
	}
	
	/* Create the push list menu */

	if (data->PushData->List==NULL)
	{
		// xreddymn Jan-10-2005 MMI-SPR-27618: Use WAP_PUSH_TITLE_MAX_LEN instead of CARD_TITLE_MAX_LEN
		data->PushData->List = ATB_wap_entry_list_create(WAP_PUSH_LIST, WAP_PUSH_MAX_MESSAGES, WAP_PUSH_TITLE_MAX_LEN, FALSE);
		for (entryIndex=0; entryIndex<WAP_PUSH_MAX_MESSAGES; entryIndex++)
		{
			data->PushData->List->Entry[entryIndex] = (char *)AUI_wap_memory_alloc(data->PushData->List->entry_size);
		}
	}

	data->PushData->list_created = FALSE;
	memset(data->PushData->list_ids, 0, WAP_PUSH_MAX_MESSAGES);

	AUI_please_wait(data->parent_win);
		
	ATB_wap_push_get_SI_info(-1, WAP_PUSH_SHOW_ALL);

	return MFW_EVENT_CONSUMED;
}


/*******************************************************************************

 $Function:    	AUI_wap_push_list_display

 $Description:	The list of pushed messages is ready; display it
 
 $Returns:		None

 $Arguments:	None
 
*******************************************************************************/

void AUI_wap_push_list_display()
{
	T_WAP_DATA *data = AUI_wap_data();

#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_list_display");
#endif

	if (!data)
	{
		return;
	}
	
	AUI_entry_list(data->PushData->List, (ListCbFunc) AUI_wap_push_list_cb, 0);
	if (!data->list_win || data->PushData->List->no_of_entries==0)
	{
		AUI_wap_push_list_destroy();
	}
	else
	{
		data->PushData->list_created = TRUE;
	}

	return;
}


/*******************************************************************************

 $Function:    	AUI_wap_push_list_destroy

 $Description:	Destroy the push message list
 
 $Returns:		None

 $Arguments:	None
 
*******************************************************************************/

void AUI_wap_push_list_destroy()
{
	T_WAP_DATA *data = AUI_wap_data();
	USHORT entryIndex;

#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_list_destroy");
#endif

	if (!data)
	{
		return;
	}

	if (!data->PushData)
	{
		return;
	}
	
	if (!data->PushData->List)
	{
		return;
	}

	/* Create the push list menu */

	for (entryIndex=0; entryIndex<WAP_PUSH_MAX_MESSAGES; entryIndex++)
	{
		AUI_wap_memory_free((void *)data->PushData->List->Entry[entryIndex], data->PushData->List->entry_size);
	}
	ATB_wap_entry_list_destroy(data->PushData->List);
	
	data->PushData->List = NULL;
	data->PushData->list_created = FALSE;

	return;
}


/*******************************************************************************

 $Function:    	AUI_wap_push_list_cb

 $Description:	Callback for push message list
 
 $Returns:		MFW_EVENT_CONSUMED

 $Arguments:	win - The current window
 				ListMenuData - The current list data
 
*******************************************************************************/

static void AUI_wap_push_list_cb(T_MFW_HND win, ListMenuData *EntryListData)
{
	T_WAP_DATA *data = AUI_wap_data();
	
	TRACE_FUNCTION("AUI_wap_push_list_cb()");

	if (!data)
	{
		return;
	}

	data->PushData->service_indication = data->PushData->list_ids[EntryListData->ListPosition];

	switch(EntryListData->Reason)
	{
		case LISTS_REASON_SEND:
			ATB_wap_push_get_SI_info(data->PushData->service_indication, 0);
			break;

		case LISTS_REASON_SELECT:
			/*  Destroy any previous options menu existing */
			if (data->options_win)
			{
				AUI_destroy(OPTIONS_WIN);
			}
            //#ifdef MMI_WAP_PUSH_ENABLED //added by ellen
			//data->options_win = bookMenuStart(data->parent_win, WAPPushListOptionsAttributes(), 0);
			data->options_win = bookMenuStart(data->parent_win, WAPPushListOptionsAttributes(), 0);
            //#endif
			SEND_EVENT(data->options_win, ADD_CALLBACK, NULL, (void *)AUI_options_cb);
			break;

		case LISTS_REASON_BACK:
			AUI_wap_push_list_destroy();
			AUI_destroy(LIST_WIN);
			break;
	}

	return;
}


/*******************************************************************************

 $Function:    	AUI_wap_push_info

 $Description:	Response from WAP Browser, returning information about a push message
 
 $Returns:		MFW_EVENT_CONSUMED

 $Arguments:	menu 	- pointer to current menu
 				item 	- pointer to current menu item
 
*******************************************************************************/

void AUI_wap_push_info(
	USHORT *Text,
	USHORT text_length,
	SHORT id,
	char *Url,
	USHORT url_length,
	UBYTE status,
	ULONG created,
	BOOL expired,
	UBYTE priority)
{
	T_WAP_DATA *data = AUI_wap_data();
	char *newURL;

	TRACE_FUNCTION("AUI_wap_push_info()");
	
	if (!data)
	{
		return;
	}

	if (!data->PushData)
	{
		return;
	}
	
	if (!data->PushData->List)
	{
		return;
	}

	/* If list is created and we're reading a single SI */

	if (data->PushData->list_created)
	{
		AUI_please_wait_destroy();
		AUI_wap_push_show(Text, text_length, id, Url, url_length);
		return;
	}
	
	/* If all of list has been sent, create the menu*/
	
	if (id==-1)
	{
		AUI_please_wait_destroy();
		AUI_wap_push_list_display();
	}

	else
	/* Otherwise, add the menu item to the list */
	{
		data->PushData->list_ids[data->PushData->List->no_of_entries] = id;
		
		newURL = (char *)AUI_wap_memory_alloc(url_length+1);
		strncpy(newURL, Url, url_length);
		newURL[url_length] = 0;
		
		if (url_length>data->PushData->List->entry_size)
		{
			newURL[data->PushData->List->entry_size] = 0;
			newURL[data->PushData->List->entry_size-1] = '.';
			newURL[data->PushData->List->entry_size-2] = '.';
		}
		
		ATB_wap_entry_add(data->PushData->List, newURL);

		AUI_wap_memory_free((void *)newURL, url_length+1);
	}

	return;
}

/*******************************************************************************

 $Function:    	AUI_wap_push_message_read

 $Description:	Read a push message
 				SPR#2086 - Added
 
 $Returns:		MFW_EVENT_CONSUMED

 $Arguments:	menu 	- pointer to current menu
 				item 	- pointer to current menu item
 
*******************************************************************************/

int AUI_wap_push_message_read(MfwMnu* menu, MfwMnuItem* item)
{
	T_WAP_DATA *data = AUI_wap_data();
	
	TRACE_FUNCTION("AUI_wap_push_message_read()");
	
	if (!data)
	{
		return;
	}

	if (!data->PushData)
	{
		return;
	}
	
	ATB_wap_push_get_SI_info(data->PushData->service_indication, 0);

	return MFW_EVENT_CONSUMED;
}


/*******************************************************************************

 $Function:    	AUI_wap_push_message_delete

 $Description:	Delete a push message
 				SPR#2086 - Added
 
 $Returns:		MFW_EVENT_CONSUMED

 $Arguments:	menu 	- pointer to current menu
 				item 	- pointer to current menu item
 
*******************************************************************************/

int AUI_wap_push_message_delete(MfwMnu* menu, MfwMnuItem* item)
{
	T_WAP_DATA *data = AUI_wap_data();
	
	TRACE_FUNCTION("AUI_wap_push_message_delete()");
	
	if (!data)
	{
		return;
	}

	/* Delete the push message */	
	ATB_wap_push_delete_SI(data->PushData->service_indication, 0);
	/* Destroy the old list */
	AUI_destroy(LIST_WIN);
	/* Save the change */
	ATB_wap_push_save_to_flash();
	/* Recreate the list */
	AUI_list_redraw(WAP_PUSH_LIST, TxtDeleted);

	return MFW_EVENT_CONSUMED;
}


/*******************************************************************************

 $Function:    	AUI_wap_push_showmessages

 $Description:	Setting specifies whether or not to display received push images
 				SPR#2086 - Added
 
 $Returns:		MFW_EVENT_CONSUMED

 $Arguments:	menu 	- pointer to current menu
 				item 	- pointer to current menu item
 
*******************************************************************************/

int AUI_wap_push_showmessages(MfwMnu* menu, MfwMnuItem* item)
{
	T_WAP_DATA	*data = AUI_wap_data();
	UBYTE PushMessages;
	T_MFW_HND setting_win;
	
#ifdef TRACE_AUIWAP	
	TRACE_FUNCTION("AUI_wap_push_showmessages");
#endif

	if (!data)
	{
		return;
	}

	setting_win = bookMenuStart(data->parent_win, WAPPushMessagesAttributes(), 0);
	PushMessages = ATB_wap_profile_setting(WAP_STATUS_PUSHMESSAGES);	// 1 or 0 depending on flag
	SEND_EVENT(setting_win, DEFAULT_OPTION, NULL, &PushMessages); // Add our own callback

	return 1;
}


/*******************************************************************************

 $Function:    	AUI_wap_push_showmessages_set

 $Description:	Set showing of push messages on or off
 
 $Returns:		MFW_EVENT_CONSUMED

 $Arguments:	menu		- pointer to the current menu
 				item		- pointer to the current menu item
 
*******************************************************************************/

int AUI_wap_push_showmessages_set(MfwMnu* menu, MfwMnuItem* item)
{
	T_WAP_DATA 		*data		= AUI_wap_data();
	
#ifdef TRACE_AUIWAP
	TRACE_FUNCTION("AUI_wap_push_showmessages_set");
#endif

	if (!data)
	{
		return;
	}

	ATB_wap_profile_setting_change(WAP_STATUS_PUSHMESSAGES, menu->lCursor[menu->level]); /* SPR#2086 */
	
	ATB_wap_profile_save(data->View);			// Save change to flash
	AUI_info_dialog(TxtChanged, NULL);  /* SPR#2086 */

	return 1;
}

