/*******************************************************************************

          Texas Instruments

********************************************************************************

 This software product is the property of Texas Instruments and may not be
 disclosed to any third party without the express permission of the owner.

********************************************************************************

 $Project name:	File Viewer
 $Module:   		File Viewer
 $File:       		MmiFileViewer.c
 $Revision:   		1.0
 $Author:   		xdeepadh - sasken India
 $Date:       		11/05/05

********************************************************************************

 Description: This module contains the functionality of File Viewer Application.

********************************************************************************
 $History: MmiFileViewer.c
 	Jan 03, 2006    REF: SPR OMAPS00060036  xdeepadh
 	Description:In file viewer, the user is not allowed to retain the old name once he/she goes to the rename file option.
 	Solution:While renaming the file,if the name already exists,  display a proper error message
 	
 	Aug 31, 2005    REF: SPR 34050 xdeepadh
	Description: The application hangs if while renaming a saved file a special character such as '?', '!' etc is used
	Solution: Error message will be displayed when ffs_rename() fails
	
	 Aug 30, 2005    REF:SPR 34049 xdeepadh
   	Description:  FileViewer: Not able to delete a file that is too big to display
   	Solution: When the file was too big to display, the file handle was closed.

	Aug 22, 2005    REF: ENH 31154 xdeepadh
   	Description: Application to Test Camera
   	Solution: Implemeted the Fileviewer to view the jpeg images.Camera Application to 
   	preview,capture and save image has been implemented.

     

 $End

*******************************************************************************/

#define ENTITY_MFW

/* includes */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>


#if defined (NEW_FRAME)
#include "typedefs.h"
#include "Vsi.h"
#include "PEI.H"
#include "custom.h"
#include "gsm.h"
#include "prim.h"

#else
#include "stddefs.h"
#include "custom.h"
#include "gsm.h"
#include "Vsi.h"

#endif

#include "message.h" /* included for aci_cmh.h .... */
#include "aci_cmh.h" /* included for mfw_sat.h, included for MmiDummy.h, included for MmiMenu.h */

#include "mfw_mfw.h"
#include "mfw_win.h"
#include "mfw_icn.h"	/* included for mfw_mnu.h */
#include "mfw_mnu.h"
#include "mfw_tim.h"
#include "mfw_kbd.h"
#include "mfw_sat.h" /* included for MmiDummy.h, included for MmiMenu.h */
#include "mfw_fv.h" //Will be enabled once the IMGIL APIS are provided.


#include "ffs/ffs_api.h"

#include "MmiBlkLangDB.h"
#include "mmiColours.h"
#include "MmiDialogs.h"
#include "MmiDummy.h" /* included for MmiMenu.h */
#include "MmiLists.h"
#include "MmiMenu.h"
#include "MmiSoftKeys.h"
#include "AUIEditor.h"
#include "MmiFileViewer.h"
#include "MmiMain.h"
#include "Mmibookshared.h"

/*local defines*/

//Events for windows
#define FV_INIT 		1
#define FV_RETURN 	2
#define FV_ABORT  	3
#define FV_EXIT 		4

//Events for editor
#define FV_EDITOR_CANCEL	 5
#define FV_EDITOR_SELECT	 6
#define FV_EDITOR_ERROR         7  //Aug 31, 2005    REF: SPR 34050 xdeepadh

//Ids for diffrent functionalities.
#define FV_OPT_ID		 100
#define FV_EDT_ID 		 101
#define FV_DEL_ID		 102

#define FV_INFO_SCRN_TIMEOUT  		1500
//Aug 31, 2005    REF: SPR 34050 xdeepadh
#define FV_FILE_NAME_LENGTH 		20 
#define FV_DIR_PATH_LENGTH 		(10 + FV_FILE_NAME_LENGTH)
#define FV_DIR_NAME 				"/mmi/jpeg/"
#define FV_MAX_NUM_FILES			10
#define FV_MAX_JPEG_SIZE 			(64  *  1024)


//Variable defines


UINT8 qcif_yuv_lum[FV_DSPL_VOLUME]; 		 //decoded and Scaled  lum buffer
UINT8 qcif_yuv_cb[FV_SCALE_VOLUME];		//decoded and Scaled cb  buffer
UINT8 qcif_yuv_cr[FV_SCALE_VOLUME];		//decoded and Scaled cr buffer

UINT8 qcif_rgb_array[ FV_DSPL_VOLUME  *  2];			//Color converted buffer to view
UINT8 rgb_img_process_array[FV_DSPL_VOLUME  *  2];  	//Color converted buffer for zoom/rotate

UINT8 qcif_yuv_process_lum[FV_DSPL_VOLUME ];		//lum buffer for rotate and zoom
UINT8 qcif_yuv_process_cb[FV_SCALE_VOLUME];		//cb buffer for rotate and zoom
UINT8 qcif_yuv_process_cr[FV_SCALE_VOLUME];		//cr buffer for rotate and zoom


/*menu data structure definition*/
typedef struct
{
  T_MMI_CONTROL     mmi_control;
  T_MFW_HND         	win;
  T_MFW_HND         	kbd;
  T_MFW_HND         	kbd_long;
  T_MFW_HND         	fv_menu;
  T_MFW_HND         	fv_option_win;
  T_MFW_HND         	parent;
  ListMenuData	 	fv_list_data;
  UBYTE				fv_file_count;
  ListCbFunc			fv_list_cb;
  UBYTE				fv_list_idx;
  
}T_fv_menu;

typedef struct
{
	/* administrative data */
	T_MMI_CONTROL   	mmi_control;
	T_MFW_HND       		win;
	SHORT          			 id;
	T_MFW_HND       		parent;
	/* associated handlers */
	T_MFW_HND      		 kbd;
	T_MFW_HND       		kbd_long;
	T_MFW_HND      		 menu;
	T_fv_menu * parent_data;
} T_fv_opt;

/*editor data structure*/
typedef struct{
	T_MMI_CONTROL		mmi_control;
	MfwHnd				win;
	MfwHnd				fv_win_editor;
	UBYTE				fv_buffer[FV_FILE_NAME_LENGTH];
}	T_FV_Editor_Data;


/*default static menu*/
static MfwMnuAttr menuFvSearchListAttr =
{
	&menuArea,
	MNU_LEFT | MNU_LIST | MNU_CUR_LINE,	/* centered page menu */
	-1,										/* use default font  */
	NULL,									/* with these items (filled in later) */
	0,										/* number of items (filled in leter)  */
	COLOUR_LIST_XX,						/* Colour */
	TxtNull,								       /* Hdr Id */
	NULL,									/* Background */
	MNUATTRSPARE							/* unused */
};

static const MfwMnuItem Mmi_fv_OPTItems [] =
{
	{0,0,0,(char *)TxtView,0,(MenuFunc)mmi_fv_view_image_start,item_flag_none},
	{0,0,0,(char *)TxtDelete,0,(MenuFunc)mmi_fv_delete_start,item_flag_none},
	{0,0,0,(char *)TxtRename,0,(MenuFunc)mmi_fv_reame_edit,item_flag_none}
	
};

static const MfwMnuAttr Mmi_fv_OPTAttrib =
{
    &SmsSend_R_OPTArea,
    MNU_LEFT | MNU_LIST | MNU_CUR_LINE,		 /* centered page menu       */
    -1,                                 						/* use default font         */
    (MfwMnuItem*)Mmi_fv_OPTItems,                    		/* with these items         */
    sizeof(Mmi_fv_OPTItems)/sizeof(MfwMnuItem), 	/* number of items     */
    COLOUR_LIST_SMS,	TxtNull, NULL, MNUATTRSPARE

};

//Variable definition
T_fv_menu	fv_menu_data = {0};
//Variable to store the  max jpeg  file count.
int 	file_count = 0;

//Array to store the Jpeg filenames.
UBYTE	jpegFiles[FV_MAX_NUM_FILES][FV_FILE_NAME_LENGTH] = {0};

//Buffer to store the jpeg image from FFS.

static UBYTE jpeg_test_array[FV_MAX_JPEG_SIZE] = {0};

T_MFW_HND fv_optwin;

UBYTE  zoom_flag,rotate_flag = 0;  //Flag to identify the rotate and zooming

//Function prototypes

static int mmi_fv_list_files_win_cb (MfwEvt e, MfwWin *w);
static void mmi_fv_menu_listmnu_cb(T_MFW_HND Parent, ListMenuData * ListData);
void mmi_fv_list_files_exec_cb (T_MFW_HND win, USHORT event, SHORT value, void * parameter);
static T_MFW_HND mmi_fv_list_files_create(T_MFW_HND parent_window, void* callfun);
static T_MFW_HND mmi_fv_test_select_file (T_MFW_HND parent_win);
static int mmi_fv_opt_mnu_cb (MfwEvt e, MfwMnu *m);
static int mmi_fv_opt_kbd_cb (MfwEvt e, MfwKbd *k);
static int mmi_fv_opt_win_cb (MfwEvt e, MfwWin *w);
void mmi_fv_opt_exec_cb (T_MFW_HND win, USHORT event, SHORT value, void * parameter);
static void mmi_fv_delete_cb (T_MFW_HND win, USHORT Identifier, SHORT reason);
static void mmi_rename_editor_cb(T_MFW_HND win, USHORT identifier, SHORT reason);
MfwRes mmi_fv_editor_destroy(T_MFW_HND win);
void mmi_fv_editor_cntrl (T_MFW_HND win, USHORT event, SHORT value, void * parameter);
T_MFW_HND mmi_fv_editor_create(T_MFW_HND parent);
static int mmi_fv_view_kbd_cb (MfwEvt e, MfwKbd *k);
static int mmi_fv_view_win_cb (MfwEvt e, MfwWin w);
static T_MFW_HND mmi_fv_view_image_create(MfwHnd parent_window);
static void mmi_fv_view_exec_cb(T_MFW_HND win, USHORT event, void *parameter);
static int mmi_fv_rotate_win_cb (MfwEvt e, MfwWin w);
static int mmi_fv_rotate_kbd_cb (MfwEvt e, MfwKbd *k);
static void mmi_fv_rotate_exec_cb(T_MFW_HND win, USHORT event, void *parameter);
static int mmi_fv_zoom_win_cb (MfwEvt e, MfwWin w);
static int mmi_fv_zoom_kbd_cb (MfwEvt e, MfwKbd *k);
static void mmi_fv_zoom_exec_cb(T_MFW_HND win, USHORT event, void *parameter);
static MfwHnd mmi_fv_test_show_info(T_MFW_HND parent,int lsk, int rsk, int str1, int str2, T_VOID_FUNC callback);
static int mmi_fv_item_mnu_cb(MfwMnu * mnu, MfwMnuItem * item);
static void mmi_fv_menu_destroy(MfwHnd window);
static T_MFW_HND mmi_fv_opt_create(MfwHnd parent_window);
GLOBAL int  mmi_fv_delete_jpeg_file(UBYTE index);
static T_MFW_HND mmi_fv_rotate_image_create(MfwHnd parent_window);
int mmi_fv_zoom_image_start(T_MFW_HND parent) ;
static T_MFW_HND mmi_fv_zoom_image_create(MfwHnd parent_window);

/*******************************************************************************

 $Function:     mmi_fv_get_file_count

 $Description:	returns the count of jpeg files

 $Returns: The file count of the files in JPG folder.

 $Arguments:None

*******************************************************************************/
UBYTE mmi_fv_get_file_count()
{
	return file_count;
}


/*******************************************************************************

 $Function:     LaunchFileViewer

 $Description:	Initiates the creation of the  jpeg file list window and post init event

 $Returns: Execution status

 $Arguments:		m 	- pointer to current menu
 				i 	- pointer to current menu item

*******************************************************************************/
GLOBAL int LaunchFileViewer( MfwMnu *m, MfwMnuItem *i )
{
	T_MFW_HND win;
	T_MFW_HND parent_win = mfw_parent(mfw_header());

	TRACE_FUNCTION ("LaunchFileViewer()");

	mfw_fv_init();
	win = mmi_fv_list_files_create(parent_win,(void*)mmi_fv_list_files_exec_cb);

	if (win NEQ NULL)
	{
		SEND_EVENT (win, FV_INIT, 0, 0);
	}
	return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:     mmi_fileviewer_list_files_create

 $Description:	Creates the window and dialog handler for the window that 
 			holds the list window

 $Returns:	Window Handler

 $Arguments:parent_window - Parent Window
 			callfun- Callback function

*******************************************************************************/

static T_MFW_HND mmi_fv_list_files_create(T_MFW_HND parent_window, void* callfun)
{
	T_fv_menu *  data = (T_fv_menu *)mfwAlloc(sizeof (T_fv_menu));
	T_MFW_WIN  * win;
	TRACE_FUNCTION ("mmi_fileviewer_list_files_create");

	/* Create window handler */

	data->win = win_create (parent_window, 0, E_WIN_VISIBLE, (T_MFW_CB)mmi_fv_list_files_win_cb);
	if (data->win EQ NULL)
	{
		return NULL;
	}

	TRACE_EVENT("list holder window created: " );
	/* connect the dialog data to the MFW-window */

	data->mmi_control.dialog = (T_DIALOG_FUNC)callfun;
	data->mmi_control.data   = data;
	win                      = ((T_MFW_HDR *)data->win)->data;
	win->user                = (void *)data;
	data->parent             = parent_window;
	winShow(data->win);
	/*
	* return window handle
	*/

	return data->win;
}

/*******************************************************************************

 $Function:     mmi_fv_list_files_exec_cb

 $Description: handler for the window that holds the list window

 $Returns:None

 $Arguments:		win :		The pointer to the window handler, so the function can reference the
							dynamic user data, if any.
 				event :		The incoming event
 				value :		Generic incoming Parameter, it will be 'event' dependant
 				parameter :	Generic incoming Parameter, it will be 'event' dependant

*******************************************************************************/
void mmi_fv_list_files_exec_cb (T_MFW_HND win, USHORT event, SHORT value, void * parameter)
{
    T_MFW_WIN   * win_data = ((T_MFW_HDR *) win)->data;
    T_fv_menu   * data = (T_fv_menu *)win_data->user;
	
    TRACE_FUNCTION ("mmi_fv_list_files_exec_cb");
	
    switch (event)
	{
    case FV_INIT:
		data->fv_menu = mmi_fv_test_select_file(win);
		if(data->fv_menu == NULL)
			mmi_fv_menu_destroy((MfwHnd)data->win);
		break;
    case FV_RETURN:
	default:
		break;
	}
}

/*******************************************************************************

 $Function:     mmi_fv_list_files_win_cb

 $Description: window callback for the window that holds the list window

 $Returns:Status

 $Arguments: 		e		- window handle event
 				w			- window


*******************************************************************************/
static int mmi_fv_list_files_win_cb (MfwEvt e, MfwWin *w)   
{
	TRACE_FUNCTION ("mmi_fv_list_files_win_cb()");
	switch (e)
	{
    case MfwWinVisible:  /* window is visible  */
		break;
    case MfwWinFocussed: /* input focus / selected   */
    case MfwWinDelete:   /* window will be deleted   */
		
    default:
		return MFW_EVENT_REJECTED;
	}
	return MFW_EVENT_CONSUMED;
}


/*******************************************************************************

 $Function:     mmi_fv_test_select_file

 $Description:	browse jpeg files and enable to select them

 $Returns: T_MFW_HND : Window Handle to the New Window, Null if failed.

 $Arguments:		parent : pointer to the parent window.


*******************************************************************************/
static T_MFW_HND mmi_fv_test_select_file (T_MFW_HND parent_win)
{
	int 	j;
	T_fv_menu	*data;
	T_MFW_HND	win  = parent_win;
	T_MFW_WIN	* win_data = ((T_MFW_HDR *) win)->data;
	
	
	TRACE_FUNCTION ("mmi_fv_test_select_file()");
	//Populate the jpeg files from ffs
	mmi_fv_init(&file_count);

	data = (T_fv_menu *)win_data->user;
	if (data->win EQ NULL)
	{
		return NULL;
	}
	//set count to 0
	data->fv_file_count = 0;
	
	//In case no files are present in ffs return info message
		if(file_count == 0)
	{
		mmi_fv_test_show_info(0,TxtNull,TxtNull, TxtFileNotLoaded, TxtNull, NULL);
			return NULL;
	}
	else
	{
		/* Create the list menu to display the results	*/
	    /* initialization of the dialog data */
		data->fv_list_data.ListPosition   = 1;/* True cursor position in list menu. */
		data->fv_list_data.Font           = 0;
		data->fv_list_data.LeftSoftKey    = TxtSoftOptions; 
		data->fv_list_data.RightSoftKey   = TxtSoftBack;
		data->fv_list_data.KeyEvents      = KEY_CLEAR | KEY_RIGHT | KEY_LEFT | KEY_MNUUP | KEY_MNUDOWN |KEY_HUP;
		data->fv_list_data.Reason         = 0;
		data->fv_list_data.Strings        = TRUE;
		data->fv_list_data.autoDestroy    = TRUE;
		data->fv_list_data.Attr           = (MfwMnuAttr *)&menuFvSearchListAttr;
		data->fv_list_data.Attr->hdrId	  = TxtFileViewer;
			data->fv_list_data.Attr->nItems   = file_count;
		data->fv_list_cb 				  = (ListCbFunc)mmi_fv_menu_listmnu_cb;
			data->fv_list_data.ListLength 	  = file_count; /* actual number of entries in list menu.    */
		data->fv_list_data.List 		  = (MfwMnuItem *)mfwAlloc(data->fv_list_data.ListLength * sizeof (MfwMnuItem));
		if (data->fv_list_data.List != (MfwMnuItem *)0)
		{
			MfwMnuItem *listItem;
			
			/* Initialise the memory to 0x00 */
			memset(data->fv_list_data.List, 0x00, data->fv_list_data.ListLength * sizeof (MfwMnuItem));
			
			for (j = 0; j <data->fv_list_data.ListLength; j++)
			{
				listItem 		   	=  &data->fv_list_data.List[j];
				listItem->str 	   	=  (char*)jpegFiles[data->fv_file_count++];
				listItem->func 	   	=  (MenuFunc)mmi_fv_item_mnu_cb;
				listItem->flagFunc 	=  (FlagFunc)item_flag_none;
			}
			
			/* Recreate the menu with the new details */
			if (listDisplayListMenu(data->win, &data->fv_list_data, data->fv_list_cb, FALSE) != LISTS_FAIL)
				return data->fv_list_data.win;
			
		}
			return data->win;
	}
	
}

/*******************************************************************************

 $Function:     mmi_fv_test_show_info

 $Description:	This function displays a information screen with the textId's

 $Returns: Dialog Window

 $Arguments:parent- parent win
 			lsk- Left Softkey
 			rsk-Right Softkey
 			str1-String
 			str2-String
 			callback- Callback function

*******************************************************************************/
static MfwHnd mmi_fv_test_show_info(T_MFW_HND parent,int lsk, int rsk, int str1, int str2, T_VOID_FUNC callback)
{
	T_DISPLAY_DATA display_info;
	
    TRACE_FUNCTION ("mmi_fv_test_show_info()");
	
	/*
	** Create a timed dialog to display the Message
	*/
	dlg_initDisplayData_TextId( &display_info, lsk, rsk, str1, str2, COLOUR_STATUS);
	dlg_initDisplayData_events( &display_info, callback, FV_INFO_SCRN_TIMEOUT, KEY_LEFT | KEY_CLEAR | KEY_HUP);
	
	return info_dialog(parent, &display_info);
}

/*******************************************************************************

 $Function:     mmi_fv_init

 $Description:	Function retrieves the files present under the /JPG directory

 $Returns: None

 $Arguments: Filecount- count of files in FFS.

*******************************************************************************/
void mmi_fv_init(int* fileCount)
{
	
	T_FFS_FD  	fd                   = 0;
	char        dir_name_p[]         = FV_DIR_NAME;
	T_FFS_DIR	dir                  = {0};
	char		dest_name_p[FV_FILE_NAME_LENGTH]     = "";
	int 		i, max_files_count = 0;
	char *fileExt;

	TRACE_FUNCTION("mmi_fv_init()");
	
	i	=	strlen(dir_name_p);
	dir_name_p[i-1] = '\0';
	
	TRACE_EVENT_P1("the dir name = %s",dir_name_p);
	
    if ((max_files_count = ffs_opendir (dir_name_p, &dir)) <= 0)
	{
		return;
	}

	//Display only 10 files
    if(max_files_count > FV_MAX_NUM_FILES) 
	   max_files_count = FV_MAX_NUM_FILES;
	
    *fileCount = max_files_count;
    TRACE_EVENT_P1("Number of files = %d",*fileCount);
	
    //read maximum of 10 file names
	for (i = 0; (ffs_readdir (&dir, dest_name_p, 100) > 0x0) && (i<max_files_count); i++)
	{
		if(i >= FV_MAX_NUM_FILES)
		 break;

		//retrieve file extension
      		fileExt  = mmi_fv_GetExtension(dest_name_p);	
		//Store only jpeg file names in array
		if ( (strcmp(fileExt, "jpg") == 0))
			{
				strcpy ((char*)jpegFiles[i], dest_name_p);
			}
		memset(dest_name_p,0,FV_FILE_NAME_LENGTH);
    	}

	 TRACE_EVENT_P1("Number of files after read = %d",*fileCount);
}

/*******************************************************************************

 $Function:     mmi_fv_item_mnu_cb

 $Description:	save index of currently selected file

 $Returns:

 $Arguments:		m 	- pointer to current menu
 				i 	- pointer to current menu item

*******************************************************************************/
static int  mmi_fv_item_mnu_cb(MfwMnu * mnu, MfwMnuItem * item)
{
	T_MFW_HND win = mfw_parent(mfw_header());
	T_MFW_WIN		* win_data = ((T_MFW_HDR *) win)->data;
	T_fv_menu *data = (T_fv_menu *)win_data->user;
	
	TRACE_FUNCTION ("mmi_fv_item_mnu_cb()");
	if (data EQ NULL)
	{
		return NULL;
	}		
	data->fv_list_idx = fv_menu_data.fv_list_idx = mnu->lCursor[mnu->level];
	data->fv_option_win = mmi_fv_opt_start(data->win,(MfwMnuAttr*)&Mmi_fv_OPTAttrib);
	return MFW_EVENT_CONSUMED;
}


/*******************************************************************************

 $Function:     mmi_fv_menu_listmnu_cb

 $Description:	destroy lists and menu in case of back softkey or menu exit

 $Returns:None 

 $Arguments:parent-
 			Listadata-

*******************************************************************************/
static void mmi_fv_menu_listmnu_cb(T_MFW_HND Parent, ListMenuData * ListData)
{
	T_MFW_WIN	*win_data	= ((T_MFW_HDR *)Parent)->data;
	T_fv_menu	*data		= (T_fv_menu *)win_data->user;
    
	TRACE_FUNCTION ("mmi_fv_menu_listmnu_cb()");
	
	if ((ListData->Reason EQ LISTS_REASON_BACK) ||
		(ListData->Reason EQ LISTS_REASON_CLEAR) ||
		(ListData->Reason EQ LISTS_REASON_HANGUP))
	{
	
		/* Free the memory set aside for ALL the devices to be listed.*/
		mfwFree((U8 *)data->fv_list_data.List, (U16)(ListData->ListLength* sizeof(MfwMnuItem)));
		listsDestroy(ListData->win);
		data->fv_list_data.List = NULL;
		mmi_fv_menu_destroy((MfwHnd)data->win);
	}

	return;
}

/*******************************************************************************

 $Function:     mmi_fv_menu_destroy

 $Description:	destroy menu window

 $Returns:None

  $Arguments:		window : pointer to the parent window.

*******************************************************************************/
static void mmi_fv_menu_destroy(MfwHnd window)
{
	T_MFW_WIN     * win_data  = ((T_MFW_HDR *)window)->data;
	T_fv_menu * data = (T_fv_menu *)win_data->user;
	
	TRACE_FUNCTION ("mmi_fv_menu_destroy()");
	
	if (window == NULL)
	{
		return;
	}
	
	if (data)
	{
		win_delete(data->win);
		/* Free Memory */
		mfwFree((void *)data, sizeof (T_fv_menu));
	}
}


/*******************************************************************************

 $Function:     mmi_fv_set_rotate_90degree

 $Description: Rotates the selected image by 90 degree.
 $Returns: Status

 $Arguments:m 	- pointer to current menu
 				i 	- pointer to current menu item

*******************************************************************************/
GLOBAL int mmi_fv_set_rotate_90degree (MfwMnu* m, MfwMnuItem* i)
{
	T_MFW_HND       win  = mfwParent(mfw_header());
	TRACE_FUNCTION ("mmi_fv_set_rotate_90degree()");
	rotate_flag = FV_ROTATE_90;
	mmi_fv_rotate_image_start(win);
	 return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:     mmi_fv_set_rotate_180degree

 $Description: Rotates the selected image by 180 degree.
 $Returns: Status

 $Arguments:m 	- pointer to current menu
 				i 	- pointer to current menu item

*******************************************************************************/
GLOBAL int mmi_fv_set_rotate_180degree (MfwMnu* m, MfwMnuItem* i)
{
	T_MFW_HND       win  = mfwParent(mfw_header());
	TRACE_FUNCTION ("mmi_fv_set_rotate_180degree()");
	rotate_flag = FV_ROTATE_180;
	mmi_fv_rotate_image_start(win);
	 return MFW_EVENT_CONSUMED;
}


/*******************************************************************************

 $Function:     mmi_fv_set_rotate_270degree

 $Description: Rotates the selected image  anti-clockwise.
 $Returns:

 $Arguments:		m 	- pointer to current menu
 				i 	- pointer to current menu item

*******************************************************************************/
GLOBAL int mmi_fv_set_rotate_270degree (MfwMnu* m, MfwMnuItem* i)
{
	T_MFW_HND       win  = mfwParent(mfw_header());
	TRACE_FUNCTION ("mmi_fv_set_rotate_270degree()");
	rotate_flag = FV_ROTATE_270;
	mmi_fv_rotate_image_start(win);
	return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:     mmi_fv_set_zoom_out

 $Description: Zooms out the selected image
 $Returns:

 $Arguments:		m 	- pointer to current menu
 				i 	- pointer to current menu item

*******************************************************************************/
GLOBAL int mmi_fv_set_zoom_out (MfwMnu* m, MfwMnuItem* i)
{
	T_MFW_HND       win  = mfwParent(mfw_header());
	TRACE_FUNCTION ("mmi_fv_set_zoom_out()");
	zoom_flag = FV_ZOOM_OUT;
	mmi_fv_zoom_image_start(win);
	return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:     mmi_fv_set_zoom_in

 $Description: Zooms in the selected image
 $Returns:

 $Arguments:		m 	- pointer to current menu
 				i 	- pointer to current menu item

*******************************************************************************/
GLOBAL int mmi_fv_set_zoom_in (MfwMnu* m, MfwMnuItem* i)
{
	T_MFW_HND       win  = mfwParent(mfw_header());
	TRACE_FUNCTION ("mmi_fv_set_zoom_in()");
	zoom_flag = FV_ZOOM_IN;
	mmi_fv_zoom_image_start(win);
	return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:    	mmi_fv_view_image_start

 $Description:	Create editor for viewing the image.
 
 $Returns:		none
 
 $Arguments:	m - menu handler
 				i - menu item selected.
*******************************************************************************/


int mmi_fv_view_image_start(MfwMnu* m, MfwMnuItem* i) 
{
	T_MFW_HND win;
	T_MFW_HND parent_win = mfw_parent(mfw_header());
	win = mmi_fv_view_image_create(parent_win);
	TRACE_FUNCTION ("mmi_fv_view_image_start");
	SEND_EVENT(win,FV_INIT,FV_OPT_ID,0);
	return MFW_EVENT_CONSUMED;

}


/*******************************************************************************

 $Function:    	mmi_fv_view_image_create

 $Description:	Create a window for viewing the image.
 
 $Returns:		window handle

 $Arguments:	parent_window - parent window.
 				
*******************************************************************************/

static T_MFW_HND mmi_fv_view_image_create(MfwHnd parent_window)
{
	T_fv_opt* data = (T_fv_opt *)ALLOC_MEMORY (sizeof (T_fv_opt));
	T_MFW_WIN  * win;

	TRACE_FUNCTION ("mmi_fv_view_image_create()");

	// Create window handler
	data->win = win_create (parent_window, 0, E_WIN_VISIBLE,(MfwCb)mmi_fv_view_win_cb);
	if (data->win EQ NULL)
	{
		return NULL;
	}

	// connect the dialog data to the MFW-window
	data->mmi_control.dialog = (T_DIALOG_FUNC)mmi_fv_view_exec_cb;
	data->mmi_control.data   = data;
	win                      = ((T_MFW_HDR *)data->win)->data;
	win->user                = (MfwUserDataPtr)data;
	data->parent             = parent_window;
	//Create keyboard handler
        data->kbd = kbd_create (data->win,KEY_ALL,(T_MFW_CB)mmi_fv_view_kbd_cb);
	
	return data->win;
}


/*******************************************************************************

 $Function:    	mmi_fv_view_exec_cb

 $Description:	Callback handler for events sent by the images editor

 $Returns:		none

 $Arguments:	win - current window
 				event - event id
 				parameter - optional data.
*******************************************************************************/

static void mmi_fv_view_exec_cb(T_MFW_HND win, USHORT event, void *parameter)
{	
	T_MFW_WIN      * win_data = ((T_MFW_HDR *) win)->data;
	T_fv_opt         * data = (T_fv_opt *)win_data->user;
	int status;

	TRACE_FUNCTION ("mmi_fv_view_exec_cb()");

	switch (event)
	{
	case FV_INIT:
		TRACE_EVENT("init");
		status = mmi_fv_read_jpeg_file(fv_menu_data.fv_list_idx);
		
		//Display the decoded jpeg image in QCIF-RGB565 format on LCD.
		if(status  == 1)
			winShow(data->win);
		else
			mmi_fv_test_show_info(0,TxtNull,TxtNull, TxtFileTooBig, TxtNull, NULL);
	break;

	case FV_RETURN:
	break;
	default:
	TRACE_EVENT("Err: Default");
	return;
	}
}


/*******************************************************************************

 $Function:    	mmi_fv_opt_start

 $Description:	called when the user press  the Option menu
 
 $Returns:		execution status

 $Arguments:	parent_window -parent window handle
 				menuAttr - Menu attributes.
 
*******************************************************************************/

T_MFW_HND mmi_fv_opt_start(T_MFW_HND parent_window, MfwMnuAttr *menuAttr)
{
    T_MFW_HND win;

    TRACE_FUNCTION ("mmi_fv_opt_start()");

    fv_optwin = (T_MFW_HND)mmi_fv_opt_create ((MfwHnd)parent_window);

    if (fv_optwin NEQ NULL)
	{
	    SEND_EVENT (fv_optwin, FV_INIT, NULL, (void *)menuAttr);
	}
    return fv_optwin;
}

/*******************************************************************************

 $Function:    	mmi_fv_opt_create

 $Description:	Create the Option window
 
 $Returns:		mfw window handle

 $Arguments:	parent_window -parent window handle
 			
*******************************************************************************/

static T_MFW_HND mmi_fv_opt_create(MfwHnd parent_window)
{
    T_fv_opt      * data = (T_fv_opt *)ALLOC_MEMORY (sizeof (T_fv_opt));
    T_MFW_WIN  * win;

    TRACE_FUNCTION ("mmi_fv_opt_create()");

    /*
     * Create window handler
     */

    data->win = 
	win_create (parent_window, 0, E_WIN_VISIBLE, (T_MFW_CB)mmi_fv_opt_win_cb);
    if (data->win EQ NULL)
	{
	    return NULL;
	}
    /* 
     * connect the dialog data to the MFW-window
     */

    data->mmi_control.dialog = (T_DIALOG_FUNC)mmi_fv_opt_exec_cb;
    data->mmi_control.data   = data;
    win                      = ((T_MFW_HDR *)data->win)->data;
    win->user                = (void *)data;
    data->parent             = parent_window;

    /*
     * return window handle
     */

    return data->win;
}

/*******************************************************************************

 $Function:    	mmi_fv_opt_destroy

 $Description:	Destroy the Option window
 
 $Returns:		mfw window handle

 $Arguments:	own_window - current window
 			
*******************************************************************************/

static void mmi_fv_opt_destroy(MfwHnd own_window)
{
    T_MFW_WIN * win_data;
    T_fv_opt     * data;

    TRACE_FUNCTION ("mmi_fv_opt_destroy()");

    if (own_window)
	{
	    win_data = ((T_MFW_HDR *)own_window)->data;
	    data = (T_fv_opt *)win_data->user;

	    if (data)
		{
		    /*
		     * Delete WIN handler
		     */ 

			if (data->kbd)
			{
				kbd_delete(data->kbd);
				data->kbd = NULL;
			}
			if (data->kbd_long)
			{
				kbd_delete(data->kbd_long);
				data->kbd_long = NULL;
			}
			if (data->menu)
			{
				mnuDelete(data->menu);
				data->menu=NULL;
			}
			
		    win_delete (data->win);

		    /*     
		     * Free Memory
		     */
		    FREE_MEMORY ((void *)data, sizeof (T_fv_opt));
		}
	    else
		{
		    TRACE_EVENT ("mmi_fv_opt_destroy() called twice");
		}
	}
}


/*******************************************************************************

 $Function:    	mmi_fv_opt_win_cb

 $Description:	Window callback function for the Option window
 
 $Returns:		execution status

 $Arguments:	e - event id
 				w - window handle 
*******************************************************************************/

static int mmi_fv_opt_win_cb (MfwEvt e, MfwWin *w)
{
    TRACE_FUNCTION ("mmi_fv_opt_win_cb()");

    switch (e)
	{
        case MfwWinVisible:  /* window is visible  */
		dspl_ClearAll();
	    displaySoftKeys(TxtSoftSelect,TxtSoftBack);
	    break;
	case MfwWinFocussed: /* input focus / selected   */
	case MfwWinDelete:   /* window will be deleted   */
        default:
	    return MFW_EVENT_REJECTED;
	}
    return MFW_EVENT_CONSUMED;
}


/*******************************************************************************

 $Function:    	mmi_fv_view_win_cb

 $Description:	Window callback function for the Option window
 
 $Returns:		execution status

 $Arguments:	e - event id
 				w - window handle 
*******************************************************************************/

static int mmi_fv_view_win_cb (MfwEvt e, MfwWin w)
{
    TRACE_FUNCTION ("mmi_fv_view_win_cb()");

    switch (e)
	{
        case MfwWinVisible:  /* window is visible  */
		dspl_ClearAll();
		displaySoftKeys(TxtSoftOptions,TxtSoftBack);
		//Display the image now.
		mmi_fv_display_image();		
	    break;
	case MfwWinFocussed: /* input focus / selected   */
	case MfwWinDelete:   /* window will be deleted   */
        default:
	    return MFW_EVENT_REJECTED;
	}
    return MFW_EVENT_CONSUMED;
}
/*******************************************************************************

 $Function:    	mmi_fv_opt_exec_cb

 $Description:	Exec callback function for the Option window
 
 $Returns:		none

 $Arguments:	win - current window
 				event - event id
 				value - unique id
 				parameter - optional data.
 			
*******************************************************************************/

void mmi_fv_opt_exec_cb (T_MFW_HND win, USHORT event, SHORT value, void * parameter)
 {
	T_MFW_WIN        * win_data = ((T_MFW_HDR *) win)->data;
	T_fv_opt * data     = (T_fv_opt *)win_data->user;
	T_MFW_HDR    * parent      =  data->parent;
	T_MFW_WIN    * win_data2   =  ((T_MFW_HDR *)parent)->data;
	T_fv_menu* parent_data =  (T_fv_menu *)win_data2->user;


    TRACE_FUNCTION ("mmi_fv_opt_exec_cb()");


    switch (event)
	{
	case FV_INIT:
		
		TRACE_EVENT("mmi_fv_opt_exec_cb() Event:5.FV_INIT");
		/* initialization of administrative data */
		data->parent_data = parent_data;		
		data->id = value;
		data->kbd =      kbdCreate(data->win,KEY_ALL,         (MfwCb)mmi_fv_opt_kbd_cb);
		data->kbd_long = kbdCreate(data->win,KEY_ALL|KEY_LONG,(MfwCb)mmi_fv_opt_kbd_cb);
		data->menu =     mnuCreate(data->win,(MfwMnuAttr*)parameter, E_MNU_ESCAPE, (MfwCb)mmi_fv_opt_mnu_cb);
		mnuLang(data->menu,mainMmiLng);
		/* put the (new) dialog window on top of the window stack */
		mnuUnhide(data->menu);
		winShow(win);
	    break;
	default:
	    return;
	}
}

/*******************************************************************************

 $Function:    	mmi_fv_opt_kbd_cb

 $Description:	Exec callback function for the Option window
 
 $Returns:		execution status

 $Arguments:	e - event id
 				k - keyboard info
*******************************************************************************/

static int mmi_fv_opt_kbd_cb (MfwEvt e, MfwKbd *k)
  {
    T_MFW_HND    win  = mfwParent(mfw_header());
    T_MFW_WIN  * win_data = ((T_MFW_HDR *)win)->data;
    T_fv_opt      * data = (T_fv_opt *)win_data->user;

    TRACE_FUNCTION ("mmi_fv_opt_kbd_cb()");

	if (e & KEY_LONG)
	{
		switch (k->code)
		{
		case KCD_HUP: /* back to previous menu */
			mnuEscape(data->menu);
		break;
		case KCD_RIGHT: /* Power Down */
			return MFW_EVENT_REJECTED; /* handled by idle */
		default: /* no response to all other keys */
			return MFW_EVENT_CONSUMED;
		}
	}
    else
	{
	    switch (k->code)
		{
		case KCD_MNUUP: /* highlight previous entry */
		    mnuUp(data->menu);
		    break;
		case KCD_MNUDOWN: /* highlight next entry */
		    mnuDown(data->menu);
		    break;
		case KCD_MNUSELECT:
		case KCD_LEFT: /* activate this entry */
		    mnuSelect(data->menu);
		    break;
		case KCD_HUP: /* back to previous menu */
		case KCD_RIGHT: /* back to previous menu */
		    mnuEscape(data->menu);
		    break;
		default: /* no response to all other keys */
		    return MFW_EVENT_CONSUMED;
		}
	}
    return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:    	mmi_fv_view_kbd_cb

 $Description:	Exec callback function for the view window
 
 $Returns:		execution status

 $Arguments:	e - event id
 				k - keyboard info
*******************************************************************************/

static int mmi_fv_view_kbd_cb (MfwEvt e, MfwKbd *k)
   {
    T_MFW_HND    win  = mfwParent(mfw_header());
    T_MFW_WIN  * win_data = ((T_MFW_HDR *)win)->data;
    T_fv_opt      * data = (T_fv_opt *)win_data->user;

    TRACE_FUNCTION ("mmi_fv_view_kbd_cb()");

	switch (k->code)
	{
	case KCD_LEFT:
		TRACE_EVENT("mmi_fv_view_kbd_cb,INFO_KCD_LEFT ");
		bookMenuStart(data->win, ViewOptionsMenuAttributes(),0);
		break;
  
	case KCD_RIGHT:
	case KCD_HUP:
		TRACE_EVENT("mmi_fv_view_kbd_cb,INFO_KCD_RIGHT ");
		mmi_fv_opt_destroy(data->win);
		break;
	default:
		
		break;
	
	}
	return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:    	mmi_fv_opt_mnu_cb

 $Description:	Menu callback function for the Option window
 
 $Returns:		none

 $Arguments:	e - event id
 				m - menu handle
*******************************************************************************/

static int mmi_fv_opt_mnu_cb (MfwEvt e, MfwMnu *m)
    
{
    T_MFW_HND    win  = mfwParent(mfw_header());
    T_MFW_WIN  * win_data = ((T_MFW_HDR *)win)->data;
    T_fv_opt      * data = (T_fv_opt *)win_data->user;
  
    TRACE_FUNCTION ("mmi_fv_opt_mnu_cb()");

    switch (e)
        {
		case E_MNU_ESCAPE: /* back to previous menu */
			mmi_fv_opt_destroy(data->win);
		break;
		default: /* in mnuCreate() only E_MNU_ESCAPE has been enabled! */
			return MFW_EVENT_REJECTED;
        }
    return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

  $Function:	  mmi_fv_rotate_start
  
  $Description:	  This function creates and displays the  Roate menu .
	
  $Returns:		  Execution status
	  
  $Arguments:	  m - menu handler
				  i - Menu item selected
		
*******************************************************************************/

 int mmi_fv_rotate_start(MfwMnu* m, MfwMnuItem* i)
{
	
    T_MFW_HND       win  = mfwParent(mfw_header());
    T_MFW_WIN     * win_data = ((T_MFW_HDR *)win)->data;
    T_fv_opt * data = (T_fv_opt *)win_data->user;
	TRACE_FUNCTION("mmi_fv_rotate_start");
	if (data->win EQ NULL)
	{
		return NULL;
	}
	
	/* Create the  Rotate menu*/
	bookMenuStart(data->win, rotateMenuAttributes(),0);
	
	return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

  $Function:	  mmi_fv_zoom_start
  
  $Description:	  This function creates and displays the  Zoom menu .
	
  $Returns:		  Execution status
	  
  $Arguments:	  m - menu handler
				  i - Menu item selected
		
*******************************************************************************/

int mmi_fv_zoom_start(MfwMnu* m, MfwMnuItem* i)
{
	
    T_MFW_HND       win  = mfwParent(mfw_header());
    T_MFW_WIN     * win_data = ((T_MFW_HDR *)win)->data;
    T_fv_opt * data = (T_fv_opt *)win_data->user;
	TRACE_FUNCTION("mmi_fv_zoom_start");
	if (data->win EQ NULL)
	{
		return NULL;
	}
	
	/* Create the  Zoom menu*/
	bookMenuStart(data->win, zoomMenuAttributes(),0);
	
	return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:     mmi_fv_GetExtension

 $Description: public function to retrieve the extension of a file

 $Returns:Extention of the filename

 $Arguments: scr- Filename

*******************************************************************************/
 char *mmi_fv_GetExtension(char *src)
{
    int i;
		TRACE_FUNCTION("mmi_fv_GetExtension");
    for(i = 0; i < strlen(src); i++){
        if(src[i] == '.'){
            return (src+i+1);
        }
    }
    return (src+i);
}


/*******************************************************************************

 $Function:     mmi_fv_delete_start

 $Description:  This function is called when the user press Delete .

 $Returns:    Execution status

 $Arguments:  m - menu handler
        i - Menu item selected

*******************************************************************************/
int mmi_fv_delete_start(MfwMnu* m, MfwMnuItem* i)
{
	T_MFW_HND     	   win      = mfwParent(mfw_header());
	 T_DISPLAY_DATA display_info;

	TRACE_FUNCTION("mmi_fv_delete_start");
	//Check whether files are present
	if ( mmi_fv_get_file_count() == 0)
		{	
			/*tell user*/
			mmi_fv_test_show_info(0,TxtNull,TxtNull, TxtEmptyList, TxtNull, NULL);
		}
		else
		{ 
			TRACE_EVENT("files are present");
			TRACE_EVENT_P1("delete_index is %d",fv_menu_data.fv_list_idx);
			TRACE_EVENT_P1("The selected file = %s",jpegFiles[fv_menu_data.fv_list_idx]);
			dlg_initDisplayData_TextId( &display_info, TxtSoftOK, TxtSoftBack, TxtDeleteConfirm, TxtNull, COLOUR_STATUS);
			dlg_initDisplayData_events( &display_info, (T_VOID_FUNC)mmi_fv_delete_cb, FOREVER, KEY_LEFT |KEY_RIGHT|KEY_CLEAR|KEY_HUP);
			info_dialog (win, &display_info);
				
		}
	return MFW_EVENT_CONSUMED;

}


/*******************************************************************************

 $Function:     mmi_fv_delete_cb

 $Description:  Exec callback function for the delete window


 $Returns:    none

 $Arguments:  win - current win
     			Identifier-
     			reason-

*******************************************************************************/
static void mmi_fv_delete_cb (T_MFW_HND win, USHORT Identifier, SHORT reason)
{
	T_MFW_WIN      * win_data = ((T_MFW_HDR *) win)->data;
	T_fv_opt         * data = (T_fv_opt *)win_data->user;
	int status;

	TRACE_FUNCTION ("mmi_fv_delete_cb()");

	switch (reason)
	{
		case INFO_KCD_LEFT:
			TRACE_EVENT("file has been left");

			//Delete the selected file from FFS.
			status=mmi_fv_delete_jpeg_file((UBYTE)fv_menu_data.fv_list_idx);
			if(status>0)
				{
					mmi_fv_test_show_info(0, TxtNull,TxtNull,TxtDeleted, TxtNull, NULL);
				}
			SEND_EVENT(data->parent, FV_INIT, Identifier, NULL);
		break;
		case INFO_KCD_RIGHT:
		case INFO_KCD_HUP:
				TRACE_EVENT("file has been back");
		break;
		default:
		break;
	}
	mmi_fv_opt_destroy(fv_optwin);
}

/*******************************************************************************

 $Function:     mmi_fv_read_jpeg_file

 $Description:	reads the contents of the file into the buffer .
 $Returns: Status

 $Arguments:index: Index of the file to be read from FFS.

*******************************************************************************/
GLOBAL int mmi_fv_read_jpeg_file(UBYTE index)
{
    T_FFS_FD	file_handle	= 0;
    SHORT		readStatus	= 1;
	//Aug 31, 2005    REF: SPR 34050 xdeepadh
	//The filename will have the full path details
     char		file_name[FV_DIR_PATH_LENGTH];
    
   	
   	TRACE_FUNCTION ("mmi_fv_read_jpeg_file()");
   	
	strcpy((char*)file_name,FV_DIR_NAME);
	strcat((char*)file_name, (const char*)jpegFiles[index]);
	
	TRACE_EVENT_P1("The selected file = %s",jpegFiles[index]);
	
	file_handle = ffs_open(file_name,FFS_O_RDONLY);
	if(file_handle <0)
	{
		readStatus = -1;  
		mmi_fv_test_show_info(0,TxtNull,TxtNull, TxtFileNotLoaded, TxtNull, NULL);
	}
	else
	{
		UINT16 Ffs_stat;
		struct stat_s  stat;
		Ffs_stat = ffs_stat(file_name,&stat);
		
		TRACE_EVENT_P1("read jpg size: %d", stat.size);
			if (Ffs_stat != 0)            
				TRACE_EVENT_P1("read jpg size: %d", stat.size);	
			if(stat.size   >=  FV_MAX_JPEG_SIZE )
				{
					TRACE_EVENT_P1("File zise too big: %d", stat.size);	
					readStatus = -1;
					//Aug 30, 2005    REF:SPR 34049 xdeepadh
					//Close the file handle
					ffs_close(file_handle);
					return readStatus;
				}
			memset(jpeg_test_array,0,FV_MAX_JPEG_SIZE);	
			Ffs_stat = ffs_read(file_handle, (UBYTE *)jpeg_test_array, stat.size);	
			TRACE_EVENT_P1("read jpg file size: %d", Ffs_stat);	
			ffs_close(file_handle);
	}
	return readStatus;
}

/*******************************************************************************

 $Function:     mmi_fv_delete_jpeg_file

 $Description:	delete the selected file  from FFS.
 $Returns:Status

 $Arguments:  index: Index of the file to be read from FFS.

*******************************************************************************/
GLOBAL int  mmi_fv_delete_jpeg_file(UBYTE index)
{
//Aug 31, 2005    REF: SPR 34050 xdeepadh
//The filename will have the full path details
    char		file_name[FV_DIR_PATH_LENGTH];
    T_FFS_RET status;
   	
   	TRACE_FUNCTION ("mmi_fv_delete_jpeg_file()");
   	
	strcpy((char*)file_name,FV_DIR_NAME);
	strcat((char*)file_name, (const char*)jpegFiles[index]);
	
	TRACE_EVENT_P1("The selected file = %s",jpegFiles[index]);
	TRACE_EVENT_P1("The deleted file = %s",file_name);
	
	status = ffs_remove(file_name);
	if (status == EFFS_OK)
	{
		//Populate the list again
		file_count = 0;
		mmi_fv_init(&file_count);
	}
	return status;
}



/*******************************************************************************

 $Function:     mmi_fv_display_image

 $Description:	The decoded and color converted image is displayed onto the screen.

 $Returns:Status

 $Arguments:None

*******************************************************************************/

GLOBAL int mmi_fv_display_image()
{
	UBYTE status = 0;
	TRACE_FUNCTION("mmi_fv_display_image()");

	status = mfw_fv_decode_image(jpeg_test_array,sizeof(jpeg_test_array));
	if(status)
	{
	TRACE_FUNCTION("mmi_fv_display_image()--OK");
		//return MFW_EVENT_CONSUMED;
	}
	else
	{
	TRACE_FUNCTION("mmi_fv_display_image()--Fail");
		//return MFW_EVENT_REJECTED;
	}

	return MFW_EVENT_CONSUMED;

}

/*******************************************************************************

 $Function:     mmi_fv_reame_edit

 $Description:  This function is called when the user press rename .

 $Returns:    Execution status

 $Arguments:  m - menu handler
        i - Menu item selected

*******************************************************************************/
GLOBAL int mmi_fv_reame_edit(MfwMnu* m, MfwMnuItem* i)
{
	T_MFW_HND win = mfw_parent(mfw_header());
	//T_MFW_WIN		* win_data = ((T_MFW_HDR *) win)->data;
	T_MFW_HND win_temp;
	//T_FV_Editor_Data * data = (T_FV_Editor_Data *)(win_data->user);

    TRACE_FUNCTION ("mmi_fv_reame_edit()");

	win_temp = mmi_fv_editor_create(win);

	if (win_temp != NULL)
	{
	    SEND_EVENT(win_temp, FV_INIT, 0, (void *)NULL);
	}
	else
	{
	    TRACE_FUNCTION ("mmi_fv_reame_edit() : Unable to create Editor Win!");
	}
	return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:    	mmi_fv_editor_create

 $Description:	Create a window for renaming the image file name
 
 $Returns:		window handle

 $Arguments:	parent - parent window.
 				
*******************************************************************************/


T_MFW_HND mmi_fv_editor_create(T_MFW_HND parent)
{
	T_FV_Editor_Data *data;
	T_MFW_WIN * win;

	TRACE_FUNCTION ("mmi_fv_editor_create");

	data = (T_FV_Editor_Data *)mfwAlloc((U16)sizeof(T_FV_Editor_Data));

	if (data == NULL)
		return data;

	data->win = winCreate(parent, 0, E_WIN_VISIBLE, NULL);

	if (data->win == NULL)
	{
		/*
		** Failed to start : Free Memory, and exit
		*/
		mfwFree((U8 *)data, (U16)sizeof(T_FV_Editor_Data));
		data = NULL;
		return data;
	}
	else
	{
		/*
		** Setup the Dialog control functions
		*/
		data->mmi_control.dialog = (T_DIALOG_FUNC)mmi_fv_editor_cntrl;
		data->mmi_control.data = data;
		
		win = ((T_MFW_HDR *)data->win)->data;
		win->user = (void *)data;
	}

	return data->win;

}

/*******************************************************************************

 $Function:    	mmi_fv_editor_cntrl

 $Description:	Exec callback function for the rename window
 
 $Returns:		none

 $Arguments:	win - current window
 				event - event id
 				value - unique id
 				parameter - optional data.
 			
*******************************************************************************/
void mmi_fv_editor_cntrl (T_MFW_HND win, USHORT event, SHORT value, void * parameter)
{
	T_MFW_WIN			* win_data = ((T_MFW_HDR *) win)->data;
	T_FV_Editor_Data * data = (T_FV_Editor_Data *)win_data->user;
	T_AUI_EDITOR_DATA	editor_data;
//Aug 31, 2005    REF: SPR 34050 xdeepadh
//The filename will have the full path details	
	char  new_filename[FV_DIR_PATH_LENGTH];
	char selected_file[FV_DIR_PATH_LENGTH];
	int status = 0;

	switch (event)
	{
		case FV_INIT:
#ifdef NEW_EDITOR
                      //Clear the editor buffer
			memset(&editor_data,'\0',sizeof(editor_data));
			// Get the name from FFS
			TRACE_EVENT_P1("The file to be renamed file = %s",jpegFiles[fv_menu_data.fv_list_idx]);
			strcpy((char *)data->fv_buffer, (const char*)jpegFiles[fv_menu_data.fv_list_idx]);
			
			/*
			** Initialise the editor
			*/
			AUI_edit_SetDefault(&editor_data);
			AUI_edit_SetBuffer(&editor_data, ATB_DCS_ASCII, data->fv_buffer, FV_FILE_NAME_LENGTH);
			AUI_edit_SetTextStr(&editor_data, TxtSoftOK, TxtDelete, TxtRename, NULL);
			AUI_edit_SetAltTextStr(&editor_data, 1, TxtNull, TRUE, TxtSoftBack);
			AUI_edit_SetEvents(&editor_data, 0, TRUE, FOREVER, (T_AUI_EDIT_CB)mmi_rename_editor_cb);
			data->fv_win_editor = AUI_edit_Start(data->win, &editor_data);
#endif
			break;

		case FV_EDITOR_CANCEL:
			/*
			** The editor will have been destroyed. Reset the Window handler
			*/
			data->fv_win_editor = NULL;
			/*
			** Destroy the editor Window
			*/
			TRACE_FUNCTION ("mmi_fv_editor_cntrl cancel");
			mmi_fv_editor_destroy(data->win);
			mmi_fv_opt_destroy(fv_optwin);
			break;

		case FV_EDITOR_ERROR:		
			//Aug 31, 2005    REF: SPR 34050 xdeepadh
			//Display the error message to the user
			mmi_fv_test_show_info(0, TxtNull,TxtNull,TxtNoExtension, TxtNull, NULL);
			/*
			** The editor will have been destroyed. Reset the Window handler
			*/
			data->fv_win_editor = NULL;
			/*
			** Destroy the editor Window
			*/
			TRACE_FUNCTION ("mmi_fv_editor_cntrl cancel");
			mmi_fv_editor_destroy(data->win);
			mmi_fv_opt_destroy(fv_optwin);
			break;

		case FV_EDITOR_SELECT:
			//copy the file name from the editor
			strcpy(new_filename,FV_DIR_NAME);
			strcat(new_filename,(char *)data->fv_buffer);
			TRACE_EVENT_P1("new_filename %s",new_filename);	
			
			strcpy(selected_file,FV_DIR_NAME);
			strcat(selected_file,(char*)jpegFiles[fv_menu_data.fv_list_idx]);
			TRACE_EVENT_P1("selected_file %s",selected_file);	

			// Rename the file in FFS.
			status = ffs_rename(selected_file,new_filename);

			TRACE_EVENT_P1("status is %d",status);	
			//Jan 03, 2006    REF: SPR OMAPS00060036  xdeepadh
			//While renaming the file,if the name already exists,  display a proper error message.
			switch(status)
				{

				case EFFS_OK:
					//Populate the list again
					file_count = 0;
					mmi_fv_init(&file_count);
					break;
					
				case  EFFS_EXISTS:
					mmi_fv_test_show_info(0, TxtNull,TxtNull,TxtFileExists, TxtNull, NULL);
					break;
				default:
					//Display the error message to the user
					mmi_fv_test_show_info(0, TxtNull,TxtNull,TxtWrongFilename, TxtNull, NULL);
				}
			
			/*
			** Destroy the Editor Window
			*/
			mmi_fv_editor_destroy(data->win);
			mmi_fv_opt_destroy(fv_optwin);
						
			break;

		default:
			TRACE_FUNCTION ("mmi_fv_editor_cntrl(), Unexpected Event!"); 
			break;
	}


}


/*******************************************************************************

 $Function:    	mmi_fv_editor_destroy

 $Description:	Destroy the  rename editor window
 
 $Returns:		mfw window handle

 $Arguments:	own_window - current window
 			
*******************************************************************************/
MfwRes mmi_fv_editor_destroy(T_MFW_HND win)
{
	T_MFW_WIN * win_data;
	T_FV_Editor_Data * data;
			
	TRACE_FUNCTION ("mmi_fv_editor_destroy");


	if (win == NULL)
		return MfwResIllHnd;

	win_data = ((T_MFW_HDR *) win)->data;
	data = (T_FV_Editor_Data *)win_data->user;

	/*
	** Destroy the Window
	*/
	win_delete(data->win);

	/*
	** Free the dynamically allocated memory
	*/
	mfwFree((U8 *)data, (U16)sizeof(T_FV_Editor_Data));

	return MfwResOk;
}

/*******************************************************************************

 $Function:		mmi_rename_editor_cb

 $Description:		This function provides the callback functionality to an editor for file viewer

 $Returns:		None

 $Arguments:	win - current window
        identifier -
        reason -
*******************************************************************************/

static void mmi_rename_editor_cb(T_MFW_HND win, USHORT identifier, SHORT reason)
{
	T_MFW_WIN		* win_data = ((T_MFW_HDR *) win)->data;
    T_FV_Editor_Data* dev_name_data = (T_FV_Editor_Data *)win_data->user;

	UBYTE			* editor_buffer = NULL;
	SHORT			max_edit_size = 0;
	T_MFW_HND		editor_win;
	T_MFW_HND		parent;
	char * fileExt;

    TRACE_FUNCTION ("mmi_rename_editor_cb()");

		editor_buffer = dev_name_data->fv_buffer;
		max_edit_size = FV_FILE_NAME_LENGTH - 1;
		editor_win = dev_name_data->fv_win_editor;
		parent = dev_name_data->win;
	
	switch (reason)
	{
		case INFO_KCD_LEFT:
		{	
			/*
			** If there is no string entered, treat this as if it was a "Cancel"
			*/
			if (editor_buffer[0] == NULL)
			{
				/*
				** Send a "Cancel" to the  window
				*/
				SEND_EVENT(parent, FV_EDITOR_CANCEL, identifier, NULL);
			}
			else
			{
				//If the file extn is not jpg, send cancel event
				fileExt = (char *)mmi_fv_GetExtension((char *)editor_buffer);
				if ( (strcmp(fileExt, "jpg") != 0))
				{
					//Aug 31, 2005    REF: SPR 34050 xdeepadh
					/*
					** Send a "Error" to the  window
					*/
					SEND_EVENT(parent, FV_EDITOR_ERROR, identifier, NULL);
				}
				else
				{
					/*
					** Send a "Select" to the  window
					*/
					SEND_EVENT(parent, FV_EDITOR_SELECT, identifier, NULL);
				}
			}
		}
		break;

		case INFO_KCD_RIGHT:
			if (strlen((char *)editor_buffer) == NULL)
			{

				/*
				** Send a "Cancel" to the  window
				*/
				SEND_EVENT(parent, FV_EDITOR_CANCEL, identifier, NULL);
			}
			break;

		case INFO_KCD_HUP:
			/*
			** Set the buffer to NULLS
			*/
			memset(editor_buffer, NULL, max_edit_size);

			/*
			** Send a "Cancel" to the   window
			*/
			SEND_EVENT(parent, FV_EDITOR_CANCEL, identifier, NULL);
			break;

		default:
			/* otherwise no action to be performed
			*/
			break;
	}
}

/*******************************************************************************

 $Function:    	mmi_fv_rotate_image_start

 $Description:	Create window for displaying the rotated image.
 
 $Returns:		none
 
 $Arguments:	m - menu handler
 				i - menu item selected.
*******************************************************************************/
int mmi_fv_rotate_image_start(T_MFW_HND parent) 
{
	T_MFW_HND win;
	win = (T_MFW_HND)mmi_fv_rotate_image_create((MfwHnd)parent);
	TRACE_FUNCTION ("mmi_fv_rotate_image_start");
	if (win NEQ NULL)
	{
		SEND_EVENT(win,FV_INIT,FV_OPT_ID,0);
	}

  	return MFW_EVENT_CONSUMED;
}


/*******************************************************************************

 $Function:    	mmi_fv_rotate_image_create

 $Description:	Create a window for rotating the image.
 
 $Returns:		window handle

 $Arguments:	parent_window - parent window.
 				
*******************************************************************************/

static T_MFW_HND mmi_fv_rotate_image_create(MfwHnd parent_window)
{
	T_fv_opt* data = (T_fv_opt *)ALLOC_MEMORY (sizeof (T_fv_opt));
	T_MFW_WIN  * win;

	TRACE_FUNCTION ("mmi_fv_rotate_image_create()");

	// Create window handler
	data->win = win_create (parent_window, 0, E_WIN_VISIBLE,(MfwCb)mmi_fv_rotate_win_cb);
	if (data->win EQ NULL)
	{
		return NULL;
	}

	// connect the dialog data to the MFW-window
	data->mmi_control.dialog = (T_DIALOG_FUNC)mmi_fv_rotate_exec_cb;
	data->mmi_control.data   = data;
	win                      = ((T_MFW_HDR *)data->win)->data;
	win->user                = (MfwUserDataPtr)data;
	data->parent             = parent_window;
	//Create keyboard handler
        data->kbd = kbd_create (data->win,KEY_ALL,(T_MFW_CB)mmi_fv_rotate_kbd_cb);
	winShow(data->win);
	return data->win;
}


/*******************************************************************************

 $Function:    	mmi_fv_rotate_exec_cb

 $Description:	Callback handler for events sent by the images editor

 $Returns:		none

 $Arguments:	win - current window
 				event - event id
 				parameter - optional data.
*******************************************************************************/

static void mmi_fv_rotate_exec_cb(T_MFW_HND win, USHORT event, void *parameter)
{	
	T_MFW_WIN      * win_data = ((T_MFW_HDR *) win)->data;
	T_fv_opt         * data = (T_fv_opt *)win_data->user;
	int status;

	TRACE_FUNCTION ("mmi_fv_rotate_exec_cb()");

	switch (event)
	{
	case FV_INIT:
		TRACE_EVENT("init");
		status = mfw_fv_rotate_image();
		if(!status)
		{
			mmi_fv_test_show_info(0,TxtNull,TxtNull, TxtError, TxtNull, NULL);
		}	
	break;

	case FV_RETURN:
				
	break;
	default:
	TRACE_EVENT("Err: Default");
	return;
	}
}

/*******************************************************************************

 $Function:    	mmi_fv_rotate_win_cb

 $Description:	Window callback function for the rotate window
 
 $Returns:		execution status

 $Arguments:	e - event id
 				w - window handle 
*******************************************************************************/

static int mmi_fv_rotate_win_cb (MfwEvt e, MfwWin w)
{
    TRACE_FUNCTION ("mmi_fv_rotate_win_cb()");

    switch (e)
	{
        case MfwWinVisible:  /* window is visible  */
		dspl_ClearAll();
	    displaySoftKeys(TxtNull,TxtSoftBack);
	    break;
	case MfwWinFocussed: /* input focus / selected   */
	case MfwWinDelete:   /* window will be deleted   */
        default:
	    return MFW_EVENT_REJECTED;
	}
    return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:    	mmi_fv_rotate_kbd_cb

 $Description:	Exec callback function for the rotate window
 
 $Returns:		execution status

 $Arguments:	e - event id
 				k - keyboard info
*******************************************************************************/

static int mmi_fv_rotate_kbd_cb (MfwEvt e, MfwKbd *k)
   {
    T_MFW_HND    win  = mfwParent(mfw_header());
    T_MFW_WIN  * win_data = ((T_MFW_HDR *)win)->data;
    T_fv_opt      * data = (T_fv_opt *)win_data->user;

    TRACE_FUNCTION ("mmi_fv_rotate_kbd_cb()");

	switch (k->code)
	{
	case KCD_LEFT:
		TRACE_EVENT("mmi_fv_rotate_kbd_cb,INFO_KCD_LEFT ");
		break;
  
	case KCD_RIGHT:
	case KCD_HUP:
		TRACE_EVENT("mmi_fv_rotate_kbd_cb,INFO_KCD_RIGHT ");
		mmi_fv_opt_destroy(data->win);
		break;
	default:
		
		break;

	}
		return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:    	mmi_fv_zoom_image_start

 $Description:	Create window for displaying the zoomed image.
 
 $Returns:		none
 
 $Arguments:	m - menu handler
 				i - menu item selected.
*******************************************************************************/
int mmi_fv_zoom_image_start(T_MFW_HND parent) 
{
	T_MFW_HND win;
	win = (T_MFW_HND)mmi_fv_zoom_image_create((MfwHnd)parent);
	TRACE_FUNCTION ("mmi_fv_zoom_image_start");
	if (win NEQ NULL)
	{
		SEND_EVENT(win,FV_INIT,FV_OPT_ID,0);
	}
	return MFW_EVENT_CONSUMED;

}

/*******************************************************************************

 $Function:    	mmi_fv_zoom_image_create

 $Description:	Create a window for zooming the image.
 
 $Returns:		window handle

 $Arguments:	parent_window - parent window.
 				
*******************************************************************************/

static T_MFW_HND mmi_fv_zoom_image_create(MfwHnd parent_window)
{
	T_fv_opt* data = (T_fv_opt *)ALLOC_MEMORY (sizeof (T_fv_opt));
	T_MFW_WIN  * win;

	TRACE_FUNCTION ("mmi_fv_zoom_image_create()");

	// Create window handler
	data->win = win_create (parent_window, 0, E_WIN_VISIBLE,(MfwCb)mmi_fv_zoom_win_cb);
	if (data->win EQ NULL)
	{
		return NULL;
	}

	// connect the dialog data to the MFW-window
	data->mmi_control.dialog = (T_DIALOG_FUNC)mmi_fv_zoom_exec_cb;
	data->mmi_control.data   = data;
	win                      = ((T_MFW_HDR *)data->win)->data;
	win->user                = (MfwUserDataPtr)data;
	data->parent             = parent_window;
	//Create keyboard handler
        data->kbd = kbd_create (data->win,KEY_ALL,(T_MFW_CB)mmi_fv_zoom_kbd_cb);
	winShow(data->win);
	return data->win;
}


/*******************************************************************************

 $Function:    	mmi_fv_zoom_exec_cb

 $Description:	Callback handler for events sent by the images editor

 $Returns:		none

 $Arguments:	win - current window
 				event - event id
 				parameter - optional data.
*******************************************************************************/

static void mmi_fv_zoom_exec_cb(T_MFW_HND win, USHORT event, void *parameter)
{	
	T_MFW_WIN      * win_data = ((T_MFW_HDR *) win)->data;
	T_fv_opt         * data = (T_fv_opt *)win_data->user;
	int status;

	TRACE_FUNCTION ("mmi_fv_zoom_exec_cb()");

	switch (event)
	{
	case FV_INIT:
		TRACE_EVENT("init");
		status = mfw_fv_zoom_image();
		if(!status)
		{
			mmi_fv_test_show_info(0,TxtNull,TxtNull, TxtError, TxtNull, NULL);
		}	
			
	break;

	case FV_RETURN:
		
	break;
	default:
	TRACE_EVENT("Err: Default");
	return;
	}
}

/*******************************************************************************

 $Function:    	mmi_fv_zoom_win_cb

 $Description:	Window callback function for the Option window
 
 $Returns:		execution status

 $Arguments:	e - event id
 				w - window handle 
*******************************************************************************/

static int mmi_fv_zoom_win_cb (MfwEvt e, MfwWin w)
{
    TRACE_FUNCTION ("mmi_fv_zoom_win_cb()");

    switch (e)
	{
        case MfwWinVisible:  /* window is visible  */
		dspl_ClearAll();
	    displaySoftKeys(TxtNull,TxtSoftBack);
	    break;
	case MfwWinFocussed: /* input focus / selected   */
	case MfwWinDelete:   /* window will be deleted   */
        default:
	    return MFW_EVENT_REJECTED;
	}
    return MFW_EVENT_CONSUMED;
}

/*******************************************************************************

 $Function:    	mmi_fv_zoom_kbd_cb

 $Description:	Exec callback function for the zoom window
 
 $Returns:		execution status

 $Arguments:	e - event id
 				k - keyboard info
*******************************************************************************/

static int mmi_fv_zoom_kbd_cb (MfwEvt e, MfwKbd *k)
   {
    T_MFW_HND    win  = mfwParent(mfw_header());
    T_MFW_WIN  * win_data = ((T_MFW_HDR *)win)->data;
    T_fv_opt      * data = (T_fv_opt *)win_data->user;

    TRACE_FUNCTION ("mmi_fv_zoom_kbd_cb()");

	switch (k->code)
	{
	case KCD_LEFT:
		TRACE_EVENT("mmi_fv_zoom_kbd_cb,INFO_KCD_LEFT ");
		break;
  
	case KCD_RIGHT:
	case KCD_HUP:
		TRACE_EVENT("mmi_fv_zoom_kbd_cb,INFO_KCD_RIGHT ");
		mmi_fv_opt_destroy(data->win);
		break;
	default:
		
		break;
	
	}
	return MFW_EVENT_CONSUMED;
}


