/*
  +------------------------------------------------------------------------------
  |  File:       alr_steps.cpp
  +------------------------------------------------------------------------------
  |              Copyright 2003 Texas Instruments
  |              All rights reserved.
  |
  |              This file is confidential and a trade secret of Texas Instruments.
  |              The receipt of or possession of this file does not convey
  |              any rights to reproduce or disclose its contents or to
  |              manufacture, use, or sell anything it may describe, in
  |              whole, or in part, without the specific written consent of
  |              Texas Instruments.
  +------------------------------------------------------------------------------
  | Purpose:     Contains test steps for alr
  |
  +------------------------------------------------------------------------------
*/
#include "alr_constraints.h"
#include "alr_steps.h"



/*------------------------Following is for ALR000-------------------------------
 *
 * Description:	The ALR is configured.
 *-------------------------------------------------------------------------------*/

T_STEP filter_and_routings_gsm_900__alr000()
{
  BEGIN_STEP ("filter_and_routings_gsm_900__alr000")
  {
    COMMAND("TAP REDIRECT CLEAR");
    COMMAND("PL REDIRECT CLEAR");
    COMMAND("PL REDIRECT RR TAP");
    COMMAND("PL REDIRECT DL TAP");
    COMMAND("PL REDIRECT L1 TAP");
    COMMAND("TAP REDIRECT TAP PL");
    COMMAND("PL CONFIG STD=1");
#ifdef  GPRS
    COMMAND("PL REDIRECT GRR TAP");
#endif // GPRS
    SEND  ( em_l1_event_req_2()  );
    //COMMAND("TAP PARKING ");

  }
}

/*------------------------Following is for ALR001-------------------------------
 *
 * Description:	RR starts a cell selection with power measurements with interruption.
 * Four channels in the
 * order 23, 14, 124, 1 have acceptable fieldstrength values.
 *--------------------------------------------------------------------------------*/

T_STEP initiation_by_rr_4_channels_available_alr001()
{
  BEGIN_STEP ("initiation_by_rr_4_channels_available_alr001")
  {
    filter_and_routings_gsm_900__alr000();

    SEND  ( mph_power_req_1()           );
    AWAIT ( mphc_init_l1_req_1()        );
    SEND  ( mphc_init_l1_con_1()        );
    AWAIT ( mphc_rxlev_req()            );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
    AWAIT ( mphc_rxlev_req()            );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
    AWAIT ( mphc_rxlev_req()            );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
    AWAIT ( mphc_rxlev_req()            );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
    AWAIT ( mphc_rxlev_req()            );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
    AWAIT ( mph_power_cnf_1()           );
  }
}

/*------------------------Following is for ALR002-------------------------------
 *
 * Description:	RR starts a cell selection with power measurements with interruption.
 * Eight channels in the
 * order 23, 637, 14, 580, 124, 885, 1, 512 have acceptable fieldstrength values.
 *--------------------------------------------------------------------------------*/

T_STEP initiation_by_rr_8_channels_available_alr002()
{
  BEGIN_STEP ("initiation_by_rr_4_channels_available_alr001")
  {
    filter_and_routings_dualband_gsm_900_dcs_1800__alr600();

    SEND  ( mph_power_req_4()     );
    AWAIT ( mphc_init_l1_req_4()  );
    SEND  ( mphc_init_l1_con_1()  );
    AWAIT ( mphc_rxlev_req()      );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
    AWAIT ( mphc_rxlev_req()    );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
    AWAIT ( mphc_rxlev_req()    );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
    AWAIT ( mphc_rxlev_req()    );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
    AWAIT ( mphc_rxlev_req()    );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
    AWAIT ( mph_power_cnf_5()                 );
  }
}

/*------------------------Following is for ALR1018-------------------------------
 *
 * Description:RR configured for idle mode and recieves measurements 
 * for the neighbouring cells and serving cell
 *
 *-------------------------------------------------------------------------------*/

T_STEP configuring_ms_for_idle_mode_and_meas()
{
  BEGIN_STEP ("configuring_ms_for_idle_mode_and_meas")
  {
      SEND  ( mph_classmark_req_3()          );
      SEND  ( mph_idle_req_10()              );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_start_ccch_req_10()       );
      AWAIT ( mphc_scell_nbcch_req_full()    );
      SEND  ( mph_identity_req_1()           );
      SEND  ( mph_cbch_req_1()               );
      SEND  ( mph_neighbourcell_req_7A('A')  );
      AWAIT ( mphc_rxlev_periodic_req_21()   );
      SEND  ( mphc_rxlev_periodic_ind_9A(50,40)        );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_512)   );
      SEND  ( mphc_ncell_sync_ind(ARFCN_512,FN_OFF_ARFCN_512,TIME_ALIGNMT_512,BSIC_10));
      AWAIT ( mphc_ncell_bcch_req(ARFCN_512,FN_OFF_ARFCN_512,TIME_ALIGNMT_512,BSIC_10));
      SEND  ( mphc_rxlev_periodic_ind_9A(50,40)  );
      SEND  ( mph_meas_rep_req(NC_START_MEAS)    );
      SEND  ( mphc_rxlev_periodic_ind_9A(40,30)  );
      MUTE  (10);
      SEND  ( mphc_rxlev_periodic_ind_9A(50,40)  );
      AWAIT ( mph_measurement_ind_36()           );
      MUTE  (10);
      SEND  ( mphc_rxlev_periodic_ind_9A(40,30)  );
      MUTE  (10);
      SEND  ( mph_meas_rep_req(NC_REPORT_MEAS)   );
      AWAIT ( mph_measurement_ind_36()           );
      AWAIT ( mph_meas_rep_cnf_sc()              );
      SEND  ( mph_meas_rep_req(NC_STOP_MEAS)     );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_512) );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_512));
      SEND  ( mph_meas_rep_req(NC_START_MEAS)    );
      SEND  ( mphc_rxlev_periodic_ind_9A(40,30)  );
      SEND  ( mphc_rxlev_periodic_ind_9A(50,40)  );
      AWAIT ( mph_measurement_ind_36()           );
      SEND  ( mphc_rxlev_periodic_ind_9A(40,30)  );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_512)    );
      SEND  ( mph_meas_rep_req(NC_REPORT_MEAS)   );
      AWAIT ( mph_meas_rep_cnf_sc_n1()           );
  }
}

/*------------------------Following is for ALR003-------------------------------
 *
 * Description:	The carrier with the highest fieldstrength (channel 23) is selected
 * for synchronizing to frequency correction burst and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_first_channel_alr003()
{
  BEGIN_STEP ("find_bcch_carrier_first_channel_alr003")
  {

    SEND  ( mph_bsic_req(ARFCN_23_STD_900)  );
    AWAIT ( mphc_network_sync_req(ARFCN_23, SM_WIDE_MODE) );
    SEND  ( mphc_network_sync_ind_1()       );
    AWAIT ( mph_bsic_cnf_1()                );
    AWAIT ( mphc_new_scell_req_1()          );
    SEND  ( mphc_new_scell_con_1()          );
    AWAIT ( mphc_scell_nbcch_req_full()     );
  }
}

/*------------------------Following is for ALR004-------------------------------
 *
 * Description:	The carrier with the highest fieldstrength (channel 23) is selected for synchronizing to
 * frequency correction burst and synchron burst. The attempt failed. A second request is
 * started for the next strongest channel (channel 14).
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_first_channel_failed_then_second_channel_alr004()
{
  BEGIN_STEP ("find_bcch_carrier_first_channel_failed_then_second_channel_alr004")
  {
    SEND  ( mph_bsic_req(ARFCN_23_STD_900)         );
    AWAIT ( mphc_network_sync_req(ARFCN_23, SM_WIDE_MODE)        );
    SEND  ( mphc_network_sync_ind_failed(ARFCN_23) );
    AWAIT ( mph_bsic_cnf_failed(ARFCN_23_STD_900)  );
    SEND  ( mph_bsic_req(ARFCN_14_STD_900)         );
    AWAIT ( mphc_network_sync_req(ARFCN_14, SM_WIDE_MODE)        );
  }
}



/*------------------------Following is for ALR006-------------------------------
 *
 * Description:	The next best channel (channel 14) is selected for synchronizing to frequency correction
 * burst and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_second_channel_alr006()
{
  BEGIN_STEP ("find_bcch_carrier_second_channel_alr006")
  {
    SEND  ( mphc_network_sync_ind_6()   );
    AWAIT ( mph_bsic_cnf_6()            );
    AWAIT ( mphc_new_scell_req_2()      );
    SEND  ( mphc_new_scell_con_1()      );
    AWAIT ( mphc_scell_nbcch_req_full() );
  }
}



/*------------------------Following is for ALR007-------------------------------
 *
 * Description:	The BCCH data blocks are read for the channel 23.
 *-------------------------------------------------------------------------------*/

T_STEP read_bcch_data_alr007()
{
  BEGIN_STEP ("read_bcch_data_alr007")
  {
    SEND  ( mphc_data_ind_1()              );
    AWAIT ( mph_unitdata_ind_si1(ARFCN_23) );
    SEND  ( mphc_data_ind_2()              );
    AWAIT ( mph_unitdata_ind_si2(ARFCN_23) );
    SEND  ( mphc_data_ind_3()              );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_23) );
    SEND  ( mphc_data_ind_4()              );
    AWAIT ( mph_unitdata_ind_si4(ARFCN_23) );
  }
}


/*------------------------Following is for ALR009-------------------------------
 *
 * Description:	RR rejects the BCCH carrier. The next best channel (channel 124) is selected for
 * synchronizing to frequency correction burst and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP rr_rejects_bcch_carrier_try_third_channel_alr009()
{
  BEGIN_STEP ("rr_rejects_bcch_carrier_try_third_channel_alr009")
  {
    SEND  ( mph_bsic_req(ARFCN_124_STD_900)  );
    AWAIT ( mphc_stop_scell_bcch_req()       );
    AWAIT ( mphc_network_sync_req(ARFCN_124, SM_NARROW_MODE) );
    SEND  ( mphc_network_sync_ind_7()        );
    AWAIT ( mph_bsic_cnf_7()                 );
    AWAIT ( mphc_new_scell_req_3()           );
    SEND  ( mphc_new_scell_con_1()           );
    AWAIT ( mphc_scell_nbcch_req_full()      );
    SEND  ( mphc_data_ind_6()                );
    AWAIT ( mph_unitdata_ind_si1(ARFCN_124)  );
  }
}



/*------------------------Following is for ALR010-------------------------------
 *
 * Description:	RR rejects the BCCH carrier. The next best channel (channel 1) is selected for synchronising
 * to frequency correction burst and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP rr_rejects_bcch_carrier_try_fourth_channel_alr010()
{
  BEGIN_STEP ("rr_rejects_bcch_carrier_try_fourth_channel_alr010")
  {
    SEND  ( mph_bsic_req(ARFCN_1_STD_900)  );
    AWAIT ( mphc_stop_scell_bcch_req()     );
    AWAIT ( mphc_network_sync_req(ARFCN_1, SM_NARROW_MODE) );
    SEND  ( mphc_network_sync_ind_8()      );
    AWAIT ( mph_bsic_cnf_8()               );
    AWAIT ( mphc_new_scell_req_4()         );
    SEND  ( mphc_new_scell_con_1()         );
    AWAIT ( mphc_scell_nbcch_req_full()    );
    SEND  ( mphc_data_ind_7()              );
    AWAIT ( mph_unitdata_ind_si1(ARFCN_1)  );
  }
}



/*------------------------Following is for ALR012-------------------------------
 *
 * Description:	RR selects the channel 14 after reading the BCCH carrier.
 * Variant A: with neighbourcell list
 * Variant B: with empty neighbour cell list
 * Variant C: like A, but non-combined CCCH
 *-------------------------------------------------------------------------------*/

T_STEP rr_select_second_channel_alr012(CHAR variant)
{
  BEGIN_STEP ("rr_select_second_channel_alr012")
  {
    SEND  ( mph_classmark_req_1()              );
    SEND  ( mph_idle_req_1(variant)            );
    AWAIT ( mphc_stop_scell_bcch_req()         );
    SEND  ( mphc_stop_scell_bcch_con()         );
    AWAIT ( mphc_start_ccch_req_1(variant)     );
    AWAIT ( mphc_scell_nbcch_req_full()        );
    SEND  ( mph_identity_req_1()               );
    SEND  ( mph_cbch_req_1()                   );
    SEND  ( mph_neighbourcell_req_1(variant)   );
    AWAIT ( mphc_rxlev_periodic_req_1(variant) );
  }
}



/*------------------------Following is for ALR013-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 *-------------------------------------------------------------------------------*/

T_STEP rr_select_first_channel_alr013()
{
  BEGIN_STEP ("rr_select_first_channel_alr013")
  {
    SEND  ( mph_classmark_req_1()       );
    SEND  ( mph_idle_req_2()            );
    AWAIT ( mphc_stop_scell_bcch_req()  );
    AWAIT ( mphc_start_ccch_req_2()     );
    AWAIT ( mphc_scell_nbcch_req_full() );
    SEND  ( mph_identity_req_1()        );
    SEND  ( mph_cbch_req_1()            );
    SEND  ( mph_neighbourcell_req_2()   );
    AWAIT ( mphc_rxlev_periodic_req_2() );
  }
}


T_STEP generating_six_sync_failures()
{
  BEGIN_STEP ("rr_select_first_channel_alr013")
  {
    /* 1st Sync Failure */

    SEND  ( mph_bsic_req(ARFCN_124_STD_900)         );
    AWAIT(mphc_stop_scell_bcch_req()                );
    AWAIT (mphc_network_sync_req(ARFCN_124, SM_NARROW_MODE));
    SEND  ( mphc_network_sync_ind_failed(ARFCN_124) );
    AWAIT ( mph_bsic_cnf_failed(ARFCN_124_STD_900)  );

    /* 2nd Sync Failure */

    SEND  ( mph_bsic_req(ARFCN_23_STD_900)         );
    AWAIT (mphc_network_sync_req(ARFCN_23, SM_NARROW_MODE));
    SEND  ( mphc_network_sync_ind_failed(ARFCN_23) );
    AWAIT ( mph_bsic_cnf_failed(ARFCN_23_STD_900)  );

    /* 3rd Sync Failure */

    SEND  ( mph_bsic_req(ARFCN_124_STD_900)         );
    AWAIT (mphc_network_sync_req(ARFCN_124, SM_NARROW_MODE));
    SEND  ( mphc_network_sync_ind_failed(ARFCN_124) );
    AWAIT ( mph_bsic_cnf_failed(ARFCN_124_STD_900)  );

    /* 4th Sync Failure */

    SEND  ( mph_bsic_req(ARFCN_23_STD_900)         );
    AWAIT (mphc_network_sync_req(ARFCN_23, SM_NARROW_MODE));
    SEND  ( mphc_network_sync_ind_failed(ARFCN_23) );
    AWAIT ( mph_bsic_cnf_failed(ARFCN_23_STD_900)  );

    /* 5th Sync Failure */

    SEND  ( mph_bsic_req(ARFCN_124_STD_900)         );
    AWAIT (mphc_network_sync_req(ARFCN_124, SM_NARROW_MODE));
    SEND  ( mphc_network_sync_ind_failed(ARFCN_124) );
    AWAIT ( mph_bsic_cnf_failed(ARFCN_124_STD_900)  );

    /* 6th Sync Failure */

    SEND  ( mph_bsic_req(ARFCN_23_STD_900)         );
    AWAIT (mphc_network_sync_req(ARFCN_23, SM_NARROW_MODE));
    SEND  ( mphc_network_sync_ind_failed(ARFCN_23) );
    AWAIT ( mph_bsic_cnf_failed(ARFCN_23_STD_900)  );
  }
}


T_STEP sending_error_data_in_sacch()
{
  BEGIN_STEP ("sending_error_data_in_sacch")
  {
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
    SEND  (mphc_data_ind_sacch_error()    );
  }
}

/*--------------------------------------------------------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 *-------------------------------------------------------------------------------*/

T_STEP rr_select_first_channel_alr013_gprs()
{
  BEGIN_STEP ("rr_select_first_channel_alr013_gprs")
  {
    SEND  ( mph_classmark_req_1()       );
    SEND  ( mph_idle_req_1g()           );
    AWAIT ( mphc_stop_scell_bcch_req()  );
    AWAIT ( mphc_start_ccch_req_2()     );
    AWAIT ( mphc_scell_nbcch_req_full() );
    SEND  ( mph_identity_req_1()        );
    SEND  ( mph_cbch_req_1()            );
    SEND  ( mph_neighbourcell_req_2()   );
    AWAIT ( mphc_rxlev_periodic_req_2() );
  }
}
/*-------------------------------------------------------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 *              This is a cell with PBCCH.
 *-------------------------------------------------------------------------------*/

T_STEP rr_select_first_channel_gprs_pbcch(T_VAR_NMO var)
{
  BEGIN_STEP ("rr_select_first_channel_alr013_gprs")
  {
    SEND  ( mph_classmark_req_1()             );
    SEND  ( mph_idle_req_4g()                 );
    AWAIT ( mphc_stop_scell_bcch_req()        );
    SEND  ( mphc_stop_scell_bcch_con()        );
    SEND  ( mph_identity_req_1()              );
#if defined GPRS
    SEND  ( mph_mon_ctrl_req(ENTER_PIM_PBCCH) );
    if(var == VAR_NMO_3) {
      SEND  ( mph_mon_ctrl_req(START_MON_CCCH) );
      AWAIT ( mphc_start_ccch_req_8()          );
    }
#else
    AWAIT ( mphc_start_ccch_req_11('A')       );
    AWAIT ( mphc_scell_nbcch_req_full()       );
    SEND  ( mphc_data_ind_24('J')             );  // paging for an unknown TMSI to change page mode
    AWAIT ( mphc_start_ccch_req_8()           );
    AWAIT ( mphc_scell_nbcch_req_periodic()   );
    SEND  ( mph_neighbourcell_req_8()         );
    AWAIT ( mphc_rxlev_periodic_req_8()       );
#endif
  }
}
/*------------------------Following is for ALR020-------------------------------
 *
 * Description:	The idle mode is configured by RR. Layer 1 is configured with MPHC_START_CCCH_REQ.
 *-------------------------------------------------------------------------------*/

T_STEP initiation_with_paging_reorganisation_alr020(CHAR variant)
{
  BEGIN_STEP ("initiation_with_paging_reorganisation_alr020")
  {
    SEND  ( mph_idle_req_3(variant)        );
    AWAIT ( mphc_stop_scell_bcch_req()     );
    AWAIT ( mphc_start_ccch_req_3(variant) );
    AWAIT ( mphc_scell_nbcch_req_full()    );
  }
}



/*------------------------Following is for ALR022-------------------------------
 *
 * Description:	The idle mode is configured with Paging Reorganisation. After reception of a new paging
 * mode this information is forwarded to PL. The message contain the following page modi:
 * Variant A: Extended Paging
 * Variant B: Paging Reorganisation
 * Variant C: Same as before
 * No reaction is expected.
 *-------------------------------------------------------------------------------*/

T_STEP page_mode_change_paging_reorganisation_alr022(CHAR variant)
{
  BEGIN_STEP ("page_mode_change_paging_reorganisation_alr022")
  {
    SEND ( mphc_data_ind_8(variant) );
    if ( variant == 'A' )
    {
      SEND  ( mphc_data_ind_9() );
      AWAIT ( mphc_start_ccch_req(BS_PA_MFRMS_7, BS_AG_BLKS_RES_2, COMB_CCCH_COMB, CCCH_GROUP_0, PG_23, PBI_0, PGM_EXTENDED ) );
    }
    else
    {
      MUTE (3000);
    }
  }
}



/*------------------------Following is for ALR025-------------------------------
 *
 * Description:	The idle mode has been configured for normal paging. After a swap to Paging
 * Reorganisation RR re-configures normal paging.
 *-------------------------------------------------------------------------------*/

T_STEP page_mode_change_back_to_normal_paging_alr025()
{
  BEGIN_STEP ("page_mode_change_back_to_normal_paging_alr025")
  {
    SEND  ( mphc_data_ind_11()              );
    AWAIT ( mphc_start_ccch_req_6()         );
    AWAIT ( mphc_scell_nbcch_req_periodic() );
  }
}



/*------------------------Following is for ALR026-------------------------------
 *
 * Description:	The idle mode is configured with Normal Paging. The messages contain the following page
 * modi:
 * Variant A: Normal Paging
 * Variant B: Same as before.
 * No reaction is exspected.
 *-------------------------------------------------------------------------------*/

T_STEP page_mode_change_normal_paging_alr026(CHAR variant)
{
  BEGIN_STEP ("page_mode_change_normal_paging_alr026")
  {
    SEND ( mphc_data_ind_12(variant) );
    MUTE (3000);
  }
}



/*------------------------Following is for ALR027-------------------------------
 *
 * Description:	The idle mode has been configured for normal paging.
 *              After a swap to Extended Paging this mode is configured a second time.
 *              Layer 1 does not need to be configured the second time, because
 *              its a continues process in ALR(MPHC).
 * Variant A: Extended Paging
 * Variant B: Same as before
 *-------------------------------------------------------------------------------*/

T_STEP page_mode_change_extended_paging_a_second_time_alr027(CHAR variant)
{
  BEGIN_STEP ("page_mode_change_extended_paging_a_second_time_alr027")
  {
    SEND  ( mphc_data_ind_9()                );
    AWAIT ( mphc_start_ccch_req_4()          );
    AWAIT ( mphc_scell_nbcch_req_periodic()  );
    SEND  ( mphc_data_ind_13(variant)        );
    MUTE  (3000);
  }
}



/*------------------------Following is for ALR030-------------------------------
 *
 * Description:	The idle mode is configured for different multiframe periods.
 * Variant A:	bs_pa_mfrms = 2
 * Variant B:	bs_pa_mfrms = 3
 * Variant C:	bs_pa_mfrms = 4
 * Variant D:	bs_pa_mfrms = 5
 * Variant E:	bs_pa_mfrms = 6
 * Variant F:	bs_pa_mfrms = 7
 * Variant G:	bs_pa_mfrms = 8
 * Variant H:	bs_pa_mfrms = 9
 *-------------------------------------------------------------------------------*/

T_STEP measurement_reporting_configuration_alr030(CHAR variant)
{
  BEGIN_STEP ("measurement_reporting_configuration_alr030")
  {
    SEND  ( mph_idle_req_4(variant)        );
    AWAIT ( mphc_stop_scell_bcch_req()     );
    AWAIT ( mphc_start_ccch_req_7(variant) );
    AWAIT ( mphc_scell_nbcch_req_full()    );
    SEND  ( mph_neighbourcell_req_3()      );
    AWAIT ( mphc_rxlev_periodic_req_3()    );
  }
}


/*------------------------Following is for ALR9806-------------------------------*/ 

T_STEP measurement_reporting_configuration_alr9806()
{
  BEGIN_STEP ("measurement_reporting_configuration_alr9806")
  {
    SEND  ( mph_idle_req_3('B')        );
    AWAIT ( mphc_stop_scell_bcch_req()     );
    AWAIT ( mphc_start_ccch_req_14()       );
    AWAIT ( mphc_scell_nbcch_req_full()    );
    SEND  ( mph_neighbourcell_req_3()      );
    AWAIT ( mphc_rxlev_periodic_req_3()    );

    /* After Cell Reselection configuring the 
     * PL with the newly found cell 
     */
    SEND  ( mph_idle_req_resel()           );
    AWAIT ( mphc_stop_scell_bcch_req()     );
    AWAIT ( mphc_stop_ccch_req()           );
    AWAIT ( mphc_stop_rxlev_periodic_req() );
    AWAIT ( mphc_stop_scell_bcch_req()     );
    AWAIT ( mphc_new_scell_req_1           );

    /* After receiving the SI3 configuring the 
     * PL with the Paging Group.
     */
    SEND  ( mph_idle_req_cell_sel()        );
    AWAIT ( mphc_stop_scell_bcch_req()     );
    AWAIT ( mphc_start_ccch_req_14()       );
    AWAIT ( mphc_scell_nbcch_req_full()    );
    AWAIT ( mphc_rxlev_periodic_req_18()   );
  }
}



T_STEP measurement_reporting_configuration_nwctrl(CHAR variant)
{
  BEGIN_STEP ("measurement_reporting_configuration_nwctrl")
  {
    SEND  ( mph_idle_req_4(variant)                 );
    AWAIT ( mphc_stop_scell_bcch_req()              );
    AWAIT ( mphc_start_ccch_req_7(variant)          );
    AWAIT ( mphc_scell_nbcch_req_full()             );
    SEND  ( mph_neighbourcell_req_2()               );
    AWAIT ( mphc_rxlev_periodic_req_2()             );
    SEND  ( mphc_rxlev_periodic_ind_9(50,40,30,20)  );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)    );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)   );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)  );
    SEND  ( mphc_ncell_sync_ind(ARFCN_1,FN_OFF_ARFCN_1,TIME_ALIGNMT_1,BSIC_16));
    SEND  ( mphc_ncell_sync_ind(ARFCN_14,FN_OFF_ARFCN_14,TIME_ALIGNMT_14,BSIC_16));
    SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1));
    AWAIT ( mphc_ncell_bcch_req(ARFCN_1,FN_OFF_ARFCN_1,TIME_ALIGNMT_1,BSIC_16));
    AWAIT ( mphc_ncell_bcch_req(ARFCN_14,FN_OFF_ARFCN_14,TIME_ALIGNMT_14,BSIC_16));
    AWAIT ( mphc_ncell_bcch_req(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1));
    SEND  ( mphc_rxlev_periodic_ind_9(50,40,30,20)  );
    MUTE(500);
  }
}

/*------------------------Following is for ALR046-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 * Variant A: with neighbourcell list (serving cell not included)
 * Variant B: with neighbourcell list (serving cell included)
 * Variant C: with empty neighbour cell list.
 * Variant D: same like A, but with ncc_permitted = 4
 *-------------------------------------------------------------------------------*/

T_STEP definition_of_bcch_allocation_alr046(CHAR variant)
{
  BEGIN_STEP ("definition_of_bcch_allocation_alr046")
  {
    SEND  ( mph_classmark_req_1()              );
    SEND  ( mph_idle_req_5(variant)            );
    AWAIT ( mphc_stop_scell_bcch_req()         );
    AWAIT ( mphc_start_ccch_req_9()            );
    AWAIT ( mphc_scell_nbcch_req_full()        );
    SEND  ( mph_identity_req_1()               );
    SEND  ( mph_cbch_req_1()                   );
    SEND  ( mph_neighbourcell_req_4(variant)   );
    AWAIT ( mphc_rxlev_periodic_req_4(variant) );
  }
}

/*------------------------Following is for ALR055-------------------------------
 *
 * Description:	RR starts the sending of channel request messages. Idle mode is stopped and the access
 * mode is started. There is one retransmission configured.
 *-------------------------------------------------------------------------------*/

T_STEP start_of_sending_channel_request_messages_gsm_900__alr055()
{
  BEGIN_STEP ("start_of_sending_channel_request_messages_gsm_900__alr055")
  {
    SEND  ( mph_random_access_req_1()        );
    AWAIT ( mphc_stop_rxlev_periodic_req()   );
    AWAIT ( mphc_stop_scell_bcch_req()       );
    AWAIT ( mphc_scell_nbcch_req_full()      );
    AWAIT ( mphc_ra_req_1()                  );
    SEND  ( mphc_ra_con_1()                  );
    AWAIT ( mph_random_access_cnf_1()        );
    AWAIT ( mphc_ra_req_2()                  );
  }
}



/*------------------------Following is for ALR056-------------------------------
 *
 * Description:	An immediate assignment for the mobile station receives. Configuration of the dedicated
 * mode is processed by RR. This stops the access link mode.
 *-------------------------------------------------------------------------------*/

T_STEP immediate_assignment_for_the_mobile_station_alr056()
{
  BEGIN_STEP ("immediate_assignment_for_the_mobile_station_alr056")
  {
    SEND  ( mphc_data_ind_20()           );
    AWAIT ( mph_unitdata_ind_10()        );
    SEND  ( mph_dedicated_req_1()        );
    AWAIT ( mphc_stop_scell_bcch_req()   );
    AWAIT ( mphc_stop_ccch_req()         );
    AWAIT ( mphc_stop_ra_req_1()         );
    AWAIT ( mphc_immed_assign_req_1()    );
    SEND  ( mphc_immed_assign_con_1()    );
    AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );
  }
}



/*------------------------Following is for ALR062-------------------------------
 *
 * Description:	The extended measurement procedure according GSM 05.08, section 10.1.5 is started.
 *
 *-------------------------------------------------------------------------------*/

T_STEP extended_measurement_reporting_start_procedure_alr062()
{
  BEGIN_STEP ("extended_measurement_reporting_start_procedure_alr062")
  {
    SEND  ( mph_ext_meas_req_1() );
    AWAIT ( mphc_init_l1_req_2() );
    SEND  ( mphc_init_l1_con_1() );
    AWAIT ( mphc_rxlev_req()     );
  }
}



/*------------------------Following is for ALR063-------------------------------
 *
 * Description:	The extended measurement procedure passes the step of the power measurement.
 *
 *-------------------------------------------------------------------------------*/

T_STEP extended_measurement_reporting_successful_power_measurement_alr063()
{
  BEGIN_STEP ("extended_measurement_reporting_successful_power_measurement_alr063")
  {
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );

    AWAIT ( mphc_rxlev_req()     );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );

    AWAIT ( mphc_rxlev_req()     );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );

    AWAIT ( mphc_rxlev_req()     );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );

    AWAIT ( mphc_rxlev_req()     );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );

    AWAIT ( mph_ext_meas_cnf_1()                    );
    SEND  ( mph_bsic_req(CONST_1023)                );
    AWAIT ( mphc_ncell_sync_req_initial(CONST_1023) );

  /*if page mode = REORG, L1 would not react,
    therefore page mode NORMAL is configured and
    BCCH reading is reduced*/
    stop_page_mode_reorg_before_hplnm_search_procedure();
  }
}



/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 * Variant A: IMSI mobile identity 1, any channel needed
 * Variant B: IMSI mobile identity 2, any channel needed
 * Variant C: TMSI mobile identity 1, TCH/F channel needed
 * Variant D: TMSI mobile identity 2, TCH/F channel needed
 * Variant E: IMSI mobile identity 1, SDCCH channel needed
 * Variant F: IMSI mobile identity 2, SDCCH channel needed
 * Variant G: TMSI mobile identity 1, Dual Rate channel needed
 * Variant H: TMSI mobile identity 2, Dual Rate channel needed
 *-------------------------------------------------------------------------------*/

T_STEP paging_req_1_imsi_tmsi_for_ms_alr071(CHAR variant)
{
  BEGIN_STEP ("paging_req_1_imsi_tmsi_for_ms_alr071")
  {
    SEND  ( mphc_data_ind_24(variant)        );
    AWAIT ( mph_paging_ind_1(variant)        );
    AWAIT ( mphc_start_ccch_req_8()          );
    AWAIT ( mphc_scell_nbcch_req_periodic()  );
  }
}



/*------------------------Following is for ALR072-------------------------------
 *
 * Description:	The MS is in idle mode. It receives paging request type 1 messages.
 *              The message content is not forwarded to RR, because the message
 *              content doesnt match to the MS identities.
 * Variant A: wrong IMSI mobile identity 1
 * Variant B: wrong TMSI mobile identity 1
 * Variant C: wrong IMSI mobile identity 2
 * Variant D: wrong TMSI mobile identity 2
 * Variant E: wrong type mobile identity 2
 *-------------------------------------------------------------------------------*/

T_STEP paging_req_1_not_for_ms_alr072(CHAR variant)
{
  BEGIN_STEP ("paging_req_1_not_for_ms_alr072")
  {
    SEND ( mphc_data_ind_25(variant) );
  }
}



/*------------------------Following is for ALR073-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding to the
 *              actual IMSI or TMSI. A paging indication is forwarded to RR.
 *              The IMSI is less than 15 digits and the TMSI has only one valid byte,
 *              but is signalled in different ways by the infrastructure
 * Variant A: Short TMSI mobile identity 1, any channel needed, 1 byte length from the network
 * Variant B: Short TMSI mobile identity 2, TCH/F needed, 3 bytes length from the network
 * Variant C: Short IMSI mobile identity 1, SDCCH needed, 10 digits
 * Variant D: Short IMSI mobile identity 2, Dual Rate needed, 10 digits
 * Variant E: Short TMSI mobile identity 1, any channel needed, 2 bytes length from the network
 * Variant F: Short TMSI mobile identity 2, TCH/F needed, 4 bytes length from the network
 *-------------------------------------------------------------------------------*/

T_STEP paging_req_1_short_imsi_tmsi_for_ms_alr073(CHAR variant)
{
  BEGIN_STEP ("paging_req_1_short_imsi_tmsi_for_ms_alr073")
  {
    SEND  ( mph_identity_req_2()             );
    SEND  ( mphc_data_ind_26(variant)        );
    AWAIT ( mph_paging_ind_2(variant)        );
    AWAIT ( mphc_start_ccch_req_8()          );
    AWAIT ( mphc_scell_nbcch_req_periodic()  );
  }
}



/*------------------------Following is for ALR075-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 or 2.
 *              A paging indication is forwarded to RR.
 * Variant A: TMSI mobile identity 1, any channel needed
 * Variant B: TMSI mobile identity 1, SDCCH needed
 * Variant C: TMSI mobile identity 1, TCH/F needed
 * Variant D: TMSI mobile identity 1, Dual Rate needed
 * Variant E: TMSI mobile identity 2, any channel needed
 * Variant F: TMSI mobile identity 2, SDCCH needed
 * Variant G: TMSI mobile identity 2, TCH/F needed
 * Variant H: TMSI mobile identity 2, Dual Rate needed
 *-------------------------------------------------------------------------------*/

T_STEP paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075(CHAR variant)
{
  BEGIN_STEP ("paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075")
  {
    SEND  ( mphc_data_ind_28(variant)       );
    AWAIT ( mph_paging_ind_3(variant)       );
    AWAIT ( mphc_start_ccch_req_8()         );
    AWAIT ( mphc_scell_nbcch_req_periodic() );
  }
}



/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant A: IMSI mobile identity 3, any channel needed
 * Variant B: IMSI mobile identity 3, SDCCH needed
 * Variant C: IMSI mobile identity 3, TCH/F needed
 * Variant D: IMSI mobile identity 3, Dual Rate needed
 * Variant E: IMSI mobile identity 3, no channel indication
 * Variant F: TMSI mobile identity 3, any channel needed
 * Variant G: TMSI mobile identity 3, SDCCH needed,
 * Variant H: TMSI mobile identity 3, TCH/F needed
 * Variant I: TMSI mobile identity 3, Dual Rate needed
 * Variant J: TMSI mobile identity 3, no channel indication
 *-------------------------------------------------------------------------------*/

T_STEP paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077(CHAR variant)
{
  BEGIN_STEP ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077")
  {
    SEND  ( mphc_data_ind_30(variant)       );
    AWAIT ( mph_paging_ind_4(variant)       );
    AWAIT ( mphc_start_ccch_req_8()         );
    AWAIT ( mphc_scell_nbcch_req_periodic() );
  }
}



/*------------------------Following is for ALR078-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR. The IMSI is less than 15 digits and the TMSI has only
 *              one valid byte, but is signalled in different ways by the infrastructure
 * Variant A: Short IMSI mobile identity 3, any channel needed, 10 digits
 * Variant B: Short TMSI mobile identity 3, SDCCH needed, 1 byte from the network
 * Variant C: Short TMSI mobile identity 3, TCH/F needed, 2 bytes from the network
 * Variant D: Short TMSI mobile identity 3, Dual Rate needed, 3 bytes from the network
 * Variant E: Short TMSI mobile identity 3, No channel indication, 4 bytes from the network
 *-------------------------------------------------------------------------------*/

T_STEP paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078(CHAR variant)
{
  BEGIN_STEP ("paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078")
  {
    SEND  ( mph_identity_req_2()            );
    SEND  ( mphc_data_ind_31(variant)       );
    AWAIT ( mph_paging_ind_5(variant)       );
    AWAIT ( mphc_start_ccch_req_8()         );
    AWAIT ( mphc_scell_nbcch_req_periodic() );
  }
}



/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant A: TMSI mobile identity 1, any channel needed
 * Variant B: TMSI mobile identity 1, SDCCH needed
 * Variant C: TMSI mobile identity 1, TCH/F needed
 * Variant D: TMSI mobile identity 1, Dual Rate needed
 * Variant E: TMSI mobile identity 2, any channel needed
 * Variant F: TMSI mobile identity 2, SDCCH needed
 * Variant G: TMSI mobile identity 2, TCH/F needed
 * Variant H: TMSI mobile identity 2, Dual Rate needed
 * Variant I: TMSI mobile identity 3, any channel needed
 * Variant J: TMSI mobile identity 3, SDCCH needed
 * Variant K: TMSI mobile identity 3, TCH/F needed
 * Variant L: TMSI mobile identity 3, Dual Rate needed
 * Variant M: TMSI mobile identity 3, no channel indication
 * Variant N: TMSI mobile identity 4, any channel needed
 * Variant O: TMSI mobile identity 4, SDCCH needed
 * Variant P: TMSI mobile identity 4, TCH/F needed
 * Variant Q: TMSI mobile identity 4, Dual Rate needed
 * Variant R: TMSI mobile identity 4, no channel indication
 *-------------------------------------------------------------------------------*/

T_STEP paging_req_3_tmsi_for_ms_alr080(CHAR variant)
{
  BEGIN_STEP ("paging_req_3_tmsi_for_ms_alr080")
  {
    SEND  ( mphc_data_ind_33(variant)       );
    AWAIT ( mph_paging_ind_6(variant)       );
    AWAIT ( mphc_start_ccch_req_8()         );
    AWAIT ( mphc_scell_nbcch_req_periodic() );
  }
}

/*------------------------Following is for ALR088-------------------------------
 *
 * Description: ALR starts ncell measurements for cells 23, 14, 124, 1,
 *              then RR requests a full list power measurement. Four channels
 *              in the order 23, 14, 124, 1 have acceptable fieldstrength values.
 *              Then the non-serving cell carrier with the highest fieldstrength
 *              (channel 14) is selected for synchronizing to frequency correction
 *              burst and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_second_channel_alr088()
{
  BEGIN_STEP ("find_bcch_carrier_second_channel_alr088")
  {
    SEND  ( mphc_rxlev_periodic_ind_2()            );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)  );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124) );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)   );
    SEND  ( mphc_rxlev_periodic_ind_2()            );
    AWAIT ( mph_measurement_ind_18()               );
    SEND  ( mphc_rxlev_periodic_ind_2()            );
    SEND  ( mphc_ncell_sync_ind (ARFCN_14)         );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_14)         );
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)      );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)     );
    SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
    SEND  ( mphc_ncell_sync_ind_3()                );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1));
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124)     );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)    );
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1)       );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)      );
    SEND  ( mphc_rxlev_periodic_ind_2()            );
    AWAIT ( mph_measurement_ind_19()               );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_1)          );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_14)         );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_124)        );
    SEND  ( mph_power_req_2()                      );
    AWAIT ( mphc_rxlev_req()                       );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)            );
    AWAIT ( mphc_rxlev_req()                       );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)            );
    AWAIT ( mphc_rxlev_req()                       );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)            );
    AWAIT ( mphc_rxlev_req()                       );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)            );
    AWAIT ( mphc_rxlev_req()                       );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)            );
    AWAIT ( mph_power_cnf_3()                      );
    SEND  ( mph_bsic_req(ARFCN_14)                 );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)  );

    stop_page_mode_reorg_before_hplnm_search_procedure();
  }
}

/*------------------------Following is for ALR091-------------------------------
 *
 * Description:	The BCCH data blocks are read for the channel 14.
 *-------------------------------------------------------------------------------*/

T_STEP read_bcch_data_alr091()
{
  BEGIN_STEP ("read_bcch_data_alr091")
  {
    SEND  ( mphc_ncell_sync_ind (ARFCN_14)        );
    AWAIT ( mph_bsic_cnf_14()                     );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_14)        );
    SEND  ( mphc_ncell_bcch_ind_si2(ARFCN_14)     );
    AWAIT ( mph_unitdata_ind_si2(ARFCN_14)        );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)    );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_14)        );
    SEND  ( mphc_ncell_bcch_ind_invalid(ARFCN_14) );
    AWAIT ( mph_error_ind_3()                     );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)    );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_14)        );
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)     );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_14)        );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)    );
  }
}

/*------------------------Following is for ALR094-------------------------------
 *
 * Description:	RR rejects the BCCH carrier. The next best channel (channel 124)
 *              is selected for synchronizing to frequency correction
 *              burst and synchron burst. This is also rejedted. The next best
 *              channel (channel 1) is selected for synchronising
 *              to frequency correction burst and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP rr_rejects_bcch_carrier_try_fourth_channel_alr094()
{
  BEGIN_STEP ("rr_rejects_bcch_carrier_try_fourth_channel_alr094")
  {
    SEND  ( mph_bsic_req(ARFCN_124)                );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124) );
    SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                );
    AWAIT ( mph_bsic_cnf_15()                      );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124)     );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_124)        );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)    );
    SEND  ( mph_bsic_req(ARFCN_1)                  );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)   );
    SEND  ( mphc_ncell_sync_ind_3()                );
    AWAIT ( mph_bsic_cnf_16()                      );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1));
    SEND  ( mphc_ncell_bcch_ind_si2(ARFCN_1)       );
    AWAIT ( mph_unitdata_ind_si2(ARFCN_1)          );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)      );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1));
  }
}

/*------------------------Following is for ALR154-------------------------------
 *
 * Description:	A handover is performed. It is checked that the serving cell
 *              channel numbers are correctly set.
 * Variant A:	AMR not signalled
 * Variant B:	AMR signalled
 *-------------------------------------------------------------------------------*/

T_STEP handover_serving_cell_parameter_alr154(CHAR variant)
{
  BEGIN_STEP ("handover_serving_cell_parameter_alr154")
  {
    SEND  ( mph_dedicated_req_6(variant) );
    AWAIT ( mphc_async_ho_req_3(variant) );
    SEND  ( mphc_async_ho_con_1()        );
    SEND  ( mphc_handover_finished_1(HO_COMPLETE) );
    AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK)      );
  }
}

/*------------------------Following is for ALR155-------------------------------
 *
 * Description:	The MS has entered the dedicated mode. The BA list is updated
 *              by neighbour cell requests generated by RR from SI 5
 *              Every frequency of the MPH_NEIGHBOURCELL_REQ must occur within
 *              the MPHC_UPDATE_BA_LIST plus the serving cell (if not already
 *              included). In the MPHC_UPDATE_BA_LIST the first up to 8
 *              frequencies must be in the same order as in the
 *              MPH_NEIGHBOURCELL_REQ, the order of the other depends on the
 *              replacing policy of ALR.
 *-------------------------------------------------------------------------------*/
T_STEP neighbour_cell_in_dedicated_alr155()
{
  BEGIN_STEP ("neighbour_cell_in_dedicated_alr155")
  {
    SEND  ( mph_neighbourcell_req_16() );
    AWAIT ( mphc_update_ba_list_4()    );
    SEND  ( mph_neighbourcell_req_17() );
    AWAIT ( mphc_update_ba_list_5()    );
  }
}

/*------------------------Following is for ALR155-------------------------------
 *
 * Description:	A handover has been performed. The BA list is updated for the
 *              new cell
 *              Every frequency of the MPH_NEIGHBOURCELL_REQ must occur within
 *              the MPHC_UPDATE_BA_LIST plus the serving cell (if not already
 *              included). In the MPHC_UPDATE_BA_LIST the first up to 8
 *              frequencies must be in the same order as in the
 *              MPH_NEIGHBOURCELL_REQ, the order of the other depends on the
 *              replacing policy of ALR.
 *-------------------------------------------------------------------------------*/
T_STEP neighbour_cell_after_ho_alr156()
{
  BEGIN_STEP ("neighbour_cell_after_ho_alr156")
  {
    SEND  ( mph_neighbourcell_req_18() );
    AWAIT ( mphc_update_ba_list_6()    );
    SEND  ( mph_neighbourcell_req_19() );
    AWAIT ( mphc_update_ba_list_7()    );
  }
}

/*------------------------Following is for ALR200-------------------------------
 *
 * Description:	The ALR is configured for DCS 1800.
 *-------------------------------------------------------------------------------*/

T_STEP filter_and_routings_dcs_1800__alr200()
{
  BEGIN_STEP ("filter_and_routings_dcs_1800__alr200")
  {
    //COMMAND("TAP RESET");
    //COMMAND("PL RESET");
    COMMAND("TAP REDIRECT CLEAR");
    COMMAND("PL REDIRECT CLEAR");
    COMMAND("PL REDIRECT RR TAP");
    COMMAND("PL REDIRECT DL TAP");
    COMMAND("PL REDIRECT MMI TAP");
    COMMAND("PL REDIRECT L1 TAP");
    COMMAND("TAP REDIRECT TAP PL");
    COMMAND("PL CONFIG STD=4");
    SEND  ( em_l1_event_req_2()  );
    //COMMAND("TAP PARKING ");
    COMMAND("PL REDIRECT MMI *1111110******** NULL");

  }
}



/*------------------------Following is for ALR201-------------------------------
 *
 * Description:	RR starts a cell selection. Four channels in the order 637, 580,
 *              885, 512 have acceptable fieldstrength values.
 *-------------------------------------------------------------------------------*/

T_STEP initiation_by_rr_4_channels_available_alr201()
{
  BEGIN_STEP ("initiation_by_rr_4_channels_available_alr201")
  {
    filter_and_routings_dcs_1800__alr200();

    SEND  ( mph_power_req_3()                );
    AWAIT ( mphc_init_l1_req_3()             );
    SEND  ( mphc_init_l1_con_1()             );
    AWAIT ( mphc_rxlev_req()                 );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800) );
    AWAIT ( mphc_rxlev_req()                 );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800) );
    AWAIT ( mphc_rxlev_req()                 );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800) );
    AWAIT ( mphc_rxlev_req()                 );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800) );
    AWAIT ( mphc_rxlev_req()                 );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800) );
    AWAIT ( mph_power_cnf_4()                );
  }
}



/*------------------------Following is for ALR203-------------------------------
 *
 * Description:	The carrier with the highest fieldstrength (channel 637) is
 *              selected for synchronizing to frequency correction burst and
 *              synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_first_channel_alr203()
{
  BEGIN_STEP ("find_bcch_carrier_first_channel_alr203")
  {
    SEND  ( mph_bsic_req(ARFCN_637_STD_1800) );
    AWAIT ( mphc_network_sync_req(ARFCN_637, SM_WIDE_MODE) );
    SEND  ( mphc_network_sync_ind_9()        );
    AWAIT ( mph_bsic_cnf_17()                );
    AWAIT ( mphc_new_scell_req_5()           );
    SEND  ( mphc_new_scell_con_1()           );
    AWAIT ( mphc_scell_nbcch_req_full()      );
  }
}



/*------------------------Following is for ALR204-------------------------------
 *
 * Description:	The carrier with the highest fieldstrength (channel 637)
 *              is selected for synchronizing to frequency correction burst and
 *              synchron burst. The attempt failed. A second request is started
 *              for the next strongest channel (channel 580).
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_first_channel_failed_then_second_channel_alr204()
{
  BEGIN_STEP ("find_bcch_carrier_first_channel_failed_then_second_channel_alr204")
  {
    SEND  ( mph_bsic_req(ARFCN_637_STD_1800)        );
    AWAIT ( mphc_network_sync_req(ARFCN_637, SM_WIDE_MODE)        );
    SEND  ( mphc_network_sync_ind_failed(ARFCN_637) );
    AWAIT ( mph_bsic_cnf_failed(ARFCN_637_STD_1800) );
    SEND  ( mph_bsic_req(ARFCN_580_STD_1800)        );
    AWAIT ( mphc_network_sync_req(ARFCN_580, SM_WIDE_MODE)        );
  }
}



/*------------------------Following is for ALR206-------------------------------
 *
 * Description:	The next best channel (channel 14) is selected for synchronizing
 * to frequency correction burst and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_second_channel_alr206()
{
  BEGIN_STEP ("find_bcch_carrier_second_channel_alr206")
  {
    SEND  ( mphc_network_sync_ind_14()  );
    AWAIT ( mph_bsic_cnf_22()           );
    AWAIT ( mphc_new_scell_req_6()      );
    SEND  ( mphc_new_scell_con_1()      );
    AWAIT ( mphc_scell_nbcch_req_full() );
  }
}



/*------------------------Following is for ALR207-------------------------------
 *
 * Description:	The BCCH data blocks are read for the channel 23.
 *-------------------------------------------------------------------------------*/

T_STEP read_bcch_data_alr207()
{
  BEGIN_STEP ("read_bcch_data_alr207")
  {
    SEND  ( mphc_data_ind_42()              );
    AWAIT ( mph_unitdata_ind_si1(ARFCN_637) );
    SEND  ( mphc_data_ind_43()              );
    AWAIT ( mph_unitdata_ind_si2(ARFCN_637) );
    SEND  ( mphc_data_ind_44()              );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_637) );
    SEND  ( mphc_data_ind_45()              );
    AWAIT ( mph_unitdata_ind_si4(ARFCN_637) );
  }
}



/*------------------------Following is for ALR209-------------------------------
 *
 * Description:	RR rejects the BCCH carrier. The next best channel (channel 124)
 *              is selected for synchronizing to frequency correction burst
 *              and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP rr_rejects_bcch_carrier_try_third_channel_alr209()
{
  BEGIN_STEP ("rr_rejects_bcch_carrier_try_third_channel_alr209")
  {
    SEND  ( mph_bsic_req(ARFCN_885_STD_1800) );
    AWAIT ( mphc_stop_scell_bcch_req()       );
    AWAIT ( mphc_network_sync_req(ARFCN_885, SM_NARROW_MODE) );
    SEND  ( mphc_network_sync_ind_15()       );
    AWAIT ( mph_bsic_cnf_23()                );
    AWAIT ( mphc_new_scell_req_7()           );
    SEND  ( mphc_new_scell_con_1()           );
    AWAIT ( mphc_scell_nbcch_req_full()      );
    SEND  ( mphc_data_ind_47()               );
    AWAIT ( mph_unitdata_ind_si1(ARFCN_885)  );
  }
}



/*------------------------Following is for ALR210-------------------------------
 *
 * Description:	RR rejects the BCCH carrier. The next best channel (channel 1)
 *              is selected for synchronising to frequency correction burst
 *              and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP rr_rejects_bcch_carrier_try_fourth_channel_alr210()
{
  BEGIN_STEP ("rr_rejects_bcch_carrier_try_fourth_channel_alr210")
  {
    SEND  ( mph_bsic_req(ARFCN_512_STD_1800) );
    AWAIT ( mphc_stop_scell_bcch_req()       );
    AWAIT ( mphc_network_sync_req(ARFCN_512, SM_NARROW_MODE) );
    SEND  ( mphc_network_sync_ind_16()       );
    AWAIT ( mph_bsic_cnf_24()                );
    AWAIT ( mphc_new_scell_req_8()           );
    SEND  ( mphc_new_scell_con_1()           );
    AWAIT ( mphc_scell_nbcch_req_full()      );
    SEND  ( mphc_data_ind_48()               );
    AWAIT ( mph_unitdata_ind_si1(ARFCN_512)  );
  }
}



/*------------------------Following is for ALR212-------------------------------
 *
 * Description:	RR selects the channel 14 after reading the BCCH carrier.
 * Variant A: with neighbourcell list
 * Variant B: with empty neighbour cell list
 * Variant C: like A, but non-combined CCCH
 *-------------------------------------------------------------------------------*/

T_STEP rr_select_second_channel_alr212(CHAR variant)
{
  BEGIN_STEP ("rr_select_second_channel_alr212")
  {
    SEND  ( mph_classmark_req_2()              );
    SEND  ( mph_idle_req_7(variant)            );
    AWAIT ( mphc_stop_scell_bcch_req()         );
    AWAIT ( mphc_start_ccch_req_11(variant)    );
    AWAIT ( mphc_scell_nbcch_req_full()        );
    SEND  ( mph_identity_req_1()               );
    SEND  ( mph_cbch_req_1()                   );
    SEND  ( mph_neighbourcell_req_5(variant)   );
    AWAIT ( mphc_rxlev_periodic_req_5(variant) );
  }
}

/*------------------------Following is for ALR600-------------------------------
 *
 * Description:	The ALR is configured for Dualband GSM 900 / DCS 1800.
 *-------------------------------------------------------------------------------*/

T_STEP filter_and_routings_dualband_gsm_900_dcs_1800__alr600()
{
  BEGIN_STEP ("filter_and_routings_dualband_gsm_900_dcs_1800__alr600")
  {
    //COMMAND("TAP RESET");
    //COMMAND("PL RESET");
    COMMAND("TAP REDIRECT CLEAR");
    COMMAND("PL REDIRECT CLEAR");
    COMMAND("PL REDIRECT RR TAP");
    COMMAND("PL REDIRECT DL TAP");
    COMMAND("PL REDIRECT MMI TAP");
    COMMAND("PL REDIRECT L1 TAP");
    COMMAND("TAP REDIRECT TAP PL");
#ifdef  GPRS
    COMMAND("PL REDIRECT GRR TAP");
#endif // GPRS
    COMMAND("PL CONFIG STD=5");
    SEND  ( em_l1_event_req_2()  );
    //COMMAND("TAP PARKING ");
    COMMAND("PL REDIRECT MMI *1111110******** NULL");

  }
}



/*------------------------Following is for ALR601-------------------------------
 *
 * Description:	RR starts a cell selection. Eight channels in the order 23, 637, 14, 580, 124, 885, 1, 512 have
 * acceptable fieldstrength values.
 *-------------------------------------------------------------------------------*/

T_STEP initiation_by_rr_8_channels_available_alr601()
{
  BEGIN_STEP ("initiation_by_rr_8_channels_available_alr601")
  {
    filter_and_routings_dualband_gsm_900_dcs_1800__alr600();

    SEND  ( mph_power_req_4()              );
    AWAIT ( mphc_init_l1_req_4()           );
    SEND  ( mphc_init_l1_con_1()           );
    AWAIT ( mphc_rxlev_req()               );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL) );
    AWAIT ( mphc_rxlev_req()               );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL) );
    AWAIT ( mphc_rxlev_req()               );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL) );
    AWAIT ( mphc_rxlev_req()               );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL) );
    AWAIT ( mphc_rxlev_req()               );
    SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL) );
    AWAIT ( mph_power_cnf_5()              );
  }
}



/*------------------------Following is for ALR603-------------------------------
 *
 * Description:	The carrier with the highest fieldstrength (channel 23) is selected
 *              for synchronizing to frequency correction burst and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_first_channel_alr603()
{
  BEGIN_STEP ("find_bcch_carrier_first_channel_alr603")
  {
    SEND  ( mph_bsic_req(ARFCN_23_STD_DUAL) );
    AWAIT ( mphc_network_sync_req(ARFCN_23, SM_WIDE_MODE) );
    SEND  ( mphc_network_sync_ind_1()       );
    AWAIT ( mph_bsic_cnf_25()               );
    AWAIT ( mphc_new_scell_req_1()          );
    SEND  ( mphc_new_scell_con_1()          );
    AWAIT ( mphc_scell_nbcch_req_full()     );
  }
}

/*------------------------Following is for ALR604-------------------------------
 *
 * Description:	The carrier with the highest fieldstrength (channel 23) is selected
 *              for synchronizing to frequency correction burst and synchron burst.
 *              The attempt failed. A second request is started for the next
 *              strongest channel (channel 637).
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_first_channel_failed_then_second_channel_alr604()
{
  BEGIN_STEP ("find_bcch_carrier_first_channel_failed_then_second_channel_alr604")
  {
    SEND  ( mph_bsic_req(ARFCN_23_STD_DUAL)        );
    AWAIT ( mphc_network_sync_req(ARFCN_23, SM_WIDE_MODE)        );
    SEND  ( mphc_network_sync_ind_failed(ARFCN_23) );
    AWAIT ( mph_bsic_cnf_failed(ARFCN_23_STD_DUAL) );
    SEND  ( mph_bsic_req(ARFCN_637_STD_DUAL)       );
    AWAIT ( mphc_network_sync_req(ARFCN_637, SM_WIDE_MODE)       );
  }
}



/*------------------------Following is for ALR606-------------------------------
 *
 * Description:	The next best channel (channel 637) is selected for synchronizing
 *              to frequency correction burst and synchron burst.
 *-------------------------------------------------------------------------------*/

T_STEP find_bcch_carrier_second_channel_alr606()
{
  BEGIN_STEP ("find_bcch_carrier_second_channel_alr606")
  {
    SEND  ( mphc_network_sync_ind_17()  );
    AWAIT ( mph_bsic_cnf_33()           );
    AWAIT ( mphc_new_scell_req_9()      );
    SEND  ( mphc_new_scell_con_1()      );
    AWAIT ( mphc_scell_nbcch_req_full() );
  }
}



/*------------------------Following is for ALR607-------------------------------
 *
 * Description:	The BCCH data blocks are read for the channel 23.
 *-------------------------------------------------------------------------------*/

T_STEP read_bcch_data_alr607()
{
  BEGIN_STEP ("read_bcch_data_alr607")
  {
    SEND  ( mphc_data_ind_1()              );
    AWAIT ( mph_unitdata_ind_si1(ARFCN_23) );
    SEND  ( mphc_data_ind_2()              );
    AWAIT ( mph_unitdata_ind_si2(ARFCN_23) );
    SEND  ( mphc_data_ind_3()              );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_23) );
    SEND  ( mphc_data_ind_4()              );
    AWAIT ( mph_unitdata_ind_si4(ARFCN_23) );
  }
}

/*------------------------Following is for ALR612-------------------------------
 *
 * Description:	RR selects the channel 637 after reading the BCCH carrier.
 * Variant A: with neighbourcell list
 * Variant B: with empty neighbour cell list
 *-------------------------------------------------------------------------------*/

T_STEP rr_select_second_channel_alr612(CHAR variant)
{
  BEGIN_STEP ("rr_select_second_channel_alr612")
  {
    SEND  ( mph_classmark_req_3()              );
    SEND  ( mph_idle_req_9()                   );
    AWAIT ( mphc_stop_scell_bcch_req()         );
    AWAIT ( mphc_start_ccch_req_10()           );
    AWAIT ( mphc_scell_nbcch_req_full()        );
    SEND  ( mph_identity_req_1()               );
    SEND  ( mph_cbch_req_1()                   );
    SEND  ( mph_neighbourcell_req_7(variant)   );
    AWAIT ( mphc_rxlev_periodic_req_7(variant) );
  }
}

/*------------------------Following is for ALR800-------------------------------
 *
 * Description:	The CBCH channel is configured. Then MMI requests reading of CBCH
 *              for the message identifier 3, 7 and 11 to 13.
 *-------------------------------------------------------------------------------*/

T_STEP configuration_cbch_followed_by_mmi_request_alr800(CHAR variant)
{
  BEGIN_STEP ("configuration_cbch_followed_by_mmi_request_alr800")
  {
    SEND  ( mph_classmark_req_1()           );
    SEND  ( mph_idle_req_14()               );
    AWAIT ( mphc_stop_scell_bcch_req()      );
    AWAIT ( mphc_start_ccch_req_10()        );
    AWAIT ( mphc_scell_nbcch_req_full()     );
    SEND  ( mph_cbch_req_2(variant)         );
    SEND  ( mmi_cbch_req_1()                );
    AWAIT ( mphc_config_cbch_req_1(variant) );
    AWAIT ( mphc_cbch_schedule_req_1()      );
    AWAIT ( mphc_cbch_schedule_req_2()      );
  }
}



/*------------------------Following is for ALR801-------------------------------
 *
 * Description:	MMI requests reading of CBCH for the message identifier
 *              3, 7 and 11 to 13. Then the CBCH channel is configured and
 *              reading of CBCH starts.
 *-------------------------------------------------------------------------------*/

T_STEP mmi_request_followed_by_configuration_of_cbch_alr801(CHAR variant)
{
  BEGIN_STEP ("mmi_request_followed_by_configuration_of_cbch_alr801")
  {
    SEND  ( mph_classmark_req_1()           );
    SEND  ( mph_idle_req_14()               );
    AWAIT ( mphc_stop_scell_bcch_req()      );
    AWAIT ( mphc_start_ccch_req_10()        );
    AWAIT ( mphc_scell_nbcch_req_full()     );
    SEND  ( mmi_cbch_req_2(variant)         );
    SEND  ( mph_cbch_req_3(variant)         );
    AWAIT ( mphc_config_cbch_req_2(variant) );
    AWAIT ( mphc_cbch_schedule_req_1()      );
    AWAIT ( mphc_cbch_schedule_req_2()      );
  }
}



/*------------------------Following is for ALR803-------------------------------
 *
 * Description:	ALR receives an expected CBCH message. The content is forwarded
 *              to MMI.
 * Variant A: single message identifier (7)
 * Variant B: message identifier specified by range (11), lower boundary
 * Variant C: message identifier specified by range (12), middle of the range
 * Variant D: message identifier specified by range (13), upper boundary
 * Variant E: single message identifier (7) with update number 2
 *-------------------------------------------------------------------------------*/

T_STEP reception_of_expected_cbch_message_alr803(CHAR variant)
{
  BEGIN_STEP ("reception_of_expected_cbch_message_alr803")
  {
    SEND  ( mphc_data_ind_54(variant) );
    AWAIT ( mphc_cbch_info_req_1()    );
    SEND  ( mphc_data_ind_55()        );
    SEND  ( mphc_data_ind_56()        );
    SEND  ( mphc_data_ind_57()        );
    AWAIT ( mmi_cbch_ind_1(variant)   );
  }
}
/*------------------------Following is for ALR808-------------------------------
 *
 * Description:	ALR receives an CBCH Schedule message.
                            variant a : entry into DRX mode
                            variant b : 2nd DRX mode
 *-------------------------------------------------------------------------------*/

T_STEP reception_of_cbch_schedule_message_alr809(CHAR variant)
{
  BEGIN_STEP ("reception_of_expected_cbch_message_alr809")
  {
      SEND   ( mphc_data_ind_67());
      AWAIT ( mphc_cbch_info_req_1());
      SEND   ( mphc_data_ind_60());
      SEND   ( mphc_data_ind_61());
      SEND   ( mphc_data_ind_62());
      AWAIT (mphc_cbch_schedule_req_4(variant));
  }
}


/*------------------------Following is for ALR960-------------------------------
 *
 * Description:	RR selects Channel 23 as serving cell after reading the
 *              BCCH carrier. On receipt of a list of neighbour cells
 *              (MPH-NEIGHBOURCELL request primitive) followed by
 *              the power measurements of these cells
 *              (MPHC-RXLEV-PERIODIC indication primitive), ALR starts a 10-second
 *              timer and requests Layer 1 to commence synchronization of these cells.
 *              On expiry of this timer, ALR requests Layer 1 to stop synchronization
 *              of these cells and forwards a list of all synchronized neighbour cells
 *              to be measured to Layer 1 (MPHC-NCELL-LIST-SYNC request primitive).
 *-------------------------------------------------------------------------------*/

T_STEP power_measurement_e_otd_on_idle_mode__alr960()
{
  BEGIN_STEP ("power_measurement_e_otd_on_idle_mode__alr960")
  {
    SEND  ( mph_classmark_req_3()                  );
    SEND  ( mph_idle_req_22()                      );
    AWAIT ( mphc_stop_scell_bcch_req()             );
    AWAIT ( mphc_start_ccch_req_9()                );
    AWAIT ( mphc_scell_nbcch_req_full()            );
    SEND  ( mph_neighbourcell_req_15()             );
    AWAIT ( mphc_rxlev_periodic_req_20()           );
    SEND  ( mphc_rxlev_periodic_ind_3()            );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_637) );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_25)  );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)  );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_512) );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_580) );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_885) );
    AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124) );
    SEND  ( mphc_ncell_sync_ind_33()               );
    SEND  ( mphc_rxlev_periodic_ind_3()            );
    AWAIT ( mph_measurement_ind_32()               );
    SEND  ( mphc_ncell_sync_ind_34()               );
    AWAIT ( mphc_ncell_bcch_req (ARFCN_637)        );
    SEND  ( mphc_ncell_sync_ind_35()               );
    AWAIT ( mphc_ncell_bcch_req_30()               );
    SEND  ( mphc_ncell_sync_ind_36()               );
    AWAIT ( mphc_ncell_bcch_req_25()               );
    SEND  ( mphc_ncell_sync_ind_37()               );
    AWAIT ( mphc_ncell_bcch_req_31()               );
    SEND  ( mphc_ncell_sync_ind_38()               );
    AWAIT ( mphc_ncell_bcch_req_32()               );
    SEND  ( mphc_ncell_sync_ind_39()               );
    AWAIT ( mphc_ncell_bcch_req_33()               );
    SEND  ( mphc_rxlev_periodic_ind_3()            );
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_637)     );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_637)    );
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_25)      );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_25)     );
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)      );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)     );
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_512)     );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_512)    );
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_580)     );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_580)    );
    SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_885)     );
    AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_885)    );
    SEND  ( mphc_rxlev_periodic_ind_3()            );
    SEND  ( mphc_rxlev_periodic_ind_3()            );
    AWAIT ( mph_measurement_ind_33()               );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_14)         );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_25)         );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_512)        );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_580)        );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_637)        );
    AWAIT ( mph_unitdata_ind_si3(ARFCN_885)        );
    SEND  ( mphc_rxlev_periodic_ind_3()            );
    SEND  ( mphc_rxlev_periodic_ind_3()            );
    SEND  ( mphc_rxlev_periodic_ind_3()            );
    AWAIT ( mphc_stop_ncell_sync_req_all()         );
    SEND  ( mphc_stop_ncell_sync_con()             );
    AWAIT ( mphc_ncell_list_sync_req_3()           );
    AWAIT ( mph_measurement_ind_33()               );
  }
}



/*------------------------Following is for ALR970-------------------------------
 *
 * Description:	A dedicated channel is assigned in preparation for E-OTD measurement.
 *-------------------------------------------------------------------------------*/

T_STEP connection_established_e_otd_dedicated_mode__alr970()
{
  BEGIN_STEP ("connection_established_e_otd_dedicated_mode__alr970")
  {
    SEND  ( mph_random_access_req_1()        );
    AWAIT ( mphc_stop_rxlev_periodic_req()   );
    SEND  ( mphc_stop_rxlev_periodic_con_1() );
    AWAIT ( mphc_stop_ncell_sync_req_all()   );
    SEND  ( mphc_stop_ncell_sync_con()       );
    AWAIT ( mphc_stop_scell_bcch_req()       );
    SEND  ( mphc_stop_scell_bcch_con()       );
    AWAIT ( mphc_scell_nbcch_req_full()      );
    AWAIT ( mphc_ra_req_5()                  );
    SEND  ( mphc_ra_con_1()                  );
    AWAIT ( mph_random_access_cnf_1()        );
    AWAIT ( mphc_ra_req_6()                  );
    SEND  ( mphc_data_ind_20()               );
    AWAIT ( mph_unitdata_ind_10()            );
    SEND  ( mph_dedicated_req_1()            );
    AWAIT ( mphc_stop_scell_bcch_req()       );
    AWAIT ( mphc_stop_ccch_req()             );
    AWAIT ( mphc_stop_ra_req_1()             );
    AWAIT ( mphc_immed_assign_req_1()        );
    SEND  ( mphc_immed_assign_con_1()        );
    AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );
  }
}



/*------------------------Following is for ALR972-------------------------------
 *
 * Description:	RR requests E-OTD measurement of a list of cells forwarded in the
 *              MPH-NCELL-POS request primitive, whereupon ALR forwards this list
 *              to Layer 1 (MPHC-NCELL-LIST-SYNC request primitive).
 * Variant A: all cells in the list are current neighbour cells and are thus known to ALR,
 * Variant B: contains two cell (arfcn 516 and 525 respectively) which are unknown.
 *-------------------------------------------------------------------------------*/

T_STEP e_otd_measurement_started_dedicated_mode__alr972(CHAR variant)
{
  BEGIN_STEP ("e_otd_measurement_started_dedicated_mode__alr972")
  {
    SEND  ( mph_ncell_pos_req_2(variant)        );
    AWAIT ( mphc_ncell_list_sync_req_4(variant) );
  }
}

/*------------------------Following is for ALRG007-------------------------------
 *
 * Description:	In dedicated mode ALR is suspended during a PDCH Assignment procedure.
 * Reference:   TS04.08 V8.10.0, section 3.4.19 Assignment to a Packet Data channel.
 *-------------------------------------------------------------------------------*/

T_STEP pdch_assignment_alrg007()
{
  BEGIN_STEP ("pdch_assignment_alrg007")
  {

    SEND  ( mph_dedicated_req_2g()  );
    AWAIT ( mphc_stop_dedicated_req() );
    SEND  ( mphc_stop_dedicated_con()  );
    AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );
    //this does not work, but there has to be
    //a CCCH request to enter idle mode before going to PTM.
    /*This needs a implementation check through GRR-RR-ALR*/
    /*AWAIT ( mphc_start_ccch_req_8() );*/

  }
}

/*-----------------------------------------------------------------------------
 *
 * Description: Start a MO call while plmn search is active.
 *
 *-------------------------------------------------------------------------------*/

T_STEP abort_search_due_to_mo_call(T_VAR_ABORT_PLMN_SEARCH var)
{
  BEGIN_STEP ("abort search due to mo call")
  {
    SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH) );
    switch (var)
    {
      case VAR_PLMN_POWER_CNF:
        AWAIT ( mphc_stop_rxlev_req()              );
        break;
      case VAR_PLMN_BSIC_CNF:
        AWAIT ( mphc_stop_ncell_sync_req(ARFCN_32) );
        break;
      case VAR_PLMN_SI:
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32) );
        break;
    }

    SEND  ( mph_random_access_req_1()         );
    AWAIT ( mphc_stop_rxlev_periodic_req()    ); // not necessary
    AWAIT ( mphc_stop_scell_bcch_req()        ); // not necessary
    AWAIT ( mphc_start_ccch_req_1('A')        );
    AWAIT ( mphc_scell_nbcch_req_full()       );
    AWAIT ( mphc_ra_req_1()                   );
  }
}

/*-----------------------------------------------------------------------------
 *
 * Description: GRR leaves Packet Idle Mode, the search is aborted.
 *
 *-------------------------------------------------------------------------------*/
T_STEP abort_search_due_to_leaving_pim(T_VAR_ABORT_PLMN_SEARCH var)
{
  BEGIN_STEP ("abort search due leaving PIM")
  {
    SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)   );
    switch (var)
    {
      case VAR_PLMN_POWER_CNF:
        AWAIT ( mphc_stop_rxlev_req()              );
        break;
      case VAR_PLMN_BSIC_CNF:
        AWAIT ( mphc_stop_ncell_sync_req(ARFCN_32) );
        break;
      case VAR_PLMN_SI:
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32) );
        break;
    }
#if defined GPRS
    SEND  ( mph_mon_ctrl_req(LEAVING_PIM_PBCCH) );
    AWAIT ( mphc_stop_ccch_req()                );
#endif
  }
}

/*-----------------------------------------------------------------------------
 *
 * Description: GRR starts a cell reselection, the search is aborted.
 *
 *-------------------------------------------------------------------------------*/
T_STEP abort_search_due_to_cell_resel(T_VAR_ABORT_PLMN_SEARCH var)
{
  BEGIN_STEP ("abort search due to cell reselection")
  {
    SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH) );
    switch (var)
    {
      case VAR_PLMN_POWER_CNF:
        AWAIT ( mphc_stop_rxlev_req()              );
        break;
      case VAR_PLMN_BSIC_CNF:
        AWAIT ( mphc_stop_ncell_sync_req(ARFCN_32) );
        break;
      case VAR_PLMN_SI:
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32) );
        break;
    }
    SEND  ( mph_idle_req_15g()                );
    AWAIT ( mphc_stop_ccch_req()              );
    AWAIT ( mphc_stop_rxlev_periodic_req()    ); // not necessary
    AWAIT ( mphc_stop_scell_bcch_req()        ); // not necessary
    AWAIT ( mphc_new_scell_req_10()           );
  }
}

/*-----------------------------------------------------------------------------
 *
 * Description: GRR requests synchronization for 3 ncells.
 *
 *-------------------------------------------------------------------------------*/
T_STEP nc_pbcch_initial_sync()
{
  BEGIN_STEP ("PBCCH, initial synchronization to 3 ncells")
  {
      SEND  ( mph_neighbourcell_req_3g()             );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)   );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)  );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124) );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)                );
      AWAIT ( mph_measurement_ind_1g()               );
      SEND  ( mphc_ncell_sync_ind_failed(ARFCN_124)  );
      AWAIT ( mph_measurement_ind_2g()               );
      SEND  ( mph_neighbourcell_req_4g()             );
      AWAIT ( mph_measurement_ind_4g()               );
      AWAIT ( mphc_stop_ncell_sync_req(ARFCN_1)      );
      AWAIT ( mphc_ncell_sync_req_4g()               );
      SEND  ( mphc_ncell_sync_ind_3g()               );
      AWAIT ( mph_measurement_ind_3g()               );
  }
}

/*-----------------------------------------------------------------------------
 *
 * Description: GRR starts a cell selection, the search is aborted.
 *
 *-------------------------------------------------------------------------------*/

T_STEP abort_search_due_to_cell_sel(T_VAR_ABORT_PLMN_SEARCH var)
{
  BEGIN_STEP ("abort search due to cell selection")
  {
    SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH) );
    switch (var)
    {
      case VAR_PLMN_POWER_CNF:
        AWAIT ( mphc_stop_rxlev_req()              );
        break;
      case VAR_PLMN_BSIC_CNF:
        AWAIT ( mphc_stop_ncell_sync_req(ARFCN_32) );
        break;
      case VAR_PLMN_SI:
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32) );
        break;
    }
    SEND  ( mph_power_req_1()                 );
    AWAIT ( mphc_stop_ccch_req()              );
    AWAIT ( mphc_stop_rxlev_periodic_req()    ); // not necessary
    AWAIT ( mphc_stop_scell_bcch_req()        ); // not necessary
    AWAIT ( mphc_init_l1_req_1() );
    SEND  ( mphc_init_l1_con_1() );
    AWAIT ( mphc_rxlev_req()   );
    //SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)   );
  }
}

#ifdef FF_EGPRS
T_STEP start_of_sending_channel_request_messages_gsm_900__alr9803()
{
  BEGIN_STEP ("start_of_sending_channel_request_messages_gsm_900__alr9803")
  {
    SEND  ( mph_random_access_req_1()        );
    AWAIT ( mphc_stop_rxlev_periodic_req()   );
    AWAIT (mphc_stop_ncell_bcch_req_emr());
    AWAIT ( mphc_stop_scell_bcch_req()       );
    AWAIT ( mphc_scell_nbcch_req_full()      );
    AWAIT ( mphc_ra_req_1()                  );
    SEND  ( mphc_ra_con_1()                  );
    AWAIT ( mph_random_access_cnf_1()        );
    AWAIT ( mphc_ra_req_2()                  );
  }
}

T_STEP rr_cell_selection_emr (char idle_var,char rxlev_var,char emr_var )
{
  BEGIN_STEP ("rr_cell_selection_emr")
  {
      int i=15;
      filter_and_routings_dualband_gsm_900_dcs_1800__alr600();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();

      SEND  ( mph_classmark_req_1()       );
      SEND  ( mph_idle_req_4(idle_var) );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_start_ccch_req_7('H') );

      AWAIT ( mphc_scell_nbcch_req_full()    ); /*acquires both N/E BCCH*/
      SEND  ( mph_identity_req_1()        );
      SEND  ( mph_identity_req_1()        );
      SEND  ( mph_cbch_req_1()            );

      if (rxlev_var == 'A')
      {
        SEND  ( mph_neighbourcell_req_3()      );
        AWAIT ( mphc_rxlev_periodic_req_3()    );
        SEND  ( mphc_rxlev_periodic_ind_1() );      /* 1 */
        SEND  ( mphc_rxlev_periodic_ind_1() );
        AWAIT ( mph_measurement_ind_emr_1()     );
        SEND  ( mphc_rxlev_periodic_ind_1() );
      }
      else if (rxlev_var == 'B')
      {
        SEND  ( mph_neighbourcell_req_3()      );
        AWAIT ( mphc_rxlev_periodic_req_3()    );
      }
      else
      {
        SEND  ( mph_neighbourcell_req_emr_1()      );
        AWAIT ( mphc_rxlev_periodic_req_emr_1()    );
      }

      if (emr_var != 'Z')
      {
        while ( i)
        {
          SEND (mphc_data_ind_si2quater(L2_CHANNEL_EBCCH) );
          AWAIT (mph_unitdata_ind_si2q(ARFCN_23) );
          i--;
        }
      }

      if (rxlev_var == 'A')
      {
        SEND  ( mphc_rxlev_periodic_ind_1() );
        AWAIT ( mph_measurement_ind_emr_1()     );
      }

      if (emr_var != 'Z')
      {
        SEND (mph_enhpara_update_req_1 (emr_var));
        SEND (mph_mon_ctrl_req_1(STOP_MON_BCCH,UPDATE_SI2QUATER) );
      }
  }
}
#endif  // FF_EGPRS
/*-------------------For Test Case 1013A-E----------------------------------------------------------
 *
 * Description: Setting state of ALR in Dedicated for Handovers.
 *
 *-------------------------------------------------------------------------------*/

T_STEP initial_setting_state_to_dedicated()
{
  BEGIN_STEP ("initial_setting_state_to_dedicated")
  {
    filter_and_routings_gsm_900__alr000();
    initiation_by_rr_4_channels_available_alr001();
    find_bcch_carrier_first_channel_alr003();
    read_bcch_data_alr007();
    rr_select_first_channel_alr013();
    start_of_sending_channel_request_messages_gsm_900__alr055();
    immediate_assignment_for_the_mobile_station_alr056();
    neighbour_cell_in_dedicated_alr155();
    SEND  ( mphc_meas_report_4()      );
    AWAIT ( mphc_ncell_sync_req_fn(ARFCN_124) );
    AWAIT ( mphc_ncell_sync_req_fn(ARFCN_1) );
    SEND  ( mphc_ncell_sync_ind_fn(ARFCN_124));
    SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1));
    AWAIT(mph_measurement_ind_dummy(ARFCN_23,25));
  }
}


/*-----------------------------------------------------------------------------
 *
 * Description: After a HPLMN search the paging is configured to REORG,
 *              if REORG was set before HPLMN search.
 *
 * Variant: depending on configuration before HPLNM search procedure
 *
 *-------------------------------------------------------------------------------*/

T_STEP continue_page_mode_reorg_after_hplnm_search_procedure(CHAR variant)
{
  BEGIN_STEP ("continue page mode reorg after HPLMN search")
  {

    switch (variant)
    {
    case 'A':
      AWAIT ( mphc_start_ccch_req_7('H')              ); /*Paging REORG */
      break;
    case 'B':/*not used*/
      AWAIT ( mphc_start_ccch_req_2()                 ); /*Paging REORG */
      break;
    default:
      AWAIT ( mphc_start_ccch_req_14()                ); /*SKIP Parameter - actual not used*/
    }

      AWAIT ( mphc_scell_nbcch_req_full()             ); /*Read complete BCCH every time*/
  }
}

/*-----------------------------------------------------------------------------
 *
 * Description: HPLMN search cannot be performed if page mode REORG is configured.
 *              Therefore REORG is stopped as long as HPLNM search procedure needs.
 *
 *-------------------------------------------------------------------------------*/

T_STEP stop_page_mode_reorg_before_hplnm_search_procedure()
{
  BEGIN_STEP ("stop page mode reorg before HPLMN search")
  {

    AWAIT ( mphc_start_ccch_req_normal()            ); /*Paging NORMAL*/
    AWAIT ( mphc_scell_nbcch_req_periodic()         ); /*read BCCH not every time*/
  }
}

/*-----------------------------------------------------------------------------
 *
 * Description:  Measurement Report is sent by L1. This is followed by sync reading on the strongest
 *                    6 cells. Random Access procedure is then started.
 *
 *-------------------------------------------------------------------------------*/

T_STEP meas_ind_with_start_sync_followed_by_rach_procedure()
{
  BEGIN_STEP ("meas indiction sync requests followed by rach")
  {
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)  );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)  );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)  );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)   ); // sync for 14
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)   );

      SEND  ( mph_random_access_req_1()        );
      AWAIT ( mphc_stop_rxlev_periodic_req()   );
      AWAIT ( mphc_stop_ncell_sync_req_1()     );
      AWAIT ( mphc_stop_ncell_bcch_req_1());
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_scell_nbcch_req_full()      );
      AWAIT ( mphc_ra_req_1()                  );

      SEND  ( mphc_ra_con_1()                  );
      AWAIT ( mph_random_access_cnf_1()        );
      AWAIT ( mphc_ra_req_2()                  );
  }
}

/*-----------------------------------------------------------------------------
 *
 * Description:  Rxlevel Measurement based on Search
 *
 *-------------------------------------------------------------------------------*/

T_STEP rxlevel_measurement_based_on_search_parallel(CHAR search_mode, CHAR rxlev_idx)
{
  BEGIN_STEP (" Rxlevel Based on Search")
  {
    AWAIT ( mphc_init_l1_req(STD_DUAL_EGSM));
    SEND  ( mphc_init_l1_con_1());
    switch(search_mode) 
    {
      case NORMAL_SEARCH_MODE:
      case FULL_SEARCH_MODE  :
        /* Full & Normal Search will do the measurement 5 Times */
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
      case FAST_SEARCH_MODE :
      case BLACK_LIST_SEARCH_MODE:
        /* FAST & Blacklist  Search will do the measurement only once */
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        break;
    }
  }
}

/*-----------------------------------------------------------------------------
 *
 * Description:  Rxlevel Measurement based on Search
 *
 *-------------------------------------------------------------------------------*/

T_STEP rxlevel_measurement_based_on_search_quadband(CHAR search_mode, CHAR rxlev_idx, CHAR pchinterrupt)
{
  BEGIN_STEP (" Rxlevel Based on Search")
  {
    switch(search_mode) 
    {
      case NORMAL_SEARCH_MODE:
      case FULL_SEARCH_MODE  :
        /* Full & Normal Search will do the measurement 5 Times */
        AWAIT ( mphc_init_l1_req(STD_DUAL_EGSM));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );

        AWAIT ( mphc_init_l1_req(STD_DUAL_US));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );

        AWAIT ( mphc_init_l1_req(STD_DUAL_EGSM));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );

        AWAIT ( mphc_init_l1_req(STD_DUAL_US));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );

        AWAIT ( mphc_init_l1_req(STD_DUAL_EGSM));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );

        AWAIT ( mphc_init_l1_req(STD_DUAL_US));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );

        AWAIT ( mphc_init_l1_req(STD_DUAL_EGSM));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );

        AWAIT ( mphc_init_l1_req(STD_DUAL_US));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
      case FAST_SEARCH_MODE :
        /* FAST Search will do the measurement only once */
        AWAIT ( mphc_init_l1_req(STD_DUAL_EGSM));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );

        AWAIT ( mphc_init_l1_req(STD_DUAL_US));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        break;
      case BLACK_LIST_SEARCH_MODE:
        /* Blacklist  Search will do the measurement only once */
        AWAIT ( mphc_init_l1_req(STD_DUAL_EGSM));
        SEND  ( mphc_init_l1_con_1());
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        break;
    }
  }
}


/*-----------------------------------------------------------------------------
 *
 * Description:  Rxlevel Measurement based on Search
 *
 *-------------------------------------------------------------------------------*/

T_STEP rxlevel_measurement_based_on_search_american(CHAR search_mode, CHAR band, CHAR rxlev_idx)
{
  BEGIN_STEP (" Rxlevel Based on Search")
  {
    AWAIT ( mphc_init_l1_req(band)      );
    SEND  ( mphc_init_l1_con_1()        );
    switch(search_mode) 
    {
      case NORMAL_SEARCH_MODE:
      case FULL_SEARCH_MODE  :
        /* Full & Normal Search will do the measurement 5 Times */
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
      case FAST_SEARCH_MODE :
      case BLACK_LIST_SEARCH_MODE:
        /* FAST & Blacklist  Search will do the measurement only once */
        AWAIT ( mphc_rxlev_req()            );
        SEND  ( mphc_rxlev_ind(rxlev_idx)   );
        break;
    }
  }
}
