/*
  +------------------------------------------------------------------------------
  |  File:       alr_cases.cpp
  +------------------------------------------------------------------------------
  |              Copyright 2003 Texas Instruments
  |              All rights reserved.
  |
  |              This file is confidential and a trade secret of Texas Instruments.
  |              The receipt of or possession of this file does not convey
  |              any rights to reproduce or disclose its contents or to
  |              manufacture, use, or sell anything it may describe, in
  |              whole, or in part, without the specific written consent of
  |              Texas Instruments.
  +------------------------------------------------------------------------------
  | Purpose:     Contains test cases for alr
  |
  +------------------------------------------------------------------------------
*/
#include "alr_constraints.h"
#include "alr_steps.h"
/*------------------------------------------------------------------------------
 *                      TC STATUS
 *  Fail                 Reason                               Issue
 *----------------------------------------------------------------------------*/

/*------------------------ handling instructions ---------------------------------
 * It is not necessary to generate the CCDINC and TDCINC files as well the
 * TDCLIB libraries twice for GSM or GPRS, because the makcdg tool does not make
 * a distinction between them. But you have to create both version of the stacks
 * (e.g. G23_SMI_GPRS.exe and MS_TI_FD.exe) which need the matching ccddata.lib
 * (e.g. \gpf\PROJECT\gprs\lib\nucwin\ccddata.lib and
 * \gpf\PROJECT\gsm\lib\nucwin\ccddata.lib).
 *
 * To create the GSM and GPRS related files you have to do
 * - set prost=gsm (or call \gpf\initvars gsm MS D: \g23m\condat %programfiles%)
 * - create CCDINC, TDCINC, TDCLIB, \gpf\bin\ccddata_dll.dll and
 *   \gpf\PROJECT\gsm\lib\nucwin\ccddata.lib with makcdg
 * - build GSM stack with
 *   msdev %TESTROOT%\int\msdev\gsm.dsw /MAKE "MS_TI_FD - Win32 Debug"
 *
 * - set prost=gprs
 * - create \gpf\PROJECT\gprs\lib\nucwin\ccddata.lib with makcdg
 * - build GPRS stack with
 *   msdev %TESTROOT%\int\msdev\gprs.dsw /MAKE "G23_SMI_GPRS - Win32 Debug"
 * - build all GSM and GPRS test cases with
 *   msdev %TESTROOT%\ms\src\alr\alr_test.dsw /MAKE "alr_test - Win32 Debug"
 *------------------------------------------------------------------------------*/
/*------------------------ GLOBAL OPTIONS-----------------------------------------
 * The following options are global options of the entity which affect most
 * testcases. Most of the time they are identical to the respective feature flags
 * Each testcase which depends on one of these options, should implement
 * a loop which tests all relevant combinations of options, ie:
 *  a ncell testcase should depend on the GPRS,STD and EOTD. And should be run
 *  for each possible combination
 *  a test for NON_DRX should only depend on GPRS....
 */
typedef enum {
  //GO_EOTD_ON,
  //GO_EOTD_OFF,
  //GO_GPRS_ON,
  //GO_GPRS_OFF,
  //GO_STD,
  GO_EOTD_ON_GPRS_OFF,
  GO_EOTD_OFF_GPRS_OFF,
  GO_EOTD_OFF_GPRS_ON,
  GO_EOTD_ON_GPRS_ON
} T_E_GLOBAL_OPTIONS;

#if defined(GPRS)
#define TRAP_FOR_DEACT U8 failed=0; \
    tdc_usi_trace ("pre-amble: TC compiled for GPRS stack");\
    TRAP
#else // GPRS
#define TRAP_FOR_DEACT U8 failed=0; \
    tdc_usi_trace ("pre-amble: TC compiled for GSM only stack");\
    TRAP
#endif  // GPRS
#define HANDLE_DEACT ONFAIL {failed =1;}\
    tdc_usi_trace ("post-amble: TC %s", failed?"FAILED":"PASSED");\
    COMMAND("PL REDIRECT CLEAR");\
    SEND (mph_deactivate_req());\
    COMMAND ("PL MEMORY");/*check mem leaks*/\
    COMMAND ("PL CONFIG STD=0");/*to force reread of PCM*/\
    if (failed) {\
      /*COMMAND("TAP TIMEOUT 200");*/\
      /*this will always fail!*/\
      AWAIT  ( mph_deactivate_req()  );\
    }

/*------------------------Following is for ALR002-------------------------------
 *
 * Description:	RR starts a cell selection. No suitable channel is available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR002()
{
  BEGIN_CASE ("initiation_by_rr_no_channels_available_alr002")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      SEND  ( mph_power_req_1()           );
      AWAIT ( mphc_init_l1_req_1()        );
      SEND  ( mphc_init_l1_con_1()        );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mph_power_cnf_2()           );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR003-------------------------------
 *
 * Description:	The FB/SB search mode switch between wideband and narrowband
 *
 *               CASE 1 : Initial Cell Selection - Wideband Mode
 *               CASE 2 : After the 1st Sync     - Narrowband Mode
 *               CASE 3 : After 6 Sync failure   - Wideband Mode
 *-------------------------------------------------------------------------------*/

T_CASE ALR003()
{
  BEGIN_CASE ("initiation_by_rr_no_channels_available_alr002")
  {
    TRAP_FOR_DEACT {
    initiation_by_rr_4_channels_available_alr001();

    /* Search Mode Checking for Wide Band */

    SEND  ( mph_bsic_req(ARFCN_23_STD_900)  );
    AWAIT ( mphc_network_sync_req(ARFCN_23, SM_WIDE_MODE) );
    SEND  ( mphc_network_sync_ind_1()       );
    AWAIT ( mph_bsic_cnf_1()                );
    AWAIT ( mphc_new_scell_req_1()          );
    SEND  ( mphc_new_scell_con_1()          );
    AWAIT ( mphc_scell_nbcch_req_full()     );

    generating_six_sync_failures();

    /* After the SIX SYNC Failures */

    SEND  ( mph_bsic_req(ARFCN_1_STD_900)  );
    AWAIT ( mphc_network_sync_req(ARFCN_1, SM_WIDE_MODE));

    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR03A-------------------------------
 *
 * Description:	Wideband switching while getting RADIO LINK Failure
 *               CASE 4 : After Radio Link failure   - Wideband Mode
 *-------------------------------------------------------------------------------*/

T_CASE ALR003A()
{
  BEGIN_CASE ("find_bcch_carrier_all_channels_failed_then_error_indication_alr005")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      rr_select_second_channel_alr012('A');

      SEND  ( mph_random_access_req_1()        );
      AWAIT ( mphc_stop_rxlev_periodic_req()   );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_scell_nbcch_req_full()      );
      AWAIT ( mphc_ra_req_1()                  );
      SEND  ( mphc_ra_con_1()                  );
      AWAIT ( mph_random_access_cnf_1()        );
      AWAIT ( mphc_ra_req_2()                  );
      SEND  ( mphc_data_ind_36()               );
      AWAIT ( mph_unitdata_ind_15()            );
      SEND  ( mph_dedicated_req_3()            );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_stop_ccch_req()             );
      AWAIT ( mphc_stop_ra_req_1()             );
      AWAIT ( mphc_immed_assign_req_2()        );
      SEND  ( mphc_immed_assign_con_1()        );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );
      SEND  ( mph_freq_redef_req_1()           );
      AWAIT ( mphc_change_frequency_1()        );
      SEND  ( mph_dedicated_req_4()            );
      AWAIT ( mphc_channel_assign_req_1()      );
      SEND  ( mphc_channel_assign_con_1()      );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );
      SEND  ( mph_dedicated_fail_req_1()       );
      AWAIT ( mphc_channel_assign_req_2()      );
      SEND  ( mphc_channel_assign_con_1()      );
      AWAIT ( mph_dedicated_fail_cnf_1()       );
      SEND  ( mphc_data_ind_37()               );
      AWAIT ( mph_unitdata_ind_16()            );
      SEND  ( mphc_data_ind_38()               );
      AWAIT ( mph_unitdata_ind_17()            );
      SEND  ( mphc_data_ind_39()               );
      AWAIT ( mph_unitdata_ind_18()            );
      SEND  ( mphc_data_ind_40()               );
      AWAIT ( ph_data_ind_1()                  );
      SEND  ( mphc_data_ind_37()               );
      SEND  ( mphc_data_ind_41()               );
      AWAIT ( mph_unitdata_ind_19()            );

      /*
       * Sending Invalid Block on SACCH to decrement the counter act_rlt
       * Intially the counter will be 22 the "Radio Link Failure" will be trigged
       * once the counter reaches 0
       */

      sending_error_data_in_sacch();

      /*
       *  RR receives the "Radio Link Failure"
       */

      AWAIT (mph_error_ind_radio_link_fail()   );

      SEND  (mph_bsic_req(ARFCN_1_STD_900)     );
      AWAIT (mphc_stop_rxlev_periodic_req()    );
      AWAIT (mphc_stop_dedicated_req());

      AWAIT (mphc_network_sync_req(ARFCN_1, SM_WIDE_MODE));

      MUTE(500);


    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR03B-------------------------------
 *
 * Description:	Wideband switching after Down Link Error
 *            CASE 5 : After receiving Downlink Error   - Wideband Mode
 *-------------------------------------------------------------------------------*/

T_CASE ALR003B()
{
  BEGIN_CASE ("Switching between Narrow to wideband band in case of downlink failure")
  {
    TRAP_FOR_DEACT {

      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      SEND  ( mphc_data_ind_11()               );
      AWAIT ( mphc_start_ccch_req_8()          );
      AWAIT ( mphc_scell_nbcch_req_periodic()  );
      SEND  ( mphc_data_ind_11()               );
      SEND  ( mphc_data_ind_11()               );

      /* Sending Invalid Block to generate Downlink Error */

      SEND  ( mphc_data_ind_19()               );
      SEND  ( mphc_data_ind_19()               );
      SEND  ( mphc_data_ind_19()               );

      /* Down Link Error Received in RR */

      AWAIT ( mph_error_ind_2()                );

      SEND  ( mph_random_access_req_1()        );
      SEND  (mph_bsic_req(ARFCN_1_STD_900)     );
      AWAIT (mphc_stop_rxlev_periodic_req()    );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      SEND  (mph_bsic_req(ARFCN_1_STD_900)     );
      AWAIT (mphc_start_ccch_req_14()          );
      AWAIT ( mphc_scell_nbcch_req_full()      );
      AWAIT ( mphc_ra_req_1()                  );
      SEND  ( mphc_ra_con_1()                  );
      AWAIT ( mph_random_access_cnf_1()        );
      AWAIT ( mphc_ra_req_2()                  );
      SEND  ( mphc_data_ind_36()               );
      AWAIT ( mph_unitdata_ind_15()            );
      SEND  ( mph_dedicated_req_3()            );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_stop_ccch_req()             );
      AWAIT ( mphc_stop_ra_req_1()             );
      AWAIT ( mphc_immed_assign_req_2()        );
      SEND  ( mphc_immed_assign_con_1()        );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );

      SEND  (mph_bsic_req(ARFCN_14_STD_900)    );
      AWAIT (mphc_stop_rxlev_periodic_req()    );
      AWAIT (mphc_stop_dedicated_req()         );

      AWAIT (mphc_network_sync_req(ARFCN_14, SM_WIDE_MODE));

      MUTE(500);

    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR03C-------------------------------
 *
 * Description:	Wideband switching after BCCH Error
 *            CASE 6 : After receiving some BCCH Error   - Wideband Mode
 *-------------------------------------------------------------------------------*/

T_CASE ALR003C()
{
  BEGIN_CASE ("Switching between wideband to Narrow band in case of after some BCCH ERROR")
  {
    TRAP_FOR_DEACT {

      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();

      /* Sending Invalid BCCH block to decrement the bcch fail count to reach 0 */

      SEND  ( mphc_data_ind_5()         );
      AWAIT ( mph_error_ind_1()         );
      SEND  ( mphc_data_ind_5()         );
      AWAIT ( mph_error_ind_1()         );
      SEND  ( mphc_data_ind_5()         );
      AWAIT ( mph_error_ind_1()         );
      SEND  ( mphc_data_ind_5()         );
      AWAIT ( mph_error_ind_1()         );

      SEND  ( mphc_data_ind_5()         );
      AWAIT ( mph_error_ind_1()         );
      SEND  ( mphc_data_ind_5()         );
      AWAIT ( mph_error_ind_1()         );
      SEND  ( mphc_data_ind_5()         );
      AWAIT ( mph_error_ind_1()         );
      SEND  ( mphc_data_ind_5()         );
      AWAIT ( mph_error_ind_1()         );

      SEND  (mph_bsic_req(ARFCN_24_STD_900));

      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT (mphc_network_sync_req(ARFCN_24, SM_WIDE_MODE));

      MUTE(500);

    } HANDLE_DEACT;
  }
}
/*------------------------Following is for ALR005-------------------------------
 *
 * Description:	All synchronization attempts to the available channels are failed.
 * RR is informed by an error indication with the cause no BCCH available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR005()
{
  BEGIN_CASE ("find_bcch_carrier_all_channels_failed_then_error_indication_alr005")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();

      SEND  ( mphc_network_sync_ind_failed(ARFCN_14)  );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_14_STD_900)   );
      SEND  ( mph_bsic_req(ARFCN_124_STD_900)         );
      AWAIT ( mphc_network_sync_req(ARFCN_124, SM_WIDE_MODE)        );
      SEND  ( mphc_network_sync_ind_failed(ARFCN_124) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_124_STD_900)  );
      SEND  ( mph_bsic_req(ARFCN_1_STD_900)           );
      AWAIT ( mphc_network_sync_req(ARFCN_1, SM_WIDE_MODE)          );
      SEND  ( mphc_network_sync_ind_failed(ARFCN_1)   );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_1_STD_900)    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR008-------------------------------
 *
 * Description:	A failed BCCH data block is read.
 *-------------------------------------------------------------------------------*/

T_CASE ALR008()
{
  BEGIN_CASE ("read_failed_bcch_data_alr008")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();

      SEND  ( mphc_data_ind_5() );
      AWAIT ( mph_error_ind_1() );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR011-------------------------------
 *
 * Description:	RR rejects the BCCH carrier. No further channel is available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR011()
{
  BEGIN_CASE ("rr_rejects_bcch_carrier_no_further_channel_available_alr011")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      rr_rejects_bcch_carrier_try_third_channel_alr009();
      rr_rejects_bcch_carrier_try_fourth_channel_alr010();

      SEND ( mph_bsic_req(NOT_PRESENT_16BIT) );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR012-------------------------------
 *
 * Description:	RR selects the channel 14 after reading the BCCH carrier.
 * Variant B: with empty neighbour cell list
 *-------------------------------------------------------------------------------*/

T_CASE ALR012B()
{
  BEGIN_CASE ("rr_select_second_channel_alr012b")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      rr_select_second_channel_alr012('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR012-------------------------------
 *
 * Description:	RR selects the channel 14 after reading the BCCH carrier.
 * Variant C: with neighbourcell list, but non-combined CCCH
 *-------------------------------------------------------------------------------*/

T_CASE ALR012C()
{
  BEGIN_CASE ("rr_select_second_channel_alr012c")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      rr_select_second_channel_alr012('C');
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR015-------------------------------
 *
 * Description:	RR has started a cell selection. During power measurements a new
 *              activation of cell selection is started by RR.
 *              The power measurement is restarted.
 *-------------------------------------------------------------------------------*/

T_CASE ALR015()
{
  BEGIN_CASE ("re_initiation_of_cell_selection_during_measurements_alr015")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();

      SEND  ( mph_power_req_1()           );
      AWAIT ( mphc_init_l1_req_1()        );
      SEND  ( mphc_init_l1_con_1()        );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mph_power_req_1()           );
      AWAIT ( mphc_stop_rxlev_req()       );
      AWAIT ( mphc_init_l1_req_1()        );
      SEND  ( mphc_init_l1_con_1()        );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mph_power_cnf_3()           );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR017-------------------------------
 *
 * Description:	During BCCH reading RR restarts a cell selection.
 *              This leads to a new full power measurement cycle.
 *-------------------------------------------------------------------------------*/

T_CASE ALR017()
{
  BEGIN_CASE ("re_initiation_of_cell_selection_during_bcch_reading_alr017")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();

      SEND  ( mph_sync_req(CS_STOP_BCCH_READING)             );
      AWAIT ( mphc_stop_scell_bcch_req() );
      SEND  ( mph_power_req_1()            );
      AWAIT ( mphc_init_l1_req_1()         );
      SEND  ( mphc_init_l1_con_1()         );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1)           );
      AWAIT ( mph_power_cnf_3()            );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR018-------------------------------
 *
 * Description:	A failed BCCH data block is read.
 *-------------------------------------------------------------------------------*/

T_CASE ALR018()
{
  BEGIN_CASE ("alr008_engineering_mode_alr018")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();

      COMMAND("PL REDIRECT MMI TAP");
      SEND  ( em_l1_event_req_1()  );
      SEND  ( mphc_data_ind_5()    );
      AWAIT ( em_data_ind_1()      );
      AWAIT ( mph_error_ind_1()    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR019-------------------------------
 *
 * Description:	A request is made to sync to a network. There is a timeout when
 *              no response is received by ALR.
 *              After informing RR that there is no BCCH, RR instructs ALR to
 *              check the next BCCH. This again, fails.
 *              This ensures the timer is correctly set up for
 *              subsequent BCCH search's.
 *-------------------------------------------------------------------------------*/

T_CASE ALR019()
{
  BEGIN_CASE ("search_for_bcch_carrier_takes_too_long_alr019")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();

      SEND  ( mph_bsic_req(ARFCN_23_STD_900)               );
      AWAIT ( mphc_network_sync_req(ARFCN_23, SM_WIDE_MODE)      );
      AWAIT ( mphc_stop_network_sync_req_1() );
      SEND  ( mphc_stop_network_sync_con_1() );
      AWAIT ( mph_bsic_cnf_9()               );
      SEND  ( mph_bsic_req(ARFCN_24_STD_900)               );
      AWAIT ( mphc_network_sync_req(ARFCN_24, SM_WIDE_MODE)      );
      AWAIT ( mphc_stop_network_sync_req_1() );
      SEND  ( mphc_stop_network_sync_con_1() );
      AWAIT ( mph_bsic_cnf_9()               );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR019A------------------------------
 *
 * Description:	A request is made to sync to a network. There is a timeout when
 *              no response is received by ALR. But L1 sends a SYNC_IND at the
 *              same time as the STOP_NETWORK_SYNC_REQ is sent. In this case
 *              L1 does not send a confirmation for the stop request and the
 *              MS should use the SYNC_IND for the two states CS_STOP_SYNC
 *              and CS_NW_SYNC_TIMEOUT in the same manner as for the normal
 *              state CS_ACTIVE_SYNC.
 *----------------------------------------------------------------------------*/

T_CASE ALR019A()
{
  BEGIN_CASE ("ALR019A: Search for BCCH carrier takes too long, crossed STOP_NETWORK_REQ and SYNC_IND")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();

      SEND  ( mph_bsic_req(ARFCN_23_STD_900)               );
      AWAIT ( mphc_network_sync_req(ARFCN_23, SM_WIDE_MODE)      );
      AWAIT ( mphc_stop_network_sync_req_1() );
      /* crossed STOP_NETWORK_REQ and SYNC_IND, no stop confirmation */
      SEND  ( mphc_network_sync_ind_1()       );
      AWAIT ( mph_bsic_cnf_1()                );
      AWAIT ( mphc_new_scell_req_1()          );
      SEND  ( mphc_new_scell_con_1()          );
      AWAIT ( mphc_scell_nbcch_req_full()     );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR022-------------------------------
 *
 * Description:	The idle mode is configured with Paging Reorganisation. After
 *              reception of a new paging mode this information is forwarded to PL.
 *              The message contain the following page modi:
 * Variant A: Extended Paging
 * No reaction is expected.
 *-------------------------------------------------------------------------------*/


/*------------------------Following is for ALR022-------------------------------
 *
 * Description:	The idle mode is configured with Paging Reorganisation. After
 *              reception of a new paging mode this information is forwarded to PL.
 *              The message contain the following page modi:
 * Variant A: Extended Paging
 * No reaction is expected.
 *-------------------------------------------------------------------------------*/

T_CASE ALR022A()
{
  BEGIN_CASE ("page_mode_change_paging_reorganisation_alr022a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('A');
      page_mode_change_paging_reorganisation_alr022('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR022-------------------------------
 *
 * Description:	The idle mode is configured with Paging Reorganisation. After
 *              reception of a new paging mode this information is forwarded to PL.
 *              The message contain the following page modi:
 * Variant B: Paging Reorganisation
 * No reaction is expected.
 *-------------------------------------------------------------------------------*/

T_CASE ALR022B()
{
  BEGIN_CASE ("page_mode_change_paging_reorganisation_alr022b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('A');
      page_mode_change_paging_reorganisation_alr022('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR022-------------------------------
 *
 * Description:	The idle mode is configured with Paging Reorganisation. After
 *              reception of a new paging mode this information is forwarded to PL.
 *              The message contain the following page modi:
 * Variant C: Same as before
 * No reaction is expected.
 *-------------------------------------------------------------------------------*/

T_CASE ALR022C()
{
  BEGIN_CASE ("page_mode_change_paging_reorganisation_alr022c")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('A');
      page_mode_change_paging_reorganisation_alr022('C');
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR023-------------------------------
 *
 * Description:	The idle mode is configured with Normal Paging. After reception of
 *              the new paging mode Extended Paging this information is forwarded
 *              to the lower layer. The new paging mode is configured for layer 1.
 *-------------------------------------------------------------------------------*/

T_CASE ALR023()
{
  BEGIN_CASE ("page_mode_change_normal_paging_to_extended_paging_alr023")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();

      SEND  ( mphc_data_ind_9()               );
      AWAIT ( mphc_start_ccch_req_4()         );
      AWAIT ( mphc_scell_nbcch_req_periodic() );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR023 A-------------------------------
 *
 * Description:	1. MS is camped ON the Scell.
 *              2. Page Mode REorg
 *              3. SI 1, SI 2, SI 3 recevied
 *              4. Page Mode Normal Paging
 *              5. TEST CHECKS if we continue reading all SI messages, but revert to reading
 *                  page blocks corresponding to normal page mode.
 *              6. SI 4 received
 *              7. Entire SI set received. TEST CHECKS if L1 is configured to read periodic SI's.
 *-------------------------------------------------------------------------------*/

T_CASE ALR023A()
{
  BEGIN_CASE ("page_mode_change_reorg_to_normal_with_SI4_not_read_alr023A")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();

      /*
       * page Mode Reorg
       */
      SEND ( mphc_data_ind_8('B')            );
      AWAIT ( mphc_start_ccch_req_3('B')     );
      AWAIT ( mphc_scell_nbcch_req_full()    );


      /*
       * SI 1, SI 2, and SI3
       */
      SEND  ( mphc_data_ind_1()              );
      SEND  ( mphc_data_ind_2()              );
      SEND  ( mphc_data_ind_3()              );

      /*
       * Normal page mode
       */
      SEND  ( mphc_data_ind_11()             );
      AWAIT ( mphc_start_ccch_req_6()        );
      AWAIT ( mphc_scell_nbcch_req_full()    );

      /*
       * SI4
       */
      SEND  ( mphc_data_ind_4()              );
      AWAIT ( mphc_scell_nbcch_req_periodic());
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR023 B-------------------------------
 *
 * Description:	1. MS is camped ON the Scell.
 *              2. Page Mode REorg
 *              3. SI 1, SI 2, SI 4 recevied
 *              4. Page Mode Normal Paging
 *              5. TEST CHECKS if we continue reading all SI messages, but revert to reading
 *                  page blocks corresponding to normal page mode.
 *              6. SI 3 received
 *              7. Entire SI set received. TEST CHECKS if L1 is configured to read periodic SI's.
 *-------------------------------------------------------------------------------*/

T_CASE ALR023B()
{
  BEGIN_CASE ("page_mode_change_reorg_to_normal_with_SI3_not_read_alr023B")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();

       /*
       * page Mode Reorg
       */
      SEND ( mphc_data_ind_8('B')            );
      AWAIT ( mphc_start_ccch_req_3('B')     );
      AWAIT ( mphc_scell_nbcch_req_full()    );

      /*
       * SI 1, SI 2, and SI4
       */
      SEND  ( mphc_data_ind_1()              );
      SEND  ( mphc_data_ind_2()              );
      SEND  ( mphc_data_ind_4()              );

      /*
       * Normal page mode
       */
      SEND  ( mphc_data_ind_11()             );
      AWAIT ( mphc_start_ccch_req_6()        );
      AWAIT ( mphc_scell_nbcch_req_full()    );

       /*
       * SI3
       */
      SEND  ( mphc_data_ind_3()              );
      AWAIT ( mphc_scell_nbcch_req_periodic());
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR023 C-------------------------------
 *
 * Description:	1. MS is camped ON the Scell.
 *              2. Page Mode REorg
 *              3. SI 1, SI 2, SI 4 recevied
 *              4. Page Mode = Extended Page
 *              5. TEST CHECKS if we continue reading all SI messages, but revert to reading
 *                  page blocks corresponding to Extended page mode.
 *              6. SI 3 received
 *              7. Entire SI set received. TEST CHECKS if L1 is configured to read periodic SI's.
 *-------------------------------------------------------------------------------*/

T_CASE ALR023C()
{
  BEGIN_CASE ("page_mode_change_reorg_to_extended_with_SI3_not_read_alr023C")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();

      /*
       * page Mode Reorg
       */
       SEND ( mphc_data_ind_8('B')            );
      AWAIT ( mphc_start_ccch_req_3('B')     );
      AWAIT ( mphc_scell_nbcch_req_full()    );

      /*
       * SI 1, SI 2, and SI4
       */
      SEND  ( mphc_data_ind_1()              );
      SEND  ( mphc_data_ind_2()              );
      SEND  ( mphc_data_ind_4()              );

      /*
       * Extended page mode
       */
      SEND  ( mphc_data_ind_9()              );
      AWAIT ( mphc_start_ccch_req_4()        );
      AWAIT ( mphc_scell_nbcch_req_full()    );

      /*
       * SI3
       */
      SEND  ( mphc_data_ind_3()              );
      AWAIT ( mphc_scell_nbcch_req_periodic());
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR024-------------------------------
 *
 * Description:	The idle mode is configured with Normal Paging. After reception
 *              of the new paging mode Paging Reorganisation this information is
 *              forwarded to PL. The new paging mode is configured for layer 1.
 *-------------------------------------------------------------------------------*/

T_CASE ALR024()
{
  BEGIN_CASE ("page_mode_change_normal_paging_to_paging_reorganisation_alr024")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();

      SEND  ( mphc_data_ind_10()             );
      AWAIT ( mphc_start_ccch_req_6_reorg()  );
      AWAIT ( mphc_scell_nbcch_req_full()    );
     } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR026-------------------------------
 *
 * Description:	The idle mode is configured with Normal Paging. The messages
 *              contain the following page modi:
 * Variant A: Normal Paging
 * No reaction is exspected.
 *-------------------------------------------------------------------------------*/

T_CASE ALR026A()
{
  BEGIN_CASE ("page_mode_change_normal_paging_alr026a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();
      page_mode_change_normal_paging_alr026('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR026-------------------------------
 *
 * Description:	The idle mode is configured with Normal Paging. The messages
 *              contain the following page modi:
 * Variant B: Same as before.
 * No reaction is exspected.
 *-------------------------------------------------------------------------------*/

T_CASE ALR026B()
{
  BEGIN_CASE ("page_mode_change_normal_paging_alr026b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();
      page_mode_change_normal_paging_alr026('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR027-------------------------------
 *
 * Description:	The idle mode has been configured for normal paging.
 *              After a swap to Extended Paging this mode is configured a second time.
 *              Layer 1 does not need to be configured the second time, because
 *              its a continues process in ALR(MPHC).
 * Variant A: Extended Paging
 *-------------------------------------------------------------------------------*/

T_CASE ALR027A()
{
  BEGIN_CASE ("page_mode_change_extended_paging_a_second_time_alr027a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();
      page_mode_change_extended_paging_a_second_time_alr027('A');
    } HANDLE_DEACT;
  }
}
/*------------------------Following is for ALR027-------------------------------
 *
 * Description:	The idle mode has been configured for normal paging.
 *              After a swap to Extended Paging this mode is configured a second time.
 *              Layer 1 does not need to be configured the second time, because
 *              its a continues process in ALR(MPHC).
 * Variant B: Same as before
 *-------------------------------------------------------------------------------*/

T_CASE ALR027B()
{
  BEGIN_CASE ("page_mode_change_extended_paging_a_second_time_alr027b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();
      page_mode_change_extended_paging_a_second_time_alr027('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR028-------------------------------
 *
 * Description:	The idle mode has been configured for normal paging. After a swap
 *              to extended paging paging reorganisation is configured.
 *-------------------------------------------------------------------------------*/

T_CASE ALR028()
{
  BEGIN_CASE ("page_mode_change_extended_paging_to_paging_reorganisation_alr028")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();
      SEND  ( mphc_data_ind_9()               );
      AWAIT ( mphc_start_ccch_req_4()         );
      AWAIT ( mphc_scell_nbcch_req_periodic() );
      SEND  ( mphc_data_ind_10()              );
      AWAIT ( mphc_start_ccch_req_4_reorg()   );
      AWAIT ( mphc_scell_nbcch_req_full()        );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR029-------------------------------
 *
 * Description:	The idle mode has been configured for normal paging. After
 *              reception of the new paging mode normal paging this mode
 *              is configured.
 *-------------------------------------------------------------------------------*/

T_CASE ALR029()
{
  BEGIN_CASE ("page_mode_change_extended_paging_to_normal_paging_alr029")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');
      page_mode_change_back_to_normal_paging_alr025();

      SEND  ( mphc_data_ind_9()               );
      AWAIT ( mphc_start_ccch_req_4()         );
      AWAIT ( mphc_scell_nbcch_req_periodic() );
      SEND  ( mphc_data_ind_11()              );
      AWAIT ( mphc_start_ccch_req_6()         );
      AWAIT ( mphc_scell_nbcch_req_periodic() );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR031-------------------------------
 *
 * Description:	Measurement reporting for the serving cell is tested. The multiframe
 *              period is set to two multiframes. It is exspected that the
 *              initial report to RR is send after eleven reports of layer 1
 *              and successive reports are send to RR after ten reports.
 *              The number of TDMA frames between measurement reports to RR is
 *              102 TDMA frames which is equal to two multiframes.
 *-------------------------------------------------------------------------------*/

T_CASE ALR031()
{
  BEGIN_CASE ("measurement_reporting_bs_pa_mfrms_2_alr031")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('A');

      meas_rep_bs_pa_mfrms_2();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR032-------------------------------
 *
 * Description:	Measurement reporting for the serving cell is tested. The multiframe
 *              period is set to three multiframes. It is exspected that the initial
 *              report to RR is send after eight reports of layer 1 and successive
 *              reports are send to RR after seven reports. The number of TDMA frames
 *              between measurement reports to RR is 153 TDMA frames which is equal to
 *              three multiframes.
 *-------------------------------------------------------------------------------*/

T_CASE ALR032()
{
  BEGIN_CASE ("measurement_reporting_bs_pa_mfrms_3_alr032")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('B');

      meas_rep_bs_pa_mfrms_3();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR033-------------------------------
 *
 * Description:	Measurement reporting for the serving cell is tested. The multiframe
 *              period is set to four multiframes. It is exspected that the initial
 *              report to RR is send after six reports of layer 1 and successive
 *              reports are send to RR after five reports. The number of TDMA frames
 *              between measurement reports to RR is 204 TDMA frames which is
 *              equal to four multiframes.
 *-------------------------------------------------------------------------------*/

T_CASE ALR033()
{
  BEGIN_CASE ("measurement_reporting_bs_pa_mfrms_4_alr033")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('C');

      meas_rep_bs_pa_mfrms_4();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR035-------------------------------
 *
 * Description:	Measurement reporting for the serving cell is tested. The multiframe
 *              period is set to six multiframes. It is exspected that the initial
 *              report to RR is send after five reports of layer 1 and successive
 *              reports are send to RR after three reports. The number of TDMA frames
 *              between measurement reports to RR is 306 TDMA frames which is equal
 *              to six multiframes.
 *-------------------------------------------------------------------------------*/

T_CASE ALR035()
{
  BEGIN_CASE ("measurement_reporting_bs_pa_mfrms_6_alr035")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('E');

      meas_rep_bs_pa_mfrms_6();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR037-------------------------------
 *
 * Description:	Measurement reporting for the serving cell is tested. The multiframe
 *              period is set to eight multiframes. It is exspected that the initial
 *              report to RR is send after five reports of layer 1 and successive
 *              reports are send to RR after two reports. The number of TDMA frames
 *              between measurement reports to RR is 408 TDMA frames which is equal to
 *              eight multiframes.
 *-------------------------------------------------------------------------------*/

T_CASE ALR037()
{
  BEGIN_CASE ("measurement_reporting_bs_pa_mfrms_8_alr037")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('G');

      meas_rep_bs_pa_mfrms_8();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR039-------------------------------
 *
 * Description:	For a multiframe period of five multiframes it is expected that
 *              all 25 reports from layer 1 reading of BCCH information is started.
 *              Nine reports are received in the preamble. All four reports a
 *              measurement report is send to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR039()
{
  BEGIN_CASE ("bcch_reading_bs_pa_mfrms_5_alr039")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('D');
      meas_rep_bs_pa_mfrms_5();

      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_6()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_6()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_6()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_6()     );
      SEND  ( mphc_data_ind_1()           );
      SEND  ( mphc_data_ind_2()           );
      SEND  ( mphc_data_ind_14()          );
      SEND  ( mphc_data_ind_4()           );
      SEND  ( mphc_data_ind_15()          );
      SEND  ( mphc_data_ind_16()          );
      SEND  ( mphc_data_ind_17()          );
      SEND  ( mphc_data_ind_18()          );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR040-------------------------------
 *
 * Description:	Measurement reporting for the serving cell is tested. The multiframe
 *              period is set to seven multiframes. It is exspected that the initial
 *              report to RR is send after five reports of layer 1 and successive
 *              reports are send to RR after three reports. The number of TDMA frames
 *              between measurement reports to RR is 357 TDMA frames which is equal to
 *              seven multiframes.
 *-------------------------------------------------------------------------------*/

T_CASE ALR040()
{
  BEGIN_CASE ("bcch_reading_bs_pa_mfrms_7_alr040")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('F');

      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_3()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_9()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_9()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_9()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_9()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_data_ind_1()           );
      SEND  ( mphc_data_ind_2()           );
      SEND  ( mphc_data_ind_14()          );
      SEND  ( mphc_data_ind_4()           );
      SEND  ( mphc_data_ind_15()          );
      SEND  ( mphc_data_ind_16()          );
      SEND  ( mphc_data_ind_17()          );
      SEND  ( mphc_data_ind_18()          );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR041-------------------------------
 *
 * Description:	Measurement reporting for the serving cell is tested. The multiframe
 *              period is set to nine multiframes. It is exspected that the initial
 *              report to RR is send after five reports of layer 1 and successive
 *              reports are send to RR after two reports. The number of TDMA frames
 *              between measurement reports to RR is 459 TDMA frames which is equal
 *              to nine multiframes.
 *-------------------------------------------------------------------------------*/

T_CASE ALR041()
{
  BEGIN_CASE ("bcch_reading_bs_pa_mfrms_9_alr041")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('H');

      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_7()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_12()    );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_12()    );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_12()    );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_12()    );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_data_ind_1()           );
      SEND  ( mphc_data_ind_2()           );
      SEND  ( mphc_data_ind_14()          );
      SEND  ( mphc_data_ind_4()           );
      SEND  ( mphc_data_ind_15()          );
      SEND  ( mphc_data_ind_16()          );
      SEND  ( mphc_data_ind_17()          );
      SEND  ( mphc_data_ind_18()          );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR042-------------------------------
 *
 * Description:	The downlink timeout value is initialised with a value of 10
 *              (bs_pa_mfrms equal to 9). For each invalid paging message the
 *              counter is decremented by four. After three invalid
 *              messages a downlink error shall be indicated to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR042()
{
  BEGIN_CASE ("receive_invalid_paging_messages_alr042")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      SEND  ( mphc_data_ind_19()   );
      SEND  ( mphc_data_ind_19()   );
      SEND  ( mphc_data_ind_19()   );
      AWAIT ( mph_error_ind_2()    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR043-------------------------------
 *
 * Description:	The downlink timeout value is initialised with a value of 10
 *              (bs_pa_mfrms equal to 9). For each invalid paging message the
 *              counter is decremented by four. For each valid paging
 *              message the counter is incremented by one. It is checked that the
 *              counter is not incremented more than the initial value.
 *-------------------------------------------------------------------------------*/

T_CASE ALR043()
{
  BEGIN_CASE ("test_upper_limit_of_downlink_timeout_value_alr043")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      SEND  ( mphc_data_ind_11()              );
      AWAIT ( mphc_start_ccch_req_8()         );
      AWAIT ( mphc_scell_nbcch_req_periodic() );
      SEND  ( mphc_data_ind_11()              );
      SEND  ( mphc_data_ind_11()              );
      SEND  ( mphc_data_ind_19()              );
      SEND  ( mphc_data_ind_19()              );
      SEND  ( mphc_data_ind_19()              );
      AWAIT ( mph_error_ind_2()               );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR044-------------------------------
 *
 * Description:	The idle mode is stopped if a normal cell selection is initiated.
 *-------------------------------------------------------------------------------*/

T_CASE ALR044()
{
  BEGIN_CASE ("stop_idle_mode_by_normal_cell_selection_alr044")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      SEND  ( mph_power_req_1()              );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_stop_ccch_req()           );
      AWAIT ( mphc_stop_rxlev_periodic_req() );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_init_l1_req_1()           );
      SEND  ( mphc_init_l1_con_1()           );
      AWAIT ( mphc_rxlev_req()               );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR046-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 * Variant B: with neighbourcell list (serving cell included)
 *-------------------------------------------------------------------------------*/

T_CASE ALR046B()
{
  BEGIN_CASE ("definition_of_bcch_allocation_alr046b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR046-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 * Variant C: with empty neighbour cell list.
 *-------------------------------------------------------------------------------*/

T_CASE ALR046C()
{
  BEGIN_CASE ("definition_of_bcch_allocation_alr046c")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('C');
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR048-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and
 *              the neighbour cells 1, 14 and 124.
 *              The fieldstrength is 56 for channel 23, 12 for channel 1,
 *              44 for channel 14 and 25 for channel 124 (all values in GSM range).
 *              The ranking for the neighbour cells is 14, 124 and at least channel 1.
 *              Each reports contains two fieldstrength values per channel.
 *              The multiframe period is set to 6. The first measurement report
 *              is send to RR after five reports from PL. Then after each
 *              three reports from PL a measurement report is send to RR.
 *              Synchronisation to channel 14 fails.
 *-------------------------------------------------------------------------------*/

T_CASE ALR048()
{
  BEGIN_CASE ("synchronisation_to_neighbour_cells_failed_alr048")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      sync_to_ncell_14_fails_once();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR053-------------------------------
 *
 * Description: ALR requests successive reading of BCCH for all neighbour cells.
 *              This failed for channel 14.
 *-------------------------------------------------------------------------------*/

T_CASE ALR053()
{
  BEGIN_CASE ("reading_of_neighbour_cell_bcch_failed_alr053")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      ncell_bcch_read_fails_for_ncell_14();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR057-------------------------------
 *
 * Description:	An immediate assignment reject for the mobile station is received.
 *              Sending of channel request bursts is stopped. The mobile station
 *              listens to the downlink CCCH. Thereafter an immediate assignment
 *              message is received.
 *-------------------------------------------------------------------------------*/

T_CASE ALR057()
{
  BEGIN_CASE ("immediate_assignment_reject_for_the_mobile_station_alr057")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();
      SEND  ( mphc_data_ind_21()           );
      AWAIT ( mph_unitdata_ind_11()        );
      SEND  ( mph_random_access_req_2()    );
      AWAIT ( mphc_stop_ra_req_1()         );
      SEND  ( mphc_data_ind_22()           );
      AWAIT ( mph_unitdata_ind_11()        );
      SEND  ( mph_dedicated_req_1()        );
      AWAIT ( mphc_stop_scell_bcch_req() );
      AWAIT ( mphc_stop_ccch_req()       );
      AWAIT ( mphc_immed_assign_req_1()    );
      SEND  ( mphc_immed_assign_con_1()    );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR058-------------------------------
 *
 * Description:	After timeout of T3126 the mobile station goes back to idle mode.
 *-------------------------------------------------------------------------------*/

T_CASE ALR058()
{
  BEGIN_CASE ("t3126_expiry_back_to_idle_mode_alr058")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();

      SEND  ( mphc_ra_con_1()           );
      AWAIT ( mph_random_access_cnf_1() );
      SEND  ( mph_idle_req_6()          );
      AWAIT ( mphc_stop_ra_req_1()      );
      AWAIT ( mphc_start_ccch_req_10()  );
      AWAIT ( mphc_scell_nbcch_req_full()  );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR061-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the
 *              neighbour cells 1, 14 and 124.
 *              The fieldstrength is 56 for channel 23, 12 for channel 1,
 *              44 for channel 14 and 25 for channel 124 (all values in GSM range).
 *              The ranking for the neighbour cells is 14, 124 and channel 1.
 *              Each reports contains two fieldstrength values per channel.
 *              The multiframe period is set to 6. The first measurement report
 *              is send to RR after five reports from PL. Then after each three
 *              reports from PL a measurement report is send to RR. The NCC permitted
 *              check for neighbour cell 124 fails. The cell shall be excluded
 *              from further attempts.
 *-------------------------------------------------------------------------------*/

T_CASE ALR061()
{
  BEGIN_CASE ("ncell_synch_ncc_permitted_check_alr061")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('D');

      ncell_sync_ncc_permitted_check();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR064-------------------------------
 *
 * Description:	During the extended measurement procedure the BSIC of
 *              a carrier is successfully obtained.
 *-------------------------------------------------------------------------------*/

T_CASE ALR064()
{
  BEGIN_CASE ("extended_measurement_reporting_successful_end_alr064")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('H');
      extended_measurement_reporting_start_procedure_alr062();
      extended_measurement_reporting_successful_power_measurement_alr063();

      SEND  ( mphc_ncell_sync_ind_8()           );
      AWAIT ( mph_bsic_cnf_10()                 );
      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH) );

      continue_page_mode_reorg_after_hplnm_search_procedure('A');

      AWAIT ( mph_sync_ind_1()                  );
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR065-------------------------------
 *
 * Description:	The extended measurement procedure is stopped during the
 *              power maesurement.
 *--------------------------------------------------------------------------------*/

T_CASE ALR065()
{
  BEGIN_CASE ("extended_measurement_reporting_premature_end_during_power_measurement_alr065")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('H');
      extended_measurement_reporting_start_procedure_alr062();

      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH));
      AWAIT ( mphc_stop_rxlev_req());

      SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)) ;
      AWAIT ( mph_sync_ind_1());
      SEND  ( mphc_stop_rxlev_con()) ;
    } HANDLE_DEACT;
  }

}

/*------------------------Following is for ALR066-------------------------------
 *
 * Description:	The extended measurement procedure is stopped during the
 *              power maesurement.
 *--------------------------------------------------------------------------------*/

T_CASE ALR066()
{
  BEGIN_CASE ("extended_measurement_reporting_premature_end_during_power_measurement_alr066")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('H');
      extended_measurement_reporting_start_procedure_alr062();

      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH));
      AWAIT ( mphc_stop_rxlev_req());

      SEND  ( mphc_stop_rxlev_con()) ;
      AWAIT ( mph_sync_ind_1());
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR067-------------------------------
 *
 * Description:	The extended measurement procedure is stopped during the
 *              carrier synchronosation attempt.
 *-------------------------------------------------------------------------------*/

T_CASE ALR067()
{
  BEGIN_CASE ("extended_measurement_reporting_premature_end_during_bsic_scan_alr067")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('H');
      extended_measurement_reporting_start_procedure_alr062();
      extended_measurement_reporting_successful_power_measurement_alr063();

      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH));

      continue_page_mode_reorg_after_hplnm_search_procedure('A');

      AWAIT ( mphc_stop_ncell_sync_req(CONST_1023));

      SEND  ( mphc_ncell_sync_ind_8());
      AWAIT ( mph_sync_ind_1());

      SEND  ( mphc_stop_ncell_sync_con()) ;
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR068-------------------------------
 *
 * Description:	The extended measurement procedure is stopped during the
 *              carrier synchronosation attempt.
 *-------------------------------------------------------------------------------*/

T_CASE ALR068()
{
  BEGIN_CASE ("extended_measurement_reporting_premature_end_during_bsic_scan_alr068")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('H');
      extended_measurement_reporting_start_procedure_alr062();
      extended_measurement_reporting_successful_power_measurement_alr063();

      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH));

      continue_page_mode_reorg_after_hplnm_search_procedure('A');

      AWAIT ( mphc_stop_ncell_sync_req(CONST_1023));

      SEND  ( mphc_stop_ncell_sync_con()) ;
      AWAIT ( mph_sync_ind_1());
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR069-------------------------------
 *
 * Description:	The extended measurement procedure is stopped
 *-------------------------------------------------------------------------------*/

T_CASE ALR069()
{
  BEGIN_CASE ("extended_measurement_reporting_premature_end")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('H');
      extended_measurement_reporting_start_procedure_alr062();

      SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );
      AWAIT ( mphc_rxlev_req()     );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );
      AWAIT ( mphc_rxlev_req()     );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );
      AWAIT ( mphc_rxlev_req()     );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );
      AWAIT ( mphc_rxlev_req()     );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_7)             );
      AWAIT ( mph_ext_meas_cnf_1()                    );

      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH));
      AWAIT ( mph_sync_ind_1());
    } HANDLE_DEACT;
  }
}
/*------------------------Following is for ALR070-------------------------------
 *
 * Description:	The MS is in idle mode. It receives an empty paging message.
 *              The message content is not forwarded to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR070()
{
  BEGIN_CASE ("paging_req_1_empty_paging_message_alr070")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      SEND  ( mphc_data_ind_23()   );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 * Variant A: IMSI mobile identity 1, any channel needed
 *-------------------------------------------------------------------------------*/

T_CASE ALR071A()
{
  BEGIN_CASE ("paging_req_1_imsi_tmsi_for_ms_alr071a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_imsi_tmsi_for_ms_alr071('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 * Variant B: IMSI mobile identity 2, any channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR071B()
{
  BEGIN_CASE ("paging_req_1_imsi_tmsi_for_ms_alr071b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_imsi_tmsi_for_ms_alr071('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 * Variant C: TMSI mobile identity 1, TCH/F channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR071C()
{
  BEGIN_CASE ("paging_req_1_imsi_tmsi_for_ms_alr071c")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_imsi_tmsi_for_ms_alr071('C');
    } HANDLE_DEACT;
  }
}
/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 * Variant D: TMSI mobile identity 2, TCH/F channel needed
 *-------------------------------------------------------------------------------*/

T_CASE ALR071D()
{
  BEGIN_CASE ("paging_req_1_imsi_tmsi_for_ms_alr071d")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_imsi_tmsi_for_ms_alr071('D');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 * Variant E: IMSI mobile identity 1, SDCCH channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR071E()
{
  BEGIN_CASE ("paging_req_1_imsi_tmsi_for_ms_alr071e")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_imsi_tmsi_for_ms_alr071('E');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 * Variant F: IMSI mobile identity 2, SDCCH channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR071F()
{
  BEGIN_CASE ("paging_req_1_imsi_tmsi_for_ms_alr071f")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_imsi_tmsi_for_ms_alr071('F');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 * Variant G: TMSI mobile identity 1, Dual Rate channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR071G()
{
  BEGIN_CASE ("paging_req_1_imsi_tmsi_for_ms_alr071g")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_imsi_tmsi_for_ms_alr071('G');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 * Variant H: TMSI mobile identity 2, Dual Rate channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR071H()
{
  BEGIN_CASE ("paging_req_1_imsi_tmsi_for_ms_alr071h")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_imsi_tmsi_for_ms_alr071('H');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR071-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual IMSI or TMSI. A paging indication is forwarded to RR.
 *-------------------------------------------------------------------------------*/
T_CASE ALR071I()
{
  BEGIN_CASE ("paging_req_1_imsi_tmsi_for_ms_alr071i")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      SEND  ( mphc_data_ind_24('I')        );
      #ifdef GPRS
      AWAIT ( mph_paging_ind_1('I')        );
      #else // GPRS
        AWAIT ( mph_paging_ind_1('A')        );
      #endif // GPRS
    } HANDLE_DEACT;
  }
}

T_CASE ALR072A()
{
  BEGIN_CASE ("paging_req_1_not_for_ms_alr072a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_not_for_ms_alr072('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR072-------------------------------
 *
 * Description:	The MS is in idle mode. It receives paging request type 1 messages.
 *              The message content is not forwarded to RR, because the message
 *              content doesnt match to the MS identities.
 * Variant B: wrong TMSI mobile identity 1
 *-------------------------------------------------------------------------------*/
T_CASE ALR072B()
{
  BEGIN_CASE ("paging_req_1_not_for_ms_alr072b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_not_for_ms_alr072('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR072-------------------------------
 *
 * Description:	The MS is in idle mode. It receives paging request type 1 messages.
 *              The message content is not forwarded to RR, because the message
 *              content doesnt match to the MS identities.
 * Variant C: wrong IMSI mobile identity 2
 *-------------------------------------------------------------------------------*/
T_CASE ALR072C()
{
  BEGIN_CASE ("paging_req_1_not_for_ms_alr072c")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_not_for_ms_alr072('C');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR072-------------------------------
 *
 * Description:	The MS is in idle mode. It receives paging request type 1 messages.
 *              The message content is not forwarded to RR, because the message
 *              content doesnt match to the MS identities.
 * Variant D: wrong TMSI mobile identity 2
 *-------------------------------------------------------------------------------*/
T_CASE ALR072D()
{
  BEGIN_CASE ("paging_req_1_not_for_ms_alr072d")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_not_for_ms_alr072('D');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR072-------------------------------
 *
 * Description:	The MS is in idle mode. It receives paging request type 1 messages.
 *              The message content is not forwarded to RR, because the message
 *              content doesnt match to the MS identities.
 * Variant E: wrong type mobile identity 2
 *-------------------------------------------------------------------------------*/
T_CASE ALR072E()
{
  BEGIN_CASE ("paging_req_1_not_for_ms_alr072e")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_not_for_ms_alr072('E');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR073-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding to the
 *              actual IMSI or TMSI. A paging indication is forwarded to RR.
 *              The IMSI is less than 15 digits and the TMSI has only one valid byte,
 *              but is signalled in different ways by the infrastructure
 * Variant A: Short TMSI mobile identity 1, any channel needed, 1 byte length from the network
 *-------------------------------------------------------------------------------*/
T_CASE ALR073A()
{
  BEGIN_CASE ("paging_req_1_short_imsi_tmsi_for_ms_alr073a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_short_imsi_tmsi_for_ms_alr073('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR073-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding to the
 *              actual IMSI or TMSI. A paging indication is forwarded to RR.
 *              The IMSI is less than 15 digits and the TMSI has only one valid byte,
 *              but is signalled in different ways by the infrastructure
 * Variant B: Short TMSI mobile identity 2, TCH/F needed, 3 bytes length from the network
 *-------------------------------------------------------------------------------*/
T_CASE ALR073B()
{
  BEGIN_CASE ("paging_req_1_short_imsi_tmsi_for_ms_alr073b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_short_imsi_tmsi_for_ms_alr073('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR073-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding to the
 *              actual IMSI or TMSI. A paging indication is forwarded to RR.
 *              The IMSI is less than 15 digits and the TMSI has only one valid byte,
 *              but is signalled in different ways by the infrastructure
 * Variant C: Short IMSI mobile identity 1, SDCCH needed, 10 digits
 *-------------------------------------------------------------------------------*/
T_CASE ALR073C()
{
  BEGIN_CASE ("paging_req_1_short_imsi_tmsi_for_ms_alr073c")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_short_imsi_tmsi_for_ms_alr073('C');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR073-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding to the
 *              actual IMSI or TMSI. A paging indication is forwarded to RR.
 *              The IMSI is less than 15 digits and the TMSI has only one valid byte,
 *              but is signalled in different ways by the infrastructure
 * Variant D: Short IMSI mobile identity 2, Dual Rate needed, 10 digits
 *-------------------------------------------------------------------------------*/
T_CASE ALR073D()
{
  BEGIN_CASE ("paging_req_1_short_imsi_tmsi_for_ms_alr073d")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_short_imsi_tmsi_for_ms_alr073('D');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR073-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding to the
 *              actual IMSI or TMSI. A paging indication is forwarded to RR.
 *              The IMSI is less than 15 digits and the TMSI has only one valid byte,
 *              but is signalled in different ways by the infrastructure
 * Variant E: Short TMSI mobile identity 1, any channel needed, 2 bytes length from the network
 *-------------------------------------------------------------------------------*/
T_CASE ALR073E()
{
  BEGIN_CASE ("paging_req_1_short_imsi_tmsi_for_ms_alr073e")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_short_imsi_tmsi_for_ms_alr073('E');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR073-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding to the
 *              actual IMSI or TMSI. A paging indication is forwarded to RR.
 *              The IMSI is less than 15 digits and the TMSI has only one valid byte,
 *              but is signalled in different ways by the infrastructure
 * Variant F: Short TMSI mobile identity 2, TCH/F needed, 4 bytes length from the network
 *-------------------------------------------------------------------------------*/
T_CASE ALR073F()
{
  BEGIN_CASE ("paging_req_1_short_imsi_tmsi_for_ms_alr073f")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_1_short_imsi_tmsi_for_ms_alr073('F');
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR074-------------------------------
 *
 * Description:	The MS is in idle mode. It receives an empty paging message.
 *              The message content is not forwarded to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR074()
{
  BEGIN_CASE ("paging_req_2_empty_paging_message_alr074")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      SEND  (mphc_data_ind_27()    );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR075-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 or 2.
 *              A paging indication is forwarded to RR.
 * Variant A: TMSI mobile identity 1, any channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR075A()
{
  BEGIN_CASE ("paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR075-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 or 2.
 *              A paging indication is forwarded to RR.
 * Variant B: TMSI mobile identity 1, SDCCH needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR075B()
{
  BEGIN_CASE ("paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR075-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 or 2.
 *              A paging indication is forwarded to RR.
 * Variant C: TMSI mobile identity 1, TCH/F needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR075C()
{
  BEGIN_CASE ("paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075c")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075('C');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR075-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 or 2.
 *              A paging indication is forwarded to RR.
 * Variant D: TMSI mobile identity 1, Dual Rate needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR075D()
{
  BEGIN_CASE ("paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075d")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075('D');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR075-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 or 2.
 *              A paging indication is forwarded to RR.
 * Variant E: TMSI mobile identity 2, any channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR075E()
{
  BEGIN_CASE ("paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075e")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075('E');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR075-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 or 2.
 *              A paging indication is forwarded to RR.
 * Variant F: TMSI mobile identity 2, SDCCH needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR075F()
{
  BEGIN_CASE ("paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075f")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075('F');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR075-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 or 2.
 *              A paging indication is forwarded to RR.
 * Variant G: TMSI mobile identity 2, TCH/F needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR075G()
{
  BEGIN_CASE ("paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075g")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075('G');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR075-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 or 2.
 *              A paging indication is forwarded to RR.
 * Variant H: TMSI mobile identity 2, Dual Rate needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR075H()
{
  BEGIN_CASE ("paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075h")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_tmsi_for_ms_mobile_identity_1_or_2__alr075('H');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR076-------------------------------
 *
 * Description:	The MS is in idle mode. It receives paging request type 2 messages.
 *              The message content is not forwarded to RR,
 *              because the message content doesnt match to the MS identities.
 ---------------------------------------------------------------------------------*/
T_CASE ALR076()
{
  BEGIN_CASE ("paging_req_2_not_for_ms_alr076")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      SEND  ( mphc_data_ind_29()   );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant A: IMSI mobile identity 3, any channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR077A()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant B: IMSI mobile identity 3, SDCCH needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR077B()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant C: IMSI mobile identity 3, TCH/F needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR077C()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077c")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('C');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant D: IMSI mobile identity 3, Dual Rate needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR077D()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077d")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('D');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant E: IMSI mobile identity 3, no channel indication
 *-------------------------------------------------------------------------------*/
T_CASE ALR077E()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077e")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('E');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant F: TMSI mobile identity 3, any channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR077F()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077f")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('F');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant G: TMSI mobile identity 3, SDCCH needed,
 *-------------------------------------------------------------------------------*/
T_CASE ALR077G()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077g")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('G');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant H: TMSI mobile identity 3, TCH/F needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR077H()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077h")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('H');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant I: TMSI mobile identity 3, Dual Rate needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR077I()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077i")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('I');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR077-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR.
 * Variant J: TMSI mobile identity 3, no channel indication
 *-------------------------------------------------------------------------------*/
T_CASE ALR077J()
{
  BEGIN_CASE ("paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077j")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_imsi_tmsi_for_ms_mobile_identity_3__alr077('J');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR078-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR. The IMSI is less than 15 digits and the TMSI has only
 *              one valid byte, but is signalled in different ways by the infrastructure
 * Variant A: Short IMSI mobile identity 3, any channel needed, 10 digits
 *-------------------------------------------------------------------------------*/
T_CASE ALR078A()
{
  BEGIN_CASE ("paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR078-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR. The IMSI is less than 15 digits and the TMSI has only
 *              one valid byte, but is signalled in different ways by the infrastructure
 * Variant B: Short TMSI mobile identity 3, SDCCH needed, 1 byte from the network
 *-------------------------------------------------------------------------------*/
T_CASE ALR078B()
{
  BEGIN_CASE ("paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR078-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR. The IMSI is less than 15 digits and the TMSI has only
 *              one valid byte, but is signalled in different ways by the infrastructure
 * Variant C: Short TMSI mobile identity 3, TCH/F needed, 2 bytes from the network
 *-------------------------------------------------------------------------------*/
T_CASE ALR078C()
{
  BEGIN_CASE ("paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078c")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078('C');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR078-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR. The IMSI is less than 15 digits and the TMSI has only
 *              one valid byte, but is signalled in different ways by the infrastructure
 * Variant D: Short TMSI mobile identity 3, Dual Rate needed, 3 bytes from the network
 *-------------------------------------------------------------------------------*/
T_CASE ALR078D()
{
  BEGIN_CASE ("paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078d")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078('D');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR078-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging request type 2 message
 *              corresponding to the actual IMSI or TMSI in information element in
 *              the information element mobile identity 3. A paging indication is
 *              forwarded to RR. The IMSI is less than 15 digits and the TMSI has only
 *              one valid byte, but is signalled in different ways by the infrastructure
 * Variant E: Short TMSI mobile identity 3, No channel indication, 4 bytes from the network
 *-------------------------------------------------------------------------------*/
T_CASE ALR078E()
{
  BEGIN_CASE ("paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078e")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_2_short_imsi_tmsi_for_ms_mobile_identity_3__alr078('E');
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR079-------------------------------
 *
 * Description:	The MS is in idle mode. It receives an empty paging message.
 *              The message content is not forwarded to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR079()
{
  BEGIN_CASE ("paging_req_3_empty_paging_message_alr079")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      SEND  ( mphc_data_ind_32()   );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant A: TMSI mobile identity 1, any channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080A()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant B: TMSI mobile identity 1, SDCCH needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080B()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant C: TMSI mobile identity 1, TCH/F needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080C()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080c")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('C');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant D: TMSI mobile identity 1, Dual Rate needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080D()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080d")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('D');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant E: TMSI mobile identity 2, any channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080E()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080e")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('E');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant F: TMSI mobile identity 2, SDCCH needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080F()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080f")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('F');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant G: TMSI mobile identity 2, TCH/F needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080G()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080g")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('G');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant H: TMSI mobile identity 2, Dual Rate needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080H()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080h")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('H');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant I: TMSI mobile identity 3, any channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080I()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080i")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('I');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant J: TMSI mobile identity 3, SDCCH needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080J()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080j")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('J');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant K: TMSI mobile identity 3, TCH/F needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080K()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080k")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('K');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant L: TMSI mobile identity 3, Dual Rate needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080L()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080l")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('L');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant M: TMSI mobile identity 3, no channel indication
 *-------------------------------------------------------------------------------*/
T_CASE ALR080M()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080m")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('M');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant N: TMSI mobile identity 4, any channel needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080N()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080n")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('N');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant O: TMSI mobile identity 4, SDCCH needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080O()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080o")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('O');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant P: TMSI mobile identity 4, TCH/F needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080P()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080p")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('P');
    } HANDLE_DEACT;
  }
}
/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant Q: TMSI mobile identity 4, Dual Rate needed
 *-------------------------------------------------------------------------------*/
T_CASE ALR080Q()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080q")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('Q');
    } HANDLE_DEACT;
  }
}
/*------------------------Following is for ALR080-------------------------------
 *
 * Description:	The MS is in idle mode. It receives a paging message corresponding
 *              to the actual TMSI in the information elements mobile identity 1 to 4.
 *              A paging indication is forwarded to RR.
 * Variant R: TMSI mobile identity 4, no channel indication
 *-------------------------------------------------------------------------------*/
T_CASE ALR080R()
{
  BEGIN_CASE ("paging_req_3_tmsi_for_ms_alr080r")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      paging_req_3_tmsi_for_ms_alr080('R');
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR081-------------------------------
 *
 * Description:	The MS is in idle mode. It receives paging request type 3 messages.
 *              The message content is not forwarded to RR, because the message
 *              content doesnt match to the MS identities.
 *-------------------------------------------------------------------------------*/

T_CASE ALR081()
{
  BEGIN_CASE ("paging_req_3_not_for_ms_alr081")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      SEND  ( mphc_data_ind_34()   );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR082-------------------------------
 *
 * Description:	For a multiframe period of five multiframes it is expected that
 *              all 25 reports from layer 1 reading of BCCH information is started.
 *              Nine reports are received in the preamble. All four reports a
 *              measurement report is send to RR. Changed System Information
 *              Messages are forwarded to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR082()
{
  BEGIN_CASE ("bcch_reading_changed_sys_infos_alr082")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('D');
      meas_rep_bs_pa_mfrms_5();

      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_6()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_6()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_6()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_6()     );
      SEND  ( mphc_data_ind_1()           );
      SEND  ( mphc_data_ind_2()           );
      SEND  ( mphc_data_ind_35()          );
      AWAIT ( mph_unitdata_ind_si1_chg(ARFCN_23)       );
      SEND  ( mphc_data_ind_4()           );
      SEND  ( mphc_data_ind_15()          );
      SEND  ( mphc_data_ind_1()           );
      AWAIT ( mph_unitdata_ind_si1(ARFCN_23)        );
      SEND  ( mphc_data_ind_17()          );
      SEND  ( mphc_data_ind_18()          );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR086-------------------------------
 *
 * Description:	RR starts a full list power measurement.
 *              No suitable channel is available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR086()
{
  BEGIN_CASE ("initiation_by_rr_no_channels_available_alr086")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      SEND  ( mph_power_req_2()    );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mph_power_cnf_2()    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR089-------------------------------
 *
 * Description:	All synchronization attempts to the available channels fail. RR
 *              is informed by an error indication with the cause no BCCH available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR089()
{
  BEGIN_CASE ("find_bcch_carrier_all_channels_failed_then_error_indication_alr089")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      find_bcch_carrier_second_channel_alr088();

      SEND  ( mphc_ncell_sync_ind_failed(ARFCN_14)  );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_14)        );
      SEND  ( mph_bsic_req(ARFCN_124)         );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)  );
      SEND  ( mphc_ncell_sync_ind_failed(ARFCN_124) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_124)        );
      SEND  ( mph_bsic_req(ARFCN_1)        );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)  );
      SEND  ( mphc_ncell_sync_ind_failed(ARFCN_1) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_1)        );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR095-------------------------------
 *
 * Description:	RR rejects the BCCH carrier. No further channel is available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR095()
{
  BEGIN_CASE ("rr_rejects_bcch_carrier_no_further_channel_available_alr095")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      find_bcch_carrier_second_channel_alr088();
      read_bcch_data_alr091();
      rr_rejects_bcch_carrier_try_fourth_channel_alr094();

      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)             );
      continue_page_mode_reorg_after_hplnm_search_procedure('B');

      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1) );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR100-------------------------------
 *
 * Description:	The successful case of the EMO procedure is performed.
 *              No interfering signals are received.
 *-------------------------------------------------------------------------------*/

T_CASE ALR100()
{
  BEGIN_CASE ("emo_normal_case_alr100")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();

      SEND  ( mph_emo_req_1()           );
      AWAIT ( mphc_update_ba_list_1()   );
      SEND  ( mphc_meas_report_1()      );
      AWAIT ( mph_emo_meas_ind_1()      );
      SEND  ( mph_neighbourcell_req_2() );
      AWAIT ( mphc_update_ba_list_2()   );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR101-------------------------------
 *
 * Description:	The successful case of the EMO procedure is performed.
 *-------------------------------------------------------------------------------*/

T_CASE ALR101()
{
  BEGIN_CASE ("emo_delayed_invalid_and_duplicated_ba_id_values_alr101")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();

      SEND  ( mph_emo_req_1()           );
      AWAIT ( mphc_update_ba_list_1()   );
      SEND  ( mphc_meas_report_2()      );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)   );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)   );
      AWAIT ( mph_measurement_ind_20()  );
      SEND  ( mph_emo_req_2()           );
      AWAIT ( mphc_update_ba_list_3()   );
      SEND  ( mphc_meas_report_1()      );
      SEND  ( mphc_meas_report_3()      );
      AWAIT ( mph_emo_meas_ind_2()      );
      SEND  ( mph_neighbourcell_req_2() );
      AWAIT ( mphc_update_ba_list_2()   );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR101-------------------------------
 *
 * Description:This test checks the scenario when a new strong cell is reported, an
 * all ongoing LIST SYNC REQ is stopped. A SYNC REQ for the new cell is sent. As
 * soon as sync_ind for this new cell is received, The Interupted LIST SYNC REQ is 
 * again.
 *-------------------------------------------------------------------------------*/

T_CASE ALR102()
{
  BEGIN_CASE ("ALR102: Synchronization to new strong cell synch confirmation in DEDICATED state")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();
      SEND  ( mph_neighbourcell_req_2() );
      AWAIT ( mphc_update_ba_list_2()   );

      SEND  ( mphc_meas_report_4()      );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_124));
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_1) );

      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,25)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_124) );

      // tim_ncsync expiry
      wait_ncsync_dedicated_102(20);
      AWAIT ( mphc_ncell_list_sync_req_1d()      );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1)   );

      SEND  ( mphc_meas_report_5()              );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );
      AWAIT ( mphc_stop_ncell_sync_req_all()    );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_14)   );

      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_14) );

      AWAIT ( mphc_ncell_list_sync_req_1e()     );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_124) );

      // tim_ncsync expiry
      wait_ncsync_dedicated_102a(20);
      AWAIT ( mphc_ncell_list_sync_req_1f()     );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1)   );

      SEND  ( mphc_meas_report_5()              );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_124) );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_14)  );   

      SEND  ( mphc_meas_report_5()              );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );  

      SEND  ( mphc_meas_report_4()      );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  ); 



    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR101-------------------------------
 *
 * Description:This test checks the scenario when a new strong cell is reported, an
 * all ongoing LIST SYNC REQ is stopped. A SYNC REQ for the new cell is sent. As
 * soon as sync_ind for this new cell is received, The Interupted LIST SYNC REQ is sent
 * again. This case checks condition when more than one NCELL Resync request was pending
 * when the new storng cell was reported.
 *-------------------------------------------------------------------------------*/

T_CASE ALR103()
{
  BEGIN_CASE ("ALR103: Synchronization to new strong cell synch confirmation in DEDICATED state")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();
      SEND  ( mph_neighbourcell_req_2() );
      AWAIT ( mphc_update_ba_list_2()   );

      SEND  ( mphc_meas_report_4()      );

      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_124)   );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_1) );

      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_124) );

      // tim_ncsync expiry
      wait_ncsync_dedicated_102(20);
      AWAIT ( mphc_ncell_list_sync_req_1d()      );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_meas_report_5()              );

      AWAIT ( mphc_stop_ncell_sync_req_all()    );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_14)   );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_14) );

      AWAIT ( mphc_ncell_list_sync_req_1d()      );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_124) );

      // tim_ncsync expiry
      wait_ncsync_dedicated_102(20);
      AWAIT ( mphc_ncell_list_sync_req_1h()      );

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR101-------------------------------
 *
 * Description:This test checks the scenario when a new strong cell is reported, an
 * all ongoing LIST SYNC REQ is stopped. A SYNC REQ for the new cell is sent. As
 * soon as sync_ind for this new cell is received, The Interupted LIST SYNC REQ is sent
 * again. This case checks the boundry condition when maximum 6 cells are included in
 * the list sync request.
 *-------------------------------------------------------------------------------*/

T_CASE ALR104()
{
  BEGIN_CASE ("emo_delayed_invalid_and_duplicated_ba_id_values_alr101")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();
      SEND  ( mph_neighbourcell_req_2a() );
      AWAIT ( mphc_update_ba_list_2a()   );

      SEND  ( mphc_meas_report_6()      );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_1)   );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_11) );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_15) );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_20) );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_24) );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_25) );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_11)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_15)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_20)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_24)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_25)   );

      /*  tim_ncsync expiry */
      wait_ncsync_dedicated_102b(20);
      AWAIT ( mphc_ncell_list_sync_req_6()      );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_11)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_15)   );

      SEND  ( mphc_meas_report_6a()      );

      AWAIT ( mphc_stop_ncell_sync_req_all()    );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_26)   );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_26)   );


      AWAIT ( mphc_ncell_list_sync_req_6a()      );



    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR101-------------------------------
 *
 * Description:This test checks the scenario when a new strong cell is reported, an
 * all ongoing LIST SYNC REQ is stopped. A SYNC REQ for the new cell is sent. As
 * soon as sync_ind for this new cell is received, The Interupted LIST SYNC REQ is sent
 * again. This case checks the condition when one of the synchronized cell is replaced
 * by new stong cell.
 *-------------------------------------------------------------------------------*/

T_CASE ALR105()
{
  BEGIN_CASE ("emo_delayed_invalid_and_duplicated_ba_id_values_alr101")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();
      SEND  ( mph_neighbourcell_req_2a() );
      AWAIT ( mphc_update_ba_list_2a()   );

      SEND  ( mphc_meas_report_6()      );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_1)   );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_11) );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_15) );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_20) );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_24) );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_25) );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_11)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_15)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_20)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_24)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_25)   );

      /*  tim_ncsync expiry */
      wait_ncsync_dedicated_102b(20);
      AWAIT ( mphc_ncell_list_sync_req_6()      );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_1)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_11)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_15)   );


      SEND  ( mphc_meas_report_6()      );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_25)   );

      SEND  ( mphc_meas_report_6a()      );
      AWAIT ( mphc_stop_ncell_sync_req_all()    );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_26)   );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)  );

      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_26)   );

      AWAIT ( mphc_ncell_list_sync_req_6a()      );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_20)   );
      SEND  ( mphc_ncell_sync_ind_fn(ARFCN_24)   );

    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR150-------------------------------
 *
 * Description:	A non-synchronized handover for the mobile station is
 *              configured without starting time.
 *-------------------------------------------------------------------------------*/

T_CASE ALR150()
{
  BEGIN_CASE ("non_synchronized_handover_alr150")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      meas_ind_with_start_sync_followed_by_rach_procedure();
      immediate_assignment_for_the_mobile_station_alr056();

      SEND  ( mph_dedicated_req_2() );
      AWAIT ( mphc_async_ho_req_1(HOT_REGULAR) );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR152-------------------------------
 *
 * Description:	The type approval testcase 26.6.13.8 is performed.
 *-------------------------------------------------------------------------------*/

T_CASE ALR152()
{
  BEGIN_CASE ("fta_26_6_13_8_alr152")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      meas_ind_with_start_sync_followed_by_rach_procedure();
      immediate_assignment_for_the_mobile_station_alr056();

      SEND  ( mph_freq_redef_req_2()        );
      AWAIT ( mphc_change_frequency_2()     );
      SEND  ( mph_dedicated_req_5()         );
      AWAIT ( mphc_async_ho_req_2()         );
      SEND  ( mphc_async_ho_con_1()         );
      SEND  ( mphc_handover_finished_1(HO_COMPLETE) );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK)      );
      SEND  ( mph_dedicated_fail_req_1()    );
      AWAIT ( mphc_handover_fail_req_1()    );
      SEND  ( mphc_handover_fail_con_1()    );
      AWAIT ( mph_dedicated_fail_cnf_1()    );
      SEND  ( mphc_change_frequency_con_1() );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR153-------------------------------
 *
 * Description:	The MS is in dedicated mode. It receives a system info
 *              type 5 message (forwarded to RR), a system info type 5bis
 *              message (forwarded to RR), a system info type 6 message
 *              (forwarded to RR), an I-frame for SMS (forwarded to DL),
 *              the first system info type 5 message again (compared and
 *              not forwarded by ALR) and a changed system info type 6
 *              message (forwarded to RR).
 *-------------------------------------------------------------------------------*/

T_CASE ALR153()
{
  BEGIN_CASE ("sacch_downlink_messages_alr153")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      rr_select_second_channel_alr012('A');

      SEND  ( mph_random_access_req_1()        );
      AWAIT ( mphc_stop_rxlev_periodic_req()   );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_scell_nbcch_req_full()      );
      AWAIT ( mphc_ra_req_1()                  );
      SEND  ( mphc_ra_con_1()                  );
      AWAIT ( mph_random_access_cnf_1()        );
      AWAIT ( mphc_ra_req_2()                  );
      SEND  ( mphc_data_ind_36()               );
      AWAIT ( mph_unitdata_ind_15()            );
      SEND  ( mph_dedicated_req_3()            );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_stop_ccch_req()             );
      AWAIT ( mphc_stop_ra_req_1()             );
      AWAIT ( mphc_immed_assign_req_2()        );
      SEND  ( mphc_immed_assign_con_1()        );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );
      SEND  ( mph_freq_redef_req_1()           );
      AWAIT ( mphc_change_frequency_1()        );
      SEND  ( mph_dedicated_req_4()            );
      AWAIT ( mphc_channel_assign_req_1()      );
      SEND  ( mphc_channel_assign_con_1()      );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );
      SEND  ( mph_dedicated_fail_req_1()       );
      AWAIT ( mphc_channel_assign_req_2()      );
      SEND  ( mphc_channel_assign_con_1()      );
      AWAIT ( mph_dedicated_fail_cnf_1()       );
      SEND  ( mphc_data_ind_37()               );
      AWAIT ( mph_unitdata_ind_16()            );
      SEND  ( mphc_data_ind_38()               );
      AWAIT ( mph_unitdata_ind_17()            );
      SEND  ( mphc_data_ind_39()               );
      AWAIT ( mph_unitdata_ind_18()            );
      SEND  ( mphc_data_ind_40()               );
      AWAIT ( ph_data_ind_1()                  );
      SEND  ( mphc_data_ind_37()               );
      SEND  ( mphc_data_ind_41()               );
      AWAIT ( mph_unitdata_ind_19()            );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR154-------------------------------
 *
 * Description:	A handover is performed. It is checked that the serving cell
 *              channel numbers are correctly set.
 * Variant A:	AMR not signalled
 *-------------------------------------------------------------------------------*/
T_CASE ALR154A()
{
  BEGIN_CASE ("handover_serving_cell_parameter_alr154a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      meas_ind_with_start_sync_followed_by_rach_procedure();
      immediate_assignment_for_the_mobile_station_alr056();
      handover_serving_cell_parameter_alr154('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR154-------------------------------
 *
 * Description:	A handover is performed. It is checked that the serving cell
 *              channel numbers are correctly set.
 * Variant B:	AMR signalled
 *-------------------------------------------------------------------------------*/
T_CASE ALR154B()
{
  BEGIN_CASE ("handover_serving_cell_parameter_alr154b")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      meas_ind_with_start_sync_followed_by_rach_procedure();
      immediate_assignment_for_the_mobile_station_alr056();

      handover_serving_cell_parameter_alr154('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR154-------------------------------
 *
 * Description:	A handover is performed. The target cell has been sync
 *
 *-------------------------------------------------------------------------------*/
T_CASE ALR154C()
{
  BEGIN_CASE ("handover_target_cell_synchronized")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      meas_ind_with_start_sync_followed_by_rach_procedure();
      immediate_assignment_for_the_mobile_station_alr056();
      handover_serving_cell_parameter_alr154('C');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR154-------------------------------
 *
 * Description:	A handover is performed. The target cell has not been sync
 *
 *-------------------------------------------------------------------------------*/
T_CASE ALR154D()
{
  BEGIN_CASE ("handover_target_cell_not_synchronized")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      meas_ind_with_start_sync_followed_by_rach_procedure();
      /*Take MS to dedicated state*/
      immediate_assignment_for_the_mobile_station_alr056();
      /* Start Handover*/
      SEND   (mph_dedicated_req_6('D'));
      #ifdef  FF_EGPRS
        AWAIT ( mphc_async_ho_req_3('D') );
        SEND  ( mphc_handover_finished_1(HO_SYNC_FAIL) );
        AWAIT ( mph_dedicated_cnf_1(DEDI_RES_SYNC_FAIL));
      #else // FF_EGPRS
        AWAIT ( mph_dedicated_cnf_1(DEDI_RES_CELL_NOT_SYNC));
      #endif  // FF_EGPRS
    } HANDLE_DEACT;
  }
}

#ifdef FF_EGPRS
/*------------------------Following is for ALR155-------------------------------
 *
 * Description: Blind Handover-success case
 *
 *-------------------------------------------------------------------------------*/
T_CASE ALR155()
{
  BEGIN_CASE ("Blind Handover-success case")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      meas_ind_with_start_sync_followed_by_rach_procedure();
      /*Take MS to dedicated state*/
      immediate_assignment_for_the_mobile_station_alr056();
      /* Start Handover*/
      SEND   (mph_dedicated_req_6('D'));
      AWAIT ( mphc_async_ho_req_3('D') );
      SEND  ( mphc_async_ho_con_1()        );
      SEND  ( mphc_handover_finished_1(HO_COMPLETE) );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK));
      /*Verify that timing info is updated*/
      SEND (mphc_meas_report_4() );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)       );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_1,0x19)           );
      SEND (mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1) );
      SEND (mphc_meas_report_4() );
      AWAIT ( mph_measurement_ind_bho_1(ARFCN_1,0x19)           );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR156-------------------------------
 *
 * Description: Blind Handover-Failure case
 *
 *-------------------------------------------------------------------------------*/
T_CASE ALR156()
{
  BEGIN_CASE ("Blind Handover-Failure case")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      meas_ind_with_start_sync_followed_by_rach_procedure();
      /*Take MS to dedicated state*/
      immediate_assignment_for_the_mobile_station_alr056();
      /*Timing before handover*/
      SEND(mphc_meas_report_5());
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)       );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)           );
      /* Start Handover*/
      SEND   (mph_dedicated_req_6('D'));
      AWAIT (mphc_stop_ncell_sync_req_dummy() );
      AWAIT ( mphc_async_ho_req_3('D') );
      SEND  ( mphc_async_ho_con_1()        );
      SEND  ( mphc_handover_finished_1(HO_SYNC_FAIL) );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_SYNC_FAIL));
      /*HO failed : Revert to old channels*/
      SEND  ( mph_dedicated_fail_req_1()    );
      AWAIT ( mphc_handover_fail_req_1()    );
      SEND  ( mphc_handover_fail_con_1()    );
      AWAIT ( mph_dedicated_fail_cnf_1()    );
      /*Verify that timing info has not changed*/
      SEND(mphc_meas_report_5());
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)       );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)           );
      SEND (mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1) );
    } HANDLE_DEACT;
  }
}
#endif // FF_EGPRS

/*------------------------Following is for ALR157-------------------------------
 *
 * Description:	This test case checks a fix for bug reported with issue #12103
 *              The BA list contains the serving cell 23 and
 *              the neighbour cells 1, 14 and 124.
 *              The fieldstrength is 56 for channel 23, 12 for channel 1,
 *              44 for channel 14 and 25 for channel 124 (all values in GSM range).
 *              The ranking for the neighbour cells is 14, 124 and at least channel 1.
 *              Each reports contains two fieldstrength values per channel.
 *              The multiframe period is set to 6. The first measurement report
 *              is send to RR after five reports from PL. Then after each
 *              three reports from PL a measurement report is send to RR.
 *              One synchronisation attempt to channel 14 fails and is repeated (ALR048)
 *              Then a connection is established and a non synchroneous handover
 *              is performed, which fails. The handover is repeated later
 *              (and fails again). The second handover must use the same timing
 *              information as the first one.
 *              Later the call is released and RR decides to start a cell
 *              reselection on serving cell 23
 *-------------------------------------------------------------------------------*/
T_CASE ALR157()
{
  BEGIN_CASE ("handover_fails_alr157")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');
      sync_to_ncell_14_fails_once();

      SEND  ( mph_random_access_req_1()          );
      AWAIT ( mphc_stop_rxlev_periodic_req()     );
      AWAIT ( mphc_stop_scell_bcch_req()         );
      AWAIT ( mphc_scell_nbcch_req_full()        );
      AWAIT ( mphc_ra_req_1()                    );
      SEND  ( mphc_ra_con_1()                    );
      AWAIT ( mph_random_access_cnf_1()          );
      AWAIT ( mphc_ra_req_2()                    );
      immediate_assignment_for_the_mobile_station_alr056();
      SEND  ( mph_neighbourcell_req_4('A') );
      AWAIT ( mphc_update_ba_list_2()    );
      MUTE  (500);
      SEND  ( mph_dedicated_req_6('C') );
      AWAIT ( mphc_async_ho_req_3('C') );
      SEND  ( mphc_async_ho_con_1()        );
      SEND  ( mphc_handover_finished_1(HO_TIMEOUT));
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_TIMEOUT));
      SEND  ( mph_dedicated_fail_req_1()    );
      AWAIT ( mphc_handover_fail_req_1()    );
      SEND  ( mphc_handover_fail_con_1()    );
      AWAIT ( mph_dedicated_fail_cnf_1()    );
      MUTE (1000);
      SEND  ( mph_dedicated_req_6('C') );
      AWAIT ( mphc_async_ho_req_3('C') );
      SEND  ( mphc_async_ho_con_1()        );
      SEND  ( mphc_handover_finished_1(HO_TIMEOUT));
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_TIMEOUT));
      SEND  ( mph_dedicated_fail_req_1()    );
      AWAIT ( mphc_handover_fail_req_1()    );
      SEND  ( mphc_handover_fail_con_1()    );
      AWAIT ( mph_dedicated_fail_cnf_1()    );
      MUTE (1000);
      SEND  ( mph_stop_dedicated_req()      );
      AWAIT ( mphc_stop_dedicated_req()     );
      SEND  ( mphc_stop_dedicated_con()     );
      AWAIT ( mph_stop_dedicated_cnf()      );
      SEND  ( mph_idle_req_5 ('E')          );
      AWAIT ( mphc_new_scell_req_1()        );
    } HANDLE_DEACT;
  }
}
/*------------------------Following is for ALR158-------------------------------
 *
 * Description:	This test case checks a fix for bug reported with issue #16068
 *              The BA list contains the serving cell 23 and
 *              the neighbour cells 1, 14 and 124.
 *              The fieldstrength is 56 for channel 23, 12 for channel 1,
 *              44 for channel 14 and 25 for channel 124 (all values in GSM range).
 *              The ranking for the neighbour cells is 14, 124 and at least channel 1.
 *              Each reports contains two fieldstrength values per channel.
 *              The multiframe period is set to 6. The first measurement report
 *              is send to RR after five reports from PL. Then after each
 *              three reports from PL a measurement report is send to RR.
 *              One synchronisation attempt to channel 14 fails and is repeated (ALR048)
 *              Then a connection is established and a non synchroneous handover
 *              to cell 124 is performed, which succeeds.
 *              The new BA list no longer contains the (strong) cell 14.
 *              The confirmation of neighbour cell synchronization must not
 *              include the cell 14, but only cells belonging to the BA list
 *              For any reason the synchronization of cell 23 lasts very long.
 *              Therefore it is't included into the MPH_MEASUREMENT_IND 10 seconds
 *              later. Again 10 seconds later the synchonization attempt is stopped
 *              and a new one is started.
 *-------------------------------------------------------------------------------*/
T_CASE ALR158()
{
  BEGIN_CASE ("handover_fails_alr158")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');
      sync_to_ncell_14_fails_once();

      SEND  ( mph_random_access_req_1()          );
      AWAIT ( mphc_stop_rxlev_periodic_req()     );
      AWAIT ( mphc_stop_scell_bcch_req()         );
      AWAIT ( mphc_scell_nbcch_req_full()        );
      AWAIT ( mphc_ra_req_1()                    );
      SEND  ( mphc_ra_con_1()                    );
      AWAIT ( mph_random_access_cnf_1()          );
      AWAIT ( mphc_ra_req_2()                    );
      immediate_assignment_for_the_mobile_station_alr056();
      SEND  ( mph_neighbourcell_req_4('A')       );
      AWAIT ( mphc_update_ba_list_2()            );

      MUTE  (500);
      SEND  ( mph_dedicated_req_6('E')           );
      AWAIT ( mphc_async_ho_req_3('E')           );
      SEND  ( mphc_async_ho_con_1()              );
      SEND  ( mphc_handover_finished_1(HO_COMPLETE));
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK)   );
      SEND  ( mph_neighbourcell_req_1('A')       );
      AWAIT ( mphc_update_ba_list_8()            );
      wait_ncsync_dedicated(18);
      AWAIT ( mphc_ncell_list_sync_req_1b()      );
      AWAIT ( mph_measurement_ind_34()           );
      SEND  ( mphc_ncell_sync_ind_3a()           );

      wait_ncsync_dedicated(20);
      AWAIT ( mph_measurement_ind_34()           );
      MUTE (500);
      wait_ncsync_dedicated(22);
      AWAIT ( mphc_stop_ncell_sync_req_all()     );
      AWAIT ( mphc_ncell_list_sync_req_1b()      );
      AWAIT ( mph_measurement_ind_34()           );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR160-------------------------------
 *
 * Description:	A handover is performed. Before the handover there was a large
 *              BA list of 20 entries in use. After the handover a different list
 *              of further 20 enries is received. The overflow case of the ALR
 *              BA list is tested
 *-------------------------------------------------------------------------------*/
T_CASE ALR160()
{
  BEGIN_CASE ("handover_full_ba_list_alr160")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      meas_ind_with_start_sync_followed_by_rach_procedure();
      immediate_assignment_for_the_mobile_station_alr056();
      neighbour_cell_in_dedicated_alr155();
      handover_serving_cell_parameter_alr154('B');
      neighbour_cell_after_ho_alr156();
    } HANDLE_DEACT;
  }
}




/*------------------------Following is for ALR202-------------------------------
 *
 * Description:	RR starts a cell selection. No channel is available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR202()
{
  BEGIN_CASE ("initiation_by_rr_no_channels_available_alr202")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_dcs_1800__alr200();
      SEND  ( mph_power_req_3()    );
      AWAIT ( mphc_init_l1_req_3() );
      SEND  ( mphc_init_l1_con_1() );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2)   );
      AWAIT ( mph_power_cnf_2()    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR205-------------------------------
 *
 * Description:	All synchronization attempts to the available channels are failed.
 *              RR is informed by an error indication with the cause no
 *              BCCH available.
 ---------------------------------------------------------------------------------*/

T_CASE ALR205()
{
  BEGIN_CASE ("find_bcch_carrier_all_channels_failed_then_error_indication_alr205")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr201();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr204();

      SEND  ( mphc_network_sync_ind_failed(ARFCN_580) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_580_STD_1800) );
      SEND  ( mph_bsic_req(ARFCN_885_STD_1800)        );
      AWAIT ( mphc_network_sync_req(ARFCN_885, SM_WIDE_MODE)        );
      SEND  ( mphc_network_sync_ind_failed(ARFCN_885) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_885_STD_1800) );
      SEND  ( mph_bsic_req(ARFCN_512_STD_1800)        );
      AWAIT ( mphc_network_sync_req(ARFCN_512, SM_WIDE_MODE)        );
      SEND  ( mphc_network_sync_ind_failed(ARFCN_512) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_512_STD_1800) );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR208-------------------------------
 *
 * Description:	A failed BCCH data block is read.
 *-------------------------------------------------------------------------------*/

T_CASE ALR208()
{
  BEGIN_CASE ("read_failed_bcch_data_alr208")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr201();
      find_bcch_carrier_first_channel_alr203();
      read_bcch_data_alr207();

      SEND  ( mphc_data_ind_46()   );
      AWAIT ( mph_error_ind_4()    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR211-------------------------------
 *
 * Description:	RR rejects the BCCH carrier. No further channel is available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR211()
{
  BEGIN_CASE ("rr_rejects_bcch_carrier_no_further_channel_available_alr211")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr201();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr204();
      find_bcch_carrier_second_channel_alr206();
      rr_rejects_bcch_carrier_try_third_channel_alr209();
      rr_rejects_bcch_carrier_try_fourth_channel_alr210();

      SEND  ( mph_bsic_req(NOT_PRESENT_16BIT)     );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR212-------------------------------
 *
 * Description:	RR selects the channel 14 after reading the BCCH carrier.
 * Variant A: with neighbourcell list
 *-------------------------------------------------------------------------------*/
T_CASE ALR212A()
{
  BEGIN_CASE ("rr_select_second_channel_alr212a")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr201();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr204();
      find_bcch_carrier_second_channel_alr206();
      rr_select_second_channel_alr212('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR212-------------------------------
 *
 * Description:	RR selects the channel 14 after reading the BCCH carrier.
 * Variant B: with empty neighbour cell list
 *-------------------------------------------------------------------------------*/
T_CASE ALR212B()
{
  BEGIN_CASE ("rr_select_second_channel_alr212b")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr201();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr204();
      find_bcch_carrier_second_channel_alr206();
      rr_select_second_channel_alr212('B');
    } HANDLE_DEACT;
  }
}
/*------------------------Following is for ALR212-------------------------------
 *
 * Description:	RR selects the channel 14 after reading the BCCH carrier.
 * Variant C: with neighbourcell list, but non-combined CCCH
 *-------------------------------------------------------------------------------*/
T_CASE ALR212C()
{
  BEGIN_CASE ("rr_select_second_channel_alr212c")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr201();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr204();
      find_bcch_carrier_second_channel_alr206();
      rr_select_second_channel_alr212('C');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR215-------------------------------
 *
 * Description:	RR has started a cell selection. During power measurements a
 *              new activation of cell selection is started by RR.
 *              The power measurement is restarted.
 *-------------------------------------------------------------------------------*/
T_CASE ALR215()
{
  BEGIN_CASE ("re_initiation_of_cell_selection_during_measurements_alr215")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_dcs_1800__alr200();

      SEND  ( mph_power_req_3()     );
      AWAIT ( mphc_init_l1_req_3()  );
      SEND  ( mphc_init_l1_con_1()  );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mph_power_req_3()     );
      AWAIT ( mphc_stop_rxlev_req() );
      AWAIT ( mphc_init_l1_req_3()  );
      SEND  ( mphc_init_l1_con_1()  );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)    );
      AWAIT ( mph_power_cnf_4()     );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR217-------------------------------
 *
 * Description:	During BCCH reading RR restarts a cell selection. This
 *              leads to a new full power measurement cycle.
 *-------------------------------------------------------------------------------*/

T_CASE ALR217()
{
  BEGIN_CASE ("re_initiation_of_cell_selection_during_bcch_reading_alr217")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr201();
      find_bcch_carrier_first_channel_alr203();
      read_bcch_data_alr207();

      SEND  ( mph_sync_req(CS_STOP_BCCH_READING)             );
      AWAIT ( mphc_stop_scell_bcch_req() );
      SEND  ( mph_power_req_3()            );
      AWAIT ( mphc_init_l1_req_3()         );
      SEND  ( mphc_init_l1_con_1()         );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_3_1800)           );
      AWAIT ( mph_power_cnf_4()            );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR244-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 *            	The idle mode is stopped if a normal cell selection is initiated.
 *-------------------------------------------------------------------------------*/

T_CASE ALR244()
{
  BEGIN_CASE ("stop_idle_mode_by_normal_cell_selection_alr244")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr201();
      find_bcch_carrier_first_channel_alr203();

      SEND  ( mph_classmark_req_1()              );
      SEND  ( mph_idle_req_8()                   );
      AWAIT ( mphc_stop_scell_bcch_req()         );
      AWAIT ( mphc_start_ccch_req_2()            );
      AWAIT ( mphc_scell_nbcch_req_full()        );
      SEND  ( mph_identity_req_1()               );
      SEND  ( mph_cbch_req_1()                   );
      SEND  ( mph_neighbourcell_req_6()          );
      AWAIT ( mphc_rxlev_periodic_req_6()        );
      SEND  ( mph_power_req_3()                  );
      AWAIT ( mphc_stop_scell_bcch_req()         );
      AWAIT ( mphc_stop_ccch_req()               );
      AWAIT ( mphc_stop_rxlev_periodic_req()     );
      AWAIT ( mphc_stop_scell_bcch_req()         );
      AWAIT ( mphc_init_l1_req_3()               );
      SEND  ( mphc_init_l1_con_1()               );
      AWAIT ( mphc_rxlev_req()                   );
    } HANDLE_DEACT;
  }
}

/*------------------------------------------------------------------------------
 *
 * Description:	The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. L1 cannot find any cells therefore MM
 *              receives a PLMN list with the current PLMN.
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR250()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, empty power cnf")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()  );
      AWAIT ( mphc_rxlev_req() );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req() );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req() );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req() );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req() );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mph_power_cnf_2()  );
      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}

/*------------------------------------------------------------------------------
 *
 * Description:	The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. The measurement results from L1 only contains a cell
 *              which is in the BA list and not in the stored BCCH list. Therefore
 *              it will be given low priority. For a parallel PLMN search only
 *              high and mid priority cells are synchronized therefore MM
 *              receives a PLMN list with the current PLMN.
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR251()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, only ncell")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()           );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8) );
      AWAIT ( mph_power_cnf_one_ncell()   );
      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)            );
      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}

/*------------------------------------------------------------------------------
 *
 * Description:	The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. L1 cannot synchronize to the cells therefore MM
 *              receives a PLMN list with the current PLMN.
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR252()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, sync fails")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                     );
      AWAIT ( mphc_rxlev_req()                      );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)           );
      AWAIT ( mphc_rxlev_req()                      );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)           );
      AWAIT ( mphc_rxlev_req()                      );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)           );
      AWAIT ( mphc_rxlev_req()                      );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)           );
      AWAIT ( mphc_rxlev_req()                      );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)           );
      AWAIT ( mph_power_cnf_one_ncell()             ); // for ALR its no ncell,
                                                        // so just use the same as above
      SEND  ( mph_bsic_req(ARFCN_32)                );
      //AWAIT ( mphc_network_sync_req(ARFCN_32)        );
      //SEND  ( mphc_network_sync_ind_failed(ARFCN_32) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32) );
      SEND  ( mphc_ncell_sync_ind_failed(ARFCN_32)  );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_32)         );
      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                      );
      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. L1 synchronizes to the cells and the BCCH is read.
 *              MM receives one PLMN.
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR253()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, one PLMN found")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       ); // for ALR its no ncell,
                                                                 // so just use the same as above
      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );
      SEND  ( mphc_ncell_sync_ind_cell32()                    );
      AWAIT ( mph_bsic_cnf_cell32()                           );
      AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)   );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                  );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)              );
      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)               );
      MUTE  ( 500 );

    } HANDLE_DEACT;
  }
}

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf the search is aborted.
 *              by a call.
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR254()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, abort due to call")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );

      abort_search_due_to_mo_call(VAR_PLMN_POWER_CNF);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf the search is aborted.
 *              due to a transition to leaving PIM.
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR255()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, abort due to leaving PIM")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );

      abort_search_due_to_leaving_pim(VAR_PLMN_POWER_CNF);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf the search is aborted.
 *              due to a cell reselection.
 *
 *-------------------------------------------------------------------------------*/
#if defined GPRS
T_CASE ALR256()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, abort due to a cell reselection")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      nc_pbcch_initial_sync(); // this is needed for doing the cell resel correctly

      SEND  ( mph_power_req_2()           );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8) );
      AWAIT ( mphc_rxlev_req()            );

      abort_search_due_to_cell_resel(VAR_PLMN_POWER_CNF);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf the search is aborted.
 *              due to a cell selection.
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR257()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, abort due to a cell selection")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()           );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8) );
      AWAIT ( mphc_rxlev_req()            );

      abort_search_due_to_cell_sel(VAR_PLMN_POWER_CNF);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}

/*-------------------------------------------------------------------------------
 *
 * The two testcases are different in RR but for ALR there is no difference
 * between MO and MT call.
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf the search is aborted.
 *              due to a CS paging on PCCCH.
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf the search is aborted.
 *              due to a CS paging on CCCH.
 *
 *-------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf the search is aborted
 *              due to a call.
 *
 *-------------------------------------------------------------------------------*/
T_CASE ALR258()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, bsic-cnf, abort due to call")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       ); // for ALR its no ncell,
                                                                 // so just use the same as above
      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );

      abort_search_due_to_mo_call(VAR_PLMN_BSIC_CNF);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf the search is aborted
 *              due to a leaving PIM.
 *
 *-------------------------------------------------------------------------------*/
T_CASE ALR259()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, bsic-cnf, abort due to leaving PIM")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       ); // for ALR its no ncell,
                                                                 // so just use the same as above
      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );

      abort_search_due_to_leaving_pim(VAR_PLMN_BSIC_CNF);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf the search is aborted.
 *              due to a cell reselection.
 *
 *-------------------------------------------------------------------------------*/
#if defined GPRS
T_CASE ALR260()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, bsic-cnf, abort due to a cell reselection")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      nc_pbcch_initial_sync(); // this is needed for doing the cell resel correctly

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       ); // for ALR its no ncell,
                                                                 // so just use the same as above
      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );

      abort_search_due_to_cell_resel(VAR_PLMN_BSIC_CNF);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf the search is aborted.
 *              due to a cell selection.
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR261()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, bsic-cnf, abort due to a cell selection")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       ); // for ALR its no ncell,
                                                                 // so just use the same as above
      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );

      abort_search_due_to_cell_sel(VAR_PLMN_BSIC_CNF);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
/*-------------------------------------------------------------------------------
 *
 * The two testcases are different in RR but for ALR there is no difference
 * between MO and MT call.
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf the search is aborted.
 *              due to a CS paging on CCCH
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf the search is aborted.
 *              due to a CS paging on PCCCH
 *
 *-------------------------------------------------------------------------------*/

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the SIs the search is aborted due
 *              to call
 *
 *-------------------------------------------------------------------------------*/
T_CASE ALR262()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, SI reading, abort due to call")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       );

      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );
      SEND  ( mphc_ncell_sync_ind_cell32()                    );
      AWAIT ( mph_bsic_cnf_cell32()                           );
      AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)   );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                  );

      abort_search_due_to_mo_call(VAR_PLMN_SI);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the SIs the search is aborted due
 *              to leaving PIM
 *
 *-------------------------------------------------------------------------------*/
T_CASE ALR263()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, SI reading, abort due to leaving PIM")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       );

      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );
      SEND  ( mphc_ncell_sync_ind_cell32()                    );
      AWAIT ( mph_bsic_cnf_cell32()                           );
      AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)   );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                  );

      abort_search_due_to_leaving_pim(VAR_PLMN_SI);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the SIs the search is aborted due
 *              to cell reselection
 *
 *-------------------------------------------------------------------------------*/
#if defined GPRS
T_CASE ALR264()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, SI reading, abort due to cell reselection")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      nc_pbcch_initial_sync(); // this is needed for doing the cell resel correctly

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       );

      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );
      SEND  ( mphc_ncell_sync_ind_cell32()                    );
      AWAIT ( mph_bsic_cnf_cell32()                           );
      AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)   );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                  );

      abort_search_due_to_cell_resel(VAR_PLMN_SI);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the SIs the search is aborted due
 *              to cell selection
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR265()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, SI reading, abort due to cell selection")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       );

      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );
      SEND  ( mphc_ncell_sync_ind_cell32()                    );
      AWAIT ( mph_bsic_cnf_cell32()                           );
      AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)   );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                  );

      abort_search_due_to_cell_sel(VAR_PLMN_SI);

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf the CCCH reading is
 *              configured.
 *
 *-------------------------------------------------------------------------------*/

T_CASE ALR266()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, CCCH reading is configured")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
#if defined GPRS
      SEND  ( mph_mon_ctrl_req(START_MON_CCCH)                );
      AWAIT ( mphc_start_ccch_req_8()                         );
#endif //GPRS
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       );

      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );
      SEND  ( mphc_ncell_sync_ind_cell32()                    );
      AWAIT ( mph_bsic_cnf_cell32()                           );
      AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)   );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                  );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)              );
      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)               );

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf the CCCH reading is
 *              configured.
 *
 *-------------------------------------------------------------------------------*/
#if defined GPRS
T_CASE ALR267()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, CCCH reading is configured II")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );

      SEND  ( mph_mon_ctrl_req(START_MON_CCCH)                );

      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );

      SEND  ( mph_mon_ctrl_req(STOP_MON_CCCH)                 );
      AWAIT ( mphc_stop_ccch_req()                            );

      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       );

      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );
      SEND  ( mphc_ncell_sync_ind_cell32()                    );
      AWAIT ( mph_bsic_cnf_cell32()                           );
      AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)   );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                  );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)              );
      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)               );

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf the CCCH reading is
 *              configured.
 *
 *-------------------------------------------------------------------------------*/
#if defined GPRS
T_CASE ALR268()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, bsic-cnf, CCCH reading is configured")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );

      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );

      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       );

      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );

      SEND  ( mph_mon_ctrl_req(START_MON_CCCH)                );
      AWAIT ( mphc_start_ccch_req_8()                         );

      SEND  ( mphc_ncell_sync_ind_cell32()                    );
      AWAIT ( mph_bsic_cnf_cell32()                           );
      AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)   );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                  );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)              );
      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)               );

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the SIs the CCCH reading is
 *              configured.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR269()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, SI reading, CCCH reading is configured")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

      SEND  ( mph_power_req_2()                               );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );

      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mphc_rxlev_req()                                );

      SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                     );
      AWAIT ( mph_power_cnf_one_ncell()                       );

      SEND  ( mph_bsic_req(ARFCN_32)                          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)           );

      SEND  ( mphc_ncell_sync_ind_cell32()                    );
      AWAIT ( mph_bsic_cnf_cell32()                           );
      AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)   );

      SEND  ( mph_mon_ctrl_req(START_MON_CCCH)                );
      AWAIT ( mphc_start_ccch_req_8()                         );

      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                  );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)              );

      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)               );

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/
#if defined GPRS
T_CASE ALR270()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])             );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                   );
        if(i != 1) {
          SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                            );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],1)         );
        }
        if(i == 1) {
          SEND  ( mphc_ncell_sync_ind_cell32()                       );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],3)         );
        }

        SEND  ( mph_power_req_2()                                    );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mph_neighbourcell_req_plmn_search(NOT_PRESENT_16BIT) );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mph_power_cnf_one_ncell()                            );

        SEND  ( mph_bsic_req(ARFCN_32)                               );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                );

        SEND  ( mphc_ncell_sync_ind_cell32()                         );
        AWAIT ( mph_bsic_cnf_cell32()                                );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)        );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                    );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                       );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                   );
        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                    );

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR271()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])             );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                   );
        if(i != 1) {
          SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                            );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],1)         );
        }
        if(i == 1) {
          SEND  ( mphc_ncell_sync_ind_cell32()                       );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],3)         );
        }

        SEND  ( mph_power_req_2()                                    );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i],c2[i])       );
        if(i==0) SEND  ( mphc_ncell_sync_req_initial(ARFCN_25));
        if(i==3) SEND  ( mphc_ncell_sync_req_initial(ARFCN_32));

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mph_power_cnf_one_ncell()                            );

        SEND  ( mph_bsic_req(ARFCN_32)                               );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                );

        SEND  ( mphc_ncell_sync_ind_cell32()                         );
        AWAIT ( mph_bsic_cnf_cell32()                                );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)        );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                    );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                       );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                   );
        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                    );

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR272()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])                  );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                        );
        SEND  ( mphc_ncell_sync_ind_failed(c1[i])                         );
        if(i != 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }
        if(i == 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }

        SEND  ( mph_power_req_2()                                         );
        AWAIT ( mphc_rxlev_req()                                          );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );

        SEND  ( mph_neighbourcell_req_plmn_search(NOT_PRESENT_16BIT)      );
        AWAIT ( mph_measurement_ind_plmn_search  (c1[i],MPH_BSIC_UNKNOWN) );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mph_power_cnf_one_ncell()                                 );

        SEND  ( mph_bsic_req(ARFCN_32)                                    );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                     );

        SEND  ( mphc_ncell_sync_ind_cell32()                              );
        AWAIT ( mph_bsic_cnf_cell32()                                     );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)             );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                         );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                            );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                        );
        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                         );

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the power-cnf GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR273()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, power-cnf, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])                  );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                        );
        SEND  ( mphc_ncell_sync_ind_failed(c1[i])                         );
        if(i != 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }
        if(i == 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }

        SEND  ( mph_power_req_2()                                    );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i],c2[i])       );
        MUTE  (500);

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mph_power_cnf_one_ncell()                            );

        SEND  ( mph_bsic_req(ARFCN_32)                               );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                );

        SEND  ( mphc_ncell_sync_ind_cell32()                         );
        AWAIT ( mph_bsic_cnf_cell32()                                );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)        );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                    );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                       );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                   );
        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                    );

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR274()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, bsic-cnf, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d", i, c1[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        START_TIMEOUT(9500);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])             );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                   );
        if(i != 1) {
          SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                            );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],1)         );
        }
        if(i == 1) {
          SEND  ( mphc_ncell_sync_ind_cell32()                       );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],3)         );
        }

        SEND  ( mph_power_req_2()                                    );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mph_power_cnf_one_ncell()                            );

        SEND  ( mph_bsic_req(ARFCN_32)                               );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                );

        SEND  ( mph_neighbourcell_req_plmn_search(NOT_PRESENT_16BIT) );

        SEND  ( mphc_ncell_sync_ind_cell32()                         );
        AWAIT ( mph_bsic_cnf_cell32()                                );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)        );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                    );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                       );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                   );
        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                    );

        WAIT_TIMEOUT();
        /*This message is sent when GRR inform RR with RRGRR_NCELL_SYNC_REQ(SYNC_RECONFIRM)
        GRR maintain the 10 sec timer through measurement counter*/
        SEND  ( mph_neighbourcell_req_sync_reconfirm()             );
        // here C1 should be removed
        AWAIT ( mph_measurement_ind_plmn_search(c1[i],MPH_BSIC_UNKNOWN));

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR275()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, bsic-cnf, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        START_TIMEOUT(10000);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])             );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                   );
        if(i != 1) {
          SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                            );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],1)         );
        }
        if(i == 1) {
          SEND  ( mphc_ncell_sync_ind_cell32()                       );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],3)         );
        }

        SEND  ( mph_power_req_2()                                    );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mph_power_cnf_one_ncell()                            );

        SEND  ( mph_bsic_req(ARFCN_32)                               );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                );

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i],c2[i])       );
        MUTE  (500);

        SEND  ( mphc_ncell_sync_ind_cell32()                         );
        AWAIT ( mph_bsic_cnf_cell32()                                );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)        );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                    );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                       );
        if(i==3) SEND ( mphc_ncell_sync_ind_cell32() );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                   );
        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                    );


        WAIT_TIMEOUT();
        if(i==0) {
          /*This message mph_neighbourcell_req_sync_reconfirm is sent when GRR 
          inform RR with RRGRR_NCELL_SYNC_REQ(SYNC_RECONFIRM)
          GRR maintain the 10 sec timer through measurement counter*/
          SEND  ( mph_neighbourcell_req_sync_reconfirm()             );
          AWAIT ( mphc_ncell_list_sync_req_cell32d()                 );
          SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                  );
          AWAIT ( mphc_ncell_sync_req_initial(ARFCN_25)              );
          SEND  ( mphc_ncell_sync_ind (ARFCN_25)                     );
          AWAIT ( mph_measurement_ind_plmn_search(c2[i],1)           );
        }
        if(i==1) {
          SEND  ( mph_neighbourcell_req_sync_reconfirm()             );
          AWAIT ( mphc_ncell_list_sync_req_cell32a()    );
        }
        if(i==2) {
          SEND  ( mph_neighbourcell_req_sync_reconfirm()             );
          AWAIT ( mphc_ncell_list_sync_req_cell32d()       );
          SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                  );
          AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)    );
          SEND  ( mphc_ncell_sync_ind_cell32()             );
          AWAIT ( mph_measurement_ind_plmn_search(c2[i],3) );
        }

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR276()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, bsic-cnf, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])                  );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                        );
        SEND  ( mphc_ncell_sync_ind_failed(c1[i])                         );
        if(i != 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }
        if(i == 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }

        SEND  ( mph_power_req_2()                                         );
        AWAIT ( mphc_rxlev_req()                                          );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mph_power_cnf_one_ncell()                                 );

        SEND  ( mph_bsic_req(ARFCN_32)                                    );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                     );

        SEND  ( mph_neighbourcell_req_plmn_search(NOT_PRESENT_16BIT)      );
        AWAIT ( mph_measurement_ind_plmn_search  (c1[i],MPH_BSIC_UNKNOWN) );

        SEND  ( mphc_ncell_sync_ind_cell32()                              );
        AWAIT ( mph_bsic_cnf_cell32()                                     );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)             );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                         );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                            );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                        );
        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                         );

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the bsic-cnf GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR277()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, bsic-cnf, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        START_TIMEOUT(10000);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])                  );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                        );
        SEND  ( mphc_ncell_sync_ind_failed(c1[i])                         );
        if(i != 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }
        if(i == 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }

        SEND  ( mph_power_req_2()                                    );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mph_power_cnf_one_ncell()                            );

        SEND  ( mph_bsic_req(ARFCN_32)                               );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                );

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i],c2[i])       );
        MUTE  ( 500 );

        SEND  ( mphc_ncell_sync_ind_cell32()                         );
        AWAIT ( mph_bsic_cnf_cell32()                                );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)        );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                    );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                       );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                   );
        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                    );

        WAIT_TIMEOUT();
        if(i==0) {
          /*This message mph_neighbourcell_req_sync_reconfirm is sent when GRR 
          inform RR with RRGRR_NCELL_SYNC_REQ(SYNC_RECONFIRM)
          GRR maintain the 10 sec timer through measurement counter*/
          SEND  ( mph_neighbourcell_req_sync_reconfirm()   );
          AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)   );
          AWAIT ( mphc_ncell_sync_req_initial(ARFCN_25)    );
          SEND  ( mphc_ncell_sync_ind (ARFCN_25)           );
          AWAIT ( mph_measurement_ind_plmn_search(c2[i],1) );
        }
        if(i==1) {
          SEND  ( mph_neighbourcell_req_sync_reconfirm()   );
          AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)    );
        }
        if(i==2) {
          SEND  ( mph_neighbourcell_req_sync_reconfirm()   );
          AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)   );
          AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)    );
          SEND  ( mphc_ncell_sync_ind_cell32()             );
          AWAIT ( mph_measurement_ind_plmn_search(c2[i],3) );
        }

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the SIs GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR278()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, SI reading, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])             );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                   );
        if(i != 1) {
          SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                            );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],1)         );
        }
        if(i == 1) {
          SEND  ( mphc_ncell_sync_ind_cell32()                       );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],3)         );
        }

        SEND  ( mph_power_req_2()                                    );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mph_power_cnf_one_ncell()                            );

        SEND  ( mph_bsic_req(ARFCN_32)                               );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                );

        SEND  ( mphc_ncell_sync_ind_cell32()                         );
        AWAIT ( mph_bsic_cnf_cell32()                                );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)        );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                    );

        SEND  ( mph_neighbourcell_req_plmn_search(NOT_PRESENT_16BIT) );

        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                       );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                   );

        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                    );

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the Sis GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR279()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, SI reading, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])             );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                   );
        if(i != 1) {
          SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)                            );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],1)         );
        }
        if(i == 1) {
          SEND  ( mphc_ncell_sync_ind_cell32()                       );
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],3)         );
        }

        SEND  ( mph_power_req_2()                                    );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mph_power_cnf_one_ncell()                            );

        SEND  ( mph_bsic_req(ARFCN_32)                               );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                );

        SEND  ( mphc_ncell_sync_ind_cell32()                         );
        AWAIT ( mph_bsic_cnf_cell32()                                );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)        );

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i],c2[i])       );
        MUTE  ( 500 );

        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                    );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                       );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                   );
        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                    );

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS

/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the SIs GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR280()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, SI reading, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])                  );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                        );
        SEND  ( mphc_ncell_sync_ind_failed(c1[i])                         );
        if(i != 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }
        if(i == 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }

        SEND  ( mph_power_req_2()                                         );
        AWAIT ( mphc_rxlev_req()                                          );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mphc_rxlev_req()                                          );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                               );
        AWAIT ( mph_power_cnf_one_ncell()                                 );

        SEND  ( mph_bsic_req(ARFCN_32)                                    );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                     );

        SEND  ( mphc_ncell_sync_ind_cell32()                              );
        AWAIT ( mph_bsic_cnf_cell32()                                     );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)             );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                         );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                            );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                        );

        SEND  ( mph_neighbourcell_req_plmn_search(NOT_PRESENT_16BIT)      );
        AWAIT ( mph_measurement_ind_plmn_search  (c1[i],MPH_BSIC_UNKNOWN) );

        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                         );

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif //GPRS
/*-------------------------------------------------------------------------------
 *
 * Description: The MS is camped on a cell with PBCCH. Then MM starts a parallel
 *              PLMN search. While waiting for the Sis GRR updates the
 *              list of best ncells.
 *
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR281()
{
  BEGIN_CASE ("PBCCH, parallel PLMN search, SI reading, ncell list updated")
  {
    int i = 0;
    U16 c1[] = { ARFCN_124, ARFCN_32,          ARFCN_124 };
    U16 c2[] = { ARFCN_25,  NOT_PRESENT_16BIT, ARFCN_32  };

    for (;i<sizeof(c1)/sizeof(c1[0]);i++) {
      tdc_trace("Configuration i=%d, Cell1=%d Cell2=%d", i, c1[i], c2[i]);
      TRAP_FOR_DEACT {
        filter_and_routings_gsm_900__alr000();
        initiation_by_rr_4_channels_available_alr001();
        find_bcch_carrier_first_channel_alr003();
        read_bcch_data_alr007();
        rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i])                  );
        AWAIT ( mphc_ncell_sync_req_initial(c1[i])                        );
        SEND  ( mphc_ncell_sync_ind_failed(c1[i])                         );
        if(i != 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }
        if(i == 1) {
          AWAIT ( mph_measurement_ind_plmn_search  (c1[i],NOT_PRESENT_8BIT) );
        }

        SEND  ( mph_power_req_2()                                    );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );
        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mphc_rxlev_req()                                     );

        SEND  ( mphc_rxlev_ind(RXLEV_IDX_8)                          );
        AWAIT ( mph_power_cnf_one_ncell()                            );

        SEND  ( mph_bsic_req(ARFCN_32)                               );
        AWAIT ( mphc_ncell_sync_req_initial(ARFCN_32)                );

        SEND  ( mphc_ncell_sync_ind_cell32()                         );
        AWAIT ( mph_bsic_cnf_cell32()                                );
        AWAIT ( mphc_ncell_bcch_req_cell32(NCELL_BCCH_SI_3_4)        );
        SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_32)                    );
        AWAIT ( mph_unitdata_ind_si3(ARFCN_32)                       );
        AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_32)                   );

        SEND  ( mph_neighbourcell_req_plmn_search(c1[i],c2[i])       );
        MUTE  ( 500 );

        SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)                    );

        MUTE  ( 500 );
      } HANDLE_DEACT;
    }
  }
}
#endif  // GPRS

/*------------------------Following is for ALR602-------------------------------
 *
 * Description:	RR starts a cell selection. No channel is available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR602()
{
  BEGIN_CASE ("initiation_by_rr_no_channels_available_alr602")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_dualband_gsm_900_dcs_1800__alr600();

      SEND  ( mph_power_req_4()    );
      AWAIT ( mphc_init_l1_req_4() );
      SEND  ( mphc_init_l1_con_1() );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2_DUAL)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2_DUAL)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2_DUAL)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2_DUAL)   );
      AWAIT ( mphc_rxlev_req()   );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2_DUAL)   );
      AWAIT ( mph_power_cnf_2()    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR605-------------------------------
 *
 * Description:	All synchronization attempts to the available channels are failed.
 *              RR is informed by an error indication with the cause no BCCH available.
 ---------------------------------------------------------------------------------*/

T_CASE ALR605()
{
  BEGIN_CASE ("find_bcch_carrier_all_channels_failed_then_error_indication_alr605")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr604();

      SEND  ( mphc_network_sync_ind_failed(ARFCN_637) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_637_STD_DUAL) );

      SEND  ( mph_bsic_req(ARFCN_14_STD_DUAL)         );
      AWAIT ( mphc_network_sync_req(ARFCN_14, SM_WIDE_MODE)         );
      SEND  ( mphc_network_sync_ind_failed(ARFCN_14)  );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_14_STD_DUAL)  );

      SEND  ( mph_bsic_req(ARFCN_580_STD_DUAL)        );
      AWAIT ( mphc_network_sync_req(ARFCN_580, SM_WIDE_MODE)        );
      SEND  ( mphc_network_sync_ind_failed(ARFCN_580) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_580_STD_DUAL) );

      SEND  ( mph_bsic_req(ARFCN_124_STD_DUAL)        );
      AWAIT ( mphc_network_sync_req(ARFCN_124, SM_WIDE_MODE)        );
      SEND  ( mphc_network_sync_ind_failed(ARFCN_124) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_124_STD_DUAL) );

      SEND  ( mph_bsic_req(ARFCN_885_STD_DUAL)        );
      AWAIT ( mphc_network_sync_req(ARFCN_885, SM_WIDE_MODE)        );
      SEND  ( mphc_network_sync_ind_failed(ARFCN_885) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_885_STD_DUAL) );

      SEND  ( mph_bsic_req(ARFCN_512_STD_DUAL)        );
      AWAIT ( mphc_network_sync_req(ARFCN_512, SM_WIDE_MODE)        );
      SEND  ( mphc_network_sync_ind_failed(ARFCN_512) );
      AWAIT ( mph_bsic_cnf_failed(ARFCN_512_STD_DUAL) );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR608-------------------------------
 *
 * Description:	A failed BCCH data block is read.
 *-------------------------------------------------------------------------------*/

T_CASE ALR608()
{
  BEGIN_CASE ("read_failed_bcch_data_alr608")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();

      SEND  ( mphc_data_ind_5()    );
      AWAIT ( mph_error_ind_1()    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR611-------------------------------
 *
 * Description:	RR rejects the BCCH carrier. The next best channel (channel 14) is
 *              selected for synchronizing to frequency correction burst and
 *              synchron burst. RR rejects the BCCH carrier. The next best
 *              channel (channel 580) is selected for synchronizing to
 *              frequency correction burst and synchron burst.
 *              RR rejects the BCCH carrier. No further channel is available.
 *-------------------------------------------------------------------------------*/

T_CASE ALR611()
{
  BEGIN_CASE ("rr_rejects_bcch_carrier_no_further_channel_available_alr611")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr604();
      find_bcch_carrier_second_channel_alr606();

      SEND  ( mph_bsic_req(ARFCN_14_STD_DUAL)            );
      AWAIT ( mphc_stop_scell_bcch_req() );
      AWAIT ( mphc_network_sync_req(ARFCN_14, SM_NARROW_MODE)    );
      SEND  ( mphc_network_sync_ind_6()    );
      AWAIT ( mph_bsic_cnf_34()            );
      AWAIT ( mphc_new_scell_req_2()       );
      SEND  ( mphc_new_scell_con_1()       );
      AWAIT ( mphc_scell_nbcch_req_full()     );
      SEND  ( mphc_data_ind_49()           );
      AWAIT ( mph_unitdata_ind_si1(ARFCN_14)        );
      SEND  ( mph_bsic_req(ARFCN_580_STD_DUAL)            );
      AWAIT ( mphc_stop_scell_bcch_req() );
      AWAIT ( mphc_network_sync_req(ARFCN_580, SM_NARROW_MODE)    );
      SEND  ( mphc_network_sync_ind_14()   );
      AWAIT ( mph_bsic_cnf_35()            );
      AWAIT ( mphc_new_scell_req_6()       );
      SEND  ( mphc_new_scell_con_1()       );
      AWAIT ( mphc_scell_nbcch_req_full()     );
      SEND  ( mphc_data_ind_50()           );
      AWAIT ( mph_unitdata_ind_si1(ARFCN_580)        );
      SEND  ( mph_bsic_req(NOT_PRESENT_16BIT)             );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR612-------------------------------
 *
 * Description:	RR selects the channel 637 after reading the BCCH carrier.
 * Variant A: with neighbourcell list
 *-------------------------------------------------------------------------------*/
T_CASE ALR612A()
{
  BEGIN_CASE ("rr_select_second_channel_alr612a")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr604();
      find_bcch_carrier_second_channel_alr606();
      rr_select_second_channel_alr612('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR612-------------------------------
 *
 * Description:	RR selects the channel 637 after reading the BCCH carrier.
 * Variant B: with empty neighbour cell list
 *-------------------------------------------------------------------------------*/
T_CASE ALR612B()
{
  BEGIN_CASE ("rr_select_second_channel_alr612b")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr604();
      find_bcch_carrier_second_channel_alr606();
      rr_select_second_channel_alr612('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR615-------------------------------
 *
 * Description:	RR has started a cell selection. During power measurements
 *              a new activation of cell selection is started by RR.
 *              The power measurement is restarted.
 *-------------------------------------------------------------------------------*/

T_CASE ALR615()
{
  BEGIN_CASE ("re_initiation_of_cell_selection_during_measurements_alr615")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_dualband_gsm_900_dcs_1800__alr600();

      SEND  ( mph_power_req_4()     );
      AWAIT ( mphc_init_l1_req_4()  );
      SEND  ( mphc_init_l1_con_1()  );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mph_power_req_4()     );
      AWAIT ( mphc_stop_rxlev_req() );
      AWAIT ( mphc_init_l1_req_4()  );
      SEND  ( mphc_init_l1_con_1()  );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
      AWAIT ( mphc_rxlev_req()    );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)    );
      AWAIT ( mph_power_cnf_5()     );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR617-------------------------------
 *
 * Description:	During BCCH reading RR restarts a cell selection. This leads
 *              to a new full power measurement cycle.
 *-------------------------------------------------------------------------------*/

T_CASE ALR617()
{
  BEGIN_CASE ("re_initiation_of_cell_selection_during_bcch_reading_alr617")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();

      SEND  ( mph_sync_req(CS_STOP_BCCH_READING)             );
      AWAIT ( mphc_stop_scell_bcch_req() );
      SEND  ( mph_power_req_4()            );
      AWAIT ( mphc_init_l1_req_4()         );
      SEND  ( mphc_init_l1_con_1()         );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)           );
      AWAIT ( mphc_rxlev_req()           );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_DUAL)           );
      AWAIT ( mph_power_cnf_5()            );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR644-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 *            	The idle mode is stopped if a normal cell selection is initiated.
 *-------------------------------------------------------------------------------*/

T_CASE ALR644()
{
  BEGIN_CASE ("stop_idle_mode_by_normal_cell_selection_alr644")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();

      SEND  ( mph_classmark_req_3()          );
      SEND  ( mph_idle_req_10()              );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_start_ccch_req_10()       );
      AWAIT ( mphc_scell_nbcch_req_full()    );
      SEND  ( mph_identity_req_1()           );
      SEND  ( mph_cbch_req_1()               );
      SEND  ( mph_neighbourcell_req_8()      );
      AWAIT ( mphc_rxlev_periodic_req_8()    );
      SEND  ( mph_power_req_4()              );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_stop_ccch_req()           );
      AWAIT ( mphc_stop_rxlev_periodic_req() );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_init_l1_req_4()           );
      SEND  ( mphc_init_l1_con_1()           );
      AWAIT ( mphc_rxlev_req()               );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR650-------------------------------
 *
 * Description:	The multiband parameter is set to 0, that means the neighbourcells
 *              are ranked after the fieldstrength. The serving cell is 23
 *              (that means in the GSM 900 frequency band). The neighbourcell list
 *              contains eight channels : 1, 14, 25, 124, 512 580, 637 and 885.
 *              It is expected, that ALR starts synchronization to the six
 *              strongest cells 637, 25, 14, 512, 580, 885. This channels must
 *              be included in the measurement report to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR650()
{
  BEGIN_CASE ("multiband_0_serving_cell_is_gsm_900_8_channels_alr650")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();

      SEND  ( mph_classmark_req_3()                  );
      SEND  ( mph_idle_req_11()                      );
      AWAIT ( mphc_stop_scell_bcch_req()             );
      AWAIT ( mphc_start_ccch_req_9()                );
      AWAIT ( mphc_scell_nbcch_req_full()            );
      SEND  ( mph_neighbourcell_req_9()              );
      AWAIT ( mphc_rxlev_periodic_req_9()            );

      ncell_reporting_multiband_0_scell_is_gsm900();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR651-------------------------------
 *
 * Description:	The multiband parameter is set to 0, that means the neighbourcells
 *              are ranked after the fieldstrength. The serving cell is 578
 *              (that means in the DCS 1800 frequency band). The neighbourcell list
 *              contains eight channels : 1, 14, 25, 124, 512 580, 637 and 885. It is
 *              expected, that ALR starts synchronization to the six strongest cells
 *              14, 25, 512, 580, 637, 885. These channels must be included in
 *              the measurement report to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR651()
{
  BEGIN_CASE ("multiband_0_serving_cell_is_dcs_1800_8_channels_alr651")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();

      SEND  ( mph_classmark_req_3()                  );
      SEND  ( mph_idle_req_12()                      );
      AWAIT ( mphc_stop_scell_bcch_req()             );
      AWAIT ( mphc_start_ccch_req_9()                );
      AWAIT ( mphc_scell_nbcch_req_full()            );
      SEND  ( mph_neighbourcell_req_9()              );
      AWAIT ( mphc_rxlev_periodic_req_10()           );

      ncell_reporting_multiband_0_scell_is_gsm1800();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR652-------------------------------
 *
 * Description:	The multiband parameter is set to 1, that means at least one of
 *              the neighbourcells is member of the non-serving cell band
 *              (that means in the DCS 1800 frequency band). The serving cell
 *              is 23 (that means in the GSM 900 frequency band). The neighbourcell
 *              list contains eight channels : 1, 11, 14, 25,87, 124, 512 and 885.
 *              It is expected, that ALR starts synchronization to the six
 *              cells 14, 87, 25, 11, 1 and 512. This channels must be included
 *              in the measurement report to RR. ALR shall not synchronize to 885,
 *              although the cell has a better fieldstrength than most channels
 *              of the GSM 900 frequency band.
 *-------------------------------------------------------------------------------*/

T_CASE ALR652()
{
  BEGIN_CASE ("multiband_1_serving_cell_is_gsm_900_8_channels_alr652")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();

      SEND  ( mph_classmark_req_3()         );
      SEND  ( mph_idle_req_11()             );
      AWAIT ( mphc_stop_scell_bcch_req()    );
      AWAIT ( mphc_start_ccch_req_9()       );
      AWAIT ( mphc_scell_nbcch_req_full()   );
      SEND  ( mph_neighbourcell_req_10()    );
      AWAIT ( mphc_rxlev_periodic_req_11()  );

      ncell_reporting_multiband_1_scell_is_gsm900();
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR653-------------------------------
 *
 * Description:	The multiband parameter is set to 1, that at least one of the
 *              neighbourcells is member of the non-serving cell band (that means
 *              in the DCS 1800 frequency band). The serving cell is 23 (that means
 *              in the GSM 900 frequency band). The neighbourcell list contains four
 *              channels : 1, 14, 512 and 885. It is expected, that ALR starts
 *              synchronization to the six cells 1, 14, 512 and 885.
 *              This channels must be included in the measurement report to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR653()
{
  BEGIN_CASE ("multiband_1_serving_cell_is_gsm_900_4_channels_alr653")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();

      SEND  ( mph_classmark_req_3()         );
      SEND  ( mph_idle_req_11()             );
      AWAIT ( mphc_stop_scell_bcch_req()    );
      AWAIT ( mphc_start_ccch_req_9()       );
      AWAIT ( mphc_scell_nbcch_req_full()   );
      SEND  ( mph_neighbourcell_req_11()    );
      AWAIT ( mphc_rxlev_periodic_req_12()  );

      ncell_reporting_multiband_1_scell_is_gsm900_4ch();

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR654-------------------------------
 *
 * Description:	The multiband parameter is set to 1, that at least one of the
 *              neighbourcells is member of the non-serving cell band (that means
 *              in the GSM 900 frequency band). The serving cell is 637 (that means
 *              in the DCS 1800 frequency band). The neighbourcell list contains eight
 *              channels: 1, 14, 512, 513, 600, 700, 810 and 885. It is expected,
 *              that ALR starts synchronization to the six cells 14, 513, 600,
 *              700, 810 and 885. This channels must be included in the measurement
 *              report to RR. ALR shall not synchronize to 1, although the cell has
 *              a better fieldstrength than the most channels of the DCS 1800
 *              frequency band.
 *-------------------------------------------------------------------------------*/

T_CASE ALR654()
{
  BEGIN_CASE ("multiband_1_serving_cell_is_dcs_1800_8_channels_alr654")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();

      SEND  ( mph_classmark_req_3()                  );
      SEND  ( mph_idle_req_13()                      );
      AWAIT ( mphc_stop_scell_bcch_req()             );
      AWAIT ( mphc_start_ccch_req_9()                );
      AWAIT ( mphc_scell_nbcch_req_full()            );
      SEND  ( mph_neighbourcell_req_12()             );
      AWAIT ( mphc_rxlev_periodic_req_13()           );
      SEND  ( mphc_rxlev_periodic_ind_7()            );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)  );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_700) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_600) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_513) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_810) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_885) );
      SEND  ( mphc_rxlev_periodic_ind_7()            );
      AWAIT ( mph_measurement_ind_26()               );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)         );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)         );
      SEND  ( mphc_ncell_sync_ind (ARFCN_700)        );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_700)        );
      SEND  ( mphc_ncell_sync_ind (ARFCN_600)        );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_600)        );
      SEND  ( mphc_ncell_sync_ind (ARFCN_513)        );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_513)        );
      SEND  ( mphc_ncell_sync_ind (ARFCN_810)        );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_810)        );
      SEND  ( mphc_ncell_sync_ind (ARFCN_885)        );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_885)        );
      SEND  ( mphc_rxlev_periodic_ind_7()            );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)     );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_700)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_700)    );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_600)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_600)    );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_513)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_513)    );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_810)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_810)    );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_885)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_885)    );
      SEND  ( mphc_rxlev_periodic_ind_7()            );
      SEND  ( mphc_rxlev_periodic_ind_7()            );
      AWAIT ( mph_measurement_ind_27()               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)         );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_513)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_600)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_700)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_810)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_885)        );
      SEND  ( mphc_rxlev_periodic_ind_7()            );
      SEND  ( mphc_rxlev_periodic_ind_7()            );
      SEND  ( mphc_rxlev_periodic_ind_7()            );
      AWAIT ( mph_measurement_ind_27()               );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR655-------------------------------
 *
 * Description:	The multiband parameter is set to 1, that at least one of the
 *              neighbourcells is member of the non-serving cell band (that means
 *              in the GSM 900 frequency band). The serving cell is 637 (that means
 *              in the DCS 1800 frequency band). The neighbourcell list contains
 *              four channels: 1, 14, 512 and 885. It is expected, that ALR starts
 *              synchronization to the six cells 1, 14, 512 and 885. This channels
 *              must be included in the measurement report to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR655()
{
  BEGIN_CASE ("multiband_1_serving_cell_is_dcs_1800_4_channels_alr655")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();

      SEND  ( mph_classmark_req_3()                  );
      SEND  ( mph_idle_req_13()                      );
      AWAIT ( mphc_stop_scell_bcch_req()             );
      AWAIT ( mphc_start_ccch_req_9()                );
      AWAIT ( mphc_scell_nbcch_req_full()            );
      SEND  ( mph_neighbourcell_req_11()             );
      AWAIT ( mphc_rxlev_periodic_req_14()           );
      SEND  ( mphc_rxlev_periodic_ind_8()            );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)  );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_512) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_885) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)   );
      SEND  ( mphc_rxlev_periodic_ind_8()            );
      AWAIT ( mph_measurement_ind_26()               );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)                );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)                );
      SEND  ( mphc_ncell_sync_ind (ARFCN_512)               );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_512)               );
      SEND  ( mphc_ncell_sync_ind (ARFCN_885)               );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_885)               );
      SEND  ( mphc_ncell_sync_ind (ARFCN_1)               );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1)               );
      SEND  ( mphc_rxlev_periodic_ind_8()            );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)     );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_512)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_512)    );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_885)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_885)    );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1)       );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)      );
      SEND  ( mphc_rxlev_periodic_ind_8()            );
      SEND  ( mphc_rxlev_periodic_ind_8()            );
      AWAIT ( mph_measurement_ind_28()               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1)          );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)         );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_512)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_885)        );
      SEND  ( mphc_rxlev_periodic_ind_8()            );
      SEND  ( mphc_rxlev_periodic_ind_8()            );
      SEND  ( mphc_rxlev_periodic_ind_8()            );
      AWAIT ( mph_measurement_ind_28()               );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR700-------------------------------
 *
 * Description:	The idle mode is configured with Paging Reorganisation. After
 *              reception of the paging mode Paging Reorganisation no change
 *              occurs (received with an immediate assignment extended
 *              message. Then the mobile is paged with a paging request 2 message.
 *              It is expected that the paging is detected by the mobile.
 *-------------------------------------------------------------------------------*/

T_CASE ALR700()
{
  BEGIN_CASE ("page_mode_change_according_26_6_2_3_1_alr700")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      SEND  ( mphc_data_ind_51()              );
      SEND  ( mphc_data_ind_52()              );
      AWAIT ( mph_paging_ind_7()              );
      AWAIT ( mphc_start_ccch_req_8()         );
      AWAIT ( mphc_scell_nbcch_req_periodic() );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR701-------------------------------
 *
 * Description:	A handover is performed. It is checked that the serving cell
 *              channel numbers are correctly set.
 *-------------------------------------------------------------------------------*/

T_CASE ALR701()
{
  BEGIN_CASE ("channel_mode_modify_request_amr_half_rate_alr701")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();

      SEND  ( mph_channel_mode_req_1()         );
      AWAIT ( mphc_channel_mode_modify_req_1() );
      SEND  ( mphc_channel_mode_modify_con_1() );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR800-------------------------------
 *
 * Description:	The CBCH channel is configured. Then MMI requests reading of CBCH
 *              for the message identifier 3, 7 and 11 to 13.
 *-------------------------------------------------------------------------------*/
T_CASE ALR800A()
{
  BEGIN_CASE ("configuration_cbch_followed_by_mmi_request_alr800a")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      configuration_cbch_followed_by_mmi_request_alr800('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR800-------------------------------
 *
 * Description:	The CBCH channel is configured. Then MMI requests reading of CBCH
 *              for the message identifier 3, 7 and 11 to 13.
 *-------------------------------------------------------------------------------*/
T_CASE ALR800B()
{
  BEGIN_CASE ("configuration_cbch_followed_by_mmi_request_alr800b")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      configuration_cbch_followed_by_mmi_request_alr800('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR801-------------------------------
 *
 * Description:	MMI requests reading of CBCH for the message identifier
 *              3, 7 and 11 to 13. Then the CBCH channel is configured and
 *              reading of CBCH starts.
 *-------------------------------------------------------------------------------*/
T_CASE ALR801B()
{
  BEGIN_CASE ("mmi_request_followed_by_configuration_of_cbch_alr801b")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR801-------------------------------
 *
 * Description:	MMI requests reading of CBCH for the message identifier
 *              3, 7 and 11 to 13. Then the CBCH channel is configured and
 *              reading of CBCH starts.
 *-------------------------------------------------------------------------------*/
T_CASE ALR801C()
{
  BEGIN_CASE ("mmi_request_followed_by_configuration_of_cbch_alr801c")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('C');
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR802-------------------------------
 *
 * Description:	ALR receives a NULL message. No further reaction is expected.
 *-------------------------------------------------------------------------------*/

T_CASE ALR802()
{
  BEGIN_CASE ("reception_of_null_message_alr802")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      SEND  ( mphc_data_ind_53()   );
      MUTE(3000);
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR803-------------------------------
 *
 * Description:	ALR receives an expected CBCH message. The content is forwarded
 *              to MMI.
 * Variant A: single message identifier (7)
 *-------------------------------------------------------------------------------*/
T_CASE ALR803A()
{
  BEGIN_CASE ("reception_of_expected_cbch_message_alr803a")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      reception_of_expected_cbch_message_alr803('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR803-------------------------------
 *
 * Description:	ALR receives an expected CBCH message. The content is forwarded
 *              to MMI.
 * Variant B: message identifier specified by range (11), lower boundary
 *-------------------------------------------------------------------------------*/
T_CASE ALR803B()
{
  BEGIN_CASE ("reception_of_expected_cbch_message_alr803b")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      reception_of_expected_cbch_message_alr803('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR803-------------------------------
 *
 * Description:	ALR receives an expected CBCH message. The content is forwarded
 *              to MMI.
 * Variant C: message identifier specified by range (12), middle of the range
 *-------------------------------------------------------------------------------*/
T_CASE ALR803C()
{
  BEGIN_CASE ("reception_of_expected_cbch_message_alr803c")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      reception_of_expected_cbch_message_alr803('C');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR803-------------------------------
 *
 * Description:	ALR receives an expected CBCH message. The content is forwarded
 *              to MMI.
 * Variant D: message identifier specified by range (13), upper boundary
 *-------------------------------------------------------------------------------*/
T_CASE ALR803D()
{
  BEGIN_CASE ("reception_of_expected_cbch_message_alr803d")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      reception_of_expected_cbch_message_alr803('D');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR804-------------------------------
 *
 * Description:	ALR receives an unexpected CBCH message. No reaction is expected
 *              after receiving the first block.
 *-------------------------------------------------------------------------------*/

T_CASE ALR804()
{
  BEGIN_CASE ("reception_of_unexpected_cbch_message_alr804")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');

      SEND  ( mphc_data_ind_58()   );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR805-------------------------------
 *
 * Description:	ALR receives an unscheduled SCHEDULE message. The content is used
 *              for going into DRX mode.
 *-------------------------------------------------------------------------------*/

T_CASE ALR805()
{
  BEGIN_CASE ("reception_of_unscheduled_schedule_message_alr805")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');

      SEND  ( mphc_data_ind_59()         );
      AWAIT ( mphc_cbch_info_req_1()     );
      SEND  ( mphc_data_ind_60()         );
      SEND  ( mphc_data_ind_61()         );
      SEND  ( mphc_data_ind_62()         );
      AWAIT ( mphc_cbch_schedule_req_3() );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR806-------------------------------
 *
 * Description:	to test handling of update number in CBCH messages
 *-------------------------------------------------------------------------------*/

T_CASE ALR806()
{
  BEGIN_CASE ("handling_of_update_number_in_normal_cbch_NO_drx_state_alr806")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      reception_of_expected_cbch_message_alr803('A');
      AWAIT ( mphc_cbch_info_req_2());

      /* valid update number checked */
      reception_of_expected_cbch_message_alr803('E');
      AWAIT ( mphc_cbch_info_req_2());

      /* invalid update number checked */
      SEND  ( mphc_data_ind_54('F'));
      MUTE  ( 500 );

    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR807-------------------------------
 *
 * Description:	to test handling of invalid sequence numbers
 *-------------------------------------------------------------------------------*/

T_CASE ALR807()
{
  BEGIN_CASE ("handling_of__invalid_sequence_number_in_normal_cbch_in_no_drx_state_alr807")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");

      /* TB0 of Normal CBCH */
      SEND  ( mphc_data_ind_54('A'));
      AWAIT ( mphc_cbch_info_req_1());
      /* Invalid sequence number in Normal CBCH */
      SEND  ( mphc_data_ind_66());
      AWAIT ( mphc_cbch_info_req_2()    );

      reception_of_expected_cbch_message_alr803('A');

    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR808-------------------------------
 *
 * Description:	to test handling of inverted sequence numbers
 *-------------------------------------------------------------------------------*/

T_CASE ALR808()
{
  BEGIN_CASE ("handling_of_inverted_sequence_number_in_normal_cbch_in_no_drx_state_alr808")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");

      /* TB0 of Normal CBCH message*/
      SEND  ( mphc_data_ind_54('A'));
      AWAIT ( mphc_cbch_info_req_1());
      /* TB3 of Normal CBCH message*/
      SEND  ( mphc_data_ind_57());
      AWAIT ( mphc_cbch_info_req_2());

      /* TB0 of Normal CBCH message*/
      SEND  ( mphc_data_ind_54('A'));
      AWAIT ( mphc_cbch_info_req_1());
      /* TB2 of Normal CBCH message*/
      SEND  ( mphc_data_ind_56());
      AWAIT ( mphc_cbch_info_req_2());

      reception_of_expected_cbch_message_alr803('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR809-------------------------------
 *
 * Description:	to test handling of invalid sequence numbers in first CBCH Schedule message
 *-------------------------------------------------------------------------------*/

T_CASE ALR809()
{
  BEGIN_CASE ("handling_of__invalid_sequence_number_in_cbch_schedule_message_in_NO_drx_state_alr809")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");

      /* TB0 of CBCH Schedule message*/
      SEND  ( mphc_data_ind_67());
      AWAIT ( mphc_cbch_info_req_1());
      /* TB1 of CBCH Schedule message*/
      SEND  ( mphc_data_ind_60());
      /* invalid sequence number */
      SEND  ( mphc_data_ind_66());
      AWAIT ( mphc_cbch_info_req_2());

      reception_of_cbch_schedule_message_alr809('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR810-------------------------------
 *
 * Description:	to test handling of inverted sequence numbers
 *                         on a CBCH Schedule message in NO_DRX states
 *-------------------------------------------------------------------------------*/

T_CASE ALR810()
{
  BEGIN_CASE ("handling_of_inverted_sequence_number_in_cbch_schedule_message_in_NO_drx_state_alr810")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      /* TB0 of CBCH Schedule message*/
      SEND  ( mphc_data_ind_67());
      AWAIT ( mphc_cbch_info_req_1());
      /* TB3 of CBCH Schedule message*/
      SEND  ( mphc_data_ind_62());
      AWAIT ( mphc_cbch_info_req_2());

      /* TB0 of CBCH Schedule message*/
      SEND  ( mphc_data_ind_67());
      AWAIT ( mphc_cbch_info_req_1());
      /* TB2 of CBCH Schedule message*/
      SEND  ( mphc_data_ind_61());
      AWAIT ( mphc_cbch_info_req_2());

      reception_of_cbch_schedule_message_alr809('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR811-------------------------------
 *
 * Description:	Enter DRX state
 *				1) Receive CBCH messages as per to new CBCH blocks specified in
 *				the CBCH Schedule new message bitmap
 *				2) A scheduled CBCH Schedule message is recvd with invalid sequence number
 *-------------------------------------------------------------------------------*/

T_CASE ALR811()
{
  BEGIN_CASE ("handling_of__invalid_sequence_number_in_cbch_schedule_message_in_drx_state_alr811")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      /* Enter DRX State */
      reception_of_cbch_schedule_message_alr809('A');

      /* Reception of 4 Normal CBCH message, some marked NEW and some OLD*/
      reception_of_expected_cbch_message_alr803('A');
      AWAIT ( mphc_cbch_info_req_2());
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));

      /* CBCH schedule message expected */
      MUTE(2000);

      /* TB0 of CBCH Schedule message*/
      SEND  ( mphc_data_ind_67());
      AWAIT ( mphc_cbch_info_req_1());
      SEND  ( mphc_data_ind_60());
      /* invalid sequence number */
      SEND  ( mphc_data_ind_66());
      AWAIT ( mphc_cbch_info_req_2());
      AWAIT ( mphc_cbch_schedule_req_1());
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR812-------------------------------
 *
 * Description:	Enter DRX state
 *				1) Receive CBCH messages as per to new CBCH blocks specified in
 *				the CBCH Schedule new message bitmap
 *				2) A scheduled CBCH Schedule message is recvd with inverted sequence number
 *-------------------------------------------------------------------------------*/

T_CASE ALR812()
{
  BEGIN_CASE ("handling_of_inverted_sequence_number_in_cbch_schedule_message_in_drx_state_alr812")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      /* Enter DRX State */
      reception_of_cbch_schedule_message_alr809('A');

      /* Reception of 5 Normal CBCH message, some marked NEW and some OLD*/
      reception_of_expected_cbch_message_alr803('A');
      AWAIT ( mphc_cbch_info_req_2());
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));

      /* CBCH schedule message expected */
      MUTE (2000);


      /* TB0 of CBCH Schedule message*/
      SEND  ( mphc_data_ind_67());
      AWAIT ( mphc_cbch_info_req_1());
      SEND  ( mphc_data_ind_60());
      /* inverted sequence number */
      SEND  ( mphc_data_ind_62());
      AWAIT ( mphc_cbch_info_req_2());
      AWAIT ( mphc_cbch_schedule_req_1());
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR813-------------------------------
 *
 * Description:	Enter DRX state
 *				1) invalid sequence number in normal cbch
 *				2) inverted sequence number in normal cbch
 *				3) Check entry into 2nd DRX mode
 *-------------------------------------------------------------------------------*/

T_CASE ALR813()
{
  BEGIN_CASE ("handling_of_error_in_sequence_number_in_normal_cbch_in_drx_state_alr813")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      /* Enter DRX */
      reception_of_cbch_schedule_message_alr809('A');

      /* invalid sequence error */
      SEND  ( mphc_data_ind_54('A'));
      AWAIT ( mphc_cbch_info_req_1());
      SEND  ( mphc_data_ind_66());
      AWAIT ( mphc_cbch_info_req_2()    );

      /* inverted sequence number */
      SEND  ( mphc_data_ind_54('A'));
      AWAIT ( mphc_cbch_info_req_1());
      SEND  ( mphc_data_ind_56());
      AWAIT ( mphc_cbch_info_req_2());

      reception_of_expected_cbch_message_alr803('A');
      AWAIT ( mphc_cbch_info_req_2());
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      MUTE  (2000);

      /* Entry into 2nd DRX mode */
      reception_of_cbch_schedule_message_alr809('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR814-------------------------------
 *
 * Description:	Enter DRX state
 * 				1) handling of cb_error_ind
 *				2) Check entry into 2nd DRX mode
 *-------------------------------------------------------------------------------*/

T_CASE ALR814()
{
  BEGIN_CASE ("handling_of_errors_in_L2_frame_in_normal_cbch_in_drx_state_alr814")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      /* Enter DRX */
      reception_of_cbch_schedule_message_alr809('A');

      SEND  ( mphc_data_ind_54('A'));
      AWAIT ( mphc_cbch_info_req_1());
      /* error_flag = TURE */
      SEND  ( mphc_data_ind_68('A'));
      AWAIT ( mphc_cbch_info_req_2()    );

      reception_of_expected_cbch_message_alr803('A');
      AWAIT ( mphc_cbch_info_req_2());
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      MUTE  (2000);

      /* Entry into 2nd DRX mode */
      reception_of_cbch_schedule_message_alr809('B');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR815-------------------------------
 *
 * Description:	Enter DRX state
 * 				1) handling of inverted sequence error in unscheduled schedule message
 *                         2) handling of invalid sequence error in unscheduled schedule message
 *				3) Check entry into 2nd DRX mode
 *-------------------------------------------------------------------------------*/

T_CASE ALR815()
{
  BEGIN_CASE ("handling_of_error_in_unscheduled_cbch_schedule_message_in_drx_state_alr815")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");

      /* Enter DRX State */
      reception_of_cbch_schedule_message_alr809('A');

      /* inverted sequence number in an unscheduled schedule message */
      SEND  ( mphc_data_ind_59()         );
      AWAIT ( mphc_cbch_info_req_1()     );
      SEND  ( mphc_data_ind_61()         );
      AWAIT ( mphc_cbch_info_req_2()    );

      /* invalid sequence number in an unscheduled schedule message */
      SEND  ( mphc_data_ind_59()         );
      AWAIT ( mphc_cbch_info_req_1()     );
      SEND  ( mphc_data_ind_66()         );
      AWAIT ( mphc_cbch_info_req_2()    );

      reception_of_expected_cbch_message_alr803('A');
      AWAIT ( mphc_cbch_info_req_2());
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      MUTE  (2000);


      /* Entry into 2nd DRX  */
      reception_of_cbch_schedule_message_alr809('B');
    } HANDLE_DEACT;
  }
}
/*------------------------Following is for ALR816-------------------------------
 *
 * Description:	Enter DRX state
 * 				1) Recv all new messages
 *				2) Recv a NORMAL CBCH message when a scheduled schedule message is
 *                             expected
 *                         3) NO_DRX_STATE should be entered
 *                         4) Recv a CBCH Schedule message
 *                         5) Enter DRX mode (First DRX )
 *-------------------------------------------------------------------------------*/

T_CASE ALR816()
{
  BEGIN_CASE ("handling_of_missing_scheduled_cbch_schedule_message_816")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");

      /* Enter DRX */
      reception_of_cbch_schedule_message_alr809('A');

      /* Recv 5 Scheduled Normal CBCH messages */
      reception_of_expected_cbch_message_alr803('A');
      AWAIT ( mphc_cbch_info_req_2());
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));


      /* CBCH Schedule message expected */
      MUTE  (2000);

      /* Normal CBCH message recvd instead of CBCH Schedule message */
      SEND  ( mphc_data_ind_54('A'));
      /* Enter NO DRX State */
      AWAIT ( mphc_cbch_schedule_req_1());

      /* Enter DRX mode (First DRX ) */
      reception_of_cbch_schedule_message_alr809('A');
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR817-------------------------------
 *
 * Description:	Enter DRX state
 *				1) invalid sequence number in normal cbch
 *				2) inverted sequence number in normal cbch
 *                         3) reception of messages in extended CBCH channel, NOR_DRX mode
 *				4) Check entry into 2nd DRX mode for normal CBCH Channel
 *-------------------------------------------------------------------------------*/

T_CASE ALR817()
{
  BEGIN_CASE ("simultaneous_operation_of_normal_and_extended_cbch_channels_alr817")
  {
   TRAP_FOR_DEACT {
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_failed_then_second_channel_alr004();
      find_bcch_carrier_second_channel_alr006();
      mmi_request_followed_by_configuration_of_cbch_alr801('A');
      COMMAND("PL REDIRECT MMI TAP");
      /* Enter DRX */
      reception_of_cbch_schedule_message_alr809('A');

      /* invalid sequence error in NORMAL CBCH*/
      SEND  ( mphc_data_ind_54('A'));
      AWAIT ( mphc_cbch_info_req_1());
      SEND  ( mphc_data_ind_66());
      AWAIT ( mphc_cbch_info_req_2()    );

      /* inverted sequence number in NORMAL CBCH*/
      SEND  ( mphc_data_ind_54('A'));
      AWAIT ( mphc_cbch_info_req_1());
      SEND  ( mphc_data_ind_56());
      AWAIT ( mphc_cbch_info_req_2());

      /* reception of message in extended CBCH channel */
      SEND  ( mphc_data_ind_69() );
      AWAIT ( mphc_cbch_info_req_3()    );
      SEND  ( mphc_data_ind_70()        );
      SEND  ( mphc_data_ind_71()        );
      SEND  ( mphc_data_ind_72()        );
      AWAIT ( mmi_cbch_ind_1('B')   );
      AWAIT ( mphc_cbch_info_req_2());

      /* NORMAL CBCH */
      reception_of_expected_cbch_message_alr803('A');
      AWAIT ( mphc_cbch_info_req_2());

      /* error reading extended CBCH channel */
      SEND  ( mphc_data_ind_68('B'));

      /* Normal CBCH, message already read*/
      SEND  ( mphc_data_ind_54('A'));
      SEND  ( mphc_data_ind_54('A'));
      MUTE  (2000);

      /* Entry into 2nd DRX mode in NORMAL CBCH*/
      reception_of_cbch_schedule_message_alr809('B');
    } HANDLE_DEACT;
  }
}

T_CASE ALR818()
{
  char variant = 'A';
  BEGIN_CASE ("geographical_scope_handling_in_cbch_messages_alr818")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      SEND  ( mphc_data_ind_73());
      AWAIT ( mph_unitdata_ind_si3(ARFCN_23, CELL_IDENT_2));
      definition_of_bcch_allocation_alr046('A');
      SEND  ( mphc_data_ind_73());

      /* SCELL = ARFCN_23 */

      COMMAND("PL REDIRECT MMI TAP");

      /* configure CBCH reading*/
      SEND  ( mmi_cbch_req_2(variant));
      SEND  ( mph_cbch_req_3(variant));
      AWAIT ( mphc_config_cbch_req_2(variant));
      AWAIT ( mphc_cbch_schedule_req_1());
      AWAIT ( mphc_cbch_schedule_req_2());

      /* reception of message id = 11 . Scope = CELL_WIDE ONLY */
      reception_of_expected_cbch_message_alr803('B');
      AWAIT ( mphc_cbch_info_req_2());

      /* reception of message id = 7 . Scope = Location Area Wide */
      SEND  ( mphc_data_ind_54('G'));
      AWAIT ( mphc_cbch_info_req_1());
      SEND  ( mphc_data_ind_55());
      SEND  ( mphc_data_ind_56());
      SEND  ( mphc_data_ind_57());
      AWAIT ( mmi_cbch_ind_1('F'));
      AWAIT ( mphc_cbch_info_req_2());

      /* Measurements and NCell sync/Bcch reading */
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14));
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124));
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1));
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mph_measurement_ind_13());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_ncell_sync_ind (ARFCN_14));
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14));
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14));
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1));
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_ncell_sync_ind_3());
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1));
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124));
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1));
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mph_measurement_ind_14());
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1));
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14));
      AWAIT ( mph_unitdata_ind_si3(ARFCN_124));

      wait_ncsync_idle_2_14(6);
      
      AWAIT ( mphc_ncell_list_sync_req_1a());
      AWAIT ( mph_measurement_ind_14());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mph_measurement_ind_14());

      SEND  ( mphc_ncell_sync_ind (ARFCN_14));
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1));
      SEND  ( mphc_ncell_sync_ind_3());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mph_measurement_ind_14());

      /* Cell Reselection to ARFCN_14, same Location area as ARFCN_23 */
      SEND  ( mph_idle_req_15());
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT  ( mphc_stop_ccch_req());
      AWAIT ( mphc_stop_rxlev_periodic_req());
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mphc_stop_cbch_req());
      AWAIT ( mphc_new_scell_req_10());
      SEND  ( mphc_new_scell_con_1());
      AWAIT ( mphc_start_ccch_req_5());
      AWAIT ( mphc_scell_nbcch_req_full());
      SEND  ( mphc_data_ind_63());
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14));
      AWAIT ( mphc_start_ccch_req_12());
      SEND  ( mphc_data_ind_64());
      AWAIT ( mph_unitdata_ind_si1(ARFCN_14));
      SEND  ( mphc_data_ind_11());
      AWAIT ( mphc_start_ccch_req_13());
      SEND  ( mphc_data_ind_65());
      AWAIT ( mph_unitdata_ind_si2(ARFCN_14));
      SEND  ( mph_idle_req_16());
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mphc_scell_nbcch_req_periodic());
      AWAIT ( mphc_rxlev_periodic_req_15());

      /* Start CBCH reading on the new SC */
      SEND  ( mph_cbch_req_3(variant));
      AWAIT ( mphc_config_cbch_req_2(variant));
      AWAIT ( mphc_cbch_schedule_req_1());
      AWAIT ( mphc_cbch_schedule_req_2());


      SEND  ( mph_neighbourcell_req_13());
      AWAIT ( mphc_rxlev_periodic_req_16());

      /* reception of message id = 7 . Scope = Location Area Wide.
      This message was already read in the same LA.
      ALR should not read TC 1 ,2 and 3*/
      SEND  ( mphc_data_ind_54('G'));
      MUTE(1000);

      /* Resend CBCH message_id = 11 */
      reception_of_expected_cbch_message_alr803('B');
      /* This message should be passed up to MMI */
      AWAIT ( mphc_cbch_info_req_2());



    } HANDLE_DEACT;
  }
}


T_CASE ALR819()
{
  char variant = 'A';
  BEGIN_CASE ("geographical_scope_handling_in_cbch_messages_alr819")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      SEND  ( mphc_data_ind_73());
      AWAIT ( mph_unitdata_ind_si3(ARFCN_23, CELL_IDENT_2));
      definition_of_bcch_allocation_alr046('A');
      SEND  ( mphc_data_ind_73());

      /* SCELL = ARFCN_23 */

      COMMAND("PL REDIRECT MMI TAP");

      /* configure CBCH reading*/
      SEND  ( mph_cbch_req_3(variant));
      SEND  ( mmi_cbch_req_2(variant));
      AWAIT ( mphc_config_cbch_req_2(variant));
      AWAIT ( mphc_cbch_schedule_req_1());
      AWAIT ( mphc_cbch_schedule_req_2());

      /* reception of message id = 11 . Scope = CELL_WIDE ONLY */
      reception_of_expected_cbch_message_alr803('B');
      AWAIT ( mphc_cbch_info_req_2());

      /* reception of message id = 7 . Scope = Location Area Wide */
      SEND  ( mphc_data_ind_54('G'));
      AWAIT ( mphc_cbch_info_req_1());
      SEND  ( mphc_data_ind_55());
      SEND  ( mphc_data_ind_56());
      SEND  ( mphc_data_ind_57());
      AWAIT ( mmi_cbch_ind_1('F'));
      AWAIT ( mphc_cbch_info_req_2());

      /* Measurements and NCell sync/Bcch reading */
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14));
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124));
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1));
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mph_measurement_ind_13());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_ncell_sync_ind (ARFCN_14));
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14));
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14));
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1));
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_ncell_sync_ind_3());
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1));
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124));
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1));
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mph_measurement_ind_14());
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1));
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14));
      AWAIT ( mph_unitdata_ind_si3(ARFCN_124));

      wait_ncsync_idle_2_14(6);
      AWAIT ( mphc_ncell_list_sync_req_1a());

      AWAIT ( mph_measurement_ind_14());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mph_measurement_ind_14());
      SEND  ( mphc_ncell_sync_ind (ARFCN_14));
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1));
      SEND  ( mphc_ncell_sync_ind_3());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_rxlev_periodic_ind_2());
      SEND  ( mphc_rxlev_periodic_ind_2());
      AWAIT ( mph_measurement_ind_14());

      /* Cell Reselection to ARFCN_14, same Location area as ARFCN_23 */
      SEND  ( mph_idle_req_15());
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mphc_stop_ccch_req());
      AWAIT ( mphc_stop_rxlev_periodic_req());
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mphc_stop_cbch_req());
      AWAIT ( mphc_new_scell_req_10());
      SEND  ( mphc_new_scell_con_1());
      AWAIT ( mphc_start_ccch_req_5());
      AWAIT ( mphc_scell_nbcch_req_full());
      SEND  ( mphc_data_ind_63());
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14));
      AWAIT ( mphc_start_ccch_req_12());
      SEND  ( mphc_data_ind_64());
      AWAIT ( mph_unitdata_ind_si1(ARFCN_14));
      SEND  ( mphc_data_ind_11());
      AWAIT ( mphc_start_ccch_req_13());
      SEND  ( mphc_data_ind_65());
      AWAIT ( mph_unitdata_ind_si2(ARFCN_14));
      SEND  ( mph_idle_req_16());
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mphc_scell_nbcch_req_periodic());
      AWAIT ( mphc_rxlev_periodic_req_15());

      /* Start CBCH reading on the new SC */
      SEND  ( mph_cbch_req_3(variant));
      AWAIT ( mphc_config_cbch_req_2(variant));
      AWAIT ( mphc_cbch_schedule_req_1());
      AWAIT ( mphc_cbch_schedule_req_2());


      SEND  ( mph_neighbourcell_req_13());
      AWAIT ( mphc_rxlev_periodic_req_16());

      /* reception of message id = 7 . Scope = Location Area Wide.
	   This message was already read in the same LA.
	   ALR should not read TC 1 ,2 and 3*/
      SEND  ( mphc_data_ind_54('G'));
      MUTE(1000);

      /* Resend CBCH message_id = 11 */
      reception_of_expected_cbch_message_alr803('B');
      /* This message should be passed up to MMI */
      AWAIT ( mphc_cbch_info_req_2());



    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR900-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the
 *              neighbour cells 1, 14 and 124. The fieldstrength is 56 for
 *              channel 23, 12 for channel 1, 44 for channel 14 and 25 for
 *              channel 124 (all values in GSM range). The ranking for the
 *              neighbour cells is 14, 124 and channel 1. Each reports contains
 *              two fieldstrength values per channel. The multiframe period
 *              is set to 6.The first measurement report is send to RR after
 *              five reports from PL. Then after each three reports from
 *              PL a measurement report is send to RR.
 *              RR starts a cell reselection to neighbourcell 14.
 *              After the end of cell reselection a new neighbourcell list is
 *              forwarded to ALR with the neighbourcells 2 and 15. It is
 *              expected that layer 1 is configured with the new serving cell 14
 *              and the neighbourcells 2 and 15.
 *-------------------------------------------------------------------------------*/

T_CASE ALR900()
{
  BEGIN_CASE ("successful_case_alr900")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      SEND  ( mphc_rxlev_periodic_ind_2()      );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)          );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)          );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      AWAIT ( mph_measurement_ind_13()         );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)          );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)          );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)          );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)     );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)          );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_ncell_sync_ind_3()          );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1)          );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124)          );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)     );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1)          );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)     );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      AWAIT ( mph_measurement_ind_14()         );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1)             );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)             );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_124)             );
      wait_ncsync_idle_2_14(6);
      AWAIT ( mphc_ncell_list_sync_req_1a()    );

      AWAIT ( mph_measurement_ind_14()         );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      AWAIT ( mph_measurement_ind_14()         );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)          );
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)          );
      SEND  ( mphc_ncell_sync_ind_3()          );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      SEND  ( mphc_rxlev_periodic_ind_2()      );
      AWAIT ( mph_measurement_ind_14()         );
      SEND  ( mph_idle_req_15()                );
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mphc_stop_ccch_req()             );
      AWAIT ( mphc_stop_rxlev_periodic_req() );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_new_scell_req_10()          );
      SEND  ( mphc_new_scell_con_1()           );
      AWAIT ( mphc_start_ccch_req_5()          );
      AWAIT ( mphc_scell_nbcch_req_full()         );
      SEND  ( mphc_data_ind_63()               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)             );
      AWAIT ( mphc_start_ccch_req_12()         );
      SEND  ( mphc_data_ind_64()               );
      AWAIT ( mph_unitdata_ind_si1(ARFCN_14)            );
      SEND  ( mphc_data_ind_11()               );
      AWAIT ( mphc_start_ccch_req_13()         );
      SEND  ( mphc_data_ind_65()               );
      AWAIT ( mph_unitdata_ind_si2(ARFCN_14)            );
      SEND  ( mph_idle_req_16()                );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_scell_nbcch_req_periodic()  );
      AWAIT ( mphc_rxlev_periodic_req_15()     );
      SEND  ( mph_neighbourcell_req_13()       );
      AWAIT ( mphc_rxlev_periodic_req_16()     );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR901-------------------------------
 *
 * Description:	The mobile leaves the dedicated mode to the current
 *              serving cell 23. A cell reselection is performed and the idle
 *              mode is configured.
 *-------------------------------------------------------------------------------*/

T_CASE ALR901()
{
  BEGIN_CASE ("cell_reselection_after_dedicated_mode_alr901")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();
      SEND  ( mph_stop_dedicated_req()     );
      AWAIT ( mphc_stop_dedicated_req()    );
      SEND  ( mphc_stop_dedicated_con()    );
      AWAIT ( mph_stop_dedicated_cnf()     );
      SEND  ( mph_idle_req_17()            );
      AWAIT ( mphc_new_scell_req_1()       );
      SEND  ( mphc_new_scell_con_1()       );
      AWAIT ( mphc_start_ccch_req_5()      );
      AWAIT ( mphc_scell_nbcch_req_full()  );
      SEND  ( mphc_data_ind_1()            );
      AWAIT ( mph_unitdata_ind_si1(ARFCN_23)        );
      SEND  ( mphc_data_ind_2()            );
      AWAIT ( mph_unitdata_ind_si2(ARFCN_23)        );
      SEND  ( mph_idle_req_10()            );
      AWAIT ( mphc_stop_scell_bcch_req()   );
      AWAIT ( mphc_start_ccch_req_10()     );
      AWAIT ( mphc_scell_nbcch_req_full()  );
      AWAIT ( mphc_rxlev_periodic_req_15() );
      SEND  ( mph_neighbourcell_req_13()   );
      AWAIT ( mphc_rxlev_periodic_req_17() );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR902-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the neighbour
 *              cells 1, 14 and 124. The fieldstrength is 56 for channel 23, 12
 *              for channel 1, 44 for channel 14 and 25 for channel 124
 *              (all values in GSM range). The ranking for the neighbour cells
 *              is 14, 124 and channel 1. Each report contains two fieldstrength
 *              values per channel. The multiframe period is set to 6.
 *              The first measurement report is send to RR after five reports
 *              from PL. Then after each three reports from PL a measurement
 *              report is send to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR902()
{
  BEGIN_CASE ("synchronisation_to_neighbour_cells_successful_sys_info_4__alr902")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_29()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_ncell_sync_ind_3()       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_30()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_bcch_ind_si4(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)  );
      SEND  ( mphc_ncell_bcch_ind_si4(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)  );
      SEND  ( mphc_ncell_bcch_ind_si4(ARFCN_124)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)  );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_14()      );
      AWAIT ( mph_unitdata_ind_si4(ARFCN_1)         );
      AWAIT ( mph_unitdata_ind_si4(ARFCN_14)         );
      AWAIT ( mph_unitdata_ind_si4(ARFCN_124)         );
      wait_ncsync_idle_2_14(5);
      AWAIT ( mphc_ncell_list_sync_req_1a() );
      AWAIT ( mph_measurement_ind_14()      );
      wait_ncsync_idle_2_14(2);
      AWAIT ( mph_measurement_ind_14()      );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR903-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the neighbour
 *              cells 1, 14 and 124. The fieldstrength is 56 for channel
 *              23, 12 for channel 1, 44 for channel 14 and 25 for channel 124
 *              (all values in GSM range). The ranking for the neighbour cells is 14,
 *              124 and channel 1. Each report contains two fieldstrength
 *              values per channel. The multiframe period is set to 6.
 *              The first measurement report is send to RR after five reports
 *              from PL. Then after each three reports from PL a measurement
 *              report is send to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR903()
{
  BEGIN_CASE ("synchronisation_to_neighbour_cells_successful_sys_info_4_and_7__alr903")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_29()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_ncell_sync_ind_3()       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_30()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_bcch_ind_si4_acs(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)  );
      AWAIT ( mphc_ncell_bcch_req_22()      );
      SEND  ( mphc_ncell_bcch_ind_si4_acs(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)  );
      AWAIT ( mphc_ncell_bcch_req_23()      );
      SEND  ( mphc_ncell_bcch_ind_si4_acs(ARFCN_124)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)  );
      AWAIT ( mphc_ncell_bcch_req_24()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_bcch_ind_si7(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)  );
      SEND  ( mphc_ncell_bcch_ind_si7(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)  );
      SEND  ( mphc_ncell_bcch_ind_si7(ARFCN_124)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)  );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_14()      );
      AWAIT ( mph_unitdata_ind_si4_diff_cs(ARFCN_1)         );
      AWAIT ( mph_unitdata_ind_si7(ARFCN_1)         );
      AWAIT ( mph_unitdata_ind_si4_diff_cs(ARFCN_14)         );
      AWAIT ( mph_unitdata_ind_si7(ARFCN_14)         );
      AWAIT ( mph_unitdata_ind_si4_diff_cs(ARFCN_124)         );
      AWAIT ( mph_unitdata_ind_si7(ARFCN_124)         );
      wait_ncsync_idle_2_14(5);
      AWAIT ( mphc_ncell_list_sync_req_1a() );
      AWAIT ( mph_measurement_ind_14()      );
      wait_ncsync_idle_2_14(2);
      AWAIT ( mph_measurement_ind_14()      );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR904-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the neighbour
 *              cells 1, 14 and 124. The fieldstrength is 56 for
 *              channel 23, 12 for channel 1, 44 for channel 14 and 25 for
 *              channel 124 (all values in GSM range). The ranking for the
 *              neighbour cells is 14, 124 and channel 1. Each report contains
 *              two fieldstrength values per channel. The multiframe period is
 *              set to 6. The first measurement report is send to RR after five
 *              reports from PL. Then after each three reports from PL a measurement
 *              report is send to RR.
 *------------------------------------------------------------------------------*/

T_CASE ALR904()
{
  BEGIN_CASE ("synchronisation_to_neighbour_cells_successful_sys_info_4_and_8__alr904")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_29()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_ncell_sync_ind_3()       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_30()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_bcch_ind_si4_acs(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)  );
      AWAIT ( mphc_ncell_bcch_req_22()      );
      SEND  ( mphc_ncell_bcch_ind_si4_acs(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)  );
      AWAIT ( mphc_ncell_bcch_req_23()      );
      SEND  ( mphc_ncell_bcch_ind_si4_acs(ARFCN_124)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)  );
      AWAIT ( mphc_ncell_bcch_req_24()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_30()      );
      SEND  ( mphc_ncell_bcch_ind_si8(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)  );
      SEND  ( mphc_ncell_bcch_ind_si8(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)  );
      SEND  ( mphc_ncell_bcch_ind_si8(ARFCN_124)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)  );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_14()      );
      AWAIT ( mph_unitdata_ind_si4_diff_cs(ARFCN_1)         );
      AWAIT ( mph_unitdata_ind_si8(ARFCN_1)         );
      AWAIT ( mph_unitdata_ind_si4_diff_cs(ARFCN_14)         );
      AWAIT ( mph_unitdata_ind_si8(ARFCN_14)         );
      AWAIT ( mph_unitdata_ind_si4_diff_cs(ARFCN_124)         );
      AWAIT ( mph_unitdata_ind_si8(ARFCN_124)         );
      wait_ncsync_idle_2_14(4);
      AWAIT ( mphc_ncell_list_sync_req_1a() );

      AWAIT ( mph_measurement_ind_14()      );
      wait_ncsync_idle_2_14(3);
      AWAIT ( mph_measurement_ind_14()      );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR905-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the neighbour
 *              cells 1, 14 and 124. The fieldstrength is 56 for channel 23,
 *              12 for channel 1, 44 for channel 14 and 25 for channel 124
 *              (all values in GSM range). The ranking for the neighbour cells is 14,
 *              124 and channel 1. Each report contains two fieldstrength
 *              values per channel. The multiframe period is set to 6.
 *              The first measurement report is send to RR after five reports
 *              from PL. Then after each three reports from PL a measurement
 *              report is send to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR905()
{
  BEGIN_CASE ("synchronisation_to_neighbour_cells_successful_unexpected_msg_and_sys_info_3__alr905")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_29()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_ncell_sync_ind_3()       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_30()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_bcch_ind_si2(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)  );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)       );
      SEND  ( mphc_ncell_bcch_ind_si7(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)  );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1)       );
      SEND  ( mphc_ncell_bcch_ind_si8(ARFCN_124)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)  );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_30()      );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)       );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)  );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1)       );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)  );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124)       );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)  );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_14()      );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1)          );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)          );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_124)          );
      wait_ncsync_idle_2_14(4);
      AWAIT ( mphc_ncell_list_sync_req_1a() );
      AWAIT ( mph_measurement_ind_14()      );
      wait_ncsync_idle_2_14(3);
      AWAIT ( mph_measurement_ind_14()      );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR906-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the neighbour
 *              cells 1, 14 and 124. The fieldstrength is 56 for channel 23,
 *              12 for channel 1, 44 for channel 14 and 25 for channel 124
 *              (all values in GSM range). The ranking for the neighbour cells is 14,
 *              124 and channel 1. Each report contains two fieldstrength values
 *              per channel. The multiframe period is set to 6. The first measurement
 *              report is send to RR after five reports from PL. Then after each three
 *              reports from PL a measurement report is send to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR906()
{
  BEGIN_CASE ("synchronisation_to_neighbour_cells_successful_read_error_and_sys_info_3__alr906")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_29()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_ncell_sync_ind_3()       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1)       );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_30()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_bcch_ind_invalid(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)  );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)       );
      SEND  ( mphc_ncell_bcch_ind_invalid(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)  );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1)       );
      SEND  ( mphc_ncell_bcch_ind_invalid(ARFCN_124)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)  );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_30()      );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)       );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)  );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1)       );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)  );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124)       );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)  );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_14()      );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1)          );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)          );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_124)          );
      wait_ncsync_idle_2_14(4);
      AWAIT ( mphc_ncell_list_sync_req_1a() );
      AWAIT ( mph_measurement_ind_14()      );
      wait_ncsync_idle_2_14(3);
      AWAIT ( mph_measurement_ind_14()      );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR907-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the neighbour cells 1,
 *              14 and 124. The fieldstrength is 56 for channel 23, 12 for channel 1,
 *              44 for channel 14 and 25 for channel 124 (all values in GSM range).
 *              The ranking for the neighbour cells is 14, 124 and channel 1.
 *              Each reports contains two fieldstrength values per channel.
 *              The multiframe period is set to 6. The first measurement report
 *              is send to RR after five reports from PL. Then after each three
 *              reports from PL a measurement report is send to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR907()
{
  BEGIN_CASE ("synchronisation_to_neighbour_cells_successful_sys_info_3__alr907")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14));
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124));
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1));
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_29()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind (ARFCN_14) );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14) );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1));
      AWAIT ( mphc_ncell_bcch_req (ARFCN_124, FN_OFF_ARFCN_124, TIME_ALIGNMT_124));
      SEND  ( mphc_ncell_sync_ind_3()       );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_1, FN_OFF_ARFCN_1, TIME_ALIGNMT_1));
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_30()      );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14));
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1));
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124));
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      SEND  ( mphc_rxlev_periodic_ind_2()   );
      AWAIT ( mph_measurement_ind_14()      );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1) );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14) );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_124));
      wait_ncsync_idle_2_14(5);
      AWAIT ( mphc_ncell_list_sync_req_1a() );
      AWAIT ( mph_measurement_ind_14()      );
      wait_ncsync_idle_2_14(2);
      AWAIT ( mph_measurement_ind_14()      );
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR908-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the neighbour
 *              cells 1, 14 and 124. The MS is in Limited service on 23.
 *              Cell 14 belongs to the requested PLMN. Parallel search is started.
 *              The Rxlev of 14 is zero before starting parallel search(TREG expiry).
 *              However, the Rxlev of 14 goes up just before the start of parallel
 *              search and is detected and reported by ALR in MPH_POWER_CNF to RR.
 *              MS reaches full service on cell 14.
 *-------------------------------------------------------------------------------*/

T_CASE ALR908()
{
  BEGIN_CASE ("parallel_search_in_limited_service__alr908")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('A');

      START_TIMEOUT(29500);

      // cell 14 is not active yet. NC status of cell 14 in
      // nc_data wil be IDLE after processing this primitive
      SEND  ( mphc_rxlev_periodic_ind_9(30,0,0,0) );

      // cell 14 is activated
      // Start parallel search
      SEND  ( mph_power_req_2()           );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_1) );
      AWAIT ( mph_power_cnf_1()           );

      // synchronize on cell 14
      SEND  ( mph_bsic_req(ARFCN_14)      );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)       );
      AWAIT ( mphc_start_ccch_req_14()                    );
      AWAIT ( mphc_scell_nbcch_req_periodic()             );

      //cell 14 BSIC is provided here. This is tored in nc_data also.
      SEND  ( mphc_ncell_sync_ind (ARFCN_14)              );
      AWAIT ( mph_bsic_cnf_14()                           );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_14)              );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)           );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14, CELL_IDENT_1));
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)          );
      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH)           );
      AWAIT ( mphc_start_ccch_req_14()                    );
      AWAIT ( mphc_scell_nbcch_req_full() );
      SEND  ( mph_idle_req_15());
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mphc_stop_ccch_req());
      AWAIT ( mphc_stop_rxlev_periodic_req());
      AWAIT ( mphc_stop_scell_bcch_req());

      // CEL_R on 14. BSIC stored in nc_data is used here
      AWAIT ( mphc_new_scell_req_10());

    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR925-------------------------------
 *
 * Description:	The mobile leaves the dedicated mode after some moments to
 *              the current serving cell 23. A cell selection is performed
 *              and the idle mode is configured.
 *-------------------------------------------------------------------------------*/

T_CASE ALR925()
{
  BEGIN_CASE ("cell_selection_after_dedicated_mode_alr925")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      initiation_with_paging_reorganisation_alr020('B');

      SEND  ( mphc_data_ind_11()               );
      AWAIT ( mphc_start_ccch_req_6()          );
      AWAIT ( mphc_scell_nbcch_req_periodic()  );
      SEND  ( mphc_data_ind_9()                );
      AWAIT ( mphc_start_ccch_req_4()          );
      AWAIT ( mphc_scell_nbcch_req_periodic()  );
      SEND  ( mph_random_access_req_1()        );
      AWAIT ( mphc_stop_rxlev_periodic_req()   );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_start_ccch_req_3('B')       );
      AWAIT ( mphc_scell_nbcch_req_full()      );
      AWAIT ( mphc_ra_req_3()                  );
      SEND  ( mphc_ra_con_1()                  );
      AWAIT ( mph_random_access_cnf_1()        );
      AWAIT ( mphc_ra_req_4()                  );
      SEND  ( mphc_data_ind_20()               );
      AWAIT ( mph_unitdata_ind_10()            );
      SEND  ( mph_dedicated_req_1()            );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_stop_ccch_req()             );
      AWAIT ( mphc_stop_ra_req_1()             );
      AWAIT ( mphc_immed_assign_req_1()        );
      SEND  ( mphc_immed_assign_con_1()        );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK) );
      SEND  ( mph_stop_dedicated_req()         );
      AWAIT ( mphc_stop_dedicated_req()        );
      SEND  ( mphc_stop_dedicated_con()        );
      AWAIT ( mph_stop_dedicated_cnf()         );
      SEND  ( mph_idle_req_2()                 );
      AWAIT ( mphc_start_ccch_req_5()          );
      AWAIT ( mphc_scell_nbcch_req_full()      );
      AWAIT ( mphc_rxlev_periodic_req_18()     );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR952-------------------------------
 *
 * Description:	After receiving the power levels of the neighbour cells
 *              (MPHC-RXLEV-PERIODIC indication primitive), synchronization of
 *              neighbour cells begins (MPHC-NCELL-SYNC request/indication primitives).
 *              After 30 seconds (third expiry of TIM_NCSYNC timer), synchronization
 *              is halted and a list of neighbour cells for synchronization is sent
 *              to Layer 1 (MPHC-NCELL-LIST-SYNC request primitive), following which
 *              synchronization is recommenced.
 *
 *              After receiving the power levels of the neighbour cells (MPHC-RXLEV-PERIODIC
 *              indication primitive), synchronization of neighbour cells begins
 *              (MPHC-NCELL-SYNC request/indication primitives). After 30 seconds
 *              (third expiry of TIM_NCSYNC timer), synchronization is halted and a
 *              list of neighbour cells for synchronization is sent to Layer 1
 *              (MPHC-NCELL-LIST-SYNC request primitive), following which synchronization is
 *              recommenced.
 *              Cell reselection is then requested by RR (MPH-IDLE request primitive),
 *              nominating a cell which has been reassigned to a different BTS.
 *              ALR deselects the current serving cell (MPHC-STOP-SCELL_BCCH request
 *              primitive and selects the new serving cell (MPHC-NEW-SCELL request primitive).
 *-------------------------------------------------------------------------------*/

T_CASE ALR952()
{
  BEGIN_CASE ("power_measurement_with_change_of_bs_e_otd_off_idle_mode__alr952")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      SEND  ( mphc_rxlev_periodic_ind_2()           );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14) );
      SEND  ( mphc_ncell_sync_ind_24()              );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124));
      SEND  ( mphc_ncell_sync_ind_25()              );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)  );
      SEND  ( mphc_ncell_sync_ind_26()              );
      AWAIT ( mphc_ncell_bcch_req_25()              );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)     );
      AWAIT ( mphc_ncell_bcch_req_26()              );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124)    );
      AWAIT ( mphc_ncell_bcch_req_27()              );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)    );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)   );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)     );
      SEND  ( mphc_rxlev_periodic_ind_2()           );
      AWAIT ( mph_measurement_ind_31()              );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1)         );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_124)       );
      wait_ncsync_idle_2_31(8);
      AWAIT ( mphc_ncell_list_sync_req_1()          );
      AWAIT ( mph_measurement_ind_31()              );
      SEND  ( mphc_ncell_sync_ind_24()              );
      SEND  ( mphc_ncell_sync_ind_25()              );
      SEND  ( mphc_ncell_sync_ind_27()              );
      AWAIT ( mphc_ncell_bcch_req_28()              );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)     );
      SEND  ( mphc_rxlev_periodic_ind_2()           );
      SEND  ( mphc_rxlev_periodic_ind_2()           );
      AWAIT ( mph_measurement_ind_31()              );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1)         );
      wait_ncsync_idle_2_31(8);
      AWAIT ( mphc_ncell_list_sync_req_1()          );
      AWAIT ( mph_measurement_ind_31()              );
      SEND  ( mph_idle_req_20()                     );
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mphc_stop_ccch_req()                  );
      AWAIT ( mphc_stop_rxlev_periodic_req()        );
      AWAIT ( mphc_stop_ncell_sync_req_all()        );
      AWAIT ( mphc_stop_scell_bcch_req()            );
      AWAIT ( mphc_new_scell_req_12()               );
      SEND  ( mphc_new_scell_con_1()                );
      AWAIT ( mphc_start_ccch_req_14()              );
      AWAIT ( mphc_scell_nbcch_req_full()           );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR952A-------------------------------
 *
 * Description: After receiving the power levels of the neighbour cells
 *              (MPHC-RXLEV-PERIODIC indication primitive), synchronization of
 *              neighbour cells begins (MPHC-NCELL-SYNC request/indication primitives).
 *              After 30 seconds (third expiry of TIM_NCSYNC timer), synchronization
 *              is halted and a list of neighbour cells for synchronization is sent
 *              to Layer 1 (MPHC-NCELL-LIST-SYNC request primitive).
 *              Synchronization lost is indicated for all neighbouring cells.
 *
 *              Cell reselection is then requested by RR (MPH-IDLE request primitive),
 *-------------------------------------------------------------------------------*/

T_CASE ALR952A()
{
  BEGIN_CASE ("power_measurement_with_loss_of_ncell_sync_idle_mode__alr952a")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      START_TIMEOUT(29500);
      SEND  ( mphc_rxlev_periodic_ind_2()           );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14) );
      SEND  ( mphc_ncell_sync_ind_24()              );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124));
      SEND  ( mphc_ncell_sync_ind_25()              );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)  );
      SEND  ( mphc_ncell_sync_ind_26()              );
      AWAIT ( mphc_ncell_bcch_req_25()              );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)     );
      AWAIT ( mphc_ncell_bcch_req_26()              );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_124)    );
      AWAIT ( mphc_ncell_bcch_req_27()              );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_1)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)    );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_124)   );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_1)     );
      SEND  ( mphc_rxlev_periodic_ind_2()           );
      AWAIT ( mph_measurement_ind_31()              );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_1)         );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_124)       );
      WAIT_TIMEOUT();
      AWAIT ( mphc_ncell_list_sync_req_1()          );
      SEND  ( mphc_ncell_sync_ind_failed(ARFCN_14)       );
      SEND  ( mphc_ncell_sync_ind_failed(ARFCN_124)       );
      SEND  ( mphc_ncell_sync_ind_failed(ARFCN_1)       );
      SEND  ( mphc_data_ind_19()              );
      SEND  ( mphc_data_ind_19()              );
      SEND  ( mphc_data_ind_19()              );
      AWAIT ( mph_error_ind_2()               );
      SEND  ( mph_idle_req_15()               );
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT  ( mphc_stop_ccch_req());
      AWAIT ( mphc_stop_rxlev_periodic_req());
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mph_error_ind_5()               );
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR954-------------------------------
 *
 * Description:	RR selects Channel 23 as serving cell after reading the BCCH carrier.
 *              After receiving the power levels of the neighbour cells
 *              (MPHC-RXLEV-PERIODIC indication primitive), synchronization of the
 *              six strongest neighbour cells begins (MPHC-NCELL-SYNC request/indication
 *              primitives). After 30 seconds (third expiry of TIM_NCSYNC timer),
 *              synchronization is halted and a list of neighbour cells for synchronization
 *              is sent to Layer 1 (MPHC-NCELL-LIST-SYNC request primitive),
 *              following which synchronization is recommenced.
 *-------------------------------------------------------------------------------*/

T_CASE ALR954()
{
  BEGIN_CASE ("power_measurement_on_8_neighbour_cells_e_otd_off_idle_mode__alr954")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();
      SEND  ( mph_classmark_req_3()                  );
      SEND  ( mph_idle_req_21()                      );
      AWAIT ( mphc_stop_scell_bcch_req()             );
      AWAIT ( mphc_start_ccch_req_9()                );
      AWAIT ( mphc_scell_nbcch_req_full()            );
      SEND  ( mph_neighbourcell_req_15()             );
      AWAIT ( mphc_rxlev_periodic_req_20()           );

      SEND  ( mphc_rxlev_periodic_ind_3());
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_637) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_25)  );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)  );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_512) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_580) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_885) );
      SEND  ( mphc_rxlev_periodic_ind_3()            );
      AWAIT ( mph_measurement_ind_32()               );
      SEND  ( mphc_ncell_sync_ind_28()               );
      AWAIT ( mphc_ncell_bcch_req (ARFCN_637)               );
      SEND  ( mphc_ncell_sync_ind_29()               );
      AWAIT ( mphc_ncell_bcch_req_30()               );
      SEND  ( mphc_ncell_sync_ind_24()               );
      AWAIT ( mphc_ncell_bcch_req_25()               );
      SEND  ( mphc_ncell_sync_ind_30()               );
      AWAIT ( mphc_ncell_bcch_req_31()               );
      SEND  ( mphc_ncell_sync_ind_31()               );
      AWAIT ( mphc_ncell_bcch_req_32()               );
      SEND  ( mphc_ncell_sync_ind_32()               );
      AWAIT ( mphc_ncell_bcch_req_33()               );
      SEND  ( mphc_rxlev_periodic_ind_3()            );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_637)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_637)    );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_25)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_25)     );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_14)      );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_14)     );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_512)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_512)    );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_580)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_580)    );
      SEND  ( mphc_ncell_bcch_ind_si3(ARFCN_885)     );
      AWAIT ( mphc_stop_ncell_bcch_req(ARFCN_885)    );
      SEND  ( mphc_rxlev_periodic_ind_3()            );
      SEND  ( mphc_rxlev_periodic_ind_3()            );
      AWAIT ( mph_measurement_ind_33()               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)         );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_25)         );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_512)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_580)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_637)        );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_885)        );
      wait_ncsync_idle_3_33(6);

      AWAIT ( mphc_ncell_list_sync_req_2()           );
      AWAIT ( mph_measurement_ind_33()               );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR962-------------------------------
 *
 * Description:	Completion of the E-OTD cell monitoring procedure is signalled
 *              by the receipt of a second MPHC-NCELL-SYNC indication primitive
 *              for the serving cell, whereupon the measurement
 *              results are forwarded to RR (MPH-NCELL-POS indication primitive).
 *-------------------------------------------------------------------------------*/

T_CASE ALR962()
{
  BEGIN_CASE ("power_measurement_e_otd_on_idle_mode__alr962")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();
      power_measurement_e_otd_on_idle_mode__alr960();

      SEND  ( mphc_ncell_sync_ind_40() );
      SEND  ( mphc_ncell_sync_ind_41() );
      SEND  ( mphc_ncell_sync_ind_42() );
      SEND  ( mphc_ncell_sync_ind_43() );
      SEND  ( mphc_ncell_sync_ind_44() );
      SEND  ( mphc_ncell_sync_ind_45() );
      SEND  ( mphc_ncell_sync_ind_46() );
      SEND  ( mphc_ncell_sync_ind_47() );
      AWAIT ( mph_ncell_pos_ind_1()    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR964-------------------------------
 *
 * Description:	ALR requests Layer 1 to stop synchronization of neighbour cells
 *              and starts a 10-second timer. As Layer 1 fails to respond the
 *              request is repeated on each expiry of the 10-second TIM_NCSYNC timer.
 *
 * XXX This testcase expects different behaviour than is implemented in the code.
 * But it is not clear what is correct.
 *-------------------------------------------------------------------------------*/
/*
  T_CASE ALR964()
  {
  BEGIN_CASE ("power_measurement_e_otd_on_no_response_from_layer_1_idle_mode__alr964")
  {
  initiation_by_rr_8_channels_available_alr601();
  find_bcch_carrier_first_channel_alr603();
  read_bcch_data_alr607();
  power_measurement_e_otd_on_idle_mode__alr960();
  AWAIT ( mphc_stop_ncell_sync_req_all());
  AWAIT ( mphc_ncell_list_sync_req_3());
  AWAIT ( mphc_stop_ncell_sync_req_all());
  AWAIT ( mphc_ncell_list_sync_req_3());
  AWAIT ( mphc_stop_ncell_sync_req_all());
  AWAIT ( mphc_ncell_list_sync_req_3());
  AWAIT ( mphc_stop_ncell_sync_req_all());
  AWAIT ( mphc_ncell_list_sync_req_3());
  }
  }
*/

/*------------------------Following is for ALR968-------------------------------
 *
 * Description:	RR request cell reselection (MPH-IDLE request primitive) following
 *              commencement of E-OTD measurement. E-OTD measurement is halted
 *              (MPHC-STOP-NCELL-SYNC request primitive), the current serving cell
 *              is deselected (MPHC-STOP-SCELL-BCCH request primitive) and ALR
 *              requests the new serving cell (MPHC-NEW-SCELL request primitive).
 *-------------------------------------------------------------------------------*/

T_CASE ALR968()
{
  BEGIN_CASE ("cell_reselection_started_during_e_otd_measurement_idle_mode__alr968")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();
      power_measurement_e_otd_on_idle_mode__alr960();

      SEND  ( mph_idle_req_23()                );
      AWAIT ( mphc_stop_scell_bcch_req());
      AWAIT ( mphc_stop_ccch_req()             );
      AWAIT ( mphc_stop_rxlev_periodic_req() );
      AWAIT ( mphc_stop_ncell_sync_req_all()     );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_new_scell_req_13()          );
      SEND  ( mphc_new_scell_con_1()           );
      AWAIT ( mphc_start_ccch_req_14()         );
      AWAIT ( mphc_scell_nbcch_req_full()         );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR974-------------------------------
 *
 * Description: Following commencement of handover (receipt of MPH-DEDICATED
 *              request primitive), ALR immediately requests Layer 1 to cease
 *              E-OTD measurement (MPHC-STOP-NCELL-SYNC request primitive)
 *              and proceeds with the handover.
 *----------------------------------------------------------------------------*/

T_CASE ALR974()
{
  BEGIN_CASE ("e_otd_measurement_halted_due_to_commencement_of_handover_alr974")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607(); // ARFCN=23
      power_measurement_e_otd_on_idle_mode__alr960();
      connection_established_e_otd_dedicated_mode__alr970();
      e_otd_measurement_started_dedicated_mode__alr972('A');

      SEND  ( mph_dedicated_req_2()); // -> ARFCN=14
      AWAIT ( mphc_stop_ncell_sync_req_all());
      AWAIT ( mphc_async_ho_req_1(HOT_BLIND));
      SEND  ( mphc_async_ho_con_1());

      SEND  ( mphc_handover_finished_1(HO_COMPLETE) );
      /*
       * This does not seem to be 100% correct, but it is just one sync
       * attempt too much so it should not hurt
       */
      AWAIT ( mphc_ncell_list_sync_req_4('C') );
      AWAIT ( mph_dedicated_cnf_1(DEDI_RES_OK)      );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR976-------------------------------
 *
 * Description: On confirmation from Layer 1 that sychronization of neighbour
 *              cells has ceased (reception of MPHC-STOP-NCELL-SYNC confirmation
 *              primitive). ALR forwards a list of the neighbour cells selected
 *              for E-OTD measurement to Layer 1 (MPHC-NCELL-LIST-SYNC request
 *              primitive). Layer 1 subsequently forwards the measurements for
 *              the serving cell and the neighbour cells requested
 *              (MPHC-NCELL-SYNC indication primitive). The process is complete
 *              when a second measurement for the serving cell is received,
 *              whereupon ALR forwards the measurement results to RR
 *              (MPH-NCELL-POS indication primitive).
 *----------------------------------------------------------------------------*/

T_CASE ALR976()
{
  BEGIN_CASE ("e_otd_measurement_performed_dedicated_mode__alr976")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();
      power_measurement_e_otd_on_idle_mode__alr960();
      connection_established_e_otd_dedicated_mode__alr970();
      e_otd_measurement_started_dedicated_mode__alr972('A');

      SEND  ( mphc_ncell_sync_ind_40() );
      SEND  ( mphc_ncell_sync_ind_41() );
      SEND  ( mphc_ncell_sync_ind_42() );
      SEND  ( mphc_ncell_sync_ind_43() );
      SEND  ( mphc_ncell_sync_ind_44() );
      SEND  ( mphc_ncell_sync_ind_45() );
      SEND  ( mphc_ncell_sync_ind_46() );
      SEND  ( mphc_ncell_sync_ind_47() );
      AWAIT ( mph_ncell_pos_ind_3()    );
    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR978-------------------------------
 *
 * Description:	On confirmation from Layer 1 that sychronization of neighbour cells
 *              has ceased (reception of MPHC-STOP-NCELL-SYNC confirmation primitive)
 *              ALR forwards a list of the neighbour cells selected for E-OTD measurement
 *              to Layer 1 (MPHC-NCELL-LIST-SYNC request primitive). Layer 1 subsequently
 *              forwards the measurements for the serving cell and the neighbour cells
 *              requested (MPHC-NCELL-SYNC indication primitive). The process is complete
 *              when a second measurement for the serving cell is received, whereupon ALR
 *              forwards the measurement results to RR (MPH-NCELL-POS indication primitive).
 *-------------------------------------------------------------------------------*/

T_CASE ALR978()
{
  BEGIN_CASE ("e_otd_measurement_performed_unknown_arfcns_dedicated_mode__alr978")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr601();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr607();
      power_measurement_e_otd_on_idle_mode__alr960();
      connection_established_e_otd_dedicated_mode__alr970();
      e_otd_measurement_started_dedicated_mode__alr972('B');

      SEND  ( mphc_ncell_sync_ind_40() );
      SEND  ( mphc_ncell_sync_ind_41() );
      SEND  ( mphc_ncell_sync_ind_42() );
      SEND  ( mphc_ncell_sync_ind_43() );
      SEND  ( mphc_ncell_sync_ind_44() );
      SEND  ( mphc_ncell_sync_ind_45() );
      SEND  ( mphc_ncell_sync_ind_46() );
      SEND  ( mphc_ncell_sync_ind_48() );
      SEND  ( mphc_ncell_sync_ind_49() );
      SEND  ( mphc_ncell_sync_ind_47() );
      /* This finds a bug in the code!! */
      AWAIT ( mph_ncell_pos_ind_4()    );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALRG007-------------------------------
 *
 * Description:	ALR needs to check to see if it is in PIM before it can configure
 *              Layer 1 for CBCH. Any other state and ALR must copy the new
 *              CBCH settings before discarding the request.
 *--------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR1007()
{
  BEGIN_CASE ("cbch_in_PIM_only_alrg007")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();

      rr_select_first_channel_gprs_pbcch(VAR_NMO_1);

      SEND  ( mph_cbch_req_2('A')                 );
      SEND  ( mph_mon_ctrl_req(ENTER_PIM_PBCCH)   );
      SEND  ( mmi_cbch_req_1()                    );
      AWAIT ( mphc_config_cbch_req_1('A')         );
      AWAIT ( mphc_cbch_schedule_req_1()          );
      AWAIT ( mphc_cbch_schedule_req_2()          );

      SEND  ( mph_mon_ctrl_req(LEAVING_PIM_PBCCH) );
      SEND  ( mmi_cbch_req_1()                    );
      AWAIT ( mphc_stop_cbch_req()                );
      MUTE(1000);

      SEND  ( mph_mon_ctrl_req(ENTER_PTM_PBCCH)   );
      SEND  ( mmi_cbch_req_1()                    );
      MUTE(1000);

      SEND  ( mph_mon_ctrl_req(LEAVING_PTM_PBCCH) );
      SEND  ( mmi_cbch_req_1()                    );
      MUTE(1000);

      SEND  ( mph_mon_ctrl_req(LEAVING_PAM_PBCCH) );
      SEND  ( mmi_cbch_req_1()                    );
      MUTE(1000);
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*------------------------Following is for ALRG008-------------------------------
 *
 * Description:	In dedicated mode ALR is suspended during a PDCH Assignment procedure.
 *              After a successful TBF establishment, ALR is requested to enter a mode
 *              which is compatible to Packet Transfer Mode.
 *--------------------------------------------------------------------------------*/
#if defined GPRS
T_CASE ALR1008()
{
  BEGIN_CASE ("pdch_assignment_success_start_tbf_establishment_alrg008")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013_gprs();

      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();

      pdch_assignment_alrg007();
      SEND ( mph_idle_req_2g() );
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*------------------------Following is for ALRG009-------------------------------
 *
 * Description:	After an unsuccessful TBF establishment, ALR is requested to switch
 *              back to the 'old' channel.
 *---------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR1009()
{
  BEGIN_CASE ("pdch_assignment_error_during_tbf_establishment_alrg009")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013_gprs();

      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();

      pdch_assignment_alrg007();

      SEND  ( mph_dedicated_fail_req_1g()  );
      AWAIT ( mphc_immed_assign_req_1()    );
      SEND  ( mphc_immed_assign_con_1()    );
      AWAIT ( mphc_channel_assign_req_1g() );
      SEND  ( mphc_channel_assign_con_1g() );
      AWAIT ( mph_dedicated_fail_cnf_1g()  );
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*------------------------Following is for ALRG010-------------------------------
 *
 * Description:	In dedicated mode ALR is requested to synchronize to a new cell.
 *              The new cell ARFCN and BSIC is provided by the network.
 *-------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR1010()
{
  BEGIN_CASE ("network_controlled_change_cell_alrg010")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013_gprs();

      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();

      SEND  ( mph_stop_dedicated_req()      );
      AWAIT ( mphc_stop_dedicated_req()     );
      SEND  ( mphc_stop_dedicated_con()     );
      AWAIT ( mph_stop_dedicated_cnf()      );

      SEND  ( mph_bsic_req(ARFCN_14)        );
      /* Measurements are stopped before tuning to new cell*/
      AWAIT ( mphc_stop_rxlev_periodic_req() );

      //AWAIT ( mphc_stop_dedicated_1()          );

      AWAIT ( mphc_network_sync_req(ARFCN_14, SM_NARROW_MODE));
      SEND  ( mphc_network_sync_ind_6()        );
      AWAIT ( mph_bsic_cnf_6()                );
      AWAIT ( mphc_new_scell_req_2()           );
      SEND  ( mphc_new_scell_con_1()           );
      AWAIT ( mphc_scell_nbcch_req_full()      );
      SEND  ( mphc_data_ind_49()               );
      AWAIT ( mph_unitdata_ind_si1(ARFCN_14)   );
      SEND  ( mphc_data_ind_65()               );
      AWAIT ( mph_unitdata_ind_si2(ARFCN_14)   );
      SEND  ( mph_idle_req_1('A')              );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_start_ccch_req_1('A')       );
      AWAIT ( mphc_scell_nbcch_req_full()      );
      SEND  ( mph_neighbourcell_req_2()        );
      AWAIT ( mphc_rxlev_periodic_req_2i()      );
    } HANDLE_DEACT;
  }
}
#endif //GPRS
/*------------------------Following is for ALRG011-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 *              This cell has PBCCH. In this test the ncell handling for PBCCH
 *              is tested.
 *
 *---------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR1011()
{
  BEGIN_CASE ("GPRS PBCCH, ncell synchronization")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();

      rr_select_first_channel_gprs_pbcch(VAR_NMO_3);

      nc_pbcch_initial_sync();
      /*This message mph_neighbourcell_req_sync_reconfirm is sent when GRR 
      inform RR with RRGRR_NCELL_SYNC_REQ(SYNC_RECONFIRM)
      GRR maintain the 10 sec timer through measurement counter*/
      SEND(mph_neighbourcell_req_sync_reconfirm());


      TIMEOUT (10000);

      AWAIT ( mphc_ncell_list_sync_req_1g() );
      SEND  ( mph_mon_ctrl_req(LEAVING_PIM_PBCCH)          );
      AWAIT ( mphc_stop_ccch_req()        );
      AWAIT ( mphc_stop_ncell_sync_req_all()  );
      SEND  ( mph_mon_ctrl_req(LEAVING_PAM_PBCCH)          );
      SEND  ( mph_mon_ctrl_req(ENTER_PTM_PBCCH)          );
      AWAIT ( mphc_ncell_list_sync_req_1g() );
    } HANDLE_DEACT;
  }
}
#endif //GPRS


/*------------------------Following is for ALRG012-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 *              The cell has GPRS. ALR has to listen to two PTMSIs after
 *              the IDENTITY-REQ.
 *
 *---------------------------------------------------------------------------------*/

#if defined GPRS
T_CASE ALR1012()
{
  BEGIN_CASE ("GPRS BCCH, 2x PTMSI check")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();

      rr_select_first_channel_alr013_gprs();

      SEND  ( mph_identity_req_2g() );
      SEND  ( mphc_data_ind_8g()    );
      AWAIT ( mph_paging_ind_1g()   );
      AWAIT ( mphc_start_ccch_req_8 () );
      SEND  ( mphc_data_ind_9g()    );
      SEND  ( mphc_data_ind_10g()   );
      AWAIT ( mph_paging_ind_1g()   );
    } HANDLE_DEACT;
  }
}


#endif //GPRS

/*------------------------Following is for ALR1013A-------------------------------
 *
 * Description: A Async handover is performed. Before the handover there was a large
 *              BA list of 20 entries in use. Handover is requested for the carrier for which
 *              the sync has already been decoded. ALR shall send Handover type as HOT_REGULAR
 *-------------------------------------------------------------------------------*/
T_CASE ALR1013A()
{
  BEGIN_CASE ("handover_full_ba_list_alr1013A")
  {
    TRAP_FOR_DEACT {
      initial_setting_state_to_dedicated();
      SEND(mph_dedicated_req_fn(MODE_ASYNC_HANDOVER, ARFCN_30, BSIC_1));
      #ifdef FF_EGPRS
        AWAIT ( mphc_async_ho_req_ho_type_fn(HOT_BLIND, 0,0) );
        SEND  ( mphc_async_ho_con_1()        );
        SEND  ( mphc_handover_finished_fn(HO_COMPLETE)   );
        AWAIT ( mph_dedicated_cnf_fn(DEDI_RES_OK)        );
      #else // FF_EGPRS
        AWAIT ( mph_dedicated_cnf_fn(DEDI_RES_CELL_NOT_SYNC) );
      #endif // FF_EGPRS

    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR1013B-------------------------------
 *
 * Description: A Async handover is performed. Before the handover there was a large
 *              BA list of 20 entries in use. Handover is requested for one of the carrier in the BA list
 *              but the sync was not decoded. ALR shall send Handover type as HOT_BLIND.
 *-------------------------------------------------------------------------------*/
T_CASE ALR1013B()
{
  BEGIN_CASE ("handover_full_ba_list_alr1013B")
  {
    TRAP_FOR_DEACT {
      initial_setting_state_to_dedicated(); /*NO sync was decoded for ARFCN_1, though this was the carrier of BA list*/
      SEND(mph_dedicated_req_fn(MODE_ASYNC_HANDOVER, ARFCN_1, BSIC_1));
      AWAIT ( mphc_async_ho_req_ho_type_fn(HOT_REGULAR,0,0) );
      SEND  ( mphc_async_ho_con_1()        );
      SEND  ( mphc_handover_finished_fn(HO_COMPLETE)   );
      AWAIT ( mph_dedicated_cnf_fn(DEDI_RES_OK)        );

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR1013C-------------------------------
 *
 * Description: A Async handover is performed. Before the handover there was a large
 *              BA list of 20 entries in use. Handover is requested for one of the carrier in the BA list
 *              but the sync was not decoded. ALR shall send Handover type as HOT_BLIND. If sync at
*              this carrier fails L1 shall send handover finished message with HO_SYNC_FAIL
 *-------------------------------------------------------------------------------*/
T_CASE ALR1013C()
{
  BEGIN_CASE ("handover_full_ba_list_alr1013C")
  {
    TRAP_FOR_DEACT {
      initial_setting_state_to_dedicated(); /*NO sync was decoded for ARFCN_2, this was not the carrier of BA list*/
      SEND(mph_dedicated_req_fn(MODE_ASYNC_HANDOVER, ARFCN_2, BSIC_1));
      #ifdef FF_EGPRS
        AWAIT ( mphc_async_ho_req_ho_type_fn(HOT_BLIND,0,0)  );
        SEND  ( mphc_async_ho_con_1()                        );
        SEND  ( mphc_handover_finished_fn(HO_SYNC_FAIL)      );
        AWAIT ( mph_dedicated_cnf_fn(DEDI_RES_SYNC_FAIL)     );
      #else // FF_EGPRS
        AWAIT ( mph_dedicated_cnf_fn(DEDI_RES_CELL_NOT_SYNC) );
      #endif // FF_EGPRS
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR1013D-------------------------------
 *
 * Description: A Sync handover is performed. Before the handover there was a large
 *              BA list of 20 entries in use. Handover is requested for one of the carrier in the BA list
 *              but the sync was not decoded. ALR shall send Handover type as HOT_BLIND.
 *-------------------------------------------------------------------------------*/
T_CASE ALR1013D()
{
  BEGIN_CASE ("handover_full_ba_list_alr1013B")
  {
    TRAP_FOR_DEACT {
      initial_setting_state_to_dedicated(); /*NO sync was decoded for ARFCN_1, though this was the carrier of BA list*/
      SEND(mph_dedicated_req_fn(MODE_SYNC_HANDOVER, ARFCN_1, BSIC_1));
      AWAIT ( mphc_sync_ho_req_ho_type_fn(HOT_REGULAR, 0,0) );
      SEND  ( mphc_async_ho_con_1()        );
      SEND  ( mphc_handover_finished_fn(HO_COMPLETE)   );
      AWAIT ( mph_dedicated_cnf_fn(DEDI_RES_OK)        );

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR1013E-------------------------------
 *
 * Description: A Pre Sync handover is performed. Before the handover there was a large
 *              BA list of 20 entries in use. Handover is requested for one of the carrier in the BA list
 *              but the sync was not decoded. ALR shall send Handover type as HOT_BLIND.
 *-------------------------------------------------------------------------------*/
T_CASE ALR1013E()
{
  BEGIN_CASE ("handover_full_ba_list_alr1013E")
  {
    TRAP_FOR_DEACT {
      initial_setting_state_to_dedicated(); /*NO sync was decoded for ARFCN_1, though this was the carrier of BA list*/
      SEND(mph_dedicated_req_fn(MODE_PRE_SYNC_HANDOVER, ARFCN_1, BSIC_1));
      AWAIT ( mphc_presync_ho_req_ho_type_fn(HOT_REGULAR, 0,0) );
      SEND  ( mphc_async_ho_con_1()        );
      SEND  ( mphc_handover_finished_fn(HO_COMPLETE)   );
      AWAIT ( mph_dedicated_cnf_fn(DEDI_RES_OK)        );

    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR1020-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the
 *              neighbour cells 1, 14 and 124.
 *              The fieldstrength is 56 for channel 23, 12 for channel 1,
 *              44 for channel 14 and 25 for channel 124 (all values in GSM range).
 *              The ranking for the neighbour cells is 14, 124 and channel 1.
 *              Each reports contains two fieldstrength values per channel.
 *              The multiframe period is set to 6. The first measurement report
 *              is send to RR after five reports from PL. Then after each three
 *              reports from PL a measurement report is send to RR. The NCC permitted
 *              check for neighbour cell 124 fails. The cell not excluded
 *              from further attempts it in IDLE state.It should try to sync after 10 secs
 *-------------------------------------------------------------------------------*/

T_CASE ALR1020()
{
  BEGIN_CASE ("ncell_synch_ncc_permitted_check_alr1020")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('D');
      SEND  ( mphc_rxlev_periodic_ind_2()  );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)      );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)      );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)      );
      SEND  ( mphc_rxlev_periodic_ind_2()  );
      AWAIT ( mph_measurement_ind_13()     );
      SEND  ( mphc_rxlev_periodic_ind_2()  );
      SEND  ( mphc_ncell_sync_ind(ARFCN_14,FN_OFF_ARFCN_14,TIME_ALIGNMT_14,BSIC_16)      );
      AWAIT ( mphc_ncell_bcch_req_4()      );
      SEND  ( mphc_rxlev_periodic_ind_2()  );
      SEND  ( mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_1)      );
      SEND  ( mphc_ncell_sync_ind(ARFCN_1,FN_OFF_ARFCN_1,TIME_ALIGNMT_1,BSIC_16)      );
      AWAIT ( mphc_ncell_bcch_req_5()      );
      SEND  ( mphc_rxlev_periodic_ind_2()  );
      SEND  ( mphc_rxlev_periodic_ind_2()  );
      SEND  ( mphc_rxlev_periodic_ind_2()  );
      AWAIT ( mph_measurement_ind_13()     );
      SEND  ( mphc_rxlev_periodic_ind_2()  );
      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_124) );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR1020A-------------------------------
 *
 * Description:	The BA list contains the serving cell 23 and the
 *              neighbour cells 1, 14 and 124.
 *              The fieldstrength is 56 for channel 23, 20 for channel 1,
 *              46 for channel 14 and 30 for channel 124 (all values in GSM range).
 *              The ranking for the neighbour cells is 14, 124 and channel 1.
 *              The NCC permitted check for neighbour cell 1 fails. The cell not excluded
 *              from further attempts it in DEDICATED state.
 *              It should try to sync after 30 secs.
 *-------------------------------------------------------------------------------*/

T_CASE ALR1020A()
{
  BEGIN_CASE ("ncell_synch_ncc_permitted_check_alr1020")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      definition_of_bcch_allocation_alr046('D');
      

      SEND  ( mph_random_access_req_1()          );
      AWAIT ( mphc_stop_rxlev_periodic_req()     );
      AWAIT ( mphc_stop_scell_bcch_req()         );
      AWAIT ( mphc_scell_nbcch_req_full()        );
      AWAIT ( mphc_ra_req_1()                    );
      SEND  ( mphc_ra_con_1()                    );
      AWAIT ( mph_random_access_cnf_1()          );
      AWAIT ( mphc_ra_req_2()                    );
      immediate_assignment_for_the_mobile_station_alr056();
      SEND  ( mph_neighbourcell_req_4('A') );
      AWAIT ( mphc_update_ba_list_2()    );
      MUTE  (500);

      SEND (mphc_meas_report_4a() );      
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_14)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_124)       );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1)       );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)           );


      SEND (mphc_meas_report_4a() ); 
      SEND (mphc_ncell_sync_ind(ARFCN_14,FN_OFF_ARFCN_14,TIME_ALIGNMT_14,BSIC_16) );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)           );
      
      SEND (mphc_meas_report_4a() ); 
      SEND (mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_16) );
      SEND (mphc_ncell_sync_ind(ARFCN_1,FN_OFF_ARFCN_1,TIME_ALIGNMT_1,BSIC_1) );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19)           );

      wait_ncsync_dedicated_1020A(18);

      AWAIT ( mphc_ncell_list_sync_req_1c () );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19) );
      SEND (mphc_ncell_sync_ind(ARFCN_14,FN_OFF_ARFCN_14,TIME_ALIGNMT_14,BSIC_16) );
      SEND (mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_16) );  

      wait_ncsync_dedicated_1020A(21);

      AWAIT ( mphc_ncell_list_sync_req_1c () );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19) );
      SEND (mphc_ncell_sync_ind(ARFCN_14,FN_OFF_ARFCN_14,TIME_ALIGNMT_14,BSIC_16) );
      SEND (mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_16) ); 

      wait_ncsync_dedicated_1020A(21);

      AWAIT ( mphc_ncell_list_sync_req_1c () );
      AWAIT ( mph_measurement_ind_dummy(ARFCN_23,0x19) );
      SEND (mphc_ncell_sync_ind(ARFCN_14,FN_OFF_ARFCN_14,TIME_ALIGNMT_14,BSIC_16) );
      SEND (mphc_ncell_sync_ind(ARFCN_124,FN_OFF_ARFCN_124,TIME_ALIGNMT_124,BSIC_16) ); 
 

      AWAIT ( mphc_ncell_sync_req_fn(ARFCN_1) );

    } HANDLE_DEACT;
  }
}


#ifdef  GPRS
/*------------------------Following is for ALR1015-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 *              The cell has GPRS. No neighbour cells measured.
 *
 *---------------------------------------------------------------------------------*/
T_CASE ALR1015()
{
  BEGIN_CASE ("MPH_MEAS_REP_REQ")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('C');
      SEND  ( mph_meas_rep_req(NC_START_MEAS)           );
      SEND  ( mphc_rxlev_periodic_ind_9(40,30,20,10)    );
      SEND  ( mphc_rxlev_periodic_ind_9(40,30,20,10)    );
      AWAIT ( mph_measurement_ind_35()                  );
      SEND  ( mph_meas_rep_req(NC_REPORT_MEAS)          );
      AWAIT ( mph_meas_rep_cnf_sc()                     );
      SEND  ( mph_meas_rep_req(NC_STOP_MEAS)            );
    } HANDLE_DEACT;
  }
}
#endif  // GPRS

#ifdef  GPRS
/*------------------------Following is for ALRG1016-------------------------------
 *
 * Description:	RR selects the channel 23 after reading the BCCH carrier.
 *              The cell has GPRS. Neighbour cell list consists of 3 cells
 *              1, 14, 124
 *
 *---------------------------------------------------------------------------------*/
T_CASE ALR1016()
{
  BEGIN_CASE ("MPH_MEAS_REP_REQ")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_nwctrl('A');

      /* Check that NC report does not contain any neighbour cells in
	 RD_BCCH_P_RR state */
      SEND  (mph_meas_rep_req(NC_START_MEAS)          );
      SEND  ( mphc_rxlev_periodic_ind_9(40,30,20,10)  );
      MUTE  (10);
      SEND  ( mphc_rxlev_periodic_ind_9(50,40,30,20)  );
      MUTE  (10);
      SEND  ( mphc_rxlev_periodic_ind_9(40,30,20,10)  );
      MUTE  (10);
      SEND  (mph_meas_rep_req(NC_REPORT_MEAS)         );
      AWAIT (mph_meas_rep_cnf_sc()                    );

      /* Send SI3 for the neighbour cells and request report again */
      SEND  (mph_meas_rep_req(NC_STOP_MEAS)           );
      SEND  (mphc_ncell_bcch_ind_si3(ARFCN_1)         );
      SEND  (mphc_ncell_bcch_ind_si3(ARFCN_14)        );
      SEND  (mphc_ncell_bcch_ind_si3(ARFCN_124)       );
      AWAIT (mphc_stop_ncell_bcch_req(ARFCN_1)        );
      AWAIT (mphc_stop_ncell_bcch_req(ARFCN_14)       );
      AWAIT (mphc_stop_ncell_bcch_req(ARFCN_124)      );
      SEND  (mph_meas_rep_req(NC_START_MEAS)          );
      SEND  (mphc_rxlev_periodic_ind_9(40,30,20,10)   );
      SEND  (mphc_rxlev_periodic_ind_9(50,40,30,20)   );
      AWAIT (mph_measurement_ind_35()                 );
      SEND  (mphc_rxlev_periodic_ind_9(40,30,20,10)   );
      AWAIT (mph_unitdata_ind_si3(ARFCN_1)            );
      AWAIT (mph_unitdata_ind_si3(ARFCN_14)           );
      AWAIT (mph_unitdata_ind_si3(ARFCN_124)          );
      SEND  (mph_meas_rep_req(NC_REPORT_MEAS)         );
      AWAIT (mph_meas_rep_cnf_sc_n3()                 );

      MUTE(1000);

    }HANDLE_DEACT;
  }
}
/*------------------------Following is for ALRG1018-------------------------------
 *
 * Description: RR selects the channel 23 after reading the BCCH carrier.
 *              The cell has GPRS. Neighbour cell list consists of one cell
 *              512.This tc tests the measurement reports in PTM state when
 *              PBCCH is not present
 *
 *---------------------------------------------------------------------------------*/

T_CASE ALR1018()
{
  BEGIN_CASE ("Meas_reports_in_PTM_state_alr1018")
  {
    TRAP_FOR_DEACT {
      initiation_by_rr_8_channels_available_alr002();
      find_bcch_carrier_first_channel_alr603();
      read_bcch_data_alr007();
      configuring_ms_for_idle_mode_and_meas();
      SEND  ( mph_random_access_req_1()        );
      AWAIT ( mphc_stop_rxlev_periodic_req()   );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      AWAIT ( mphc_scell_nbcch_req_full()      );
      AWAIT ( mphc_ra_req_3()                  );
      SEND  ( mphc_ra_con_1()                  );
      AWAIT ( mph_random_access_cnf_1()        );
      AWAIT ( mphc_ra_req_4()                  );
      immediate_assignment_for_the_mobile_station_alr056();
      pdch_assignment_alrg007();
      SEND  ( mph_idle_req_2g() );
      AWAIT ( mphc_stop_rxlev_periodic_req()   );
      AWAIT ( mphc_stop_scell_bcch_req()       );
      SEND  ( mph_mon_ctrl_req(ENTER_PTM_BCCH) );
      SEND  ( mph_meas_rep_req(NC_START_MEAS)  );
      SEND  ( prim_tb_meas_ind_1()             );
      MUTE(100);
      SEND  ( prim_tb_meas_ind_2()             );
      MUTE(100);
      SEND  ( prim_tb_meas_ind_3()             );
      MUTE(100);
      SEND  ( prim_tb_meas_ind_1()             );
      MUTE(100);
      SEND  ( prim_tb_meas_ind_2()             );
      AWAIT ( mph_measurement_ind_36A()        );
      SEND  ( mph_meas_rep_req(NC_REPORT_MEAS) );
      AWAIT ( mph_meas_rep_cnf_sc_n1()         );
    }HANDLE_DEACT;
  }
}
#endif  // GPRS

#ifdef  FF_EGPRS
/*------------------------Following is for ALR9800-------------------------------
 *
 * Description: Acquisition of SI-2quater when scheduled on EBCCH and when
                GPRS is not supported
 *-------------------------------------------------------------------------------*/

T_CASE ALR9800()
{
  BEGIN_CASE ("EMR CASE: Periodic SI-2quater acquisition on EBCCH, GPRS not supported ")
  {
    TRAP_FOR_DEACT {
      rr_cell_selection_emr('I','A','A');
      SEND  ( mphc_rxlev_periodic_ind_1() );      /* 3 */
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_emr_1()     );

      SEND  ( mphc_rxlev_periodic_ind_1() );      /* 4 */
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_emr_1()     );

      SEND  ( mphc_rxlev_periodic_ind_1() );      /* 5 */
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_emr_1()     );

      SEND  ( mphc_rxlev_periodic_ind_1() );      /* 6 */
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_emr_1()     );
      AWAIT (mphc_scell_ebcch_req_si2q() );

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif  // FF_EGPRS

#ifdef  FF_EGPRS
/*------------------------Following is for ALR9801-------------------------------
 *
 * Description: Acquisition of SI-2quater when scheduled on NBCCH and when
                GPRS is not supported
 *-------------------------------------------------------------------------------*/

T_CASE ALR9801()
{
  BEGIN_CASE ("EMR CASE: Periodic SI-2quater acquisition on NBCCH, GPRS not supported ")
  {
    TRAP_FOR_DEACT {
      rr_cell_selection_emr('J','A','A');

      SEND  ( mphc_rxlev_periodic_ind_1() );      /* 3 */
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_emr_1()     );

      SEND  ( mphc_rxlev_periodic_ind_1() );      /* 4 */
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_emr_1()     );

      SEND  ( mphc_rxlev_periodic_ind_1() );      /* 5 */
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_emr_1()     );

      SEND  ( mphc_rxlev_periodic_ind_1() );      /* 6 */
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_emr_1()     );

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif  //FF_EGPRS

#ifdef  FF_EGPRS
/*------------------------Following is for ALR9802-------------------------------
 *
 * Description: Acquisition of SI-2quater when scheduled on EBCCH and when
                GPRS is supported
 *-------------------------------------------------------------------------------*/

T_CASE ALR9802()
{
  BEGIN_CASE ("EMR CASE: Periodic SI-2quater acquisition on EBCCH, GPRS supported ")
  {
    TRAP_FOR_DEACT {
      rr_cell_selection_emr('K','B','A');
      /*Indication from SI-13 to update SI-2 and it's cousins*/
      SEND (mph_mon_ctrl_req_1(START_MON_NBCCH,UPDATE_SI2_SI2BIS_OR_SI2TER) );
      AWAIT (mphc_scell_nbcch_req_1() );
      SEND  ( mphc_data_ind_2() );
      AWAIT ( mph_unitdata_ind_si2(ARFCN_23) );
      AWAIT (mphc_scell_ebcch_req_si2q() );

      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif  // FF_EGPRS

#ifdef  FF_EGPRS
/*------------------------Following is for ALR9803-------------------------------
 *
 * Description: Acquisition of SI-2quater when scheduled on EBCCH after cell reselection,
 *              then periodic acquisition when GPRS is not supported
 *-------------------------------------------------------------------------------*/

T_CASE ALR9803()
{
  BEGIN_CASE ("EMR CASE: Periodic SI-2quater acquisition on EBCCH, GPRS not supported ")
  {
    TRAP_FOR_DEACT {
      rr_cell_selection_emr('H','B','Z');

      /*cell reselection starts*/
      SEND  ( mph_idle_req_15_si2q(MODE_CELL_RESELECTION));
      AWAIT ( mphc_stop_rxlev_periodic_req() );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_new_scell_req_20()          );
      SEND  ( mphc_new_scell_con_1()           );
      AWAIT ( mphc_start_ccch_req_5()          );
      AWAIT ( mphc_scell_nbcch_req_full()         );
      SEND  ( mphc_data_ind_63()               );
      AWAIT ( mph_unitdata_ind_si3(ARFCN_14)             );
      AWAIT ( mphc_start_ccch_req_12()         );
      SEND  ( mphc_data_ind_64()               );
      AWAIT ( mph_unitdata_ind_si1(ARFCN_14)            );
      SEND  ( mphc_data_ind_11()               );
      AWAIT ( mphc_start_ccch_req_13()         );
      SEND  ( mphc_data_ind_65()               );
      AWAIT ( mph_unitdata_ind_si2(ARFCN_14)            );
      SEND  ( mph_idle_req_15_si2q(MODE_CELL_SELECTION));
      SEND  ( mph_neighbourcell_req_13()       );
      AWAIT ( mphc_stop_scell_bcch_req()     );
      AWAIT ( mphc_start_ccch_req_14()         );
      AWAIT (mphc_scell_ebcch_req_si2q() );
      AWAIT ( mphc_rxlev_periodic_req_18()     );
      AWAIT ( mphc_rxlev_periodic_req_18()     );

      SEND (mph_mon_ctrl_req_1(STOP_MON_BCCH,UPDATE_SI2QUATER) );
      AWAIT ( mphc_scell_nbcch_req_periodic()  );
      /*


      MUTE  ( 500 );*/
    } HANDLE_DEACT;
  }
}
#endif  // FF_EGPRS

#ifdef  FF_EGPRS
/*------------------------Following is for ALR9804-------------------------------
 *
 * Description: To test Enhanced reporting using both BA(BCCH) and BA(SACCH) in succession
 *-------------------------------------------------------------------------------*/

T_CASE ALR9804()
{
  BEGIN_CASE ("To test Enhanced reporting using both BA(BCCH) and BA(SACCH) in succession")
  {
    TRAP_FOR_DEACT {
      /*Step 1: MS moved to Idle state, SI-2quater acquired, enhanced reporting
        is enabled in ALR on BA(BCCH) */
      rr_cell_selection_emr('J','C','B');
      SEND  ( mphc_rxlev_periodic_ind_emr_1() );
      AWAIT (mphc_ncell_list_sync_req_emr());
      SEND (mphc_ncell_sync_ind_emr(ARFCN_59, 20,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_59));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_73, 21,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_73));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_90, 22,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_90));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_108,24,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_108));

      SEND  ( mphc_rxlev_periodic_ind_emr_1() );

      SEND (mphc_ncell_sync_ind_emr(ARFCN_1, 16,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_15, 17,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_30, 18,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_30));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_46,19,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_46));

      SEND  ( mphc_rxlev_periodic_ind_emr_1() );
      TIMEOUT(100);
      AWAIT ( mph_measurement_ind_emr_1() );

      /*Step 2: MS moved to dedicated state and continues to measure
        and starts reporting on BA(BCCH) */
      start_of_sending_channel_request_messages_gsm_900__alr9803();
      immediate_assignment_for_the_mobile_station_alr056();

      SEND (mphc_meas_report_emr('A') );
      AWAIT ( mph_measurement_ind_emr_1() );

      /*Step 3: MS received BA(SACCH), ALR configured with Neighbour cell request for BA(SACCH),
        Reporting is reset to 'Normal' and MS starts normal reporting on BA(SACCH) */
      SEND (mph_neighbourcell_req_emr_2() );
      AWAIT (mphc_update_ba_list_emr() );
      SEND (mph_enhpara_update_req_1 ('C')); /*report type NORMAL*/
      SEND (mphc_meas_report_emr('B') ); /*Meas report on BA(SACCH)*/
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_580) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_584) );
      AWAIT (mphc_ncell_sync_req_initial(ARFCN_513) );
      AWAIT (mph_measurement_ind_emr_1());


      SEND (mphc_ncell_sync_ind_emr(ARFCN_580, 28,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_584, 48,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_513, 17,SB_FOUND));
      SEND (mphc_meas_report_emr('B') ); /*Meas report on BA(SACCH)*/

      AWAIT (mph_measurement_ind_emr_1());
      /*Step 3: Configure ALR for enhanced reporting with new GSM NC list
        Assuming that MI messages have been received by RR through DL
        we now inject Enhanced measurement parameters to ALR */
      SEND (mph_enhpara_update_req_1 ('D'));
      SEND (mphc_meas_report_emr('B') ); /*Meas report on BA(SACCH)*/
      AWAIT (mphc_ncell_list_sync_req_emr());
      AWAIT (mph_measurement_ind_emr_1());
      SEND (mphc_ncell_sync_ind_emr(ARFCN_581, 29,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_578, 27,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_525, 26,SB_FOUND));
      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}
#endif  // FF_EGPRS

#ifdef  FF_EGPRS
/*------------------------Following is for ALR9805-------------------------------
 *
 * Description: To test Enhanced reporting using BA(SACCH) with cells of low report priority
 *              and change in ranking at second report
 *-------------------------------------------------------------------------------*/

T_CASE ALR9805()
{
  BEGIN_CASE ("To test Enhanced reporting using BA(SACCH) with cells of low report priority")
  {
    TRAP_FOR_DEACT {
      rr_cell_selection_emr('J','C','Z');
      SEND  ( mphc_rxlev_periodic_ind_emr_1() );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_108) );
      AWAIT (mphc_ncell_sync_req_initial(ARFCN_90) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_73) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_59) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_46) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_30) );


      SEND (mphc_ncell_sync_ind_emr(ARFCN_59, 20,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_59));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_73, 21,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_73));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_90, 22,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_90));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_108,24,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_108));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_30, 18,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_30));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_46,19,SB_FOUND));
      AWAIT (mphc_ncell_bcch_req_emr(ARFCN_46));


      /*Step 2: MS moved to dedicated state and continues to measure
        and starts reporting on BA(BCCH) */
      start_of_sending_channel_request_messages_gsm_900__alr9803();
      immediate_assignment_for_the_mobile_station_alr056();

      SEND (mphc_meas_report_emr('A') );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_1) );
      AWAIT ( mph_measurement_ind_emr_1() );

      /*Step 3: MS received BA(SACCH), ALR configured with Neighbour cell request for BA(SACCH),
        Reporting is reset to 'Normal' and MS starts normal reporting on BA(SACCH) */
      SEND (mph_neighbourcell_req_emr_2() );
      AWAIT (mphc_update_ba_list_emr() );

      SEND (mphc_meas_report_emr('B') ); /*Meas report on BA(SACCH)*/
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_580) );
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_584) );
      AWAIT (mphc_ncell_sync_req_initial(ARFCN_513) );
      /*
      AWAIT ( mphc_ncell_sync_req_initial(ARFCN_581) );
            AWAIT ( mphc_ncell_sync_req_initial(ARFCN_578) );
            AWAIT ( mphc_ncell_sync_req_initial(ARFCN_525) ); */

      AWAIT (mph_measurement_ind_emr_1());
      SEND (mphc_ncell_sync_ind_emr(ARFCN_580, 28,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_584, 48,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_513, 17,SB_FOUND));
      /*
      SEND (mphc_ncell_sync_ind_emr(ARFCN_581, 29,SB_FOUND));
            SEND (mphc_ncell_sync_ind_emr(ARFCN_578, 27,SB_FOUND));
            SEND (mphc_ncell_sync_ind_emr(ARFCN_525, 26,SB_FOUND));*/

      SEND (mphc_meas_report_emr('B') ); /*Meas report on BA(SACCH)*/

      AWAIT (mph_measurement_ind_emr_1());
      /*Step 3: Configure ALR for enhanced reporting with new GSM NC list
        Assuming that MI messages have been received by RR through DL
        we now inject Enhanced measurement parameters to ALR
      Here, cells are of mixed priority, some low and some high*/
      SEND (mph_enhpara_update_req_1 ('E'));
      SEND (mphc_meas_report_emr('B') ); /*Meas report on BA(SACCH)*/
      AWAIT (mphc_ncell_list_sync_req_emr());
      SEND (mphc_ncell_sync_ind_emr(ARFCN_581, 29,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_578, 27,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_525, 26,SB_FOUND));
      AWAIT (mph_measurement_ind_emr_1());
      SEND (mphc_meas_report_emr('C') );
      AWAIT (mphc_ncell_list_sync_req_emr());
      AWAIT (mph_measurement_ind_emr_1());
      SEND (mphc_ncell_sync_ind_emr(ARFCN_512, 16,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_516, 25,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_582, 30,SB_FOUND));
      SEND (mphc_ncell_sync_ind_emr(ARFCN_583, 31,SB_FOUND));
      SEND (mphc_meas_report_emr('C') );
      AWAIT (mph_measurement_ind_emr_1());
      MUTE  ( 500 );
    } HANDLE_DEACT;
  }
}


#endif  // FF_EGPRS

/*------------------------Following is for ALR9806-------------------------------
 * Description:
 *        Measurement Reporting for the following Scenario
 *          In case of Cell Reselection the MPH_IDLE_REQ (MODE_CELL_RESELECTION)
 *        will be sent two times from RR to ALR. Because during the
 *        cell reselection the Paging Group configuration might not be known
 *        by the MS. MS has to read the SI3 to find the Paging Group. The second
 *        IDLE REQ will be sent from RR to ALR to configure the Paging group.
 *-------------------------------------------------------------------------------*/


T_CASE ALR9806()
{
  BEGIN_CASE ("measurement_reporting_after_cell_reselection_bs_pa_mfrms_4_alr9806")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();

      /*
       * Fist IDLE_REQ configure PL with Cell Reselection Mode
       * Second IDLE_REQ configure PL with Cell Selction Mode [PGM Config]
       */

      measurement_reporting_configuration_alr9806();

      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_7()     );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      SEND  ( mphc_rxlev_periodic_ind_1() );
      AWAIT ( mph_measurement_ind_7()     );

      MUTE (1000);

    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9806a-------------------------------
 * Description: Cell selection.with FULL Search.mode will do measure
 *                the carrier strength 5 times and the average will be
 *                reported to RR through MPH_POWER_CNF
 *-------------------------------------------------------------------------------*/

T_CASE ALR9806A()
{
  BEGIN_CASE ("initiation_by_rr_with_full_search_alr9806a")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      /* FULL Search will do the measurement 5 Times */
      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'X')     );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_2, PCH_INTERRUPT);

      AWAIT ( mph_power_cnf_2()           );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9806b-------------------------------
 * Description: Cell selection.with Normal Search.mode will do measure
 *                the carrier strength 5 times and the average will be
 *                reported to RR through MPH_POWER_CNF
 *-------------------------------------------------------------------------------*/

T_CASE ALR9806B()
{
  BEGIN_CASE ("initiation_by_rr_with_normal_search_alr9806b")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      /* Normal Search will do the measurement 5 Times */
      SEND  ( mph_power_req_variant(NORMAL_SEARCH_MODE, 'X')     );
      rxlevel_measurement_based_on_search_quadband(NORMAL_SEARCH_MODE, RXLEV_IDX_2, PCH_INTERRUPT);

      AWAIT ( mph_power_cnf_2()           );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9806c-------------------------------
 * Description: Cell selection.with FAST Search.mode will do measure
 *                the carrier strength only once and the strengh will be
 *                reported to RR through MPH_POWER_CNF
 *                and will measure the rxlevel
 *-------------------------------------------------------------------------------*/

T_CASE ALR9806C()
{
  BEGIN_CASE ("initiation_by_rr_with_fast_search_alr9806c")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      /* FAST Search will do the measurement only once   */
      SEND  ( mph_power_req_variant(FAST_SEARCH_MODE, 'X')     );
      rxlevel_measurement_based_on_search_quadband(FAST_SEARCH_MODE, RXLEV_IDX_2, PCH_INTERRUPT);

      AWAIT ( mph_power_cnf_2()           );
      MUTE(1000);
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9806d-------------------------------
 * Description: RR starts a cell selection.with BLACKLIST Search.mode
 *                and will measure the rxlevel only once
 *-------------------------------------------------------------------------------*/

T_CASE ALR9806D()
{
  BEGIN_CASE ("initiation_by_rr_with_Blacklist_search_alr9806D")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      /* Blacklist Search will do the measurement only once */
      SEND  ( mph_power_req_variant(BLACK_LIST_SEARCH_MODE, 'X')     );
      rxlevel_measurement_based_on_search_quadband(BLACK_LIST_SEARCH_MODE, RXLEV_IDX_2, PCH_INTERRUPT);

      /* Blacklist Search will report only the inactive carrier list to RR
       * The number of channels in the POWER_CNF will be always 0
       */
      AWAIT(mph_power_cnf_2());

      MUTE(1000);
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR9807-------------------------------
 * Description:
 *        (1) Whitelist carriers provided by the RR needs to be reported on top
 *            of the MPH_POWER_CNF in decreasing order, other carriers will
 *            follow the whitelist carriers in decreasing order.
 *
 *            Whitelist carriers are : ARFCN_14_STD_900, ARFCN_1_STD_900
 *
 *        (2) Whitelist carriers provided by the RR needs to be reported on top
 *            of the MPH_POWER_CNF in decreasing order, other carriers will
 *            follow the whitelist carriers in decreasing order. If the
 *            whitelist carriers reported by RR is more than 32 needs to be
 *            ignored by ALR.
 *        (3) Cariers reported by L1 are 0, then the MPH_POWER_CNF should report
 *            0 carriers to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9807()
{
  BEGIN_CASE ("whitelist_validation_case_alr9807")
  {
    TRAP_FOR_DEACT {
      /* Scenario 1 */
      filter_and_routings_gsm_900__alr000();
      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'B')         );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_3_new()       );
      /* Scenario 2 */
      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'C') );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_9, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_32_whitelist()       );
      /* Scenario 3 */
      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'C') );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_10, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_2()       );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9808-------------------------------
 * Description:          (Scenarios)
 *      (1) Carriers which has the rxlevel lower than the LOWER_THRESHOLD
 *            should be reported in inactive_carrier_list of MPH_POWER_CNF
 *      (2) Carriers which are reported as blacklist carriers by RR shall not
 *            participate in MPH_POWER_CNF (NORMAL SEARCH)
 *      (3) Carriers which are reported as blacklist carriers by RR shall not
 *            participate in MPH_POWER_CNF (FAST SEARCH)
 *      (4) Carriers which are reported as blacklist carriers by RR shall not
 *            participate in MPH_POWER_CNF (BLACK LIST SEARCH)
 *      (5) Carriers which are reported as blacklist carriers  by RR shall
 *            participate in MPH_POWER_CNF only in case of FULLSEARCH
 *      (6) Same carriers in Black list and other region white list
 *          Ex:  Same Data in European Region Blacklist and American Region white list
 *               could be send in MPH_POWER_REQ.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9808()
{
  BEGIN_CASE ("blacklist_validation_case_alr9808")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      /* Scenario 1 */
      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'D') );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_7, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_black()       );

      /* Scenario 2 */
      SEND  ( mph_power_req_variant(NORMAL_SEARCH_MODE, 'F')         );
      rxlevel_measurement_based_on_search_quadband(NORMAL_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_same_white_black()       );

      /* Scenario 3 */
      SEND  ( mph_power_req_variant(FAST_SEARCH_MODE, 'F')         );
      rxlevel_measurement_based_on_search_quadband(FAST_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_same_white_black()       );

      /* Scenario 4 */
      SEND  ( mph_power_req_variant(BLACK_LIST_SEARCH_MODE, 'F')         );
      rxlevel_measurement_based_on_search_quadband(BLACK_LIST_SEARCH_MODE, RXLEV_IDX_7, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_variant('D')       );

      /* Scenario 5 */
      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'D') );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_7, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_black()       );

      /* Scenario 6 */
      SEND  ( mph_power_req_variant(NORMAL_SEARCH_MODE, 'M')         );
      rxlevel_measurement_based_on_search_quadband(NORMAL_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_3_new()       );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9809-------------------------------
 * Description:
 *        (Scenario 1 & 2)
 *               Rxlevel for FULL/NORMAL SEARCH (measurement will be taken
 *               only once) should be reported by the same measured value
 *               {ARFCN, Measured value }       : { 23, 280},{124, 125}
 *               Resultant (POWER_CNF) rxlevel : { 23, 56}, {124, 25}
 *               [In POWER_CNF rxlevel is an U8 so instead of 280 value 24]
 *               [(280) 1 0001 1000 - 9th bit 1 will not be received]
 *        (Scenario 3 )
 *               Rxlevel for FAST SEARCH (measurement will be taken
 *               only once) should be reported by the same measured value
 *               {ARFCN, Measured value }       : { 23, 280},{124, 125}
 *               Resultant (POWER_CNF) rxlevel : { 23, 24}, {124, 125}
 *               [In POWER_CNF rxlevel is an U8 so instead of 280 value 24]
 *               [(280) 1 0001 1000 - 9th bit 1 will not be received]
 *-------------------------------------------------------------------------------*/

T_CASE ALR9809()
{
  BEGIN_CASE ("rxlevel_validation_case_alr9809")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      /* Scenario - 1 = FULL_SEARCH */
      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'F')         );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_3_new()       );
      /* Scenario - 2 = NORMAL_SEARCH */
      SEND  ( mph_power_req_variant(NORMAL_SEARCH_MODE, 'F')         );
      rxlevel_measurement_based_on_search_quadband(NORMAL_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_rxlevel('A')       );
      /* Scenario - 3 = FAST_SEARCH */
      SEND  ( mph_power_req_variant(FAST_SEARCH_MODE, 'F')         );
      rxlevel_measurement_based_on_search_quadband(FAST_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_rxlevel('B')       );
      /*
       * In case of blacklist search the number of channels in MPH_POWER_CNF will be always
       *  0. So testing rxlevel is not needed in case of BLACK_LIST_SEARCH.
       */
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9810-------------------------------
 *   Description:  (Scenario)
 *      (1) Test case for STD_850 (American band) with US BIT
 *      (2) Test case for STD_850 (American band) with US BIT & Blacklist Carriers
 *-------------------------------------------------------------------------------*/


T_CASE ALR9810()
{
  BEGIN_CASE ("validation_case_for_american_band_alr9810")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      /* Scenario 1 - US BIT */
      SEND  ( mph_power_req_variant(FAST_SEARCH_MODE, 'G')         );
      rxlevel_measurement_based_on_search_american(FAST_SEARCH_MODE, STD_850, RXLEV_IDX_1);
      AWAIT ( mph_power_cnf_american()       );
      /* Scenario 2 - Blacklisted carriers */
      SEND  ( mph_power_req_variant(FAST_SEARCH_MODE, 'H')         );
      rxlevel_measurement_based_on_search_american(FAST_SEARCH_MODE, STD_850, RXLEV_IDX_1);
      AWAIT ( mph_power_cnf_american1()       );
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9811-------------------------------
 * Description: Parallel Search type with FULL_SEARCH_MODE
 *      The parallel type will be used only in case of the MS is in idle mode,
 *   and the other task would not get disturbed while doing the power measurement.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9811()
{
  BEGIN_CASE ("parallel_search_cs_improvement_cases_alr9811")
  {
    TRAP_FOR_DEACT {
      /* To move the MS to IDLE mode to start parallel search */
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      /* Send a Parallel Search type with FULL_SEARCH_MODE */
      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'A')     );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_1, NO_PCH_INTERRUPT);

      AWAIT(mph_power_cnf_3_new());

      /*  The following primitives are to ensure that after the parallel search
       *  ALR is still in IDLE State.
       */
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();
      neighbour_cell_in_dedicated_alr155();

    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9812-------------------------------
 * Description: Parallel Search type with NORMAL_SEARCH_MODE
 *      The parallel type will be used only in case of the MS is in idle mode,
 * and the other task would not get disturbed while doing the power measurement.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9812()
{
  BEGIN_CASE ("parallel_search_cs_improvement_cases_alr9812")
  {
    TRAP_FOR_DEACT {
      /* To move the MS to IDLE mode to start parallel search */
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      /* Send a Parallel Search type with NORMAL_SEARCH_MODE */
      SEND  ( mph_power_req_variant(NORMAL_SEARCH_MODE, 'A')     );
      rxlevel_measurement_based_on_search_quadband(NORMAL_SEARCH_MODE, RXLEV_IDX_1, NO_PCH_INTERRUPT);

      AWAIT(mph_power_cnf_rxlevel('A'));

      /*  The following primitives are to ensure that after the parallel search
       *  ALR is still in IDLE State.
       */
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();
      neighbour_cell_in_dedicated_alr155();
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9813-------------------------------
 * Description: Parallel Search type with FAST_SEARCH_MODE
 *      The parallel type will be used only in case of the MS is in idle mode,
 * and the other task would not get disturbed while doing the power measurement.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9813()
{
  BEGIN_CASE ("parallel_search_cs_improvement_cases_alr9813")
  {
    TRAP_FOR_DEACT {
      /* To move the MS to IDLE mode to start parallel search */
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      /* Send a Parallel Search type with FAST_SEARCH_MODE */
      SEND  ( mph_power_req_variant(FAST_SEARCH_MODE, 'A')     );
      rxlevel_measurement_based_on_search_quadband(FAST_SEARCH_MODE, RXLEV_IDX_1, NO_PCH_INTERRUPT);

      AWAIT(mph_power_cnf_rxlevel('B'));

      /*  The following primitives are to ensure that after the parallel search
       *  ALR is still in IDLE State.
       */
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();
      neighbour_cell_in_dedicated_alr155();
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR9814-------------------------------
 * Description: Parallel Search type with BLACK_LIST_SEARCH_MODE
 *      The parallel type will be used only in case of the MS is in idle mode,
 * and the other task would not get disturbed while doing the power measurement.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9814()
{
  BEGIN_CASE ("parallel_search_cs_improvement_cases_alr9814")
  {
    TRAP_FOR_DEACT {
      /* To move the MS to IDLE mode to start parallel search */
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      rr_select_first_channel_alr013();

      /* Send a Parallel Search type with BLACK_LIST_SEARCH_MODE */
      SEND  ( mph_power_req_variant(BLACK_LIST_SEARCH_MODE, 'A')     );
      rxlevel_measurement_based_on_search_quadband(BLACK_LIST_SEARCH_MODE, RXLEV_IDX_1, NO_PCH_INTERRUPT);

      AWAIT(mph_power_cnf_variant('X'));

      /*  The following primitives are to ensure that after the parallel search
       *  ALR is still in IDLE State.
       */
      start_of_sending_channel_request_messages_gsm_900__alr055();
      immediate_assignment_for_the_mobile_station_alr056();
      neighbour_cell_in_dedicated_alr155();
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR9815-------------------------------
 * Description: General testcase scenarios to test the Cell Selection Improvement
 *-------------------------------------------------------------------------------*/

T_CASE ALR9815()
{
  BEGIN_CASE ("general_cell_selection_imporvement_cases_alr9815")
  {
    TRAP_FOR_DEACT {
      /* Scenario 1 - All carriers are lower than min rxlevel */
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'J')     );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_11, PCH_INTERRUPT);

      /* The POWER_CNF number of channels should be 0 and all the carriers should
       * added in to the inactive carriers.
       */
      AWAIT(mph_power_cnf_variant('B'));

      /* Scenario 2 - Whitelist reported by RR is not present in L1 Power measurement */
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'L')     );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);

      /* The POWER_CNF number of channels should be 0 and all the carriers should
       * added in to the inactive carriers.
       */

      AWAIT(mph_power_cnf_variant('C'));

      /* Scenario 3 - White list invalid */
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'J')     );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);

      AWAIT ( mph_power_cnf_variant('C') );

      /* Scenario 4 - White list valid and NO white list carriers */
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'X')     );
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);

      AWAIT (mph_power_cnf_variant('C'));
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9816-------------------------------
 * Description:
 *      ALR will forward the measurement report to RR only the Neighbor Cell state
 * is in "NC_IDLE". The following test case is to test the ALR, whether it's
 * forwarding the Measurement report to RR after the parallel search is over.
 *
 *      As per the existing code after the parallel search the MPH_BSIC_REQ has
 * to be sent to ALR, to enable the ALR to forward the MEASUREMENT_IND to RR.
 * But in case of Blacklist search the number of carriers will be always 0 and
 * there will not be any MPH_BSIC_REQ sent to ALR.  In ALR if the search type is
 * Blacklist search the NC state will be changed to NC_IDLE, before sending the
 * MPH_POWER_CNF to RR.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9816()
{
  BEGIN_CASE ("nc_idle_after_blacklist_search_cs_improvement_cases_alr9816")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      initiation_by_rr_4_channels_available_alr001();
      find_bcch_carrier_first_channel_alr003();
      read_bcch_data_alr007();
      measurement_reporting_configuration_alr030('A');

      meas_rep_bs_pa_mfrms_2();

      /* Blacklist Search will do the measurement only once */
      SEND  ( mph_power_req_variant(BLACK_LIST_SEARCH_MODE, 'A') );
      rxlevel_measurement_based_on_search_quadband(BLACK_LIST_SEARCH_MODE, RXLEV_IDX_1, NO_PCH_INTERRUPT);

      /* Blacklist Search will report the inactive carrier list to RR */
      AWAIT(mph_power_cnf_variant('X'));

      meas_rep_bs_pa_mfrms_2();

    } HANDLE_DEACT;
  }
}

/* CSImp Phase II Testcases Starts from here */

/*------------------------Following is for ALR9817-------------------------------
 * Description:
 *      First 40 Carriers in all bands should get priority to participarte in
 *      MPH_POWER_CNF.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9817()
{
  BEGIN_CASE ("all_bands_first_40_get_priority_alr9817")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      SEND  (mph_power_req_variant(FULL_SEARCH_MODE, 'K')     );

      /* L1 will report more than 80 carriers to ALR */
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_12, PCH_INTERRUPT);

      /* MPH_POER_CNF maximum carrier will be 80 */
      AWAIT(mph_power_cnf_overlap1());
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9818-------------------------------
 * Description:
 *      With some wrong carriers given by L1 (ALR will skip these carriers).
 *-------------------------------------------------------------------------------*/

T_CASE ALR9818()
{
  BEGIN_CASE ("some_wrong_carriers_reported_by_l1_alr9818")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'K')     );

      /* L1 will report more than exactly 160 carriers  to ALR */
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_13, PCH_INTERRUPT);

      /* MPH_POER_CNF maximum carrier will be 160 */
      AWAIT ( mph_power_cnf_overlap());
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR9819-------------------------------
 * Description:
 *     More than sixty carrier per band should get removed from the MPH_POWER_CNF
 *     and the first 40 carriers for all band should get priority.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9819()
{
  BEGIN_CASE ("more_than_60_carrier_per_band_should_get_removed_alr9819")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

      SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'K'));

      /* L1 will report more than exactly 160 carriers  to ALR */
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_14, PCH_INTERRUPT);

      /* MPH_POER_CNF maximum carrier will be 160 */
      AWAIT(mph_power_cnf_overlap());
    } HANDLE_DEACT;
  }
}

/*------------------------Following is for ALR9820-------------------------------
 * Description:
 *  ALL Whitelist carriers should get priority to participate in the MPH_POWER_CNF.
 *  After adding the whitelist carriers to POWER_CNF it should be removed from the
 *  power array during sorting
 *-------------------------------------------------------------------------------*/

T_CASE ALR9820()
{
  BEGIN_CASE ("whitelist_also_should_fall_in_first_40_carriers_alr9820")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

	  SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'C') );

      /* L1 will report more than exactly 160 carriers  to ALR */
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_14, PCH_INTERRUPT);

      /* MPH_POER_CNF maximum carrier will be 160 */
      AWAIT(mph_power_cnf_overlap());
    } HANDLE_DEACT;
  }
}


/*------------------------Following is for ALR9821-------------------------------
 * Description:
 *       Inactive carriers will be get removed before sorting.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9821()
{
  BEGIN_CASE ("inactive_carriers_get_removed_alr9821")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");
      filter_and_routings_gsm_900__alr000();

	  SEND  ( mph_power_req_variant(FULL_SEARCH_MODE, 'C') );

      /* L1 will report more than exactly 160 carriers  to ALR */
      rxlevel_measurement_based_on_search_quadband(FULL_SEARCH_MODE, RXLEV_IDX_15, PCH_INTERRUPT);

      /* MPH_POER_CNF maximum carrier will be 160 */
      AWAIT(mph_power_cnf_overlap());
    } HANDLE_DEACT;
  }
}

/* CSImp Phase II Testcases ends here */


/*------------------------Following is for ALR9822-------------------------------
 * Description:
 *         MS starts Non Parallel cell selection. 
 *         ALR performs a Network Sync.
 *         Meanwhile, Manual search is initiated by the user.
 *         To test the TIM_NW_SYNC_GUARD Timer Stop [CQ : 32448]
 *-------------------------------------------------------------------------------*/

T_CASE ALR9822()
{
  BEGIN_CASE ("TIM_NW_SYNC_GUARD_expiry_during_manual_search_alr9822")
  {
    TRAP_FOR_DEACT {
      COMMAND("TAP RESET");
      COMMAND("PL RESET");

      /* Non Parallel cell selection */
      initiation_by_rr_4_channels_available_alr001();

      SEND  ( mph_bsic_req(ARFCN_23_STD_900)  );
      /* ALR performs a Network Sync. */
      AWAIT ( mphc_network_sync_req(ARFCN_23, SM_WIDE_MODE) );

      /* Manual search is initiated by the user */
      SEND  ( mph_sync_req(CS_STOP_PLMN_SEARCH));

      AWAIT ( mphc_stop_network_sync_req_1() );
      SEND  ( mphc_stop_network_sync_con_1() );

      SEND  ( mph_power_req_1()           );
      AWAIT ( mphc_init_l1_req_1()        );
      SEND  ( mphc_init_l1_con_1()        );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mphc_rxlev_req()            );
      SEND  ( mphc_rxlev_ind(RXLEV_IDX_2) );
      AWAIT ( mph_power_cnf_2()           );

    } HANDLE_DEACT;
  }
}



/*------------------------Following is for ALR9823-------------------------------
 * Description:          
 *      Carriers which has the rxlevel lower than the LOWER_THRESHOLD
 *      should be reported in inactive_carrier_list of MPH_POWER_CNF
 *      This TC tests addotion of carriers to inactive_carrier_list
 *      in case of different LOWER_THRESHOLD values for different bands.
 *-------------------------------------------------------------------------------*/

T_CASE ALR9823()
{
  BEGIN_CASE ("lower_rxlevel_thr_for_different_bands_alr9823")
  {
    TRAP_FOR_DEACT {
      filter_and_routings_gsm_900__alr000();
      SEND  ( mph_power_req_variant(NORMAL_SEARCH_MODE, 'O')         );
      rxlevel_measurement_based_on_search_quadband(NORMAL_SEARCH_MODE, RXLEV_IDX_1, PCH_INTERRUPT);
      AWAIT ( mph_power_cnf_lrxt()       );

    } HANDLE_DEACT;
  }
}
