/* 
+----------------------------------------------------------------------------- 
|  Project :  GSM-PS (6147)
|  Modul   :  PSA_SMSP
+----------------------------------------------------------------------------- 
|  Copyright 2002 Texas Instruments Berlin, AG 
|                 All rights reserved. 
| 
|                 This file is confidential and a trade secret of Texas 
|                 Instruments Berlin, AG 
|                 The receipt of or possession of this file does not convey 
|                 any rights to reproduce or disclose its contents or to 
|                 manufacture, use, or sell anything it may describe, in 
|                 whole, or in part, without the specific written consent of 
|                 Texas Instruments Berlin, AG. 
+----------------------------------------------------------------------------- 
|  Purpose :  This module defines the processing functions for the
|             primitives send to the protocol stack adapter by 
|             short message service. 
+----------------------------------------------------------------------------- 
*/ 

#ifndef PSA_SMSP_C
#define PSA_SMSP_C
#endif

#include "aci_all.h"

/*==== INCLUDES ===================================================*/
#include "aci_cmh.h"
#include "ati_cmd.h"
#include "aci_cmd.h"


#ifdef FAX_AND_DATA
#include "aci_fd.h"
#endif    /* of #ifdef FAX_AND_DATA */

#include "aci.h"
#include "psa.h"
#include "psa_sms.h"
#include "cmh.h"
#include "cmh_sms.h"
#include "aoc.h"
#ifdef SIM_PERS
#include "cl_imei.h"
#include "aci_ext_pers.h"
#include "aci_slock.h"
#endif

#ifdef GPRS
#include "dti.h"
#include "dti_conn_mng.h"
#include "dti_cntrl_mng.h"
#include "gaci.h"
#include "gaci_cmh.h"
#include "psa_sm.h"

#include "cmh_sm.h"
#endif  /* GPRS */
/*==== CONSTANTS ==================================================*/


/*==== TYPES ======================================================*/


/*==== EXPORT =====================================================*/


/*==== VARIABLES ==================================================*/
#ifndef PIN_LEN
#define PIN_LEN (8)
#endif
#ifdef SIM_PERS_OTA
#define CTRL_KEY_LEN PIN_LEN+1
#define SMS_DATA_IDX 41
#define IMEI_MISMATCH "0xFFFFFFFF"
#endif

/*==== FUNCTIONS ==================================================*/


/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_message_ind   |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_MESSAGE_IND primitive send by SMS.
            this indicates an incoming SMS.

*/

GLOBAL const void psa_mnsms_message_ind 
                               ( T_MNSMS_MESSAGE_IND *mnsms_message_ind )
{
  /* (was psa_mnsms_alert_ind) */
  TRACE_FUNCTION ("psa_mnsms_message_ind()");


  if (smsShrdPrm.smsStat EQ SMS_STATE_INITIALISING)
  {
    TRACE_FUNCTION ("SMS_STATE_INITIALISING");
    cmhSMS_SMSInitState(mnsms_message_ind);
  }
  else
  {
    if (mnsms_message_ind->rec_num EQ SMS_RECORD_NOT_EXIST)
    {
#ifdef FF_CPHS      
      if ( !cmhSMS_voice_mail_ind( &(mnsms_message_ind->sms_sdu)) )
#endif /* FF_CPHS */      
      cmhSMS_SMSDeliver (mnsms_message_ind);
    }
    else
    {
      /*
       *  Incoming SMS may modify the Advice of Charge Parameters
       *  
       *  Read them again
       */
      aoc_sms ();

#ifdef FF_CPHS      
      cmhSMS_voice_mail_ind( &(mnsms_message_ind->sms_sdu));
#endif /* FF_CPHS */    
      cmhSMS_SMSMemory (mnsms_message_ind);
    }
  }

  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE (mnsms_message_ind);
}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_report_ind    |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_REPORT_IND primitive send by SMS.
            this indicates the state of the SMS Entity.

*/

GLOBAL const void psa_mnsms_report_ind 
                             ( T_MNSMS_REPORT_IND *mnsms_report_ind )
{

  TRACE_FUNCTION ("psa_mnsms_report_ind()");

  cmhSMS_Result(mnsms_report_ind);

  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE (mnsms_report_ind);

}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_status_ind    |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_STATUS_IND primitive send by SMS.
            this indicates an incoming SMS status message.

*/

GLOBAL const void psa_mnsms_status_ind ( T_MNSMS_STATUS_IND *mnsms_status_ind )
{
  TRACE_FUNCTION ("psa_mnsms_status_ind()");

  cmhSMS_SMSStatRpt (mnsms_status_ind);
  
  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE (mnsms_status_ind);
}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_delete_cnf    |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_DELETE_CNF primitive send by SMS.

*/

GLOBAL const void psa_mnsms_delete_cnf ( T_MNSMS_DELETE_CNF *mnsms_delete_cnf )
{
  TRACE_FUNCTION ("psa_mnsms_delete_cnf()");

  cmhSMS_SMSDelCnf (mnsms_delete_cnf);
  
  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE (mnsms_delete_cnf);
}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_read_cnf    |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_READ_CNF primitive send by SMS.

*/

GLOBAL const void psa_mnsms_read_cnf ( T_MNSMS_READ_CNF *mnsms_read_cnf )
{
  TRACE_FUNCTION ("psa_mnsms_read_cnf()");

  cmhSMS_SMRead (mnsms_read_cnf);
  
  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE (mnsms_read_cnf);
}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_store_cnf    |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_STORE_CNF primitive send by SMS.

*/

GLOBAL const void psa_mnsms_store_cnf ( T_MNSMS_STORE_CNF *mnsms_store_cnf )
{
  TRACE_FUNCTION ("psa_mnsms_store_cnf()");

  cmhSMS_SMSStoCnf(mnsms_store_cnf);
  
  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE (mnsms_store_cnf);
}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_submit_cnf    |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_SUBMIT_CNF primitive send by SMS.

*/

GLOBAL const void psa_mnsms_submit_cnf ( T_MNSMS_SUBMIT_CNF *mnsms_submit_cnf )
{
  TRACE_FUNCTION ("psa_mnsms_submit_cnf()");

  cmhSMS_SMSSbmCnf(mnsms_submit_cnf);

  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE (mnsms_submit_cnf);
}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_command_cnf   |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_COMMAND_CNF primitive send by SMS.

*/

GLOBAL const void psa_mnsms_command_cnf ( T_MNSMS_COMMAND_CNF *mnsms_command_cnf )
{
  TRACE_FUNCTION ("psa_mnsms_command_cnf()");

  cmhSMS_SMSCmdCnf(mnsms_command_cnf);
  
  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE (mnsms_command_cnf);
}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_error_ind     |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_ERROR_IND primitive send by SMS.

*/

GLOBAL const void psa_mnsms_error_ind ( T_MNSMS_ERROR_IND *mnsms_error_ind )
{
  TRACE_FUNCTION ("psa_mnsms_error_ind()");

  cmhSMS_SMSErrorInd(mnsms_error_ind);
  
  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE (mnsms_error_ind);
}


/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_resume_cnf    |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_RESUME_CNF primitive send by SMS.
            This confirms that the receiving of has been SMS is 
            activated.

*/

GLOBAL const void psa_mnsms_resume_cnf ( T_MNSMS_RESUME_CNF *mnsms_resume_cnf )
{
  TRACE_FUNCTION ("psa_mnsms_resume_cnf()");

  /* inform command handler */ 
  cmhSMS_SMSResumeCnf(mnsms_resume_cnf);
  
  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE(mnsms_resume_cnf);  
}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_query_cnf      |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_TEST_CNF primitive send by SMS.
            This confirms that the receiving of has been SMS is 
            activated.

*/

GLOBAL const void psa_mnsms_query_cnf ( T_MNSMS_QUERY_CNF *mnsms_query_cnf)
{
  TRACE_FUNCTION ("psa_mnsms_query_cnf()");

  /* inform command handler */ 
  cmhSMS_SMSQueryCnf( mnsms_query_cnf );
  
  /*
   *-------------------------------------------------------------------
   * free the primitive buffer
   *-------------------------------------------------------------------
   */  
  PFREE(mnsms_query_cnf);  
}


#if defined (GPRS) AND defined (DTI)
/*
+-------------------------------------------------------------------+
| PROJECT : GSM-PS (6147)         MODULE  : PSA_SMSP                |
|                                 ROUTINE : psa_mnsms_info_cnf      |
+-------------------------------------------------------------------+

  PURPOSE : processes the MNSMS_MO_SERV_CNF primitive send by SMS.
            this confirms the requested SMS message service.

*/

GLOBAL const void psa_mnsms_mo_serv_cnf 
                            ( T_MNSMS_MO_SERV_CNF *mnsms_mo_serv_cnf )
{

  TRACE_FUNCTION ("psa_mnsms_mo_serv_cnf()");

  /*
   *-------------------------------------------------------------------
   * update sms parameters and notify ACI
   *-------------------------------------------------------------------
   */ 

  cmhSM_sms_service_changed( mnsms_mo_serv_cnf->mo_sms_serv );

/*
 *-------------------------------------------------------------------
 * free the primitive buffer
 *-------------------------------------------------------------------
 */  
  PFREE (mnsms_mo_serv_cnf);

}
#endif  /* GPRS */


#ifdef SIM_PERS_OTA

/*
+------------------------------------------------------------------------------
|  Function    : psa_mnsms_OTA_decode_ind
+------------------------------------------------------------------------------
|  Description : processes the MNSMS_OTA_DECODE_IND primitive
|
|  Parameters  : mnsms_ota_decode_ind            - primitive
|                
|  Return      : void
|
+------------------------------------------------------------------------------
*/

GLOBAL const void psa_mnsms_OTA_decode_ind ( T_MNSMS_OTA_DECODE_IND *mnsms_ota_decode_ind )
{
  UBYTE lcktype;
  UBYTE ctrl_key[CTRL_KEY_LEN];
  UBYTE *imei;
  UBYTE imeiBufPtr[CL_IMEI_ISDID_SIZE];
  UBYTE ret_status = 0;
  UINT i = 0,ctr;

  TRACE_FUNCTION ("psa_mnsms_OTA_decode_ind()"); 
  
  imei = &mnsms_ota_decode_ind->tp_ud.data[73];

  /*As the IMEI is 15 digits long (see TS 23.003 [3]), 
  the sixteenth digit present here is ignored by the ME 
  during the de-personalization procedure and should be set to zero.*/

  *(imei+15) = 0;
  cl_get_imeisv(CL_IMEI_ISDID_SIZE,  imeiBufPtr, CL_IMEI_GET_SECURE_IMEI);

  while (*imei EQ imeiBufPtr[i] && i NEQ CL_IMEI_ISDID_SIZE )
  {
    imei++;
    i++;
  }
  if (i NEQ CL_IMEI_ISDID_SIZE)
    ret_status = 0xff;  //If IMEI mismatch occurs then status of all categories is set to IMEI Mismatch

  /*Extract the values from the SMS user data*/

  memset(ctrl_key,0,CTRL_KEY_LEN);
  
  for(lcktype=SIMLOCK_NETWORK,ctr=SMS_DATA_IDX; lcktype<=SIMLOCK_CORPORATE;lcktype++,ctr+=PIN_LEN)
  {
    strncpy ( (char *) ctrl_key, (char*) mnsms_ota_decode_ind->tp_ud.data[ctr], PIN_LEN);
    ctrl_key[PIN_LEN] = '\0';
    if (memcmp ((char *)ctrl_key,IMEI_MISMATCH,PIN_LEN))
      ret_status |= aci_ext_personalisation_get_status(lcktype);
    else    
    {
      if( aci_slock_unlock(lcktype, (char *) ctrl_key) EQ SIMLOCK_FAIL)
        ret_status |= aci_ext_personalisation_get_status(lcktype);
    }
    if(lcktype NEQ SIMLOCK_CORPORATE)
      ret_status  <<= 2;
  }

  imei = &mnsms_ota_decode_ind->tp_ud.data[ctr];
  {
    PALLOC (mnsms_OTA_decode_res, MNSMS_OTA_DECODE_RES);
    mnsms_OTA_decode_res->ota_status = ret_status;
    for (i=0;i<CL_IMEI_ISDID_SIZE;i++)
    {
      mnsms_OTA_decode_res->imei[i] = *imei;
      imei++;
    }
    PSEND (hCommSMS, mnsms_OTA_decode_res);
  }
  PFREE (mnsms_ota_decode_ind);
}
#endif //SIM_PERS_OTA
/*==== EOF =========================================================*/