/*
+------------------------------------------------------------------------------
|  File:       psa_pktiop.c
+------------------------------------------------------------------------------
|  Copyright 2002 Texas Instruments Berlin, AG 
|                 All rights reserved. 
| 
|                 This file is confidential and a trade secret of Texas 
|                 Instruments Berlin, AG 
|                 The receipt of or possession of this file does not convey 
|                 any rights to reproduce or disclose its contents or to 
|                 manufacture, use, or sell anything it may describe, in 
|                 whole, or in part, without the specific written consent of 
|                 Texas Instruments Berlin, AG.
+------------------------------------------------------------------------------
| Purpose:     This module defines the processing functions for the
|              primitives send to the protocol stack adapter by the
|              packet i/o managment (MNPKTIO).
+------------------------------------------------------------------------------
*/

#ifdef GPRS
#ifdef FF_PKTIO
#ifndef PSA_PKTIOP_C
#define PSA_PKTIOP_C
#endif

#include "aci_all.h"
/*==== INCLUDES ===================================================*/
#include "dti.h"      /* functionality of the dti library */
#include "aci_cmh.h"
#include "ati_cmd.h"
#include "aci_cmd.h"
#include "aci_lst.h"
#include "aci_mem.h"
#include "aci.h"
#include "psa.h"
#include "psa_pktio.h"

#include "cmh.h"
/*#include "dti_mng.h"*/
#include "dti_conn_mng.h"
#include "dti_cntrl_mng.h"
#include "cmh_pktio.h"


/*==== CONSTANTS ==================================================*/


/*==== TYPES ======================================================*/
GLOBAL T_ACI_LIST *pktio_dev_list=NULL;


/*==== EXPORT =====================================================*/


/*==== VARIABLES ==================================================*/


/*==== FUNCTIONS ==================================================*/
LOCAL BOOL find_pkt_dev_no ( UBYTE deviceNum, void * elem );
LOCAL BOOL mng_pkt_dev_list ( T_PKT_CONNECT_IND *pkt_connect_ind  );
/*
+----------------------------------------------------------------------+
| PROJECT : GPRS (8441)         MODULE  : PSA_PKTIOP                   |
| STATE   : finished            ROUTINE : find_pkt_dev_no              |
+----------------------------------------------------------------------+

  PURPOSE : search pktio device number in pktio device list
*/
LOCAL BOOL find_pkt_dev_no ( UBYTE deviceNum, void * elem )
{
  T_ACI_PKTIO *compared = (T_ACI_PKTIO *)elem;

  if (compared NEQ NULL)
    if (compared->device_no EQ deviceNum )
      return TRUE;
    return FALSE; 
}

/*
+----------------------------------------------------------------------+
| PROJECT : GPRS (8441)          MODULE  : PSA_PKTIOP                  |
| STATE   : finished             ROUTINE : mng_pkt_dev_list            |
+----------------------------------------------------------------------+

  PURPOSE : manage pktio device list
*/
LOCAL BOOL mng_pkt_dev_list ( T_PKT_CONNECT_IND *pkt_connect_ind  )
{
  T_ACI_PKTIO *  msg_ptr = NULL;

  if(pktio_dev_list EQ NULL)
  {/* there is no pktio device list */
    pktio_dev_list = new_list();
  }
  msg_ptr = find_element(pktio_dev_list, pkt_connect_ind->device_no,
                         find_pkt_dev_no);
  if(msg_ptr EQ NULL)
  {/* added new device */
    if((pkt_connect_ind->dio_dcb.convergence >= DTI_CPBLTY_CMD ) 
              AND 
       (pkt_connect_ind->dio_dcb.convergence <=(DTI_CPBLTY_CMD|DTI_CPBLTY_PKT|DTI_CPBLTY_SER)))
    {
      ACI_MALLOC(msg_ptr,sizeof(T_ACI_PKTIO));
      msg_ptr->device_no = pkt_connect_ind->device_no;
      memcpy(&msg_ptr->pktio_cap, &pkt_connect_ind->dio_dcb,sizeof(T_ACI_PKTIO_CAP));
      insert_list(pktio_dev_list,msg_ptr);
      return (TRUE);
    }
    else
    {/* neither CMD, SER nor PKT mode */
      return (FALSE);           
    }
  }
  else
  {/* new DIO capabilities for existing device */
    return (FALSE);
  }
}

/*
+----------------------------------------------------------------------+
| PROJECT : GPRS (8441)           MODULE  : PSA_PKTIOP                 |
| STATE   : finished              ROUTINE : psa_pkt_connect_ind        |
+----------------------------------------------------------------------+

  PURPOSE : processes the PKT_CONNECTION_IND primitive send by PKTIO.
                   add new device to pktio device list
                   register the DIO capabilities in ACI
*/
GLOBAL const void psa_pkt_connect_ind ( T_PKT_CONNECT_IND *pkt_connect_ind )
{
  T_ACI_PKTIO *  msg_ptr;

  TRACE_FUNCTION ("psa_pkt_connect_ind()");
  
  if(mng_pkt_dev_list(pkt_connect_ind) EQ TRUE)
  {
    msg_ptr = find_element(pktio_dev_list, pkt_connect_ind->device_no,
                                           find_pkt_dev_no);
    cmhPKT_Ind(msg_ptr);
  }
  else
  {/* second PKT_CONNECT_IND from same device or wrong convergence (mode) in 
        DIO capabilities */
    psaPKT_ConnectRej(pkt_connect_ind->device_no);
  }
  /* free the primitive buffer */
  PFREE (pkt_connect_ind);
}

/*
+----------------------------------------------------------------------+
| PROJECT : GPRS (8441)           MODULE  : PSA_PKTIOP                 |
| STATE   : finished              ROUTINE : psa_pkt_disconnect_ind     |
+----------------------------------------------------------------------+

  PURPOSE : processes the PKT_DISCONNECTION_IND primitive send by PKTIO.
            Removes the device from the pktio device list.
*/
GLOBAL const void psa_pkt_disconnect_ind ( T_PKT_DISCONNECT_IND *
                                           pkt_disconnect_ind )
{

  TRACE_FUNCTION ("psa_pkt_disconnect_ind()");    

  remove_element(pktio_dev_list, pkt_disconnect_ind->device_no, find_pkt_dev_no);

  /* free the primitive buffer */
  PFREE (pkt_disconnect_ind);
}

/*
+----------------------------------------------------------------------+
| PROJECT : GPRS (8441)           MODULE  : PSA_PKTIOP                 |
| STATE   : finished              ROUTINE : psa_pkt_dti_close_cnf      |
+----------------------------------------------------------------------+

  PURPOSE : processes the PKT_DTI_CLOSE_CNF primitive send by PKTIO.
     this confirms the dti disconnection requested by PKT_DTI_CLOSE_REQ 
*/
GLOBAL const void psa_pkt_dti_close_cnf ( T_PKT_DTI_CLOSE_CNF *
                                          pkt_dti_close_cnf )
{
  T_ACI_PKTIO * msg_ptr;
        
  TRACE_FUNCTION ("psa_pkt_dti_close_cnf()");    
  msg_ptr = find_element(pktio_dev_list, pkt_dti_close_cnf->device_no, 
                                          find_pkt_dev_no);
  /* if the device_no does not exist in the pktio_dev_list
       the primitive is ignored */
  if(msg_ptr NEQ NULL)
  {                    
    cmhPKT_Close(pkt_dti_close_cnf->device_no, DTI_CLOSE_CNF);          
  }
  /* free the primitive buffer */
  PFREE (pkt_dti_close_cnf);
}

/*
+----------------------------------------------------------------------+
| PROJECT : GPRS (8441)           MODULE  : PSA_PKTIOP                 |
| STATE   : finished              ROUTINE : psa_pkt_dti_close_ind      |
+----------------------------------------------------------------------+

  PURPOSE : processes the PKT_DTI_CLOSE_IND primitive send by PKTIO.
            this indicates the dti disconnection caused by the DIO driver
*/
GLOBAL const void psa_pkt_dti_close_ind ( T_PKT_DTI_CLOSE_IND *
                                          pkt_dti_close_ind )
{
  T_ACI_PKTIO * msg_ptr;
     
  TRACE_FUNCTION ("psa_pkt_dti_close_ind()");

  msg_ptr = find_element(pktio_dev_list, pkt_dti_close_ind->device_no, 
                                          find_pkt_dev_no);
  /* if the device_no does not exist in the pktio_dev_list
     the primitive is ignored */
  if(msg_ptr NEQ NULL)
  {            
    cmhPKT_Close(pkt_dti_close_ind->device_no, DTI_CLOSE_IND);
  } 
  /* free the primitive buffer */
  PFREE (pkt_dti_close_ind);
}

/*
+----------------------------------------------------------------------+
| PROJECT : GPRS (8441)          MODULE  : PSA_PKTIOP                  |
| STATE   : finished             ROUTINE : psa_pkt_dti_open_cnf        |
+----------------------------------------------------------------------+

  PURPOSE : processes the PKT_DTI_OPEN_CNF primitive send by PKTIO.
            this confirms the dti connection requested by PKT_DTI_OPEN_REQ
*/
GLOBAL const void psa_pkt_dti_open_cnf ( T_PKT_DTI_OPEN_CNF *
                                         pkt_dti_open_cnf )
{
  T_ACI_PKTIO * msg_ptr;
     
  TRACE_FUNCTION ("psa_pkt_dti_open_cnf()");

  msg_ptr = find_element(pktio_dev_list, pkt_dti_open_cnf->device_no, 
                                          find_pkt_dev_no);
  /* if the device_no does not exist in the pktio_dev_list
     the primitive is ignored */
  if(msg_ptr NEQ NULL)
  {
    switch(pkt_dti_open_cnf->cause)
    {
      case PKTCS_SUCCESS:
        cmhPKT_OpenCnf(pkt_dti_open_cnf->device_no,DTI_OK);
        break;
      case PKTCS_INVALID_PARAMS:
      case PKTCS_INVALID_PEER:
      case PKTCS_DISCONNECT:
      case PKTCS_INTERNAL_DRV_ERROR:
      default:
        cmhPKT_OpenCnf(pkt_dti_open_cnf->device_no,DTI_ERROR);
        break;
    }              
 }
 /* free the primitive buffer */
 PFREE (pkt_dti_open_cnf);
}

/*
+----------------------------------------------------------------------+
| PROJECT : GPRS (8441)           MODULE  : PSA_PKTIOP                 |
| STATE   : finished              ROUTINE : psa_pkt_modify_cnf         |
+----------------------------------------------------------------------+

  PURPOSE : processes the PKT_MODIFY_CNF primitive send by PKTIO.
            this confirms the modification fo channel parameters
            requested by PKT_MODIFY_REQ
*/
GLOBAL const void psa_pkt_modify_cnf ( T_PKT_MODIFY_CNF *pkt_modify_cnf )
{
  TRACE_FUNCTION ("psa_pkt_modify_cnf()");
 
  /* free the primitive buffer */
  PFREE (pkt_modify_cnf);
}
#endif /* FF_PKTIO */
#endif  /* GPRS */
/*==== EOF =========================================================*/
