/*
+-----------------------------------------------------------------------------
|  Project :  ACI
|  Modul   :  ati_src_riv
+-----------------------------------------------------------------------------
|  Copyright 2002 Texas Instruments Berlin, AG
|                 All rights reserved.
|
|                 This file is confidential and a trade secret of Texas
|                 Instruments Berlin, AG
|                 The receipt of or possession of this file does not convey
|                 any rights to reproduce or disclose its contents or to
|                 manufacture, use, or sell anything it may describe, in
|                 whole, or in part, without the specific written consent of
|                 Texas Instruments Berlin, AG.
+-----------------------------------------------------------------------------
|  Purpose :  This modul ...
+-----------------------------------------------------------------------------
*/
#ifndef ATI_SRC_RIV_C
#define ATI_SRC_RIV_C
#endif

#ifdef MFW
#define ENTITY_MFW
#else
#ifdef SMI
#define ENTITY_SMI
#else
#define ENTITY_ACI
#endif
#endif

#define ACI_MEMBER

#include "aci_all.h"
#include "pei.h"
#include "dti.h"      /* functionality of the dti library */
#include "aci.h"
#include "line_edit.h"
#include "aci_cmh.h"
#include "ati_cmd.h"
#include "aci_cmd.h"

#include "aci_lst.h"
#include "aci_mem.h"

#include "dti_conn_mng.h"
#include "dti_cntrl_mng.h"

#include "psa.h"
#include "cmh.h"

#include "psa_uart.h"
#include "ati_io.h"
#include "aci_io.h"

#include "psa_aaa.h"
#include "ati_src_riv.h"
#include "ati_src_uart.h"


EXTERN T_ACI_LIST *ati_src_list;

/*
+-----------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)              MODULE  : ati_src_riv           |
| STATE   : code                        ROUTINE : ati_src_riv_open_port |
+-----------------------------------------------------------------------+

  PURPOSE : - port_number = dev_no
*/
GLOBAL UBYTE  ati_src_riv_open_port  (UBYTE port_number, UBYTE sub_no, UBYTE capability )
{
  UBYTE                 src_id = DTI_SRC_ID_NOTPRESENT;
  UBYTE                 dti_id = DTI_DTI_ID_NOTPRESENT;
  T_DTI_ENTITY_ID       dev_id = DTI_ENTITY_AAA    ;   /* id(name) of device = RIV      */
  UBYTE                 dev_no = port_number;          /* instance of device            */
  T_ATI_SRC_PARAMS *src_params = NULL;
  T_DTI_CNTRL            *info = NULL;

  TRACE_FUNCTION("ati_src_riv_open_port()");

  ACI_MALLOC (info, sizeof (T_DTI_CNTRL));
  /*
   * if dti_cntrl_get_info_from_dev_id() returns with TRUE means
   * that the port_number is already in use, so deny this request
   */
  if (dti_cntrl_get_info_from_dev_id (dev_id, dev_no, sub_no, info) EQ FALSE)
  {
    if (BITFIELD_CHECK (capability, DTI_CPBLTY_CMD))
    {
      if (src_id NEQ 0)
      {
        T_LEDIT line;

        src_id = ati_init (ATI_SRC_TYPE_RIV, riv_src_result_cb, riv_src_line_state_cb);
        ati_switch_mode(src_id, ATI_CMD_MODE);

        src_params = find_element (ati_src_list, src_id, search_ati_src_id);

        /*
         * tell line edit that the line termination for Rivera is not default '\r'
         */
        line.S3     = 0x00; /* Rivera terminates cmd line with NULL                  */
        line.S4     = 0x0A; /* LF default value                                      */
        line.S5     = 0x08; /* BS default value                                      */
        line.smsEnd = 0x00; /* Rivera does not terminate a SMS with Ctrl-Z, but NULL */
        line.atE    = 0;
        ati_user_output_cfg[src_params->src_id].atE = 0;

        ledit_set_config (src_params->src_id, line);

        if (dti_cntrl_new_device (src_id, (UBYTE)dev_id, dev_no, sub_no, port_number, capability, DTI_DRIVER_ID_NOTPRESENT, DTI_DIO_ID_NOTPRESENT))
        {
          if (dti_cntrl_get_info_from_src_id (src_id, info))
          {
            dti_id = info->dti_id;
          }
          else
          {
            TRACE_EVENT_P2("[ERR] no info available for port_number/src_id: %d/%d", port_number, src_id);
          }
        }
        else
        {
          TRACE_EVENT_P1("[ERR] no dti_id available for port_number: %d", port_number);
        }
      }
      else
      {
        TRACE_EVENT_P1("[ERR] no srd_id available for port_number: %d", port_number);
      }
    }
    else
    {
      TRACE_EVENT_P1("[ERR] device with port_number: %d has no AT cmd capability", port_number);
    }
  }
  else
  {
    TRACE_EVENT_P1("[ERR] port_number: %d already in use", port_number);
  }

  if ((src_id NEQ DTI_SRC_ID_NOTPRESENT) AND (dti_id NEQ DTI_DTI_ID_NOTPRESENT))
  {
    TRACE_EVENT_P3("port = %d, src_id = %d, dti_id = %d", port_number, src_id, dti_id);
  }

  ACI_MFREE(info);

  return (dti_id);
}

/*
+-----------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)              MODULE  : ati_src_riv           |
| STATE   : code                        ROUTINE : ati_src_riv_close_port|
+-----------------------------------------------------------------------+

  PURPOSE : -
*/
GLOBAL void ati_src_riv_close_port  (UBYTE  port_number, UBYTE sub_no)
{
  UBYTE              src_id = DTI_SRC_ID_NOTPRESENT;
  T_DTI_ENTITY_ID    dev_id = DTI_ENTITY_AAA    ;   /* id(name) of device = RIV      */
  UBYTE              dev_no = port_number;          /* instance of device            */
  T_DTI_CNTRL         *info = NULL;

  TRACE_FUNCTION("ati_src_riv_close_port");

  ACI_MALLOC (info, sizeof (T_DTI_CNTRL));

  if (dti_cntrl_get_info_from_dev_id (dev_id, dev_no, sub_no, info))
  {
    src_id = info->src_id;

    ati_finit (src_id);

    dti_cntrl_erase_entry (info->dti_id);

    TRACE_EVENT_P1("Source nb %d deleted", src_id);
  }
  else
  {
    TRACE_EVENT_P1("there is no info to port_number %d", port_number);
  }

  ACI_MFREE(info);
}

/*
+----------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)              MODULE  : ati_src_riv          |
| STATE   : code                        ROUTINE : ati_src_riv_proc_cmd |
+----------------------------------------------------------------------+

  PURPOSE : -
*/
GLOBAL BOOL  ati_src_riv_proc_cmd  (T_AAA_CMD_REQ *cmd)
{
  UBYTE                 src_id = DTI_SRC_ID_NOTPRESENT;
  T_DTI_ENTITY_ID       dev_id = DTI_ENTITY_AAA;       /* id(name) of device = RIV      */
  UBYTE                 dev_no = cmd->port_number;     /* instance of device            */
  UBYTE                 sub_no = cmd->sub_no;          /* multiplexed channel of device */
  T_DTI_CNTRL           info;
  T_ATI_SRC_PARAMS *src_params = NULL;

  //ACI_MALLOC (info, sizeof (T_DTI_CNTRL));

  if (dti_cntrl_get_info_from_dev_id (dev_id, dev_no, sub_no, &info))
  {
    src_id = info.src_id;
    src_params = find_element (ati_src_list, src_id, search_ati_src_id);

    //ACI_MFREE (info);
    TRACE_EVENT_P1("AT command: %s", cmd->cmd_seq);

    if (src_params->text_mode EQ CMD_MODE)
    {
      TRACE_FUNCTION ("ati_src_riv_proc_cmd(): CMD MODE");
    }
    else
    {
      TRACE_FUNCTION ("ati_src_riv_proc_cmd(): TEXT MODE");
    }
#ifdef _SIMULATION_
    return (ati_execute (src_id, cmd->cmd_seq, (USHORT)(cmd->cmd_len+1)));
#else
    return (ati_execute (src_id, cmd->cmd_seq, cmd->cmd_len));
#endif
  }
  else
  {
    TRACE_EVENT_P1("ati_src_riv_proc_cmd(): there is no info to port_number %d", cmd->port_number);
  }

  //ACI_MFREE (info);
  return (FALSE);
}

/*
+-------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)              MODULE  : ati_src_riv       |
| STATE   : code                        ROUTINE : riv_src_result_cb |
+-------------------------------------------------------------------+

  PURPOSE : -
*/
GLOBAL void  riv_src_result_cb      (UBYTE              src_id,
                                   T_ATI_OUTPUT_TYPE    output_type,
                                   UBYTE             *output,
                                   USHORT               output_len)
{
  UBYTE port_number = 0xFF;
  UBYTE sub_no      = 0xFF;
  T_DTI_CNTRL *info = NULL;

  TRACE_FUNCTION("riv_src_result_cb()");

  ACI_MALLOC (info, sizeof (T_DTI_CNTRL));

  if (dti_cntrl_get_info_from_src_id (src_id, info))
  {
    port_number = info->dev_no;
    sub_no      = info->sub_no;
    ACI_MFREE(info);

    if (IS_INDICATION_OUTPUT(output_type))
    {
      psaAAA_ures_ind (port_number, sub_no, output_len, output);
      return;
    }

    if (IS_CONFIRM_OUTPUT(output_type))
    {
      psaAAA_cmd_cnf    (port_number, sub_no, output_len, output);
      return;
    }

    if (IS_NORMAL_OUTPUT(output_type))
    {
      psaAAA_pres_ind (port_number, sub_no, output_len, output);
      return;
    }
  }
  else
  {
    ACI_MFREE(info);
    TRACE_EVENT("riv_src_result_cb(): did not get port_number or unknown output_type !");
  }
}

/*
+-----------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)              MODULE  : ati_src_riv           |
| STATE   : code                        ROUTINE : riv_src_line_state_cb |
+-----------------------------------------------------------------------+

  PURPOSE : - this is just a dummy function to match
              the function parameters of ati_init()
*/
GLOBAL void  riv_src_line_state_cb (UBYTE                 src_id,
                                    T_ATI_LINE_STATE_TYPE line_state_type,
                                    ULONG                 line_state_param)
{
}


/*
+-----------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)              MODULE  : ati_src_riv           |
| STATE   : code                        ROUTINE : ati_riv_src_dti_rsp   |
+-----------------------------------------------------------------------+

  PURPOSE : -
*/
GLOBAL void ati_src_riv_dti_rsp (ULONG link_id)
{
#ifndef FF_ESIM
  dti_cntrl_connect_after_aaa_dti_rsp(EXTRACT_DTI_ID(link_id)) ;
  dti_cntrl_entity_connected (link_id, DTI_ENTITY_AAA, DTI_OK);
#endif
}

/*
+-----------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)         MODULE  : ati_src_riv                |
| STATE   : code                   ROUTINE : ati_riv_src_disconnect_rsp |
+-----------------------------------------------------------------------+

  PURPOSE : -
*/
GLOBAL void ati_src_riv_disconnect_rsp (ULONG link_id)
{
/*   T_DTI_CNTRL  dti_cntr; */

/*   if(dti_cntrl_get_info_from_dti_id(EXTRACT_DTI_ID(link_id), &dti_cntr)) */
/*   { */
/*     R_AT( RAT_OK, dti_cntr.src_id ) ( AT_CMD_CGACT ); */
/*   } */

  dti_cntrl_entity_disconnected (link_id, DTI_ENTITY_AAA);
}

