/* 
+----------------------------------------------------------------------------- 
|  Project :  BLUETOOTH
|  Modul   :  ATI_SRC_BT_C
+----------------------------------------------------------------------------- 
|  Copyright 2002 Texas Instruments Berlin, AG 
|                 All rights reserved. 
| 
|                 This file is confidential and a trade secret of Texas 
|                 Instruments Berlin, AG 
|                 The receipt of or possession of this file does not convey 
|                 any rights to reproduce or disclose its contents or to 
|                 manufacture, use, or sell anything it may describe, in 
|                 whole, or in part, without the specific written consent of 
|                 Texas Instruments Berlin, AG. 
+----------------------------------------------------------------------------- 
|  Purpose :  .
+----------------------------------------------------------------------------- 
*/ 

#ifndef ATI_SRC_BT_C
#define ATI_SRC_BT_C
#endif

#include "aci_all.h"

/*==== INCLUDES ===================================================*/
#include "aci_cmh.h"
#include "ati_cmd.h"
#include "aci_cmd.h"

#include "aci_lst.h"
#include "dti.h"
#include "dti_conn_mng.h"
#include "psa_uart.h"
#include "ati_io.h"
#include "cmh_uart.h"
#include "aci_io.h"
#include "aci_mem.h"

#if defined (BT_ADAPTER)
#include "bti_aci.h"
#include "bti_int.h"
#endif


#include "ati_src_bt.h"

/*==== CONSTANTS ==================================================*/
/*#ifdef GPRS
#define MAX_LARGE_BUFFER_LEN (600)
#define MAX_BUFFER_COUNT (15)
#else
#define MAX_LARGE_BUFFER_LEN (400)
#ifdef FAX_AND_DATA
#define MAX_BUFFER_COUNT (27)
#else
#define MAX_BUFFER_COUNT (7)
#endif
#endif
*/
/*==== TYPES ======================================================*/

/*==== EXPORT =====================================================*/

/*==== VARIABLES ==================================================*/

/*==== FUNCTIONS ==================================================*/

T_BTI_AT_RING_TYPE  get_bti_ring_type(T_ACI_CRING_TYP t);
T_BTI_AT_DCE_SIGNAL get_bti_ring_state(T_IO_DCD_STAT s);
T_BTI_AT_DCE_SIGNAL get_bti_dcd_state(T_IO_DCD_STAT s);

/*
+--------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445              MODULE  : ACI_URT            |
| STATE   : code                        ROUTINE : io_openPort        |
+--------------------------------------------------------------------+

  PURPOSE: opens an external port

*/
GLOBAL T_BTI_ACK io_openPort (T_BTI_PORT_NB port)
{
  TRACE_EVENT("io_openPort()");

  if ((port >= MAX_PORTS) || (port < 0)) /* if requested port number is higher than available ports (including 0) -> fail*/
    return BTI_NAK;

  if (bti_port_table[port].aci_src_id != 0) /* port already used */
    return BTI_NAK;

  bti_port_table[port].aci_src_id = ati_init(ATI_SRC_TYPE_BLUETOOTH, bt_src_result_cb, bt_src_line_state_cb);

  TRACE_EVENT_P2("port %d aci_src_id %d", port, bti_port_table[port].aci_src_id);

  if (bti_port_table[port].aci_src_id == 0) /* failed to get aci src id */
    return BTI_NAK;

  dti_mng_new (bti_port_table[port].aci_src_id, bti_port_table[port].aci_src_id,
                bt_src_dti_cb, TRUE, DTI_MNG_DEVICE_TYPE_BT, DTI_MNG_DATA_TYPE_CMD);

  bti_port_table[port].dti_data_ind_BTI = NULL;
  bti_port_table[port].data_offset = 0;
  bti_port_table[port].data_left = 0;
  bti_port_table[port].bti_gsm_port_info.optimal_gsm_max_packet_size = 0;
  bti_port_table[port].bti_bt_port_info.optimal_bt_max_packet_size = 0;
  bti_port_table[port].atqueue = NULL;
  bti_port_table[port].large_at_output = FALSE;
  bti_port_table[port].at_flow_on = TRUE;

  return BTI_ACK;
}

/*
+--------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445              MODULE  : ACI_URT            |
| STATE   : code                        ROUTINE : io_closePort       |
+--------------------------------------------------------------------+

  PURPOSE: closes an external port

*/
GLOBAL void io_closePort (T_BTI_PORT_NB port)
{
  TRACE_EVENT("io_closePort()");
  ati_finit(bti_port_table[port].aci_src_id);

  bti_port_table[port].aci_src_id = 0;
  bti_port_table[port].dti_data_ind_BTI = NULL;
  bti_port_table[port].data_offset = 0;
  bti_port_table[port].data_left = 0;
  bti_port_table[port].bti_gsm_port_info.optimal_gsm_max_packet_size = 0;
  bti_port_table[port].bti_bt_port_info.optimal_bt_max_packet_size = 0;
  bti_port_table[port].atqueue = NULL;
  bti_port_table[port].large_at_output = FALSE;
  bti_port_table[port].at_flow_on = TRUE;
}



/*
+--------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445              MODULE  : ACI_URT            |
| STATE   : code                        ROUTINE : get_bti_ring_type  |
+--------------------------------------------------------------------+

  PURPOSE: converting ACI type to BTI type
*/
T_BTI_AT_RING_TYPE get_bti_ring_type(T_ACI_CRING_TYP t)
{
  switch (t)
  {
    case CRING_TYP_NotPresent:
      return BTI_AT_RING_NotPresent;
    case CRING_TYP_Async:
      return BTI_AT_RING_Async;
    case CRING_TYP_Sync:
      return BTI_AT_RING_Sync;
    case CRING_TYP_RelAsync:
      return BTI_AT_RING_RelAsync;
    case CRING_TYP_RelSync:
      return BTI_AT_RING_RelSync;
    case CRING_TYP_Fax:
      return BTI_AT_RING_Fax;
    case CRING_TYP_Voice:
      return BTI_AT_RING_Voice;
    case CRING_TYP_AuxVoice:
      return BTI_AT_RING_AuxVoice;
#ifdef GPRS
    case CRING_TYP_GPRS:
      return BTI_AT_RING_GPRS;
#endif  /* GPRS */

 }
  return BTI_AT_RING_NotPresent;
}

/*
+--------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445              MODULE  : ACI_URT            |
| STATE   : code                        ROUTINE : get_bti_ring_state |
+--------------------------------------------------------------------+

  PURPOSE: converting ACI type to BTI type
*/
T_BTI_AT_DCE_SIGNAL get_bti_ring_state(T_IO_RING_STAT s)
{
   switch (s)
   {
     case IO_RING_OFF:
       TRACE_EVENT ("V.24 RING OFF");
       return BTI_AT_RING_OFF;

     case IO_RING_ON:
       TRACE_EVENT ("V.24 RING ON");
       return BTI_AT_RING_ON;
   }
}

/*
+--------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445              MODULE  : ACI_URT            |
| STATE   : code                        ROUTINE : get_bti_dcd_state  |
+--------------------------------------------------------------------+

  PURPOSE: converting ACI type to BTI type
*/
T_BTI_AT_DCE_SIGNAL get_bti_dcd_state(T_IO_DCD_STAT s)
{
   switch (s)
   {
     case IO_DCD_OFF:
       TRACE_EVENT ("V.24 DCD OFF");
       return BTI_AT_DCD_OFF;

     case IO_DCD_ON:
       TRACE_EVENT ("V.24 DCD ON");
       return BTI_AT_DCD_ON;
   }
}

/*
+--------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445              MODULE  : ACI_URT            |
| STATE   : code                        ROUTINE : ati_src_bt_proc_cmd|
+--------------------------------------------------------------------+

  PURPOSE:

*/

GLOBAL BOOL ati_src_bt_proc_cmd (T_ACI_CMD_REQ_BT *cmd)
{
  TRACE_FUNCTION ("ati_src_bt_proc_cmd ()");
  return (ati_execute (cmd->src_id, cmd->cmd_seq, cmd->cmd_len));
}

/*
+-------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445    MODULE  : PSA_UART                    |
|                            ROUTINE : bt_src_connect               |
+-------------------------------------------------------------------+

  PURPOSE :

*/
GLOBAL T_ACI_RETURN bt_src_connect (UBYTE srcId, T_DTI_ENTITY_ID unit) 
{
  T_BTI_PORT_NB port_nb;

  TRACE_EVENT("bt_src_connect()");

  port_nb = get_port_by_srcID(srcId);

  TRACE_EVENT_P3("port %d src_id %d unit %d", port_nb, bti_port_table[port_nb].aci_src_id, unit);

  dti_entity_connected(srcId, CONNECTED, UNIT_BLUETOOTH);

  return AT_CMPL;
}

/*
+-------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445    MODULE  : PSA_UART                    |
|                            ROUTINE : bt_src_disconnect            |
+-------------------------------------------------------------------+

  PURPOSE :

*/
GLOBAL T_ACI_RETURN bt_src_disconnect (UBYTE srcId)
{
  T_BTI_PORT_NB port_nb;

  port_nb = get_port_by_srcID(srcId);

  return AT_CMPL;
}

/*
+-------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445    MODULE  : PSA_UART                    |
|                            ROUTINE : bt_enqueue_command           |
+-------------------------------------------------------------------+

  PURPOSE :

*/
GLOBAL void bt_enqueue_command(T_BTI_PORT_NB port_nb, UBYTE *output, USHORT output_len, UBYTE type)
{
  T_ATI_BT_OUTPUT *cur_cmd;
  T_ATI_BT_OUTPUT *queue_end;
  int len;

  TRACE_EVENT_P2("bt_enqueue_command() %d %s",type, output);

  len = output_len + sizeof(T_ATI_BT_OUTPUT);
  ACI_MALLOC(cur_cmd, len);

  cur_cmd->output_len = output_len;
  cur_cmd->output_type = type;
  cur_cmd->next = NULL;
  memcpy (cur_cmd->output, output, output_len);
  cur_cmd->output[output_len] = 0;

  if (bti_port_table[port_nb].atqueue == NULL) /* Queue is empty */
  {
    bti_port_table[port_nb].atqueue = cur_cmd;
  }
  else /* There are already pending commands */
  {
    queue_end = bti_port_table[port_nb].atqueue;
    len = 1; /* just for debugging */

#ifdef _SIMULATION_
//    TRACE_EVENT_P1("content: %s", queue_end->output); /* just for debugging */
#endif

    while (queue_end->next != NULL) /* search end of queue */
    {
      queue_end = queue_end->next;
      len++; /* just for debugging */
#ifdef _SIMULATION_
//      TRACE_EVENT_P1("content: %s", queue_end->output); /* just for debugging */
#endif
    }
    TRACE_EVENT_P1("elements in queue: %d", len); /* just for debugging */
    queue_end->next = cur_cmd;
  }
  return;
}


/*
+-------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445    MODULE  : PSA_UART                    |
|                            ROUTINE : bt_set_output_type           |
+-------------------------------------------------------------------+

  PURPOSE :

*/
GLOBAL UBYTE bt_set_output_type(T_ATI_OUTPUT_TYPE t)
{
  if (IS_CONFIRM_OUTPUT(t))
    return ATI_BT_CONFIRM_COMMAND;
  if (IS_NORMAL_OUTPUT(t))
    return ATI_BT_NORMAL_COMMAND;
  if (IS_INDICATION_OUTPUT(t))
    return ATI_BT_INDICATION_COMMAND;

  TRACE_EVENT("ATI_BT unexpected output type");
}

/*==== CALLBACKS ==================================================*/

/*
+--------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445              MODULE  : ACI_URT            |
| STATE   : code                        ROUTINE : bt_src_result_cb   |
+--------------------------------------------------------------------+

  PURPOSE:

*/
GLOBAL void bt_src_result_cb (UBYTE             src_id,
                                T_ATI_OUTPUT_TYPE output_type,
                                UBYTE             *output,
                                USHORT            output_len)
{
  ULONG port_nb;

  port_nb = get_port_by_srcID(src_id);

  TRACE_EVENT_P1("bt_src_result_cb() pt: %d", port_nb);

/* just for test purposes */
  TRACE_EVENT((char*)output);

  /* first check, if we are allowed to send */
  if (bti_port_table[port_nb].at_flow_on == FALSE)
  {
    TRACE_EVENT("bt_src_result_cb() commands pending");
    bt_enqueue_command(port_nb, output, output_len, bt_set_output_type(output_type));
    return;
  }

  /* kk 010808
   * according to RIE NORMAL_OUTPUT and CONFIRM_OUTPUT is mutual exclusive
   * but this is not the case in the current implementation
   * addionally INDICATION_OUTPUT is never set
   * temporary solution is to check for CONFIRM first then exit if applicable
   * else check for NORMAL_OUTPUT and assume that this is also INDICATION
   * NEEDS FIX from RIE
   */
  if (IS_CONFIRM_OUTPUT(output_type))
  {
    TRACE_EVENT("bt_src_result_cb() confirm sent");
    btiaci_at_cmd_res(port_nb, (char *)output);
    return;
  }

  /* kk 010808
   * according to RIE NORMAL_OUTPUT and CONFIRM_OUTPUT is mutual exclusive
   * but this is not the case in the current implementation
   * addionally INDICATION_OUTPUT is never set
   * temporary solution is to check for CONFIRM first then exit if applicable
   * else check for NORMAL_OUTPUT and assume that this is also INDICATION
   * NEEDS FIX from RIE
   */
  if (IS_NORMAL_OUTPUT(output_type) OR IS_INDICATION_OUTPUT(output_type) OR IS_ECHO_OUTPUT(output_type))
  {
    TRACE_EVENT("bt_src_result_cb() indication/normal sent");
    bti_port_table[port_nb].at_flow_on = FALSE;
    btiaci_at_cmd_req(port_nb, (char *)output);
    return;
  }

  TRACE_EVENT("bt_src_result_cb() SOMETHING IS WRONG HERE");
}

/*
+--------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445              MODULE  : ACI_URT            |
| STATE   : code                        ROUTINE : bt_src_line_state_cb
+--------------------------------------------------------------------+

  PURPOSE:

*/
GLOBAL void bt_src_line_state_cb (UBYTE                 src_id,
                                    T_ATI_LINE_STATE_TYPE line_state_type,
                                    ULONG                  line_state_param)
{

  T_BTI_AT_RING_TYPE bt1, bt2;
  T_BTI_AT_DCE_SIGNAL signal;
  T_BTI_PORT_NB port_nb;

  T_IO_RING_PARAMS *ring;
  T_IO_DCD_STAT *dcd;

  port_nb = get_port_by_srcID(src_id);

  TRACE_EVENT_P1("bt_src_line_state_cb() pt: %d", port_nb);

  switch (line_state_type)
  {
    case ATI_LINE_STATE_RNG:
      TRACE_EVENT("line state RING");

      ring = (T_IO_RING_PARAMS *)line_state_param;

      bt1 = get_bti_ring_type(ring->b_cap_1);
      bt2 = get_bti_ring_type(ring->b_cap_2);
      signal = get_bti_ring_state(ring->ring_stat);

      btiaci_at_signal_change_req(port_nb, signal, bt1, bt2);
      break;

    case ATI_LINE_STATE_DCD:
      TRACE_EVENT("line state DCD");

      dcd = (T_IO_DCD_STAT *)line_state_param;
      signal = get_bti_dcd_state(*dcd);
      btiaci_at_signal_change_req(port_nb, signal, BTI_AT_RING_NotPresent, BTI_AT_RING_NotPresent);
      break;

    case ATI_LINE_STATE_OUTPUT_TYPE:
      /*src_infos->run_cmd = TRUE;*/
      /*src_infos->first_output = TRUE;*/

      if( line_state_param EQ ATI_OUTPUT_TYPE_LARGE )
      {
        TRACE_EVENT("large output following");
        bti_port_table[port_nb].large_at_output = TRUE;
      }
      else
      {
        TRACE_EVENT("normal output following");
        bti_port_table[port_nb].large_at_output = FALSE;
      }
      break;

    default:
      TRACE_EVENT("line state NO TYPE");
      break;
  }
}



/*
+-------------------------------------------------------------------+
| PROJECT : BLUETOOTH 8445    MODULE  : PSA_UART                    |
|                            ROUTINE : bt_src_dti_cb                |
+-------------------------------------------------------------------+

  PURPOSE : callback for dti manager.

*/

GLOBAL BOOL bt_src_dti_cb( UBYTE srcId, T_DTI_MNG_CB_TYPE result_type )
{
  TRACE_FUNCTION("bt_src_dti_cb()");

  switch(result_type)
  {
    case(CB_TYPE_CONNECTED):
      TRACE_EVENT_P1("srcID %d connected", srcId);
      break;

    case(CB_TYPE_TMP_RECONNECTED):
      TRACE_EVENT_P1("srcID %d reconnected", srcId);
      break;

    case(CB_TYPE_NEW_CONNECTED):
      TRACE_EVENT_P1("srcID %d new connected", srcId);
      break;

    case(CB_TYPE_TMP_DISCONNECTING):
      TRACE_EVENT_P1("srcID %d disconnecting", srcId);
      break;

    case(CB_TYPE_TMP_DISCONNECTED):
      TRACE_EVENT_P1("srcID %d disconnected", srcId);
      break;

    case(CB_TYPE_ERROR):
      TRACE_EVENT_P1("srcID %d ERROR", srcId);

    /* connection failed */
//kk        srcInfos->LineState = LINE_TMP_DISCONNECTED;
//k        dti_mng_owner_connect(srcId, UNIT_UART);
      break;

    default:
      break;
  }

  return TRUE;
}

