/* 
+----------------------------------------------------------------------------- 
|  Project :  GSM-F&D (8411)
|  Modul   :  ACI_BAT
+-----------------------------------------------------------------------------
|  Copyright 2002 Texas Instruments Berlin, AG
|                 All rights reserved. 
|
|                 This file is confidential and a trade secret of Texas 
|                 Instruments Berlin, AG 
|                 The receipt of or possession of this file does not convey 
|                 any rights to reproduce or disclose its contents or to 
|                 manufacture, use, or sell anything it may describe, in 
|                 whole, or in part, without the specific written consent of 
|                 Texas Instruments Berlin, AG. 
+----------------------------------------------------------------------------- 
|  Purpose :  This BAT wrapper modul is ...
| 
+----------------------------------------------------------------------------- 
*/ 

#ifdef GPRS
 
#include "aci_all.h"     /* includes prim.h, which includes p_bat.h */ 
#include "aci_cmh.h"     /* prototypes of sAT_,qAT_,tAT_    */ 
#include "aci_bat_cmh.h" /* prototypes of sBAT_,qBAT_,tBAT_ */ 
#include "ati_cmd.h"
#include "aci_cmd.h"
#include "gaci_cmh.h"
#include "aci_bat.h"
#include "aci_mem.h"

extern BOOL cmhSM_isContextActive( void );
extern void ipaddr_to_string(ULONG ipaddr, char *ipaddr_string);
extern T_ATI_RSLT setatPercentCGPCO_HEX (USHORT cid,
                                    UBYTE *pco_hex_string, USHORT len);
extern T_ATI_RSLT setatPercentCGPCO_ASCII (UBYTE srcId, USHORT cid,
                                      UBYTE *pco_ascii_string, USHORT len);


#define MAGIC_EMPTY_APN "\x0ff"
/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGDCONT     |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGDCONT     (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_BAT_pdp_cid  cid = BAT_PDP_CID_NOT_PRESENT ;
  T_GPRS_CONT_REC inputCtxt;

  TRACE_FUNCTION ("sBAT_PlusCGDCONT()");

  memset(inputCtxt.apn,0,sizeof(T_GPRS_CONT_REC));
  
  cid = cmd->params.ptr_set_plus_cgdcont->cid;
  
  if (cid EQ BAT_PDP_CID_NOT_PRESENT )
  {
    cid = GPRS_CID_OMITTED ;
  }

  if(cmd->params.ptr_set_plus_cgdcont->pdp_type EQ BAT_PDP_TYPE_IPV4)
  {
    strcpy(inputCtxt.pdp_type, "ip"); 
  }
    
  if(cmd->params.ptr_set_plus_cgdcont->v_apn)
  {
    memcpy(inputCtxt.apn,cmd->params.ptr_set_plus_cgdcont->apn,cmd->params.ptr_set_plus_cgdcont->c_apn);
  }
  
  if(cmd->params.ptr_set_plus_cgdcont->v_pdp_addr)
  {
    memcpy(inputCtxt.pdp_addr,cmd->params.ptr_set_plus_cgdcont->pdp_addr,cmd->params.ptr_set_plus_cgdcont->c_pdp_addr);
  }
  
  inputCtxt.d_comp = cmd->params.ptr_set_plus_cgdcont->d_comp;
  inputCtxt.h_comp = cmd->params.ptr_set_plus_cgdcont->h_comp;
  
  ret = sAT_PlusCGDCONT (src_infos_psi->srcId, (SHORT)cid ,&inputCtxt);
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PlusCGDCONT     |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PlusCGDCONT     (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_BAT_cmd_response resp;
  T_BAT_res_que_plus_cgdcont que_cgdcont_buffer;
  T_GPRS_CONT_REC defCtxts[MAX_CID_PLUS_EINS];
  SHORT cid_array[MAX_CID_PLUS_EINS];
  int i ;
  
  TRACE_FUNCTION ("qBAT_PlusCGDCONT()");
  resp.ctrl_response = BAT_RES_QUE_PLUS_CGDCONT;
  resp.response.ptr_que_plus_cgdcont = &que_cgdcont_buffer;
  
  ret = qAT_PlusCGDCONT (src_infos_psi->srcId, &defCtxts, cid_array);

  if (ret EQ AT_CMPL)
  {
    for ( i = 0; i < MAX_CID && cid_array[i] NEQ INVALID_CID; i++ )
    {
      resp.response.ptr_que_plus_cgdcont->cid = cid_array[i];
      if(!(utl_strcasecmp((const char *)defCtxts[i].pdp_type, (const char *)"ip")))
        resp.response.ptr_que_plus_cgdcont->pdp_type = BAT_PDP_TYPE_IPV4;
      else
        resp.response.ptr_que_plus_cgdcont->pdp_type = BAT_PDP_TYPE_NOT_PRESENT;

      resp.response.ptr_que_plus_cgdcont->c_apn = strlen(defCtxts[i].apn);
      memcpy(resp.response.ptr_que_plus_cgdcont->apn,defCtxts[i].apn,strlen(defCtxts[i].apn));
      resp.response.ptr_que_plus_cgdcont->c_pdp_addr = strlen(defCtxts[i].pdp_addr);
      memcpy(resp.response.ptr_que_plus_cgdcont->pdp_addr,defCtxts[i].pdp_addr,strlen(defCtxts[i].pdp_addr));
      resp.response.ptr_que_plus_cgdcont->d_comp = defCtxts[i].d_comp;
      resp.response.ptr_que_plus_cgdcont->h_comp = defCtxts[i].h_comp;
      aci_bat_send(src_infos_psi,&resp);
    }
    return ACI_BAT_CMPL;
  }

  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGQREQ      |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGQREQ      (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_QOS input_qos = { ( UBYTE ) GPRS_QOS_OMITTED };
  T_BAT_pdp_cid  cid = BAT_PDP_CID_NOT_PRESENT ;
  
  TRACE_FUNCTION ("sBAT_PlusCGQREQ()");
  
  cid = cmd->params.ptr_set_plus_cgqreq->cid;
  if (cid EQ BAT_PDP_CID_NOT_PRESENT )
  {
    cid = GPRS_CID_OMITTED ;
  }
  input_qos.preced = cmd->params.ptr_set_plus_cgqreq->precedence;
  input_qos.delay = cmd->params.ptr_set_plus_cgqreq->delay;
  input_qos.relclass = cmd->params.ptr_set_plus_cgqreq->reliability;
  input_qos.peak  = cmd->params.ptr_set_plus_cgqreq->peak;
  input_qos.mean = cmd->params.ptr_set_plus_cgqreq->mean;
  
  if (input_qos.preced EQ BAT_PRECEDENCE_NOT_PRESENT)
  {
    input_qos.preced = GPRS_QOS_OMITTED ;
  }
  if (input_qos.delay EQ BAT_DELAY_NOT_PRESENT)
  {
    input_qos.delay = GPRS_QOS_OMITTED ;
  }
  if (input_qos.relclass EQ BAT_RELIABILITY_NOT_PRESENT)
  {
    input_qos.relclass = GPRS_QOS_OMITTED ;
  }
  if (input_qos.peak EQ BAT_PEAK_NOT_PRESENT)
  {
    input_qos.peak= GPRS_QOS_OMITTED ;
  }
  if (input_qos.mean EQ BAT_MEAN_NOT_PRESENT)
  {
    input_qos.mean = GPRS_QOS_OMITTED ;
  }  
  ret = sAT_PlusCGQREQ (src_infos_psi->srcId, (SHORT)cid ,&input_qos);
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PlusCGQREQ      |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PlusCGQREQ      (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_QOS *pdef_qos[MAX_CID_PLUS_EINS];
  SHORT cid_array[MAX_CID_PLUS_EINS];
  int i = 0;
  T_BAT_cmd_response resp;
  T_BAT_res_que_plus_cgqreq que_cgqreq_buffer;
 
  TRACE_FUNCTION ("qBAT_PlusCGQREQ()");
  resp.ctrl_response = BAT_RES_QUE_PLUS_CGQREQ;
  resp.response.ptr_que_plus_cgqreq = &que_cgqreq_buffer;
  
  ret=qAT_PlusCGQREQ (src_infos_psi->srcId, pdef_qos, cid_array);
  if (ret EQ AT_CMPL)
  {
    for ( i = 0; i < MAX_CID && pdef_qos[i]; i++ )
    {
      resp.response.ptr_que_plus_cgqreq->cid = ((cid_array[i] == INVALID_CID) ? 
                                                BAT_PDP_CID_NOT_PRESENT : cid_array[i]);
      resp.response.ptr_que_plus_cgqreq->precedence = pdef_qos[i]->preced ;
      resp.response.ptr_que_plus_cgqreq->delay = pdef_qos[i]->delay ;
      resp.response.ptr_que_plus_cgqreq->reliability= pdef_qos[i]->relclass ;
      resp.response.ptr_que_plus_cgqreq->peak = pdef_qos[i]->peak ;
      resp.response.ptr_que_plus_cgqreq->mean = pdef_qos[i]->mean ;    
      aci_bat_send(src_infos_psi,&resp);
    }
    return ACI_BAT_CMPL;
  }
  ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_Unknown);
  return (ACI_BAT_FAIL);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGQMIN      |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGQMIN      (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_QOS input_qos = { ( UBYTE ) GPRS_QOS_OMITTED };
  T_BAT_pdp_cid  cid = BAT_PDP_CID_NOT_PRESENT ;
  
  TRACE_FUNCTION ("sBAT_PlusCGQMIN()");
  cid = cmd->params.ptr_set_plus_cgqmin->cid;
  if (cid EQ BAT_PDP_CID_NOT_PRESENT )
  {
    cid = GPRS_CID_OMITTED ;
  }
  input_qos.preced = cmd->params.ptr_set_plus_cgqmin->precedence;
  input_qos.delay = cmd->params.ptr_set_plus_cgqmin->delay;
  input_qos.relclass = cmd->params.ptr_set_plus_cgqmin->reliability;
  input_qos.peak  = cmd->params.ptr_set_plus_cgqmin->peak;
  input_qos.mean = cmd->params.ptr_set_plus_cgqmin->mean;

  if (input_qos.preced EQ BAT_PRECEDENCE_NOT_PRESENT)
  {
    input_qos.preced = GPRS_QOS_OMITTED ;
  }
  if (input_qos.delay EQ BAT_DELAY_NOT_PRESENT)
  {
    input_qos.delay = GPRS_QOS_OMITTED ;
  }
  if (input_qos.relclass EQ BAT_RELIABILITY_NOT_PRESENT)
  {
    input_qos.relclass = GPRS_QOS_OMITTED ;
  }
  if (input_qos.peak EQ BAT_PEAK_NOT_PRESENT)
  {
    input_qos.peak= GPRS_QOS_OMITTED ;
  }
  if (input_qos.mean EQ BAT_MEAN_NOT_PRESENT)
  {
    input_qos.mean = GPRS_QOS_OMITTED ;
  }
  ret = sAT_PlusCGQMIN (src_infos_psi->srcId, (SHORT)cid ,&input_qos);
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PlusCGQMIN      |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PlusCGQMIN      (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_BAT_cmd_response resp;
  T_BAT_res_que_plus_cgqmin que_cgqmin_buffer;
  T_QOS *pdef_qos[MAX_CID_PLUS_EINS];
  SHORT cid_array[MAX_CID_PLUS_EINS];
  int i = 0;
  
  TRACE_FUNCTION ("qBAT_PlusCGQMIN()");
  resp.ctrl_response = BAT_RES_QUE_PLUS_CGQMIN;
  resp.response.ptr_que_plus_cgqmin = &que_cgqmin_buffer;  
  ret=qAT_PlusCGQMIN (src_infos_psi->srcId, pdef_qos, cid_array);
  if (ret EQ AT_CMPL)
  {
    for ( i = 0; i < MAX_CID && pdef_qos[i]; i++ )
    {
      resp.response.ptr_que_plus_cgqmin->cid = (cid_array[i] == INVALID_CID) ? 
                                                BAT_PDP_CID_NOT_PRESENT : cid_array[i] ;
      resp.response.ptr_que_plus_cgqmin->precedence = pdef_qos[i]->preced;
      resp.response.ptr_que_plus_cgqmin->delay = pdef_qos[i]->delay ;
      resp.response.ptr_que_plus_cgqmin->reliability= pdef_qos[i]->relclass ;
      resp.response.ptr_que_plus_cgqmin->peak = pdef_qos[i]->peak ;
      resp.response.ptr_que_plus_cgqmin->mean = pdef_qos[i]->mean ;
      aci_bat_send(src_infos_psi,&resp);
    }
  }
  return (ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGATT       |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGATT       (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_CGATT_STATE state = CGATT_STATE_OMITTED;
 
  TRACE_FUNCTION ("sBAT_PlusCGATT()");
  
  state = cmd->params.ptr_set_plus_cgatt->state;
  ret = sAT_PlusCGATT ( src_infos_psi->srcId, state );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PlusCGATT       |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PlusCGATT       (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_BAT_cmd_response resp;
  T_BAT_res_que_plus_cgatt que_cgatt_buffer;
  T_CGATT_STATE state = CGATT_STATE_OMITTED;
  
  TRACE_FUNCTION ("qBAT_PlusCGATT()");
  resp.ctrl_response = BAT_RES_QUE_PLUS_CGATT;
  resp.response.ptr_que_plus_cgatt = &que_cgatt_buffer;
  
  ret = qAT_PlusCGATT(src_infos_psi->srcId, &state);
  if (ret EQ AT_CMPL)
  {
    resp.response.ptr_que_plus_cgatt->state = state;
    aci_bat_send(src_infos_psi,&resp); 
    return ACI_BAT_CMPL;
  }
  ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_Unknown);
  return (ACI_BAT_FAIL);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGCLASS     |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGCLASS     (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_CGCLASS_CLASS m_class = CGCLASS_CLASS_OMITTED;
  
  TRACE_FUNCTION ("sBAT_PlusCGCLASS()");
  
  m_class = (T_CGCLASS_CLASS)cmd->params.ptr_set_plus_cgclass->mobile_class;

  if (m_class EQ CGCLASS_CLASS_OMITTED)
  {
    ACI_ERR_DESC(ACI_ERR_CLASS_Cme,CME_ERR_OpNotAllow);
    return (ACI_BAT_FAIL);
  }
  ret = sAT_PlusCGCLASS ( src_infos_psi->srcId, m_class );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PlusCGCLASS     |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PlusCGCLASS     (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret;
  T_CGCLASS_CLASS m_class;
  
  TRACE_FUNCTION ("qBAT_PlusCGCLASS()");

  ret=qAT_PlusCGCLASS(src_infos_psi->srcId,&m_class);

  /*
  *   If the query completes, send the response now.
  */
  if (ret==ACI_BAT_CMPL)
  {
    T_BAT_cmd_response resp;
    T_BAT_res_que_plus_cgclass cgclass_data;
 
    resp.ctrl_response=BAT_RES_QUE_PLUS_CGCLASS;
    resp.response.ptr_que_plus_cgclass=&cgclass_data;

    /*
    *   This relies on T_BAT_mobile_class and T_CGCLASS_CLASS
    *   being identical.
    */
    resp.response.ptr_que_plus_cgclass->mobile_class=(T_BAT_mobile_class)m_class;

    aci_bat_send(src_infos_psi,&resp);
    return ACI_BAT_CMPL;
  }
  ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_Unknown);
  return (ACI_BAT_FAIL);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGACT       |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGACT       (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_CGACT_STATE state = CGACT_STATE_OMITTED;
  SHORT count = 0;
  SHORT cid_array[MAX_CID_PLUS_EINS];
  
  TRACE_FUNCTION ("sBAT_PlusCGACT()");
 
  state = cmd->params.ptr_set_plus_cgact->state;  
  if(cmd->params.ptr_set_plus_cgact->c_cid == 0)
    cid_array[0] = INVALID_CID;
  for(count=0;count<cmd->params.ptr_set_plus_cgact->c_cid;count++)
  {
    cid_array[count] = (cmd->params.ptr_set_plus_cgact->cid[count] EQ BAT_PDP_CID_NOT_PRESENT)
                   ? INVALID_CID : cmd->params.ptr_set_plus_cgact->cid[count];
  }
  if(count > 0)
    cid_array[count]= INVALID_CID;  
  ret = sAT_PlusCGACT ( src_infos_psi->srcId, state, cid_array );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PlusCGACT       |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PlusCGACT       (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;  
  SHORT cid_array[MAX_CID_PLUS_EINS];
  SHORT count = 0;
  BOOL  states[MAX_CID_PLUS_EINS];
  T_BAT_cmd_response resp;
  T_BAT_res_que_plus_cgact que_cgact_buffer;
  
  TRACE_FUNCTION ("qBAT_PlusCGACT()");
  resp.ctrl_response = BAT_RES_QUE_PLUS_CGACT;
  resp.response.ptr_que_plus_cgact = &que_cgact_buffer;  
  
  ret=qAT_PlusCGACT(src_infos_psi->srcId, states, cid_array);
  if(ret EQ AT_CMPL)
  {
    for ( count = 0; count < MAX_CID && cid_array[count] NEQ INVALID_CID; count++ )
    {
      resp.response.ptr_que_plus_cgact->state = states[count];    
      resp.response.ptr_que_plus_cgact->cid = cid_array[count];
      aci_bat_send(src_infos_psi,&resp);
    }
    return(ret);
  }
  ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_Unknown);
  return (ACI_BAT_FAIL);  
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGDATA      |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGDATA      (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  char L2P[MAX_L2P_LENGTH] = "";  
  SHORT cid_array[MAX_CID_PLUS_EINS] = { GPRS_CID_OMITTED };
  
  TRACE_FUNCTION ("sBAT_PlusCGDATA()");
  
  if(cmd->params.ptr_set_plus_cgdata->v_l2p)
  {
    memcpy(L2P,cmd->params.ptr_set_plus_cgdata->l2p,cmd->params.ptr_set_plus_cgdata->c_l2p);
  }
  else
  {
    L2P[0] = 0;
  }
 
  if(cmd->params.ptr_set_plus_cgdata->v_cid)
  {
    memcpy(cid_array,cmd->params.ptr_set_plus_cgdata->cid,2*cmd->params.ptr_set_plus_cgdata->c_cid);
    cid_array[cmd->params.ptr_set_plus_cgdata->c_cid] = INVALID_CID;
  }
  ret = sAT_PlusCGDATA (src_infos_psi->srcId, L2P, cid_array ); 
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGPADDR     |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGPADDR     (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_PDP_ADDRESS pdp_addresses[MAX_CID];
  SHORT         count = 0;
  SHORT         cid_array[MAX_CID_PLUS_EINS];
  T_BAT_cmd_set_plus_cgpaddr *cgpaddr;

  TRACE_FUNCTION ("sBAT_PlusCGPADDR()");

  /*
  *   Get a pointer to the CGPADDR data for convenience.
  */
  cgpaddr=cmd->params.ptr_set_plus_cgpaddr;
  
  if (cgpaddr->c_cid)
  {
    for (count=0;((count<(MAX_CID_PLUS_EINS-1)) && (count<cgpaddr->c_cid));count++)
      cid_array[count]=(SHORT)cgpaddr->cid[count];

    cid_array[count]=INVALID_CID;
  }
  else
  {
    cid_array[0]=GPRS_CID_OMITTED;
  }

  ret = (T_ACI_BAT_RSLT)sAT_PlusCGPADDR(src_infos_psi->srcId, cid_array, pdp_addresses );

  if (ret==ACI_BAT_CMPL)
  {
    T_BAT_cmd_response resp;
    T_BAT_res_set_plus_cgpaddr cgpaddr_data;
    UBYTE len;
 
    resp.ctrl_response=BAT_RES_SET_PLUS_CGPADDR;
    resp.response.ptr_set_plus_cgpaddr=&cgpaddr_data;

    for (count=0;count<BAT_MAX_PDP_CID;count++)
    {
      if (cid_array[count]==INVALID_CID)
        break;

      cgpaddr_data.cid=(T_BAT_pdp_cid)cid_array[count];

      len=strlen(pdp_addresses[count]);

      if (len>BAT_MAX_PDP_ADD)
        len=BAT_MAX_PDP_ADD;

      memcpy(cgpaddr_data.pdp_addr,pdp_addresses[count],len);
      cgpaddr_data.c_pdp_addr=(U8)len;

      aci_bat_send(src_infos_psi,&resp);
    }
  }

  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGANS       |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGANS       (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  char L2P[MAX_L2P_LENGTH] = "" ;
  SHORT cid = GPRS_CID_OMITTED, response = -1 ;
  
  TRACE_FUNCTION ("sBAT_PlusCGANS()");
 
    response = cmd->params.ptr_set_plus_cgans->response;
    if(cmd->params.ptr_set_plus_cgans->v_l2p)
    {
      memcpy(L2P,cmd->params.ptr_set_plus_cgans->l2p,cmd->params.ptr_set_plus_cgans->c_l2p);
    }
    else
   {
     L2P[0] = 0;
   }
    
   cid = cmd->params.ptr_set_plus_cgans->cid;
   if (cid EQ BAT_PDP_CID_NOT_PRESENT )
  {
    cid = GPRS_CID_OMITTED ;
  }    
  ret = sAT_PlusCGANS ( src_infos_psi->srcId, response, L2P, cid );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PlusCGREG       |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PlusCGREG       (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_CGREG_STAT   stat;
  USHORT         lac, cid;    
 
  TRACE_FUNCTION ("qBAT_PlusCGREG()");
 
  ret=(T_ACI_BAT_RSLT)qAT_PlusCGREG(src_infos_psi->srcId, &stat, &lac, &cid);

  if (ret==ACI_BAT_CMPL)
  {
    T_BAT_cmd_response resp;
    T_BAT_res_que_plus_cgreg que_cgreg_buffer;
 
    resp.ctrl_response = BAT_RES_QUE_PLUS_CGREG;
    resp.response.ptr_que_plus_cgreg = &que_cgreg_buffer;

    resp.response.ptr_que_plus_cgreg->stat = stat;
    resp.response.ptr_que_plus_cgreg->lac = lac;
    resp.response.ptr_que_plus_cgreg->ci = cid;
    aci_bat_send(src_infos_psi,&resp);
  }

  return(ret);  
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PercentCGREG    |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PercentCGREG    (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_P_CGREG_STAT  stat;
  USHORT          lac, cid;
 
  TRACE_FUNCTION ("qBAT_PercentCGREG()");
  ret=(T_ACI_BAT_RSLT)qAT_PercentCGREG(src_infos_psi->srcId, &stat, &lac, &cid);

  if (ret==ACI_BAT_CMPL)
  {
    T_BAT_cmd_response resp;
    T_BAT_res_que_percent_cgreg que_cgreg_buffer;

    resp.ctrl_response = BAT_RES_QUE_PERCENT_CGREG;
    resp.response.ptr_que_percent_cgreg = &que_cgreg_buffer;

    resp.response.ptr_que_percent_cgreg->stat = stat;
    resp.response.ptr_que_percent_cgreg->lac = lac;
    resp.response.ptr_que_percent_cgreg->ci = cid;
    resp.response.ptr_que_percent_cgreg->state = (cmhSM_isContextActive() ? BAT_P_CGREG_STATE_STAT_1 
                                                                        : BAT_P_CGREG_STATE_STAT_0 );
    aci_bat_send(src_infos_psi,&resp);
  }

  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : tBAT_PlusCGPADDR     |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT tBAT_PlusCGPADDR     (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  U8         count = 0;
  SHORT         cid_array[MAX_CID_PLUS_EINS];  
  int           pos = 0;
  T_BAT_cmd_response resp;
  T_BAT_res_tst_plus_cgpaddr tst_cgpaddr_buffer;
  
  TRACE_FUNCTION ("tBAT_PlusCGPADDR()");

  ret = tAT_PlusCGPADDR ( src_infos_psi->srcId, cid_array );
  resp.ctrl_response = BAT_RES_TST_PLUS_CGPADDR; 
  resp.response.ptr_tst_plus_cgpaddr = &tst_cgpaddr_buffer;
  if (ret EQ ACI_BAT_CMPL)
  {
    while ( cid_array[count] NEQ INVALID_CID )
    {
      resp.response.ptr_tst_plus_cgpaddr->cid[count]=cid_array[count];
      count ++;
    }
    if(count == 0)
      resp.response.ptr_tst_plus_cgpaddr->cid[0] = BAT_PDP_CID_NOT_PRESENT;
    
    resp.response.ptr_tst_plus_cgpaddr->c_cid = count;  
    aci_bat_send(src_infos_psi,&resp);
    return(ACI_BAT_CMPL);
  }
  ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_Unknown);
  return(ACI_BAT_FAIL);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PlusCGSMS       |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PlusCGSMS       (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_CGSMS_SERVICE service = CGSMS_SERVICE_OMITTED;
  
  TRACE_FUNCTION ("sBAT_PlusCGSMS()");
  
  service = cmd->params.ptr_set_plus_cgsms->service;
  ret = sAT_PlusCGSMS ( src_infos_psi->srcId, service );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PlusCGSMS       |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PlusCGSMS       (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_BAT_cmd_response resp;
  T_BAT_res_que_plus_cgsms que_cgsms_buffer;
  T_CGSMS_SERVICE service = CGSMS_SERVICE_OMITTED;
  
  TRACE_FUNCTION ("qBAT_PlusCGSMS()");
  
  ret = qAT_PlusCGSMS(src_infos_psi->srcId, &service);
  resp.ctrl_response = BAT_RES_QUE_PLUS_CGSMS;
  resp.response.ptr_que_plus_cgsms = &que_cgsms_buffer;
  if(ret EQ ACI_BAT_CMPL)
  {
    resp.response.ptr_que_plus_cgsms->service = service;
    aci_bat_send(src_infos_psi,&resp);
    return ret;
  }
  else
  {
    ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_Unknown);
    return (ACI_BAT_FAIL);
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PercentCGAATT   |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PercentCGAATT   (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_BAT_percent_cgaatt_att_m att_m;
  T_BAT_percent_cgaatt_det_m det_m;   
  
  TRACE_FUNCTION ("sBAT_PercentCGAATT()");
  
  att_m = cmd->params.ptr_set_percent_cgaatt->att_m;
  det_m = cmd->params.ptr_set_percent_cgaatt->det_m;  
  ret = sAT_PercentCGAATT ( src_infos_psi->srcId, att_m, det_m );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PercentCGAATT   |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PercentCGAATT   (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_BAT_percent_cgaatt_att_m att_m;
  T_BAT_percent_cgaatt_det_m det_m;   
 
  TRACE_FUNCTION ("qBAT_PercentCGAATT()");
  
  ret = (T_ACI_BAT_RSLT)qAT_PercentCGAATT(
    src_infos_psi->srcId, (T_CGAATT_ATTACH_MODE *)&att_m, (T_CGAATT_DETACH_MODE *)&det_m);
 
  if(ret==ACI_BAT_CMPL)
  {
    T_BAT_cmd_response resp;
    T_BAT_res_que_percent_cgaatt que_cgaatt_buffer;
 
    resp.ctrl_response = BAT_RES_QUE_PERCENT_CGAATT;
    resp.response.ptr_que_percent_cgaatt = &que_cgaatt_buffer;

    resp.response.ptr_que_percent_cgaatt->att_m = att_m;
    resp.response.ptr_que_percent_cgaatt->det_m = det_m;
    aci_bat_send(src_infos_psi,&resp);
  }

  return ret;
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PercentSNCNT    |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PercentSNCNT    (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_BAT_percent_sncnt_rst rst;
  
  TRACE_FUNCTION ("sBAT_PercentSNCNT()");
  /* what to do with rst? */
  rst = cmd->params.ptr_set_percent_sncnt->rst;
  ret = sAT_PercentSNCNT( src_infos_psi->srcId, TRUE );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PercentSNCNT    |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PercentSNCNT    (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{ 
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  
  TRACE_FUNCTION ("qBAT_PercentSNCNT()");
  ret = sAT_PercentSNCNT( src_infos_psi->srcId, FALSE );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PercentCGMM     |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PercentCGMM     (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  UBYTE cipher_on , tlli_handling ;
  
  TRACE_FUNCTION ("sBAT_PercentCGMM()");
  
  cipher_on = cmd->params.ptr_set_percent_cgmm->cipher_on;
  tlli_handling  = cmd->params.ptr_set_percent_cgmm->tlli_handling;

  ret = sAT_PercentCGMM( src_infos_psi->srcId, cipher_on, tlli_handling );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PercentCGPCO    |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PercentCGPCO    (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ATI_RSLT ret = ATI_FAIL;
  USHORT              cid = 0;
  T_ACI_CGPCO_FORMAT  format;
  CHAR                pco_string[BAT_MAX_CGPCO_PCO_LEN+1];
  T_ACI_CGPCO_MODE    mode;
  
  TRACE_FUNCTION ("sBAT_PercentCGPCO()");

  memset(pco_string,0,sizeof(pco_string));

  mode = cmd->params.ptr_set_percent_cgpco->mode;
  format = cmd->params.ptr_set_percent_cgpco->format;

  if (cmd->params.ptr_set_percent_cgpco->v_pco)
  {
    memcpy(pco_string,cmd->params.ptr_set_percent_cgpco->pco,cmd->params.ptr_set_percent_cgpco->c_pco);
  }
  cid = cmd->params.ptr_set_percent_cgpco->cid;
  if(cid EQ BAT_PDP_CID_NOT_PRESENT)
  {
    cid = INVALID_CID ;
  }
  if (cid > MAX_CID)
  {
    ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_OpNotSupp);
    return ACI_BAT_FAIL ;
  }
    
  switch (mode)
  {
    case CGPCO_MODE_QUERY_PCO:
      {
        ULONG gateway,dns1,dns2;
        char dns1_str[sizeof("255.255.255.255")];
        char dns2_str[sizeof("255.255.255.255")];
        char gatew_str[sizeof("255.255.255.255")];
        USHORT i;

        for (i=1;i<MAX_CID;i++)
        {
          if ((i==cid) || (cid==INVALID_CID))
          {
            gateway=dns1=dns2=0;

            /*
            *   Ignore the return value of the ACI function, which is
            *   always AT_CMPL.
            */
            qAT_PercentCGPCO(src_infos_psi->srcId,&gateway,&dns1,&dns2,i);

            if (dns1 || dns2 || gateway)
            {
              T_BAT_cmd_response resp;
              T_BAT_res_set_percent_cgpco cgpco;
 
              resp.ctrl_response=BAT_RES_SET_PERCENT_CGPCO;
              resp.response.ptr_set_percent_cgpco=&cgpco;

              ipaddr_to_string(dns1, dns1_str);
              ipaddr_to_string(dns2, dns2_str);
              ipaddr_to_string(gateway, gatew_str);

              cgpco.cid=(T_BAT_pdp_cid)i;

              sprintf((char *)cgpco.pco,"%s,%s,%s",dns1_str,dns2_str,gatew_str);
              cgpco.c_pco=strlen((char *)cgpco.pco);

              aci_bat_send(src_infos_psi,&resp);
            }
          }
        }
      }
      return (ACI_BAT_CMPL);

    case CGPCO_MODE_SET_PCO:
    {
      switch (format)
      {
        case CGPCO_FORMAT_HEX:
          ret = setatPercentCGPCO_HEX (cid, (UBYTE*)pco_string, (USHORT)strlen (pco_string));
          break;
        case CGPCO_FORMAT_ASCII:
          ret = setatPercentCGPCO_ASCII (src_infos_psi->srcId, cid, (UBYTE*)pco_string, (USHORT)strlen (pco_string));
          break;
        default:
          ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_OpNotSupp);
          break;
       }
      break;
    }
    default:
      ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_OpNotSupp);
      break;
  }

  /*
  *   Convert the ATI return value into ACI/BAT format.
  */
  switch (ret)
  {
    default:
    case ATI_FAIL:
    case ATI_FAIL_NO_OUTPUT:
      return(ACI_BAT_FAIL);

    case ATI_BUSY:
      return(ACI_BAT_BUSY);

    case ATI_EXCT:
      return(ACI_BAT_EXCT);

    case ATI_CMPL:
    case ATI_CMPL_NO_OUTPUT:
      return(ACI_BAT_CMPL);
  }
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PercentCGPPP    |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PercentCGPPP    (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_ACI_PPP_PROT  protocol;
  
  TRACE_FUNCTION ("sBAT_PercentCGPPP()");
  
  protocol = cmd->params.ptr_set_percent_cgppp->protocol;
  ret = sAT_PercentCGPPP(src_infos_psi->srcId, protocol);
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PercentCGPPP    |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PercentCGPPP    (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_ACI_PPP_PROT  protocol;
  T_BAT_cmd_response resp;
  T_BAT_res_que_percent_cgppp que_cgppp_buffer;

  TRACE_FUNCTION ("qBAT_PercentCGPPP()");
  
  ret = qAT_PercentCGPPP(src_infos_psi->srcId, &protocol);
  resp.ctrl_response = BAT_RES_QUE_PERCENT_CGPPP;
  resp.response.ptr_que_percent_cgppp = &que_cgppp_buffer;
  if( ret EQ ATI_FAIL)
  {
    ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_Unknown);
    return (ACI_BAT_FAIL);
  }
  resp.response.ptr_que_percent_cgppp->protocol = protocol;
  aci_bat_send(src_infos_psi,&resp);
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : sBAT_PercentCGCLASS  |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT sBAT_PercentCGCLASS  (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_PERCENT_CGCLASS m_class = PERCENT_CGCLASS_OMITTED;
  
  TRACE_FUNCTION ("sBAT_PercentCGCLASS()");

  m_class=(T_PERCENT_CGCLASS)cmd->params.ptr_set_percent_cgclass->p_mobile_class;

  if (m_class EQ PERCENT_CGCLASS_OMITTED)
  {
    ACI_ERR_DESC( ACI_ERR_CLASS_Cme, CME_ERR_OpNotAllow);
    return (ACI_BAT_FAIL);
  }
  ret = sAT_PercentCGCLASS ( src_infos_psi->srcId, m_class );
  return(ret);
}

/*
+--------------------------------------------------------------------+
| PROJECT : GSM-F&D (8411)            MODULE  : ACI_BAT              |
| STATE   : code                      ROUTINE : qBAT_PercentCGCLASS  |
+--------------------------------------------------------------------+

  PURPOSE : 
*/
GLOBAL T_ACI_BAT_RSLT qBAT_PercentCGCLASS  (T_ACI_DTI_PRC_PSI *src_infos_psi,
                                            T_BAT_cmd_send    *cmd)
{
  T_ACI_BAT_RSLT ret = ACI_BAT_FAIL;
  T_PERCENT_CGCLASS m_class, currentClass;
  T_BAT_cmd_response resp;
  T_BAT_res_que_percent_cgclass que_cgclass_buffer;
  
  TRACE_FUNCTION ("qBAT_PercentCGCLASS()");

  resp.ctrl_response = BAT_RES_QUE_PERCENT_CGCLASS;
  resp.response.ptr_que_percent_cgclass = &que_cgclass_buffer;
  ret = qAT_PercentCGCLASS(src_infos_psi->srcId, &m_class, &currentClass);
  if (ret EQ AT_CMPL)
  {
    resp.response.ptr_que_percent_cgclass->p_mobile_class = m_class;
    resp.response.ptr_que_percent_cgclass->current_mobile_class = currentClass;    
    aci_bat_send(src_infos_psi,&resp);
  }
  return(ret);
}

#endif /* GPRS */

