/********************************************************************************/
/*                                                                              */
/*    File Name:         ffs_pc_api.h                                           */
/*                                                                              */
/*    Purpose:           This header file contains the external constants and   */
/*                       prototypes related to the Flash File System.           */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       02/27/02        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/* (C) Copyright 2002 by Texas Instruments Incorporated, All Rights Reserved.   */
/*                                                                              */
/********************************************************************************/
#ifndef _FFS_PC_API_
#define _FFS_PC_API_

#include <windows.h>

#ifdef __cplusplus
extern "C"
{
#endif


/******************************** FILE DESCRIPTOR *******************************/
/*                                                                              */
/* Define the file descriptor.                                                  */
typedef INT32  T_FFS_FD;

	
/*********************************** FILE SIZE **********************************/
/*                                                                              */
/* Define the file size.                                                        */
typedef INT32  T_FFS_SIZE;

	
/************************************ OFFSET ************************************/
/*                                                                              */
/* Define the offset.                                                           */
typedef UINT32  T_FFS_OFFSET;

/************************************* FLAGS ************************************/
/*                                                                              */
/* Define the modes used to open the file.                                      */
typedef UINT32  T_FFS_OPEN_FLAGS;

#define FFS_O_CREATE                                  (0x00000001)
#define FFS_O_APPEND                                  (0x00000002)
#define FFS_O_EXCL                                    (0x00000004)
#define FFS_O_TRUNC                                   (0x00000008)
#define FFS_O_RDONLY                                  (0x00000010)
#define FFS_O_WRONLY                                  (0x00000020)
#define FFS_O_RDWR                                    (FFS_O_RDONLY | FFS_O_WRONLY)

/* Define the starting point for the file pointer move.                         */
typedef UINT32  T_FFS_WHENCE;

#define FFS_SEEK_SET                                  (FILE_BEGIN)
#define FFS_SEEK_CUR                                  (FILE_CURRENT)
#define FFS_SEEK_END                                  (FILE_END)


/*********************************** META-DATA **********************************/
/*                                                                              */
/* Define object types.                                                         */
typedef enum 
{
	OT_FILE	= 0x00000001,
	OT_DIR,
	OT_LINK
} T_FFS_OBJECT_TYPE;

/* Define a structure used to gather information (meta-data) about an object.   */
typedef struct 
{
	T_FFS_OBJECT_TYPE  type;	/* Object type: file, folder, ...               */
	T_FFS_SIZE         size;	/* Size of data space occupied by object.       */
} T_FFS_STAT;


/*********************************** DIRECTORY **********************************/
/*                                                                              */
/* Define a structure used to gather information about a directory.             */
typedef struct 
{ 
	WIN32_FIND_DATA  find_data;
	HANDLE			 search_handle;
} T_FFS_DIR;

	
/***************************** INTERNAL ERROR CODES *****************************/
/*                                                                              */
/* Define the internal error codes.                                             */
typedef enum
{
	EFFS_OK           = 0,		/* OK.                                          */

    EFFS_NODEVICE     = -1,		/* Flash device unknown.                        */
    EFFS_CORRUPTED    = -2,		/* File system corrupted.                       */
    EFFS_NOPREFORMAT  = -3,		/* Flash File System not preformatted.          */
    EFFS_NOFORMAT     = -4,		/* Flash File System not formatted.             */
    EFFS_BADFORMAT    = -5,		/* Format not recognized.                       */
    EFFS_MAGIC        = -6,		/* Bad magic.                                   */
    EFFS_AGAIN        = -7,		/* Not ready, try again later on.               */
    EFFS_NOSYS        = -8,		/* Function not implemented.                    */
    EFFS_DRIVER       = -9,		/* Driver error.                                */

    EFFS_NOSPACE      = -10,	/* Out of file space (no free data space).      */
    EFFS_FSFULL       = -11,	/* File system full (no free inodes).           */
    EFFS_BADNAME      = -12,	/* Bad filename.                                */
    EFFS_NOTFOUND     = -13,	/* Not found.                                   */
    EFFS_EXISTS       = -14,	/* Object already exists.                       */
    EFFS_ACCESS       = -15,	/* File access permission violation.            */
    EFFS_NAMETOOLONG  = -16,	/* Filename too long.                           */
    EFFS_INVALID      = -17,	/* Invalid argument.                            */
    EFFS_DIRNOTEMPTY  = -18,	/* Directory not empty.                         */
    EFFS_NOTADIR      = -19,	/* Object is not a directory.                   */
    EFFS_SPARE        = -20,	/* Spare.                                       */

    EFFS_FILETOOBIG   = -21,	/* File too big.                                */
    EFFS_NOTAFILE     = -22,	/* Object is not a file.                        */
    EFFS_PATHTOODEEP  = -23,	/* Path too deep.                               */
    EFFS_NUMFD        = -24,	/* Maximum number of open files reached.        */
    EFFS_BADFD        = -25,	/* Bad file descriptor.                         */
    EFFS_BADOP        = -26,	/* Bad options.                                 */
    EFFS_LOCKED       = -27,	/* File locked by another file descriptor.      */

    EFFS_TOOBIG       = -30,	/* Too big (buffer overflow).                   */
    EFFS_MSGALLOC     = -31,	/* Message allocation failed.                   */
    EFFS_MSGSEND      = -32,	/* Message send failed.                         */

    EFFS_SIBLINGLOOP  = -40,	/* Directory sibling loop.                      */
    EFFS_NOBLOCKS     = -41,	/* No more blocks.                              */
    EFFS_DBR          = -42,	/* Data reclaim did not finish.                 */
} T_FFS_RET;


/*
 * Prototypes for functions provided by the FFS simulation on the PC
 */
T_FFS_FD ffs_open       (const char        *pathname_p,
                         T_FFS_OPEN_FLAGS  flags);
T_FFS_RET ffs_close     (T_FFS_FD  fd);
T_FFS_SIZE ffs_write    (T_FFS_FD    fd,
                         void        *buffer_p,
                         T_FFS_SIZE  size);
T_FFS_SIZE ffs_read     (T_FFS_FD    fd,
                         void        *buffer_p,
                         T_FFS_SIZE  size);
T_FFS_SIZE ffs_seek     (T_FFS_FD      fd,
                         T_FFS_SIZE    offset,
                         T_FFS_WHENCE  whence);
T_FFS_RET ffs_ftruncate (T_FFS_FD      fd,
                         T_FFS_OFFSET  length);
T_FFS_RET ffs_stat      (const char  *pathname_p,
                         T_FFS_STAT  *stat_p);
T_FFS_RET ffs_remove    (const char  *pathname_p);
T_FFS_RET ffs_mkdir     (const char  *pathname_p);
T_FFS_SIZE ffs_opendir  (const char  *pathname_p,
                         T_FFS_DIR   *dir_p);
T_FFS_SIZE ffs_readdir  (T_FFS_DIR   *dir_p,
                         char        *buffer_p,
                         T_FFS_SIZE  size);
T_FFS_RET ffs_rename    (const char  *old_pathname_p,
                         const char  *new_pathname_p);

T_FFS_SIZE ffs_file_read(const char *name, void *addr, T_FFS_SIZE size);

T_FFS_SIZE ffs_init (void);

#ifdef __cplusplus
}
#endif

#endif /* #ifndef _FFS_PC_API_ */
