/******************************************************************************
 * FLUID (Flash Loader Utility Independent of Device)
 *
 * (C) Delta Technologies 2001.
 * Cleanup, modifications and extensions by Mads Meisner-Jensen, mmj@ti.com.
 *
 * Target common definitions
 *
 * $Id: target.h,v 1.3 2003/07/10 17:50:21 rlendenmann Exp $
 *
 ******************************************************************************/
#ifndef __TARGET_H__
#define __TARGET_H__
#include "protocol.h"


/******************************************************************************
 * Typedefs and Prototypes
 ******************************************************************************/

#ifndef RIVIERA_INCLUDED
typedef unsigned char  UBYTE;
typedef unsigned short UINT16;
typedef unsigned long  UINT32;
typedef signed char    INT8;
typedef signed short   INT16;
typedef signed long    INT32;
#endif

typedef unsigned char  uint8;
typedef unsigned short uint16;
typedef unsigned long  uint32;
typedef signed char    int8;
typedef signed short   int16;
typedef signed long    int32;

typedef union {
    UINT32 i;    // integer value
    UBYTE  b[4]; // byte array
} union32_t;

typedef union {
    UINT16 i;    // integer value
    UBYTE  b[2]; // byte array
} union16_t;

typedef struct {
    char   (*init)(uint32 addr);
    char   (*erase)(uint32 *addr, int size);
    char   (*program)(uint32 addr, uint16 *buf, int size);
    uint8  (*getchar)(void);
    void   (*putchar)(uint8 ch);
    void   (*debug_putstr)(uint32 uart, char *string);
    void   (*debug_putnum)(uint32 uart, unsigned int number);
    uint32 uart_base;
} method_table_t;

// Prototypes for functions in debug.c
void debug_putstr(uint32 uart, char *string);
void debug_putnum(uint32 uart, unsigned int number);


/******************************************************************************
 * Macros
 ******************************************************************************/

#define REGISTER_8_READ(addr)         (*((volatile uint8 *)  (addr)))
#define REGISTER_8_WRITE(addr, data)  (*((volatile uint8 *)  (addr)) = data)
#define REGISTER_16_READ(addr)        (*((volatile uint16 *) (addr)))
#define REGISTER_16_WRITE(addr, data) (*((volatile uint16 *) (addr)) = data)

#define BIT_SET(reg, bit) (*((volatile UINT16 *) (reg)) |=  (1 << (bit)))
#define BIT_CLR(reg, bit) (*((volatile UINT16 *) (reg)) &= ~(1 << (bit)))

#define FLASH_READ(addr)        (*(volatile uint16 *) (addr))
#define FLASH_WRITE(addr, data) (*(volatile uint16 *) (addr)) = data

#define PUTCHAR_FUNCTION_CODE(uart) \
    while (!(REGISTER_8_READ(uart + UART_LSR) & STAT_TXRDY)) ; \
    REGISTER_8_WRITE(uart + UART_THR, ch);

#define GETCHAR_FUNCTION_CODE(uart) \
    while (!(REGISTER_8_READ(uart + UART_LSR) & STAT_RXRDY)) ; \
    return (uint8) REGISTER_8_READ(uart + UART_RHR);


/******************************************************************************
 * Debug Macros
 ******************************************************************************/

#define DEBUG_PUTSTR(uart, text) ((method_table_t *)(my_addr))->debug_putstr(uart, text);
#define DEBUG_PUTNUM(uart, num)  ((method_table_t *)(my_addr))->debug_putnum(uart, num);


/******************************************************************************
 * Hardware Definitions
 ******************************************************************************/

#define UART_BASE_MODEM (0xFFFF5800)
#define UART_BASE_IRDA  (0xFFFF5000)

// ROM/Flash Config register
#define WS_REG_ROM 0x400000   

// UART register definitions.
#define UART_RHR   (0x00) /* Receive Holding Register */
#define UART_THR   (0x00) /* Transmit Holding Register */
#define UART_FCR   (0x02) /* FIFO Control Register */ 
#define UART_LCR   (0x03) /* Line Control Register */ 
#define UART_LSR   (0x05) /* Line Status Register */ 
#define UART_MDR1  (0x08) /* Mode Definition Register 1 */
#define UART_EFR   (0x02) /* Enhanced Feature Register */
#define UART_DLL   (0x00) /* Divisor Latches */
#define UART_DLH   (0x01) /* Divisor Latches */
#define UART_UIR   (0xFFFF6000)      /* UART Interface Register */

// UART status bit definitions.
#define STAT_RXRDY (0x01)
#define STAT_TXRDY (0x20)
#define STAT_TX_E  (0x40)

// UART_UIR bit definitions.
#define UART_ACCESS  (0)
#define UART_MASK_IT (1)

// UART_EFR bit definitions.
#define ENHANCED_EN  (4)

// UART command bit definitions.
#define CMD_UART_RESET   (0x07)
#define CMD_EFR_EN       (0xBF)
#define CMD_FCR_MCR_EN   (0x80)
#define CMD_FIFO_EN      (0x07)
#define CMD_UART_MODE    (0x00)
#define CMD_CHAR8_STOP1_NOPAR   (0x03)

// UART ???
#define BAUD_38400  (0x15)
#define BAUD_115200 (0x07)
#define BAUD_406250 (0x02)
#define BAUD_812500 (0x01)


// Enable/Disable ROM/Flash write strobe
#define ROM_WRITE_ENABLE()  *((volatile uint16*) WS_REG_ROM) |= 0x80;
#define ROM_WRITE_DISABLE() *((volatile uint16*) WS_REG_ROM) &= ~0x80;

#endif /* __TARGET_H__ */
