/*******************************************************************************
 *
 * COMMAND.H
 *
 * This module contains commands structures and functions to analyze a command
 * received and to build a command to send (request and confirmation).
 * 
 * (C) Texas Instruments 1998
 *
 ******************************************************************************/

#ifndef __COMMAND_H__
#define __COMMAND_H__

#include "main/sys_types.h"

#define COM_SERIAL_COMMAND_LENGTH     (256) /* In bytes. */
#define COM_MAX_NUMBER_OF_16BITS_DATA ( 64)

/*
 * Constants used in the confirmation of the COM_GET_MONITOR_ID command.
 */

enum {
    COM_MONITOR_STAND_ALONE,
    COM_MONITOR_LIBRARY,
	COM_BOOTLOADER
};

/*
 * Constants used in the confirmation of the COM_GET_CHIP_ID command.
 */

enum {
    COM_POLE112LA00,
    COM_GEMINI_EFH,
    COM_CHIP_NOT_SUPPORTED
};

/*
 * Constants used in the confirmation of the COM_GET_BOARD_ID command.
 */

enum {
    COM_A_SAMPLE_1_0,
    COM_A_SAMPLE_1_5,
    COM_EVA4,
    COM_GAIA_BOARD21
};

/*
 * Constants used in the confirmation of the COM_START_APPLICATION command.
 */

enum {
    COM_APPLICATION_IS_RUNNING,
    COM_MONITOR_IS_RUNNING
};

/*
 * List of commands.
 */

enum {
    COM_GET_MONITOR_ID,
    COM_GET_FLASH_ID,
    COM_GET_CHIP_ID,
    COM_GET_BOARD_ID,
    COM_ERASE_FLASH,
    COM_WRITE_DATA,
    COM_START_APPLICATION,
    COM_READ_PARAMETERS,
    COM_WRITE_PARAMETERS,
	COM_LOAD_APPLICATION,
	COM_SEND_RUN_ADDRESS
};

/*
 * Functions results.
 */

enum {
    COM_SUCCESS,
    COM_NO_EXECUTION,
    COM_ERASE_ERROR,
    COM_WRITE_ERROR,
    COM_PARAMETER_ERROR,
    COM_ADDRESS_ERROR,
    COM_FILE_ERROR,
	COM_NOT_SUPPORTED,
    COM_COMMAND_UNKNOWN
};


/*
 * Constants used in the bootloader.
 */

enum {
    CMD_RAM_SUCCESS,
    CMD_RAM_NO_EXECUTION,
    CMD_RAM_WRITE_ERROR,
    CMD_RAM_ADDRESS_ERROR,
    CMD_RAM_FILE_ERROR
};

/*
 * These macros allow to access to fields of each structure.
 */

#define COM_COMMAND_WORD(COMMAND) ((COMMAND)->command_word)

#define COM_RESULT(COMMAND) ((COMMAND)->result)

#define COM_MONITOR_ID(COMMAND) ((COMMAND)->param.monitor_id.identifier)

#define COM_MONITOR_VERSION_FIRST_NBR(COMMAND) ((COMMAND)->param.monitor_id.version_first_nbr)

#define COM_MONITOR_VERSION_SECOND_NBR(COMMAND) ((COMMAND)->param.monitor_id.version_second_nbr)

#define COM_FLASH_ID(COMMAND) ((COMMAND)->param.flash_id.identifier)

#define COM_BOARD_ID(COMMAND) ((COMMAND)->param.board_id.identifier)

#define COM_FILE_CHECKSUM(COMMAND) ((COMMAND)->param.checksum.file_checksum)

#define COM_FLASH_CHECKSUM(COMMAND) ((COMMAND)->param.checksum.flash_checksum)

#define COM_CHIP_ID(COMMAND) ((COMMAND)->param.chip_id.identifier)

#define COM_STARTING_STATE(COMMAND) ((COMMAND)->param.starting.state)

#define COM_READ_PARAMS_OFFSET(COMMAND) \
            ((COMMAND)->param.params_read.param_offset)

#define COM_NUMBER_OF_PARAMS_READ(COMMAND) \
            ((COMMAND)->param.params_read.number_of_params)

#define COM_READ_PARAMS_LIST_ADDRESS(COMMAND) \
            (&((COMMAND)->param.params_read.param[0]))

#define COM_WRITTEN_PARAMS_OFFSET(COMMAND) \
            ((COMMAND)->param.params_written.param_offset)

#define COM_NUMBER_OF_PARAMS_WRITTEN(COMMAND) \
            ((COMMAND)->param.params_written.number_of_params)

#define COM_WRITTEN_PARAMS_LIST_ADDRESS(COMMAND) \
            (&((COMMAND)->param.params_written.param[0]))

#define COM_FILE_RAM_CHECKSUM(COMMAND) ((COMMAND)->param.checksum_RAM.file_checksum)

#define COM_RAM_CHECKSUM(COMMAND) ((COMMAND)->param.checksum_RAM.RAM_checksum)

#define COM_ADDRESS_MSB(COMMAND) \
            ((COMMAND)->param.address.address_msb)
 
#define COM_ADDRESS_LSB(COMMAND) \
            ((COMMAND)->param.address.address_lsb)

 
 
/*
 * Parameters for each command.
 * Get software id   : no parameter.
 * Get flash id      : no parameter.
 * Get chip id       : no parameter.
 * Get board id      : no parameter.
 * Erase flash       : no parameter.
 * Write data        : no parameter.
 * Start application : no parameter.
 */

/*
 * Read parameters:
 *  - parameter offset (even),
 *  - number of parameters to read
 */

typedef struct s_com_read_params_req {
    SYS_UWORD16 param_offset;
    SYS_UWORD8  number_of_params;
} t_com_read_params_req;

/*
 * Write parameters:
 *  - parameter offset (even),
 *  - number of parameters (16 bits) to write,
 *  - list of parameters to write.
 */

typedef struct s_com_write_params_req {
    SYS_UWORD16 param_offset;
    SYS_UWORD8  number_of_params;
    SYS_UWORD16 param[COM_MAX_NUMBER_OF_16BITS_DATA];     
} t_com_write_params_req;


/*
 * Start application in RAM
 *  - file checksum,
 *  - flash checksum
 */

typedef struct s_com_send_run_address_req {
    SYS_UWORD16 address_msb;
    SYS_UWORD16 address_lsb;
} t_com_send_run_address_req;


/*
 * General request structure.
 */

typedef struct s_com_request {
    SYS_UWORD8 command_word;
    union u_req_param {
        t_com_read_params_req  params_read;
        t_com_write_params_req params_written;
		t_com_send_run_address_req address;
    } param;
} t_com_request;

/*
 * Parameters for each confirmation.
 * Erase flash      : no parameter.
 * Write parameters : no parameter.
 */

/*
 * Get monitor id:
 *  - identifier,
 *  - version.
 */

typedef struct s_com_get_monitor_id_cnf {
    SYS_UWORD8 identifier;
    SYS_UWORD8 version_first_nbr;
	SYS_UWORD8 version_second_nbr;
} t_com_get_monitor_id_cnf;

/*
 * Get flash id:
 *  - identifier.
 */

typedef struct s_com_get_flash_id_cnf {
    SYS_UWORD8 identifier;
} t_com_get_flash_id_cnf;

/*
 * Get chip id:
 *  - identifier.
 */

typedef struct s_com_get_chip_id_cnf {
    SYS_UWORD8 identifier;
} t_com_get_chip_id_cnf;

/*
 * Get board id:
 *  - identifier.
 */

typedef struct s_com_get_board_id_cnf {
    SYS_UWORD8 identifier;
} t_com_get_board_id_cnf;

/*
 * Write data:
 *  - file checksum,
 *  - flash checksum
 */

typedef struct s_com_write_data_cnf {
    SYS_UWORD16 file_checksum;
    SYS_UWORD16 flash_checksum;
} t_com_write_data_cnf;

/*
 * Start application:
 *  - state (application is running or not)
 */

typedef struct s_com_start_application_cnf {
    SYS_UWORD8 state;
} t_com_start_application_cnf;

/*
 * Read parameters:
 *  - number of words (16 bits) read,
 *  - list of words read.
 */

typedef struct s_com_read_params_cnf {
    SYS_UWORD8  number_of_params;
    SYS_UWORD16 param[COM_MAX_NUMBER_OF_16BITS_DATA];
} t_com_read_params_cnf;


/*
 * Load application
 *  - file checksum,
 *  - flash checksum
 */

typedef struct s_com_load_appli_cnf {
    SYS_UWORD16 file_checksum;
    SYS_UWORD16 RAM_checksum;
} t_com_load_appli_cnf;




/*
 * General confirmation structure.
 */

typedef struct s_com_confirmation {
    SYS_UWORD8 command_word;
    SYS_UWORD8 result;
    union u_cnf_param {
        t_com_get_monitor_id_cnf    monitor_id;
        t_com_get_flash_id_cnf      flash_id;
        t_com_get_chip_id_cnf       chip_id;
        t_com_get_board_id_cnf      board_id;
        t_com_write_data_cnf        checksum;
        t_com_start_application_cnf starting;
        t_com_read_params_cnf       params_read;
		t_com_load_appli_cnf		checksum_RAM;
    } param;
} t_com_confirmation;

extern long com_analyze_request (SYS_UWORD8 *request_received,
                                 t_com_request *request);

extern long com_analyze_confirmation (SYS_UWORD8 *confirmation_received,
                                      t_com_confirmation *confirmation);

extern long com_build_request (t_com_request *request,
                               SYS_UWORD8 *request_to_send);

extern long com_build_confirmation (t_com_confirmation *confirmation,
                                    SYS_UWORD8 *confirmation_to_send);

#endif /* __COMMAND_H__ */
