/**
 * @file	tty_env.c
 *
 * Coding of the Riviera Generic Functions,
 * except handle_message and handle_timer, which are
 * in tty_handle_message.c and tty_handle_timer.c
 *
 * @author	Frederic Turgis (f-turgis@ti.com) & Gerard Cauvy (g-cauvy@ti.com)
 * @version 0.1
 */

/*
 * History:
 *
 * 	Date       	Modification
 *  ------------------------------------
 *  01/27/2003	Create
 *
 * (C) Copyright 2003 by Texas Instruments Incorporated, All Rights Reserved
 */


#include "tty/tty_i.h"

#include "rvm/rvm_priorities.h"
#include "tty/tty_env.h"
#include <string.h>

/**
 * Pointer on the structure gathering all the global variables
 * used by TTY instance.
 */
T_TTY_ENV_CTRL_BLK *tty_env_ctrl_blk_p = NULL;


/* External declaration */
extern T_RV_RET tty_core(void);

/**
 * Called by the RV manager to learn 
 * tty requirements in terms of memory, SWEs...
 *
 * @param	swe_info	Pointer to the structure to fill
 *						containing infos related to the tty SWE.
 *
 * @return	RVM_OK
 */
T_RVM_RETURN tty_get_info(T_RVM_INFO_SWE  * swe_info)
{
	/* TTY is a Type 4 SWE. */
	swe_info->swe_type = RVM_SWE_TYPE_4;

	memcpy(swe_info->type_info.type4.swe_name, "TTY", sizeof("TTY"));
	swe_info->type_info.type4.swe_use_id = TTY_USE_ID;

	/* SWE info */
	swe_info->type_info.type4.stack_size	= TTY_STACK_SIZE;
	swe_info->type_info.type4.priority		= RVM_TTY_TASK_PRIORITY;
	swe_info->type_info.type4.version		= BUILD_VERSION_NUMBER(0,1,0);

	/* Memory bank info */
	swe_info->type_info.type4.nb_mem_bank = 1;
	memcpy(swe_info->type_info.type4.mem_bank[0].bank_name, "TTY_PRIM", sizeof("TTY_PRIM"));
	swe_info->type_info.type4.mem_bank[0].initial_params.size		= TTY_MB_PRIM_SIZE;
	swe_info->type_info.type4.mem_bank[0].initial_params.watermark	= TTY_MB_PRIM_WATERMARK;
	
	/* 
	 * Linked SWE info.
	 */
	swe_info->type_info.type4.nb_linked_swe = 0;

	/* Set the return path: NOT USED. */
	swe_info->type_info.type4.return_path.callback_func	= NULL;
	swe_info->type_info.type4.return_path.addr_id		= 0;

	/* Generic functions */
	swe_info->type_info.type4.set_info		 = tty_set_info;
	swe_info->type_info.type4.init			 = tty_init;
	swe_info->type_info.type4.core			 = tty_core;
	swe_info->type_info.type4.stop			 = tty_stop;
	swe_info->type_info.type4.kill			 = tty_kill;

	return RVM_OK;
}


/**
 * Called by the RV manager to inform the tty SWE about 
 * addr_id, return path, mb_id and error function.
 *
 * It is called only once.
 *
 * @param	addr_id			Address ID of the TTY SWE.
 *							Used to send messages to the SWE.
 * @param	return_path		Return path array of the linked SWEs.
 * @param	bk_id_table		Array of memory bank ids allocated to the SWE.
 * @param	call_back_error_ft Callback function to call in case of unrecoverable error.
 * @return	RVM_MEMORY_ERR ou RVM_OK.
 */
T_RVM_RETURN tty_set_info ( T_RVF_G_ADDR_ID		addr_id,
							T_RV_RETURN_PATH	return_path[], 
							T_RVF_MB_ID			bk_id_table[],
							T_RVM_CB_FUNC		call_back_error_ft)
{
	/* Memory bank status (red, yellow, green). */
	T_RVF_MB_STATUS mb_status;

	
	/* Create instance gathering all the variable used by TTY instance */
	mb_status = rvf_get_buf(bk_id_table[0], 
							sizeof(T_TTY_ENV_CTRL_BLK),
							(T_RVF_BUFFER**)&tty_env_ctrl_blk_p);
	if (mb_status == RVF_RED)
	{
		TTY_SEND_TRACE("TTY: Error to get memory ",RV_TRACE_LEVEL_ERROR);
		return RVM_MEMORY_ERR;
	}
	else if (mb_status == RVF_YELLOW)
	{
		/*
		 * The flag is yellow, there will soon be not enough memory anymore.
		 */
		TTY_SEND_TRACE("TTY: Getting short on memory ", RV_TRACE_LEVEL_WARNING);
	}

	/* Store the initial State; default is IDLE. */
	tty_env_ctrl_blk_p->state = TTY_IDLE;

	/* Store the address ID. */
	tty_env_ctrl_blk_p->addr_id = addr_id;

	/* Store the pointer to the error function. */
	tty_env_ctrl_blk_p->error_ft = call_back_error_ft;

	/*
	 * Store the mem bank id.
	 * Memory bank ID (mb_id) can be retrieved later using rvf_get_mb_id function.
	 */
	tty_env_ctrl_blk_p->prim_mb_id = bk_id_table[0];

	/*
	 * Array return_path: Return path of linked SWE.
	 * If the linked SWE have an API based on messages, their return path given 
	 * by the set_info function would be stored and used to communicate with them.
	 * But since the linked SWE of TTY have an API based on functions, it is not
	 * necessary to store their return path.
	 */

	return RVM_OK;
}


/**
 * Called by the RV manager to initialize the 
 * tty SWE before creating the task and calling tty_start. 
 *
 * @return	RVM_OK
 */
T_RVM_RETURN tty_init (void)
{
	/*
	 * TTY Global Initialization.
	 */

	return RVM_OK;
}


/**
 * Called by the RV manager to stop the tty SWE.
 *
 * @return	RVM_OK or RVM_INTERNAL_ERR
 */
T_RVM_RETURN tty_stop (void)
{
	/*
	 * Here we should stop the activities of the SWE
	 * It is still possible to send messages to other SWE, to unregister for example.
	 */
	TTY_SEND_TRACE("TTY: stop called", RV_TRACE_LEVEL_DEBUG_LOW);

	return RVM_OK;
}


/**
 * Called by the RV manager to kill the tty SWE,
 * after the tty_stop function has been called.
 *
 * @return	RVM_OK
 */
T_RVM_RETURN tty_kill (void)
{
	/*
	 * Here we cannot send messages anymore. We only free the last
	 * used resources, like the control block buffer.
	 */
	TTY_SEND_TRACE("TTY: kill called", RV_TRACE_LEVEL_DEBUG_LOW);

	rvf_free_buf(tty_env_ctrl_blk_p);

	return RVM_OK;
}
