/**
 * @file   mks_env.c
 *
 * Coding of the Riviera Generic Functions 
 *
 * @author   Laurent Sollier (l-sollier@ti.com)
 * @version 0.1
 */

/*
 * History:
 *
 *   Date          Author       Modification
 *  ----------------------------------------
 *  11/16/2001     L Sollier    Create
 *
 *
 * (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved
 */

#include "mks/mks_env.h"
#include "mks/mks_i.h"

#include "rvm/rvm_use_id_list.h"
#include "rvm/rvm_priorities.h"

#include <string.h>


/* External declaration */
extern T_RV_RET mks_core(void);

/**
 * Pointer on the structure gathering all the global variables
 * used by MKS instance.
 */
T_MKS_ENV_CTRL_BLK* mks_env_ctrl_blk = NULL;


/**
 * @name Riviera generic functions implementation
 *
 */
/*@{*/

/**
 * function: mks_get_info
 */
T_RVM_RETURN mks_get_info(T_RVM_INFO_SWE* swe_info)
{
   /* SWE info */
   swe_info->swe_type = RVM_SWE_TYPE_4;
   swe_info->type_info.type4.swe_use_id = MKS_USE_ID;

   memcpy( swe_info->type_info.type4.swe_name, "MKS", sizeof("MKS") );

   swe_info->type_info.type4.stack_size = MKS_STACK_SIZE;
   swe_info->type_info.type4.priority   = RVM_MKS_TASK_PRIORITY;


   /* Set the return path */
   swe_info->type_info.type4.return_path.callback_func   = NULL;
   swe_info->type_info.type4.return_path.addr_id         = 0;


   /* memory bank info */
   swe_info->type_info.type4.nb_mem_bank = 1;

   memcpy ((UINT8 *)    swe_info->type_info.type4.mem_bank[0].bank_name, "MKS_PRIM", 9);
   swe_info->type_info.type4.mem_bank[0].initial_params.size          = MKS_MB_PRIM_SIZE;
   swe_info->type_info.type4.mem_bank[0].initial_params.watermark     = MKS_MB_PRIM_WATERMARK;

   /* Linked SWE info */
   swe_info->type_info.type4.linked_swe_id[0] = KPD_USE_ID;
   swe_info->type_info.type4.nb_linked_swe = 1;

   /* generic functions */
   swe_info->type_info.type4.set_info = mks_set_info;
   swe_info->type_info.type4.init     = mks_init;
   swe_info->type_info.type4.core     = mks_core;
   swe_info->type_info.type4.stop     = mks_stop;
   swe_info->type_info.type4.kill     = mks_kill;

   return RVM_OK;
}


/**
 * function: mks_set_info
 */
T_RVM_RETURN mks_set_info (T_RVF_ADDR_ID   addr_id, 
                           T_RV_RETURN      return_path[], 
                           T_RVF_MB_ID      bk_id[],
                           T_RVM_RETURN   (*rvm_error_ft)(T_RVM_NAME       swe_name,
                                                          T_RVM_RETURN     error_cause,
                                                          T_RVM_ERROR_TYPE error_type,
                                                          T_RVM_STRING     error_msg))
{
   /* Create instance gathering all the variable used by MKS instance */
   if (rvf_get_buf(bk_id[0], 
                  sizeof(T_MKS_ENV_CTRL_BLK),
                  (T_RVF_BUFFER**)&mks_env_ctrl_blk) != RVF_GREEN)
   {
      /* The environemnt will cancel the MKS instance creation. */
      return RVM_MEMORY_ERR;
   }

   /* Store the pointer to the error function */
   mks_env_ctrl_blk->error_ft = rvm_error_ft ;
   /* Store the mem bank id. */
   mks_env_ctrl_blk->prim_id = bk_id[0];
   /* Store the address id */
   mks_env_ctrl_blk->addr_id = addr_id;

   /* return_path of linked SWE -> not used */
   return RVM_OK;
}


/**
 * function: mks_init
 */
T_RVM_RETURN mks_init(void)
{
   return RV_OK;
}


/**
 * function: mks_stop
 */
T_RVM_RETURN mks_stop(void)
{
   /* Stop MKS SWE */
   mks_stop_swe();

   return RVM_OK;
}


/**
 * function: mks_kill
 */
T_RVM_RETURN mks_kill (void)
{
   /*
    * DO NOT SEND MESSAGES
    */

   rvf_free_buf(mks_env_ctrl_blk);

   return RVM_OK;
}


/*@}*/
