/**
 * @file   lls_env.c
 *
 * Coding of the Riviera Generic Functions 
 *
 * @author   Laurent Sollier (l-sollier@ti.com)
 * @version 0.1
 */

/*
 * History:
 *
 *   Date          Author       Modification
 *  ----------------------------------------
 *  03/12/2002     L Sollier    Create
 *
 *
 * (C) Copyright 2002 by Texas Instruments Incorporated, All Rights Reserved
 */

#include "lls/lls_env.h"
#include "lls/lls_i.h"

#include "rvm/rvm_use_id_list.h"

#include <string.h>


/**
 * Pointer on the structure gathering all the global variables
 * used by LLS instance.
 */
T_LLS_ENV_CTRL_BLK* lls_env_ctrl_blk = NULL;


/**
 * @name Riviera generic functions implementation
 *
 */
/*@{*/

/**
 * function: lls_get_info
 */
T_RVM_RETURN lls_get_info(T_RVM_INFO_SWE* swe_info)
{
   /* SWE info */
   swe_info->swe_type = RVM_SWE_TYPE_1;
   swe_info->type_info.type1.swe_use_id = LLS_USE_ID;

   memcpy( swe_info->type_info.type1.swe_name, "LLS", sizeof("LLS") );

   /* Set the return path */
   swe_info->type_info.type1.return_path.callback_func   = NULL;
   swe_info->type_info.type1.return_path.addr_id         = 0;


   /* memory bank info */
   swe_info->type_info.type1.nb_mem_bank = 1;

   memcpy ((UINT8 *)    swe_info->type_info.type1.mem_bank[0].bank_name, "LLS_PRIM", 9);
   swe_info->type_info.type1.mem_bank[0].initial_params.size          = LLS_MB_PRIM_SIZE;
   swe_info->type_info.type1.mem_bank[0].initial_params.watermark     = LLS_MB_PRIM_WATERMARK;

   /* linked SWE info */
   /* this SWE does not require any SWE to run */
   swe_info->type_info.type1.nb_linked_swe = 1;
   swe_info->type_info.type1.linked_swe_id[0] = SPI_USE_ID;

   /* generic functions */
   swe_info->type_info.type1.set_info = lls_set_info;
   swe_info->type_info.type1.init     = lls_init;
   swe_info->type_info.type1.start    = lls_start;
   swe_info->type_info.type1.stop     = lls_stop;
   swe_info->type_info.type1.kill     = lls_kill;

   return RVM_OK;
}


/**
 * function: lls_set_info
 */
T_RVM_RETURN lls_set_info (T_RVF_ADDR_ID   addr_id, 
                           T_RV_RETURN     return_path[], 
                           T_RVF_MB_ID     bk_id[],
                           T_RVM_RETURN   (*rvm_error_ft)(T_RVM_NAME       swe_name,
                                                          T_RVM_RETURN     error_cause,
                                                          T_RVM_ERROR_TYPE error_type,
                                                          T_RVM_STRING     error_msg))
{
   /* Create instance gathering all the variable used by LLS instance */
   if (rvf_get_buf(bk_id[0], 
                  sizeof(T_LLS_ENV_CTRL_BLK),
                  (T_RVF_BUFFER**)&lls_env_ctrl_blk) != RVF_GREEN)
   {
      /* The environemnt will cancel the LLS instance creation. */
      return RVM_MEMORY_ERR;
   }

   /* Store the pointer to the error function */
   lls_env_ctrl_blk->error_ft = rvm_error_ft ;
   /* Store the mem bank id. */
   lls_env_ctrl_blk->prim_id = bk_id[0];
   lls_env_ctrl_blk->swe_is_initialized = FALSE;

   /* return_path of linked SWE -> not used */
   return RVM_OK;
}


/**
 * function: lls_init
 */
T_RVM_RETURN lls_init(void)
{
   if (lls_env_ctrl_blk != 0)
      lls_env_ctrl_blk->swe_is_initialized = TRUE;
   else
      return RV_INTERNAL_ERR;

   if (lls_initialize() != RV_OK)
      return RV_INTERNAL_ERR;

   return RV_OK;
}


/**
 * function: lls_start
 */
T_RVM_RETURN lls_start(void)
{
   return RV_OK;
}


/**
 * function: lls_stop
 */
T_RVM_RETURN lls_stop(void)
{

   return RVM_OK;
}


/**
 * function: lls_kill
 */
T_RVM_RETURN lls_kill (void)
{
   /*
    * DO NOT SEND MESSAGES
    */
   rvf_free_buf(lls_env_ctrl_blk);

   lls_kill_service();

   return RVM_OK;
}


/*@}*/
