/********************************************************************************/
/*                                                                              */
/*    File Name:         atp_uart_i.h                                           */
/*                                                                              */
/*    Purpose:           This header file contains the internal structures,     */
/*                       constants and prototypes related to the ATP-UART       */
/*                       interface.                                             */
/*                                                                              */
/*    Note:              none.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved.   */
/*                                                                              */
/********************************************************************************/
#ifndef _ATP_UART_I_
#define _ATP_UART_I_

#include <windows.h>
#include "atp/atp_uart_api.h"
#include "rvm/rvm_gen.h"
#include "atp/atp_messages.h"
#include "atp/atp_gsm_bt_api.h"
#include "atp/atp_gsm_gsm_api.h"


/************************************ MAILBOX ***********************************/
/*                                                                              */
/* Define the mailbox assigned to the ATP-UART interface.                       */
#define ATP_UART_MAILBOX                              (RVF_TASK_MBOX_0)

/* Define the main features of the mailbox assigned to the ATP-UART interface.  */
#define ATP_UART_ALL_EVENT_FLAGS                      (0xFFFF)
#define ATP_UART_EXPECTED_EVENT                       (RVF_TASK_MBOX_0_EVT_MASK)


/****************************** MAXIMUM PACKET SIZE *****************************/
/*                                                                              */
/* Define the maximum packet size exchanged with the AT Parser.                 */
#define ATP_UART_MAX_PACKET_SIZE                      (0x0100)


/********************************** PORT NUMBER *********************************/
/*                                                                              */
/* Define the port number dedicated to the virtual modem port opened with the   */
/* AT Parser.                                                                   */
#define ATP_UART_GSM_PORT_NB                          (0x00FF)


/*************************** CONNECTION CONTROL BLOCK ***************************/
/*                                                                              */
/* Define the default name assigned to the the COM port. Note that such a name  */
/* must be a NULL-terminated string.                                            */
#define ATP_UART_DEFAULT_COM_PORT_NAME                ("COM0")

/* Define a structure used to gather information related to the COM port as     */
/* well as the virtual modem port established with the AT Parser.               */
typedef struct
{
	T_ATP_SW_ENTITY_ID  atp_id;							/* Unique entity        */
														/* identifier assigned  */
														/* to the ATP-UART      */
														/* interface after its  */
														/* registration to the  */
														/* AT Parser.           */
	char                com_port_name[5];				/* Name associated with */
														/* the COM port.        */
	HANDLE              com_port_handle;				/* Unique handle        */
														/* assigned to the COM  */
														/* port.                */
	BOOLEAN             virtual_modem_port_established;	/* Indicate whether the */
														/* virtual modem port   */
														/* is established with  */
														/* the AT Parser.       */
	T_ATP_PORT_MODE     virtual_modem_port_mode;		/* Indicate whether the */
														/* virtual modem port   */
														/* is in 'Command' or   */
														/* 'Data' state.        */
	T_ATP_PORT_SIGNAL   control_signals;				/* RS232 control        */
														/* signals.             */
	UINT32              nb_bytes_left;					/* Number of bytes      */
														/* left.                */
	UINT8               in_buffer_p[ATP_UART_MAX_PACKET_SIZE];
} T_ATP_UART_CONN_CTRL_BLK;


/******************** GLOBAL ATP-UART INTERFACE CONTROL BLOCK *******************/
/*                                                                              */
/* Define a global structure used to gather information related to the 'Global  */
/* ATP-UART Interface Control Block', such as the address and memory bank       */
/* identifier assigned to the ATP-UART interface.                               */
typedef struct
{
	T_ATP_UART_CONN_CTRL_BLK  conn_ctrl_blk;
	T_RVF_MB_ID               mb_id;			/* 'Memory Bank' identifier     */
												/* assigned to the ATP-UART     */
												/* interface.                   */
	T_RVF_ADDR_ID             addr_id;			/* Unique address identifier    */
												/* assigned to the ATP-UART     */
												/* interface.                   */
	T_RVM_CB_FUNC             error_function_p;	/* Function to be called        */
												/* whenever any unrecoverable   */
												/* error occurs.                */
} T_ATP_UART_CTRL_BLK;

/* Define a pointer to the 'Global ATP-UART Interface Control Block'.           */
extern T_ATP_UART_CTRL_BLK  *gbl_atp_uart_ctrl_blk_p;


/****************************** UNRECOVERABLE ERROR *****************************/
/*                                                                              */
/* Define a macro used to call the function due to an unrecoverable error.      */
/*                                                                              */
/* Prototype:                                                                   */
/*   ATP_UART_UNRECOVERABLE_ERROR (T_RVM_RETURN  error_cause,                   */
/*                                 T_RVM_STRING  error_message);                */
#define ATP_UART_UNRECOVERABLE_ERROR(error_cause, \
									 error_message) \
		{ \
			(gbl_atp_uart_ctrl_blk_p->error_function_p) (ATP_UART_NAME, \
														 (error_cause), \
														 0x00000000, \
														 (error_message)); \
		}


/*************************** LOCAL FUNCTION PROTOTYPES **************************/
/*                                                                              */
/* Define the local fonction prototypes.                                        */
T_ATP_UART_ERROR_CODES atp_uart_create_com_port (T_ATP_UART_COM_PORT   com_port,
												 T_ATP_UART_BAUD_RATE  baud_rate);

T_ATP_UART_ERROR_CODES atp_uart_write_com_port (UINT8   *data_buffer_p,
												UINT32  data_size);

T_ATP_UART_ERROR_CODES atp_uart_read_com_port (UINT8   *data_buffer_p,
											   UINT32  *data_size_p);

T_ATP_UART_ERROR_CODES atp_uart_remove_com_port (void);

/********************************************************************************/

T_ATP_UART_ERROR_CODES atp_uart_handle_msg (void);

#endif
