/********************************************************************************/
/*                                                                              */
/*    File Name:         atp_uart_env.c                                         */
/*                                                                              */
/*    Purpose:           This file contains the functions used to:              */
/*                          - get some information about the ATP-UART           */
/*                            interface, such as its name or the memory         */
/*                            requirements,                                     */
/*                          - set the memory bank identifier dedicated to the   */
/*                            ATP-UART interface,                               */
/*                          - initialize the ATP-UART interface,                */
/*                          - activate the ATP-UART interface,                  */
/*                          - prepare the termination of the ATP-UART           */
/*                            interface,                                        */
/*                          - kill the ATP-UART interface.                      */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved.   */
/*                                                                              */
/********************************************************************************/
#include "atp/atp_uart_i.h"
#include "atp/atp_uart_env.h"
#include "rvm/rvm_priorities.h"
#include "rvm/rvm_use_id_list.h"


/******************** GLOBAL ATP-UART INTERFACE CONTROL BLOCK *******************/
/*                                                                              */
/* Define a pointer to the 'Global ATP-UART Interface Control Block'.           */
T_ATP_UART_CTRL_BLK  *gbl_atp_uart_ctrl_blk_p = NULL;


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_get_info                                      */
/*                                                                              */
/*    Purpose:           This function gets some information about the ATP-UART */
/*                       interface, such as its name or the required resources. */
/*                                                                              */
/*    Input Parameters:                                                         */
/*       info_software_entity_p - Points to the structure gathering information */
/*                                about the ATP-UART interface.                 */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_RVM_RETURN atp_uart_get_info (T_RVM_INFO_SWE  *info_software_entity_p)
{

/*********************** atp_uart_get_info function begins **********************/

	/* Get some information about the ATP-UART interface, such as its priority  */
	/* or the number of needed memory banks.                                    */
	info_software_entity_p->swe_type                        = RVM_SWE_TYPE_4;
	((info_software_entity_p->type_info).type4).stack_size  = ATP_UART_STACK_SIZE;
	((info_software_entity_p->type_info).type4).priority    = RVM_ATP_UART_TASK_PRIORITY;
	((info_software_entity_p->type_info).type4).nb_mem_bank = 0x01;

	/* Get the name of the ATP-UART interface.                                  */
	strcpy (((info_software_entity_p->type_info).type4).swe_name,
			ATP_UART_NAME);

	/* Get the unique entity identifier assigned to the ATP-UART interface.     */
	((info_software_entity_p->type_info).type4).swe_use_id = ATP_UART_USE_ID;

	/* Get some information about the 'Memory Bank' assigned to the ATP-UART    */
	/* interface, such as its size or the corresponding watermarks.             */
	strcpy ((((info_software_entity_p->type_info).type4).mem_bank[0]).bank_name,
			"ATP_UART");
	((((info_software_entity_p->type_info).type4).mem_bank[0]).initial_params).size      = ATP_UART_MB_PRIM_SIZE;
	((((info_software_entity_p->type_info).type4).mem_bank[0]).initial_params).watermark = ATP_UART_MB_PRIM_WATERMARK;

	/* Notify the 'Riviera Manager' of the callback functions to be used in     */
	/* order to run the ATP-UART interface.                                     */
	((info_software_entity_p->type_info).type4).set_info = atp_uart_set_info;
	((info_software_entity_p->type_info).type4).init     = atp_uart_init;
	((info_software_entity_p->type_info).type4).core     = atp_uart_core;
	((info_software_entity_p->type_info).type4).stop     = atp_uart_stop;
	((info_software_entity_p->type_info).type4).kill     = atp_uart_kill;

	/* Notify the 'Riviera Manager' of the fact that the ATP-UART interface     */
	/* requires to be layered above the AT Parser.                              */
	((info_software_entity_p->type_info).type4).nb_linked_swe    = 0x01;
	((info_software_entity_p->type_info).type4).linked_swe_id[0] = ATP_USE_ID;
	return (RVM_OK);

} /********************** End of atp_uart_get_info function *********************/


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_set_info                                      */
/*                                                                              */
/*    Purpose:           This function sets the memory bank identifiers         */
/*                       dedicated to the ATP-UART interface.                   */
/*                                                                              */
/*    Input Parameters:                                                         */
/*       addr_id            - Contains the unique address identifier assigned   */
/*                            to the ATP-UART interface.                        */
/*       return_path_p      - Points to the information regarding return paths  */
/*                            assigned to other linked software entities.       */
/*       mb_id_p            - Points to the memory bank identifiers assigned to */
/*                            the ATP-UART interface.                           */
/*       error_function_p   - Function to be called in order to report any      */
/*                            unrecoverable error.                              */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_RVM_RETURN atp_uart_set_info (T_RVF_ADDR_ID  addr_id,
								T_RV_RETURN    return_path_p[],
								T_RVF_MB_ID    *mb_id_p,
								T_RVM_CB_FUNC  error_function_p)
{
    /* Declare a local variable.                                                */
	T_RVF_MB_STATUS  mb_status = RVF_GREEN;

/*********************** atp_uart_set_info function begins **********************/

	/* Allocate memory required to store the 'Global ATP-UART Interface Control */
	/* Block'.                                                                  */
	mb_status = rvf_get_buf (mb_id_p[0],
							 sizeof (T_ATP_UART_CTRL_BLK),
							 (T_RVF_BUFFER **) (&gbl_atp_uart_ctrl_blk_p));

	/* If insufficient resources to properly run the ATP-UART interface, then   */
	/* report an internal memory error and abort.                               */
	switch (mb_status)
	{
		case RVF_GREEN:
			{

				/* Initialize the 'Global ATP-UART Interface Control Block'.    */
				memset ((T_RVF_BUFFER *) (gbl_atp_uart_ctrl_blk_p),
						0x00,
						sizeof (T_ATP_UART_CTRL_BLK));

				/* Store the memory bank identifier assigned to the ATP-UART    */
				/* interface.                                                   */
				gbl_atp_uart_ctrl_blk_p->mb_id = mb_id_p[0];

				/* Store the unique address identifier assigned to the ATP-UART */
				/* interface.                                                   */
				gbl_atp_uart_ctrl_blk_p->addr_id = addr_id;

				/* Store the function to be called whenever any unrecoverable   */
				/* error occurs.                                                */
				gbl_atp_uart_ctrl_blk_p->error_function_p = error_function_p;
				rvf_send_trace ("  ATP-UART (env). 'atp_uart_set_info' succeeded ",
								48,
								NULL_PARAM,
								RV_TRACE_LEVEL_DEBUG_LOW,
								ATP_USE_ID);
				break;
			}
		case RVF_YELLOW:
			{
				(void) rvf_free_buf ((T_RVF_BUFFER *) (gbl_atp_uart_ctrl_blk_p));
				gbl_atp_uart_ctrl_blk_p = NULL;
			}
		default:
			{
				rvf_send_trace ("  ATP-UART (env). 'atp_uart_set_info' failed ",
								45,
								NULL_PARAM,
								RV_TRACE_LEVEL_WARNING,
								ATP_USE_ID);
				return (RVM_MEMORY_ERR);
			}
	}
	return (RVM_OK);

} /********************** End of atp_uart_set_info function *********************/


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_init                                          */
/*                                                                              */
/*    Purpose:           This function initializes the ATP-UART interface.      */
/*                                                                              */
/*    Input Parameters:  None.                                                  */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_RVM_RETURN atp_uart_init (void)
{

/************************* atp_uart_init function begins ************************/

	/* Set the default name and handle assigned to the COM port. Note that the  */
	/* name must be a NULL-terminated string.                                   */
	(void) strcpy ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_name,
				   ATP_UART_DEFAULT_COM_PORT_NAME);
	(gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle = INVALID_HANDLE_VALUE;

	/* Indicate that no virtual modem port is established with the AT Parser    */
	/* for now.                                                                 */
	(gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).virtual_modem_port_established = FALSE;
	(gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).virtual_modem_port_mode        = ATP_PORT_CMD_MODE;
	(gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).control_signals                = ATP_RX_FLOW_OFF;
	rvf_send_trace ("  ATP-UART (env). 'atp_uart_init' succeeded ",
					44,
					NULL_PARAM,
					RV_TRACE_LEVEL_DEBUG_LOW,
					ATP_USE_ID);
	return (RVM_OK);

} /************************ End of atp_uart_init function ***********************/


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_core                                          */
/*                                                                              */
/*    Purpose:           This function activates the ATP-UART interface.        */
/*                                                                              */
/*    Input Parameters:  None.                                                  */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_RVM_RETURN atp_uart_core (void)
{

/************************* atp_uart_core function begins ************************/

	return (atp_uart_handle_msg ());

} /************************ End of atp_uart_core function ***********************/


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_stop                                          */
/*                                                                              */
/*    Purpose:           This function prepares the termination of the ATP-UART */
/*                       interface.                                             */
/*                                                                              */
/*    Input Parameters:  None.                                                  */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_RVM_RETURN atp_uart_stop (void)
{

/************************* atp_uart_stop function begins ************************/

	/* If the virtual modem port is opened with the AT Parser, then deactivate  */
	/* it first. If any  error occurred, then call the function used whenever   */
	/* any unrecoverable error occurs and abort.                                */
	if (((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).virtual_modem_port_established == TRUE) && \
		(atp_close_port ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).atp_id, \
						 ATP_UART_GSM_PORT_NB) != RV_OK))
	{
		ATP_UART_UNRECOVERABLE_ERROR (RVM_INTERNAL_ERR,
									  "  ATP-UART (env). 'atp_uart_stop' failed ");
		return (RVM_NOT_READY);
	}

	/* Then, if needed, deactivate the COM port currently in use. If any        */
	/* error occurred, then call the function used whenever any unrecoverable   */
	/* error occurs and abort.                                                  */
	if (((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle != INVALID_HANDLE_VALUE) && \
		(atp_uart_remove_com_port () != RV_OK))
	{
		ATP_UART_UNRECOVERABLE_ERROR (RVM_INTERNAL_ERR,
									  "  ATP-UART (env). 'atp_uart_stop' failed ");
		return (RVM_NOT_READY);
	}
	
	/* At last, withdraw the ATP-UART interface from the AT Parser. If any      */
	/* error occurred, then call the function used whenever any unrecoverable   */
	/* error occurs and abort.                                                  */
	if (atp_dereg ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).atp_id) != RV_OK)
	{
		ATP_UART_UNRECOVERABLE_ERROR (RVM_INTERNAL_ERR,
									  "  ATP-UART (env). 'atp_uart_stop' failed ");
		return (RVM_NOT_READY);
	}
	rvf_send_trace ("  ATP-UART (env). 'atp_uart_stop' succeeded ",
					44,
					NULL_PARAM,
					RV_TRACE_LEVEL_DEBUG_LOW,
					ATP_USE_ID);
	return (RVM_OK);

} /************************ End of atp_uart_stop function ***********************/


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_kill                                          */
/*                                                                              */
/*    Purpose:           This function kills the ATP-UART interface.            */
/*                                                                              */
/*    Input Parameters:  None.                                                  */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_RVM_RETURN atp_uart_kill (void)
{

/************************* atp_uart_kill function begins ************************/

	/* Free the previously allocated memory required to store the 'Global       */
	/* ATP-UART Interface Control  Block'.                                      */
	(void) rvf_free_buf ((T_RVF_BUFFER *) (gbl_atp_uart_ctrl_blk_p));
	gbl_atp_uart_ctrl_blk_p = NULL;
	rvf_send_trace ("  ATP-UART (env). 'atp_uart_kill' succeeded ",
					44,
					NULL_PARAM,
					RV_TRACE_LEVEL_DEBUG_LOW,
					ATP_USE_ID);
	return (RVM_OK);

} /************************ End of atp_uart_kill function ***********************/
