/********************************************************************************/
/*                                                                              */
/*    File Name:         atp_uart_api.c                                         */
/*                                                                              */
/*    Purpose:           This file contains the external functions related to   */
/*                       the ATP-UART interface.                                */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved.   */
/*                                                                              */
/********************************************************************************/
#include "atp/atp_uart_i.h"
#include "rvm/rvm_use_id_list.h"


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_open_com_port                                 */
/*                                                                              */
/*    Purpose:           This function instructs the ATP-UART interface to open */
/*                       the COM port.                                          */
/*                                                                              */
/*    Input Parameters:                                                         */
/*       com_port           - Contains the COM port number.                     */
/*       baud_rate          - Contains the baud rate (in bits per second).      */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_ATP_UART_ERROR_CODES atp_uart_open_com_port (T_ATP_UART_COM_PORT   com_port,
											   T_ATP_UART_BAUD_RATE  baud_rate)
{
    /* Declare a local variable.                                                */
	T_ATP_UART_OPEN_COM_PORT  *open_com_port_p = NULL;

/******************** atp_uart_open_com_port function begins ********************/

	/* Allocate memory required to instruct the ATP-UART interface to open the  */
	/* COM port. If insufficient resources, then report an internal memory      */
	/* error and abort.                                                         */
	if (rvf_get_buf (gbl_atp_uart_ctrl_blk_p->mb_id, \
					 sizeof (T_ATP_UART_OPEN_COM_PORT), \
					 (T_RVF_BUFFER **) (&open_com_port_p)) == RVF_RED)
	{
		rvf_send_trace ("  ATP-UART (api). Insufficient resources ",
						41,
						NULL_PARAM,
						RV_TRACE_LEVEL_WARNING,
						ATP_USE_ID);
		return (RV_MEMORY_ERR);
	}

	/* Fill the 'ATP-UART Open COM Port' header up.                             */
	(open_com_port_p->os_header).msg_id        = ATP_UART_OPEN_COM_PORT;
	(open_com_port_p->os_header).src_addr_id   = RVF_INVALID_ADDR_ID;
	(open_com_port_p->os_header).callback_func = NULL;

	/* Fill the 'ATP-UART Open COM Port' payload up.                            */
	open_com_port_p->com_port  = com_port;
	open_com_port_p->baud_rate = baud_rate;

	/* Send the 'ATP-UART Open COM Port' notification to the ATP-UART           */
	/* interface.                                                               */
	return (rvf_send_msg (gbl_atp_uart_ctrl_blk_p->addr_id, \
						  open_com_port_p));

} /******************* End of atp_uart_open_com_port function *******************/


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_start_gsm                                     */
/*                                                                              */
/*    Purpose:           This function instructs the ATP-UART interface to      */
/*                       initiate using the GSM protocol stack.                 */
/*                                                                              */
/*    Input Parameters:  None.                                                  */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              The GSM protocol stack is initialized by invoking the  */
/*                       following AT commands:                                 */
/*                          AT+CFUN=1 : Set Phone Functionality (Full),         */
/*                          AT+COPS=0 : Operator Selection.                     */
/*                                                                              */
/*    Revision History:                                                         */
/*       09/26/02        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_ATP_UART_ERROR_CODES atp_uart_start_gsm (void)
{
    /* Declare a local variable.                                                */
	T_ATP_UART_START_GSM  *start_gsm_p = NULL;

/********************** atp_uart_start_gsm function begins **********************/

	/* Allocate memory required to instruct the ATP-UART interface to initiate  */
	/* using the GSM protocol stack. If insufficient resources, then report an  */
	/* internal memory error and abort.                                         */
	if (rvf_get_buf (gbl_atp_uart_ctrl_blk_p->mb_id, \
					 sizeof (T_ATP_UART_START_GSM), \
					 (T_RVF_BUFFER **) (&start_gsm_p)) == RVF_RED)
	{
		rvf_send_trace ("  ATP-UART (api). Insufficient resources ",
						41,
						NULL_PARAM,
						RV_TRACE_LEVEL_WARNING,
						ATP_USE_ID);
		return (RV_MEMORY_ERR);
	}

	/* Fill the 'ATP-UART Start GSM' header up.                                 */
	(start_gsm_p->os_header).msg_id        = ATP_UART_START_GSM;
	(start_gsm_p->os_header).src_addr_id   = RVF_INVALID_ADDR_ID;
	(start_gsm_p->os_header).callback_func = NULL;

	/* Send the 'ATP-UART Start GSM' notification to the ATP-UART interface.    */
	return (rvf_send_msg (gbl_atp_uart_ctrl_blk_p->addr_id, \
						  start_gsm_p));

} /********************* End of atp_uart_start_gsm function *********************/
