/********************************************************************************/
/*                                                                              */
/*    File Name:         atp_uart.c                                             */
/*                                                                              */
/*    Purpose:           This file contains the internal functions related to   */
/*                       the ATP-UART interface and used to utilize a COM port  */
/*                       under WIN32.                                           */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       10/04/01        Pascal Pompei                                          */
/*                          - Create.                                           */
/*                                                                              */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved.   */
/*                                                                              */
/********************************************************************************/
#include "atp/atp_uart_i.h"


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_create_com_port                               */
/*                                                                              */
/*    Purpose:           This function creates a COM port.                      */
/*                                                                              */
/*    Input Parameters:                                                         */
/*       com_port           - Contains the COM port number.                     */
/*       baud_rate          - Contains the baud rate (in bits per second).      */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       02/05/01        David Lamy-Charrier                                    */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_ATP_UART_ERROR_CODES atp_uart_create_com_port (T_ATP_UART_COM_PORT   com_port,
												 T_ATP_UART_BAUD_RATE  baud_rate)
{
    /* Declare local variables.                                                 */
	DCB           control_settings;
	COMMTIMEOUTS  timeouts = {MAXDWORD, \
							  MAXDWORD, \
							  0x0000000A, \
							  0x00000000, \
							  0x00000000};	

/******************* atp_uart_create_com_port function begins *******************/

	/* First, update the name associated with the COM port.                     */
	(gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_name[3] = (char) ('0' + com_port);

	/* Then, create the COM port with the following parameters:                 */
	/*   - Read/Write access,                                                   */
	/*   - The COM port can not be shared and can not be opened again until the */
	/*     handle is closed,                                                    */
	/*   - The handle can not be inherited,                                     */
	/*   - The COM port is just opened (failed if does not exist),              */
	/*   - No specific file attributes,                                         */
	/*   - No template used.                                                    */
	(gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle = CreateFile ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_name,
																		   (GENERIC_READ | GENERIC_WRITE),
																		   0x00000000,
																		   NULL,
																		   OPEN_EXISTING,
																		   0x00000000,
																		   NULL);

	/* If any error occurred, then abort.                                       */
	if ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle == INVALID_HANDLE_VALUE)
	{
		return (RV_INVALID_PARAMETER);
	}
	
	/* Otherwise, having created the COM port, get the default setting          */
	/* information. However, if any error occurred, then remove the COM port    */
	/* and abort.                                                               */
	control_settings.DCBlength = sizeof (DCB);
	if (GetCommState ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle, \
					  &control_settings) == FALSE)
	{
		(void) atp_uart_remove_com_port ();
		return (RV_INVALID_PARAMETER);
	}

	/* Define the baud rate, as well as the DCB structure settings.             */
	control_settings.BaudRate        = baud_rate;
	control_settings.fOutxCtsFlow    = TRUE;
	control_settings.fOutxDsrFlow    = FALSE;
	control_settings.fDtrControl     = DTR_CONTROL_ENABLE;
	control_settings.fDsrSensitivity = FALSE;
	control_settings.fRtsControl     = RTS_CONTROL_HANDSHAKE;

	/* Then, configure the COM port according to the specifications of the DCB  */
	/* structure, set the timeout parameters for all read and write operations  */
	/* on the COM port and discard all characters from the output and input     */
	/* buffers. However, if any error occurred, then remove the COM port and    */
	/* abort.                                                                   */
	if ((SetCommState ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle, \
					   &control_settings) == FALSE) || \
		(SetCommTimeouts ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle, \
						  &timeouts) == FALSE) || \
		(PurgeComm ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle, \
					(PURGE_RXCLEAR | PURGE_TXCLEAR | PURGE_RXABORT | PURGE_TXABORT)) == FALSE))
	{
		(void) atp_uart_remove_com_port ();
		return (RV_INTERNAL_ERR);
	}
	return (RV_OK);

} /****************** End of atp_uart_create_com_port function ******************/


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_write_com_port                                */
/*                                                                              */
/*    Purpose:           This function writes data on the COM port.             */
/*                                                                              */
/*    Input Parameters:                                                         */
/*       data_buffer_p      - Pointer on the data to be written.                */
/*       data_size          - Contains the total byte count to be written.      */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              Buffer de-allocation is outside the scope of this      */
/*                       function.                                              */
/*                                                                              */
/*    Revision History:                                                         */
/*       02/05/01        David Lamy-Charrier                                    */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_ATP_UART_ERROR_CODES atp_uart_write_com_port (UINT8   *data_buffer_p,
												UINT32  data_size)
{
    /* Declare a local variable.                                                */
	UINT32  data_size_sent = 0x00000000;

/******************** atp_uart_write_com_port function begins *******************/

	/* Attempt writing data. If any error occurred, then abort.                 */
	if (((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle == INVALID_HANDLE_VALUE) || \
		(WriteFile ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle, \
					data_buffer_p, \
					data_size, \
					(unsigned long *) (&data_size_sent), \
					NULL) == FALSE) || \
		(data_size_sent != data_size))
	{
		return (RV_INVALID_PARAMETER);
	}
	return (RV_OK);

} /******************* End of atp_uart_write_com_port function ******************/


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_read_com_port                                 */
/*                                                                              */
/*    Purpose:           This function reads data from the COM port.            */
/*                                                                              */
/*    Input Parameters:                                                         */
/*       data_buffer_p      - Pointer on the destination buffer.                */
/*       data_size_p        - Pointer on the total byte count to be read.       */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*       data_size_p        - Contains the total byte count read.               */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Notes:             Buffer allocation is outside the scope of this         */
/*                       function Moreover, this function returns the total     */
/*                       byte count read, up to the expected value (from        */
/*                       0x00000000 to 'data_size_p').                          */
/*                                                                              */
/*    Revision History:                                                         */
/*       02/05/01        David Lamy-Charrier                                    */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_ATP_UART_ERROR_CODES atp_uart_read_com_port (UINT8   *data_buffer_p,
											   UINT32  *data_size_p)
{
    /* Declare a local variable.                                                */
	UINT32  data_size_read       = 0x00000000;
	UINT32  total_data_size_read = 0x00000000;

/******************** atp_uart_read_com_port function begins ********************/

	/* First, check whether the COM port is in use and whether the total byte   */
	/* count to be read is valid. Otherwise, abort.                             */
	if (((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle == INVALID_HANDLE_VALUE) || \
		(data_size_p == NULL))
	{
		return (RV_INVALID_PARAMETER);
	}
	
	/* Then, attempt reading data, up to the expected value. However, abort if  */
	/* no data available from the COM port or if any error occurred.            */
	do
	{
		if (ReadFile ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle, \
					  (data_buffer_p + total_data_size_read), \
					  (*data_size_p - total_data_size_read), \
					  (unsigned long *) (&data_size_read), \
					  NULL) == FALSE)
		{
			return (RV_INVALID_PARAMETER);
		}
		total_data_size_read += data_size_read;
	}
	while (data_size_read != 0x00000000);

	/* Update the total byte count read.                                        */
	*data_size_p = total_data_size_read;
	return(RV_OK);

} /******************* End of atp_uart_read_com_port function *******************/


/********************************************************************************/
/*                                                                              */
/*    Function Name:     atp_uart_remove_com_port                               */
/*                                                                              */
/*    Purpose:           This function removes the COM port.                    */
/*                                                                              */
/*    Input Parameters:  None.                                                  */
/*                                                                              */
/*    Output Parameters: None.                                                  */
/*                                                                              */
/*    Global Parameters: None.                                                  */
/*                                                                              */
/*    Note:              None.                                                  */
/*                                                                              */
/*    Revision History:                                                         */
/*       02/05/01        David Lamy-Charrier                                    */
/*                          - Create.                                           */
/*                                                                              */
/********************************************************************************/
T_ATP_UART_ERROR_CODES atp_uart_remove_com_port (void)
{

/******************* atp_uart_remove_com_port function begins *******************/

	/* Remove the COM port. If any error occurred, then abort.                  */
	if (((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle == INVALID_HANDLE_VALUE) || \
		(DeleteFile ((gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_name) == FALSE))
	{
		return (RV_INVALID_PARAMETER);
	}

	/* Set the default handle assigned to the COM port.                         */
	(gbl_atp_uart_ctrl_blk_p->conn_ctrl_blk).com_port_handle = INVALID_HANDLE_VALUE;
	return (RV_OK);

} /****************** End of atp_uart_remove_com_port function ******************/
 