/*******************************************************************************
 *
 * File Name : atp_new_i.h
 *
 * Internal definition for ATP SW entity
 *
 * (C) Texas Instruments, all rights reserved
 *
 * Version number	: 0.1      Date : 28-Feb-2000
 *
 * History			: 0.1  - Created by E. Baissus
 *
 *
 * Author           : Eric Baissus : e-baissus@ti.com
 *
 *   (C) Copyright 2000 by Texas Instruments Incorporated, All Rights Reserved
 ******************************************************************************/
#ifndef ATP_NEW_I_H
#define ATP_NEW_I_H

#include "rv/rv_general.h"
#include "rvf/rvf_api.h"
#include "atp/atp_api.h"
#include "atp/atp_messages.h"
#include "atp/atp_config.h"
#include "atp/atp_cmd.h"






/* Structure used to store information on a registered SW entity */
/*---------------------------------------------------------------*/
typedef struct
{ 
	T_ATP_SW_ENTITY_NAME sw_entity_name; // Sw entity name
	UINT8 nb_open_port; // Number of port open with the SW entity
	T_ATP_ENTITY_MODE mode; // List all the modes of the SW entity
	T_ATP_CALLBACK return_path; // return path to send event back to the SW entity
} T_ATP_SW_ENTITY_STRUCT;


/* Structure used to store information on a port */
/*---------------------------------------------------------------*/
typedef enum
{
	ATP_OPEN_PENDING,
	ATP_CMD_MODE,
	ATP_DATA_MODE
} T_ATP_PORT_STATE; // Indicate in which state is the port 


typedef struct
{
	UINT16 tx_head_size;
	UINT16 tx_trail_size;
	UINT16 rx_head_size;
	UINT16 rx_trail_size;
    T_ATP_PACKET_MODE packet_mode; // SEGMENTED means that data are stored in segmented buffer. 
									// l2cap_read_uint function must be called to fetch the data 
								    // NORMAL means that data are in a single buffer
} T_ATP_NO_COPY_PORT_INFO; // Information regarding no copy data processing


/* Every port has 2 port_end_struct : one for each SWE concerned */
#define INVALID_PORT_NB (0xFF)

typedef struct t_atp_port_end_struct 
{
	T_ATP_SW_ENTITY_ID sw_id; // SWE ID
	T_ATP_PORT_NB port_nb; // Port number 
	T_ATP_EVENT_MASK event_mask;
	T_ATP_SIGNAL_MASK signal_mask;
	T_ATP_NO_COPY_PORT_INFO no_copy_info;
	T_RVF_BUFFER_Q rx_queue; // Queue used to pipe the received packet 
							 //and wait for reading processing
	UINT32 rx_data_left; // Number of data still not read by the SW entity
	T_ATP_PORT_SIGNAL signal; // Status of the port signal
	T_RVF_MB_ID rx_mb; // MB to use when the SWE receives data from ATP 
	T_RVF_MB_ID tx_mb; // MB used by SWE when sending data to ATP
} T_ATP_PORT_END_STRUCT;


typedef UINT8 T_ATP_SW_NB; /* Which SW_ID of a port : first/initiator (0) or second(acceptor) (1) */



/* Definition of types used for handling of commands on a port */
/*-------------------------------------------------------------*/
typedef enum { READY_FOR_NEW_CMD, WAITING_FOR_RESULT } T_ATP_CMD_STATE;
	// Indicate if the port is ready to receive a new command 
	// or if it is waiting for the result of a command
typedef enum { NOT_STARTED,ON_GOING,FINISHED } T_ATP_CMD_INTERPRETATION;
	// A string can multiplex several command. If status is NOT_STARTED, the command has not
	// been yet interpreted at all by ATP. If ON_GOING, that means that interpretation started but
	// is not finished (still command pending in the string)

#define ATP_CMD_INVALID_POSITION (0xFFFF);

typedef struct
{
	T_ATP_CMD_STATE state; //READY_FOR_NEW_CMD or WAITING_FOR_RESULT
	T_ATP_CMD_INTERPRETATION status; // NOT_STARTED or ON_GOING or FINISHED
	T_ATP_TXT_CMD  cmd_txt_p; // text containing the commands : Example :'ATD1274576;' Can contains several command
	UINT16 cmd_txt_length; // length of the text  in bytes
	UINT16 next_position; // next caracter to interpret in the cmd_txt buffer
} T_ATP_CMD_STRUCT;



/* Internal struct used to store all the information related to a port */
/*---------------------------------------------------------------------*/
typedef struct t_atp_port_struct
{
	struct t_atp_port_struct * next_p; // points on next port structure/ otherwise = NULL
	T_ATP_PORT_STATE port_state; //	ATP_OPEN_PENDING or ATP_CMD_MODE or ATP_DATA_MODE
	T_ATP_PORT_END_STRUCT port_info[2]; //[0]=initiator side / [1]=target side
	T_ATP_REDIRECT_MODE redirect_mode; /* Indicates  if redirect flow is ON or OFF */
	struct t_atp_port_struct * redirect_port_p; /* pointing on the port of redirection */
	T_ATP_SW_NB redirect_port_end_nb; /* Indicates to which end of the redirecting port the flow 
						should be redirected = value = 0 or 1 */
	T_ATP_CMD_STRUCT cmd_info; // Information related to CMD handling
	T_ATP_DCE_INFO * dce_info_p; // Information related to DCE handling
} T_ATP_PORT_STRUCT;


typedef struct
{
	UINT32 first_byte;
	UINT32 last_byte;
	UINT32 next_byte_to_read;
	T_ATP_BUFFER atp_buffer_p;
} T_ATP_RX_PACKET;


/* Defines the state of the ATP SWE  (cf Riviera Manager) */
typedef enum
{
	ATP_STARTED,
	ATP_STOPPED,
	ATP_KILLED
} T_ATP_SWE_STATE;


/* Types of error handels by atp_error function */
typedef enum
{
	ATP_ERROR_MB_PRIM_RED,
	ATP_ERROR_TX_MB_RED,
	ATP_ERROR_WRONG_CALLBACK /* The callback function of the primitive has been used where as */
							  /* Mailbox should be used */
} T_ATP_ERROR_REASON;



extern T_RVF_MB_ID atp_mb_prim; // Memory bank dedicated to internal use of ATP entity
// Table gathering the pointers on the different SW entity data structure. If NULL, field is free
extern T_ATP_PORT_STRUCT * atp_first_port_p;
extern T_ATP_SW_ENTITY_ID atp_nb_sw_entity;
extern T_ATP_SWE_STATE atp_swe_state;
extern T_ATP_SW_ENTITY_STRUCT * atp_sw_entity_table_p[ATP_MAX_NB_SW_ENTITY+1]; //+1 due to GSM SW entity 


/* Registration / re-registration */
T_ATP_RET atp_dereg_all(void);
T_ATP_RET atp_init_cmd_info_struct(T_ATP_PORT_STRUCT * port_p);

/* Port handling related functions -> atp_port.c */
T_ATP_RET atp_create_port(T_ATP_PORT_STRUCT **port_pp);
T_ATP_RET atp_delete_port(T_ATP_PORT_STRUCT *port_p);
T_ATP_RET atp_get_port(T_ATP_SW_ENTITY_ID sw_id, T_ATP_PORT_NB port_nb,T_ATP_PORT_STRUCT ** port_found_pp,T_ATP_SW_NB * sw_nb_p);
T_ATP_RET atp_send_message(T_ATP_CALLBACK path,T_ATP_MESSAGE * message_p);


/* Commands handling related functions -> atp_cmd.c */
T_ATP_RET atp_cmd_init_dce_info(T_ATP_DCE_INFO * dce_info_p); // initialise dce buffer */
T_ATP_RET atp_interpret_raw_data (T_ATP_PORT_STRUCT	*port_p,
								  T_RVF_MB_ID		mb_id,
								  T_ATP_CMD_TYPE	*cmd_type_p,
								  T_ATP_CMD_NB		*cmd_nb_p,
								  T_ATP_CMD			**cmd_info_pp,
								  T_ATP_TXT_CMD		*text_pp,
								  UINT16			*text_length_p);

T_ATP_RET atp_create_data_buffer_from_cmd (T_ATP_CMD_MODE  cmd_mode,
										   UINT16          header,
										   UINT16          trailer,
										   T_ATP_DCE_INFO  *dce_info_p,
										   T_RVF_MB_ID     mb_id,
										   T_ATP_CMD_TYPE  cmd_type,
										   T_ATP_CMD_NB    cmd_nb,
										   T_ATP_TXT_CMD   text_p,
										   T_ATP_CMD       *cmd_info_p,
										   T_ATP_BUFFER    *buffer_pp,
										   UINT16          *length_p);

T_ATP_RET atp_translate_cmd_to_txt (T_ATP_CMD_TYPE  cmd_type,
									T_ATP_CMD_NB    cmd_nb,
									T_ATP_CMD       *cmd_info_p,
									T_RVF_MB_ID     mb_id,
									T_ATP_TXT_CMD   *text_pp,
									UINT16          *text_length_p);


T_ATP_RET atp_copy_buffer_from_l2cap (void * l2cap_buffer_p, void * copy_buffer_p,
									  UINT32 buffer_length);
T_ATP_RET atp_copy_buffer (UINT8   *in_buffer_p,
						   UINT8   *out_buffer_p,
						   UINT32  data_size);

/* Error functions -> atp_env.c */
void atp_error(T_ATP_ERROR_REASON atp_error_reason);

#endif
