/*******************************************************************************
 *
 * File Name : atp_env.c
 *
 * Generic functions of ATP definition
 *
 * (C) Texas Instruments, all rights reserved
 *
 * Version number	: 0.1      Date : 28-Feb-2000
 *
 * History			: 0.1  - Created by E. Baissus
 *					: 0.9 (3-May-2000) : reviewed 
 *
 * Author			: Eric Baissus : e-baissus@ti.com
 *
 *   (C) Copyright 2000 by Texas Instruments Incorporated, All Rights Reserved				
 ******************************************************************************/

//#include "rv_general.h"
#include "rvf/rvf_api.h"
#include "atp/atp_api.h"
#include "atp/atp_i.h"
#include "atp/atp_config.h"
#include "atp/atp_env.h"
#include "rvm/rvm_priorities.h"
#include "rvm/rvm_gen.h"
#include "rvm/rvm_use_id_list.h"

#include <string.h>


/* Memory bank dedicated to internal use of ATP entity (initiated in atp_init) */
T_RVF_MB_ID atp_mb_prim; 
T_RVM_RETURN (*atp_error_ft)(T_RVM_NAME swe_name, T_RVM_RETURN error_cause, \
					T_RVM_ERROR_TYPE error_type,T_RVM_STRING error_msg);

T_ATP_SWE_STATE atp_swe_state = ATP_STOPPED;


/******************************************************************************
* Function name: atp_get_info
*
* Description : Generic ATP get info  function : provide general information 
*				regarding the ATP SWE to the Riviera Manager
*				For more details, see Riviera Manager documentation
*
* Parameters :  Pointer on the info structure
*
* Return     :   Always OK 
*
* History			: 0.1 (30-May-2000) - Created
*
******************************************************************************/
T_RVM_RETURN atp_get_info (T_RVM_INFO_SWE * swe_info_p)
{
	if (swe_info_p ==  NULL)
	{
		return RVM_INVALID_PARAMETER;
	}

	/* Provide SWE USEID and type */
	swe_info_p->swe_type = RVM_SWE_TYPE_1;
	swe_info_p->type_info.type1.swe_use_id = ATP_USE_ID;
	strcpy (swe_info_p->type_info.type1.swe_name,
			"ATP");

	/* Return Path of ATP */
	swe_info_p->type_info.type1.return_path.callback_func = NULL;
	swe_info_p->type_info.type1.return_path.addr_id = 0;


	/* ATP MB */
	swe_info_p->type_info.type1.nb_mem_bank = 1;
	memcpy (swe_info_p->type_info.type1.mem_bank[0].bank_name,
			"ATP_PRIM",
			sizeof("ATP_PRIM"));
	swe_info_p->type_info.type1.mem_bank[0].initial_params.size		 = ATP_MB_PRIM_SIZE;
	swe_info_p->type_info.type1.mem_bank[0].initial_params.watermark = ATP_MB_PRIM_WATERMARK;

	/* ATP generic RVM functions */
	swe_info_p->type_info.type1.init     = &atp_init;
	swe_info_p->type_info.type1.start    = &atp_start;
	swe_info_p->type_info.type1.stop     = &atp_stop;
	swe_info_p->type_info.type1.kill     = &atp_kill;
	swe_info_p->type_info.type1.set_info = &atp_set_info;

	/* Linked SW entities : ATP does not need any other entity to run 
	Actually, other SWE will register to ATP before using its services */
	swe_info_p->type_info.type1.nb_linked_swe = 0;

	return RVM_OK;
}



/******************************************************************************
* Function name: atp_set_info
*
* Description : Generic ATP set info  function : environment use it to set a 
*               a certain number of information for ATP
*				For more details, see Riviera Manager documentation
*
* Parameters :  
*
* Return     :   Always OK 
*
* History			: 0.1 (30-May-2000) - Created
*
******************************************************************************/
T_RVM_RETURN
atp_set_info(T_RVF_ADDR_ID addr_id,
             T_RV_RETURN   return_path[],
             T_RVF_MB_ID   mb_id[],
             T_RVM_RETURN  (*rvm_error_ft) ( T_RVM_NAME swe_name,
                                             T_RVM_RETURN error_cause,
                                             T_RVM_ERROR_TYPE error_type,
                                             T_RVM_STRING error_msg) )
{

	/* Save the parameters given by environment */

	/* Bank Ids */
	atp_mb_prim = mb_id [0];

	/* Error function */
	atp_error_ft = rvm_error_ft;

	return RVM_OK;
}






/******************************************************************************
 * Function name: atp_init
 *
 * Description : This function is used to initialise the ATP entity
 *
 * Parameters : none
 *
 * Return     :   Standard error 
 *				  RV_OK 
 *
 * History			: 0.1 (29-Feb-2000) 
 *					: 0.9 (3-May-2000) : reviewed
 *					: 1.0 (30-May-2000) updated to support RVM 
 ******************************************************************************/
T_RVM_RETURN atp_init(void)
{
	UINT8 i;


	// Initialise global variables  used by ATP 
	atp_nb_sw_entity=0; // Number of SW entity which has been registered into ATP excluding GSM
	atp_first_port_p = NULL; // No port instance has been created yet

	// Initialise SW entity list table
	for(i=1;i<(ATP_MAX_NB_SW_ENTITY+1);i++)
	{
		atp_sw_entity_table_p[i]=NULL;
	}
	
	atp_swe_state=ATP_STARTED ; 
	
	rvf_send_trace("ATP : Initialisation performed ",31,NULL_PARAM,RV_TRACE_LEVEL_DEBUG_LOW, 
		ATP_USE_ID); 
	
	return RVM_OK;	
}






/******************************************************************************
 * Function name: atp_start
 *
 * Description : This function is used to start ATP (used by RVM)
 *
 * Parameters : none
 *
 * Return     :    
 *				  RVM_OK 
 *
 * History			
 *					: 1.0 (30-May-2000) created
 ******************************************************************************/
T_RVM_RETURN atp_start (void)
{

	atp_swe_state=ATP_STARTED;

	rvf_send_trace("ATP : Start performed ",31,NULL_PARAM,RV_TRACE_LEVEL_DEBUG_LOW, 
		ATP_USE_ID); 

	return RVM_OK;
}




/******************************************************************************
 * Function name: atp_stop
 *
 * Description : This function is used to stop ATP (used by RVM)
 *
 * Parameters : none
 *
 * Return     :   
 *				  RVM_OK 
 *
 * History			
 *					: 1.0 (30-May-2000) created
 ******************************************************************************/
T_RVM_RETURN atp_stop (void)
{

	atp_swe_state=ATP_STOPPED;

	return RVM_OK;
}




/******************************************************************************
 * Function name: atp_kill
 *
 * Description : This function is used to kill ATP (used by RVM)
 *
 * Parameters : none
 *
 * Return     :   
 *				  RVM_OK 
 *
 * History			
 *					: 1.0 (30-May-2000) created
 ******************************************************************************/
T_RVM_RETURN atp_kill (void)
{
	if (atp_delete_all_port() ==RV_OK)
	{
		return atp_dereg_all();
	}
	return RVM_INTERNAL_ERR;
}




/******************************************************************************
* Function name: atp_error
*
* Description : Generic ATP error function
*
* Parameters :  
*
* Return     :   no return
*
* History			: 0.1 (1-Marsh-2000) - Created
*
******************************************************************************/
void atp_error(T_ATP_ERROR_REASON atp_error_reason)
{
	
	switch (atp_error_reason)
	{
	case ATP_ERROR_MB_PRIM_RED:
		{
			
			rvf_send_trace("ATP : ATP MEMORY BANK CRASHED !!!!!!!",37,NULL_PARAM,RV_TRACE_LEVEL_ERROR, 
				ATP_USE_ID); 
			
			atp_error_ft("ATP",RVM_MEMORY_ERR,0," ATP memory bank irrecoverably crashed  ");
			break;
		}
		
	default:
		{
			rvf_send_trace("ATP : Function ATP error has been called",40,NULL_PARAM,
				RV_TRACE_LEVEL_ERROR, ATP_USE_ID); 
			
			atp_error_ft("ATP",RVM_INTERNAL_ERR,0," ATP irrecoverable error occured ");
			break;
		}		
	}	
}
	
	
	
	
