/********************************************************************************/
/*                                                                              */
/*  File Name:  audio_test_regr.c                                               */
/*                                                                              */
/*  Purpose:  This file gathers non regression tests for AUDIO.                 */
/*                                                                              */
/*  Note:   None.                                                               */
/*                                                                              */
/*  Version   0.1                                                               */
/*                                                                              */
/*  Date        Modification                                                    */
/*  ----------------------------------------------------------------------------*/
/*  14 May 2001  Create                                                         */
/*                                                                              */
/*  Author      Francois Mazard - Stephanie Gerthoux                            */
/*                                                                              */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved    */
/********************************************************************************/

/********************************************************************************/
/*                                                                              */
/*  Include files used for Audio riviera testing.                               */
/*                                                                              */
/********************************************************************************/

#include "rvf/rvf_api.h"
#include "tests/rv/rv_test_filter.h"

#if ((AUDIO_REGR == SW_COMPILED) || (AUDIO_MISC == SW_COMPILED))
  #ifndef _WINDOWS
    #include "config/swconfig.cfg"
    #include "config/sys.cfg"
    #include "config/l1sw.cfg"
    #include "config/chipset.cfg"
  #endif

  #include "l1_confg.h"

  #if (L1_GTT == 1)
    #include "l1gtt_signa.h"
    #include "tty/tty_api.h"
  #endif

  #include "rv/rv_general.h"
  #include "rvm/rvm_gen.h"
  #include "audio/audio_ffs_i.h"
  #include "audio/audio_api.h"
  #include "audio/audio_structs_i.h"
  #include "audio/audio_var_i.h"
  #include "audio/audio_messages_i.h"
  #include "rvf/rvf_target.h"
  #include "audio/audio_const_i.h"
  #include "audio/audio_error_hdlr_i.h"
  #include "audio/audio_macro_i.h"
  #include "audio/tests/audio_test_regr.h"
  #ifdef _WINDOWS
    /* include the usefull L1 header */
    #define BOOL_FLAG
    #include "l1_types.h"
    #include "l1audio_const.h"
    #include "l1audio_cust.h"
    #include "l1audio_defty.h"
    #include "l1audio_msgty.h"
    #include "l1audio_signa.h"
    #include "l1_const.h"
    #include "l1_defty.h"
    #include "l1_msgty.h"
    #include "l1_signa.h"
    #include "l1_varex.h"
  #endif
  #include "audio/tests/audio_test.h"

  #include <string.h>


  /********************************************************************************/
  /*                                                                              */
  /*  Declare function prototype used for Riviera AUDIO testing.                  */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_test_regr_return_verdict ( T_RV_REGR_ERR_TYPE error_type);

  void *audio_test_regr_wait_message(UINT16              message_type,
                                     T_RV_REGR_ERR_TYPE  *error_type);

  void *audio_test_regr_wait_message_timeout(UINT16              message_type,
                                             T_RV_REGR_ERR_TYPE  *error_type,
                                             UINT32              timeout);

  T_RV_RET audio_test_regr_send_message(UINT16              message_type,
                                        T_RV_REGR_ERR_TYPE  *error_type,
                                        UINT32              data);

    void audio_keybeep_generation( void );

#if (KEYBEEP)
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO KEYBEEP CALLBACK FUNCTION                   */
  /*                                                                              */
  /* Decription: call back funtion used for the keybeep test.                     */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_keybeep_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:KEYBEEP CALLBACK FUNCTION WITH A WRONG STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:KEYBEEP CALLBACK FUNCTION WITH A GOOD STATUS");
    }
  }

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 1                       */
  /*                                                                              */
  /* Description: Start a Keybeep and wait until it stopped automatically.        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_1 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_KEYBEEP_REQ *)p_message)->d_dur_kb   != 50)      ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt0 != 0xb512) ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt1 != 0x614a) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_1 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 2                       */
  /*                                                                              */
  /* Decription: Start a Keybeep and stop it after the L1 keybeep start           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_2 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_KEYBEEP_REQ *)p_message)->d_dur_kb   != 50)      ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt0 != 0xb512) ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt1 != 0x614a) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }


      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(500));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the keybeep with no error parameters                                    */
    /*------------------------------------------------------------------------------*/
    if ( audio_keybeep_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_2 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 3                       */
  /*                                                                              */
  /* Decription: Start a Keybeep and stop it before the L1 keybeep start          */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_3 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_KEYBEEP_REQ *)p_message)->d_dur_kb   != 50)      ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt0 != 0xb512) ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt1 != 0x614a) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }


      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif
    /*------------------------------------------------------------------------------*/
    /* stop the keybeep with no error parameters                                    */
    /*------------------------------------------------------------------------------*/
    if ( audio_keybeep_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_3 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 4                       */
  /*                                                                              */
  /* Decription: Start twice the Keybeep                                          */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_4 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_KEYBEEP_REQ *)p_message)->d_dur_kb   != 50)      ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt0 != 0xb512) ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt1 != 0x614a) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }


      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(500));
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_4 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 5                       */
  /*                                                                              */
  /* Decription: Start the keybeep and send 2 stop                                */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_5 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_KEYBEEP_REQ *)p_message)->d_dur_kb   != 50)     ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt0 != 0xb512) ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt1 != 0x614a) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the keybeep with no error parameters                                    */
    /*------------------------------------------------------------------------------*/
    if ( audio_keybeep_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the keybeep with no error parameters                                    */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    if ( audio_keybeep_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);

  }
  /******************** End of audio_regr_test_5 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 6                       */
  /*                                                                              */
  /* Decription: Check the parameters error manager                 .             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_6 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    T_RV_RETURN               return_path;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with  error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 2001; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with  error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 2000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 2500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with  error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 2000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = 0; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with  error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = 1; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_6 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 7                       */
  /*                                                                              */
  /* Decription: Start a Keybeep and wait until it stopped automatically.         */
  /*             At this moment, the stop is requested by the entity              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_7 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_KEYBEEP_REQ *)p_message)->d_dur_kb   != 50)      ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt0 != 0xb512) ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt1 != 0x614a) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }


      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* stop the keybeep with no error parameters                                    */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    if ( audio_keybeep_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);


    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_7 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 8                       */
  /*                                                                              */
  /* Decription: Start a Keybeep and wait until it stopped automatically.         */
  /*             Note the stop return is made by a call back function.            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_8 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
//	UINT8					  i;

    p_audio_test->audio_call_back_called = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id       = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_KEYBEEP_REQ *)p_message)->d_dur_kb   != 50)      ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt0 != 0xb512) ||
           (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt1 != 0x614a) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }


      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* stop the keybeep with no error parameters                                    */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = audio_keybeep_callback_test_function;
    return_path.addr_id       = 0;

    if ( audio_keybeep_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the call back function is called                                        */
    /*------------------------------------------------------------------------------*/
    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_8 function ************************/

#endif /* KEYBEEP */
#if (TONE)
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO TONES CALLBACK FUNCTION                     */
  /*                                                                              */
  /* Decription: call back funtion used for the tones test.                       */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_tones_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:TONES CALLBACK FUNCTION WITH A WRONG STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:TONES CALLBACK FUNCTION WITH A GOOD STATUS");
    }
  }

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 10                      */
  /*                                                                              */
  /* Description: Start a Tones and wait until it stopped automatically.          */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_10 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_TONES_PARAMETER   tones_parameter ;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_rep  != 0x000a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_se_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_bu_off  != 8 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_on   != 0 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_off  != 3 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_on   != 1 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_off  != 4 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_on   != 5 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(3000));
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_10 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 11                      */
  /*                                                                              */
  /* Description: Start a Tones and stop it when the L1 is started.    .          */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_11 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_TONES_PARAMETER   tones_parameter ;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = TONE_INFINITE;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_rep  != 0xffff) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_se_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_bu_off  != 8 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_on   != 0 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_off  != 3 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_on   != 1 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_off  != 4 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_on   != 5 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(5000));
    #endif
    /*------------------------------------------------------------------------------*/
    /* Request to stop the tones generation                                         */
    /*------------------------------------------------------------------------------*/
    if ( audio_tones_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
       return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_11 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 12                      */
  /*                                                                              */
  /* Decription: Start a Tones and stop it before the L1 starts                   */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_12 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_TONES_PARAMETER   tones_parameter ;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = TONE_INFINITE;  /* infinite tones */

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* Request to stop the tones generation                                         */
    /*------------------------------------------------------------------------------*/
    if ( audio_tones_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_rep  != 0xffff) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_se_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_bu_off  != 8 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_on   != 0 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_off  != 3 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_on   != 1 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_off  != 4 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_on   != 5 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_12 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 13                      */
  /*                                                                              */
  /* Decription: Start twice a Tones and wait until it stopped automatically.     */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_13 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_TONES_PARAMETER   tones_parameter ;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the tones*/
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_rep  != 0x000a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_se_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_bu_off  != 8 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_on   != 0 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_off  != 3 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_on   != 1 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_off  != 4 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_on   != 5 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #else
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif
    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_13 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 14                      */
  /*                                                                              */
  /* Decription: Start a Tones and stop twice                    .    .           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_14 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_TONES_PARAMETER   tones_parameter ;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = TONE_INFINITE;

    /* start the tones*/
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_rep  != 0xffff) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_se_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_bu_off  != 8 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_on   != 0 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_off  != 3 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_on   != 1 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_off  != 4 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_on   != 5 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);


      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(3000));
    #endif
    /*------------------------------------------------------------------------------*/
    /* Request to stop the tones generation                                         */
    /*------------------------------------------------------------------------------*/
    if ( audio_tones_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    /*------------------------------------------------------------------------------*/
    /* Request to stop the tones generation                                         */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    if ( audio_tones_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_14 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 15                      */
  /*                                                                              */
  /* Decription: Check the parameters error manager                  .            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_15 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_TONES_PARAMETER   tones_parameter ;
    T_RV_RETURN               return_path;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 100;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 1;    /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 1;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 40;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 3500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 1;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 100;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = 24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 1;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 10;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 321;  /* ms */
    tones_parameter.sequence_duration        = 420;  /* ms */
    tones_parameter.period_duration          = 523;  /* ms */
    tones_parameter.repetition               = 1;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 10;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 123;  /* ms */
    tones_parameter.repetition               = 1;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_15 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 16                      */
  /*                                                                              */
  /* Description: Start a Tones and wait until it stopped automatically.          */
  /*             And after, the entity resquest to stop the tones.                */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_16 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_TONES_PARAMETER   tones_parameter ;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_rep  != 0x000a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_se_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_bu_off  != 8 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_on   != 0 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_off  != 3 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_on   != 1 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_off  != 4 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_on   != 5 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(3000));
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* Request to stop the tones generation                                         */
    /*------------------------------------------------------------------------------*/
    if ( audio_tones_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);



    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_16 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 17                      */
  /*                                                                              */
  /* Decription: Start a Tones and wait until it stopped automatically.           */
  /*             After, the stop tone is requested and the status is returnd by   */
  /*             a callback function.                                             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_17 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_TONES_PARAMETER   tones_parameter ;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    p_audio_test->audio_call_back_called = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_rep  != 0x000a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_se_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_bu_off  != 8 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_on   != 0 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_off  != 3 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_on   != 1 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_off  != 4 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_on   != 5 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(3000));
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* stop the tones with no error parameters                                    */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = audio_tones_callback_test_function;
    return_path.addr_id = 0;

    if ( audio_tones_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the call back function is called                                        */
    /*------------------------------------------------------------------------------*/
    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_17 function ***********************/
#endif /* TONE */

#if (MELODY_E1)
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO MELODY_E1 CALLBACK FUNCTION                 */
  /*                                                                              */
  /* Decription: call back funtion used for the melody E1 test.                   */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_melody_E1_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:MELODY CALLBACK FUNCTION WITH A WRONG STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:MELODY CALLBACK FUNCTION WITH A GOOD STATUS");
    }
  }
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 39                      */
  /*                                                                              */
  /* Decription: Start a Melody E1 in loopback mode and stop it after the         */
  /* L1 Melody_E1 start                                                           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_39 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_AUDIO_MELODY_E1_STOP_PARAMETER  melody_E1_stop_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameter */
    melody_E1_parameter.loopback = AUDIO_MELODY_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/LASER");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0x0F ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(120000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the melody                                                              */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E1_stop_parameter.melody_name,"/melo/LASER");

    if ( audio_melody_E1_stop(&melody_E1_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_39 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 18                      */
  /*                                                                              */
  /* Decription: Check the parameters error manager                  .            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_18 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_RV_RETURN                 return_path;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = 10;
    strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_NORMAL_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = 10;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the melody E1                                                        */
      /*----------------------------------------------------------------------------*/
      /* fill the melody E1 parameters */
      melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
      strcpy(melody_E1_parameter.melody_name,"/melo/AFR");
      melody_E1_parameter.melody_mode = AUDIO_MELODY_NORMAL_MODE;

      /* start the melody_E1 */
      if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_ERROR)
      {
        *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_18 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 20                      */
  /*                                                                              */
  /* Description: Start a Melody_E1 and wait until it stopped automatically.      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_20 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_NORMAL_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0xFF ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      #else
        rvf_delay(RVF_MS_TO_TICKS(60000));
      #endif


      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_20 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 21                      */
  /*                                                                              */
  /* Description: Start a Melody E1 and stop it after the L1 Melody_E1 start      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_21 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_AUDIO_MELODY_E1_STOP_PARAMETER  melody_E1_stop_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameter */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0x0F ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the melody                                                              */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E1_stop_parameter.melody_name,"/melo/USA");

    if ( audio_melody_E1_stop(&melody_E1_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_21 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 22                      */
  /*                                                                              */
  /* Decription: Start a Melody_E1 and stop it before the L1 Melody_E1 start      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_22 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_AUDIO_MELODY_E1_STOP_PARAMETER  melody_E1_stop_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the Melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameter */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0x0F ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif
    /*------------------------------------------------------------------------------*/
    /* stop the Melody_E1 with no error parameters                                  */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E1_stop_parameter.melody_name,"/melo/USA");

    if ( audio_melody_E1_stop(&melody_E1_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_22 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 23                      */
  /*                                                                              */
  /* Decription: Start a Melody_E1 and stop it before the FFS initialization      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_23 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_AUDIO_MELODY_E1_STOP_PARAMETER  melody_E1_stop_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the Melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameter */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* wait before to stop the melody                                               */
    /*------------------------------------------------------------------------------*/
    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif


    /*------------------------------------------------------------------------------*/
    /* stop the Melody_E1 with no error parameters                                  */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E1_stop_parameter.melody_name,"/melo/USA");

    if ( audio_melody_E1_stop(&melody_E1_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_23 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 24                      */
  /*                                                                              */
  /* Decription: Start twice the Melody_E1                                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/

  T_RV_REGR_RET audio_regr_test_24 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0x0F ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(3000));
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the melody E1 parameter */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/LASER");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY1_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0xF0 ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 1 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY1_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY1_STOP_CON, error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* This delay is introduced in order to receive the melody1 end status before the */
      /* melody0 end status. */
      rvf_delay(10);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(15000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(60000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }


  /******************** End of audio_regr_test_24 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 25                      */
  /*                                                                              */
  /* Decription: Start the melody and send 2 stop                                 */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_25 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_AUDIO_MELODY_E1_STOP_PARAMETER  melody_E1_stop_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameter */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
        strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0x0F ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif


    /*------------------------------------------------------------------------------*/
    /* stop the melody                                                              */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E1_stop_parameter.melody_name,"/melo/USA");

    if ( audio_melody_E1_stop(&melody_E1_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #else
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the melody E1                                                           */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    strcpy(melody_E1_stop_parameter.melody_name,"/melo/LASER");

    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    if ( audio_melody_E1_stop(&melody_E1_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif
    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_25 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 26                      */
  /*                                                                              */
  /* Decription: Start a Melody and wait until it stopped automatically.          */
  /*             At this moment, the stop is requested by the entity              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_26 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_AUDIO_MELODY_E1_STOP_PARAMETER  melody_E1_stop_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/LASER");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0x0F ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif


      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* stop the melody                                                              */
      /*------------------------------------------------------------------------------*/
      /* initalize the return path, for this test the return path is a message */
      strcpy(melody_E1_stop_parameter.melody_name,"/melo/LASER");

      return_path.callback_func = NULL;
      return_path.addr_id = rvf_get_taskid();

      if ( audio_melody_E1_stop(&melody_E1_stop_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      #ifndef _WINDOWS
        rvf_delay(RVF_MS_TO_TICKS(60000));
      #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_ERROR)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_26 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 27                      */
  /*                                                                              */
  /* Decription: Start a Melody and wait until it stopped automatically.         */
  /*             Note the stop return is made by a call back function.            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_27 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_AUDIO_MELODY_E1_STOP_PARAMETER  melody_E1_stop_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    p_audio_test->audio_call_back_called = 1;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id       = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
        strcpy(melody_E1_parameter.melody_name,"/melo/LASER");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0x0F ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(50000));
    #endif


      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* stop the melody E1                                                           */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E1_stop_parameter.melody_name,"/melo/LASER");

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = audio_melody_E1_callback_test_function;
    return_path.addr_id = 0;

    if ( audio_melody_E1_stop(&melody_E1_stop_parameter,return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the call back function is called                                        */
    /*------------------------------------------------------------------------------*/

    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_27 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 28                      */
  /*                                                                              */
  /* Decription: 3 start of the Melody_E1                                         */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/

  T_RV_REGR_RET audio_regr_test_28 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;
//    T_AUDIO_MELODY_E1_STOP_PARAMETER melody_E1_stop_parameter;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
        strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0x0F ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(3000));
    #endif


    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the melody E1 parameter */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
        strcpy(melody_E1_parameter.melody_name,"/melo/LASER");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the melody E1 parameter */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
        strcpy(melody_E1_parameter.melody_name,"/melo/LASER");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY1_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0xF0 ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 1 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY1_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY1_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #else
      rvf_delay(RVF_MS_TO_TICKS(60000));
    #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);


      return(TEST_PASSED);
  }

  /******************** End of audio_regr_test_28 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 29                      */
  /*                                                                              */
  /* Decription: Start twice the Melody_E1 in normal mode.                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/

  T_RV_REGR_RET audio_regr_test_29 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/LASER");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_NORMAL_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0xFF ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif


    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the melody E1 parameter */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
        strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_NORMAL_MODE;

    /* start the melody */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_MODE_ERROR)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(5000));
    #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      return(TEST_PASSED);
  }


  /******************** End of audio_regr_test_29 function ************************/
#endif /* MELODY_E1 */

#if (VOICE_MEMO)
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO VOICE MEMO CALLBACK FUNCTION                */
  /*                                                                              */
  /* Decription: call back funtion used for the voice memo test.                  */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_vm_play_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR: VOICE MEMO PLAY CALLBACK FUNCTION WITH A WRONG STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:VOICE MEMO PLAY CALLBACK FUNCTION WITH A GOOD STATUS");
    }
  }

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 30                      */
  /*                                                                              */
  /* Decription: Start a Voice Memorization play and wait until it stopped        */
  /*             automatically.                                                   */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_30 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_PLAY_PARAMETER   vm_play_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo play                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo play parameters */
    strcpy(vm_play_parameter.memo_name,"/memo/voice");

    /* start the voice memo play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_VM_PLAY_REQ *)p_message)->session_id != 0 )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      #else
        rvf_delay(RVF_MS_TO_TICKS(10000));
      #endif


      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_30 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 31                      */
  /*                                                                              */
  /* Decription: Start a Voice Memo Play and stop it after the L1 Voive Memo      */
  /*             Play start                                                       */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_31 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_PLAY_PARAMETER vm_play_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the Voice Memo Playing task                                          */
    /*----------------------------------------------------------------------------*/
    /* fill the Voice Memo Play parameters */
    strcpy(vm_play_parameter.memo_name,"/memo/voice");

    /* start the Voice memo Play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ((T_MMI_VM_PLAY_REQ *)p_message)->session_id != 0)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(7000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the voice memo play                                                     */
    /*------------------------------------------------------------------------------*/
    if ( audio_vm_play_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_31 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 32                      */
  /*                                                                              */
  /* Decription:Start a Voice Memo Play and stop it before the L1 Voice_memo start*/
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_32 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_PLAY_PARAMETER vm_play_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the Voice memo Play                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill the Voice memo Play parameter */
    strcpy(vm_play_parameter.memo_name,"/memo/voice");

    /* start the Voice memo Play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ((T_MMI_VM_PLAY_REQ *)p_message)->session_id != 0 )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif
    /*------------------------------------------------------------------------------*/
    /* stop the Voice Memo Play with no error parameters                            */
    /*------------------------------------------------------------------------------*/
    if ( audio_vm_play_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_32 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 33                      */
  /*                                                                              */
  /* Decription: Start a Voice memo Play and stop it before the FFS initialization*/
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_33 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_PLAY_PARAMETER   vm_play_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the Voice memo play                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill the Voice memo play parameter */
    strcpy(vm_play_parameter.memo_name,"/memo/voice");

    /* start the Voice memo play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* stop the Voice memo play with no error parameters                            */
    /*------------------------------------------------------------------------------*/
    if ( audio_vm_play_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_33 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 34                      */
  /*                                                                              */
  /* Decription: Start twice a Voice Memo Play and wait until it stopped          */
  /*             automatically.                                                   */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_34 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_VM_PLAY_PARAMETER   vm_play_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the Voice memo play                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill the Voice memo play parameter */
    strcpy(vm_play_parameter.memo_name,"/memo/voice");

    /* start the Voice memo play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_VM_PLAY_REQ *)p_message)->session_id != 0 )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the voice memo play with no error parameters                         */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* start the Voice memo play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
       return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

   #ifdef _WINDOWS

    /*------------------------------------------------------------------------------*/
    /* send the L1 stop confirmation message                                        */
    /*------------------------------------------------------------------------------*/
    audio_test_regr_send_message( MMI_VM_PLAY_STOP_CON, error_type, 0 );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);


    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_34 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 35                      */
  /*                                                                              */
  /* Decription: Start the voice memo play and send 2 stop                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_35 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_PLAY_PARAMETER     vm_play_parameter;
    T_RV_RETURN                   return_path;
    void                          *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo play                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo play parameter */
    strcpy(vm_play_parameter.memo_name,"/memo/voice");

    /* start the voice memo play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_VM_PLAY_REQ *)p_message)->session_id != 0 )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #else
        rvf_delay(RVF_MS_TO_TICKS(4000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the voice memo play                                                     */
    /*------------------------------------------------------------------------------*/
    if ( audio_vm_play_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #else
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the voice memo play                                                     */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    if ( audio_vm_play_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);


    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_35 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 36                      */
  /*                                                                              */
  /* Decription: Start a Voice Memo Play and wait until it stopped automatically. */
  /*             And after, the entity resquest to stop the Voice Memo Play.      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_36 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_VM_PLAY_PARAMETER   vm_play_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the Voice memo play                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill the Voice memo play parameter */
    strcpy(vm_play_parameter.memo_name,"/memo/voice");

    /* start the Voice memo play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_VM_PLAY_REQ *)p_message)->session_id != 0 )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(9000));
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* Request to stop the voice memo play generation                               */
    /*------------------------------------------------------------------------------*/
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    if ( audio_vm_play_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_36 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 37                      */
  /*                                                                              */
  /* Decription: Start a voice memo play and wait until it stopped automatically. */
  /*             Note the stop return is made by a call back function.            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_37 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    T_RV_RETURN                     return_path;
    void                            *p_message = NULL;

    p_audio_test->audio_call_back_called = 1;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo play                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo play parameters */
    strcpy(vm_play_parameter.memo_name,"/memo/voice");

    /* start the voice memo play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_VM_PLAY_REQ *)p_message)->session_id != 0 )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif


      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* stop the voice memo play                                                     */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = audio_vm_play_callback_test_function;
    return_path.addr_id = 0;

    if ( audio_vm_play_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the call back function is called                                        */
    /*------------------------------------------------------------------------------*/

    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_37 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 38                      */
  /*                                                                              */
  /* Decription: Check the parameters error manager                  .            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_38 (T_RV_REGR_ERR_TYPE *error_type)
  {
    #ifndef _WINDOWS

      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
      T_RV_RETURN                     return_path;
      void                            *p_message = NULL;

      /* initalize the return path, for this test the return path is a message */
      return_path.callback_func = NULL;
      return_path.addr_id = rvf_get_taskid();

      /*----------------------------------------------------------------------------*/
      /* start the voice memo play                                                  */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/memo/chevre");

      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_ERROR)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_38 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 59                      */
  /*                                                                              */
  /* Decription: Start a Voice Memorization play and wait until it stopped        */
  /*             automatically. ( Voice Memo play of 150s)                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_59 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_PLAY_PARAMETER   vm_play_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo play                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo play parameters */
    strcpy(vm_play_parameter.memo_name,"/memo/voice");

    /* start the voice memo play */
    if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_PLAY_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_VM_PLAY_REQ *)p_message)->session_id != 0 )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_PLAY_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      #else
        rvf_delay(RVF_MS_TO_TICKS(165000));
      #endif


      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_59 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                     AUDIO VOICE MEMO RECORD CALLBACK FUNCTION                */
  /*                                                                              */
  /* Decription: call back funtion used for the voice memo test.                  */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_vm_record_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR: VOICE MEMO RECORD CALLBACK FUNCTION WITH A WRONG STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:VOICE MEMO RECORD CALLBACK FUNCTION WITH A GOOD STATUS");
    }
  }

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 40                      */
  /*                                                                              */
  /* Description: Start a Voice Memorization record and wait until it stopped      */
  /*             automatically.                                                   */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_40 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_RECORD_PARAMETER   vm_record_parameter;
    T_AUDIO_TONES_PARAMETER       tones_parameter;
    T_RV_RETURN                   return_path;
    void                          *p_message = NULL;
    T_RV_RET                      status;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain =0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
        if ((((T_MMI_VM_RECORD_REQ *)p_message)->dtx_used != 0)           ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->maximum_size != 9001)     ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_ul != 0x0100) ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_dl != 0x0100) ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->session_id != 0) ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_rep  != 0x000a) ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_off  != 16)     ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_se_off  != 16)     ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_bu_off  != 8 )     ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_on   != 0 )     ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_off  != 3 )     ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_on   != 1 )     ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_off  != 4 )     ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_on   != 5 )     ||
         (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        status = rvf_free_buf((T_RVF_BUFFER *)p_message);
        if (status != RVF_GREEN)
        {
          AUDIO_SEND_TRACE(" AUDIO ERROR (env). A wrong message is deallocated ",
                         RV_TRACE_LEVEL_ERROR);
        }

         *error_type = EVENT_PARAM_ERROR;
         return (audio_test_regr_return_verdict(*error_type));
      }

      status = rvf_free_buf((T_RVF_BUFFER *)p_message);
      if (status != RVF_GREEN)
      {
        AUDIO_SEND_TRACE(" AUDIO ERROR (env). A wrong message is deallocated ",
                       RV_TRACE_LEVEL_ERROR);
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_STOP_CON,
                                    error_type, 8001 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
     #else
        rvf_delay(RVF_MS_TO_TICKS(10000));
     #endif


    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)  ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 8) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)
    #endif
    {
      status = rvf_free_buf((T_RVF_BUFFER *)p_message);
      if (status != RVF_GREEN)
      {
        AUDIO_SEND_TRACE(" AUDIO ERROR (env). A wrong message is deallocated ",
                       RV_TRACE_LEVEL_ERROR);
      }

      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

      status = rvf_free_buf((T_RVF_BUFFER *)p_message);
      if (status != RVF_GREEN)
      {
        AUDIO_SEND_TRACE(" AUDIO ERROR (env). A wrong message is deallocated ",
                       RV_TRACE_LEVEL_ERROR);
      }

      return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_40 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 41                      */
  /*                                                                              */
  /* Decription: Start a Voice Memo Record and stop it after the L1 Voive Memo    */
  /*             Record start                                                     */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_41 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_RECORD_PARAMETER vm_record_parameter;
    T_AUDIO_TONES_PARAMETER     tones_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain = 0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the Voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    /* Check the message parameters */
      if ((((T_MMI_VM_RECORD_REQ *)p_message)->dtx_used != 0)           ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->maximum_size != 9001)   ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_ul != 0x0100) ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_dl != 0x0100) ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->session_id != 0) ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_rep  != 0x000a) ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_off  != 16)     ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_se_off  != 16)     ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_bu_off  != 8 )     ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_on   != 0 )     ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_off  != 3 )     ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_on   != 1 )     ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_off  != 4 )     ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_on   != 5 )     ||
        (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_off  != 7 )     )
    {
       rvf_free_buf((T_RVF_BUFFER *)p_message);
       *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the voice memo record                                                     */
    /*------------------------------------------------------------------------------*/
    if ( audio_vm_record_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_STOP_CON,
                                    error_type, 8001 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK) ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 8) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)
    #endif
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_41 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 42                      */
  /*                                                                              */
  /* Decription:Start a Voice Memo Record and stop it before the L1 Voice_memo    */
  /*            start                                                             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_42 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_RECORD_PARAMETER vm_record_parameter;
    T_AUDIO_TONES_PARAMETER     tones_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain = 0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the Voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }


    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    /* Check the message parameters */
      if ((((T_MMI_VM_RECORD_REQ *)p_message)->dtx_used != 0)           ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->maximum_size != 9001)   ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_ul != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_dl != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->session_id != 0) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_rep  != 0x000a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_se_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_bu_off  != 8 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_on   != 0 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_off  != 3 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_on   != 1 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_off  != 4 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_on   != 5 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_off  != 7 )     )

    {
       rvf_free_buf((T_RVF_BUFFER *)p_message);
       *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the Voice Memo record with no error parameters                            */
    /*------------------------------------------------------------------------------*/
    if ( audio_vm_record_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_STOP_CON,
                                    error_type, 8001 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      #else
        rvf_delay(RVF_MS_TO_TICKS(5000));
      #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK) ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 8) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)
    #endif
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_42 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 43                      */
  /*                                                                              */
  /* Decription: Start a Voice memo Record and stop it before the FFS             */
  /*             initialization                                                   */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_43 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_RECORD_PARAMETER vm_record_parameter;
    T_AUDIO_TONES_PARAMETER     tones_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain = 0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the Voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait before to stop the Voice memo record                                      */
    /*------------------------------------------------------------------------------*/
    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(500));
    #endif


    /*------------------------------------------------------------------------------*/
    /* stop the Voice memo record with no error parameters                            */
    /*------------------------------------------------------------------------------*/
    if ( audio_vm_record_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK) ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 8) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)
    #endif
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_43 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 44                      */
  /*                                                                              */
  /* Decription: Start twice a Voice Memo record and wait until it stopped        */
  /*             automatically.                                                   */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_44 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_RECORD_PARAMETER vm_record_parameter;
    T_AUDIO_TONES_PARAMETER     tones_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain = 0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the Voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ((((T_MMI_VM_RECORD_REQ *)p_message)->dtx_used != 0)           ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->maximum_size != 9001)   ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_ul != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_dl != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->session_id != 0)      ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_rep  != 0x000a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_se_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_bu_off  != 8 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_on   != 0 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_off  != 3 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_on   != 1 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_off  != 4 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_on   != 5 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_off  != 7 )     )
      {
       rvf_free_buf((T_RVF_BUFFER *)p_message);
       *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

  #endif

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record with no error parameters                       */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* start the Voice memo record */
    strcpy(vm_record_parameter.memo_name,"/memo/tibo");
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(5000));
    #endif

  /*------------------------------------------------------------------------------*/
  /* wait the status message in the good mailbox                                  */
  /*------------------------------------------------------------------------------*/
  p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                            error_type );
  if (*error_type != NO_ERR)
  {
    return (audio_test_regr_return_verdict(*error_type));
  }

  /*------------------------------------------------------------------------------*/
  /* check if it's the expected status                                            */
  /*------------------------------------------------------------------------------*/
  #ifdef _WINDOWS
    if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_ERROR) ||
         (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 0) )
  #else
    if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_ERROR)
  #endif
  {
    rvf_free_buf((T_RVF_BUFFER *)p_message);
    *error_type = EVENT_PARAM_ERROR;
    return (audio_test_regr_return_verdict(*error_type));
  }

  rvf_free_buf((T_RVF_BUFFER *)p_message);

   #ifdef _WINDOWS
    /*------------------------------------------------------------------------------*/
    /* send the L1 stop confirmation message                                        */
    /*------------------------------------------------------------------------------*/
    audio_test_regr_send_message( MMI_VM_RECORD_STOP_CON, error_type, 8001 );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
     }
    #else
      rvf_delay(RVF_MS_TO_TICKS(5000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK) ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 8) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)
    #endif
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_44 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 45                      */
  /*                                                                              */
  /* Decription: Start the voice memo record and send 2 stop                      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_45 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_AUDIO_VM_RECORD_PARAMETER vm_record_parameter;
    T_AUDIO_TONES_PARAMETER     tones_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain = 0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the Voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ((((T_MMI_VM_RECORD_REQ *)p_message)->dtx_used != 0)           ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->maximum_size != 9001)   ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_ul != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_dl != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->session_id != 0)      ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_rep  != 0x000a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_se_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_bu_off  != 8 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_on   != 0 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_off  != 3 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_on   != 1 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_off  != 4 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_on   != 5 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_off  != 7 )     )
     {
       rvf_free_buf((T_RVF_BUFFER *)p_message);
       *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
     }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #else
        rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the voice memo record                                                     */
    /*------------------------------------------------------------------------------*/
    if ( audio_vm_record_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the voice memo record                                                     */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    if ( audio_vm_record_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_ERROR) ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 0) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_ERROR)
    #endif
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);


    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_STOP_CON,
                                    error_type, 8001 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK) ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 8) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)
    #endif
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);
    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_45 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 46                      */
  /*                                                                              */
  /* Decription: Start a Voice Memo record and wait until it stopped automatically*/
  /*             And after, the entity resquest to stop the Voice Memo record.    */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_46 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_RECORD_PARAMETER vm_record_parameter;
    T_AUDIO_TONES_PARAMETER     tones_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain = 0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the Voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ((((T_MMI_VM_RECORD_REQ *)p_message)->dtx_used != 0)           ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->maximum_size != 9001)   ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_ul != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_dl != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->session_id != 0) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_rep  != 0x000a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_se_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_bu_off  != 8 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_on   != 0 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_off  != 3 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_on   != 1 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_off  != 4 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_on   != 5 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_STOP_CON,
                                    error_type, 8001 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK) ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 8) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)
    #endif
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* Request to stop the voice memo record generation                             */
    /*------------------------------------------------------------------------------*/
    if ( audio_vm_record_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_ERROR) ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 0) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_ERROR)
    #endif
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_46 function ***********************/

  /*********************************************************************************/
  /*                                                                               */
  /*                            AUDIO NON REGRESSION TEST 47                       */
  /*                                                                               */
  /* Decription: Start a voice memo record and wait until it stopped automatically */
  /*             Note the stop return is made by a call back function.             */
  /*                                                                               */
  /*                                                                               */
  /*********************************************************************************/
  T_RV_REGR_RET audio_regr_test_47 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_RECORD_PARAMETER vm_record_parameter;
    T_AUDIO_TONES_PARAMETER     tones_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    p_audio_test->audio_call_back_called = 1;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain = 0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the Voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }


    /* Check the message parameters */
      if ((((T_MMI_VM_RECORD_REQ *)p_message)->dtx_used != 0)           ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->maximum_size != 9001)   ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_ul != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->record_coeff_dl != 0x0100) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->session_id != 0) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_rep  != 0x000a) ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_pe_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_se_off  != 16)     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_bu_off  != 8 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_on   != 0 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t0_off  != 3 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_on   != 1 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t1_off  != 4 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_on   != 5 )     ||
          (((T_MMI_VM_RECORD_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_STOP_CON,
                                    error_type, 8001 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK) ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 8) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)
    #endif
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* stop the voice memo record                                                     */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = audio_vm_record_callback_test_function;
    return_path.addr_id = 0;

    if ( audio_vm_record_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the call back function is called                                        */
    /*------------------------------------------------------------------------------*/

    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_47 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 48                      */
  /*                                                                              */
  /* Decription: Check the parameters error manager                  .            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_48 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_RECORD_PARAMETER   vm_record_parameter;
    T_AUDIO_TONES_PARAMETER       tones_parameter;
    T_RV_RETURN                   return_path;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/chevre");
    vm_record_parameter.microphone_gain =0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 123;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = 3;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain =0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain =0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = -2;   /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 9;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain =0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 200;    /* ms */
    tones_parameter.tones[0].stop_tone       = 84;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_48 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 69                      */
  /*                                                                              */
  /* Description: Start a Voice Memorization record and wait until it stopped     */
  /*             automatically.                                                   */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_69 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_VM_RECORD_PARAMETER   vm_record_parameter;
    T_AUDIO_TONES_PARAMETER       tones_parameter;
    T_RV_RETURN                   return_path;
    void                          *p_message = NULL;
    T_RV_RET                      status;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the voice memo record                                                */
    /*----------------------------------------------------------------------------*/
    /* fill the voice memo record parameters */
    vm_record_parameter.compression_mode = AUDIO_VM_NO_COMPRESSION_MODE;
    vm_record_parameter.memo_duration = 150;
    strcpy(vm_record_parameter.memo_name,"/memo/voice");
    vm_record_parameter.microphone_gain =0x0100;
    vm_record_parameter.network_gain = 0x0100;

    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the voice memo record */
    if ( audio_vm_record_start(&vm_record_parameter, &tones_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_VM_RECORD_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      status = rvf_free_buf((T_RVF_BUFFER *)p_message);
      if (status != RVF_GREEN)
      {
        AUDIO_SEND_TRACE(" AUDIO ERROR (env). A wrong message is deallocated ",
                       RV_TRACE_LEVEL_ERROR);
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_VM_RECORD_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
     #else
        rvf_delay(RVF_MS_TO_TICKS(165000));
     #endif


    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_VM_RECORD_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    #ifdef _WINDOWS
      if ( (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)  ||
           (((T_AUDIO_VM_RECORD_STATUS *)p_message)->recorded_duration != 0) )
    #else
      if (((T_AUDIO_VM_RECORD_STATUS *)p_message)->status != AUDIO_OK)
    #endif
    {
      status = rvf_free_buf((T_RVF_BUFFER *)p_message);
      if (status != RVF_GREEN)
      {
        AUDIO_SEND_TRACE(" AUDIO ERROR (env). A wrong message is deallocated ",
                       RV_TRACE_LEVEL_ERROR);
      }

      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

      status = rvf_free_buf((T_RVF_BUFFER *)p_message);
      if (status != RVF_GREEN)
      {
        AUDIO_SEND_TRACE(" AUDIO ERROR (env). A wrong message is deallocated ",
                       RV_TRACE_LEVEL_ERROR);
      }

      return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_69 function ************************/

#endif /* VOICE_MEMO */
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 50                      */
  /*                                                                              */
  /* Decription: Set an audio mode with the full access write functions. Check if */
  /*             it's good with full access read functions.                       */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_50 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_FULL_ACCESS_WRITE  full_access_write_parameter;
    T_AUDIO_FULL_ACCESS_READ   full_access_read_parameter;
    T_RV_RETURN                return_path;
    void                        *p_message = NULL;
    UINT8                      i;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /********* Audio mode to set ***************/
    /* path setting */
    p_audio_test->audio_mode_1.audio_path_setting = AUDIO_GSM_VOICE_PATH;
    /* micropohne setting */
    p_audio_test->audio_mode_1.audio_microphone_setting.mode = AUDIO_MICROPHONE_HANDFREE;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.gain = -8;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.extra_gain =
      AUDIO_MICROPHONE_AUX_GAIN_4_6dB;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.output_bias =
      AUDIO_MICROPHONE_OUTPUT_BIAS_2_5V;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir.coefficient[0] = 0x4001;
    for(i=1; i<30; i++)
    {
      p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir.coefficient[i] = 0x0000;
    }
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir.coefficient[30] = 0x0001;
    /* speaker setting */
    p_audio_test->audio_mode_1.audio_speaker_setting.mode = AUDIO_SPEAKER_HANDHELD_HANDFREE;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.gain = -3;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.audio_filter =
      AUDIO_SPEAKER_FILTER_ON;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[0] = 0x4001;
    for(i=1; i<30; i++)
    {
      p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[i] = 0x0000;
    }
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[30] = 0x0001;
    /* speaker micropohne loop setting */
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain = -20;
  #if (L1_NEW_AEC)
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
      AUDIO_AEC_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.continuous_filtering =
      TRUE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.granularity_attenuation =
      0x020D;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.smoothing_coefficient =
      0x0CCC;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.max_echo_suppression_level =
      AUDIO_MAX_ECHO_12dB;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.vad_factor =
      0x4000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.absolute_threshold =
      0x0032;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_filtering =
      0x1000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_muting =
      0x1000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_visibility =
      AUDIO_AEC_VISIBILITY_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
      AUDIO_NOISE_SUPPRESSION_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
      AUDIO_NOISE_6dB;
  #else
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
      AUDIO_AEC_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_mode =
      AUDIO_LONG_ECHO;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.echo_suppression_level =
      AUDIO_ECHO_12dB;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
      AUDIO_NOISE_SUPPRESSION_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
      AUDIO_NOISE_6dB;
  #endif
    /* speaker volume setting */
    p_audio_test->speaker_volume_1.audio_speaker_level = AUDIO_SPEAKER_VOLUME_12dB;


    /************************ FULL ACCESS WRITE TEST **********************************/
    /* fill the full access write parameter */
    full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
    full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_MODE_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_AUDIO_MODE *)p_message)->audio_mode != GSM_ONLY)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_MODE_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.mode);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_MODE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_GAIN;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.extra_gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_EXTRA_GAIN;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.output_bias);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_OUTPUT_BIAS;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_FIR;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_FIR_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AUDIO_FIR_REQ *)p_message)->update_fir != UL_FIR) ||
           (((T_MMI_AUDIO_FIR_REQ *)p_message)->fir_loop != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_FIR_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.mode);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_MODE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.gain);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_GAIN;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.audio_filter);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_FILTER;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_FIR;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_FIR_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AUDIO_FIR_REQ *)p_message)->update_fir != DL_FIR) ||
           (((T_MMI_AUDIO_FIR_REQ *)p_message)->fir_loop != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_FIR_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_SIDETONE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_AEC;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AEC_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
    #if (L1_NEW_AEC)
      if ( ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x03A6 ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->cont_filter) != TRUE ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->granularity_att) != 0x020D ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->coef_smooth) != 0x0CCC ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->es_level_max) != AUDIO_MAX_ECHO_12dB ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->fact_vad) != 0x4000 ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->thrs_abs) != 0x0032 ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->fact_asd_fil) != 0x1000 ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->fact_asd_mut) != 0x1000 ) )
    #else
      if ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x01B6 )
    #endif
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AEC_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data = &(p_audio_test->speaker_volume_1);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /************************ FULL READ ACCESS TEST **********************************/

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->audio_mode_2.audio_path_setting);
    full_access_read_parameter.variable_indentifier = AUDIO_PATH_USED;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->audio_mode_1.audio_path_setting !=
        p_audio_test->audio_mode_2.audio_path_setting )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_microphone_setting.mode);
    full_access_read_parameter.variable_indentifier = AUDIO_MICROPHONE_MODE;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->audio_mode_1.audio_microphone_setting.mode !=
         p_audio_test->audio_mode_2.audio_microphone_setting.mode )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_microphone_setting.setting.handfree.gain);
    full_access_read_parameter.variable_indentifier = AUDIO_MICROPHONE_GAIN;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.gain !=
         p_audio_test->audio_mode_2.audio_microphone_setting.setting.handfree.gain )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_microphone_setting.setting.handfree.extra_gain);
    full_access_read_parameter.variable_indentifier = AUDIO_MICROPHONE_EXTRA_GAIN;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.extra_gain !=
         p_audio_test->audio_mode_2.audio_microphone_setting.setting.handfree.extra_gain )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_microphone_setting.setting.handfree.output_bias);
    full_access_read_parameter.variable_indentifier = AUDIO_MICROPHONE_OUTPUT_BIAS;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.output_bias !=
         p_audio_test->audio_mode_2.audio_microphone_setting.setting.handfree.output_bias )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_microphone_setting.setting.handfree.fir);
    full_access_read_parameter.variable_indentifier = AUDIO_MICROPHONE_FIR;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    #ifndef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected value                                             */
      /*------------------------------------------------------------------------------*/
      if ( (p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir.coefficient[0] !=
            p_audio_test->audio_mode_2.audio_microphone_setting.setting.handfree.fir.coefficient[0]) ||
            (p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir.coefficient[30] !=
            p_audio_test->audio_mode_2.audio_microphone_setting.setting.handfree.fir.coefficient[30]) )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_speaker_setting.mode);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_MODE;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->audio_mode_1.audio_speaker_setting.mode !=
         p_audio_test->audio_mode_2.audio_speaker_setting.mode )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_speaker_setting.setting.handheld_handfree.gain);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_GAIN;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.gain !=
         p_audio_test->audio_mode_2.audio_speaker_setting.setting.handheld_handfree.gain )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_speaker_setting.setting.handheld_handfree.audio_filter);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_FILTER;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.audio_filter !=
         p_audio_test->audio_mode_2.audio_speaker_setting.setting.handheld_handfree.audio_filter )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_speaker_setting.setting.handheld_handfree.fir);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_FIR;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    #ifndef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected value                                             */
      /*------------------------------------------------------------------------------*/
      if ( (p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[0] !=
            p_audio_test->audio_mode_2.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[0]) ||
          (p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[30] !=
            p_audio_test->audio_mode_2.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[30]) )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.sidetone_gain);
    full_access_read_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_SIDETONE;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain !=
         p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.sidetone_gain )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* fill the full access read parameter */
    full_access_read_parameter.data =
      &(p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec);
    full_access_read_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_AEC;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    #ifndef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected value                                             */
      /*------------------------------------------------------------------------------*/
    #if (L1_NEW_AEC)
      if ( (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.aec_enable ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.continuous_filtering !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.continuous_filtering ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.granularity_attenuation !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.granularity_attenuation ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.smoothing_coefficient !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.smoothing_coefficient ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.max_echo_suppression_level !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.max_echo_suppression_level ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.vad_factor !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.vad_factor ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.absolute_threshold !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.absolute_threshold ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_filtering !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.factor_asd_filtering ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_muting !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.factor_asd_muting ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_visibility !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.aec_visibility ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.noise_suppression_level ) )
    #else
      if ( (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.aec_enable ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_mode !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.aec_mode ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.echo_suppression_level !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.echo_suppression_level ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable ) ||
           (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level !=
            p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.noise_suppression_level ) )
    #endif
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_1.audio_speaker_level !=
         p_audio_test->speaker_volume_2.audio_speaker_level )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_50 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 51                      */
  /*                                                                              */
  /* Decription: Set an audio mode with the full access write functions. Save it  */
  /*             with audio mode save.                                            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_51 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_FULL_ACCESS_WRITE  full_access_write_parameter;
    T_RV_RETURN                return_path;
    void                        *p_message = NULL;
    UINT8                      i;
    T_AUDIO_MODE_SAVE          audio_mode_save_parameter;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /********* Audio mode to set ***************/
    /* path setting */
    p_audio_test->audio_mode_1.audio_path_setting = AUDIO_GSM_VOICE_PATH;
    /* micropohne setting */
    p_audio_test->audio_mode_1.audio_microphone_setting.mode = AUDIO_MICROPHONE_HANDHELD;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.gain = -10;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.output_bias =
      AUDIO_MICROPHONE_OUTPUT_BIAS_2_0V;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir.coefficient[0] = 0x4002;
    for(i=1; i<30; i++)
    {
      p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir.coefficient[i] = 0x0000;
    }
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir.coefficient[30] = 0x0002;
    /* speaker setting */
  #if (ANLG_FAM == 1)
    p_audio_test->audio_mode_1.audio_speaker_setting.mode = AUDIO_SPEAKER_BUZZER;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.buzzer.activate = AUDIO_SPEAKER_BUZZER_ON;
  #else
    p_audio_test->audio_mode_1.audio_speaker_setting.mode = AUDIO_SPEAKER_HANDHELD_HANDFREE;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.gain = -3;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.audio_filter =
      AUDIO_SPEAKER_FILTER_ON;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[0] = 0x4001;
    for(i=1; i<30; i++)
    {
      p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[i] = 0x0000;
    }
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[30] = 0x0001;
  #endif
    /* speaker microphone loop setting */
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain = 1;

  #if (L1_NEW_AEC)
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
      AUDIO_AEC_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.continuous_filtering =
      TRUE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.granularity_attenuation =
      0x020D;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.smoothing_coefficient =
      0x0CCC;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.max_echo_suppression_level =
      AUDIO_MAX_ECHO_6dB;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.vad_factor =
      0x4000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.absolute_threshold =
      0x0032;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_filtering =
      0x1000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_muting =
      0x1000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_visibility =
      AUDIO_AEC_VISIBILITY_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
      AUDIO_NOISE_SUPPRESSION_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
      AUDIO_NOISE_NO_LIMITATION;
  #else
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
      AUDIO_AEC_DISABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_mode =
      AUDIO_SHORT_ECHO;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.echo_suppression_level =
      AUDIO_ECHO_6dB;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
      AUDIO_NOISE_SUPPRESSION_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
      AUDIO_NOISE_NO_LIMITATION;
  #endif
    /* speaker volume setting */
    p_audio_test->speaker_volume_1.audio_speaker_level = AUDIO_SPEAKER_VOLUME_24dB;


    /************************ FULL ACCESS WRITE TEST **********************************/
    /* fill the full access write parameter */
    full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
    full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_MODE_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_AUDIO_MODE *)p_message)->audio_mode != GSM_ONLY)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_MODE_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.mode);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_MODE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_GAIN;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.output_bias);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_OUTPUT_BIAS;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_FIR;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_FIR_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AUDIO_FIR_REQ *)p_message)->update_fir != UL_FIR) ||
           (((T_MMI_AUDIO_FIR_REQ *)p_message)->fir_loop != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_FIR_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.mode);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_MODE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

  #if (ANLG_FAM == 1)
    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.buzzer.activate );
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_BUZZER_STATE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);
  #else
    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.gain);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_GAIN;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.audio_filter);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_FILTER;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_FIR;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_FIR_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AUDIO_FIR_REQ *)p_message)->update_fir != DL_FIR) ||
           (((T_MMI_AUDIO_FIR_REQ *)p_message)->fir_loop != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_FIR_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);
  #endif

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_SIDETONE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_AEC;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AEC_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
    #if (L1_NEW_AEC)
      if ( ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x0386 ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->cont_filter) != TRUE ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->granularity_att) != 0x020D ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->coef_smooth) != 0x0CCC ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->es_level_max) != AUDIO_MAX_ECHO_6dB ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->fact_vad) != 0x4000 ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->thrs_abs) != 0x0032 ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->fact_asd_fil) != 0x1000 ) ||
           ( (((T_MMI_AEC_REQ *)p_message)->fact_asd_mut) != 0x1000 ) )
    #else
      if ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x038C )
    #endif
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AEC_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data = &(p_audio_test->speaker_volume_1);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* Save the current audio mode in Flash                                         */
    /*------------------------------------------------------------------------------*/
    strcpy(audio_mode_save_parameter.audio_mode_filename,"buzzer");
    if ( audio_mode_save(&audio_mode_save_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MODE_SAVE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SAVE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* Compare if the mode save correspond to the current mode                      */
      /*------------------------------------------------------------------------------*/

    #if (ANLG_FAM == 1)
      if( (p_audio_test->audio_mode_1.audio_path_setting !=
           p_audio_test->audio_mode_2.audio_path_setting) ||
          (p_audio_test->audio_mode_1.audio_microphone_setting.mode !=
           p_audio_test->audio_mode_2.audio_microphone_setting.mode) ||
          (p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.gain !=
           p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.gain) ||
          (p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.output_bias !=
           p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.output_bias) ||
          (p_audio_test->audio_mode_1.audio_speaker_setting.mode !=
           p_audio_test->audio_mode_2.audio_speaker_setting.mode) ||
          (p_audio_test->audio_mode_1.audio_speaker_setting.setting.buzzer.activate !=
           p_audio_test->audio_mode_2.audio_speaker_setting.setting.buzzer.activate) ||
          (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain !=
           p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.sidetone_gain) ||
          (p_audio_test->speaker_volume_1.audio_speaker_level !=
           p_audio_test->speaker_volume_2.audio_speaker_level) )
    #else
      if( (p_audio_test->audio_mode_1.audio_path_setting !=
           p_audio_test->audio_mode_2.audio_path_setting) ||
          (p_audio_test->audio_mode_1.audio_microphone_setting.mode !=
           p_audio_test->audio_mode_2.audio_microphone_setting.mode) ||
          (p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.gain !=
           p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.gain) ||
          (p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.output_bias !=
           p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.output_bias) ||
          (p_audio_test->audio_mode_1.audio_speaker_setting.mode !=
           p_audio_test->audio_mode_2.audio_speaker_setting.mode) ||
          (p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.gain !=
           p_audio_test->audio_mode_2.audio_speaker_setting.setting.handheld_handfree.gain) ||
          (p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.audio_filter !=
           p_audio_test->audio_mode_2.audio_speaker_setting.setting.handheld_handfree.audio_filter) ||
          (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain !=
           p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.sidetone_gain) ||
          (p_audio_test->speaker_volume_1.audio_speaker_level !=
           p_audio_test->speaker_volume_2.audio_speaker_level) )
    #endif
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    return(TEST_PASSED);
  }

  /******************** End of audio_regr_test_51 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 52                      */
  /*                                                                              */
  /* Decription: Set an audio mode with the full access write functions. Load a   */
  /*             new one.                                                         */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_52 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_FULL_ACCESS_WRITE  full_access_write_parameter;
    T_RV_RETURN                return_path;
    void                        *p_message = NULL;
    UINT8                      i;
    T_AUDIO_MODE_SAVE          audio_mode_save_parameter;
    T_AUDIO_MODE_LOAD          audio_mode_load_parameter;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /********* Audio mode to set ***************/
    /* path setting */
    p_audio_test->audio_mode_1.audio_path_setting = AUDIO_GSM_VOICE_PATH;
    /* micropohne setting */
    p_audio_test->audio_mode_1.audio_microphone_setting.mode = AUDIO_MICROPHONE_HANDHELD;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.gain = -1;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.output_bias =
      AUDIO_MICROPHONE_OUTPUT_BIAS_2_5V;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir.coefficient[0] = 0x4002;
    for(i=1; i<30; i++)
    {
      p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir.coefficient[i] = 0x0000;
    }
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir.coefficient[30] = 0x0002;
    /* speaker setting */
    p_audio_test->audio_mode_1.audio_speaker_setting.mode = AUDIO_SPEAKER_BUZZER;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.buzzer.activate = AUDIO_SPEAKER_BUZZER_OFF;
    /* speaker micropohne loop setting */
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain = -2;

  #if (L1_NEW_AEC)
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
      AUDIO_AEC_DISABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.continuous_filtering =
      TRUE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.granularity_attenuation =
      0x020D;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.smoothing_coefficient =
      0x0CCC;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.max_echo_suppression_level =
      AUDIO_MAX_ECHO_12dB;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.vad_factor =
      0x4000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.absolute_threshold =
      0x0032;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_filtering =
      0x1000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_muting =
      0x1000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_visibility =
      AUDIO_AEC_VISIBILITY_DISABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
      AUDIO_NOISE_SUPPRESSION_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
      AUDIO_NOISE_NO_LIMITATION;
  #else
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
      AUDIO_AEC_DISABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_mode =
      AUDIO_SHORT_ECHO;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.echo_suppression_level =
      AUDIO_ECHO_6dB;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
      AUDIO_NOISE_SUPPRESSION_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
      AUDIO_NOISE_NO_LIMITATION;
  #endif
    /* speaker volume setting */
    p_audio_test->speaker_volume_1.audio_speaker_level = AUDIO_SPEAKER_VOLUME_18dB;


    /************************ FULL ACCESS WRITE TEST **********************************/
    /* fill the full access write parameter */
    full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
    full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_MODE_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_AUDIO_MODE *)p_message)->audio_mode != GSM_ONLY)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_MODE_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.mode);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_MODE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_GAIN;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.output_bias);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_OUTPUT_BIAS;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_FIR;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_FIR_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AUDIO_FIR_REQ *)p_message)->update_fir != UL_FIR) ||
           (((T_MMI_AUDIO_FIR_REQ *)p_message)->fir_loop != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_FIR_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.mode);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_MODE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.buzzer.activate );
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_BUZZER_STATE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_SIDETONE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_AEC;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AEC_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
    #if (L1_NEW_AEC)
      if ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x0184 )
    #else
      if ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x038C )
    #endif
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AEC_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data = &(p_audio_test->speaker_volume_1);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /********* Audio mode to load ***************/
      /* path setting */
      p_audio_test->audio_mode_1.audio_path_setting = AUDIO_GSM_VOICE_PATH;
      /* micropohne setting */
      p_audio_test->audio_mode_1.audio_microphone_setting.mode = AUDIO_MICROPHONE_HANDHELD;
      p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.gain = -10;
      p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.output_bias =
        AUDIO_MICROPHONE_OUTPUT_BIAS_2_0V;
      p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir.coefficient[0] = 0x4002;
      for(i=1; i<30; i++)
      {
        p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir.coefficient[i] = 0x0000;
      }
      p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.fir.coefficient[30] = 0x0002;
      /* speaker setting */
      p_audio_test->audio_mode_1.audio_speaker_setting.mode = AUDIO_SPEAKER_BUZZER;
      p_audio_test->audio_mode_1.audio_speaker_setting.setting.buzzer.activate = AUDIO_SPEAKER_BUZZER_ON;
      /* speaker microphone loop setting */
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain = 1;
    #if (L1_NEW_AEC)
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
        AUDIO_AEC_DISABLE;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.continuous_filtering =
        TRUE;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.granularity_attenuation =
        0x020D;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.smoothing_coefficient =
        0x0CCC;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.max_echo_suppression_level =
        AUDIO_MAX_ECHO_12dB;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.vad_factor =
        0x4000;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.absolute_threshold =
        0x0032;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_filtering =
        0x1000;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_muting =
        0x1000;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_visibility =
        AUDIO_AEC_VISIBILITY_ENABLE;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
        AUDIO_NOISE_SUPPRESSION_ENABLE;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
        AUDIO_NOISE_6dB;
    #else
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
        AUDIO_AEC_DISABLE;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_mode =
        AUDIO_SHORT_ECHO;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.echo_suppression_level =
        AUDIO_ECHO_6dB;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
        AUDIO_NOISE_SUPPRESSION_ENABLE;
      p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
        AUDIO_NOISE_NO_LIMITATION;
    #endif
      /* speaker volume setting */
      p_audio_test->speaker_volume_1.audio_speaker_level = AUDIO_SPEAKER_VOLUME_24dB;

      /* path setting */
      p_audio_test->audio_mode_2.audio_path_setting = 0;
      /* micropohne setting */
      p_audio_test->audio_mode_2.audio_microphone_setting.mode = 0;
      p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.gain = 0;
      p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.output_bias =
        0;
      p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.fir.coefficient[0] = 0;
      for(i=1; i<30; i++)
      {
        p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.fir.coefficient[i] = 0;
      }
      p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.fir.coefficient[30] = 0;
      /* speaker setting */
      p_audio_test->audio_mode_2.audio_speaker_setting.mode = 0;
      p_audio_test->audio_mode_2.audio_speaker_setting.setting.buzzer.activate = 0;
      /* speaker micropohne loop setting */
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.sidetone_gain = 0;
    #if (L1_NEW_AEC)
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.aec_enable =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.continuous_filtering =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.granularity_attenuation =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.smoothing_coefficient =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.max_echo_suppression_level =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.vad_factor =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.absolute_threshold =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.factor_asd_filtering =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.factor_asd_muting =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.aec_visibility =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
        0;
    #else
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.aec_enable =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.aec_mode =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.echo_suppression_level =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
        0;
      p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
        0;
    #endif
      /* speaker volume setting */
      p_audio_test->speaker_volume_2.audio_speaker_level = 0;

    #endif

    /*------------------------------------------------------------------------------*/
    /* Load the current audio mode from flash                                       */
    /*------------------------------------------------------------------------------*/
    strcpy(audio_mode_load_parameter.audio_mode_filename,"buzzer");
    if ( audio_mode_load(&audio_mode_load_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_MODE_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_AUDIO_MODE *)p_message)->audio_mode != GSM_ONLY)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_MODE_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_FIR_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AUDIO_FIR_REQ *)p_message)->update_fir != UL_FIR) ||
           (((T_MMI_AUDIO_FIR_REQ *)p_message)->fir_loop != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_FIR_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AEC_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x038C )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AEC_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif


    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MODE_LOAD_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_LOAD_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* Save the current audio mode in Flash                                         */
      /*------------------------------------------------------------------------------*/
      strcpy(audio_mode_save_parameter.audio_mode_filename,"buzzer");
      if ( audio_mode_save(&audio_mode_save_parameter, return_path) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MODE_SAVE_DONE,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_SAVE_DONE *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* Compare if the mode save correspond to the current mode                      */
      /*------------------------------------------------------------------------------*/
      if( (p_audio_test->audio_mode_1.audio_path_setting !=
           p_audio_test->audio_mode_2.audio_path_setting) ||
          (p_audio_test->audio_mode_1.audio_microphone_setting.mode !=
           p_audio_test->audio_mode_2.audio_microphone_setting.mode) ||
          (p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.gain !=
           p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.gain) ||
          (p_audio_test->audio_mode_1.audio_microphone_setting.setting.handheld.output_bias !=
           p_audio_test->audio_mode_2.audio_microphone_setting.setting.handheld.output_bias) ||
          (p_audio_test->audio_mode_1.audio_speaker_setting.mode !=
           p_audio_test->audio_mode_2.audio_speaker_setting.mode) ||
          (p_audio_test->audio_mode_1.audio_speaker_setting.setting.buzzer.activate !=
           p_audio_test->audio_mode_2.audio_speaker_setting.setting.buzzer.activate) ||
          (p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain !=
           p_audio_test->audio_mode_2.audio_microphone_speaker_loop_setting.sidetone_gain) ||
          (p_audio_test->speaker_volume_1.audio_speaker_level !=
           p_audio_test->speaker_volume_2.audio_speaker_level) )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_52 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 53                      */
  /*                                                                              */
  /* Decription: Set an audio mode with the full access write functions. Save it, */
  /*             load it and play a melody with an audio speaker volume function. */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_53 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_FULL_ACCESS_WRITE  full_access_write_parameter;
    T_AUDIO_FULL_ACCESS_READ   full_access_read_parameter;
    T_RV_RETURN                return_path;
    void                        *p_message = NULL;
    UINT8                      i;
    T_AUDIO_SPEAKER_VOLUME     volume_parameter;
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;
    T_AUDIO_MODE_SAVE          audio_mode_save_parameter;
    T_AUDIO_MODE_LOAD          audio_mode_load_parameter;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /********* Audio mode to set ***************/
    /* path setting */
    p_audio_test->audio_mode_1.audio_path_setting = AUDIO_GSM_VOICE_PATH;
    /* micropohne setting */
    p_audio_test->audio_mode_1.audio_microphone_setting.mode = AUDIO_MICROPHONE_HANDFREE;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.gain =
      AUDIO_MICROPHONE_MUTE;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.extra_gain =
      AUDIO_MICROPHONE_AUX_GAIN_4_6dB;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.output_bias =
      AUDIO_MICROPHONE_OUTPUT_BIAS_2_5V;
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir.coefficient[0] = 0x4001;
    for(i=1; i<30; i++)
    {
      p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir.coefficient[i] = 0x0000;
    }
    p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir.coefficient[30] = 0x0001;
    /* speaker setting */
    p_audio_test->audio_mode_1.audio_speaker_setting.mode = AUDIO_SPEAKER_HANDHELD_HANDFREE;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.gain = 0;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.audio_filter =
      AUDIO_SPEAKER_FILTER_ON;
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[0] = 0x4001;
    for(i=1; i<30; i++)
    {
      p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[i] = 0x0000;
    }
    p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir.coefficient[30] = 0x0001;
    /* speaker micropohne loop setting */
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain = -20;

  #if (L1_NEW_AEC)
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
      AUDIO_AEC_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.continuous_filtering =
      TRUE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.granularity_attenuation =
      0x020D;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.smoothing_coefficient =
      0x0CCC;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.max_echo_suppression_level =
      AUDIO_MAX_ECHO_0dB;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.vad_factor =
      0x4000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.absolute_threshold =
      0x0032;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_filtering =
      0x1000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.factor_asd_muting =
      0x1000;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_visibility =
      AUDIO_AEC_VISIBILITY_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
      AUDIO_NOISE_SUPPRESSION_ENABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
      AUDIO_NOISE_NO_LIMITATION;
  #else
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_enable =
      AUDIO_AEC_DISABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.aec_mode =
      AUDIO_LONG_ECHO;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.echo_suppression_level =
      AUDIO_ECHO_0dB;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_enable =
      AUDIO_NOISE_SUPPRESSION_DISABLE;
    p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec.noise_suppression_level =
      AUDIO_NOISE_NO_LIMITATION;
  #endif
    /* speaker volume setting */
    p_audio_test->speaker_volume_1.audio_speaker_level = AUDIO_SPEAKER_VOLUME_0dB;

    /************************ FULL ACCESS WRITE TEST **********************************/
    /* fill the full access write parameter */
    full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
    full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_MODE_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_AUDIO_MODE *)p_message)->audio_mode != GSM_ONLY)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_MODE_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.mode);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_MODE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_GAIN;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.extra_gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_EXTRA_GAIN;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.output_bias);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_OUTPUT_BIAS;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_setting.setting.handfree.fir);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_FIR;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_FIR_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AUDIO_FIR_REQ *)p_message)->update_fir != UL_FIR) ||
           (((T_MMI_AUDIO_FIR_REQ *)p_message)->fir_loop != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_FIR_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.mode);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_MODE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.gain);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_GAIN;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.audio_filter);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_FILTER;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_speaker_setting.setting.handheld_handfree.fir);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_FIR;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_FIR_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AUDIO_FIR_REQ *)p_message)->update_fir != DL_FIR) ||
           (((T_MMI_AUDIO_FIR_REQ *)p_message)->fir_loop != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_FIR_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.sidetone_gain);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_SIDETONE;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data =
      &(p_audio_test->audio_mode_1.audio_microphone_speaker_loop_setting.aec);
    full_access_write_parameter.variable_indentifier = AUDIO_MICROPHONE_SPEAKER_LOOP_AEC;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AEC_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
    #if (L1_NEW_AEC)
      if ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x0386 )
    #else
      if ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x0180 )
    #endif
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AEC_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access write parameter */
    full_access_write_parameter.data = &(p_audio_test->speaker_volume_1);
    full_access_write_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* Save the current audio mode in Flash                                         */
    /*------------------------------------------------------------------------------*/
    strcpy(audio_mode_save_parameter.audio_mode_filename,"ring");
    if ( audio_mode_save(&audio_mode_save_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MODE_SAVE_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SAVE_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* Load the current audio mode from flash                                       */
    /*------------------------------------------------------------------------------*/
    strcpy(audio_mode_load_parameter.audio_mode_filename,"ring");
    if ( audio_mode_load(&audio_mode_load_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_MODE_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if (((T_MMI_AUDIO_MODE *)p_message)->audio_mode != GSM_ONLY)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_MODE_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AUDIO_FIR_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
      if ( (((T_MMI_AUDIO_FIR_REQ *)p_message)->update_fir != UL_DL_FIR) ||
           (((T_MMI_AUDIO_FIR_REQ *)p_message)->fir_loop != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AUDIO_FIR_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_AEC_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /* Check the message parameters */
    #if (L1_NEW_AEC)
      if ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x0386 )
    #else
      if ( (((T_MMI_AEC_REQ *)p_message)->aec_control) != 0x0180 )
    #endif
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_AEC_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif


    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MODE_LOAD_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_LOAD_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_NORMAL_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E1_parameter.loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->oscillator_used_bitmap != 0xFF ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_SET;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_MUTE)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_INCREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_24dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_INCREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_18dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_INCREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_12dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_INCREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_6dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_INCREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_0dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_INCREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_0dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_DECREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_6dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_DECREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_12dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_DECREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_18dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_DECREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_24dB)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_DECREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_MUTE)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* Change the volume and save it in Flash                                       */
    /*------------------------------------------------------------------------------*/
    volume_parameter.volume_action = AUDIO_SPEAKER_VOLUME_DECREASE;
    volume_parameter.value = AUDIO_SPEAKER_VOLUME_MUTE;
    if ( audio_speaker_volume(volume_parameter, return_path) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SPEAKER_VOLUME_DONE,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_VOLUME_DONE *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /* fill the full access read parameter */
    full_access_read_parameter.data = &(p_audio_test->speaker_volume_2);
    full_access_read_parameter.variable_indentifier = AUDIO_SPEAKER_VOLUME_LEVEL;

    /* start a full access read */
    if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* check if it's the expected value                                             */
    /*------------------------------------------------------------------------------*/
    if ( p_audio_test->speaker_volume_2.audio_speaker_level != AUDIO_SPEAKER_VOLUME_MUTE)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(60000));
    #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E1_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }

  /******************** End of audio_regr_test_53 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 54                      */
  /*                                                                              */
  /* Decription: Set different audio path:                                        */
  /*              mode: GSM voice path                                            */
  /*              play a keybeep                                                  */
  /*              mode: BT cordless                                               */
  /*              play a keybeep                                                  */
  /*              mode: BT headset                                                */
  /*              play a keybeep                                                  */
  /*              mode: DAI decoder                                               */
  /*              play a keybeep                                                  */
  /*              mode: DAI encoder                                               */
  /*              play a keybeep                                                  */
  /*              mode: DAI acoustic                                              */
  /*              play a keybeep                                                  */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_54 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_FULL_ACCESS_WRITE  full_access_write_parameter;
    T_AUDIO_FULL_ACCESS_READ   full_access_read_parameter;
    T_RV_RETURN                return_path;
    void                        *p_message = NULL;
//    UINT8                      i;
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /************************ GSM VOICE PATH **********************************/
    /* fill the full access write parameter */
    p_audio_test->audio_mode_1.audio_path_setting = AUDIO_GSM_VOICE_PATH;
    full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
    full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

    /* start a full access write */
    if ( (audio_full_access_write (&full_access_write_parameter,
          return_path)) == AUDIO_ERROR )
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
        /*------------------------------------------------------------------------------*/
        /* wait the L1 start message in the good mailbox                                */
        /*------------------------------------------------------------------------------*/
        p_message = audio_test_regr_wait_message( MMI_AUDIO_MODE_REQ,
                                                  error_type );
        if (*error_type != NO_ERR)
        {
          return (audio_test_regr_return_verdict(*error_type));
        }

        /* Check the message parameters */
        if (((T_MMI_AUDIO_MODE *)p_message)->audio_mode != GSM_ONLY)
        {
          rvf_free_buf((T_RVF_BUFFER *)p_message);
          *error_type = EVENT_PARAM_ERROR;
          return (audio_test_regr_return_verdict(*error_type));
        }
        rvf_free_buf((T_RVF_BUFFER *)p_message);

        /*------------------------------------------------------------------------------*/
        /* send the L1 stop confirmation message                                        */
        /*------------------------------------------------------------------------------*/
        audio_test_regr_send_message( MMI_AUDIO_MODE_CON,
                                      error_type, 0 );
        if (*error_type != NO_ERR)
        {
          return (audio_test_regr_return_verdict(*error_type));
        }
      #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the full access read parameter */
      full_access_read_parameter.data = &(p_audio_test->audio_mode_2.audio_path_setting);
      full_access_read_parameter.variable_indentifier = AUDIO_PATH_USED;

      /*----------------------------------------------------------------------------*/
      /* start the keybeep with no error parameters                                 */
      /*----------------------------------------------------------------------------*/
      /* fill the keybeep parameter */
      keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
      keybeep_parameter.amplitude_beep[0] = -20; /* dB */
      keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
      keybeep_parameter.amplitude_beep[1] = -10; /* dB */
      keybeep_parameter.duration = 1000; /* ms */

      /* start the keybeep */
      if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      #ifdef _WINDOWS
        /*------------------------------------------------------------------------------*/
        /* wait the L1 start message in the good mailbox                                */
        /*------------------------------------------------------------------------------*/
        p_message = audio_test_regr_wait_message( MMI_KEYBEEP_START_REQ,
                                                  error_type );
        if (*error_type != NO_ERR)
        {
          return (audio_test_regr_return_verdict(*error_type));
        }

        /* Check the message parameters */
        if ( (((T_MMI_KEYBEEP_REQ *)p_message)->d_dur_kb   != 50)      ||
             (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt0 != 0xb512) ||
             (((T_MMI_KEYBEEP_REQ *)p_message)->d_k_x1_kt1 != 0x614a) )
        {
          rvf_free_buf((T_RVF_BUFFER *)p_message);
          *error_type = EVENT_PARAM_ERROR;
          return (audio_test_regr_return_verdict(*error_type));
        }

        rvf_free_buf((T_RVF_BUFFER *)p_message);

        /*------------------------------------------------------------------------------*/
        /* send the L1 start confirmation message                                       */
        /*------------------------------------------------------------------------------*/
        audio_test_regr_send_message( MMI_KEYBEEP_START_CON,
                                      error_type, 0 );
        if (*error_type != NO_ERR)
        {
          return (audio_test_regr_return_verdict(*error_type));
        }

        /*------------------------------------------------------------------------------*/
        /* send the L1 stop confirmation message                                        */
        /*------------------------------------------------------------------------------*/
        audio_test_regr_send_message( MMI_KEYBEEP_STOP_CON,
                                      error_type, 0 );
        if (*error_type != NO_ERR)
        {
          return (audio_test_regr_return_verdict(*error_type));
        }
      #else
        rvf_delay(RVF_MS_TO_TICKS(1000));
      #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* start a full access read */
      if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected value                                             */
      /*------------------------------------------------------------------------------*/
      if ( p_audio_test->audio_mode_1.audio_path_setting !=
          p_audio_test->audio_mode_2.audio_path_setting )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      #ifndef _WINDOWS
      /************************ BT Cordless PATH **********************************/
      /* fill the full access write parameter */
      p_audio_test->audio_mode_1.audio_path_setting = AUDIO_BLUETOOTH_CORDLESS_VOICE_PATH;
      full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
      full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access write */
      if ( (audio_full_access_write (&full_access_write_parameter,
            return_path)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the full access read parameter */
      full_access_read_parameter.data = &(p_audio_test->audio_mode_2.audio_path_setting);
      full_access_read_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access read */
      if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected value                                             */
      /*------------------------------------------------------------------------------*/
      if ( p_audio_test->audio_mode_1.audio_path_setting !=
          p_audio_test->audio_mode_2.audio_path_setting )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /************************ BT Headset PATH **********************************/
      /* fill the full access write parameter */
      p_audio_test->audio_mode_1.audio_path_setting = AUDIO_BLUETOOTH_HEADSET;
      full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
      full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access write */
      if ( (audio_full_access_write (&full_access_write_parameter,
            return_path)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the full access read parameter */
      full_access_read_parameter.data = &(p_audio_test->audio_mode_2.audio_path_setting);
      full_access_read_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access read */
      if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected value                                             */
      /*------------------------------------------------------------------------------*/
      if ( p_audio_test->audio_mode_1.audio_path_setting !=
          p_audio_test->audio_mode_2.audio_path_setting )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /************************ DAI Encoder PATH **********************************/
      /* fill the full access write parameter */
      p_audio_test->audio_mode_1.audio_path_setting = AUDIO_DAI_ENCODER;
      full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
      full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access write */
      if ( (audio_full_access_write (&full_access_write_parameter,
            return_path)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the full access read parameter */
      full_access_read_parameter.data = &(p_audio_test->audio_mode_2.audio_path_setting);
      full_access_read_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access read */
      if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected value                                             */
      /*------------------------------------------------------------------------------*/
      if ( p_audio_test->audio_mode_1.audio_path_setting !=
          p_audio_test->audio_mode_2.audio_path_setting )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /************************ DAI Decoder PATH **********************************/
      /* fill the full access write parameter */
      p_audio_test->audio_mode_1.audio_path_setting = AUDIO_DAI_DECODER;
      full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
      full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access write */
      if ( (audio_full_access_write (&full_access_write_parameter,
            return_path)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the full access read parameter */
      full_access_read_parameter.data = &(p_audio_test->audio_mode_2.audio_path_setting);
      full_access_read_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access read */
      if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected value                                             */
      /*------------------------------------------------------------------------------*/
      if ( p_audio_test->audio_mode_1.audio_path_setting !=
          p_audio_test->audio_mode_2.audio_path_setting )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /************************ DAI Acoustic PATH **********************************/
      /* fill the full access write parameter */
      p_audio_test->audio_mode_1.audio_path_setting = AUDIO_DAI_ACOUSTIC;
      full_access_write_parameter.data = &(p_audio_test->audio_mode_1.audio_path_setting);
      full_access_write_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access write */
      if ( (audio_full_access_write (&full_access_write_parameter,
            return_path)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_FULL_ACCESS_WRITE_DONE,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_FULL_ACCESS_WRITE_DONE *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the full access read parameter */
      full_access_read_parameter.data = &(p_audio_test->audio_mode_2.audio_path_setting);
      full_access_read_parameter.variable_indentifier = AUDIO_PATH_USED;

      /* start a full access read */
      if ( (audio_full_access_read (&full_access_read_parameter)) == AUDIO_ERROR )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected value                                             */
      /*------------------------------------------------------------------------------*/
      if ( p_audio_test->audio_mode_1.audio_path_setting !=
          p_audio_test->audio_mode_2.audio_path_setting )
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif /* _WINDOWS */

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_54 function ************************/

#if (TONE) && (MELODY_E1)
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 60                      */
  /*                                                                              */
  /* Decription: Compatibilities test: tone + melody E1                           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_60 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_TONES_PARAMETER   tones_parameter ;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    T_AUDIO_MELODY_E1_PARAMETER melody_E1_parameter;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the tones with no error parameters                                   */
    /*----------------------------------------------------------------------------*/
    /* fill the tones parameter */
    tones_parameter.tones[0].start_tone      = 2;    /* ms */
    tones_parameter.tones[0].stop_tone       = 64;   /* ms */
    tones_parameter.tones[0].frequency_tone  = 500;  /* Hz */
    tones_parameter.tones[0].amplitude_tone  = -24;  /* dB */
    tones_parameter.tones[1].start_tone      = 22;   /* ms */
    tones_parameter.tones[1].stop_tone       = 81;   /* ms */
    tones_parameter.tones[1].frequency_tone  = 800;  /* Hz */
    tones_parameter.tones[1].amplitude_tone  = -15;  /* dB */
    tones_parameter.tones[2].start_tone      = 110;  /* ms */
    tones_parameter.tones[2].stop_tone       = 143;  /* ms */
    tones_parameter.tones[2].frequency_tone  = 600;  /* Hz */
    tones_parameter.tones[2].amplitude_tone  = -6;   /* dB */
    tones_parameter.frame_duration           = 163;  /* ms */
    tones_parameter.sequence_duration        = 320;  /* ms */
    tones_parameter.period_duration          = 323;  /* ms */
    tones_parameter.repetition               = 10;

    /* start the tones */
    if ( audio_tones_start((T_AUDIO_TONES_PARAMETER *)(&tones_parameter), return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E1                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E1 parameters */
    melody_E1_parameter.loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E1_parameter.melody_name,"/melo/USA");
    melody_E1_parameter.melody_mode = AUDIO_MELODY_NORMAL_MODE;

    /* start the melody_E1 */
    if ( audio_melody_E1_start(&melody_E1_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_TONE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t0 != 0xec06) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t1 != 0xcf1a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_k_x1_t2 != 0xe43a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_rep  != 0x000a) ||
           (((T_MMI_TONE_REQ *)p_message)->d_pe_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_se_off  != 16)     ||
           (((T_MMI_TONE_REQ *)p_message)->d_bu_off  != 8 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_on   != 0 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t0_off  != 3 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_on   != 1 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t1_off  != 4 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_on   != 5 )     ||
           (((T_MMI_TONE_REQ *)p_message)->d_t2_off  != 7 )     )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E1_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_TONE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(3000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TONES_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TONES_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_60 function ************************/
#endif /* #if (TONE) && (MELODY_E1) */
#if (SPEECH_RECO) && (KEYBEEP)
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 61                      */
  /*                                                                              */
  /* Decription: Compatibilities test: Speech reco + Keybeep                      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_61 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_ENROLL_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    audio_keybeep_generation();
    strcpy(sr_parameter.enroll_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 1000; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 1500; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 1000; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_61 function ************************/
#endif /* #if (SPEECH_RECO) && (KEYBEEP)*/
#if (SPEECH_RECO)
  /********************************************************************************/
  /*                                                                              */
  /*                     AUDIO SPEECH_RECO_ENROLL CALLBACK FUNCTION               */
  /*                                                                              */
  /* Decription: call back funtion used for the speech reco enroll test.          */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_sr_enroll_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:SR ENROLL CALLBACK FUNCTION WITH A WRONG STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:SR ENROLL CALLBACK FUNCTION WITH A GOOD STATUS");
    }
  }
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 70                      */
  /*                                                                              */
  /* Decription: Start a speec hreco enrollment and wait until it stopped         */
  /* automatically.                                                               */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_70 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_ENROLL_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_70 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 71                      */
  /*                                                                              */
  /* Decription: Start a SR enroll and stop it after the L1 SR enroll start       */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_71 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_ENROLL_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_NO_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*-------------------------------------------------------------------------*/
    /* stop the SR enroll with no error parameters                               */
    /*---------------------------------------------------------------------------*/

    /* stop the SR enroll */
    if ( audio_sr_enroll_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_71 function ************************/


  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 72                      */
  /*                                                                              */
  /* Decription: Start a SR enroll and stop it before the L1 start                */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_72 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_ENROLL_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*---------------------------------------------------------------------------*/
    /* stop the SR enroll with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR enroll */
    if ( audio_sr_enroll_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_72 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 73                      */
  /*                                                                              */
  /* Decription: Start twice the SR_ENROLL                                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_73 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_ENROLL_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_NO_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/jerome");
    sr_parameter.record_speech = AUDIO_SR_NO_RECORD_SPEECH;

    /* start the SR enroll */
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_73 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 74                      */
  /*                                                                              */
  /* Decription: Start the sr enroll and send 2 stop                              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_74 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_ENROLL_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*---------------------------------------------------------------------------*/
    /* stop the SR enroll with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR enroll */
    if ( audio_sr_enroll_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*---------------------------------------------------------------------------*/
    /* stop the SR enroll with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR enroll */
    if ( audio_sr_enroll_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_74 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 75                      */
  /*                                                                              */
  /* Decription: Check the parameters error manager                 .             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_75 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_ENROLL_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/francois");
    sr_parameter.record_speech = 12;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the SR enroll with no error parameters                               */
      /*----------------------------------------------------------------------------*/
      /* fill the SR enroll parameter */
      strcpy(sr_parameter.enroll_name,"/sr/francois/");
      sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

      /* start the SR enroll */
      audio_keybeep_generation();
      if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_ERROR)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_75 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 76                      */
  /*                                                                              */
  /* Decription: Start a SR enroll and wait until it stopped automatically.       */
  /*             At this moment, the stop is requested by the entity              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_76 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_ENROLL_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*---------------------------------------------------------------------------*/
    /* stop the SR enroll with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR enroll */
    if ( audio_sr_enroll_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_76 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 77                      */
  /*                                                                              */
  /* Decription: Stop SR with stop return made by a call back function.           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_77 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_RV_RETURN               return_path;

    p_audio_test->audio_call_back_called = 1;

    /*-------------------------------------------------------------------------*/
    /* stop the SR enroll with no error parameters                               */
    /*---------------------------------------------------------------------------*/

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = audio_sr_enroll_callback_test_function;
    return_path.addr_id = rvf_get_taskid();

    /* stop the SR enroll */
    if ( audio_sr_enroll_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_77 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 78                      */
  /*                                                                              */
  /* Decription: Start a 6 SR enroll.                                             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_78 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_ENROLL_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /*                              francois                                      */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              eric                                          */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/eric");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              christian                                     */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/christian");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              david                                         */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/david");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              pascal                                        */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/pascal");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              stephanie                                     */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR enroll with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR enroll parameter */
    strcpy(sr_parameter.enroll_name,"/sr/stephanie");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR enroll */
    audio_keybeep_generation();
    if ( audio_sr_enroll_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_ENROLL_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_ENROLL_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_ENROLL_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_ENROLL_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_ENROLL_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_ENROLL_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/eric");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/christian");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/david");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/pascal");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/stephanie");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

      return(TEST_PASSED);
  }
  /********************************************************************************/
  /*                                                                              */
  /*                     AUDIO SPEECH_RECO_UPDATE CALLBACK FUNCTION               */
  /*                                                                              */
  /* Decription: call back funtion used for the speech reco update test.          */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_sr_update_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:SR UPDATE CALLBACK FUNCTION WITH A WRONG STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:SR UPDATE CALLBACK FUNCTION WITH A GOOD STATUS");
    }
  }
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 80                      */
  /*                                                                              */
  /* Decription: Start a speec hreco update and wait until it stopped             */
  /* automatically.                                                               */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_80 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update without check */
    p_audio_test->vocabuary_size = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_80 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 81                      */
  /*                                                                              */
  /* Decription: Start a SR update and stop it after the L1 SR update start       */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_81 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update without check */
    p_audio_test->vocabuary_size = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_NO_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != FALSE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*---------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/

    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_81 function ************************/


  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 82                      */
  /*                                                                              */
  /* Decription: Start a SR update and stop it before the L1 start                */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_82 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update without check */
    p_audio_test->vocabuary_size = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*---------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_82 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 83                      */
  /*                                                                              */
  /* Decription: Start twice the SR_UPDATE                                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_83 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update without check */
    p_audio_test->vocabuary_size = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/jerome");
    sr_parameter.record_speech = AUDIO_SR_NO_RECORD_SPEECH;

    /* start the SR update */
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_83 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 84                      */
  /*                                                                              */
  /* Decription: Start the sr update and send 2 stop                              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_84 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update without check */
    p_audio_test->vocabuary_size = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*---------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*---------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_84 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 85                      */
  /*                                                                              */
  /* Decription: Check the parameters error manager                 .             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_85 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update without check */
    p_audio_test->vocabuary_size = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = 12;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the SR update with no error parameters                               */
      /*----------------------------------------------------------------------------*/
      /* fill the SR update parameter */
      strcpy(sr_parameter.update_name,"/sr/francois/");
      sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

      /* start the SR update */
      audio_keybeep_generation();
      if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_ERROR)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*----------------------------------------------------------------------------*/
      /* start the SR update with no error parameters                               */
      /*----------------------------------------------------------------------------*/
      /* fill the SR update parameter */
      strcpy(sr_parameter.update_name,"/sr/tibo");
      sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

      /* start the SR update */
      audio_keybeep_generation();
      if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_ERROR)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_85 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 86                      */
  /*                                                                              */
  /* Decription: Start a SR update and wait until it stopped automatically.       */
  /*             At this moment, the stop is requested by the entity              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_86 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update without check */
    p_audio_test->vocabuary_size = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*---------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_86 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 87                      */
  /*                                                                              */
  /* Decription: Stop SR with stop return made by a call back function.           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_87 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_RV_RETURN               return_path;

    /* update test mode : update without check */
    p_audio_test->vocabuary_size = 1;

    p_audio_test->audio_call_back_called = 1;

    /*-------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = audio_sr_update_callback_test_function;
    return_path.addr_id = rvf_get_taskid();

    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_87 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 88                      */
  /*                                                                              */
  /* Decription: Start a 6 SR update.                                             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_88 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER vm_play_parameter;
    #endif

    /* update test mode : update without check */
    p_audio_test->vocabuary_size = 1;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /*                              francois                                      */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              eric                                          */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/eric");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              christian                                     */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/christian");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              david                                         */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/david");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              pascal                                        */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/pascal");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              stephanie                                     */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/stephanie");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_REQ*)p_message)->speech      != TRUE) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/eric");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/christian");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/david");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/pascal");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/stephanie");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

      return(TEST_PASSED);
  }

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 90                      */
  /*                                                                              */
  /* Decription: Start a speech reco update and wait until it stopped             */
  /* automatically.                                                               */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_90 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update with check */
    p_audio_test->vocabuary_size = 10;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_90 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 91                      */
  /*                                                                              */
  /* Decription: Start a SR update and stop it after the L1 SR update start       */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_91 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update with check */
    p_audio_test->vocabuary_size = 10;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_NO_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != FALSE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*-------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/

    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_81 function ************************/


  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 92                      */
  /*                                                                              */
  /* Decription: Start a SR update and stop it before the L1 start                */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_92 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update with check */
    p_audio_test->vocabuary_size = 10;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*---------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_92 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 93                      */
  /*                                                                              */
  /* Decription: Start twice the SR_UPDATE                                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_93 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update with check */
    p_audio_test->vocabuary_size = 10;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/jerome");
    sr_parameter.record_speech = AUDIO_SR_NO_RECORD_SPEECH;

    /* start the SR update */
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_93 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 94                      */
  /*                                                                              */
  /* Decription: Start the sr update and send 2 stop                              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_94 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update with check */
    p_audio_test->vocabuary_size = 10;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*---------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*---------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_94 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 95                      */
  /*                                                                              */
  /* Decription: Check the parameters error manager                 .             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_95 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update with check */
    p_audio_test->vocabuary_size = 10;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = 12;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the SR update with no error parameters                               */
      /*----------------------------------------------------------------------------*/
      /* fill the SR update parameter */
      strcpy(sr_parameter.update_name,"/sr/francois/");
      sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

      /* start the SR update */
      audio_keybeep_generation();
      if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_ERROR)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*----------------------------------------------------------------------------*/
      /* start the SR update with no error parameters                               */
      /*----------------------------------------------------------------------------*/
      /* fill the SR update parameter */
      strcpy(sr_parameter.update_name,"/sr/tibo");
      sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

      /* start the SR update */
      audio_keybeep_generation();
      if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_ERROR)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_95 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 96                      */
  /*                                                                              */
  /* Decription: Start a SR update and wait until it stopped automatically.       */
  /*             At this moment, the stop is requested by the entity              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_96 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* update test mode : update with check */
    p_audio_test->vocabuary_size = 10;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*---------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_96 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 97                      */
  /*                                                                              */
  /* Decription: Stop SR with stop return made by a call back function.           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_97 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_RV_RETURN               return_path;

    /* update test mode : update with check */
    p_audio_test->vocabuary_size = 10;

    p_audio_test->audio_call_back_called = 1;

    /*-------------------------------------------------------------------------*/
    /* stop the SR update with no error parameters                               */
    /*---------------------------------------------------------------------------*/

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = audio_sr_update_callback_test_function;
    return_path.addr_id = rvf_get_taskid();

    /* stop the SR update */
    if ( audio_sr_update_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_97 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 98                      */
  /*                                                                              */
  /* Decription: Start a 6 SR update.                                             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_98 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_UPDATE_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER vm_play_parameter;
    #endif

    /* update test mode : update with check */
    p_audio_test->vocabuary_size = 10;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /*                              francois                                      */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/francois");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              eric                                          */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/eric");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              christian                                     */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/christian");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              david                                         */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/david");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              pascal                                        */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/pascal");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*----------------------------------------------------------------------------*/
    /*                              stephanie                                     */
    /*----------------------------------------------------------------------------*/
    /*----------------------------------------------------------------------------*/
    /* start the SR update with no error parameters                               */
    /*----------------------------------------------------------------------------*/
    /* fill the SR update parameter */
    strcpy(sr_parameter.update_name,"/sr/stephanie");
    sr_parameter.record_speech = AUDIO_SR_RECORD_SPEECH;

    /* start the SR update */
    audio_keybeep_generation();
    if ( audio_sr_update_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_UPDATE_CHECK_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->word_index  != 0) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->speech      != TRUE) ||
           (((T_MMI_SR_UPDATE_CHECK_REQ*)p_message)->vocabulary_size != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_UPDATE_CHECK_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_UPDATE_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_UPDATE_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/francois");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/eric");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/christian");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/david");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/pascal");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,"/sr/stephanie");
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(5000));
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

      return(TEST_PASSED);
  }

  /********************************************************************************/
  /*                                                                              */
  /*                     AUDIO SPEECH_RECO_RECO CALLBACK FUNCTION                 */
  /*                                                                              */
  /* Decription: call back funtion used for the speech reco test.                 */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_sr_reco_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:SR RECO CALLBACK FUNCTION WITH A WRONG STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:SR RECO CALLBACK FUNCTION WITH A GOOD STATUS");
    }
  }
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 100                      */
  /*                                                                              */
  /* Decription: Start a speech reco and wait until it stopped automatically.     */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_100 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_RECO_PARAMETER   sr_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR reco with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the SR reco parameter */
    strcpy(sr_parameter.database_directory,"/sr");
    p_audio_test->vocabuary_size = 10;

    /* start the SR reco */
    audio_keybeep_generation();
    if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_RECO_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_RECO_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_RECO_REQ*)p_message)->vocabulary_size    != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_RECO_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,
        ((T_AUDIO_SR_RECO_STATUS *)p_message)->word_recognized);
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_100 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 101                     */
  /*                                                                              */
  /* Decription: Start a SR reco and stop it after the L1 SR reco start           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_101 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_RECO_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR reco with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the SR reco parameter */
    strcpy(sr_parameter.database_directory,"/sr");
    p_audio_test->vocabuary_size = 10;

    /* start the SR reco */
    audio_keybeep_generation();
    if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_RECO_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_RECO_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_RECO_REQ*)p_message)->vocabulary_size    != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*-------------------------------------------------------------------------*/
    /* stop the SR reco with no error parameters                               */
    /*---------------------------------------------------------------------------*/

    /* stop the SR reco */
    if ( audio_sr_reco_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_RECO_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_RECO_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_101 function ************************/


  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 102                     */
  /*                                                                              */
  /* Decription: Start a SR reco and stop it before the L1 start                  */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_102 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_RECO_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR reco with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the SR reco parameter */
    strcpy(sr_parameter.database_directory,"/sr");
    p_audio_test->vocabuary_size = 10;

    /* start the SR reco */
    audio_keybeep_generation();
    if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*---------------------------------------------------------------------------*/
    /* stop the SR reco with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR reco */
    if ( audio_sr_reco_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_RECO_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_RECO_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_RECO_REQ*)p_message)->vocabulary_size    != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_RECO_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_RECO_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_102 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 103                     */
  /*                                                                              */
  /* Decription: Start twice the SR_RECO                                          */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_103 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_RECO_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR reco with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the SR reco parameter */
    strcpy(sr_parameter.database_directory,"/sr");
    p_audio_test->vocabuary_size = 10;

    /* start the SR reco */
    audio_keybeep_generation();
    if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_RECO_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_RECO_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_RECO_REQ*)p_message)->vocabulary_size    != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the SR reco with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the SR reco parameter */
    strcpy(sr_parameter.database_directory,"/sr");
    p_audio_test->vocabuary_size = 10;

    /* start the SR reco */
    if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_RECO_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_RECO_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,
        ((T_AUDIO_SR_RECO_STATUS *)p_message)->word_recognized);
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_103 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 104                     */
  /*                                                                              */
  /* Decription: Start the sr reco and send 2 stop                                */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_104 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_RECO_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR reco with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the SR reco parameter */
    strcpy(sr_parameter.database_directory,"/sr");
    p_audio_test->vocabuary_size = 10;

    /* start the SR reco */
    audio_keybeep_generation();
    if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_RECO_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_RECO_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_RECO_REQ*)p_message)->vocabulary_size    != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #endif

    /*---------------------------------------------------------------------------*/
    /* stop the SR reco with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR reco */
    if ( audio_sr_reco_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*---------------------------------------------------------------------------*/
    /* stop the SR reco with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR reco */
    if ( audio_sr_reco_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_RECO_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_STOP_CON,
                                    error_type, 1 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_RECO_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_RECO_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status == AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_104 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 105                     */
  /*                                                                              */
  /* Decription: Check the parameters error manager                 .             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_105 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_RECO_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the SR reco with error parameters                                    */
      /*----------------------------------------------------------------------------*/
      /* fill the SR reco parameter */
      strcpy(sr_parameter.database_directory,"/srco");
      p_audio_test->vocabuary_size = 10;

      /* start the SR reco */
      audio_keybeep_generation();
      if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_ERROR)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the SR reco with error parameters                                    */
    /*----------------------------------------------------------------------------*/
    /* fill the SR reco parameter */
    strcpy(sr_parameter.database_directory,"/reco");
    p_audio_test->vocabuary_size = 0;

    /* start the SR reco */
    audio_keybeep_generation();
    if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the SR reco with error parameters                                    */
    /*----------------------------------------------------------------------------*/
    /* fill the SR reco parameter */
    strcpy(sr_parameter.database_directory,"/sr/");
    p_audio_test->vocabuary_size = 0;

    /* start the SR reco */
    audio_keybeep_generation();
    if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_105 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 106                     */
  /*                                                                              */
  /* Decription: Start a SR reco and wait until it stopped automatically.       */
  /*             At this moment, the stop is requested by the entity              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_106 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_SR_RECO_PARAMETER sr_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    #ifndef _WINDOWS
      T_AUDIO_VM_PLAY_PARAMETER       vm_play_parameter;
    #endif

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the SR reco with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the SR reco parameter */
    strcpy(sr_parameter.database_directory,"/sr");
    p_audio_test->vocabuary_size = 10;

    /* start the SR reco */
    audio_keybeep_generation();
    if ( audio_sr_reco_start(&sr_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_SR_RECO_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( (((T_MMI_SR_RECO_REQ*)p_message)->database_id != 0) ||
           (((T_MMI_SR_RECO_REQ*)p_message)->vocabulary_size    != 10) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_SR_RECO_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_RECO_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*---------------------------------------------------------------------------*/
    /* stop the SR reco with no error parameters                               */
    /*---------------------------------------------------------------------------*/
    /* stop the SR reco */
    if ( audio_sr_reco_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_SR_RECO_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_SR_RECO_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the voice memo play-back                                             */
      /*----------------------------------------------------------------------------*/
      /* fill the voice memo play parameters */
      strcpy(vm_play_parameter.memo_name,
        ((T_AUDIO_SR_RECO_STATUS *)p_message)->word_recognized);
      /* start the voice memo play */
      if ( audio_vm_play_start(&vm_play_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_VM_PLAY_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_VM_PLAY_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_106 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 107                     */
  /*                                                                              */
  /* Decription: Stop SR with stop return made by a call back function.           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_107 (T_RV_REGR_ERR_TYPE *error_type)
  {

    T_RV_RETURN               return_path;

    p_audio_test->audio_call_back_called = 1;

    /*-------------------------------------------------------------------------*/
    /* stop the SR reco with no error parameters                               */
    /*---------------------------------------------------------------------------*/

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = audio_sr_reco_callback_test_function;
    return_path.addr_id = rvf_get_taskid();

    /* stop the SR reco */
    if ( audio_sr_reco_stop(return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_107 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_keybeep_generation                                 */
  /*                                                                              */
  /*    Purpose:  generate a keybeep before the SR task.                          */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        none.                                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        none.                                                                 */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        27 November 2001   Francois Mazard: Creation.                         */
  /*                                                                              */
  /********************************************************************************/
  void audio_keybeep_generation( void )
  {
    T_AUDIO_KEYBEEP_PARAMETER keybeep_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;
    T_RV_REGR_ERR_TYPE        error_type;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the keybeep with no error parameters                                 */
    /*----------------------------------------------------------------------------*/
    /* fill the keybeep parameter */
    keybeep_parameter.frequency_beep[0] = 270; /* Hz */
    keybeep_parameter.amplitude_beep[0] = -20; /* dB */
    keybeep_parameter.frequency_beep[1] = 270; /* Hz */
    keybeep_parameter.amplitude_beep[1] = -10; /* dB */
    keybeep_parameter.duration = 500; /* ms */

    /* start the keybeep */
    if ( audio_keybeep_start(keybeep_parameter, return_path) != AUDIO_OK)
    {
      RV_TEST_TRACE_ERROR ("!!! TEST ERROR !!! KEYBEEP FOR SR: API FUNCTION ERROR");
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_KEYBEEP_START_REQ,
                                                &error_type );
      if (error_type != NO_ERR)
      {
        RV_TEST_TRACE_ERROR ("!!! TEST ERROR !!! KEYBEEP FOR SR: NO START REQ RECEIVE");
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_START_CON,
                                    &error_type, 0 );
      if (error_type != NO_ERR)
      {
        RV_TEST_TRACE_ERROR ("!!! TEST ERROR !!! KEYBEEP FOR SR: NO START CON SEND");
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_KEYBEEP_STOP_CON,
                                    &error_type, 0 );
      if (error_type != NO_ERR)
      {
        RV_TEST_TRACE_ERROR ("!!! TEST ERROR !!! KEYBEEP FOR SR: NO STOP CON SEND");
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_KEYBEEP_STATUS_MSG,
                                              &error_type );
    if (error_type != NO_ERR)
    {
      RV_TEST_TRACE_ERROR ("!!! TEST ERROR !!! KEYBEEP FOR SR: NO MSG STATUS");
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_KEYBEEP_STATUS *)p_message)->status != AUDIO_OK)
    {
      RV_TEST_TRACE_ERROR ("!!! TEST ERROR !!! KEYBEEP FOR SR: STATUS ERROR");
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);
  }

  /******************** End of audio_keybeep_generation function ********************/
#endif /* SPEECH_RECO */

#if (MELODY_E2)
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO MELODY_E2 CALLBACK FUNCTION                 */
  /*                                                                              */
  /* Description: call back funtion used for the melody E2 test.                  */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_melody_E2_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:MELODY E2 CALLBACK FUNCTION WITH A WRONG STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:MELODY E2 CALLBACK FUNCTION WITH A GOOD STATUS");
    }
  }

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 110                     */
  /*                                                                              */
  /* Description: Start a Melody_E2 and wait until it stopped automatically.      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_110 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER  melody_E2_parameter;
    T_RV_RETURN                  return_path;
    void                         *p_message = NULL;

    /* initialize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id       = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/

    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameters */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load file instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }


    /*************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/


    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
        p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                  error_type );
        if (*error_type != NO_ERR)
        {
          return (audio_test_regr_return_verdict(*error_type));
        }

        /* Check the message parameters */
        if ( (((T_MMI_MELODY_E2_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback ) ||
           (((T_MMI_MELODY_E2_REQ *)p_message)->session_id != 0 ) )
        {
          rvf_free_buf((T_RVF_BUFFER *)p_message);
          *error_type = EVENT_PARAM_ERROR;
          return (audio_test_regr_return_verdict(*error_type));
        }
        rvf_free_buf((T_RVF_BUFFER *)p_message);

        /*------------------------------------------------------------------------------*/
        /* send the L1 start confirmation message                                       */
        /*------------------------------------------------------------------------------*/
        audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                      error_type, 0 );
        if (*error_type != NO_ERR)
        {
          return (audio_test_regr_return_verdict(*error_type));
        }

        /*------------------------------------------------------------------------------*/
        /* send the L1 stop confirmation message                                        */
        /*------------------------------------------------------------------------------*/
        audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                      error_type, 0 );
        if (*error_type != NO_ERR)
        {
          return (audio_test_regr_return_verdict(*error_type));
        }
      #else
        rvf_delay(RVF_MS_TO_TICKS(30000));
      #endif


      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_110 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 111                     */
  /*                                                                              */
  /* Description: Start a Melody E2 and stop it after the L1 Melody_E2 start      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_111 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER       melody_E2_parameter;
    T_AUDIO_MELODY_E2_STOP_PARAMETER  melody_E2_stop_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }


    /*************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_E2_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback) ||
           ( ((T_MMI_MELODY_E2_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the melody                                                              */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E2_stop_parameter.melody_E2_name,"/meloe2/USA");

    if ( audio_melody_E2_stop(&melody_E2_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_111 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 112                     */
  /*                                                                              */
  /* Description: Start a Melody_E2 and stop it before the L1 Melody_E2 start     */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_112 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER                   melody_E2_parameter;
    T_AUDIO_MELODY_E2_STOP_PARAMETER              melody_E2_stop_parameter;
    T_RV_RETURN     return_path;
    void            *p_message = NULL;

    /* initalize the return path, for this test, the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }


    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the Melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_E2_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback) ||
           ( ((T_MMI_MELODY_E2_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the Melody_E2 with no error parameters                                  */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E2_stop_parameter.melody_E2_name,"/meloe2/USA");

    if ( audio_melody_E2_stop(&melody_E2_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_112 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 113                     */
  /*                                                                              */
  /* Decription: Start a Melody_E2 and stop it before the FFS initialization      */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_113 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER                   melody_E2_parameter;
    T_AUDIO_MELODY_E2_STOP_PARAMETER              melody_E2_stop_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }


    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the Melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }
    /*------------------------------------------------------------------------------*/
    /* wait before to stop the melody                                               */
    /*------------------------------------------------------------------------------*/
    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif


    /*------------------------------------------------------------------------------*/
    /* stop the Melody_E2 with no error parameters                                  */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E2_stop_parameter.melody_E2_name,"/meloe2/USA");

    if ( audio_melody_E2_stop(&melody_E2_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_113 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 114                     */
  /*                                                                              */
  /* Decription: Start twice the Melody_E2                                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/

  T_RV_REGR_RET audio_regr_test_114 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER                  melody_E2_parameter;
    T_RV_RETURN      return_path;
    void             *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_GAME_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }


    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(3000));
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/LASER");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY1_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_E2_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback) ||
           ( ((T_MMI_MELODY_E2_REQ *)p_message)->session_id != 1 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY1_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY1_E2_STOP_CON, error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* This delay is introduced in order to receive the melody1 end status before the */
      /* melody0 end status. */
      rvf_delay(10);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(15000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifndef _WINDOWS
      rvf_delay(RVF_MS_TO_TICKS(40000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }

  /******************** End of audio_regr_test_114 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 115                     */
  /*                                                                              */
  /* Decription: Start the melody E2 and send 2 stop                              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_115 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER                  melody_E2_parameter;
    T_AUDIO_MELODY_E2_STOP_PARAMETER             melody_E2_stop_parameter;
    T_RV_RETURN    return_path;
    void           *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }


    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif


    /*------------------------------------------------------------------------------*/
    /* stop the melody                                                              */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E2_stop_parameter.melody_E2_name,"/meloe2/USA");

    if ( audio_melody_E2_stop(&melody_E2_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #else
      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the melody E2                                                           */
    /*------------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    strcpy(melody_E2_stop_parameter.melody_E2_name,"/meloe2/LASER");

    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    if ( audio_melody_E2_stop(&melody_E2_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif
    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_115 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 116                     */
  /*                                                                              */
  /* Decription: Start a Melody and wait until it stopped automatically.          */
  /*             At this moment, the stop is requested by the entity              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_116 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER       melody_E2_parameter;
    T_AUDIO_MELODY_E2_STOP_PARAMETER  melody_E2_stop_parameter;
    T_RV_RETURN   return_path;
    void          *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif


      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* stop the melody                                                              */
      /*------------------------------------------------------------------------------*/
      /* initalize the return path, for this test the return path is a message */
      strcpy(melody_E2_stop_parameter.melody_E2_name,"/meloe2/LASER");

      return_path.callback_func = NULL;
      return_path.addr_id = rvf_get_taskid();

      if ( audio_melody_E2_stop(&melody_E2_stop_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_ERROR)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_116 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 117                     */
  /*                                                                              */
  /* Decription: Start a Melody and wait until it stopped automatically.          */
  /*             Note the stop return is made by a call back function.            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_117 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER       melody_E2_parameter;
    T_AUDIO_MELODY_E2_STOP_PARAMETER  melody_E2_stop_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    p_audio_test->audio_call_back_called = 1;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id       = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }


    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif


      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* stop the melody E2                                                           */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E2_stop_parameter.melody_E2_name,"/meloe2/LASER");

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = audio_melody_E2_callback_test_function;
    return_path.addr_id = 0;

    if ( audio_melody_E2_stop(&melody_E2_stop_parameter,return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the call back function is called                                        */
    /*------------------------------------------------------------------------------*/

    /* Wait until the audio process is done */
    while (p_audio_test->audio_call_back_called == 1)
    {
      rvf_delay(RVF_MS_TO_TICKS(500));
    }

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_117 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 118                     */
  /*                                                                              */
  /* Decription: 3 start of the Melody_E2                                         */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/

  T_RV_REGR_RET audio_regr_test_118 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER melody_E2_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;
    T_AUDIO_MELODY_E2_STOP_PARAMETER melody_E2_stop_parameter;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_GAME_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(3000));
    #endif


    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
        strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/LASER");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
        strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/LASER");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_GAME_MODE;

    /* start the melody */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY1_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
     if ( ( ((T_MMI_MELODY_E2_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback ) ||
           ( ((T_MMI_MELODY_E2_REQ *)p_message)->session_id != 1 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY1_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(10000));
    #endif

      /*------------------------------------------------------------------------------*/
      /* stop the melody                                                              */
      /*------------------------------------------------------------------------------*/
      /* initalize the return path, for this test the return path is a message */
      strcpy(melody_E2_stop_parameter.melody_E2_name,"/meloe2/LASER");

      return_path.callback_func = NULL;
      return_path.addr_id = rvf_get_taskid();

      if ( audio_melody_E2_stop(&melody_E2_stop_parameter, return_path) != AUDIO_OK)
      {
        *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY1_E2_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY1_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #else
      rvf_delay(RVF_MS_TO_TICKS(46000));
    #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);


      return(TEST_PASSED);
  }

  /******************** End of audio_regr_test_118 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 119                     */
  /*                                                                              */
  /* Decription: Start twice the Melody_E2 in normal mode.                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/

  T_RV_REGR_RET audio_regr_test_119 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER melody_E2_parameter;
    T_RV_RETURN                 return_path;
    void                        *p_message = NULL;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }


    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback ) ||
           ( ((T_MMI_MELODY_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(1000));
    #endif


    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
        strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* start the melody */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_MODE_ERROR)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(5000));
    #endif

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      return(TEST_PASSED);
  }


  /******************** End of audio_regr_test_119 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 120                     */
  /*                                                                              */
  /* Decription: Start a Melody E2 in loopback mode and stop it after the         */
  /* L1 Melody_E2 start                                                           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_120 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER       melody_E2_parameter;
    T_AUDIO_MELODY_E2_STOP_PARAMETER  melody_E2_stop_parameter;
    T_RV_RETURN    return_path;
    void           *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* fill the melody E2 parameter */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/LASER");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_GAME_MODE;

    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load file instruments                                  */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));					       }


    /* ************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /* Check the message parameters */
      if ( ( ((T_MMI_MELODY_E2_REQ *)p_message)->loopback != melody_E2_parameter.E2_loopback ) ||
           ( ((T_MMI_MELODY_E2_REQ *)p_message)->session_id != 0 ) )
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #else
      rvf_delay(RVF_MS_TO_TICKS(120000));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the melody                                                              */
    /*------------------------------------------------------------------------------*/
    strcpy(melody_E2_stop_parameter.melody_E2_name,"/meloe2/LASER");

    if ( audio_melody_E2_stop(&melody_E2_stop_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_MELODY0_E2_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_MELODY0_E2_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

    #endif

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_120 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 121                     */
  /*                                                                              */
  /* Decription: Check the parameters error manager                  .            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_121 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER  melody_E2_load_file_instr_parameter;
    T_AUDIO_MELODY_E2_PARAMETER melody_E2_parameter;
    T_RV_RETURN                 return_path;

    /* initalize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load instruments                                       */
    /*----------------------------------------------------------------------------*/

    /* fill the melody E2 load instruments parameters */
    strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"/meloe2/default");

    /* start the melody_E2 load instruments */
    if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                        != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the melody E2 load instruments                                       */
    /*----------------------------------------------------------------------------*/

    #ifndef _WINDOWS
       /* fill the melody E2 load instruments parameters */
       strcpy(melody_E2_load_file_instr_parameter.melody_E2_file_name,"toto.lsi");

       /* start the melody_E2 load instruments */
       if (audio_melody_E2_load_file_instruments(&melody_E2_load_file_instr_parameter)
                                           != AUDIO_ERROR)
       {
         *error_type = FUNCTION_ERROR;
         return (audio_test_regr_return_verdict(*error_type));
       }
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters */
    melody_E2_parameter.E2_loopback = 10;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = 10;

    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_ERROR)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifndef _WINDOWS
      /*----------------------------------------------------------------------------*/
      /* start the melody E2                                                        */
      /*----------------------------------------------------------------------------*/
      /* fill the melody E2 parameters */
      melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
      strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/AFR");
      melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;

      /* start the melody_E2 */
      if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_ERROR)
      {
        *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
      }
    #endif

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_121 function ***********************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 122                     */
  /*                                                                              */
  /* Description: Start a Melody_E2 without loading the instruments file          */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_122 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_MELODY_E2_PARAMETER  melody_E2_parameter;
    T_RV_RETURN                  return_path;
    void                         *p_message = NULL;

    /* initialize the return path, for this test the return path is a message */
    return_path.callback_func = NULL;
    return_path.addr_id       = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* fill the melody E2 parameters                                              */
    /*----------------------------------------------------------------------------*/

    /* fill the melody E2 parameters */
    melody_E2_parameter.E2_loopback = AUDIO_MELODY_NO_LOOPBACK;
    strcpy(melody_E2_parameter.melody_E2_name,"/meloe2/USA");
    melody_E2_parameter.melody_E2_mode = AUDIO_MELODY_NORMAL_MODE;


    /*************************************************************************** */
    /*----------------------------------------------------------------------------*/
    /* start the melody E2                                                        */
    /*----------------------------------------------------------------------------*/

    /* start the melody_E2 */
    if ( audio_melody_E2_start(&melody_E2_parameter, return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

      /*------------------------------------------------------------------------------*/
      /* wait the status message in the good mailbox                                  */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( AUDIO_MELODY_E2_STATUS_MSG,
                                              error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* check if it's the expected status                                            */
      /*------------------------------------------------------------------------------*/
      if (((T_AUDIO_MELODY_E2_STATUS *)p_message)->status != AUDIO_ERROR)
      {
        rvf_free_buf((T_RVF_BUFFER *)p_message);
        *error_type = EVENT_PARAM_ERROR;
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_122 function ************************/

 #endif /* MELODY_E2 */

 #if (L1_GTT == 1)
  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO TTY CALLBACK FUNCTION                       */
  /*                                                                              */
  /* Decription: call back funtion used for the keybeep test.                     */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  void audio_tty_callback_test_function (void *p_message)
  {
    if (((T_AUDIO_TTY_STATUS *)p_message)->status == AUDIO_ERROR)
    {
      p_audio_test->audio_call_back_called = EVENT_PARAM_ERROR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:TTY CALLBACK FUNCTION WITH A GOOD STATUS");
    }
    else
    {
      p_audio_test->audio_call_back_called = NO_ERR;

      RV_TEST_TRACE_LOW ("AUDIO TEST REGR:TTY CALLBACK FUNCTION WITH A WRONG STATUS");
    }
  }

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 130                     */
  /*                                                                              */
  /* Decription: Start TTY and stop it after the L1 TTY start                     */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_130 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_TTY_CONFIG_PARAMETER tty_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start TTY                                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill TTY parameter */
    tty_parameter.Mode = TTY_EXT_START;

    /* start the tty */
    if ( audio_tty_set_config(&tty_parameter, &return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_GTT_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_GTT_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_delay(RVF_MS_TO_TICKS(5000));
    #else
      rvf_delay(RVF_MS_TO_TICKS(500));
    #endif

    /*------------------------------------------------------------------------------*/
    /* stop the tty with no error parameters                                        */
    /*------------------------------------------------------------------------------*/
    /* fill TTY parameter */
    tty_parameter.Mode = TTY_STOP;

    if ( audio_tty_set_config(&tty_parameter, &return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_GTT_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_GTT_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TTY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TTY_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_130 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 131                     */
  /*                                                                              */
  /* Decription: Start TTY and stop it before the L1 tty start                    */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_131 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_TTY_CONFIG_PARAMETER tty_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start TTY                                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill TTY parameter */
    tty_parameter.Mode = TTY_EXT_START;

    /* start the tty */
    if ( audio_tty_set_config(&tty_parameter, &return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_GTT_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);
    #endif
    /*------------------------------------------------------------------------------*/
    /* stop the tty with no error parameters                                        */
    /*------------------------------------------------------------------------------*/
    /* fill TTY parameter */
    tty_parameter.Mode = TTY_STOP;

    if ( audio_tty_set_config(&tty_parameter, &return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_GTT_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_GTT_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_GTT_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TTY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TTY_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_131 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 132                     */
  /*                                                                              */
  /* Decription: Start twice TTY + stop                                           */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_132 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_TTY_CONFIG_PARAMETER tty_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start TTY                                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill TTY parameter */
    tty_parameter.Mode = TTY_EXT_START;

    /* start the tty */
    if ( audio_tty_set_config(&tty_parameter, &return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_GTT_START_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 start confirmation message                                       */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_GTT_START_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
      rvf_delay(RVF_MS_TO_TICKS(2000));
    #else
      rvf_delay(RVF_MS_TO_TICKS(500));
    #endif

    /*----------------------------------------------------------------------------*/
    /* start the tty with no error parameters                                     */
    /*----------------------------------------------------------------------------*/
    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start TTY                                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill TTY parameter */
    tty_parameter.Mode = TTY_EXT_START;

    /* start the tty */
    if ( audio_tty_set_config(&tty_parameter, &return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TTY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TTY_STATUS *)p_message)->status != AUDIO_ERROR)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    /*------------------------------------------------------------------------------*/
    /* stop the tty with no error parameters                                        */
    /*------------------------------------------------------------------------------*/
    /* fill TTY parameter */
    tty_parameter.Mode = TTY_STOP;

    if ( audio_tty_set_config(&tty_parameter, &return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 stop message in the good mailbox                                 */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_GTT_STOP_REQ,
                                                error_type );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }

      rvf_free_buf((T_RVF_BUFFER *)p_message);

      /*------------------------------------------------------------------------------*/
      /* send the L1 stop confirmation message                                        */
      /*------------------------------------------------------------------------------*/
      audio_test_regr_send_message( MMI_GTT_STOP_CON,
                                    error_type, 0 );
      if (*error_type != NO_ERR)
      {
        return (audio_test_regr_return_verdict(*error_type));
      }
    #endif
    /*------------------------------------------------------------------------------*/
    /* wait the status message in the good mailbox                                  */
    /*------------------------------------------------------------------------------*/
    p_message = audio_test_regr_wait_message( AUDIO_TTY_STATUS_MSG,
                                              error_type );
    if (*error_type != NO_ERR)
    {
      return (audio_test_regr_return_verdict(*error_type));
    }

    /*------------------------------------------------------------------------------*/
    /* check if it's the expected status                                            */
    /*------------------------------------------------------------------------------*/
    if (((T_AUDIO_TTY_STATUS *)p_message)->status != AUDIO_OK)
    {
      rvf_free_buf((T_RVF_BUFFER *)p_message);
      *error_type = EVENT_PARAM_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    rvf_free_buf((T_RVF_BUFFER *)p_message);

    return(TEST_PASSED);
  }
  /******************** End of audio_regr_test_132 function ************************/

  /********************************************************************************/
  /*                                                                              */
  /*                            AUDIO NON REGRESSION TEST 133                     */
  /*                                                                              */
  /* Decription: Start TTY and send 2 stop                                        */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RV_REGR_RET audio_regr_test_133 (T_RV_REGR_ERR_TYPE *error_type)
  {
    T_AUDIO_TTY_CONFIG_PARAMETER tty_parameter;
    T_RV_RETURN               return_path;
    void                      *p_message = NULL;

    /* initalize the return path, for this test the return path is an message */
    return_path.callback_func = NULL;
    return_path.addr_id = rvf_get_taskid();

    /*----------------------------------------------------------------------------*/
    /* start TTY                                                                  */
    /*----------------------------------------------------------------------------*/
    /* fill TTY parameter */
    tty_parameter.Mode = TTY_EXT_START;

    /* start the tty */
    if ( audio_tty_set_config(&tty_parameter, &return_path) != AUDIO_OK)
    {
      *error_type = FUNCTION_ERROR;
      return (audio_test_regr_return_verdict(*error_type));
    }

    #ifdef _WINDOWS
      /*------------------------------------------------------------------------------*/
      /* wait the L1 start message in the good mailbox                                */
      /*------------------------------------------------------------------------------*/
      p_message = audio_test_regr_wait_message( MMI_GTT_START_REQ,
                                                error_type );
      if (*error_