/****************************************************************************/
/*                                                                          */
/*  File Name:  audio_mode_save.c                                           */
/*                                                                          */
/*  Purpose:  This file contains all the functions used for audio mode      */
/*            save services.                                                */
/*                                                                          */
/*  Version   0.1                                                           */
/*                                                                          */
/*  Date        Modification                                                */
/*  ------------------------------------------------------------------------*/
/*  14 Jan 2002  Create                                                     */
/*                                                                          */
/*  Author      Francois Mazard                                             */
/*                                                                          */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved*/
/****************************************************************************/

#include "rv/rv_defined_swe.h"

#ifdef RVM_AUDIO_MAIN_SWE
  #ifndef _WINDOWS
    #include "config/swconfig.cfg"
    #include "config/sys.cfg"
    #include "config/chipset.cfg"
  #endif

  #include "l1_confg.h"
  #include "rv/rv_general.h"
  #include "rvm/rvm_gen.h"
  #include "audio/audio_ffs_i.h"
  #include "audio/audio_api.h"
  #include "audio/audio_structs_i.h"
  #include "audio/audio_error_hdlr_i.h"
  #include "audio/audio_var_i.h"
  #include "audio/audio_messages_i.h"
  #include "audio/audio_macro_i.h"
  #include "rvf/rvf_target.h"
  #include "audio/audio_const_i.h"

  /* include the usefull L1 header */
  #ifdef _WINDOWS
    #define BOOL_FLAG
    //#define CHAR_FLAG
  #endif
  #include "l1_types.h"
  #include "l1audio_const.h"
  #include "l1audio_cust.h"
  #include "l1audio_defty.h"
  #include "l1audio_msgty.h"
  #include "l1audio_signa.h"
  #if TESTMODE
   #include "l1tm_defty.h"
  #endif
  #if (L1_GTT == 1)
    #include "l1gtt_const.h"
    #include "l1gtt_defty.h"
  #endif
  #include "l1_const.h"
  #include "l1_defty.h"
  #include "l1_msgty.h"
  #include "l1_signa.h"
  #ifdef _WINDOWS
    #define L1_ASYNC_C
  #endif
  #include "l1_varex.h"

  #ifdef _WINDOWS
    #include "audio/tests/audio_test.h"
  #endif

  /* external functions */
  /* read */
  extern T_AUDIO_RET audio_mode_voice_path_read               (T_AUDIO_VOICE_PATH_SETTING *data);
  extern T_AUDIO_RET audio_mode_microphone_mode_read          (INT8  *data);
  extern T_AUDIO_RET audio_mode_microphone_gain_read          (INT8  *data);
  extern T_AUDIO_RET audio_mode_microphone_extra_gain_read    (INT8  *data);
  extern T_AUDIO_RET audio_mode_microphone_output_bias_read   (INT8  *data);
  extern T_AUDIO_RET audio_mode_microphone_fir_read           (T_AUDIO_FIR_COEF *data);
  extern T_AUDIO_RET audio_mode_speaker_mode_read             (INT8  *data);
  extern T_AUDIO_RET audio_mode_speaker_gain_read             (INT8  *data);
  extern T_AUDIO_RET audio_mode_speaker_filter_read           (INT8  *data);
  extern T_AUDIO_RET audio_mode_speaker_fir_read              (T_AUDIO_FIR_COEF *data);
  extern T_AUDIO_RET audio_mode_speaker_buzzer_read           (INT8  *data);
  extern T_AUDIO_RET audio_mode_sidetone_gain_read            (INT8  *data);
  extern T_AUDIO_RET audio_mode_aec_read                      (T_AUDIO_AEC_CFG *data);
  extern T_AUDIO_RET audio_mode_speaker_volume_read           (T_AUDIO_SPEAKER_LEVEL *data);

  /* Define a macro to simplify the code */
  #define CHECK_STATUS(function, param)     if ( (function(&(param))) == AUDIO_ERROR ) \
                                            { \
                                              return(AUDIO_ERROR); \
                                            } \

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_save_send_status                              */
  /*                                                                              */
  /*    Purpose:  This function sends the audio mdoe save status to the entity.   */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        status,                                                               */
  /*        return path                                                           */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  void audio_mode_save_send_status (T_AUDIO_RET status, T_RV_RETURN return_path)
  {
    void *p_send_message = NULL;
    T_RVF_MB_STATUS mb_status = RVF_RED;

    while (mb_status == RVF_RED)
    {
      /* allocate the message buffer */
      mb_status = rvf_get_buf (p_audio_gbl_var->mb_external,
                               sizeof(T_AUDIO_SAVE_DONE),
                               (T_RVF_BUFFER **) (&p_send_message));

      /* If insufficient resources, then report a memory error and abort.               */
      /* and wait until more ressource is given */
      if (mb_status == RVF_RED)
      {
        audio_mode_error_trace(AUDIO_ENTITY_NO_MEMORY);
        rvf_delay(RVF_MS_TO_TICKS(1000));
      }
    }
    /*fill the header of the message */
    ((T_AUDIO_SAVE_DONE *)(p_send_message))->os_hdr.msg_id =
      AUDIO_MODE_SAVE_DONE;

    /* fill the status parameters */
    ((T_AUDIO_SAVE_DONE *)(p_send_message))->status = status;

    if (return_path.callback_func == NULL)
    {
      /* send the message to the entity */
      rvf_send_msg (return_path.addr_id,
                    p_send_message);
    }
    else
    {
      /* call the callback function */
      (*return_path.callback_func)((void *)(p_send_message));
	  rvf_free_buf((T_RVF_BUFFER *)p_send_message);
    }
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_get                                           */
  /*                                                                              */
  /*    Purpose:  This function fill the audio mode structure.                    */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        Audio mode structure.                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_get(T_AUDIO_MODE *p_audio_mode)
  {
    /* Audio voice path */
    CHECK_STATUS(audio_mode_voice_path_read, p_audio_mode->audio_path_setting)

    /* Audio microphone mode */
    CHECK_STATUS(audio_mode_microphone_mode_read, p_audio_mode->audio_microphone_setting.mode)

    /* Audio microphone setting */
    switch (p_audio_mode->audio_microphone_setting.mode)
    {
      case AUDIO_MICROPHONE_HANDHELD:
      {
        CHECK_STATUS(audio_mode_microphone_gain_read, p_audio_mode->audio_microphone_setting.setting.handheld.gain)
        CHECK_STATUS(audio_mode_microphone_fir_read, p_audio_mode->audio_microphone_setting.setting.handheld.fir)
        CHECK_STATUS(audio_mode_microphone_output_bias_read, p_audio_mode->audio_microphone_setting.setting.handheld.output_bias)
        break;
      }
      case AUDIO_MICROPHONE_HANDFREE:
      {
        CHECK_STATUS(audio_mode_microphone_extra_gain_read, p_audio_mode->audio_microphone_setting.setting.handfree.extra_gain)
        CHECK_STATUS(audio_mode_microphone_fir_read, p_audio_mode->audio_microphone_setting.setting.handfree.fir)
        CHECK_STATUS(audio_mode_microphone_gain_read, p_audio_mode->audio_microphone_setting.setting.handfree.gain)
        CHECK_STATUS(audio_mode_microphone_output_bias_read, p_audio_mode->audio_microphone_setting.setting.handfree.output_bias)
        break;
      }
      case AUDIO_MICROPHONE_HEADSET:
      {
        CHECK_STATUS(audio_mode_microphone_gain_read, p_audio_mode->audio_microphone_setting.setting.headset.gain)
        CHECK_STATUS(audio_mode_microphone_fir_read, p_audio_mode->audio_microphone_setting.setting.headset.fir)
        CHECK_STATUS(audio_mode_microphone_output_bias_read, p_audio_mode->audio_microphone_setting.setting.headset.output_bias)
      }
    }

    /* Audio speaker mode */
    CHECK_STATUS(audio_mode_speaker_mode_read, p_audio_mode->audio_speaker_setting.mode)

    /* Audio speaker setting */
    switch(p_audio_mode->audio_speaker_setting.mode)
    {
      case AUDIO_SPEAKER_HANDHELD:
      {
        CHECK_STATUS(audio_mode_speaker_filter_read, p_audio_mode->audio_speaker_setting.setting.handheld.audio_filter)
        CHECK_STATUS(audio_mode_speaker_fir_read, p_audio_mode->audio_speaker_setting.setting.handheld.fir)
        CHECK_STATUS(audio_mode_speaker_gain_read, p_audio_mode->audio_speaker_setting.setting.handheld.gain)
        break;
      }
      case AUDIO_SPEAKER_HANDFREE:
      {
        CHECK_STATUS(audio_mode_speaker_filter_read, p_audio_mode->audio_speaker_setting.setting.handfree.audio_filter)
        CHECK_STATUS(audio_mode_speaker_fir_read, p_audio_mode->audio_speaker_setting.setting.handfree.fir)
        CHECK_STATUS(audio_mode_speaker_gain_read, p_audio_mode->audio_speaker_setting.setting.handfree.gain)
        break;
      }
      case AUDIO_SPEAKER_HEADSET:
      {
        CHECK_STATUS(audio_mode_speaker_filter_read, p_audio_mode->audio_speaker_setting.setting.headset.audio_filter)
        CHECK_STATUS(audio_mode_speaker_fir_read, p_audio_mode->audio_speaker_setting.setting.headset.fir)
        CHECK_STATUS(audio_mode_speaker_gain_read, p_audio_mode->audio_speaker_setting.setting.headset.gain)
        break;
      }
      case AUDIO_SPEAKER_BUZZER:
      {
        CHECK_STATUS(audio_mode_speaker_buzzer_read, p_audio_mode->audio_speaker_setting.setting.buzzer.activate)
        break;
      }
      case AUDIO_SPEAKER_HANDHELD_HANDFREE:
      {
        CHECK_STATUS(audio_mode_speaker_filter_read, p_audio_mode->audio_speaker_setting.setting.handheld_handfree.audio_filter)
        CHECK_STATUS(audio_mode_speaker_fir_read, p_audio_mode->audio_speaker_setting.setting.handheld_handfree.fir)
        CHECK_STATUS(audio_mode_speaker_gain_read, p_audio_mode->audio_speaker_setting.setting.handheld_handfree.gain)
        break;
      }
    }

    /* Audio spkear microphone loop setting */
    CHECK_STATUS(audio_mode_aec_read, p_audio_mode->audio_microphone_speaker_loop_setting.aec)
    CHECK_STATUS(audio_mode_sidetone_gain_read, p_audio_mode->audio_microphone_speaker_loop_setting.sidetone_gain)

    return(AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_volume_get                                         */
  /*                                                                              */
  /*    Purpose:  This function fill the audio volume structure.                  */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        Audio volume structure.                                               */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_volume_get(T_AUDIO_SPEAKER_LEVEL *p_audio_volume)
  {
    CHECK_STATUS(audio_mode_speaker_volume_read, *p_audio_volume)

    return(AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_save_manager                                  */
  /*                                                                              */
  /*    Purpose:  This function manage the audio mode save services.              */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        Audio message.                                                        */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  void audio_mode_save_manager (T_RV_HDR *p_message)
  {
    T_AUDIO_MODE *p_audio_mode;
    T_AUDIO_SPEAKER_LEVEL audio_volume;
    T_RVF_MB_STATUS       mb_status;

    #ifdef _WINDOWS
      INT8  *p_read, *p_write;
      UINT8 i;
    #endif
    /* allocate the buffer for the current Audio mode */
    mb_status = rvf_get_buf (p_audio_gbl_var->mb_internal,
                             sizeof (T_AUDIO_MODE),
                             (T_RVF_BUFFER **) (&p_audio_mode));
    /* If insufficient resources, then report a memory error and abort.               */
    if (mb_status == RVF_RED)
    {
      AUDIO_SEND_TRACE("AUDIO MODE SAVE: not enough memory to allocate the audio mode buffer", RV_TRACE_LEVEL_ERROR);
      audio_mode_save_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_SAVE_REQ *)p_message)->return_path);
      return;
    }

    /* Fill the audio mode structure */
    if ( (audio_mode_get(p_audio_mode)) == AUDIO_ERROR)
    {
      AUDIO_SEND_TRACE("AUDIO MODE SAVE: error in the the audio mode get function", RV_TRACE_LEVEL_ERROR);
      /* free the audio mode buffer */
      rvf_free_buf((T_RVF_BUFFER *)p_audio_mode);

      #ifndef _WINDOWS
        /* Close the files */
        ffs_close(((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_ffs_fd);
        ffs_close(((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_volume_ffs_fd);
      #endif

      /* send the status message */
      audio_mode_save_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_SAVE_REQ *)p_message)->return_path);
      return;
    }

    #ifndef _WINDOWS
      /* Save the audio mode structure to the FFS */
      if ( (ffs_write (((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_ffs_fd,
                      p_audio_mode,
                      sizeof(T_AUDIO_MODE))) < EFFS_OK )
      {
        AUDIO_SEND_TRACE("AUDIO MODE SAVE: impossible to save the current audio mode", RV_TRACE_LEVEL_ERROR);

        /* free the audio mode buffer */
        rvf_free_buf((T_RVF_BUFFER *)p_audio_mode);

        /* Close the files */
        ffs_close(((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_ffs_fd);
        ffs_close(((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_volume_ffs_fd);

        /* send the status message */
        audio_mode_save_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_SAVE_REQ *)p_message)->return_path);
        return;
      }

      /* Close the file */
      ffs_close(((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_ffs_fd);
    #else
      #if ((AUDIO_REGR == SW_COMPILED) || (AUDIO_MISC == SW_COMPILED))
        p_read = (INT8 *)(p_audio_mode);
        p_write = (INT8 *)(&(p_audio_test->audio_mode_2));

        for (i=0; i<sizeof(T_AUDIO_MODE); i++)
        {
          *p_write++ = *p_read++;
        }
      #endif
    #endif

    /* free the audio mode buffer */
    rvf_free_buf((T_RVF_BUFFER *)p_audio_mode);

    /* Fill the audio volume structure */
    if ( (audio_volume_get(&audio_volume)) == AUDIO_ERROR)
    {
      AUDIO_SEND_TRACE("AUDIO MODE SAVE: error in the the audio speaker volume get function", RV_TRACE_LEVEL_ERROR);
      #ifndef _WINDOWS
        /* Close the file */
        ffs_close(((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_volume_ffs_fd);
      #endif

      /* send the status message */
      audio_mode_save_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_SAVE_REQ *)p_message)->return_path);
      return;
    }

    #ifndef _WINDOWS
      /* Save the audio speaker volume structure to the FFS */
      if ( (ffs_write (((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_volume_ffs_fd,
                      &audio_volume,
                      sizeof(T_AUDIO_SPEAKER_LEVEL))) < EFFS_OK )
      {
        AUDIO_SEND_TRACE("AUDIO MODE SAVE: impossible to save the current speaker volume", RV_TRACE_LEVEL_ERROR);

        /* Close the files */
        ffs_close(((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_volume_ffs_fd);

        /* send the status message */
        audio_mode_save_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_SAVE_REQ *)p_message)->return_path);
        return;
      }

      /* Close the file */
      ffs_close(((T_AUDIO_MODE_SAVE_REQ *)p_message)->audio_volume_ffs_fd);
    #else
      #if ((AUDIO_REGR == SW_COMPILED) || (AUDIO_MISC == SW_COMPILED))
        p_audio_test->speaker_volume_2.audio_speaker_level = audio_volume.audio_speaker_level;
      #endif
    #endif

    /* send the status message */
    audio_mode_save_send_status (AUDIO_OK, ((T_AUDIO_MODE_SAVE_REQ *)p_message)->return_path);
  }
#endif /* RVM_AUDIO_MAIN_SWE */
