/****************************************************************************/
/*                                                                          */
/*  File Name:  audio_mode_load.c                                           */
/*                                                                          */
/*  Purpose:  This file contains all the functions used for audio mode      */
/*            load services.                                                */
/*                                                                          */
/*  Version   0.1                                                           */
/*                                                                          */
/*  Date        Modification                                                */
/*  ------------------------------------------------------------------------*/
/*  14 Jan 2002  Create                                                     */
/*                                                                          */
/*  Author      Francois Mazard                                             */
/*                                                                          */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved*/
/****************************************************************************/

#include "rv/rv_defined_swe.h"

#ifdef RVM_AUDIO_MAIN_SWE
  #ifndef _WINDOWS
    #include "config/swconfig.cfg"
    #include "config/sys.cfg"
    #include "config/chipset.cfg"
  #endif

  #include "l1_confg.h"
  #include "rv/rv_general.h"
  #include "rvm/rvm_gen.h"
  #include "audio/audio_ffs_i.h"
  #include "audio/audio_api.h"
  #include "audio/audio_structs_i.h"
  #include "audio/audio_error_hdlr_i.h"
  #include "audio/audio_var_i.h"
  #include "audio/audio_messages_i.h"
  #include "audio/audio_macro_i.h"
  #include "rvf/rvf_target.h"
  #include "audio/audio_const_i.h"

  /* include the usefull L1 header */
  #ifdef _WINDOWS
    #define BOOL_FLAG
    //#define CHAR_FLAG
  #endif
  #include "l1_types.h"
  #include "l1audio_const.h"
  #include "l1audio_cust.h"
  #include "l1audio_defty.h"
  #include "l1audio_msgty.h"
  #include "l1audio_signa.h"
  #if TESTMODE
   #include "l1tm_defty.h"
  #endif
  #if (L1_GTT == 1)
    #include "l1gtt_const.h"
    #include "l1gtt_defty.h"
  #endif
  #include "l1_const.h"
  #include "l1_defty.h"
  #include "l1_msgty.h"
  #include "l1_signa.h"
  #ifdef _WINDOWS
    #define L1_ASYNC_C
  #endif
  #include "l1_varex.h"

  #include "ffs/ffs_api.h"

  #ifdef _WINDOWS
    #include "audio/tests/audio_test.h"
  #endif

#include <string.h>

  /* external functions */
  /* write */
  extern T_AUDIO_RET audio_mode_voice_path_write               (T_AUDIO_VOICE_PATH_SETTING *data,
                                                                UINT8 *message_to_confirm);
  extern T_AUDIO_RET audio_mode_microphone_mode_write          (INT8  *data);
  extern T_AUDIO_RET audio_mode_microphone_gain_write          (INT8  *data);
  extern T_AUDIO_RET audio_mode_microphone_extra_gain_write    (INT8  *data);
  extern T_AUDIO_RET audio_mode_microphone_output_bias_write   (INT8  *data);
  extern T_AUDIO_RET audio_mode_speaker_mode_write             (INT8  *data);
  extern T_AUDIO_RET audio_mode_speaker_gain_write             (INT8  *data);
  extern T_AUDIO_RET audio_mode_speaker_filter_write           (INT8  *data);
  extern T_AUDIO_RET audio_mode_speaker_buzzer_write           (INT8  *data);
  extern T_AUDIO_RET audio_mode_sidetone_gain_write            (INT8  *data);
  extern T_AUDIO_RET audio_mode_aec_write                      (T_AUDIO_AEC_CFG *data);
  extern T_AUDIO_RET audio_mode_speaker_volume_write           (T_AUDIO_SPEAKER_LEVEL *data);
  extern T_AUDIO_RET audio_mode_speaker_microphone_fir_write   (T_AUDIO_FIR_COEF *data_speaker,
                                                                T_AUDIO_FIR_COEF *data_microphone);


  /* Define a macro to simplify the code */
  #define CHECK_STATUS(function, param)     if ( (function(&(param))) == AUDIO_ERROR ) \
                                            { \
                                              return(AUDIO_ERROR); \
                                            } \

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_load_send_status                              */
  /*                                                                              */
  /*    Purpose:  This function sends the audio mdoe load status to the entity.   */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        status,                                                               */
  /*        return path                                                           */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  void audio_mode_load_send_status (T_AUDIO_RET status, T_RV_RETURN return_path)
  {
    void *p_send_message = NULL;
    T_RVF_MB_STATUS mb_status = RVF_RED;

    while (mb_status == RVF_RED)
    {
      /* allocate the message buffer */
      mb_status = rvf_get_buf (p_audio_gbl_var->mb_external,
                               sizeof (T_AUDIO_LOAD_DONE),
                               (T_RVF_BUFFER **) (&p_send_message));

      /* If insufficient resources, then report a memory error and abort.               */
      /* and wait until more ressource is given */
      if (mb_status == RVF_RED)
      {
        audio_mode_error_trace(AUDIO_ENTITY_NO_MEMORY);
        rvf_delay(RVF_MS_TO_TICKS(1000));
      }
    }
    /*fill the header of the message */
    ((T_AUDIO_LOAD_DONE *)(p_send_message))->os_hdr.msg_id =
      AUDIO_MODE_LOAD_DONE;

    /* fill the status parameters */
    ((T_AUDIO_LOAD_DONE *)(p_send_message))->status = status;

    if (return_path.callback_func == NULL)
    {
      /* send the message to the entity */
      rvf_send_msg (return_path.addr_id,
                    p_send_message);
    }
    else
    {
      /* call the callback function */
      (*return_path.callback_func)((void *)(p_send_message));
	  rvf_free_buf((T_RVF_BUFFER *)p_send_message);
    }
  }
  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_set                                           */
  /*                                                                              */
  /*    Purpose:  This function set the audio mode structure to the mobile.       */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        Audio mode structure.                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_set(T_AUDIO_MODE *p_audio_mode, UINT8 *message_to_confirm)
  {
    T_AUDIO_FIR_COEF *data_speaker = NULL, *data_microphone = NULL;

    *message_to_confirm = 0;


    /* Audio voice path */
    if (audio_mode_voice_path_write (&p_audio_mode->audio_path_setting, message_to_confirm) == AUDIO_ERROR)
    {
      return(AUDIO_ERROR);
    }

    /* CR33671 : wrong sequence of message for disabling AQI in TTY calls */
    /* Audio speaker microphone loop setting */
    CHECK_STATUS(audio_mode_aec_write, p_audio_mode->audio_microphone_speaker_loop_setting.aec)
    /* confirm the AEC message */
    *message_to_confirm += 1;


    /* Audio microphone mode */
    CHECK_STATUS(audio_mode_microphone_mode_write, p_audio_mode->audio_microphone_setting.mode)

    /* Audio micropohne setting */
    switch (p_audio_mode->audio_microphone_setting.mode)
    {
      case AUDIO_MICROPHONE_HANDHELD:
      {
        CHECK_STATUS(audio_mode_microphone_gain_write, p_audio_mode->audio_microphone_setting.setting.handheld.gain)
        data_microphone = &(p_audio_mode->audio_microphone_setting.setting.handheld.fir);
        CHECK_STATUS(audio_mode_microphone_output_bias_write, p_audio_mode->audio_microphone_setting.setting.handheld.output_bias)
        break;
      }
      case AUDIO_MICROPHONE_HANDFREE:
      {
        CHECK_STATUS(audio_mode_microphone_extra_gain_write, p_audio_mode->audio_microphone_setting.setting.handfree.extra_gain)
        data_microphone = &(p_audio_mode->audio_microphone_setting.setting.handfree.fir);
        CHECK_STATUS(audio_mode_microphone_gain_write, p_audio_mode->audio_microphone_setting.setting.handfree.gain)
        CHECK_STATUS(audio_mode_microphone_output_bias_write, p_audio_mode->audio_microphone_setting.setting.handfree.output_bias)
        break;
      }
      case AUDIO_MICROPHONE_HEADSET:
      {
        CHECK_STATUS(audio_mode_microphone_gain_write, p_audio_mode->audio_microphone_setting.setting.headset.gain)
        data_microphone = &(p_audio_mode->audio_microphone_setting.setting.headset.fir);
        CHECK_STATUS(audio_mode_microphone_output_bias_write, p_audio_mode->audio_microphone_setting.setting.headset.output_bias)
      }
    }

    /* Audio speaker mode */
    CHECK_STATUS(audio_mode_speaker_mode_write, p_audio_mode->audio_speaker_setting.mode)

    /* Audio speaker setting */
    switch(p_audio_mode->audio_speaker_setting.mode)
    {
      case AUDIO_SPEAKER_HANDHELD:
      {
        CHECK_STATUS(audio_mode_speaker_filter_write, p_audio_mode->audio_speaker_setting.setting.handheld.audio_filter)
        data_speaker = &(p_audio_mode->audio_speaker_setting.setting.handheld.fir);
        CHECK_STATUS(audio_mode_speaker_gain_write, p_audio_mode->audio_speaker_setting.setting.handheld.gain)
        break;
      }
      case AUDIO_SPEAKER_HANDFREE:
      {
        CHECK_STATUS(audio_mode_speaker_filter_write, p_audio_mode->audio_speaker_setting.setting.handfree.audio_filter)
        data_speaker = &(p_audio_mode->audio_speaker_setting.setting.handfree.fir);
        CHECK_STATUS(audio_mode_speaker_gain_write, p_audio_mode->audio_speaker_setting.setting.handfree.gain)
        break;
      }
      case AUDIO_SPEAKER_HEADSET:
      {
        CHECK_STATUS(audio_mode_speaker_filter_write, p_audio_mode->audio_speaker_setting.setting.headset.audio_filter)
        data_speaker = &(p_audio_mode->audio_speaker_setting.setting.headset.fir);
        CHECK_STATUS(audio_mode_speaker_gain_write, p_audio_mode->audio_speaker_setting.setting.headset.gain)
        break;
      }
      case AUDIO_SPEAKER_BUZZER:
      {
        CHECK_STATUS(audio_mode_speaker_buzzer_write, p_audio_mode->audio_speaker_setting.setting.buzzer.activate)
        break;
      }
      case AUDIO_SPEAKER_HANDHELD_HANDFREE:
      {
        CHECK_STATUS(audio_mode_speaker_filter_write, p_audio_mode->audio_speaker_setting.setting.handheld_handfree.audio_filter)
        data_speaker = &(p_audio_mode->audio_speaker_setting.setting.handheld_handfree.fir);
        CHECK_STATUS(audio_mode_speaker_gain_write, p_audio_mode->audio_speaker_setting.setting.handheld_handfree.gain)
        break;
      }
    }

    /* Set the speaker and microphone FIR */
    if ( (audio_mode_speaker_microphone_fir_write(data_speaker, data_microphone))
          == AUDIO_ERROR )
    {
      return(AUDIO_ERROR);
    }
    /* confirm the FIR message */
    *message_to_confirm += 1;

    CHECK_STATUS(audio_mode_sidetone_gain_write, p_audio_mode->audio_microphone_speaker_loop_setting.sidetone_gain)


    return(AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_volume_set                                         */
  /*                                                                              */
  /*    Purpose:  This function sets the audio volume.                            */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        Audio volume structure.                                               */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_volume_set(T_AUDIO_SPEAKER_LEVEL *p_audio_volume)
  {
    CHECK_STATUS(audio_mode_speaker_volume_write, *p_audio_volume)

    return(AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_load_manager                                  */
  /*                                                                              */
  /*    Purpose:  This function manages the audio mode load services.             */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        Audio message.                                                        */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  void audio_mode_load_manager (T_RV_HDR *p_message)
  {
    T_AUDIO_SPEAKER_LEVEL audio_volume;
    T_RVF_MB_STATUS       mb_status;
    UINT8                 message_to_confirm;
    #ifdef _WINDOWS
      INT8  *p_read, *p_write;
      UINT8 i;
    #endif

    switch (p_audio_gbl_var->audio_mode_var.audio_mode_load_var.state)
    {
      case AUDIO_MODE_LOAD_IDLE:
      {
        /* allocate the buffer for the current Audio mode */
        mb_status = rvf_get_buf (p_audio_gbl_var->mb_internal,
                                 sizeof (T_AUDIO_MODE),
                                 (T_RVF_BUFFER **) (&(p_audio_gbl_var->audio_mode_var.audio_mode_load_var.p_audio_mode)));
        /* If insufficient resources, then report a memory error and abort.               */
        if (mb_status == RVF_RED)
        {
          AUDIO_SEND_TRACE("AUDIO MODE LOAD: not enough memory to allocate the audio mode buffer", RV_TRACE_LEVEL_ERROR);
          audio_mode_load_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_LOAD_REQ *)p_message)->return_path);
         return;
        }

        #ifdef _WINDOWS
          #if ((AUDIO_REGR == SW_COMPILED) || (AUDIO_MISC == SW_COMPILED))
            p_read = (INT8 *)(&(p_audio_test->audio_mode_1));
            p_write = (INT8 *)(p_audio_gbl_var->audio_mode_var.audio_mode_load_var.p_audio_mode);

            for (i=0; i<sizeof(T_AUDIO_MODE); i++)
            {
              *p_write++ = *p_read++;
            }
          #endif
        #else
          /* Load the audio mode structure from the FFS */

          if ( ffs_read (((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_ffs_fd,
                         (void *)(p_audio_gbl_var->audio_mode_var.audio_mode_load_var.p_audio_mode),
                         sizeof(T_AUDIO_MODE)) < EFFS_OK )
          {
            AUDIO_SEND_TRACE("AUDIO MODE LOAD: impossible to load the current audio mode", RV_TRACE_LEVEL_ERROR);

            /* Close the files */
            ffs_close(((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_ffs_fd);
            ffs_close(((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_volume_ffs_fd);

            /* free the audio mode buffer */
            rvf_free_buf((T_RVF_BUFFER *)(p_audio_gbl_var->audio_mode_var.audio_mode_load_var.p_audio_mode));

            /* send the status message */
            audio_mode_load_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_LOAD_REQ *)p_message)->return_path);
            return;
          }
        #endif

        /* Set the audio mode structure */
        if ( (audio_mode_set((p_audio_gbl_var->audio_mode_var.audio_mode_load_var.p_audio_mode),
              &message_to_confirm)) == AUDIO_ERROR)
        {
          AUDIO_SEND_TRACE("AUDIO MODE LOAD: error in the the audio mode set function", RV_TRACE_LEVEL_ERROR);
          /* free the audio mode buffer */
          rvf_free_buf((T_RVF_BUFFER *)(p_audio_gbl_var->audio_mode_var.audio_mode_load_var.p_audio_mode));

          #ifndef _WINDOWS
            /* Close the files */
            ffs_close(((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_ffs_fd);
            ffs_close(((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_volume_ffs_fd);
          #endif

          /* send the status message */
          audio_mode_load_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_LOAD_REQ *)p_message)->return_path);
          return;
        }

        /* Calculate the number of confirmation message to receive */
        p_audio_gbl_var->audio_mode_var.audio_mode_load_var.number_of_message = message_to_confirm;

        #ifndef _WINDOWS
          /* Close the file */
          ffs_close(((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_ffs_fd);
        #endif

        #ifndef _WINDOWS
          /* Load the audio speaker volume structure from the FFS */
          if ( (ffs_read (((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_volume_ffs_fd,
                          (void *)(&audio_volume),
                          sizeof(T_AUDIO_SPEAKER_LEVEL))) < EFFS_OK )
          {
            AUDIO_SEND_TRACE("AUDIO MODE LOAD: impossible to load the current speaker volume", RV_TRACE_LEVEL_ERROR);

            /* Close the file */
            ffs_close(((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_volume_ffs_fd);

            /* send the status message */
            audio_mode_load_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_LOAD_REQ *)p_message)->return_path);
            return;
          }
        #else
          #if ((AUDIO_REGR == SW_COMPILED) || (AUDIO_MISC == SW_COMPILED))
            audio_volume.audio_speaker_level = p_audio_test->speaker_volume_1.audio_speaker_level;
          #endif
        #endif

        /* Fill the audio volume structure */
        if ( (audio_volume_set(&audio_volume)) == AUDIO_ERROR)
        {
          AUDIO_SEND_TRACE("AUDIO MODE LOAD: error in the the audio speaker volume set function", RV_TRACE_LEVEL_ERROR);
          #ifndef _WINDOWS
            /* Close the file */
            ffs_close(((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_volume_ffs_fd);
          #endif

          /* send the status message */
          audio_mode_load_send_status (AUDIO_ERROR, ((T_AUDIO_MODE_LOAD_REQ *)p_message)->return_path);
          return;
        }
        #ifndef _WINDOWS
          /* Close the file */
          ffs_close(((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_volume_ffs_fd);
        #endif

        /* Save the return path */
        p_audio_gbl_var->audio_mode_var.audio_mode_load_var.return_path.callback_func =
          ((T_AUDIO_MODE_LOAD_REQ *)p_message)->return_path.callback_func;
        p_audio_gbl_var->audio_mode_var.audio_mode_load_var.return_path.addr_id =
          ((T_AUDIO_MODE_LOAD_REQ *)p_message)->return_path.addr_id;

        /* Save the name of the volume path name */
        strcpy(p_audio_gbl_var->audio_mode_var.audio_volume_var.audio_volume_path_name,
              ((T_AUDIO_MODE_LOAD_REQ *)p_message)->audio_volume_path_name);

        /* change the state */
        p_audio_gbl_var->audio_mode_var.audio_mode_load_var.state =
          AUDIO_MODE_LOAD_WAIT_MESSAGE_CONF;
        break;
      }
      case AUDIO_MODE_LOAD_WAIT_MESSAGE_CONF:
      {
        p_audio_gbl_var->audio_mode_var.audio_mode_load_var.number_of_message--;

        if (p_audio_gbl_var->audio_mode_var.audio_mode_load_var.number_of_message == 0)
        {
          /* free the audio mode buffer */
          rvf_free_buf((T_RVF_BUFFER *)(p_audio_gbl_var->audio_mode_var.audio_mode_load_var.p_audio_mode));

          /* send the status message */
          audio_mode_load_send_status (AUDIO_OK,
            p_audio_gbl_var->audio_mode_var.audio_mode_load_var.return_path);

          /* change the state */
          p_audio_gbl_var->audio_mode_var.audio_mode_load_var.state =
            AUDIO_MODE_LOAD_IDLE;
        }
        break;
      }
    }
  }
#endif /* RVM_AUDIO_MAIN_SWE */
