/****************************************************************************/
/*                                                                          */
/*  File Name:  audio_mode_drivers.c                                        */
/*                                                                          */
/*  Purpose:  This file contains all the drivers used for audio mode        */
/*            services.                                                     */
/*                                                                          */
/*  Version   0.1                                                           */
/*                                                                          */
/*  Date        Modification                                                */
/*  ------------------------------------------------------------------------*/
/*  14 Jan 2002  Create                                                     */
/*                                                                          */
/*  Author      Francois Mazard                                             */
/*                                                                          */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved*/
/****************************************************************************/

#include "rv/rv_defined_swe.h"

#ifdef RVM_AUDIO_MAIN_SWE
  #ifndef _WINDOWS
    #include "config/swconfig.cfg"
    #include "config/sys.cfg"
    #include "config/l1sw.cfg"
    #include "config/chipset.cfg"
  #endif

  #include "l1_confg.h"
  #include "rv/rv_general.h"
  #include "rvm/rvm_gen.h"
  #include "audio/audio_features_i.h"
  #include "audio/audio_ffs_i.h"
  #include "audio/audio_api.h"
  #include "audio/audio_structs_i.h"
  #include "audio/audio_error_hdlr_i.h"
  #include "audio/audio_var_i.h"
  #include "audio/audio_messages_i.h"
  #include "audio/audio_macro_i.h"
  #include "rvf/rvf_target.h"
  #include "audio/audio_const_i.h"
  #include "audio/audio_mode_i.h"

  /* include the usefull L1 header */
  #ifdef _WINDOWS
    #define BOOL_FLAG
    //#define CHAR_FLAG
  #endif
  #include "l1_types.h"
  #include "l1audio_const.h"
  #include "l1audio_cust.h"
  #include "l1audio_defty.h"
  #include "l1audio_msgty.h"
  #include "l1audio_signa.h"
  #if TESTMODE
   #include "l1tm_defty.h"
  #endif
  #if (L1_GTT == 1)
    #include "l1gtt_const.h"
    #include "l1gtt_defty.h"
  #endif
  #include "l1_const.h"
  #include "l1_defty.h"
  #include "l1_msgty.h"
  #include "l1_signa.h"
  #ifdef _WINDOWS
    #define L1_ASYNC_C
  #endif
  #include "l1_varex.h"

  /* write drivers */
  T_AUDIO_RET audio_mode_voice_path_write               (T_AUDIO_VOICE_PATH_SETTING *data,
                                                         UINT8 *message_to_confirm);
  T_AUDIO_RET audio_mode_microphone_mode_write          (INT8  *data);
  T_AUDIO_RET audio_mode_microphone_gain_write          (INT8  *data);
  T_AUDIO_RET audio_mode_microphone_extra_gain_write    (INT8  *data);
  T_AUDIO_RET audio_mode_microphone_output_bias_write   (INT8  *data);
  T_AUDIO_RET audio_mode_speaker_mode_write             (INT8  *data);
  T_AUDIO_RET audio_mode_speaker_gain_write             (INT8  *data);
  T_AUDIO_RET audio_mode_speaker_filter_write           (INT8  *data);
  T_AUDIO_RET audio_mode_speaker_buzzer_write           (INT8  *data);
  T_AUDIO_RET audio_mode_sidetone_gain_write            (INT8  *data);
  T_AUDIO_RET audio_mode_aec_write                      (T_AUDIO_AEC_CFG *data);
  T_AUDIO_RET audio_mode_speaker_volume_write           (T_AUDIO_SPEAKER_LEVEL *data);
  T_AUDIO_RET audio_mode_speaker_microphone_fir_write   (T_AUDIO_FIR_COEF *data_speaker,
                                                         T_AUDIO_FIR_COEF *data_microphone);

  /* read drivers */
  T_AUDIO_RET audio_mode_voice_path_read               (T_AUDIO_VOICE_PATH_SETTING *data);
  T_AUDIO_RET audio_mode_microphone_mode_read          (INT8  *data);
  T_AUDIO_RET audio_mode_microphone_gain_read          (INT8  *data);
  T_AUDIO_RET audio_mode_microphone_extra_gain_read    (INT8  *data);
  T_AUDIO_RET audio_mode_microphone_output_bias_read   (INT8  *data);
  T_AUDIO_RET audio_mode_microphone_fir_read           (T_AUDIO_FIR_COEF *data);
  T_AUDIO_RET audio_mode_speaker_mode_read             (INT8  *data);
  T_AUDIO_RET audio_mode_speaker_gain_read             (INT8  *data);
  T_AUDIO_RET audio_mode_speaker_filter_read           (INT8  *data);
  T_AUDIO_RET audio_mode_speaker_fir_read              (T_AUDIO_FIR_COEF *data);
  T_AUDIO_RET audio_mode_speaker_buzzer_read           (INT8  *data);
  T_AUDIO_RET audio_mode_sidetone_gain_read            (INT8  *data);
  T_AUDIO_RET audio_mode_aec_read                      (T_AUDIO_AEC_CFG *data);
  T_AUDIO_RET audio_mode_speaker_volume_read           (T_AUDIO_SPEAKER_LEVEL *data);

  /* external functions */
  extern void ABB_CAL_UlVolume   (UWORD8 pga_index);
  extern void ABB_CAL_DlVolume   (UWORD8 volume_index, UWORD8 pga_index);
  extern void ABB_UlVolume       (UWORD8 volume_index);
  extern void ABB_DlVolume       (UWORD8 volume_index);
  extern void ABB_DlMute         (UWORD8 mute);
  extern void ABB_SideTone       (UWORD8 volume_index);
  extern void ABB_Audio_Config   (UWORD16 data);
  extern void ABB_Audio_Config_2 (UWORD16 data);
  extern void ABB_UlMute         (BOOL mute);

  /* external variable */
  /* Uplink PGA gain is coded on 5 bits, corresponding to -12 dB to +12 dB in 1dB steps */
  extern const UWORD8 ABB_uplink_PGA_gain[];
  /* Downlink volume: mute, -24dB to 0dB in 6dB steps */
  extern const UWORD8 ABB_volume_control_gain[];
  /* Downlink PGA gain is coded on 4 bits, corresponding to -6dB to 6dB in 1dB steps */
  extern const UWORD8 ABB_downlink_PGA_gain[];
  /* Side tone level: mute, -23dB to +4dB in 3dB steps */
  extern const UWORD8 ABB_sidetone_gain[];

  /********************************************************************************/
  /**********************     WRITE DRIVERS          ******************************/
  /********************************************************************************/

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_voice_path_write                              */
  /*                                                                              */
  /*    Purpose:  This function set a new voice path.                             */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new voice path.                                                       */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        number of message to confirm                                          */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_voice_path_write (T_AUDIO_VOICE_PATH_SETTING *data, UINT8 *message_to_confirm)
  {
    void *p_send_message = NULL;
    T_AUDIO_VOICE_PATH_SETTING previous_audio_mode_path;

    *message_to_confirm = 0;

    /* Read if the current mode is a DAI mode */
    if (audio_mode_voice_path_read(&previous_audio_mode_path) == AUDIO_ERROR)
    {
      AUDIO_SEND_TRACE("AUDIO MODE WRITE: voice path: error to read the current audio mode path",
        RV_TRACE_LEVEL_ERROR);
      return(AUDIO_ERROR);
    }

    switch (*data)
    {
      case AUDIO_GSM_VOICE_PATH:
      {
        if ( (previous_audio_mode_path == AUDIO_DAI_ENCODER) ||
             (previous_audio_mode_path == AUDIO_DAI_DECODER) ||
             (previous_audio_mode_path == AUDIO_DAI_ACOUSTIC) )
        {
          /* The current mode is a DAI test so it need to be stopped */
          /* Send the stop DAI message to the L1*/
          /* allocate the buffer for the message to the L1 */
          p_send_message = audio_allocate_l1_message(0);
          if (p_send_message != NULL)
          {
            /* send the start command to the audio L1 */
            audio_send_l1_message(OML1_STOP_DAI_TEST_REQ, p_send_message);
          }
          else
          {
            AUDIO_SEND_TRACE("AUDIO MODE WRITE: voice path: the stop DAI message isn't send",
              RV_TRACE_LEVEL_ERROR);
            return(AUDIO_ERROR);
          }
          /* confirm the DAI stop message */
          *message_to_confirm += 1;
        }

        #if (AUDIO_MODE)
          /* Send the Audio path configuration message */
          /* allocate the buffer for the message to the L1 */
          p_send_message = audio_allocate_l1_message(sizeof(T_MMI_AUDIO_MODE));
          if (p_send_message != NULL)
          {
            /* Fill the parameter */
            ((T_MMI_AUDIO_MODE *)p_send_message)->audio_mode = *data;

            /* send the start command to the audio L1 */
            audio_send_l1_message(MMI_AUDIO_MODE_REQ, p_send_message);
          }
          else
          {
            AUDIO_SEND_TRACE("AUDIO MODE WRITE: voice path: the audio mode message isn't send",
              RV_TRACE_LEVEL_ERROR);
            return(AUDIO_ERROR);
          }
          /* confirm the AUDIO MODE message */
          *message_to_confirm += 1;
        #endif
        break;
      }
      #if (AUDIO_MODE)
        case AUDIO_BLUETOOTH_CORDLESS_VOICE_PATH:
        case AUDIO_BLUETOOTH_HEADSET:
        {
          if ( (previous_audio_mode_path == AUDIO_DAI_ENCODER) ||
               (previous_audio_mode_path == AUDIO_DAI_DECODER) ||
               (previous_audio_mode_path == AUDIO_DAI_ACOUSTIC) )
          {
            /* The current mode is a DAI test so it need to be stopped */
            /* Send the stop DAI message to the L1*/
            /* allocate the buffer for the message to the L1 */
            p_send_message = audio_allocate_l1_message(0);
            if (p_send_message != NULL)
            {
              /* send the start command to the audio L1 */
              audio_send_l1_message(OML1_STOP_DAI_TEST_REQ, p_send_message);
            }
            else
            {
              AUDIO_SEND_TRACE("AUDIO MODE WRITE: voice path: the stop DAI message isn't send",
                RV_TRACE_LEVEL_ERROR);
              return(AUDIO_ERROR);
            }
            /* confirm the DAI stop message */
            *message_to_confirm += 1;
          }

          /* Send the Audio path configuration message */
          /* allocate the buffer for the message to the L1 */
          p_send_message = audio_allocate_l1_message(sizeof(T_MMI_AUDIO_MODE));
          if (p_send_message != NULL)
          {
            /* Fill the parameter */
            ((T_MMI_AUDIO_MODE *)p_send_message)->audio_mode = *data;

            /* send the start command to the audio L1 */
            audio_send_l1_message(MMI_AUDIO_MODE_REQ, p_send_message);
          }
          else
          {
            AUDIO_SEND_TRACE("AUDIO MODE WRITE: voice path: the audio mode message isn't send",
              RV_TRACE_LEVEL_ERROR);
            return(AUDIO_ERROR);
          }
          /* confirm the AUDIO MODE message */
          *message_to_confirm += 1;
          break;
        }
      #endif
      case AUDIO_DAI_ENCODER :
      case AUDIO_DAI_DECODER :
      case AUDIO_DAI_ACOUSTIC :
      {
        /* Send the DAI start message */
        /* allocate the buffer for the message to the L1 */
        p_send_message = audio_allocate_l1_message(sizeof(T_OML1_START_DAI_TEST_REQ));
        if (p_send_message != NULL)
        {
          /* Fill the parameter */
          ((T_OML1_START_DAI_TEST_REQ *)p_send_message)->tested_device = (UINT8)(*data - 10);

          /* send the start command to the audio L1 */
          audio_send_l1_message(OML1_START_DAI_TEST_REQ, p_send_message);
        }
        else
        {
          AUDIO_SEND_TRACE("AUDIO MODE WRITE: voice path: the start DAI message isn't send",
            RV_TRACE_LEVEL_ERROR);
          return(AUDIO_ERROR);
        }
        /* confirm the DAI start message */
        *message_to_confirm += 1;
        break;
      }
      default :
      {
          AUDIO_SEND_TRACE_PARAM("AUDIO MODE WRITE: voice path: this mode isn't supported",
            *data,
            RV_TRACE_LEVEL_ERROR);
          return(AUDIO_ERROR);
          break;
      }
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_microphone_mode_write                         */
  /*                                                                              */
  /*    Purpose:  This function set a new microphone mode.                        */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new microphone mode.                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_microphone_mode_write (INT8  *data)
  {
    UINT16 vbcr;
    #if (ANLG_FAM == 2)
      UINT16 vbcr2;
    #endif

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
      vbcr2 = (l1s_dsp_com.dsp_ndb_ptr->d_vbctrl2 >> 6);
    #endif

    switch (*data)
    {
      case AUDIO_MICROPHONE_HANDHELD:
      {
        #if (ANLG_FAM == 2)
          /* Reset the Head set configuration */
          vbcr2 &= (~(AUDIO_VBCR2_MICBIASEL | AUDIO_VBCR2_MICNAUX));
          ABB_Audio_Config_2(vbcr2);
        #endif

        vbcr |= AUDIO_VBCR_VULSWITCH;
        ABB_Audio_Config(vbcr);
        break;
      }
      case AUDIO_MICROPHONE_HEADSET:
      {
        #if (ANLG_FAM == 2)
          /* Set the auxilary input */
          vbcr &= ~(AUDIO_VBCR_VULSWITCH);
          ABB_Audio_Config(vbcr);

          vbcr2 |= (AUDIO_VBCR2_MICBIASEL | AUDIO_VBCR2_MICNAUX);
          ABB_Audio_Config_2(vbcr2);
        #else
          AUDIO_SEND_TRACE_PARAM("AUDIO MODE WRITE: microphone mode: this mode isn't supported",
            *data,
            RV_TRACE_LEVEL_ERROR);
          return(AUDIO_ERROR);
        #endif
        break;
      }
      case AUDIO_MICROPHONE_HANDFREE:
      {
        #if (ANLG_FAM == 2)
          /* Reset the Head set configuration */
          vbcr2 &= (~(AUDIO_VBCR2_MICBIASEL | AUDIO_VBCR2_MICNAUX));
          ABB_Audio_Config_2(vbcr2);
        #endif

        vbcr &= ~(AUDIO_VBCR_VULSWITCH);
        ABB_Audio_Config(vbcr);
        break;
      }
      default :
      {
        AUDIO_SEND_TRACE_PARAM("AUDIO MODE WRITE: microphone mode: this mode isn't supported",
          *data,
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
        break;
      }
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_microphone_gain_write                         */
  /*                                                                              */
  /*    Purpose:  This function set a new microphone gain.                        */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new microphone gain.                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_microphone_gain_write (INT8  *data)
  {
    if (*data == AUDIO_MICROPHONE_MUTE)
    {
      ABB_UlMute(TRUE);
    }
    else
    {
      /* Unmute the micropohne */
      ABB_UlMute(FALSE);
      ABB_CAL_UlVolume( (UINT8)(*data + 12) );
    }

    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_microphone_extra_gain_write                   */
  /*                                                                              */
  /*    Purpose:  This function set a new microphone extra gain.                  */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new microphone extra gain.                                            */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_microphone_extra_gain_write (INT8  *data)
  {
    UINT16 vbcr;

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
    #endif

    switch (*data)
    {
      case AUDIO_MICROPHONE_AUX_GAIN_28_2dB:
      {
        vbcr |= (AUDIO_VBCR_VBDFAUXG);
        ABB_Audio_Config(vbcr);
        break;
      }
      case AUDIO_MICROPHONE_AUX_GAIN_4_6dB:
      {
        vbcr &= ~(AUDIO_VBCR_VBDFAUXG);
        ABB_Audio_Config(vbcr);
        break;
      }
      default :
      {
        AUDIO_SEND_TRACE_PARAM("AUDIO MODE WRITE: microphone extra gain: this extra gain isn't supported",
          *data,
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
        break;
      }
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_microphone_output_bias_write                  */
  /*                                                                              */
  /*    Purpose:  This function set a new microphone output bias.                 */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new microphone output bias.                                           */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_microphone_output_bias_write (INT8  *data)
  {
    UINT16 vbcr;

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
    #endif

    switch (*data)
    {
      case AUDIO_MICROPHONE_OUTPUT_BIAS_2_5V:
      {
        vbcr |= (AUDIO_VBCR_MICBIAS);
        ABB_Audio_Config(vbcr);
        break;
      }
      case AUDIO_MICROPHONE_OUTPUT_BIAS_2_0V:
      {
        vbcr &= ~(AUDIO_VBCR_MICBIAS);
        ABB_Audio_Config(vbcr);
        break;
      }
      default :
      {
        AUDIO_SEND_TRACE_PARAM("AUDIO MODE WRITE: microphone output bias: this output bias isn't supported",
          *data,
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
        break;
      }
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_speaker_mode_write                            */
  /*                                                                              */
  /*    Purpose:  This function set a new speaker mode.                           */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new speaker mode.                                                     */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_mode_write (INT8  *data)
  {
    UINT16 vbcr;
    #if (ANLG_FAM == 2)
      UINT16 vbcr2;
    #endif

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
      vbcr2 = (l1s_dsp_com.dsp_ndb_ptr->d_vbctrl2 >> 6);
    #endif

    switch (*data)
    {
      case AUDIO_SPEAKER_HANDHELD:
      {
        #if (ANLG_FAM == 2)
          /* Reset the Head set configuration */
          vbcr2 &= (~(AUDIO_VBCR2_VDLHSO));
          ABB_Audio_Config_2(vbcr2);
        #endif

        #if (ANLG_FAM != 3)
          vbcr |= (AUDIO_VBCR_VDLEAR);
          vbcr &= ~(AUDIO_VBCR_VDLAUX);
          ABB_Audio_Config(vbcr);
        #endif

        break;
      }
      case AUDIO_SPEAKER_HANDFREE:
      {
        #if (ANLG_FAM == 2)
          /* Reset the Head set configuration */
          vbcr2 &= (~(AUDIO_VBCR2_VDLHSO));
          ABB_Audio_Config_2(vbcr2);
        #endif

        #if (ANLG_FAM != 3)
          vbcr |= (AUDIO_VBCR_VDLAUX);
          vbcr &= ~(AUDIO_VBCR_VDLEAR);
          ABB_Audio_Config(vbcr);
        #endif

        break;
      }
      case AUDIO_SPEAKER_HEADSET:
      {
        #if (ANLG_FAM == 2)
          /* Reset aux output amplifier and the ear amplifier */
          vbcr &= ~(AUDIO_VBCR_VDLEAR | AUDIO_VBCR_VDLAUX);
          ABB_Audio_Config(vbcr);

          vbcr2 |= (AUDIO_VBCR2_VDLHSO);
          ABB_Audio_Config_2(vbcr2);
        #else
          AUDIO_SEND_TRACE_PARAM("AUDIO MODE WRITE: speaker mode: this mode isn't supported",
            *data,
            RV_TRACE_LEVEL_ERROR);
          return(AUDIO_ERROR);
        #endif
        break;
      }
      case AUDIO_SPEAKER_BUZZER:
      {
        #if (ANLG_FAM == 2)
          /* Reset the Head set configuration */
          vbcr2 &= (~(AUDIO_VBCR2_VDLHSO));
          ABB_Audio_Config_2(vbcr2);
        #endif

        /* Reset aux output amplifier and the ear amplifier */
        #if (ANLG_FAM != 3)
          vbcr &= ~(AUDIO_VBCR_VDLEAR | AUDIO_VBCR_VDLAUX);
          ABB_Audio_Config(vbcr);
        #endif

        break;
      }
      case AUDIO_SPEAKER_HANDHELD_HANDFREE:
      {
        #if (ANLG_FAM == 2)
          /* Reset the Head set configuration */
          vbcr2 &= (~(AUDIO_VBCR2_VDLHSO));
          ABB_Audio_Config_2(vbcr2);
        #endif

        #if (ANLG_FAM != 3)
          vbcr |= (AUDIO_VBCR_VDLAUX | AUDIO_VBCR_VDLEAR);
          ABB_Audio_Config(vbcr);
        #endif

        break;
      }
      default:
      {
        AUDIO_SEND_TRACE_PARAM("AUDIO MODE WRITE: speaker mode: this mode isn't supported",
          *data,
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
        break;
      }
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_speaker_gain_write                            */
  /*                                                                              */
  /*    Purpose:  This function set a new speaker gain.                           */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new speaker gain.                                                     */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_gain_write (INT8  *data)
  {
    T_AUDIO_SPEAKER_LEVEL volume;

    audio_mode_speaker_volume_read (&volume);

    ABB_CAL_DlVolume (volume.audio_speaker_level, (UINT8)(*data + 6));
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_speaker_filter_write                          */
  /*                                                                              */
  /*    Purpose:  This function enable/disable the DL filter.                     */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new DL filter state.                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_filter_write (INT8  *data)
  {
    UINT16 vbcr;

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
    #endif

    switch (*data)
    {
      case AUDIO_SPEAKER_FILTER_ON:
      {
        vbcr &= ~(AUDIO_VBCR_VFBYP);
        ABB_Audio_Config(vbcr);
        break;
      }
      case AUDIO_SPEAKER_FILTER_OFF:
      {
        vbcr |= (AUDIO_VBCR_VFBYP);
        ABB_Audio_Config(vbcr);
        break;
      }
      default :
      {
        AUDIO_SEND_TRACE_PARAM("AUDIO MODE WRITE: speaker filter: this state isn't supported",
          *data,
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
        break;
      }
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:  audio_mode_speaker_buzzer_write                           */
  /*                                                                              */
  /*    Purpose:  This function enable/disable the buzzer.                        */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new buzzer state.                                                     */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_buzzer_write (INT8  *data)
  {
    #if (ANLG_FAM== 1)
      UINT16 vbcr;

      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);

      switch (*data)
      {
        case AUDIO_SPEAKER_BUZZER_OFF:
        {
          /* Reset aux output amplifier and the ear amplifier */
          vbcr &= ~(AUDIO_VBCR_VDLEAR | AUDIO_VBCR_VDLAUX);

          vbcr &= ~(AUDIO_VBCR_VBUZ);
          ABB_Audio_Config(vbcr);
          break;
        }
        case AUDIO_SPEAKER_BUZZER_ON:
        {
          /* Reset aux output amplifier and the ear amplifier */
          vbcr &= ~(AUDIO_VBCR_VDLEAR | AUDIO_VBCR_VDLAUX);

          vbcr |= (AUDIO_VBCR_VBUZ);
          ABB_Audio_Config(vbcr);
          break;
        }
        default :
        {
          AUDIO_SEND_TRACE_PARAM("AUDIO MODE WRITE: buzzer: this state isn't supported",
            *data,
            RV_TRACE_LEVEL_ERROR);
          return(AUDIO_ERROR);
          break;
        }
      }
      return (AUDIO_OK);
    #endif
    #if (ANLG_FAM == 2)
      AUDIO_SEND_TRACE("AUDIO MODE WRITE: buzzer: this feature isn't supported by the current analog base band",
        RV_TRACE_LEVEL_ERROR);
      return(AUDIO_ERROR);
    #endif
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:  audio_mode_sidetone_gain_write                            */
  /*                                                                              */
  /*    Purpose:  This function set the sidetone gain.                            */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new sidetone gain.                                                    */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_sidetone_gain_write (INT8  *data)
  {
    UINT8  sidetone_index;

    sidetone_index = (UINT8)(25*((*data + 26)/3));

    ABB_SideTone (sidetone_index);
    return (AUDIO_OK);
  }


  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:  audio_mode_aec_write                                      */
  /*                                                                              */
  /*    Purpose:  This function set the AEC.                                      */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new AEC setting.                                                      */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_aec_write (T_AUDIO_AEC_CFG *data)
  {
    #if (L1_NEW_AEC)
      void *p_send_message = NULL;

      /* Send the FIR configuration message */
      /* allocate the buffer for the message to the L1 */
      p_send_message = audio_allocate_l1_message(sizeof(T_MMI_AEC_REQ));
      if (p_send_message != NULL)
      {
        /* Fill the parameter */
        ((T_MMI_AEC_REQ *)p_send_message)->aec_control =
                           (UINT16)( (data->aec_enable | 0x0080) |
                           (data->noise_suppression_enable | 0x0100) |
                           (data->noise_suppression_level) |
                           (data->aec_visibility) );

        ((T_MMI_AEC_REQ *)p_send_message)->cont_filter     = data->continuous_filtering;
        ((T_MMI_AEC_REQ *)p_send_message)->granularity_att = data->granularity_attenuation;
        ((T_MMI_AEC_REQ *)p_send_message)->coef_smooth     = data->smoothing_coefficient;
        ((T_MMI_AEC_REQ *)p_send_message)->es_level_max    = data->max_echo_suppression_level;
        ((T_MMI_AEC_REQ *)p_send_message)->fact_vad        = data->vad_factor;
        ((T_MMI_AEC_REQ *)p_send_message)->thrs_abs        = data->absolute_threshold;
        ((T_MMI_AEC_REQ *)p_send_message)->fact_asd_fil    = data->factor_asd_filtering;
        ((T_MMI_AEC_REQ *)p_send_message)->fact_asd_mut    = data->factor_asd_muting;

        /* send the start command to the audio L1 */
        audio_send_l1_message(MMI_AEC_REQ, p_send_message);
      }
      else
      {
        AUDIO_SEND_TRACE("AUDIO MODE WRITE: AEC: the audio AEC message isn't sent",
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
      }
    #elif (AEC)
        void *p_send_message = NULL;

      /* Send the FIR configuration message */
      /* allocate the buffer for the message to the L1 */
      p_send_message = audio_allocate_l1_message(sizeof(T_MMI_AEC_REQ));
      if (p_send_message != NULL)
      {
        /* Fill the parameter */
        ((T_MMI_AEC_REQ *)p_send_message)->aec_control =
                 (UINT16)( (data->aec_enable | 0x0080) |
                           (data->aec_mode) |
                           (data->echo_suppression_level) |
                           (data->noise_suppression_enable | 0x0100) |
                           (data->noise_suppression_level) );

        /* send the start command to the audio L1 */
        audio_send_l1_message(MMI_AEC_REQ, p_send_message);
      }
      else
      {
        AUDIO_SEND_TRACE("AUDIO MODE WRITE: AEC: the audio AEC message isn't sent",
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
      }
    #else
      AUDIO_SEND_TRACE("AUDIO MODE WRITE: AEC: feature not supported",
        RV_TRACE_LEVEL_ERROR);
      return(AUDIO_ERROR);
    #endif
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:  audio_mode_speaker_volume_write                           */
  /*                                                                              */
  /*    Purpose:  This function set the speaker volume.                           */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new speaker volume.                                                   */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_volume_write (T_AUDIO_SPEAKER_LEVEL *data)
  {
    ABB_DlVolume (data->audio_speaker_level);

    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_speaker_microphone_fir_write                  */
  /*                                                                              */
  /*    Purpose:  This function set the speaker and microphone FIR.               */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        new speaker or/and microphone FIR.                                    */
  /*        Note: if one of this two pointer equals NULL, it means corresponding  */
  /*              FIR doesn't need to be updated.                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        The data buffer must be deallocate only when the FIR confirmation is  */
  /*        received.                                                             */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_microphone_fir_write   (T_AUDIO_FIR_COEF *data_speaker,
                                                         T_AUDIO_FIR_COEF *data_microphone)
  {
    #if (FIR)
      void *p_send_message = NULL;

      if ( (data_speaker == NULL) &&
           (data_microphone == NULL) )
      {
        AUDIO_SEND_TRACE("AUDIO MODE WRITE: FIR: wrong arguments",
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
      }

      /* Send the FIR configuration message */
      /* allocate the buffer for the message to the L1 */
      p_send_message = audio_allocate_l1_message(sizeof(T_MMI_AUDIO_FIR_REQ));
      if (p_send_message != NULL)
      {
        if (data_speaker == NULL)
        {
          /* Fill the parameter */
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->update_fir = UL_FIR;
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->fir_loop = FALSE;
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->fir_dl_coefficient = NULL;
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->fir_ul_coefficient = &(data_microphone->coefficient[0]);
        }
        else
        if (data_microphone == NULL)
        {
          /* Fill the parameter */
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->update_fir = DL_FIR;
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->fir_loop = FALSE;
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->fir_dl_coefficient = &(data_speaker->coefficient[0]);
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->fir_ul_coefficient = NULL;
        }
        else
        {
          /* Fill the parameter */
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->update_fir = UL_DL_FIR;
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->fir_loop = FALSE;
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->fir_dl_coefficient = &(data_speaker->coefficient[0]);
          ((T_MMI_AUDIO_FIR_REQ *)p_send_message)->fir_ul_coefficient = &(data_microphone->coefficient[0]);
        }

        /* send the start command to the audio L1 */
        audio_send_l1_message(MMI_AUDIO_FIR_REQ, p_send_message);
      }
      else
      {
        AUDIO_SEND_TRACE("AUDIO MODE WRITE: FIR: the audio FIR message isn't send",
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
      }
    #else
      AUDIO_SEND_TRACE("AUDIO MODE WRITE: FIR: FIR isn't supported",
        RV_TRACE_LEVEL_ERROR);
      return(AUDIO_ERROR);
    #endif

    return (AUDIO_OK);
  }

  /********************************************************************************/
  /**********************     READ DRIVERS           ******************************/
  /********************************************************************************/

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_voice_path_read                               */
  /*                                                                              */
  /*    Purpose:  This function reads the current voice path.                     */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current path voice.                                                   */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_voice_path_read  (T_AUDIO_VOICE_PATH_SETTING *data)
  {
    #ifndef _WINDOWS
      INT16 audio_init, dai_mode;

      dai_mode   = (((l1s_dsp_com.dsp_ndb_ptr->d_dai_onoff)>>11) & 0x0003);
      audio_init = l1s_dsp_com.dsp_ndb_ptr->d_audio_init;

      // Note :
      //         tested_device  | dai_mode (bit 11 and 12)  |    test
      //         ---------------|---------------------------|----------------
      //              0         |     0                     |  no test
      //              1         |     2                     |  speech decoder
      //              2         |     1                     |  speech encoder
      //              3         |     0                     |  no test
      //              4         |     3                     |  Acouustic devices

      if(dai_mode == 0)
      /* There's no DAI mode */
      {
        #if (AUDIO_MODE)
          if (audio_init & B_GSM_ONLY)
          {
            *data = AUDIO_GSM_VOICE_PATH;
          }
          else
          if (audio_init & B_BT_CORDLESS)
          {
            *data = AUDIO_BLUETOOTH_CORDLESS_VOICE_PATH;
          }
          else
          if (audio_init & B_BT_HEADSET)
          {
            *data = AUDIO_BLUETOOTH_HEADSET;
          }
          else
          {
            AUDIO_SEND_TRACE_PARAM("AUDIO MODE READ: voice path: error in audio path mode",
              audio_init,
              RV_TRACE_LEVEL_ERROR);
            return(AUDIO_ERROR);
          }
        #else
          *data = AUDIO_GSM_VOICE_PATH;
        #endif
      }
      else
      {
        switch (dai_mode)
        {
          case 1:
          {
            *data = AUDIO_DAI_DECODER;
            break;
          }
          case 2:
          {
            *data = AUDIO_DAI_ENCODER;
            break;
          }
          case 3:
          {
            *data = AUDIO_DAI_ACOUSTIC;
            break;
          }
          default :
          {
            AUDIO_SEND_TRACE_PARAM("AUDIO MODE READ: voice path: error in DAI mode",
              dai_mode,
              RV_TRACE_LEVEL_ERROR);
            return(AUDIO_ERROR);
            break;
          }
        }
      }
  #else
    *data = AUDIO_GSM_VOICE_PATH;
  #endif
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_microphone_mode_read                          */
  /*                                                                              */
  /*    Purpose:  This function reads the current microphone mode.                */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Microphone mode.                                                      */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_microphone_mode_read  (INT8  *data)
  {
    UINT16 vbcr;
    #if (ANLG_FAM == 2)
      UINT16 vbcr2;
    #endif

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
      vbcr2 = (l1s_dsp_com.dsp_ndb_ptr->d_vbctrl2 >> 6);
    #endif

    #if (ANLG_FAM == 2)
      if (vbcr2 & (AUDIO_VBCR2_MICBIASEL | AUDIO_VBCR2_MICNAUX))
      {
        *data = AUDIO_MICROPHONE_HEADSET;
      }
      else
    #endif
    /* No headset mode */
    {
      if (vbcr & AUDIO_VBCR_VULSWITCH)
      {
        *data = AUDIO_MICROPHONE_HANDHELD;
      }
      else
      {
        *data = AUDIO_MICROPHONE_HANDFREE;
      }
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_microphone_gain_read                          */
  /*                                                                              */
  /*    Purpose:  This function reads the current microphone gain.                */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current microphone gain.                                              */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_microphone_gain_read  (INT8  *data)
  {
    UINT8 i = 1;
    UINT16 vbur, vulpg;


    vbur = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbuctrl >> 6);
    if (vbur & AUDIO_VBUR_DXEN)
    {
      *data = AUDIO_MICROPHONE_MUTE;
    }
    else
    /* Microphone not muted */
    {
      vulpg = (UINT16)(vbur & AUDIO_VBUR_VULPG);
      while (ABB_uplink_PGA_gain[i] != vulpg)
      {
        i++;
        if (i == 25)
        {
          AUDIO_SEND_TRACE_PARAM("AUDIO MODE READ: microphone gain: error in gain value",
            vulpg,
            RV_TRACE_LEVEL_ERROR);
          return(AUDIO_ERROR);
        }
      }
      *data = (INT8)(i - 12);
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_microphone_extra_gain_read                    */
  /*                                                                              */
  /*    Purpose:  This function reads the current microphone extra gain.          */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current microphone extra gain.                                        */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_microphone_extra_gain_read  (INT8  *data)
  {
    UINT16 vbcr;

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
    #endif

    if (vbcr & AUDIO_VBCR_VBDFAUXG)
    {
      *data = AUDIO_MICROPHONE_AUX_GAIN_28_2dB;
    }
    else
    {
      *data = AUDIO_MICROPHONE_AUX_GAIN_4_6dB;
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_microphone_output_bias_read                   */
  /*                                                                              */
  /*    Purpose:  This function reads the current microphone output bias.         */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current microphone output bias.                                       */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_microphone_output_bias_read  (INT8  *data)
  {
    UINT16 vbcr;

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
    #endif

    if (vbcr & AUDIO_VBCR_MICBIAS)
    {
      *data = AUDIO_MICROPHONE_OUTPUT_BIAS_2_5V;
    }
    else
    {
      *data = AUDIO_MICROPHONE_OUTPUT_BIAS_2_0V;
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_microphone_fir_read                           */
  /*                                                                              */
  /*    Purpose:  This function reads the current microphone FIR.                 */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current microphone FIR.                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_microphone_fir_read  (T_AUDIO_FIR_COEF *data)
  {
    #if (FIR)
      UINT8 i;

      for (i=0; i<31; i++)
      {
        #if (DSP == 33) || (DSP == 34) || (DSP == 35) || (DSP == 36)
          // For this DSP code the FIR coefficients are in API param memory
          data->coefficient[i] = l1s_dsp_com.dsp_param_ptr->a_fir31_uplink[i];
        #else
          data->coefficient[i] = l1s_dsp_com.dsp_ndb_ptr->a_fir31_uplink[i];
        #endif
      }
      return (AUDIO_OK);
    #else
      AUDIO_SEND_TRACE("AUDIO MODE READ: microphone FIR: FIR isn't supported",
        RV_TRACE_LEVEL_ERROR);
      return(AUDIO_ERROR);
    #endif
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_speaker_mode_read                             */
  /*                                                                              */
  /*    Purpose:  This function reads the current speaker mode.                   */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None                                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current speaker mode.                                                 */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_mode_read  (INT8  *data)
  {
    UINT16 vbcr;
    #if (ANLG_FAM == 2)
      UINT16 vbcr2;
    #endif

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
      vbcr2 = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl2 >> 6);
    #endif

    #if (ANLG_FAM == 2)
      if (vbcr2 & AUDIO_VBCR2_VDLHSO)
      {
        *data = AUDIO_SPEAKER_HEADSET;
      }
      else
    #endif
    /* No headset mode */
    #if (ANLG_FAM != 3)
      if ( (vbcr & AUDIO_VBCR_VDLEAR) &&
           (vbcr & AUDIO_VBCR_VDLAUX) )
      {
        *data = AUDIO_SPEAKER_HANDHELD_HANDFREE;
      }
      else
      if (vbcr & AUDIO_VBCR_VDLEAR)
      {
        *data = AUDIO_SPEAKER_HANDHELD;
      }
      else
      if (vbcr & AUDIO_VBCR_VDLAUX)
      {
        *data = AUDIO_SPEAKER_HANDFREE;
      }
      else
      {
        *data = AUDIO_SPEAKER_BUZZER;
      }
    #endif
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_speaker_gain_read                             */
  /*                                                                              */
  /*    Purpose:  This function reads the current speaker gain.                   */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None                                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current speaker gain.                                                 */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_gain_read  (INT8  *data)
  {
    UINT16 vbdr, vdlpg;
    UINT8 i=0;

    vbdr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbdctrl >> 6);
    vdlpg = (UINT16)(vbdr & AUDIO_VBDR_VDLPG);

    while (ABB_downlink_PGA_gain[i] != vdlpg)
    {
      i++;
      if (i == 13)
      {
        AUDIO_SEND_TRACE_PARAM("AUDIO MODE READ: speaker gain: error in gain value",
          vdlpg,
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
      }
    }
    *data = (INT8)(i - 6);
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_speaker_filter_read                           */
  /*                                                                              */
  /*    Purpose:  This function reads the current state of the DL filter.         */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None                                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current DL filter state.                                              */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_filter_read  (INT8  *data)
  {
    UINT16 vbcr;

    #if (ANLG_FAM == 1)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);
    #elif (ANLG_FAM == 2)
      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl1 >> 6);
    #endif

    if (vbcr & AUDIO_VBCR_VFBYP)
    {
      *data = AUDIO_SPEAKER_FILTER_OFF;
    }
    else
    {
      *data = AUDIO_SPEAKER_FILTER_ON;
    }
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:   audio_mode_speaker_fir_read                              */
  /*                                                                              */
  /*    Purpose:  This function reads the speaker FIR.                            */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current speaker FIR.                                                  */
  /*                                                                              */
  /*    Note:                                                                     */
  /*        None.                                                                 */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_fir_read  (T_AUDIO_FIR_COEF *data)
  {
    #if (FIR)
      UINT8 i;

      for (i=0; i<31; i++)
      {
        #if (DSP == 33) || (DSP == 34) || (DSP == 35) || (DSP == 36)
          // For this DSP code the FIR coefficients are in API param memory
          data->coefficient[i] = l1s_dsp_com.dsp_param_ptr->a_fir31_downlink[i];
        #else
          data->coefficient[i] = l1s_dsp_com.dsp_ndb_ptr->a_fir31_downlink[i];
        #endif
      }
      return (AUDIO_OK);
    #else
      AUDIO_SEND_TRACE("AUDIO MODE READ: speaker FIR: FIR isn't supported",
        RV_TRACE_LEVEL_ERROR);
      return(AUDIO_ERROR);
    #endif
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:  audio_mode_speaker_buzzer_read                            */
  /*                                                                              */
  /*    Purpose:  This function reads the current state of the buzzer.            */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current buzzer state.                                                 */
  /*                                                                              */
  /*    Note:                                                                     */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_buzzer_read  (INT8  *data)
  {
    #if (ANLG_FAM== 1)
      UINT16 vbcr;

      vbcr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbctrl >> 6);

      if (vbcr & AUDIO_VBCR_VBUZ)
      {
        *data = AUDIO_SPEAKER_BUZZER_ON;
      }
      else
      {
        *data = AUDIO_SPEAKER_BUZZER_OFF;
      }
      return (AUDIO_OK);
    #endif
    #if (ANLG_FAM == 2)
      AUDIO_SEND_TRACE("AUDIO MODE READ: buzzer: this feature isn't supported by the current analog base band",
        RV_TRACE_LEVEL_ERROR);
      return(AUDIO_ERROR);
    #endif
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:  audio_mode_sidetone_gain_read                             */
  /*                                                                              */
  /*    Purpose:  This function reads the current sidetone gain.                  */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current sidetone gain.                                                */
  /*                                                                              */
  /*    Note:                                                                     */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_sidetone_gain_read  (INT8  *data)
  {
    UINT16 vbur, vdlst;
    UINT8 i=0;

    vbur = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbuctrl >> 11);
    vdlst = (UINT16)(vbur & AUDIO_VBUR_VDLST);

    while (ABB_sidetone_gain[i] != vdlst)
    {
      i++;
      if (i == 10)
      {
        AUDIO_SEND_TRACE_PARAM("AUDIO MODE READ: sidetone gain: error in gain value",
          vdlst,
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
      }
    }
    *data = (INT8)((3 * i) - 26);
    return (AUDIO_OK);
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:  audio_mode_aec_read                                       */
  /*                                                                              */
  /*    Purpose:  This function reads the current AEC setting.                    */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current AEC setting.                                                  */
  /*                                                                              */
  /*    Note:                                                                     */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_aec_read  (T_AUDIO_AEC_CFG *data)
  {
    #if (L1_NEW_AEC)
      UINT16 aec_ctrl;

      aec_ctrl = l1s.aec.aec_control;

      data->aec_enable               = (UINT16)(aec_ctrl & AUDIO_AEC_ENABLE);
      data->aec_visibility           = (UINT16)(aec_ctrl & AUDIO_AEC_VISIBILITY_ENABLE);
      data->noise_suppression_enable = (UINT16)(aec_ctrl & AUDIO_NOISE_SUPPRESSION_ENABLE);
      data->noise_suppression_level  = (UINT16)(aec_ctrl & AUDIO_NOISE_18dB);

      data->continuous_filtering       = l1s_dsp_com.dsp_ndb_ptr->d_cont_filter;
      data->granularity_attenuation    = l1s_dsp_com.dsp_ndb_ptr->d_granularity_att;
      data->smoothing_coefficient      = l1s_dsp_com.dsp_ndb_ptr->d_coef_smooth;
      data->max_echo_suppression_level = l1s_dsp_com.dsp_ndb_ptr->d_es_level_max;
      data->vad_factor                 = l1s_dsp_com.dsp_ndb_ptr->d_fact_vad;
      data->absolute_threshold         = l1s_dsp_com.dsp_ndb_ptr->d_thrs_abs;
      data->factor_asd_filtering       = l1s_dsp_com.dsp_ndb_ptr->d_fact_asd_fil;
      data->factor_asd_muting          = l1s_dsp_com.dsp_ndb_ptr->d_fact_asd_mut;

      return (AUDIO_OK);

    #elif (AEC)
      UINT16 aec_ctrl;

      aec_ctrl = l1s.aec.aec_control;

      data->aec_enable = (UINT16)(aec_ctrl & AUDIO_AEC_ENABLE);
      data->aec_mode = (UINT16)(aec_ctrl & AUDIO_SHORT_ECHO);
      data->echo_suppression_level = (UINT16)(aec_ctrl & AUDIO_ECHO_18dB);
      data->noise_suppression_enable = (UINT16)(aec_ctrl & AUDIO_NOISE_SUPPRESSION_ENABLE);
      data->noise_suppression_level = (UINT16)(aec_ctrl & AUDIO_NOISE_18dB);
      return (AUDIO_OK);
    #else
      AUDIO_SEND_TRACE("AUDIO MODE READ: AEC: feature not supported",
        RV_TRACE_LEVEL_ERROR);
      return(AUDIO_ERROR);
    #endif
  }

  /********************************************************************************/
  /*                                                                              */
  /*    Function Name:  audio_mode_speaker_volume_read                            */
  /*                                                                              */
  /*    Purpose:  This function reads the current speaker volume.                 */
  /*                                                                              */
  /*    Input Parameters:                                                         */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Output Parameters:                                                        */
  /*        Status.                                                               */
  /*        Current speaker volume.                                               */
  /*                                                                              */
  /*    Note:                                                                     */
  /*       None.                                                                  */
  /*                                                                              */
  /*    Revision History:                                                         */
  /*        None.                                                                 */
  /*                                                                              */
  /********************************************************************************/
  T_AUDIO_RET audio_mode_speaker_volume_read  (T_AUDIO_SPEAKER_LEVEL *data)
  {
    UINT16 vbdr, volctl;
    UINT8 i=0;

    vbdr = (UINT16)(l1s_dsp_com.dsp_ndb_ptr->d_vbdctrl >> 10);
    volctl = (UINT16)(vbdr & AUDIO_VBDR_VOLCTL);

    while (ABB_volume_control_gain[i] != volctl)
    {
      i++;
      if (i == 6)
      {
        AUDIO_SEND_TRACE_PARAM("AUDIO MODE READ: speaker volume: error in volume value",
          volctl,
          RV_TRACE_LEVEL_ERROR);
        return(AUDIO_ERROR);
      }
    }
    if ( (i == 0) ||
         (i == 1) )
    {
      data->audio_speaker_level = i;
    }
    else
    {
      data->audio_speaker_level = (UINT8)((i - 1) * 50);
    }
    return (AUDIO_OK);
  }
#endif /* RVM_AUDIO_MAIN_SWE */
