/****************************************************************************/
/*                                                                          */
/*  File Name:  audio_env.c                                                 */
/*                                                                          */
/*  Purpose:  This file contains routines that will be called in order to:  */
/*            - notify the Bluetooth Environment of the AUDIO's             */
/*              Memory Banks requirements,                                  */
/*            - set AUDIO's task and memory banks IDs                       */
/*            - initialize all the AUDIO's data structures,                 */
/*            - start the AUDIO's task                                      */
/*            - stop the AUDIO's task                                       */
/*            - kill the AUDIO's task                                       */
/*                                                                          */
/*  Version   0.1                                                           */
/*                                                                          */
/*  Date         Modification                                               */
/*  ------------------------------------                                    */
/*  18 May 2001  Create                                                     */
/*                                                                          */
/*  Author       Francois Mazard - Stephanie Gerthoux                       */
/*                                                                          */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved*/
/****************************************************************************/

#include "rv/rv_defined_swe.h"
#ifdef RVM_AUDIO_MAIN_SWE
  #ifndef _WINDOWS
    #include "config/swconfig.cfg"
	#include "config/sys.cfg"
	#include "config/chipset.cfg"
  #endif

  #include "l1_confg.h"
  #include "rvm/rvm_use_id_list.h"
  #include "rv/rv_general.h"
  #include "rvm/rvm_gen.h"
  #include "rvm/rvm_priorities.h"
  #include "audio/audio_ffs_i.h"
  #include "audio/audio_api.h"
  #include "audio/audio_structs_i.h"
  #include "audio/audio_env_i.h"
  #include "rvf/rvf_target.h"
  #include "audio/audio_const_i.h"
  #include "audio/audio_macro_i.h"
  #include "audio/audio_var_i.h"

  #include <string.h>

  extern T_RV_RET audio_core(void);

  /* Initialisation of the pointer to the Global Environment Control block */
  T_AUDIO_ENV_CTRL_BLK *p_audio_gbl_var = NULL;


  /********************************************************************************/
  /* Function   : audio_get_info                                                  */
  /*                                                                              */
  /* Description : This function is called by the Bt Environment to learn         */
  /*               audio requirements in terms of memory, SWEs...                 */
  /*                                                                              */
  /* Parameters  : T_RVM_INFO_SWE  * swe_info: pointer to the structure to fill   */
  /*               containing infos related to the Audio SWE.                     */
  /*                                                                              */
  /* Return      :  T_BTE_RETURN                                                  */
  /*                                                                              */
  /* History      : 0.1 (11-May-2001)                                             */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RVM_RETURN audio_get_info(T_RVM_INFO_SWE  *infoSWEnt)
  {

    /* SWE info */
    infoSWEnt->swe_type = RVM_SWE_TYPE_4;
    infoSWEnt->type_info.type4.swe_use_id = AUDIO_USE_ID;

    memcpy( infoSWEnt->type_info.type4.swe_name, "AUDIO", sizeof("AUDIO") );

    infoSWEnt->type_info.type4.stack_size = AUDIO_STACK_SIZE;
    infoSWEnt->type_info.type4.priority   = RVM_AUDIO_TASK_PRIORITY;


    /* Set the return path */
    infoSWEnt->type_info.type4.return_path.callback_func  = NULL;
    infoSWEnt->type_info.type4.return_path.addr_id        = 0;


    /* memory bank info */
    #if (MELODY_E1) || (MELODY_E2) || (VOICE_MEMO) || (SPEECH_RECO)
      infoSWEnt->type_info.type4.nb_mem_bank = 0x03;
    #else
      infoSWEnt->type_info.type4.nb_mem_bank = 0x02;
    #endif

    /* Memory bank used to receive/send the message to/from the entity */
    memcpy ((UINT8 *) ((infoSWEnt->type_info.type4.mem_bank[0]).bank_name), "AUDIO_EXTERN",sizeof("AUDIO_EXTERN"));
    ((infoSWEnt->type_info.type4.mem_bank[0]).initial_params).size          = AUDIO_MB_EXTERNAL_SIZE;
    /* Set a watermark to ensure that biggest status message (Speech Reco) can always be sent */
    ((infoSWEnt->type_info.type4.mem_bank[0]).initial_params).watermark     = AUDIO_MB_EXTERNAL_WATERMARK - sizeof(T_AUDIO_SR_RECO_STATUS);

    /* Memory bank used to send/receive the message to/from the L1 */
    memcpy ((UINT8 *) ((infoSWEnt->type_info.type4.mem_bank[1]).bank_name), "AUDIO_INTERN",sizeof("AUDIO_INTERN"));
    ((infoSWEnt->type_info.type4.mem_bank[1]).initial_params).size          = AUDIO_MB_INTERNAL_SIZE;
    ((infoSWEnt->type_info.type4.mem_bank[1]).initial_params).watermark     = AUDIO_MB_INTERNAL_WATERMARK;

    #if (MELODY_E1) || (MELODY_E2) || (VOICE_MEMO) || (SPEECH_RECO)
      /* Memory bank used to download data to/from the flash */
      memcpy ((UINT8 *) ((infoSWEnt->type_info.type4.mem_bank[2]).bank_name), "AUDIO_FFS",sizeof("AUDIO_FFS"));
      ((infoSWEnt->type_info.type4.mem_bank[2]).initial_params).size          = AUDIO_MB_FFS_SIZE;
      ((infoSWEnt->type_info.type4.mem_bank[2]).initial_params).watermark     = AUDIO_MB_FFS_WATERMARK;
    #endif

    /* linked SWE info */
    #ifdef _WINDOWS
      /* this SWE requires no SWE to run in PC environement*/
      infoSWEnt->type_info.type4.nb_linked_swe   = 0x00;
    #else
      #if (MELODY_E1) || (MELODY_E2) || (VOICE_MEMO) || (SPEECH_RECO)
        /* this SWE requires the FFS SWE to run */
        infoSWEnt->type_info.type4.nb_linked_swe   = 0x01;
        infoSWEnt->type_info.type4.linked_swe_id[0]= FFS_USE_ID;
      #else
        infoSWEnt->type_info.type4.nb_linked_swe   = 0x00;
      #endif
      #if (L1_GTT == 1)
        infoSWEnt->type_info.type4.linked_swe_id[infoSWEnt->type_info.type4.nb_linked_swe] = TTY_USE_ID;
        (infoSWEnt->type_info.type4.nb_linked_swe)++;
      #endif
    #endif

    /* generic functions */
    infoSWEnt->type_info.type4.set_info = audio_set_info;
    infoSWEnt->type_info.type4.init     = audio_init;
    infoSWEnt->type_info.type4.core     = audio_core;
    infoSWEnt->type_info.type4.stop     = audio_stop;
    infoSWEnt->type_info.type4.kill     = audio_kill;

    return (RV_OK);
  } /**************** End of audio_get_info function ******************************/

  /********************************************************************************/
  /* Function   : audio_set_info                                                  */
  /*                                                                              */
  /* Description : This function is called by the RV Environment to inform        */
  /*               the audio SWE about task_id, mb_id and error function.         */
  /*                                                                              */
  /* Parameters  : - T_RVM_TASK_ID  taskId[]: array of task_id.                  */
  /*               - task_id[0] contains audio task_id.                           */
  /*               - T_RVF_MB_ID mb_id[]: array of memory bank ids.               */
  /*                - callback function to call in case of unrecoverable error.   */
  /*                                                                              */
  /* Return      : T_RVM_RETURN                                                   */
  /*                                                                              */
  /* History    : 0.1 (11-May-2001 )                                              */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RVM_RETURN audio_set_info(T_RVF_ADDR_ID addrId,
                              T_RV_RETURN   return_path[],
                              T_RVF_MB_ID   mbId[],
                              T_RVM_RETURN  (*callBackFct) ( T_RVM_NAME SWEntName,
                                                             T_RVM_RETURN errorCause,
                                                             T_RVM_ERROR_TYPE errorType,
                                                             T_RVM_STRING errorMsg) )
  {

    /* Declare local variable.*/
    T_RVF_MB_STATUS mb_status = RVF_GREEN;

    #if (VOICE_MEMO) || (MELODY_E1)
      UINT8   i;
    #endif

    /* Allocate memory required to store the Global Environment control Block. */
    mb_status = rvf_get_buf(mbId[1],
                            sizeof(T_AUDIO_ENV_CTRL_BLK),
                            (T_RVF_BUFFER **) & p_audio_gbl_var);

    /* If insufficient resources to properly run the AUDIO's task, then abort. */
    switch (mb_status)
    {
      case RVF_GREEN:
        {
          /* Initialize the Global Environment Control Block */
          memset((UINT8 *) p_audio_gbl_var,
                 0x00,
                 sizeof (T_AUDIO_ENV_CTRL_BLK));

          /* Store the memory bank IDs assigned to the AUDIO */
          p_audio_gbl_var->mb_external = mbId[0];
          p_audio_gbl_var->mb_internal = mbId[1];
          p_audio_gbl_var->mb_audio_ffs = mbId[2];

          /* Store the address ID assigned to the AUDIO */
          p_audio_gbl_var->addrId = addrId;

          /* Store the address ID of the audio FFS task */
          p_audio_gbl_var->audio_ffs_addrId = addrId;

          /* Store the function to be called whenever any unrecoverable */
          /* error occurs.                                              */
          p_audio_gbl_var->callBackFct = callBackFct;
          AUDIO_SEND_TRACE(" AUDIO (env). AUDIO's information set ",
                   RV_TRACE_LEVEL_DEBUG_LOW);
          break;
        }
      case RVF_YELLOW:
        {
          rvf_free_buf((T_RVF_BUFFER *) p_audio_gbl_var);
          AUDIO_TRACE_WARNING(" AUDIO memory warning (orange memory)");
          return (RV_MEMORY_ERR);
        }
      default:
        {
          AUDIO_TRACE_WARNING(" AUDIO memory warning (red memory)");
          return (RV_MEMORY_ERR);
        }
    } /* switch (mb_status) */


    /* Initialization of the audio keybeep state */
    #if (KEYBEEP)
      p_audio_gbl_var->keybeep.state = AUDIO_KEYBEEP_IDLE;
    #endif
    /* Initialization of the audio tones state */
    #if (TONE)
      p_audio_gbl_var->tones.state = AUDIO_TONES_IDLE;
    #endif
    /* Initialization of the audio melody E1 state */
    #if (MELODY_E1)
      p_audio_gbl_var->melody_E1_0.state = AUDIO_MELODY_E1_IDLE;
      p_audio_gbl_var->melody_E1_1.state = AUDIO_MELODY_E1_IDLE;
    #endif
    /* Initialization of the audio melody E2 state */
    #if (MELODY_E2)
      p_audio_gbl_var->melody_E2_0.state = AUDIO_MELODY_E2_IDLE;
      p_audio_gbl_var->melody_E2_1.state = AUDIO_MELODY_E2_IDLE;

      /* Initalize the flag to valie the .lsi file */
      p_audio_gbl_var->melody_E2_load_file_instruments.file_downloaded = FALSE;
    #endif
    /* Initialization of the audio voice memorization state */
    #if (VOICE_MEMO)
      p_audio_gbl_var->vm_play.state    = AUDIO_VM_PLAY_IDLE;
      p_audio_gbl_var->vm_record.state  = AUDIO_VM_RECORD_IDLE;
    #endif
    #if (L1_MIDI==1)
      p_audio_gbl_var->midi.state = AUDIO_IDLE;
    #endif
    #if (L1_AUDIO_DRIVER)
      for (i = 0; i < AUDIO_DRIVER_MAX_CHANNEL; i++)
      {
        p_audio_gbl_var->audio_driver_session[i].session_info.state = AUDIO_DRIVER_CHANNEL_WAIT_INIT;
        p_audio_gbl_var->audio_driver_session[i].session_req.nb_buffer = 0;
      }
    #endif
    #if (AUDIO_MEM_MANAGER)
      p_audio_gbl_var->audio_vm_amr_record.state = AUDIO_IDLE;
      p_audio_gbl_var->audio_vm_amr_play.state = AUDIO_IDLE;
      for (i = 0; i < AUDIO_MEM_MAX_CHANNEL; i++)
      {
        p_audio_gbl_var->audio_mem_session[i].session_info.state = AUDIO_MEM_IDLE;
      }
    #endif
    #if (VOICE_MEMO) || (MELODY_E1) || (MELODY_E2)
      for (i=0; i< AUDIO_FFS_MAX_CHANNEL; i++)
      {
        p_audio_gbl_var->audio_ffs_session[i].session_req.valid_channel;
      }
    #endif
    /* Initialization of the audio speech reco state machine */
    #if (SPEECH_RECO)
      p_audio_gbl_var->speech_reco.sr_enroll.state = AUDIO_SR_ENROLL_IDLE;
      p_audio_gbl_var->speech_reco.sr_update.state = AUDIO_SR_UPDATE_IDLE;
      p_audio_gbl_var->speech_reco.sr_reco.state   = AUDIO_SR_RECO_IDLE;
    #endif

    /* intialize the sate machine of audio mode managers */
    p_audio_gbl_var->audio_mode_var.full_access_write_var.state = AUDIO_MODE_FULL_WRITE_IDLE;
    p_audio_gbl_var->audio_mode_var.audio_mode_load_var.state = AUDIO_MODE_LOAD_IDLE;
    /* initialize the name ofthe current volume file */
    p_audio_gbl_var->audio_mode_var.audio_volume_var.audio_volume_path_name[0] = 0;

    return (RV_OK);
  } /*************** End of audio_set_info function ********************************/



  /********************************************************************************/
  /* Function   : audio_init                                                      */
  /*                                                                              */
  /* Description : This function is called by the BT Environment to initialize the*/
  /*               audio SWE before creating the task and calling audio_start.    */
  /*                                                                              */
  /* Parameters  : None                                                           */
  /*                                                                              */
  /* Return      : T_BTE_RETURN                                                   */
  /*                                                                              */
  /* History    : 0.1 (20-August-2000)                                            */
  /*                                                                              */
  /*                                                                              */
  /********************************************************************************/
  T_RVM_RETURN audio_init(void)
  {
    /* Put global audio initialization */

    return RV_OK;
  }


  /*************************************************************************************/
  /* Function	  : audio_stop                                                           */
  /*                                                                                   */
  /* Description : This function is called by the BT Environment to stop the audio SWE.*/
  /*                                                                                   */
  /* Parameters  : None                                                                */
  /*                                                                                   */
  /* Return      : T_BTE_RETURN                                                        */
  /*                                                                                   */
  /* History    : 0.1 (11-May-2001)                                                    */
  /*                                                                                   */
  /*                                                                                   */
  /*************************************************************************************/
  T_RVM_RETURN audio_stop(void)
  {
    /* other SWEs have not been killed yet, audio can send messages to other SWEs      */

    return RV_OK;
  }


  /*************************************************************************************/
  /* Function   : audio_kill                                                           */
  /*                                                                                   */
  /* Description : This function is called by the BT Environment to kill the audio     */
  /*               SWE, after the audio_stop function has been called.                 */
  /*                                                                                   */
  /* Parameters  : None                                                                */
  /*                                                                                   */
  /* Return      : T_BTE_RETURN                                                        */
  /*                                                                                   */
  /* History     : 0.1 (11-May-2001)                                                   */
  /*                                                                                   */
  /*                                                                                   */
  /*************************************************************************************/
  T_RVM_RETURN audio_kill (void)
  {
    /* free all memory buffer previously allocated */
    rvf_free_buf ((T_RVF_BUFFER *) p_audio_gbl_var);
    return RV_OK;
  }

#endif /* #ifdef RVM_AUDIO_MAIN_SWE */
