/****************************************************************************/
/*                                                                          */
/*  File Name:  audio_api.h                                                 */
/*                                                                          */
/*  Purpose:  This file contains data structures and functions prototypes   */
/*            used to send events to the AUDIO SWE.                         */
/*                                                                          */
/*  Version   0.1                                                           */
/*                                                                          */
/*  Date         Modification                                               */
/*  ------------------------------------                                    */
/*  18 May 2001  Create                                                     */
/*                                                                          */
/*  Author       Francois Mazard - Stephanie Gerthoux                       */
/*                                                                          */
/* (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved*/
/****************************************************************************/



  #ifndef __AUDIO_API_H_
    #define __AUDIO_API_H_

    #ifdef __cplusplus
      extern "C"
        {
    #endif

    #define AUDIO_EVENTS_MASK                       (0x3000)
    #define AUDIO_EVENT_EXTERN                      (0x0C00)

    /* Immediat return */
    typedef INT8 T_AUDIO_RET;

    /* possible values */
    #define AUDIO_OK                     (0)
    #define AUDIO_ERROR                  (-1)
    #define AUDIO_MODE_ERROR             (-2)
    /* Bad acquisition of the word. The word is too long or too short */
    #define AUDIO_BAD_ACQUISITION (-3)
    /* The DSP task to acquire the word takes to much time */
    #define AUDIO_TIME_OUT        (-4)
    /* Bad update of the model. The model from the database is too different */
    /* than the model built during the acquisition. */
    #define AUDIO_BAD_UPDATE      (-5)
    /* This word is out of vocabulary or the best words are too close  */
    #define AUDIO_BAD_RECOGNITION (-6)
    /* A word generated by the CTO algorithm is the best word. */
    #define AUDIO_CTO_WORD        (-7)
    /* The best word isn't the word to update. */
    #define AUDIO_CHECK_ERROR     (-8)
    #define AUDIO_NOT_ENOUGH_OSCILLATOR  (-9)

  /******************************* AUDIO DRIVER *****************************/
    #define AUDIO_VM_AMR_RECORD_SESSION_ID    (10)
    #define AUDIO_VM_AMR_PLAY_SESSION_ID      (11)
    #define AUDIO_MIDI_SESSION_ID             (12)

  /******************************* KEY BEEP REGISTRATION *****************************/
    /* Definition of no_beep */
    #define NO_BEEP     (0)

    /* Parameters */
    typedef struct
    {
      UINT16  frequency_beep[2]; /* Frequency of the 2 beeps */
      INT8    amplitude_beep[2];  /* Amplitude of the 2 beeps */
      UINT16  duration;
    } T_AUDIO_KEYBEEP_PARAMETER;

    /* Prototype */
    T_AUDIO_RET audio_keybeep_start (T_AUDIO_KEYBEEP_PARAMETER parameter,
                          T_RV_RETURN return_path);

    /* Event return */
    #define AUDIO_KEYBEEP_STATUS_MSG      (0x0001 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
    typedef struct
    {
      T_RV_HDR  os_hdr;
      INT8     status;
    } T_AUDIO_KEYBEEP_STATUS;

    /* Prototype */
    T_AUDIO_RET audio_keybeep_stop (T_RV_RETURN return_path);

    /******************************* TONES REGISTRATION *****************************/

    /* Definition of no_tone and tone_infinite */
    #define NO_TONE          (0)
    #define TONE_INFINITE    (0xFFFF)

    /* Parameters */
    typedef struct
    {
      UINT16  start_tone;        /* start date of the tone */
      UINT16  stop_tone;         /* stop date of the tone */
      UINT16  frequency_tone;    /* frequency of the tone */
      INT8    amplitude_tone;    /* amplitude of the tone */
    } T_AUDIO_TONE_DESC;

    typedef struct
    {
      T_AUDIO_TONE_DESC  tones[3];           /* Description of the 3 tones */
      UINT16              frame_duration;     /* Duration of the tones frame */
      UINT16              sequence_duration;  /* Duration of the sequence */
      UINT16              period_duration;    /* Duration of the period */
      UINT16              repetition;         /* number of period */
    } T_AUDIO_TONES_PARAMETER;

      /* Prototype */
      T_AUDIO_RET audio_tones_start (T_AUDIO_TONES_PARAMETER* p_parameter,
                            T_RV_RETURN return_path);

      /* Event return */
      #define AUDIO_TONES_STATUS_MSG              (0x0002 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_TONES_STATUS;

      /* Prototype */
      T_AUDIO_RET audio_tones_stop (T_RV_RETURN return_path);

      /******************************* AUDIO DRIVER REGISTRATION ************************/

      T_AUDIO_RET audio_driver_start_session(UINT8 channel_id, T_RV_RETURN notification_return_path);

      T_AUDIO_RET audio_driver_stop_session(UINT8 channel_id);

      T_AUDIO_RET audio_driver_free_session(UINT8 channel_id, T_RV_RETURN return_path);

      T_AUDIO_RET audio_driver_get_play_buffer(UINT8 channel_id, UINT8 **pp_buffer);
      T_AUDIO_RET audio_driver_play_buffer(UINT8 channel_id, UINT8 *p_buffer);

      #define AUDIO_DRIVER_INIT_STATUS_MSG          (0x0003 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR                      os_hdr;
        INT8                          status;
        UINT8                         channel_id;
        UINT8                         session_id;
      } T_AUDIO_DRIVER_INIT_STATUS;

      #define AUDIO_DRIVER_NOTIFICATION_MSG         (0x0004 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR header;
        UINT8    channel_id;
        UINT16   *p_buffer;
      }
      T_AUDIO_DRIVER_NOTIFICATION;

      #define AUDIO_DRIVER_LAST_NOTIFICATION_MSG    (0x0005 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR header;
        UINT8    channel_id;
        UINT16   *p_buffer;
        UINT32   recorded_size;
      }
      T_AUDIO_DRIVER_LAST_NOTIFICATION;

      #define AUDIO_DRIVER_STATUS_MSG               (0x0006 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR                      os_hdr;
        INT8                          status;
        UINT8                         status_type;
        UINT8                         channel_id;
      } T_AUDIO_DRIVER_STATUS;

      #define AUDIO_START_STATUS         (1)
      #define AUDIO_STOP_STATUS          (2)
      #define AUDIO_FREE_STATUS          (3)

      typedef struct
      {
        UINT16                 buffer_size;
        UINT8                  nb_buffer;
      } T_AUDIO_DRIVER_PARAMETER;

      /******************************* MELODY AND VOICE MEMO REGISTRATION *****************************/
      /* Definition of the maximum size of the path name */
      #define AUDIO_PATH_NAME_MAX_SIZE                 (20)
      /* Definition of the maximum number of instruments */
      #define AUDIO_MELODY_E2_MAX_NUMBER_OF_INSTRUMENT (128)

      /******************************* MELODY REGISTRATION *****************************/

      /* Definition of loopback melody mode */
      #define AUDIO_MELODY_NO_LOOPBACK       (0)
      #define AUDIO_MELODY_LOOPBACK          (1)

      /* Definition ofthe mode of melody */
      #define AUDIO_MELODY_GAME_MODE        (0)
      #define AUDIO_MELODY_NORMAL_MODE      (1)

      /******** Melody E1 ********/

      /* Parameters */
      typedef struct
      {
        char melody_name[AUDIO_PATH_NAME_MAX_SIZE]; /* File name of the melody */
        BOOLEAN loopback;                     /* the melody is played indefinitely */
        BOOLEAN melody_mode;                  /* mode of the melody */
      } T_AUDIO_MELODY_E1_PARAMETER;

      typedef struct
      {
        char melody_name[AUDIO_PATH_NAME_MAX_SIZE]; /* File name of the melody */
      }
      T_AUDIO_MELODY_E1_STOP_PARAMETER;

      /* Prototype */
      T_AUDIO_RET audio_melody_E1_start (T_AUDIO_MELODY_E1_PARAMETER *parameter,
                                         T_RV_RETURN return_path);

      /* Event return */
      #define AUDIO_MELODY_E1_STATUS_MSG              (0x0007 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_MELODY_E1_STATUS;

      /* Prototype */
      T_AUDIO_RET audio_melody_E1_stop (T_AUDIO_MELODY_E1_STOP_PARAMETER *parameter,
                                        T_RV_RETURN return_path);

      /******** Melody E2 ********/

      /* Parameters */
      typedef struct
      {
        char    melody_E2_name[AUDIO_PATH_NAME_MAX_SIZE]; /* File name of the melody */
        BOOLEAN E2_loopback;                              /* the melody is played indefinitely */
        BOOLEAN melody_E2_mode;                           /* mode of the melody (Game/Normal)*/
      } T_AUDIO_MELODY_E2_PARAMETER;

      typedef struct
      {
        char melody_E2_name[AUDIO_PATH_NAME_MAX_SIZE]; /* File name of the melody */
      } T_AUDIO_MELODY_E2_STOP_PARAMETER;

      typedef struct
      {
        char melody_E2_file_name[AUDIO_PATH_NAME_MAX_SIZE]; /* File name of the melody */
      } T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER;


      /* Prototype */
      T_AUDIO_RET audio_melody_E2_load_file_instruments (
                 T_AUDIO_MELODY_E2_LOAD_FILE_INSTR_PARAMETER *p_parameter);

      T_AUDIO_RET audio_melody_E2_start (T_AUDIO_MELODY_E2_PARAMETER *p_parameter,
                                         T_RV_RETURN return_path);
      T_AUDIO_RET audio_melody_E2_stop (T_AUDIO_MELODY_E2_STOP_PARAMETER *parameter,
                                        T_RV_RETURN return_path);

      /* Event return */
      /* For melody_E2 */
      #define AUDIO_MELODY_E2_STATUS_MSG              (0x0008 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)

      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_MELODY_E2_STATUS;



      /******************************* VOICE MEMO REGISTRATION *****************************/

      /* ---------------------------- */
      /*     Voice memo PLAY          */
      /* ---------------------------- */

      /* Parameters */
      typedef struct
      {
        char memo_name[AUDIO_PATH_NAME_MAX_SIZE]; /* File name of the Voice Memo play */
      } T_AUDIO_VM_PLAY_PARAMETER;

      /* Prototype */
      T_AUDIO_RET audio_vm_play_start(T_AUDIO_VM_PLAY_PARAMETER *p_parameter,
                                         T_RV_RETURN return_path);

      /* Event return */
      #define AUDIO_VM_PLAY_STATUS_MSG              (0x0009 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR   os_hdr;
        INT8      status;
      } T_AUDIO_VM_PLAY_STATUS;

      /* Prototype */
      T_AUDIO_RET audio_vm_play_stop (T_RV_RETURN return_path);

      /* ---------------------------- */
      /*     Voice memo RECORD        */
      /* ---------------------------- */

      /* Definition of compression mode */
      #define AUDIO_VM_NO_COMPRESSION_MODE       (0)
      #define AUDIO_VM_COMPRESSION_MODE          (1)

      /* Parameters */
      typedef struct
      {
        char    memo_name[AUDIO_PATH_NAME_MAX_SIZE]; /* File name of the Voice Memo play */
        UINT32  memo_duration;    /* maximum duration of the voice memo */
        BOOL    compression_mode; /* activate the compression */
        UINT16  microphone_gain;  /* recording gain applies to microphone */
        UINT16  network_gain;     /* gain applies to the network voice */
      } T_AUDIO_VM_RECORD_PARAMETER;

      /* Prototype */
      T_AUDIO_RET audio_vm_record_start( T_AUDIO_VM_RECORD_PARAMETER *p_record_parameter,
                                         T_AUDIO_TONES_PARAMETER *p_tones_parameter,
                                         T_RV_RETURN return_path);

      /* Event return */
      #define AUDIO_VM_RECORD_STATUS_MSG              (0x000A | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
        UINT16    recorded_duration;
      } T_AUDIO_VM_RECORD_STATUS;

      /* Prototype */
      T_AUDIO_RET audio_vm_record_stop (T_RV_RETURN return_path);

      /******************************* VOICE MEMO DRIVER AMR REGISTRATION *****************************/

      /* ---------------------------- */
      /*     Voice memo AMR PLAY      */
      /* ---------------------------- */

      /* Parameters */
      typedef struct
      {
        char    memo_name[AUDIO_PATH_NAME_MAX_SIZE]; /* File name of the Voice Memo AMR play */
      } T_AUDIO_AMR_PLAY_FROM_FFS_PARAMETER;

      typedef T_AUDIO_AMR_PLAY_FROM_FFS_PARAMETER T_AUDIO_MMS_PLAY_FROM_FFS_PARAMETER;

      typedef struct
      {
        UINT8  *p_buffer; /* pointer on buffer to copy from */
		UINT16 buffer_size; /* size of buffer */
      } T_AUDIO_AMR_PLAY_FROM_RAM_PARAMETER;

      #define AUDIO_AMR_PLAY_FROM_MEM_STATUS_MSG            (0x000B | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_AMR_PLAY_STATUS;

      /* Prototype */
      T_AUDIO_RET audio_amr_play_from_ffs_start (T_AUDIO_AMR_PLAY_FROM_FFS_PARAMETER *p_record_parameter,
                                                 T_RV_RETURN return_path);
      T_AUDIO_RET audio_amr_play_from_ffs_stop  (void);

      T_AUDIO_RET audio_mms_play_from_ffs_start (T_AUDIO_MMS_PLAY_FROM_FFS_PARAMETER *p_record_parameter,
                                                 T_RV_RETURN return_path);
      T_AUDIO_RET audio_mms_play_from_ffs_stop  (void);

      T_AUDIO_RET audio_amr_play_from_ram_start (T_AUDIO_AMR_PLAY_FROM_RAM_PARAMETER *p_record_parameter,
                                                 T_RV_RETURN return_path);
      T_AUDIO_RET audio_amr_play_from_ram_stop  (void);

      T_AUDIO_RET audio_driver_init_vm_amr_play_session(T_AUDIO_DRIVER_PARAMETER *p_driver_parameter, T_RV_RETURN return_path);

      /* ---------------------------- */
      /*     Voice memo AMR RECORD    */
      /* ---------------------------- */

      /* Definition of compression mode */
      #define AUDIO_AMR_NO_COMPRESSION_MODE       (0)
      #define AUDIO_AMR_COMPRESSION_MODE          (1)

      /* Definition of AMR vocoders */
      #define AUDIO_AMR_VOCODER_4_75              (0)
      #define AUDIO_AMR_VOCODER_5_15              (1)
      #define AUDIO_AMR_VOCODER_5_90              (2)
      #define AUDIO_AMR_VOCODER_6_70              (3)
      #define AUDIO_AMR_VOCODER_7_40              (4)
      #define AUDIO_AMR_VOCODER_7_95              (5)
      #define AUDIO_AMR_VOCODER_10_2              (6)
      #define AUDIO_AMR_VOCODER_12_2              (7)

      /* Parameters */
      typedef struct
      {
        char    memo_name[AUDIO_PATH_NAME_MAX_SIZE]; /* File name of the Voice Memo AMR play */
        UINT32  memo_duration;    /* maximum duration of the voice memo */
        BOOL    compression_mode; /* activate the compression */
        UINT16  microphone_gain;  /* recording gain applies to microphone */
        UINT8   amr_vocoder;      /* AMR codec */
      } T_AUDIO_AMR_RECORD_TO_FFS_PARAMETER;

      typedef T_AUDIO_AMR_RECORD_TO_FFS_PARAMETER T_AUDIO_MMS_RECORD_TO_FFS_PARAMETER;

      typedef struct
      {
        UINT8   *p_buffer; /* pointer on buffer to copy to, length must be > memo_duration */
        UINT32  memo_duration;    /* maximum duration of the voice memo */
        BOOL    compression_mode; /* activate the compression */
        UINT16  microphone_gain;  /* recording gain applies to microphone */
        UINT8   amr_vocoder;      /* AMR codec */
      } T_AUDIO_AMR_RECORD_TO_RAM_PARAMETER;

      /* Parameters */
      typedef struct
      {
        UINT32  memo_duration;    /* maximum duration of the voice memo */
        BOOL    compression_mode; /* activate the compression */
        UINT16  microphone_gain;  /* recording gain applies to microphone */
        UINT8   amr_vocoder;      /* AMR codec */
      } T_AUDIO_DRIVER_VM_AMR_RECORD_PARAMETER;

      #define AUDIO_AMR_RECORD_TO_MEM_STATUS_MSG               (0x000C | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
        UINT32    recorded_duration;
      } T_AUDIO_AMR_RECORD_STATUS;

      /* Prototype */
      T_AUDIO_RET audio_amr_record_to_ffs_start (T_AUDIO_AMR_RECORD_TO_FFS_PARAMETER *p_record_parameter,
                                                 T_RV_RETURN return_path);
      T_AUDIO_RET audio_amr_record_to_ffs_stop  (void);

      T_AUDIO_RET audio_mms_record_to_ffs_start (T_AUDIO_MMS_RECORD_TO_FFS_PARAMETER *p_record_parameter,
                                                 T_RV_RETURN return_path);
      T_AUDIO_RET audio_mms_record_to_ffs_stop  (void);

      T_AUDIO_RET audio_amr_record_to_ram_start (T_AUDIO_AMR_RECORD_TO_RAM_PARAMETER *p_record_parameter,
                                                 T_RV_RETURN return_path);
      T_AUDIO_RET audio_amr_record_to_ram_stop  (void);

      T_AUDIO_RET audio_driver_init_vm_amr_record_session(T_AUDIO_DRIVER_VM_AMR_RECORD_PARAMETER *p_record_parameter,
                                                          T_AUDIO_DRIVER_PARAMETER *p_driver_parameter,
                                                          T_RV_RETURN return_path);

      /******************************* AUDIO SPEECH RECOGNITION REGISTRATION ************************/

      /* ---------------------------- */
      /*     Start Enrollment         */
      /* ---------------------------- */

      /* Symbolic constant */
      #define AUDIO_SR_RECORD_SPEECH      (1)
      #define AUDIO_SR_NO_RECORD_SPEECH   (0)

      /* maximum number of vocabulary name */
      #define AUDIO_SR_MAX_VOCABULARY_NAME    (10)

      /* Parameters */
      typedef struct {
        char  enroll_name[AUDIO_PATH_NAME_MAX_SIZE];  /* name and path of the word to enroll */
        BOOL  record_speech;  /* speech sample must be recorded or not */
      }T_AUDIO_SR_ENROLL_PARAMETER;

      /* Prototype */
      T_AUDIO_RET audio_sr_enroll_start (T_AUDIO_SR_ENROLL_PARAMETER *p_parameter,
                                         T_RV_RETURN return_path);

      /* Event return */
      #define AUDIO_SR_ENROLL_STATUS_MSG              (0x000D | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_SR_ENROLL_STATUS;

      /* ---------------------------- */
      /*     Stop Enrollment          */
      /* ---------------------------- */

      /* Symbolic constant */

      /* Parameters */

      /* Prototype */
      T_AUDIO_RET audio_sr_enroll_stop ( T_RV_RETURN return_path );

      /* Event return */

      /* ---------------------------- */
      /*     Start Update             */
      /* ---------------------------- */

      /* Symbolic constant */

      /* Parameters */
      typedef struct {
        char  update_name[AUDIO_PATH_NAME_MAX_SIZE];  /* name and path of the word to update */
        BOOL  record_speech;  /* speech sample must be recorded or not */
      }T_AUDIO_SR_UPDATE_PARAMETER;

      /* Prototype */
      T_AUDIO_RET audio_sr_update_start (T_AUDIO_SR_UPDATE_PARAMETER *p_parameter,
                                         T_RV_RETURN return_path);

      /* Event return */
      #define AUDIO_SR_UPDATE_STATUS_MSG              (0x000E | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_SR_UPDATE_STATUS;

      /* ---------------------------- */
      /*     Stop Update              */
      /* ---------------------------- */

      /* Symbolic constant */

      /* Parameters */

      /* Prototype */
      T_AUDIO_RET audio_sr_update_stop ( T_RV_RETURN return_path );

      /* Event return */

      /* ---------------------------- */
      /*     Start Update             */
      /* ---------------------------- */

      /* Symbolic constant */

      /* Parameters */
      typedef struct {
        /* directory of the vocabulary database */
        char  database_directory[AUDIO_PATH_NAME_MAX_SIZE];
      }T_AUDIO_SR_RECO_PARAMETER;

      /* Prototype */
      T_AUDIO_RET audio_sr_reco_start (T_AUDIO_SR_RECO_PARAMETER *p_parameter,
                                         T_RV_RETURN return_path);

      /* Event return */
      #define AUDIO_SR_RECO_STATUS_MSG              (0x000F | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct {
        T_RV_HDR  os_hdr;
        /* status about the recognition task */
        INT8      status;
        /*path ofthe recognized word */
        char word_recognized[AUDIO_PATH_NAME_MAX_SIZE];
      } T_AUDIO_SR_RECO_STATUS;

      /* ---------------------------- */
      /*     Stop reco                */
      /* ---------------------------- */

      /* Symbolic constant */

      /* Parameters */

      /* Prototype */
      T_AUDIO_RET audio_sr_reco_stop ( T_RV_RETURN return_path );

      /* Event return */



    /******************************* MIDI REGISTRATION *****************************/

      // Parameters
      typedef struct
      {
        char midi_name[AUDIO_PATH_NAME_MAX_SIZE]; // File name of the MIDI to play
      } T_AUDIO_MIDI_PARAMETER;

      // Prototype
      T_AUDIO_RET audio_midi_start(T_AUDIO_MIDI_PARAMETER *parameter, T_RV_RETURN *p_return_path);

      // Event return
      #define AUDIO_MIDI_STATUS_MSG      (0x0010 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR os_hdr;
        INT8     status;
      } T_AUDIO_MIDI_STATUS;

      // Prototype
      T_AUDIO_RET audio_midi_stop(void);

      T_AUDIO_RET audio_driver_init_midi_session(T_AUDIO_DRIVER_PARAMETER *p_driver_parameter, T_RV_RETURN *p_return_path);


      /******************************* AUDIO CONFIGURATION REGISTRATION *****************************/

      /******************/
      /*** Voice path ***/
      /******************/

      /* Parameters */

      /* Voice path selection */
      #define AUDIO_GSM_VOICE_PATH                (0)
      #define AUDIO_BLUETOOTH_CORDLESS_VOICE_PATH (1)
      #define AUDIO_BLUETOOTH_HEADSET             (2)
      #define AUDIO_DAI_ENCODER                   (12)
      #define AUDIO_DAI_DECODER                   (11)
      #define AUDIO_DAI_ACOUSTIC                  (14)

      /* Structure */
      /* audio path used */
      typedef UINT8 T_AUDIO_VOICE_PATH_SETTING;

      /********************************/
      /*** Microphone configuration ***/
      /********************************/

      /* Parameters */
      /* Mode */
      #define AUDIO_MICROPHONE_HANDHELD   (0)
      #define AUDIO_MICROPHONE_HANDFREE   (1)
      #define AUDIO_MICROPHONE_HEADSET    (2)

      /* gain */
      #define AUDIO_MICROPHONE_MUTE       (13)

      /* output_bias */
      #define AUDIO_MICROPHONE_OUTPUT_BIAS_2_0V  (0)
      #define AUDIO_MICROPHONE_OUTPUT_BIAS_2_5V  (1)

      /* extra gain */
      #define AUDIO_MICROPHONE_AUX_GAIN_28_2dB   (0)
      #define AUDIO_MICROPHONE_AUX_GAIN_4_6dB    (1)

      /* Structure */
      typedef struct
      {
        UINT16 coefficient[31];
      }
      T_AUDIO_FIR_COEF;

      typedef struct
      {
        /* gain of the microphone */
        INT8    gain;
        /* microphone output bias */
        INT8    output_bias;
        /* coefficients of the microphone FIR */
        T_AUDIO_FIR_COEF    fir;
      }
      T_AUDIO_MICROPHONE_MODE_HANDHELD;

      typedef struct {
        /* gain of the microphone */
        INT8    gain;
        /* extra gain of the microphone */
        INT8    extra_gain;
        /* microphone output bias */
        INT8    output_bias;
        /* coefficients of the microphone FIR */
        T_AUDIO_FIR_COEF    fir;
      }
      T_AUDIO_MICROPHONE_MODE_HANDFREE;

      typedef struct {
        /* gain of the microphone */
        INT8    gain;
        /* microphone output bias */
        INT8    output_bias;
        /* coefficients of the microphone FIR */
        T_AUDIO_FIR_COEF    fir;
      }
      T_AUDIO_MICROPHONE_MODE_HEADSET;

      typedef union
      {
        /* handheld mode parameters */
        T_AUDIO_MICROPHONE_MODE_HANDHELD handheld;
        /* handfree mode parameters */
        T_AUDIO_MICROPHONE_MODE_HANDFREE handfree;
        /* headset mode parameters */
        T_AUDIO_MICROPHONE_MODE_HEADSET  headset;
      }
      T_AUDIO_MICROPHONE_MODE;

      typedef struct
      {
        /* mode of the microphone */
        INT8                     mode;
        /* Setting of the current mode */
        T_AUDIO_MICROPHONE_MODE  setting;
      }
      T_AUDIO_MICROPHONE_SETTING;

      /*****************************/
      /*** Speaker configuration ***/
      /*****************************/

      /* Paratemers */
      /* Mode*/
      #define AUDIO_SPEAKER_HANDHELD          (0)
      #define AUDIO_SPEAKER_HANDFREE          (1)
      #define AUDIO_SPEAKER_HEADSET           (2)
      #define AUDIO_SPEAKER_BUZZER            (3)
      #define AUDIO_SPEAKER_HANDHELD_HANDFREE (4)

      /* filter */
      #define AUDIO_SPEAKER_FILTER_ON   (0)
      #define AUDIO_SPEAKER_FILTER_OFF  (1)

      /* Buzzer */
      #define AUDIO_SPEAKER_BUZZER_ON   (0)
      #define AUDIO_SPEAKER_BUZZER_OFF  (1)

      /* Structure */
      typedef struct
      {
        /* gain of the speaker */
        INT8  gain;
        /* use the audio filter */
        INT8  audio_filter;
        /* coefficients of the speaker FIR */
        T_AUDIO_FIR_COEF    fir;
      }
      T_AUDIO_SPEAKER_MODE_HANDHELD;

      typedef struct
      {
        /* gain of the speaker */
        INT8   gain;
        /* use the audio filter */
        INT8   audio_filter;
        /* coefficients of the speaker FIR */
        T_AUDIO_FIR_COEF    fir;
      }
      T_AUDIO_SPEAKER_MODE_HANDFREE;

      typedef struct
      {
        /* gain of the speaker */
        INT8   gain;
        /* use the audio filter */
        INT8   audio_filter;
        /* coefficients of the speaker FIR */
        T_AUDIO_FIR_COEF    fir;
      }
      T_AUDIO_SPEAKER_MODE_HEADSET;

      typedef struct
      {
        /* activate the buzzer */
        INT8  activate;
      }
      T_AUDIO_SPEAKER_MODE_BUZZER;

      typedef struct
      {
        /* gain of the speaker */
        INT8   gain;
        /* use the audio filter */
        INT8   audio_filter;
        /* coefficients of the speaker FIR */
        T_AUDIO_FIR_COEF    fir;
      }
      T_AUDIO_SPEAKER_MODE_HANDHELD_HANDFREE;

      typedef union
      {
        /* handheld mode parameters */
        T_AUDIO_SPEAKER_MODE_HANDHELD handheld;
        /* handfree mode parameters */
        T_AUDIO_SPEAKER_MODE_HANDFREE handfree;
        /* headset mode parameters */
        T_AUDIO_SPEAKER_MODE_HEADSET  headset;
        /* buzzer mode parameters */
        T_AUDIO_SPEAKER_MODE_BUZZER   buzzer;
        /* handheld and handfree mode parameters */
        T_AUDIO_SPEAKER_MODE_HANDHELD_HANDFREE handheld_handfree;
      }
      T_AUDIO_SPEAKER_MODE;

      typedef struct
      {
        /* mode of the speaker */
        INT8                      mode;
        /* Setting of the current mode */
        T_AUDIO_SPEAKER_MODE      setting;
      }
      T_AUDIO_SPEAKER_SETTING;

      /********************************************/
      /*** Mirophone-Speaker loop configuration ***/
      /********************************************/

      /* sidetone */
      #define AUDIO_SIDETONE_OPEN             (-26)

      /* AEC enable */
      #define AUDIO_AEC_ENABLE                (0x0002)
      #define AUDIO_AEC_DISABLE               (0x0000)

      /* AEC visibility */
      #define AUDIO_AEC_VISIBILITY_ENABLE     (0x0200)
      #define AUDIO_AEC_VISIBILITY_DISABLE    (0x0000)

      /* AEC mode */
      #define AUDIO_SHORT_ECHO                (0x0200)
      #define AUDIO_LONG_ECHO                 (0x0000)

      /* echo suppression level */
    #if (L1_NEW_AEC)
      #define AUDIO_MAX_ECHO_0dB              (0x7FFF)
      #define AUDIO_MAX_ECHO_2dB              (0x65AA)
      #define AUDIO_MAX_ECHO_3dB              (0x59AD)
      #define AUDIO_MAX_ECHO_6dB              (0x4000)
      #define AUDIO_MAX_ECHO_12dB             (0x1FFF)
      #define AUDIO_MAX_ECHO_18dB             (0x0FFF)
      #define AUDIO_MAX_ECHO_24dB             (0x07FF)
    #else
      #define AUDIO_ECHO_0dB                  (0x0000)
      #define AUDIO_ECHO_6dB                  (0x0008)
      #define AUDIO_ECHO_12dB                 (0x0010)
      #define AUDIO_ECHO_18dB                 (0x0018)
    #endif

      /* noise suppression enable */
      #define AUDIO_NOISE_SUPPRESSION_ENABLE  (0x0004)
      #define AUDIO_NOISE_SUPPRESSION_DISABLE (0x0000)

      /* noise suppression level */
      #define AUDIO_NOISE_NO_LIMITATION       (0x0000)
      #define AUDIO_NOISE_6dB                 (0x0020)
      #define AUDIO_NOISE_12dB                (0x0040)
      #define AUDIO_NOISE_18dB                (0x0060)

      typedef struct
      {
        /* Enable the AEC */
        UINT16    aec_enable;
      #if (L1_NEW_AEC)
        BOOL   continuous_filtering;
        UINT16 granularity_attenuation;
        UINT16 smoothing_coefficient;
        UINT16 max_echo_suppression_level;
        UINT16 vad_factor;
        UINT16 absolute_threshold;
        UINT16 factor_asd_filtering;
        UINT16 factor_asd_muting;
        UINT16 aec_visibility; // UINT16 and not boolean because in L1, it is in aec_control
      #else
        /* Mode of the AEC */
        UINT16    aec_mode;
        /* level of the echo cancellation */
        UINT16    echo_suppression_level;
      #endif
        /* enable the noise suppression */
        UINT16    noise_suppression_enable;
        /* level of the noise suppression */
        UINT16    noise_suppression_level;
      }
      T_AUDIO_AEC_CFG;

      typedef struct
      {
        /* gain of the sidetone */
        INT8                          sidetone_gain;
        /* configuration of the acoustic echo cancellation */
        T_AUDIO_AEC_CFG               aec;
      }
      T_AUDIO_MICROPHONE_SPEAKER_LOOP_SETTING;

      /****************************************/
      /*      audio mode file struture        */
      /* this file is in the folder /AUDCFG/  */
      /* and has the extention .CFG           */
      /****************************************/
      typedef struct
      {
        /* group of setting to define the audio path used */
        T_AUDIO_VOICE_PATH_SETTING              audio_path_setting;
        /* group of setting to configure the audio path of the microphone */
        T_AUDIO_MICROPHONE_SETTING              audio_microphone_setting;
        /* group of setting to configure the audio path of the speaker */
        T_AUDIO_SPEAKER_SETTING                 audio_speaker_setting;
        /* group of setting to configure the audio mode involved */
        /* in tr microhpone and speaker loop */
        T_AUDIO_MICROPHONE_SPEAKER_LOOP_SETTING audio_microphone_speaker_loop_setting;
      }
      T_AUDIO_MODE;

      /****************************************/
      /*      audio volume file struture      */
      /* this file is in the folder /AUDCFG/  */
      /* and has the extention .VOL           */
      /****************************************/
      /* speaker volume action */
      #define AUDIO_SPEAKER_VOLUME_INCREASE     (0)
      #define AUDIO_SPEAKER_VOLUME_DECREASE     (1)
      #define AUDIO_SPEAKER_VOLUME_SET          (2)

      /* speaker volume value */
      #define AUDIO_SPEAKER_VOLUME_MUTE         (0)
      #define AUDIO_SPEAKER_VOLUME_24dB         (1)
      #define AUDIO_SPEAKER_VOLUME_18dB         (50)
      #define AUDIO_SPEAKER_VOLUME_12dB         (100)
      #define AUDIO_SPEAKER_VOLUME_6dB          (150)
      #define AUDIO_SPEAKER_VOLUME_0dB          (200)
      typedef struct
      {
        /* volume of the audio speaker */
        UINT8      audio_speaker_level;
      }
      T_AUDIO_SPEAKER_LEVEL;

      /******************/
      /* MMI family API */
      /******************/


      /* structure */
      /* maximum number of number of the filname characters */
      #define AUDIO_MODE_FILENAME_MAX_SIZE    (10)

      typedef struct
      {
        char  audio_mode_filename[AUDIO_MODE_FILENAME_MAX_SIZE];
      }
      T_AUDIO_MODE_LOAD;

      typedef struct
      {
        char  audio_mode_filename[AUDIO_MODE_FILENAME_MAX_SIZE];
      }
      T_AUDIO_MODE_SAVE;

      typedef struct
      {
        INT8    volume_action;
        UINT8   value;
      }
      T_AUDIO_SPEAKER_VOLUME;

      /* Prototypes of MMI family*/
      T_AUDIO_RET audio_mode_load   (T_AUDIO_MODE_LOAD  *p_parameter,
                                     T_RV_RETURN        return_path);

      T_AUDIO_RET audio_mode_save   (T_AUDIO_MODE_SAVE  *p_parameter,
                                     T_RV_RETURN        return_path);

      T_AUDIO_RET audio_speaker_volume ( T_AUDIO_SPEAKER_VOLUME volume,
                                         T_RV_RETURN            return_path);

      /* Event return of the MMI family API*/
      #define AUDIO_MODE_LOAD_DONE              (0x0010 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_LOAD_DONE;

      #define AUDIO_MODE_SAVE_DONE              (0x0011 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_SAVE_DONE;

      #define AUDIO_SPEAKER_VOLUME_DONE         (0x0012 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_VOLUME_DONE;


      /**********************/
      /* Full Access Family */
      /**********************/

      /* structure */
      typedef struct
      {
        UINT8   variable_indentifier;
        void    *data;
      }
      T_AUDIO_FULL_ACCESS_WRITE;

      typedef struct
      {
        UINT8   variable_indentifier;
        void    *data;
      }
      T_AUDIO_FULL_ACCESS_READ;

      /* parameter */
      #define AUDIO_PATH_USED                         (0)
      #define AUDIO_MICROPHONE_MODE                   (1)
      #define AUDIO_MICROPHONE_GAIN                   (2)
      #define AUDIO_MICROPHONE_EXTRA_GAIN             (3)
      #define AUDIO_MICROPHONE_OUTPUT_BIAS            (4)
      #define AUDIO_MICROPHONE_FIR                    (5)
      #define AUDIO_SPEAKER_MODE                      (6)
      #define AUDIO_SPEAKER_GAIN                      (7)
      #define AUDIO_SPEAKER_FILTER                    (8)
      #define AUDIO_SPEAKER_FIR                       (9)
      #define AUDIO_SPEAKER_BUZZER_STATE              (10)
      #define AUDIO_MICROPHONE_SPEAKER_LOOP_SIDETONE  (11)
      #define AUDIO_MICROPHONE_SPEAKER_LOOP_AEC       (12)
      #define AUDIO_SPEAKER_VOLUME_LEVEL              (13)


      /* prototype */
      T_AUDIO_RET audio_full_access_write (T_AUDIO_FULL_ACCESS_WRITE  *p_parameter,
                                            T_RV_RETURN                           return_path);

      T_AUDIO_RET audio_full_access_read (T_AUDIO_FULL_ACCESS_READ    *p_parameter);


      /* Event returned ofthe Full Access Family */
      #define AUDIO_FULL_ACCESS_WRITE_DONE              (0x0013 | AUDIO_EVENT_EXTERN | AUDIO_EVENTS_MASK)
      typedef struct
      {
        T_RV_HDR  os_hdr;
        INT8      status;
      } T_AUDIO_FULL_ACCESS_WRITE_DONE;

    #ifdef __cplusplus
      }
    #endif

  #endif /* __AUDIO_API_H_ */


