/*******************************************************************************
 *
 * SERIALSWITCH_CORE.H
 *
 * This module allows managing the use of the serial ports of TI GSM Evaluation
 * Boards.
 * An application may have to send several serial data flows. The board on which
 * the application is running may have one or several devices. The purpose of
 * this module is to establish connections between the serial data flows and the
 * serial devices at runtime, when the application is started.
 *
 * (C) Texas Instruments 1999 - 2003
 *
 ******************************************************************************/
/*
 * 17/12/03
 * Duplication of serialswitch.h for L1 standalone only
 *
 ******************************************************************************/

#ifndef __SERIALSWITCH_CORE_H__
#define __SERIALSWITCH_CORE_H__

#ifndef _WINDOWS
  #include "l1sw.cfg"
  #include "swconfig.cfg"
  #include "chipset.cfg"
#endif

#ifndef __MONITOR__

#if (OP_L1_STANDALONE == 0)
  #include "main/sys_types.h"
#else
  #include "sys_types.h"
#endif

#include "traceswitch.h"
#endif /* __MONITOR__ */

#ifndef C_EXTERN
  #if (OP_L1_STANDALONE)
    #define C_EXTERN  extern
  #else
    #define C_EXTERN
  #endif
#endif

/*
 * Constants used to identify the serial data flows.
 */

#define SER_FLOW_1  (0)
#define SER_FLOW_2  (1)
#define SER_FLOW_3  (2)
#define SER_FLOW_4  (3)

#define SER_PROTOCOL_STACK (SER_FLOW_1)
#define SER_LAYER_1        (SER_FLOW_2) 
#define SER_FAX_DATA       (SER_FLOW_3)
#define SER_BLUETOOTH_HCI  (SER_FLOW_4)

#define SER_HWTEST         (SER_FLOW_1)
#define SER_SERIAL_TEST_1  (SER_FLOW_3)

 
#define SER_MAX_NUMBER_OF_FLOWS  (4)

#define SER_MAX_NUMBER_OF_CFG  (16)

/*
 * Type used to define the various drivers configuration
 * available, according to the UART devices.
 */

typedef enum {
    /* Trace Flow */
    DUMMY_TRACE,             /* = 0 */
    UART_IRDA_TRACE,         /* = 1 */
    UART_MODEM_TRACE,        /* = 2 */
    #if (CHIPSET == 12)
      UART_MODEM2_TRACE,     /* = 3 */
    #endif
    /* AT-Commands/Fax & Data Flow */
    DUMMY_FAX_DATA = 4,      /* = 4 */
    /* UART IrDA F&D Driver, not supported - should be = 5 */
    UART_MODEM_FAX_DATA = 6, /* = 6 */
    #if (CHIPSET == 12)
      /* UART Modem2 F&D Driver, not supported - should be = 7 */
    #endif
    /* Bluetooth HCI Flow */
    DUMMY_BT_HCI = 8,        /* = 8 */
    UART_IRDA_BT_HCI,        /* = 9 */
    UART_MODEM_BT_HCI        /* = A */
    #if (CHIPSET == 12)
      , UART_MODEM2_BT_HCI   /* = B */
    #endif
} T_SerialDriver;

/*
 * Type used to describe a defined serial configuration;
 * Each field is a 4 bits field representing one serial flow.
 *
 * T_DefinedSerialConfig : [ flow_1 | flow_2 | flow_3 | flow_4 ]
 *                         15    12 11     8  7     4  3     0
 */

typedef struct {

    unsigned int flow_4 :4;
    unsigned int flow_3 :4;
    unsigned int flow_2 :4;
    unsigned int flow_1 :4;
    
} T_DefinedSerialConfig;


/*
 * Type used to describe all serial configuration informations
 * of a defined application:
 *     - the default configuration to set up, if the current one is
 *       not valid,
 *     - the number of allowed serial configurations,
 *     - the entire allowed serial configurations.
 */

typedef struct {

    T_DefinedSerialConfig default_config;
	SYS_UWORD8         num_config;
	T_DefinedSerialConfig allowed_config[SER_MAX_NUMBER_OF_CFG];

} T_AppliSerialInfo;


/*
 * Functions prototypes.
 */

#ifndef __MONITOR__

C_EXTERN  void SER_InitSerialConfig (T_AppliSerialInfo *serial_info);

C_EXTERN SYS_BOOL SER_UartSleepStatus (void);

C_EXTERN void SER_WakeUpUarts (void);

void SER_restart_uart_sleep_timer (void);

void SER_activate_timer_hisr (void);

C_EXTERN  void SER_tr_Init (int serial_data_flow,
                              T_tr_Baudrate baudrate,
                              void (callback_function (void)));

C_EXTERN SYS_UWORD32 SER_tr_ReadNChars (int serial_data_flow,
                                        char *buffer,
                                        SYS_UWORD32 chars_to_read);

C_EXTERN SYS_UWORD32 SER_tr_ReadNBytes (int serial_data_flow,
                                        char *buffer,
                                        SYS_UWORD32 chars_to_read,
                                        SYS_BOOL *eof_detected);

C_EXTERN SYS_UWORD32 SER_tr_WriteNChars (int serial_data_flow,
                                         char *buffer,
                                         SYS_UWORD32 chars_to_write);

C_EXTERN SYS_UWORD32 SER_tr_EncapsulateNChars (int serial_data_flow,
                                               char *buffer,
                                               SYS_UWORD32 chars_to_write);

C_EXTERN SYS_UWORD32 SER_tr_WriteNBytes (int serial_data_flow,
                                         SYS_UWORD8 *buffer,
                                         SYS_UWORD32 chars_to_write);

C_EXTERN void SER_tr_WriteChar (int serial_data_flow,
                                char character);

C_EXTERN SYS_BOOL SER_tr_EnterSleep (int serial_data_flow);

C_EXTERN void SER_tr_WakeUp (int serial_data_flow);

C_EXTERN void SER_tr_WriteString (int serial_data_flow,
                                  char *buffer);


#if ((CHIPSET == 2) || (CHIPSET == 3))
   C_EXTERN void SER_uart_handler (void);
#elif ((CHIPSET == 4) || (CHIPSET == 5) || (CHIPSET == 6) || (CHIPSET == 7) || (CHIPSET == 8) || (CHIPSET == 9) || (CHIPSET == 10) || (CHIPSET == 11) || (CHIPSET == 12))
   C_EXTERN void SER_uart_modem_handler (void);
   C_EXTERN void SER_uart_irda_handler (void);
#endif
#if (CHIPSET == 12)
   C_EXTERN void SER_uart_modem2_handler (void);
#endif

#endif /* __MONITOR__ */

#undef C_EXTERN

#endif /* __SERIALSWITCH_CORE_H__ */
