/*
 * ARMIO.C 
 *
 *
 * Control diagnostic bits
 *
 * Reference : GCS207
 *
 */


#include "l1sw.cfg"
#include "swconfig.cfg"
#ifdef BLUETOOTH_INCLUDED
#include "btemobile.cfg"
#endif
#if (OP_L1_STANDALONE == 1)
  #include "l1_macro.h"
  #include "l1_confg.h"
#endif
#include "board.cfg"
#include "chipset.cfg"

#if (OP_L1_STANDALONE == 0)
  #include "main/sys_types.h"
#else
  #include "sys_types.h"
#endif


#include "memif/mem.h"

#include "inth/iq.h"
#include "armio/armio.h"
#include "abb/abb.h"	 // for AI_Power function : to be removed, use ABB_Power_Off in abb.c file instead !!!

#if (CHIPSET != 12)
/*
 * AI_EnableBit
 *
 * Enable ARMIO input/output bit (see CLKM module specification)
 */
void AI_EnableBit(int bit)
{
  *((volatile SYS_UWORD16 *) CLKM_IO_CNTL) |= (1<<bit); 
}

/*
 * AI_DisableBit
 *
 * Disable ARMIO input/output bit (see CLKM module specification)
 */
void AI_DisableBit(int bit)
{
  *((volatile SYS_UWORD16 *) CLKM_IO_CNTL) &= ~(1<<bit); 
}

#endif /* CHIPSET != 12 */

/*
 * AI_SetBit
 *
 * Switch-on one bit
 */
void AI_SetBit(int bit)
{
   *((volatile SYS_UWORD16 *) ARMIO_OUT) |= (1<<bit); 
}

/*
 * AI_ResetBit
 *
 * Switch-off one bit
 */
void AI_ResetBit(int bit)
{
   *((volatile SYS_UWORD16 *) ARMIO_OUT) &= ~(1<<bit); 
}

/*
 * AI_ConfigBitAsOutput
 *
 * Set this bit as an output
 */
void AI_ConfigBitAsOutput(int bit)
{
   *((volatile SYS_UWORD16 *) ARMIO_IO_CNTL) &= ~(1<<bit); 
}

/*
 * AI_ConfigBitAsInput
 *
 * Set this bit as an input
 */
void AI_ConfigBitAsInput(int bit)
{
   *((volatile SYS_UWORD16 *) ARMIO_IO_CNTL) |= (1<<bit); 
}


/*
 * AI_ReadBit
 *
 * Read value in register
 */
SYS_BOOL AI_ReadBit(int bit)
{
   if ((*((volatile SYS_UWORD16 *) ARMIO_IN)) & (1<<bit))
      return (1);
   else
      return (0);
}

/*
 * AI_Power
 *
 * Switch-on or off the board
 *
 * Parameters : SYS_UWORD8 power: 1 to power-on (maintain power)
 *                                0 to power-off
 *
 */
#if (OP_L1_STANDALONE == 0)
void AI_Power(SYS_UWORD8 power)
{
  if (power == 0) 
  {
	ABB_Power_Off();
  }
}
#endif

/*
 * AI_ResetIoConfig
 *
 * Reset all default IO configurations
 *
 */
void AI_ResetIoConfig(void)
{
   *((volatile SYS_UWORD16 *) ARMIO_IO_CNTL) = 0xFFFF; // all bits are inputs
#if (CHIPSET != 12)
   *((volatile SYS_UWORD16 *) CLKM_IO_CNTL) = 0;       // default config
  #endif /* CHIPSET != 12 */
}


/*
 * AI_ClockEnable
 *
 * Enable ARMIO clock module
 *
 */
void AI_ClockEnable(void)
{
   *((volatile SYS_UWORD16 *) ARMIO_CNTL_REG) |= ARMIO_CLOCKEN;    // set to 1 bit 5
}


#if (BOARD == 7)
/*
 * AI_InitIOConfig
 *
 * Configure all GPIOs at initialization in order to optimize the power consumption 
 * of the B-Sample :
 *  - select IOs 8,9,10,11,12 and 13 on the pins instead of MCSI and MCUEN signals.
 *  - configure these IOs in output high.
 *  - configure the IOs 0 and 1 in output low.
 */
void AI_InitIOConfig(void)
{
  // reset the IOs config
  AI_ResetIoConfig();

  // CLKM_IO_CNTL register configuration :
  // select IOs 8,9,10,11,12 and 13 on the pins instead of MCSI and MCUEN signals.
  #if (CHIPSET != 12)
  AI_EnableBit(4);
  #endif

  /* Bits 5,6,7,8 are used to output I/O 9,10,11,12 or MCSI pins */
  /* If Bluetooth, IO should be disabled, outputting MCSI used for Bluetooth voice */
#ifdef BTEMOBILE
  #if (CHIPSET != 12)
  AI_DisableBit(5);
  AI_DisableBit(6);
  AI_DisableBit(7);
  AI_DisableBit(8);
  #endif
#else
  #if (CHIPSET != 12)
  AI_EnableBit(5);
  AI_EnableBit(6);
  AI_EnableBit(7);
  AI_EnableBit(8);
#endif
#endif

  #if (CHIPSET != 12)
  AI_EnableBit(9);
  #endif

  // ARMIO_OUT register configuration :
  // reset the general output latchs.
  *((volatile SYS_UWORD16 *) ARMIO_OUT) = 0x3F00;

  // ARMIO_CNTL_REG register configuration :
  // set IOs 0,1,8,9,10,11,12 and 13 as ouputs.
  AI_ConfigBitAsOutput(0);	// vibrator
  AI_ConfigBitAsOutput(1);  // LCD_A0
  AI_ConfigBitAsOutput(8);
  AI_ConfigBitAsOutput(9);
  AI_ConfigBitAsOutput(10);
  AI_ConfigBitAsOutput(11);
  AI_ConfigBitAsOutput(12);
  AI_ConfigBitAsOutput(13);
}

#elif ((BOARD == 8) || (BOARD == 9) || (BOARD == 40) || (BOARD == 41) || (BOARD == 42) || (BOARD == 43) || (BOARD == 45))
/*
 * AI_InitIOConfig
 *
 * Configure all GPIOs at initialization in order to optimize the power consumption 
 * of the C-Sample :
 *  - select IOs 8,9,10,11,12 and 13 on the pins instead of MCSI and MCUEN signals.
 *  - configure these IOs in output high.
 *  - configure the IOs 0 (Vibrator LED) and 1 (LCD_A0) in output low.
 */
void AI_InitIOConfig(void)
{
  // reset the IOs config
  AI_ResetIoConfig();

  // CLKM_IO_CNTL register configuration :
  // select IOs 6,8,9,10,11,12 and 13 on the pins instead of MCSI and MCUEN signals.
  #if (CHIPSET != 12)
  AI_EnableBit(2);
  AI_EnableBit(4);
  #endif

  /* Bits 5,6,7,8 are used to output I/O 9,10,11,12 or MCSI pins */
  /* If Bluetooth, IO should be disabled, outputting MCSI used for Bluetooth voice */
#ifdef BTEMOBILE
  #if (CHIPSET != 12)
  AI_DisableBit(5);
  AI_DisableBit(6);
  AI_DisableBit(7);
  AI_DisableBit(8);
  #endif
#else
  #if (CHIPSET != 12)
  AI_EnableBit(5);
  AI_EnableBit(6);
  AI_EnableBit(7);
  AI_EnableBit(8);
#endif
#endif

  #if (CHIPSET != 12)
  AI_EnableBit(9);
  #endif

  // ARMIO_OUT register configuration :
  // set IOs 8,9,10,11,12 and 13 as high
  // set IOs 0 to 7 as low
  #if ((BOARD == 8) || (BOARD == 9))
    *((volatile SYS_UWORD16 *) ARMIO_OUT) = 0x3F00;
    
    // ARMIO_CNTL_REG register configuration :
    // set IOs 0,1,6,8,9,10,11,12 and 13 as ouputs.
    AI_ConfigBitAsOutput(0);	
    AI_ConfigBitAsOutput(1);
    AI_ConfigBitAsOutput(6);
    AI_ConfigBitAsOutput(8);
    AI_ConfigBitAsOutput(9);
    AI_ConfigBitAsOutput(10);
    AI_ConfigBitAsOutput(11);
    AI_ConfigBitAsOutput(12);
    AI_ConfigBitAsOutput(13);
  #elif ((BOARD == 40) || (BOARD == 41) || (BOARD == 42) || (BOARD == 43) || (BOARD == 45))
    // set IOs 1 and 8 to 13 as high
    // set IOs 0 and 2 to 7 as low
    // On D-Sample GPIO 1 must be set to high to enable the audio amplifier.
    #if (OP_L1_STANDALONE == 0)
// CC test    
#if 1 // Dmitriy: GPIO 1 is the interrupt for the ext host, set it to 0
      *((volatile SYS_UWORD16 *) ARMIO_OUT) = 0x3F00;
#else
      *((volatile SYS_UWORD16 *) ARMIO_OUT) = 0x3F02;
#endif
      //*((volatile SYS_UWORD16 *) ARMIO_OUT) = 0x3F01;
// end 
    #else
      *((volatile SYS_UWORD16 *) ARMIO_OUT) = 0x3F01;
    #endif

    // ARMIO_CNTL_REG register configuration :
    // set IOs 1,2,5,7,9,14 and 15 as ouputs.
// CC test 0316
    AI_ConfigBitAsOutput(1);	
// end
    AI_ConfigBitAsOutput(2);
    AI_ConfigBitAsOutput(5);
    AI_ConfigBitAsOutput(7);
    AI_ConfigBitAsOutput(9);
    AI_ConfigBitAsOutput(14);
    AI_ConfigBitAsOutput(15);
  #endif
}

/*
 * AI_SelectIOForIT
 *
 * Select which IO will be used to generate an interrupt.
 * 'Edge' specifies if interrup must be detected on falling or rising edge.
 *
 * Warning: parameters are not checked.
 */
 
void AI_SelectIOForIT (SYS_UWORD16 Pin, SYS_UWORD16 Edge)
{
  #if (CHIPSET == 12)
    /*
     *  Update INTERRUPT_LEVEL_REG with Edge configuration on Pin selection
     */
    GPIO_INTERRUPT_LEVEL_REG = (Edge & 0x0001) << Pin;

    /*
     *  Update INTERRUPT_MASK_REG to enable interrupt generation on Pin selection
     */
    GPIO_INTERRUPT_MASK_REG = 1 << Pin;
  #else
    /*
     * Bit SET_GPIO_EVENT_MODE (bit 0) is set to enable the GPIO event mode.
     */
     
    *((volatile SYS_UWORD16 *) ARMIO_GPIO_EVENT_MODE) = (Pin << 1) + (Edge << 5) + 1;
  #endif
}

#if (CHIPSET != 12)
/*
 * AI_CheckITSource
 *
 * Check if the interrupt specified by 'Source' is active or not.
 *
 * Output: 0: IT is not active
 *         1: IT is active
 *
 * Warning: parameters are not checked.
 *
 * Warning: If the keypad and GPIO interrupts may occur the GPIO interrupt
 *          must be checked first because the GPIO status bit is reset when
 *          the register is read.
 */
 
int  AI_CheckITSource (SYS_UWORD16 Source)
{
    return (*((volatile SYS_UWORD16 *) ARMIO_KBD_GPIO_INT) & Source ? 1 : 0);
}

/*
 * AI_UnmaskIT
 *
 * Unmask the IT specified by 'Source' (keyboard or GPIO).
 *
 * Warning: parameters are not checked.
 */
 
void AI_UnmaskIT (SYS_UWORD16 Source)
{
    *((volatile SYS_UWORD16 *) ARMIO_KBD_GPIO_MASKIT) &= ~Source;
}

/*
 * AI_MaskIT
 *
 * Mask the IT specified by 'Source' (keyboard or GPIO).
 *
 * Warning: parameters are not checked.
 */
 
void AI_MaskIT (SYS_UWORD16 Source)
{
    *((volatile SYS_UWORD16 *) ARMIO_KBD_GPIO_MASKIT) |= Source;
}
#endif /* CHIPSET != 12 */

#if (CHIPSET == 12)

  void AI_MaskIT(SYS_UWORD16 d_io_number) {
    GPIO_INTERRUPT_MASK_REG |= (1 << d_io_number);
  } /* f_gpio_mask_it() */
  
  void AI_UnmaskIT(SYS_UWORD16 d_io_number) {
    GPIO_INTERRUPT_MASK_REG &= ~(1 << d_io_number);
  } /* f_gpio_unmask_it() */

#endif

#endif /* BOARD 8, 9, 40, 41, 42, 43 or 45*/
