/**
                                                                          
	@file:	r2d_inits.c	
	
	@author Christophe Favergeon                              
                                                                          
    @version	0.1

    Purpose:	Initializations routines for R2D
	
*/

/*
																			
 	Date       	Modification												
  ------------------------------------									
    03/05/2002	Create		
	    
																			
 (C) Copyright 2002 by Texas Instruments Incorporated, All Rights Reserved
*/


#include "rv/general.h"

#include "rvm/rvm_gen.h"
#include "rvm/rvm_ext_priorities.h"
#include "rvm/rvm_use_id_list.h"

#include "r2d/r2d_pool_size.h"	/* Stack & Memory Bank sizes definitions */

#include "r2d/r2d_config.h"
#include "r2d/r2d.h"
#include "r2d/r2d_i.h"
#include "r2d/r2d_env.h"
#include "r2d/r2d_independance_layer.h"
#include "r2d/r2d_messages.h"
#include "r2d/uwire.h"


#ifndef _WINDOWS
#ifndef _SIZE_T
#define _SIZE_T
typedef unsigned int size_t;
#endif
#endif

#ifndef _WINDOWS
void *memcpy(void *s1, const void *s2, register size_t n);
void *memmove(void *s1, const void *s2, size_t n);
#endif

extern T_R2D_ERROR r2d_font_init(void);
extern void r2d_font_clean(void);extern void r2d_core(UINT32 param);


/* global memory bank ID */
extern T_RVF_MB_ID r2d_mb_id;

/* Global standard graphical context */

extern T_R2D_GC_PTR r2d_g_lcd_gc;



#if (R2D_ASM == R2D_ON)
extern T_RVF_MUTEX * r2d_g_blit_mutex;
extern UINT32 r2d_g_old_foreground_pixelvalue;
extern UINT32 r2d_g_old_background_pixelvalue;
#endif

extern T_RVF_MUTEX * r2d_g_global_mutex;



#if (R2D_REFRESH == R2D_VERTICAL)
  extern UINT32 r2d_lcd_memory_words[(((R2D_WIDTH*R2D_MWHEIGHT)+1)<<R2D_LONGSIZE_FACTOR)];
#else
  extern UINT32 r2d_lcd_memory_words[(((R2D_MWWIDTH*R2D_HEIGHT)+1)<<R2D_LONGSIZE_FACTOR)];
#endif

extern INT16 r2d_update_ul_x,r2d_update_ul_y,r2d_update_br_x,r2d_update_br_y;

extern void r2d_reinit_update_region(void);

void (*IND_r2d_reinit_update_region)(void)=r2d_reinit_update_region;

extern T_R2D_FONT_DESCRIPTION *r2d_g_font_configuration;


#if (R2D_EMBEDDED_LCD == R2D_PC_COLOR_LCD)
#include "r2d/lcds/colorpc/r2d_task_init_i.c"
#endif

#if (R2D_EMBEDDED_LCD == R2D_PC_CSAMPLE)
#include "r2d/lcds/pc_csample/r2d_task_init_i.c"
#endif

#if (R2D_EMBEDDED_LCD == R2D_PC_DSAMPLE)
#include "r2d/lcds/pc_dsample/r2d_task_init_i.c"
#endif

#if (R2D_EMBEDDED_LCD == R2D_MIRRORED_LCD)
#include "r2d/lcds/simple/r2d_task_init_i.c"
#endif

#if (R2D_EMBEDDED_LCD == R2D_CUSTOMER_LCD)
#include "r2d/lcds/customer/r2d_task_init_i.c"
#endif

#if (R2D_EMBEDDED_LCD == R2D_HORIZONTAL_LCD)
#include "r2d/lcds/horizontal/r2d_task_init_i.c"
#endif

#if (R2D_EMBEDDED_LCD == R2D_SIMPLE_LCD)
#include "r2d/lcds/simple/r2d_task_init_i.c"
#endif

#if (R2D_EMBEDDED_LCD == R2D_BOARD_COLOR_LCD)
#include "r2d/lcds/colorboard/r2d_task_init_i.c"
#endif

#if (R2D_EMBEDDED_LCD == R2D_BOARD_DSAMPLE)
#include "r2d/lcds/d_sample/r2d_task_init_i.c"
#endif

#if (R2D_EMBEDDED_LCD == R2D_BOARD_BW_DSAMPLE)
#include "r2d/lcds/bw_d_sample/r2d_task_init_i.c"
#endif

UINT32 r2d_get_memory_bank_size()
{
	return(R2D_MB_PRIM_SIZE);
}

#if ((!defined DEPENDENCY) && (R2D_MB_PRIM_SIZE != R2D_MB1_SIZE))
  /* If the R2D Memory Bank sizes defined in rvf_pool_size.h differs from */
  /* the one returned by the previous function, compilation is STOPPED. */
  #error R2D Memory Bank size mismatch!
#endif



////////////////////////////////////////
//
// Software entity initialization

// Allocate the LCD framebuffer and dithering matrix
// Clear the framebuffer to 0
// (Which does not mean intensity 0 since the meaning
// of 0 is dependent on the LCD).

static void r2d_lcd_init(void)
{
   UINT16 length;
//   UINT32 *p;
    T_RVF_RET err;
   

   R2D_MALLOC(r2d_mb_id,T_R2D_FRAMEBUFFER,sizeof(T_R2D_FRAMEBUFFER),r2d_g_framebuffer);
   
   if (r2d_g_framebuffer !=NULL)
   {
   // length+1 is used instead of length because in the refresh loop
   // one may need to read one word past the end of the real datas
#if (R2D_REFRESH==R2D_VERTICAL)
   // Column is contiguous in memory
   length=R2D_WIDTH*R2D_MWHEIGHT;
   //R2D_MALLOC(r2d_mb_id,UINT32,((length+1)<<R2D_LONGSIZE_FACTOR),r2d_g_framebuffer->p_memory_words);
   r2d_g_framebuffer->p_memory_words=r2d_lcd_memory_words;
   
   // No deletion at the end since memory_word is NOT a dynamical object
   r2d_g_framebuffer->refcount=-1;

   // 0 is a special LCD kind : size of the framebuffer
   // is hardcoded to quick pixel generation
   r2d_g_framebuffer->kind=0;
   r2d_g_framebuffer->p_frame_buffer_end=r2d_g_framebuffer->p_memory_words+length;
   r2d_g_framebuffer->width=R2D_WIDTH;
   r2d_g_framebuffer->height=R2D_HEIGHT;
#else
    // Line is contiguous in memory
    length=R2D_HEIGHT*R2D_MWWIDTH;
    r2d_g_framebuffer->p_memory_words=r2d_lcd_memory_words;
	
	r2d_g_framebuffer->refcount=-1;
// 0 is a special LCD kind : size of the framebuffer
   // is hardcoded to quick pixel generation
    r2d_g_framebuffer->kind=0;    
    r2d_g_framebuffer->p_frame_buffer_end=r2d_g_framebuffer->p_memory_words+length;
    r2d_g_framebuffer->width=R2D_WIDTH;
    r2d_g_framebuffer->height=R2D_HEIGHT;
#endif

		R2D_MALLOC(r2d_mb_id,T_RVF_MUTEX,sizeof(T_RVF_MUTEX),r2d_g_framebuffer->mutex);
   if (r2d_g_framebuffer->mutex!=NULL)
	{
			  err=rvf_initialize_mutex(r2d_g_framebuffer->mutex);
			  if (err!=RVF_OK)
			  {
				 R2D_FREE(r2d_g_framebuffer->mutex);
				 R2D_FREE(r2d_g_framebuffer);
				 r2d_g_framebuffer=NULL;
				 
			  }
	}
	else
	{
				R2D_FREE(r2d_g_framebuffer);
				r2d_g_framebuffer=NULL;
				
	}

   }
    IND_r2d_reinit_update_region();


}

T_RVM_RETURN r2d_cleanup(void)
{
	/*
	** free all memory buffer previously allocated.
	** DO NOT SEND MESSAGES
	*/
	r2d_refresh_task_kill();
	#if (R2D_ASM == R2D_ON)
	  rvf_delete_mutex(r2d_g_blit_mutex);
	  R2D_FREE(r2d_g_blit_mutex);
    #endif
	rvf_delete_mutex(r2d_g_global_mutex);
	R2D_FREE(r2d_g_global_mutex);
	r2d_release_context(r2d_g_lcd_gc);
	r2d_release_framebuffer(r2d_g_framebuffer);
    R2D_FREE(r2d_g_dithering_matrix);
	R2D_FREE(r2d_g_font_configuration);
	
	return(RV_OK);
}



/******************************************************************************
* Function	  : xxx_init
*
* Description : This function is called by the RV manager to initialize the 
*				xxx SWE before creating the task and calling xxx_start. 
*
* Parameters  : None
*
* Return      : T_RVM_RETURN
* 
* History	  : 0.1 (20-August-2000)
*									
*
******************************************************************************/
T_RVM_RETURN r2d_initializations(void)
{
	/*
	** Put local SWE initialization here
	*/

	T_R2D_ERROR err;

    r2d_lcd_init();
	if (r2d_g_framebuffer==NULL)
	 goto r2d_init_error;

    r2d_dithering_init();
	if (r2d_g_dithering_matrix==NULL)
	{
		r2d_release_framebuffer(r2d_g_framebuffer);
		goto r2d_init_error;
	}

	R2D_MALLOC(r2d_mb_id,T_R2D_FONT_DESCRIPTION,
		sizeof(T_R2D_FONT_DESCRIPTION)*36,r2d_g_font_configuration);

	if (r2d_g_font_configuration ==NULL)
	{
		r2d_release_framebuffer(r2d_g_framebuffer);
        R2D_FREE(r2d_g_dithering_matrix);
		goto r2d_init_error;
    }

    err=r2d_font_init();
    if (err!=R2D_OK)
	{
		r2d_release_framebuffer(r2d_g_framebuffer);
        R2D_FREE(r2d_g_dithering_matrix);
		goto r2d_init_error;
	}


	r2d_g_default_font_framebuffer=r2d_g_font_configuration[R2D_MEDIUM_SYSTEM_NORMAL].framebuffer;
    r2d_g_default_font_metrics=r2d_g_font_configuration[R2D_MEDIUM_SYSTEM_NORMAL].metrics;
    r2d_g_default_font_table=r2d_g_font_configuration[R2D_MEDIUM_SYSTEM_NORMAL].table;

	r2d_refresh_task_init();

	r2d_g_lcd_gc=r2d_new_lcd_context(r2d_mb_id);
	if (r2d_g_lcd_gc==NULL)
	{
		r2d_font_clean();
		r2d_release_framebuffer(r2d_g_framebuffer);
        R2D_FREE(r2d_g_dithering_matrix);
		goto r2d_init_error;
	}

	r2d_g_event_was_sent=FALSE;
	r2d_g_refresh_disabled=0;

	#if (R2D_ASM == R2D_ON)

	R2D_MALLOC(r2d_mb_id,T_RVF_MUTEX,sizeof(T_RVF_MUTEX),r2d_g_blit_mutex);
    if (r2d_g_blit_mutex==NULL)
	{
		r2d_font_clean();
		r2d_release_framebuffer(r2d_g_framebuffer);
        R2D_FREE(r2d_g_dithering_matrix);
		goto r2d_init_error;
	}

	err=rvf_initialize_mutex(r2d_g_blit_mutex);
    if (err!=RVF_OK)
	{
		R2D_FREE(r2d_g_blit_mutex);
		r2d_font_clean();
		r2d_release_framebuffer(r2d_g_framebuffer);
        R2D_FREE(r2d_g_dithering_matrix);
		goto r2d_init_error;
	}

	// Mode of initial asm blit rect
	r2d_g_old_mode=R2D_COPY_MODE;
    r2d_g_old_use_foreground=FALSE;
    r2d_g_old_srcdst=R2D_LCDLCD;
	r2d_g_old_foreground_pixelvalue=((T_R2D_GC*)r2d_g_lcd_gc)->foreground_pixel_value;
	r2d_g_old_background_pixelvalue=((T_R2D_GC*)r2d_g_lcd_gc)->background_pixel_value;

#endif

	R2D_MALLOC(r2d_mb_id,T_RVF_MUTEX,sizeof(T_RVF_MUTEX),r2d_g_global_mutex);
    if (r2d_g_global_mutex==NULL)
	{
		r2d_font_clean();
		r2d_release_framebuffer(r2d_g_framebuffer);
        R2D_FREE(r2d_g_dithering_matrix);
#if (R2D_ASM == R2D_ON)
		rvf_delete_mutex(r2d_g_blit_mutex);
	    R2D_FREE(r2d_g_blit_mutex);
#endif
		goto r2d_init_error;
	}

	err=rvf_initialize_mutex(r2d_g_global_mutex);
    if (err!=RVF_OK)
	{
		R2D_FREE(r2d_g_global_mutex);
		r2d_font_clean();
		r2d_release_framebuffer(r2d_g_framebuffer);
        R2D_FREE(r2d_g_dithering_matrix);
#if (R2D_ASM == R2D_ON)
		rvf_delete_mutex(r2d_g_blit_mutex);
	    R2D_FREE(r2d_g_blit_mutex);
#endif
		goto r2d_init_error;
	}
	

	return RV_OK;
	
    
r2d_init_error:	return RV_MEMORY_ERR;
}

