/***************************************************************************
                                                                          
  Name        R2DI.h                                                   
                                                                          
  Function    Riviera 2D system : Private API                       
                                                                          
  Date        Modification                                                 
  -----------------------                                                 
  06/12/2001    Create                                                       
 **************************************************************************
  History       
                                                                          
****************************************************************************/


#ifndef __R2DI_H_
#define __R2DI_H_

#include "r2d/r2d.h"
#include "rvf/rvf_api.h"


/***************************************

     CONSTANTS

***************************************/


#if (R2D_ASM == R2D_ON)
//////////////////////////////////////////
//
// R2D ASM API
//

enum{
  R2D_LCDLCD=1,
  R2D_COLORCOLOR=2,
  R2D_COLORLCD=3,
  R2D_LCDCOLOR=4
};

#endif


/***************************************

     TYPES

***************************************/

////////////////////////////////////////
//
//  Misc
//

typedef UINT32 R2D_BOOLEAN; // To avoid veneer bug a bool on 16 bits is introduced

////////////////////////////////////////
//
// Arc region
//

// Region delimited by vectors whose scalar
// product with (sa,sb) and (ea,eb) is (+,+)
// of (-,+) or (+,-)
typedef struct R2D_ARC_REGION
{
	INT32 sa,sb,ea,eb;
    INT16 one_sector;
	BOOLEAN is_null;
} T_R2D_ARC_REGION;

////////////////////////////////////////
//
// Textures
//
typedef struct R2D_FREE_TEXTURE
{
	INT16 refcount;
	INT16 size; // Exponent of a power of 2
	T_R2D_ARGB_COLOR *pattern;
} T_R2D_FREE_TEXTURE;

typedef struct R2D_ANCHORED_TEXTURE
{
	INT16 refcount;
	INT16 size; // Exponent of a power of 2
	UINT32 *pattern;
} T_R2D_ANCHORED_TEXTURE;

///////////////////////////////////////
//
// FRAMEBUFFER
//


typedef struct R2D_FRAMEBUFFER
{
   // Refcounting
     INT16 refcount;
   // List of memory words containing the pixels
     UINT32 *p_memory_words;
   // End of frameBuffer used to prevent out of
   // framebuffer access
   // (It is the last word contained in the framebuffer)
   // (but that word is used for loops and is never drawn)
     UINT32 *p_frame_buffer_end;
   // Kind of framebuffer (LCD or 24 bits)
     T_R2D_FRAMEBUFFER_KIND kind;
   // Dimensions of frameBuffer
     UINT16 width,height;
   // Semaphore for concurrent accesses to the framebuffer
	 T_RVF_MUTEX * mutex;
} T_R2D_FRAMEBUFFER;

// Font cache framebuffer has other fields required to build
// the glyph from a succession of Unicode codes
typedef struct R2D_FONT_CACHE_FRAMEBUFFER
{
   // Refcounting
     INT16 refcount;
   // List of memory words containing the pixels
     UINT32 *p_memory_words;
   // End of frameBuffer used to prevent out of
   // framebuffer access
   // (It is the last word contained in the framebuffer)
     UINT32 *p_frame_buffer_end;
   // Kind of framebuffer (LCD or 24 bits)
     T_R2D_FRAMEBUFFER_KIND kind;
   // Dimensions of frameBuffer
     UINT16 width,height;
   // Semaphore for concurrent accesses to the framebuffer
	 T_RVF_MUTEX * mutex;
} T_R2D_FONT_CACHE_FRAMEBUFFER;


////////////////////////////////////////
//
// LCD API
//

typedef UINT32 (*T_R2D_DRAWING_OP)(UINT32 old,UINT32 value);


typedef T_R2D_DRAWING_OP T_R2D_DRAWING_OPERATORS[10];

////////////////////////////////////////
//
// OBJECTS
//

// Metaclass object describing the classes


typedef struct R2D_CLASS_SHAPE T_R2D_CLASS_SHAPE;
typedef struct R2D_SHAPE T_R2D_SHAPE;

typedef void (*T_R2D_shape_release_funct)(T_R2D_SHAPE *self);
typedef void (*T_R2D_shape_draw_funct)(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
typedef void (*T_R2D_shape_fill_funct)(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
typedef T_R2D_SHAPE_PTR (*T_R2D_shape_clone_funct)(T_RVF_MB_ID bank,T_R2D_SHAPE *self);
typedef void (*T_R2D_shape_translate_funct)(T_R2D_SHAPE *self,INT16 dx,INT16 dy);
typedef BOOLEAN (*T_R2D_shape_point_testing_funct)(T_R2D_SHAPE *self,INT16 x,INT16 y);


struct R2D_CLASS_SHAPE
{
  void *parent_class;
  T_R2D_shape_release_funct release;
  T_R2D_shape_draw_funct draw;
  T_R2D_shape_fill_funct fill;
  T_R2D_shape_clone_funct clone;
  T_R2D_shape_translate_funct translate;
  T_R2D_shape_point_testing_funct point_in_shape;
};

////////////////////////////////////////
//
// SHAPES OBJECTS
//

struct R2D_SHAPE
{
  INT16 refcount;
  const T_R2D_CLASS_SHAPE *p_r2d_class;
};

typedef struct R2D_RECT
{
  INT16 refcount;
  const T_R2D_CLASS_SHAPE *p_r2d_class;
  INT16 ul_x,ul_y,br_x,br_y;
} T_R2D_RECTANGLE;

typedef struct R2D_CIRCLE
{
  INT16 refcount;
  const T_R2D_CLASS_SHAPE *p_r2d_class;
  INT16 x,y,r;
} T_R2D_CIRCLE;

typedef T_R2D_RECTANGLE T_R2D_ELLIPSE;

typedef struct R2D_ROUNDRECT
{
  INT16 refcount;
  const T_R2D_CLASS_SHAPE *p_r2d_class;
  INT16 ul_x,ul_y,br_x,br_y;
  INT16 h,v;
} T_R2D_ROUND_RECTANGLE;

typedef struct R2D_ARC
{
  INT16 refcount;
  const T_R2D_CLASS_SHAPE *p_r2d_class;
  INT16 ul_x,ul_y,br_x,br_y;
  INT16 start_angle,stop_angle;
} T_R2D_ARC;

typedef struct R2D_TEXT
{
  INT16 refcount;
  const T_R2D_CLASS_SHAPE *p_r2d_class;
  INT16 x,y;
  T_R2D_UTF16 *p_text;
} T_R2D_TEXT;


////////////////////////////////////////
//
// GRAPHIC CONTEXT
//

typedef struct R2D_GC
{
   // Refcounting
     INT16 refcount;
   // The framebuffer
     T_R2D_FRAMEBUFFER_PTR p_frame_buffer;
   // Clip rectangle (in graphic context coordinates)
     T_R2D_SHAPE *p_clipping_shape;
   // Clipping shape in scanning coordinates
	 T_R2D_SHAPE *p_s_clipping_shape;
   // Foreground pixel value
   //  (on 24 bits per pixel. Conversion done on the fly
   //  according to the kind of framebuffer)
     T_R2D_ARGB_COLOR foreground_color;
   // Cache for computed pixel value corresponding
   // to previous color. Used for LCD with dithering off
   // to speed drawing
     UINT32 foreground_pixel_value;
   // Cache for foreground dithering used when dithering ON
     UINT32 *p_foreground_dithered_cache;
   // Background pixel value
   //  (on 24 bits per pixel. Conversion done on the fly
   //  according to the kind of framebuffer)
     T_R2D_ARGB_COLOR background_color;
   // Cache for computed pixel value corresponding
   // to previous color. Used for LCD with dithering off
   // to speed drawing
     UINT32 background_pixel_value;
   // Cache for dithering used when dithering ON
     UINT32 *p_background_dithered_cache;
   // Background texture
	 T_R2D_ANCHORED_TEXTURE *background_texture;
   // Pen position
     INT16 pen_x,pen_y;
   // pen_size in pixels
     INT16 pen_size;
   // Drawing mode
     INT16 drawing_mode;
   // Drawing operator for above drawing mode
	 T_R2D_DRAWING_OP drawing_op;
   // Context origin (H coordinates of upper-left corner of screen)
     INT16 org_x,org_y;
   // Context origin (S coordinates for texture of the point corresponding
   //                 to (0,0) H coordinates)
	 INT16 s_org_x,s_org_y;
   // Dash state
     BOOLEAN dash;
   // Text face
     T_R2D_TEXT_FACE text_face;
   // Text style
   //  (List of types requested by the user)
     T_R2D_TEXT_STYLE text_style;
   // Internal text style
   //  (list of style effect which must be computed since
   //  there are not embedded into the fonts)
     T_R2D_TEXT_STYLE internal_text_style;
   // Text size
     T_R2D_TEXT_SIZE text_size;
   // Font framebuffer
     T_R2D_FRAMEBUFFER *font_frame_buffer;
   // Font metrics pointer
     INT32 *font_metrics;
   // Font transcoding table
	 INT32 *font_table;
   // Script mode (left to right by defaulr)
	 T_R2D_SCRIPT_MODE script_mode;
} T_R2D_GC;

////////////////////////////////////////
//
// FONT CONFIG
//

typedef INT32 *T_R2D_FONT_METRICS;
typedef INT32 *T_R2D_FONT_TABLE;

typedef struct R2D_FONT_DESCRIPTION
{
	T_R2D_FONT_TABLE table;
	T_R2D_FONT_METRICS metrics;
	T_R2D_FRAMEBUFFER_PTR framebuffer;
	BOOLEAN compute_bold;
} T_R2D_FONT_DESCRIPTION;


/***************************************

     FUNCTIONS

***************************************/

////////////////////////////////////////
//
//    TOOL
//    MACROS
//

// One should never call following function directly
// because when refcount is reaching 0 an object
// has generally subobject torelease before releasing
// itself

// Decrement refcount and autodelete if required
#define r2d_release(p) { \
    if (p) \
    { \
      if ((((T_R2D_REFCOUNT*)(p))->refcount)>0) { \
       ((T_R2D_REFCOUNT*)(p))->refcount--; \
       if ((((T_R2D_REFCOUNT*)(p))->refcount==0)) \
          R2D_FREE(p); \
	  } \
    } \
   }


// Return refcount of an object
#define R2D_REFCOUNT(p) (((T_R2D_REFCOUNT*)(p))->refcount)

// Compute dithered value
// lightness is either the gray intensity (for monochrome)
// or the component intensity.
// It is coded on at most 8 bits
#if (R2D_DITHERING == R2D_ON)

  #define r2d_dithering_group_and_lightness() \
  lightness = lightness >> (8 - R2D_DITHERING_DIMENSION - R2D_PIXEL_DEPTH); \
        dithering_group=lightness >> R2D_DITHERING_DIMENSION; \
 \
        lightness=lightness & R2D_DITHERED_MASK;

  #define r2d_dithered_value(x,y) \
  \
        dvalue=dithering_group; \
        if (lightness>=r2d_get_dithering_matrix_entry(r2d_g_dithering_matrix,x,y)) \
          dvalue+=1; \
\
        if (dvalue) \
          dvalue--;
#else
  #define r2d_dithering_group_and_lightness()
  #define r2d_dithered_value(x,y)
#endif

extern INT32 r2d_ptree_find(INT32 *tree,INT32 unicode);

void r2d_arc_region_x_reflect(T_R2D_ARC_REGION *rgn);
void r2d_arc_region_y_reflect(T_R2D_ARC_REGION *rgn);
void r2d_arc_region_diagonal_reflect(T_R2D_ARC_REGION *rgn);
BOOLEAN r2d_check_is_in_arc_region(INT16 x,INT16 y,T_R2D_ARC_REGION *rgn);
INT32 r2d_get_point_side(INT16 x,INT16 y,INT32 sa,INT32 sb);

////////////////////////////////////////
//
//    LCD
//    FUNCTIONS
//

extern void r2d_convert_foreground_color(T_R2D_GC *gc,UINT32 color);
extern void r2d_convert_background_color(T_R2D_GC *gc,UINT32 color);
extern BOOLEAN r2d_lcd_foreground_pixel(UINT32 lcd_value,T_R2D_GC_PTR src_gc);
extern UINT32 r2d_color_to_lcd(UINT32 pixel_value,INT16 x,INT16 y);
extern UINT32 r2d_lcd_to_color(UINT32 color);

////////////////////////////////////////
//
//    SHAPE LOW LEVEL
//    FUNCTIONS
//

void r2d_release_rectangle(T_R2D_SHAPE *self);
void r2d_draw_rectangle_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
void r2d_fill_rectangle_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
T_R2D_SHAPE_PTR r2d_clone_rectangle_shape(T_RVF_MB_ID bank,T_R2D_SHAPE *self);

void r2d_release_circle(T_R2D_SHAPE *self);
void r2d_draw_circle_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
void r2d_fill_circle_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
T_R2D_SHAPE_PTR r2d_clone_circle_shape(T_RVF_MB_ID bank,T_R2D_SHAPE *self);

void r2d_release_ellipse(T_R2D_SHAPE *self);
void r2d_draw_ellipse_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
void r2d_fill_ellipse_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);

void r2d_release_round_rectangle(T_R2D_SHAPE *self);
void r2d_draw_round_rectangle_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
void r2d_fill_round_rectangle_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
T_R2D_SHAPE_PTR r2d_clone_round_rectangle_shape(T_RVF_MB_ID bank,T_R2D_SHAPE *self);

void r2d_release_arc(T_R2D_SHAPE *self);
void r2d_draw_arc_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
void r2d_fill_arc_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
T_R2D_SHAPE_PTR r2d_clone_arc_shape(T_RVF_MB_ID bank,T_R2D_SHAPE *self);

void r2d_release_text(T_R2D_SHAPE *self);
void r2d_draw_text_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
void r2d_fill_text_shape(T_R2D_SHAPE *self,T_R2D_GC_PTR gc);
T_R2D_SHAPE_PTR r2d_clone_text_shape(T_RVF_MB_ID bank,T_R2D_SHAPE *self);

void r2d_translate_rectangle_shape(T_R2D_SHAPE *self,INT16 dx,INT16 dy);
void r2d_translate_point_shape(T_R2D_SHAPE *self,INT16 dx,INT16 dy);

BOOLEAN r2d_point_in_rectangle_shape(T_R2D_SHAPE *self,INT16 x,INT16 y);
BOOLEAN r2d_point_in_circle_shape(T_R2D_SHAPE *self,INT16 x,INT16 y);
BOOLEAN r2d_point_in_ellipse_shape(T_R2D_SHAPE *self,INT16 x,INT16 y);
BOOLEAN r2d_point_in_round_rectangle_shape(T_R2D_SHAPE *self,INT16 x,INT16 y);
BOOLEAN r2d_point_in_not_supported_shape(T_R2D_SHAPE *self,INT16 x,INT16 y);

////////////////////////////////////////
//
//    FRAMEBUFFER ACCESS
//    FUNCTIONS
//



// Write a pixel at position x,y with graphic context gc
// pixel_value must be compatible with the framebuffer owned
// by the graphic context. No color conversion is done
// here
extern void r2d_write_lcd_pixel(T_R2D_GC *gc,INT16 x,INT16 y,UINT32 pixel_value);
extern void r2d_write_lcd_line(T_R2D_GC* gc,INT16 x,INT16 y,INT16 nb,R2D_BOOLEAN
background);
extern void r2d_arc_write_lcd_line(T_R2D_GC* gc,INT16 x,INT16 y,INT16 org_x,INT16 org_y,INT16 nb,
								   T_R2D_ARC_REGION *rgn,R2D_BOOLEAN
background);


extern T_R2D_DRAWING_OP r2d_get_drawing_op(T_R2D_GC *gc,T_R2D_DRAWING_MODE mode);
#if (R2D_ASM == R2D_ON)
extern UINT32 * r2d_get_asm_drawing_op(T_R2D_GC *gc,T_R2D_DRAWING_MODE mode);

#endif

////////////////////////////////////////
//
//    COLOR
//    FUNCTIONS
//

// color conversion : on the fly dithering and color scaling
// with final conversion to the format used by the LCD if needed
// The dithered matrix or the pixel_value are cached into the gc
// structure
extern void r2d_convert_foreground_color(T_R2D_GC *gc,UINT32 color);
extern void r2d_convert_background_color(T_R2D_GC *gc,UINT32 color);

// Get value at dithering matrix position
#define r2d_get_dithering_matrix_entry(matrix,x,y) \
 *(matrix+(x&1)+(y&1)*2)

// Set element (x,y) of dithering or dithered matrix
extern void r2d_set_dithering_matrix_entry(UINT32 *matrix,UINT32 elem,INT16 x,INT16 y);

extern void r2d_convert_foreground_color_color(T_R2D_GC *gc,UINT32 color);
extern void r2d_convert_background_color_color(T_R2D_GC *gc,UINT32 color);

extern UINT16 r2d_max_color(UINT16 red,UINT16 green, UINT16 blue);
// Get min component
extern UINT16 r2d_min_color(UINT16 red,UINT16 green, UINT16 blue);

extern UINT32 r2d_get_color_pixel_value(T_R2D_GC *gc,INT16 x,INT16 y);

////////////////////////////////////////
//
//    TEXT
//    FUNCTIONS
//

#define r2d_get_char_x(p) (((INT32*)p)[0])

////////////////////////////////////////
//
//    COPY
//    FUNCTIONS
//

#if (R2D_ASM == R2D_OFF)

void           r2d_blit_lcd_to_lcd(T_R2D_GC_PTR src_gc,T_R2D_GC_PTR dst_gc,
                 T_R2D_SHAPE_PTR src_rectangle, T_R2D_SHAPE_PTR dst_rectangle,
				 R2D_BOOLEAN use_foreground_color);

void           r2d_blit_color_to_color(T_R2D_GC_PTR src_gc,T_R2D_GC_PTR dst_gc,
                 T_R2D_SHAPE_PTR src_rectangle, T_R2D_SHAPE_PTR dst_rectangle,
				 R2D_BOOLEAN use_foreground_color);

void           r2d_blit_lcd_to_color(T_R2D_GC_PTR src_gc,T_R2D_GC_PTR dst_gc,
                 T_R2D_SHAPE_PTR src_rectangle, T_R2D_SHAPE_PTR dst_rectangle,
				 R2D_BOOLEAN use_foreground_color);

void           r2d_blit_color_to_lcd(T_R2D_GC_PTR src_gc,T_R2D_GC_PTR dst_gc,
                 T_R2D_SHAPE_PTR src_rectangle, T_R2D_SHAPE_PTR dst_rectangle,
				 R2D_BOOLEAN use_foreground_color);
#else

void           r2d_blit_asm(T_R2D_GC_PTR src_gc,T_R2D_GC_PTR dst_gc,
                             T_R2D_SHAPE_PTR src_rectangle, 
							 T_R2D_SHAPE_PTR dst_rectangle,
							 R2D_BOOLEAN use_foreground_color,INT32 srcdst);
#endif


/***************************
  
	 DEFINES

***************************/

/* FOR SEARCH INSIDE PATRICIA TREE */
#define R2D_PTREE_TYPE 3
#define R2D_PTREE_BRANCHING 0
#define R2D_PTREE_PREFIX 1
#define R2D_PTREE_UNICODE_BLOCK 0
#define R2D_PTREE_GLYPH_INDEX 1
#define R2D_PTREE_CONVERSION 2
#define R2D_PTREE_LEFT 2
#define R2D_PTREE_RIGHT 3
#define R2D_PTREE_CONVERSION 2
#define R2D_PTREE_T_LEAF 0
#define R2D_PTREE_SHIFT 4

/* FOR FONT DESCRIPTION */

#define R2D_SMALL_SYSTEM_NORMAL 0
#define R2D_SMALL_SYSTEM_BOLD 1
#define R2D_SMALL_SYSTEM_ITALIC 2
#define R2D_SMALL_SYSTEM_BOLD_ITALIC 3

#define R2D_MEDIUM_SYSTEM_NORMAL 4
#define R2D_MEDIUM_SYSTEM_BOLD 5
#define R2D_MEDIUM_SYSTEM_ITALIC 6
#define R2D_MEDIUM_SYSTEM_BOLD_ITALIC 7

#define R2D_BIG_SYSTEM_NORMAL 8
#define R2D_BIG_SYSTEM_BOLD 9
#define R2D_BIG_SYSTEM_ITALIC 10
#define R2D_BIG_SYSTEM_BOLD_ITALIC 11

#define R2D_SMALL_TYPEWRITER_NORMAL 12
#define R2D_SMALL_TYPEWRITER_BOLD 13
#define R2D_SMALL_TYPEWRITER_ITALIC 14
#define R2D_SMALL_TYPEWRITER_BOLD_ITALIC 15

#define R2D_MEDIUM_TYPEWRITER_NORMAL 16
#define R2D_MEDIUM_TYPEWRITER_BOLD 17
#define R2D_MEDIUM_TYPEWRITER_ITALIC 18
#define R2D_MEDIUM_TYPEWRITER_BOLD_ITALIC 19

#define R2D_BIG_TYPEWRITER_NORMAL 20
#define R2D_BIG_TYPEWRITER_BOLD 21
#define R2D_BIG_TYPEWRITER_ITALIC 22
#define R2D_BIG_TYPEWRITER_BOLD_ITALIC 23

#define R2D_SMALL_PROPORTIONAL_NORMAL 24
#define R2D_SMALL_PROPORTIONAL_BOLD 25
#define R2D_SMALL_PROPORTIONAL_ITALIC 26
#define R2D_SMALL_PROPORTIONAL_BOLD_ITALIC 27

#define R2D_MEDIUM_PROPORTIONAL_NORMAL 28
#define R2D_MEDIUM_PROPORTIONAL_BOLD 29
#define R2D_MEDIUM_PROPORTIONAL_ITALIC 30
#define R2D_MEDIUM_PROPORTIONAL_BOLD_ITALIC 31

#define R2D_BIG_PROPORTIONAL_NORMAL 32
#define R2D_BIG_PROPORTIONAL_BOLD 33
#define R2D_BIG_PROPORTIONAL_ITALIC 34
#define R2D_BIG_PROPORTIONAL_BOLD_ITALIC 35


/***************************

     GLOBALS

****************************/

// and for reentrancy they should be read only if possible
// or protected by a semaphore otherwise

#if (R2D_ASM == R2D_ON)
extern T_RVF_MUTEX * r2d_g_blit_mutex;
extern T_R2D_DRAWING_MODE r2d_g_old_mode;
extern R2D_BOOLEAN r2d_g_old_use_foreground;
extern INT32 r2d_g_old_srcdst;
#endif

// Global mutex for access to global variables
extern T_RVF_MUTEX * r2d_g_global_mutex;


// Drawing operators (C version)
extern const T_R2D_DRAWING_OPERATORS r2d_g_lcd_operators,r2d_g_color_operators;

extern UINT32 *r2d_g_asm_lcd_operators[],*r2d_g_asm_color_operators[];

// LCD framebuffer
extern T_R2D_FRAMEBUFFER* r2d_g_framebuffer;

// For fonts 

extern T_R2D_FONT_DESCRIPTION *r2d_g_font_configuration;

// (default pointers for initialization of graphical context)
extern T_R2D_FRAMEBUFFER *r2d_g_default_font_framebuffer;
extern INT32 *r2d_g_default_font_metrics;
extern INT32 *r2d_g_default_font_table;


// Dithering matrix
extern UINT32 *r2d_g_dithering_matrix;

// Class globals
extern const T_R2D_CLASS_SHAPE r2d_class_rectangle;
extern const T_R2D_CLASS_SHAPE r2d_class_circle;
extern const T_R2D_CLASS_SHAPE r2d_class_ellipse;
extern const T_R2D_CLASS_SHAPE r2d_class_round_rectangle;
extern const T_R2D_CLASS_SHAPE r2d_class_arc;
extern const T_R2D_CLASS_SHAPE r2d_class_text;

// Memory bank id
extern T_RVF_MB_ID r2d_mb_id;

// Global LCD graphical context
extern T_R2D_GC_PTR r2d_g_lcd_gc;

// Global flag to check if event required
extern BOOLEAN r2d_g_event_was_sent;

// Global flag to disable refresh
extern INT16 r2d_g_refresh_disabled;

/* global addr id */
extern T_RVF_ADDR_ID r2d_addr_id;


#endif

