/********************************************************************************
*																				*
*   lcd_messages.c	: contains functions used by Upper layer to send messages	*
*					  to LCD mailbox, and by LCD to send events to Upper layer  *
*																				*
*	Project			: 															*
*		                                            					    	*
*   Author         	: Davide Carpegna 				    						*
*   Version number	: 1.0														*
*   Date        	: 29 September 2000											*		
*																				*
*   (C) Copyright 2000 by Texas Instruments Incorporated, All Rights Reserved	*		
*********************************************************************************/

#include <string.h>

#include "r2d/lcd_messages.h"
#include "rvm/rvm_use_id_list.h"

/* Global variables */
typedef struct DRIVER_GBL_INFO
{
	T_RVF_MB_ID		prim_id;
	T_RVF_ADDR_ID	addr_id;
	UINT8		rtc_mailbox;
} T_DRIVER_GBL_INFO;

T_DRIVER_GBL_INFO * DRIVER_GBL_INFO_PTR = NULL;

UINT8 DrvTaskReady = 0;


/********************************************************************************/
/*    Function Name: LCD_Reset													*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_reset message into lcd mailbox			*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:	None													*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_reset(void)
{

		/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);

    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

		return (RV_OK);
}

/********************************************************************************/
/*    Function Name: lcd_clear													*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_clear message into lcd mailbox			*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:	None													*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET.                                                             */
/********************************************************************************/


T_RV_RET lcd_clear(void)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_CLEAR				*msg_ptr;
	

		/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);


	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_CLEAR ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
	
		msg_ptr->os_hdr.msg_id = LCD_CLEAR_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
	{	
		rvf_send_trace("LCD_Clear function FAILED ",26, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}

}

/********************************************************************************/
/*    Function Name: LCD_Init													*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_init message into lcd mailbox			*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:	None					                                */
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_init(void)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_INIT				*msg_ptr;
	
		/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
	{	LCD_Init_ll();
		return (RV_NOT_READY);
	}
    if (DrvTaskReady == 0)
	{	LCD_Init_ll();
	    return(RV_NOT_READY);
	}


	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_INIT ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		
		
		msg_ptr->os_hdr.msg_id = LCD_INIT_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);				

		return (RV_OK);
	}
	else 	
	{	rvf_send_trace("LCD_Init function FAILED ",25, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_WriteString		                                    */
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_write_string message into lcd mailbox	*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:	row, column, string, mode                               */
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_write_string(UINT8 row, UINT8 column, char *string, T_VIDEO_MODE mode)
{

	UINT8					length;
	T_RVF_MB_STATUS			mb_status;
    T_LCD_WRITE_STRING      *msg_ptr;
	
	/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
	{
		return (RV_NOT_READY);
	}
    if (DrvTaskReady == 0)
	{
		return (RV_NOT_READY);
	}

	/* the string must fit into one row, so the maximum string length is 14		*/
	/* character starting from column 0											*/
	length = (UINT8) strlen (string);
	if((length*6+column)>DISP_PIXELWIDTH)
	{
		rvf_send_trace("LCD_WriteString - too long string ",34, NULL_PARAM,RV_TRACE_LEVEL_WARNING, R2D_USE_ID );	
		return (RV_INVALID_PARAMETER);
	}

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_WRITE_STRING ) + length, (void **) &msg_ptr);
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->row = row;
		msg_ptr->column = column * 6;
		msg_ptr->mode = mode;

		/* the first byte indicates the string length ('\x00' excepted), while 	*/
		/* the message is included beyond this parameter						*/
		msg_ptr->string = length;
		memcpy (&(msg_ptr->string) + 1,
				string,
				length);
		
		msg_ptr->os_hdr.msg_id = LCD_WRITE_STRING_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
		{	
		rvf_send_trace("LCD_WriteString function FAILED ",32, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}

}


/********************************************************************************/
/*    Function Name: LCD_WriteCenter		                                    */
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_write_center message into lcd mailbox	*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:	string	                                                */
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_write_center( char *string)
{

	UINT8					length;
	T_RVF_MB_STATUS			mb_status;
    T_LCD_WRITE_CENTER      *msg_ptr;
	
	/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
	{
		return (RV_NOT_READY);
	}
	
    if (DrvTaskReady == 0)
	{
	    return(RV_NOT_READY);
	}

	/* the string must fit into one row, so the maximum string length is 14		*/
	/* character starting from column 0											*/
	length = (UINT8) strlen (string);
	if((length*6+X_CENTER-(length/2)*DISP_PAGEHEIGHT)>DISP_PIXELWIDTH)
	{
		rvf_send_trace("LCD_WriteCenter - too long string ",34, NULL_PARAM,RV_TRACE_LEVEL_WARNING, R2D_USE_ID );	
		return (RV_INVALID_PARAMETER);
	}

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_WRITE_CENTER ) + length, (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{

		/* the first byte indicates the length ('\x00' excepted), while the		*/
		/* message is included beyond this parameter							*/
		msg_ptr->string = length;
		memcpy (&(msg_ptr->string) + 1,
				string,
				length);
		
		msg_ptr->os_hdr.msg_id = LCD_WRITE_CENTER_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
	{	
		rvf_send_trace("LCD_WriteCenter function FAILED ",32, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_ClearString		                                    */
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_clear_strring message into lcd mailbox	*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:	row					                                    */
/*    Output Parameters: 				            							*/
/*        T_RV_RET.                                                             */
/********************************************************************************/


T_RV_RET lcd_clear_string(UINT8 row)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_CLEAR_STRING      *msg_ptr;
	
	/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);


	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_CLEAR_STRING ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->row = row;
	
		
		msg_ptr->os_hdr.msg_id = LCD_CLEAR_STRING_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
			{	
		rvf_send_trace("LCD_ClearString function FAILED ",32, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_ClearWord			                                    */
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_clear_word message into lcd mailbox	*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:	row, column, length                                     */
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_clear_word(UINT8 row, UINT8 column, UINT16 length)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_CLEAR_WORD      *msg_ptr;
	
	 /* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);
	

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_CLEAR_WORD ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->row = row;
		msg_ptr->column = column * 6;
		msg_ptr->length = length;
		
		msg_ptr->os_hdr.msg_id = LCD_CLEAR_WORD_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
			{	
		rvf_send_trace("LCD_ClearWord function FAILED ",31, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}


/********************************************************************************/
/*    Function Name: LCD_PutPixel												*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_put_pixel message into lcd mailbox		*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:x,y														*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_put_pixel(UINT8 x, UINT8 y)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_PUT_PIXEL      *msg_ptr;
	
	  /* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_PUT_PIXEL ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->x = x;
		msg_ptr->y = y;
		
		msg_ptr->os_hdr.msg_id = LCD_PUT_PIXEL_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
			{	
		rvf_send_trace("LCD_PutPixel function FAILED ",29, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_ClearPixel												*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_clear_pixel message into lcd mailbox	*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:x,y														*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_clear_pixel(UINT8 x, UINT8 y)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_CLEAR_PIXEL      *msg_ptr;
	
	  /* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_CLEAR_PIXEL ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->x = x;
		msg_ptr->y = y;
		
		msg_ptr->os_hdr.msg_id = LCD_CLEAR_PIXEL_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
			{	
		rvf_send_trace("LCD_ClearPixel function FAILED ",31, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_DrawLine										    	*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_draw_line message into lcd mailbox		*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:x1,y1, x2,y2												*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_draw_line(UINT8 x1, UINT8 y1, UINT8 x2, UINT8 y2)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_DRAW_LINE	      *msg_ptr;
	
	  /* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_DRAW_LINE ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->x1 = x1;
		msg_ptr->y1 = y1;
		msg_ptr->x2 = x2;
		msg_ptr->y2 = y2;

		msg_ptr->os_hdr.msg_id = LCD_DRAW_LINE_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
			{	
		rvf_send_trace("LCD_DrawLine function FAILED ",29, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}
/********************************************************************************/
/*    Function Name: LCD_ClearLine												*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_clear_line message into lcd mailbox	*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:x1,y1, x2,y2												*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_clear_line(UINT8 x1, UINT8 y1, UINT8 x2, UINT8 y2)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_CLEAR_LINE	      *msg_ptr;
	
	  /* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_CLEAR_LINE ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->x1 = x1;
		msg_ptr->y1 = y1;
		msg_ptr->x2 = x2;
		msg_ptr->y2 = y2;

		msg_ptr->os_hdr.msg_id = LCD_CLEAR_LINE_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
			{	
		rvf_send_trace("LCD_ClearLine function FAILED ",30, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_DrawRectangle											*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_draw_rectangle message into lcd mailbox*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:x1,y1, x2,y2												*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_draw_rectangle(UINT8 x1, UINT8 y1, UINT8 x2, UINT8 y2)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_DRAW_RECTANGLE	      *msg_ptr;
	
	  /* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_DRAW_RECTANGLE ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->x1 = x1;
		msg_ptr->y1 = y1;
		msg_ptr->x2 = x2;
		msg_ptr->y2 = y2;

		msg_ptr->os_hdr.msg_id = LCD_DRAW_RECTANGLE_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
			{	
		rvf_send_trace("LCD_DrawRectangle function FAILED ",34, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}
/********************************************************************************/
/*    Function Name: LCD_ClearRectangle											*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_clear_rectangle message into lcd mailbox*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:x1,y1, x2,y2												*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_clear_rectangle(UINT8 x1, UINT8 y1, UINT8 x2, UINT8 y2)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_CLEAR_RECTANGLE	      *msg_ptr;
	
	 /* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_CLEAR_RECTANGLE ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->x1 = x1;
		msg_ptr->y1 = y1;
		msg_ptr->x2 = x2;
		msg_ptr->y2 = y2;

		msg_ptr->os_hdr.msg_id = LCD_CLEAR_RECTANGLE_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
			{	
		rvf_send_trace("LCD_ClearRectangle function FAILED ",35, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_DrawIcon						    					*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_display_icon message into lcd mailbox	*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:x, y, icon_id											*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_draw_icon(UINT8 x,UINT8 y,UINT8 icon_id)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_DRAW_ICON			*msg_ptr;
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

	/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
	{	
		return (RV_NOT_READY);
	}

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_DRAW_ICON ), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->icon_id = icon_id;
		msg_ptr->x = x;
		msg_ptr->y = y;

		msg_ptr->os_hdr.msg_id = LCD_DRAW_ICON_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
			{	
		rvf_send_trace("LCD_DrawIcon function FAILED ",29, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_RefreshBloc								    		*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_draw_bloc message into lcd mailbox		*/	
/*																				*/		
/*                                                                              */
/*    Input Parameters:table,x_dim,y_dim,x_pos,y_pos							*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_refresh_bloc(char *table,UINT8 x_dim,UINT8 y_dim,UINT8 x_pos,UINT8 y_pos)
{

	T_RVF_MB_STATUS			mb_status;
    T_LCD_REFRESH_BLOC	      *msg_ptr;
	

	/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);

    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_REFRESH_BLOC), (void **) &msg_ptr);	

	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		
		msg_ptr->table = table;
		msg_ptr->x_dim = x_dim;
		msg_ptr->y_dim = y_dim;
		msg_ptr->x_pos = x_pos;
		msg_ptr->y_pos = y_pos;

		msg_ptr->os_hdr.msg_id = LCD_REFRESH_BLOC_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
	{	
		rvf_send_trace("LCD_RefreshBloc function FAILED ",32, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_PixBlt													*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_refresh_bloc message into lcd mailbox	*/	
/*																				*/		
/*                                                                              */
/*    Input Parameters:table,x_dim,y_dim,x_pos,y_pos,table						*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_pix_blt(int x_pos,int y_pos,int x_dim,int y_dim,char *table)
{
	T_RVF_MB_STATUS			mb_status1,mb_status2;
    T_LCD_DRAW_BLOC	    	*msg_ptr;
	int  					i;
	char					*table_to_send;  //pointer to the table created to copy the transmitted data
	

	 /* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);
	
    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);
	
	if((x_pos>83)||(y_pos>48))
	{
		rvf_send_trace("LCD - image outside lcd",23, NULL_PARAM,RV_TRACE_LEVEL_WARNING, R2D_USE_ID );	
		return(RV_OK);
	}
     
	mb_status1 = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_DRAW_BLOC), (void **) &msg_ptr);	
	
	if((mb_status1==RVF_RED))//memory allocation failed
	{	
		rvf_send_trace("LCD_PixBlt function FAILED ",27, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
    }

	//dynamic allocation of the copy table
	mb_status2 = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, (x_dim*y_dim)/8, (void **) &table_to_send);

   	if((mb_status2==RVF_RED))//memory allocation failed
	  {	
		rvf_free_buf(msg_ptr);
		rvf_send_trace("LCD_PixBlt function FAILED ",27, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
      }

	else 
	{  
		for(i=0;i<((x_dim*y_dim/8)-1);i++)
			table_to_send[i]=table[i];
	
		msg_ptr->table = table_to_send;
		msg_ptr->x_dim = x_dim;
		msg_ptr->y_dim = y_dim;
		msg_ptr->x_pos = x_pos;
		msg_ptr->y_pos = y_pos;

		msg_ptr->os_hdr.msg_id = LCD_DRAW_BLOC_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);	

		return (RV_OK);
	}
}

/********************************************************************************/
/*    Function Name: LCD_Enable													*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_enable message into lcd mailbox		*/	
/*																				*/		
/*                                                                              */
/*    Input Parameters:	x														*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_enable(UINT8 x)
{

	/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);

    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

		return (RV_OK);
}

/********************************************************************************/
/*    Function Name: LCD_SetPower												*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_set_power message into lcd mailbox		*/	
/*																				*/		
/*                                                                              */
/*    Input Parameters:	None													*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_set_power()
{

	T_RVF_MB_STATUS		mb_status;
    T_LCD_SET_POWER		*msg_ptr;
	
	/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
	{	LCD_Init_ll();
		return (RV_NOT_READY);
	}
    if (DrvTaskReady == 0)
	{
     	LCD_Init_ll();
		return (RV_NOT_READY);
	}
	

	mb_status = rvf_get_buf (DRIVER_GBL_INFO_PTR->prim_id, sizeof (T_LCD_SET_POWER), (void **) &msg_ptr);	
	
	if ((mb_status == RVF_GREEN) || (mb_status == RVF_YELLOW)) /* Memory allocation success */
	{		

		msg_ptr->os_hdr.msg_id = LCD_SET_POWER_EVT;
		
		rvf_send_msg (DRIVER_GBL_INFO_PTR->addr_id, msg_ptr);		

		return (RV_OK);
	}
	else 
		{	
		rvf_send_trace("LCD_SetPower function FAILED ",29, NULL_PARAM, 
						RV_TRACE_LEVEL_WARNING, R2D_USE_ID );
		return(RV_MEMORY_ERR);
	}
}

/********************************************************************************/
/*    Function Name: LCD_Cursor													*/
/*                                                                              */
/*    Purpose:called by Appli, sends lcd_cursor message into lcd mailbox		*/	
/*																				*/		
/*                                                                              */
/*    Input Parameters:x,y														*/
/*    Output Parameters: 				            							*/
/*        T_RV_RET                                                              */
/********************************************************************************/


T_RV_RET lcd_cursor(UINT8 y,UINT8 x)
{

	/* check if the driver has been started */
	if ( DRIVER_GBL_INFO_PTR == NULL)
		return (RV_NOT_READY);

    if (DrvTaskReady == 0)
	    return(RV_NOT_READY);

		return (RV_OK);
}


		
