/************************************************************************************
* lcd_functions.c :            contains low level driver for the lcd				*
*																					*	
*																					*
*	Author: Davide Carpegna															*
*																					*
*	version: 1.0																	*
*																					*
*	Date: 22/09/2000																*
*   (C) Copyright 2000 by Texas Instruments Incorporated, All Rights Reserved		*
*                                                                                   *
* --------------------------------------------------------------------------------- *
*                                                                                   *
*   History:                                                                        *
*                                                                                   *
*   10/18/2001 - Updated for R2D by Christophe Favergeon		                    * 
*                                                                                   *		
************************************************************************************/

#include "rv/general.h"
#include "rvf/rvf_api.h"
#include "r2d/r2d_config.h"
#include "r2d/r2d.h"
#include "r2d/lcd_functions.h"
#include <string.h>

extern T_R2D_GC_PTR r2d_g_lcd_gc;
extern T_RVF_MB_ID r2d_mb_id;


/********************  FUNCTIONS  **********************************************/
								


/********************************************************************************/
/*    Function Name: LCD_Clear_ll												*/
/*                                                                              */
/*    Purpose:low level driver to clear all the lcd								*/
/*																				*/		
/*                                                                              */
/*    Input Parameters:	None													*/
/*    Output Parameters: 				            							*/
/*        None		                                                            */
/********************************************************************************/

void LCD_Clear_ll (void)
{
   	if (r2d_g_lcd_gc!=NULL)
	  r2d_erase(r2d_g_lcd_gc);
}

/********************************************************************************/
/*    Function Name: LCD_Init_ll												*/
/*                                                                              */
/*    Purpose:low level driver to init the lcd		    						*/
/*    The LCD is initialized by R2D. This routine is now just only clearing     */
/*    the LCD                                                                   */
/*																				*/		
/*                                                                              */
/*    Input Parameters:	None													*/
/*    Output Parameters: 				            							*/
/*        None		                                                            */
/********************************************************************************/

void LCD_Init_ll(void)
{
	
  if (r2d_g_lcd_gc!=NULL)
	 r2d_erase(r2d_g_lcd_gc);
	
}

/********************************************************************************/
/*    Function Name: LCD_Set_Column												*/
/*                                                                              */
/*    Purpose:low level driver to revert the axis in the lcd		    	    */
/*			  revert the horizontal coordinate 									*/		
/*                                                                              */
/*    Input Parameters:	UINT8 x													*/
/*    Output Parameters: 				            							*/
/*        UINT8		                                                            */
/********************************************************************************/

UINT8 LCD_Set_Column(UINT8 x)
{
	return (83-x);
}

/********************************************************************************/
/*    Function Name: LCD_Set_Row_Pixel				    						*/
/*                                                                              */
/*    Purpose:low level driver to revert the axis in the lcd		    	    */
/*			   revert the vertical coordinates in pixel	    					*/		
/*                                                                              */
/*    Input Parameters:	UINT8 y													*/
/*    Output Parameters: 				            							*/
/*        UINT8		                                                            */
/********************************************************************************/

UINT8 LCD_Set_Row_Pixel(UINT8 y)
{
	return (47-y);
}
/********************************************************************************/
/*    Function Name: LCD_Set_Row							    				*/
/*                                                                              */
/*    Purpose:low level driver to revert the axis in the lcd		    	    */
/*			  revert the vertical coordinates in row							*/		
/*                                                                              */
/*    Input Parameters:	UINT8 y													*/
/*    Output Parameters: 				            							*/
/*        UINT8		                                                            */
/********************************************************************************/

UINT8 LCD_Set_Row(UINT8 y)
{
	return (5-y);
}


/********************************************************************************/
/*    Function Name: LCD_Write_String											*/
/*                                                                              */
/*    Purpose:low level driver to write a string in the lcd		        	    */
/*																				*/		
/*                                                                              */
/*    Input Parameters:	row,column,string videomode 							*/
/*    Output Parameters: 				            							*/
/*        None 		                                                            */
/********************************************************************************/

void LCD_Write_String(UINT8 row, UINT8 column, char *string, T_VIDEO_MODE mode)
{
//    UINT8 j,l,k,X,Y;
//	char asciicharacter;
//	char new_char;

	INT16 ascent,descent,leading;//,h;
	T_R2D_UTF16 *text;

	if (r2d_g_lcd_gc!=NULL)
	{

    r2d_get_font_info(r2d_g_lcd_gc,&ascent,&descent,&leading);
	
   
	r2d_set_drawing_mode(r2d_g_lcd_gc,R2D_COPY_MODE);
    
    text=r2d_new_unicode_from_cstring(r2d_mb_id,(unsigned char*)string);

	
	if (mode)
	{
		T_R2D_ARGB_COLOR bcol,fcol;
//		T_R2D_DRAWING_MODE mode;
		UINT16 text_width;

		bcol=r2d_get_background_color(r2d_g_lcd_gc);
		fcol=r2d_get_foreground_color(r2d_g_lcd_gc);
		r2d_set_background_color_with_argb(r2d_g_lcd_gc,0,0,0,0);
		r2d_set_foreground_color_with_argb(r2d_g_lcd_gc,0,255,255,255);
        
        r2d_get_text_width(r2d_g_lcd_gc,text,&text_width);

        r2d_draw_text(r2d_g_lcd_gc,(INT16)(column*6),(INT16)((row+1)*8),text);
		r2d_set_background_color(r2d_g_lcd_gc,bcol);
		r2d_set_foreground_color(r2d_g_lcd_gc,fcol);
	}
	else
        r2d_draw_text(r2d_g_lcd_gc,(INT16)(column*6),(INT16)((row+1)*8),text);
    rvf_free_buf((void*)text);
	}
}

/********************************************************************************/
/*    Function Name: LCD_Write_Center							    			*/
/*                                                                              */
/*    Purpose:low level driver to write a string in the screen center	   	    */
/*																				*/		
/*                                                                              */
/*    Input Parameters:	string													*/
/*    Output Parameters: 				            							*/
/*        None		                                                            */
/********************************************************************************/

void LCD_Write_Center( char *string)
{
  
  UINT16 i;
  INT16 j;
  char *p;
  j=0;
  p=string;
  while(*p!='\0')
  {
	  j++;
	  p++;
  }

  i=(R2D_WIDTH-((j<<2)+(j<<1)))>>1;
  if (i<0)
	  i=0;
  i=((i*0x2A)+0x80)>>8; // i/6 in Q8
  LCD_Write_String(Y_CENTER,(UINT8) i,string,NORMAL);
}  




















