/******************************************************************************
 * Power Task (pwr)
 * Design and coding by Svend Kristian Lindholm, skl@ti.com
 *
 * Environment (Riviera) functions
 *
 * $Id: pwr_env.c 1.2 Wed, 20 Aug 2003 12:54:50 +0200 skl $
 *
 ******************************************************************************/

#include "lcc/lcc.h"
#include "lcc/lcc_env.h"
#include "lcc/lcc_task.h"
#include "lcc/lcc_trace.h"
#include "lcc/lcc_handle_timers.h"

#include "rv/rv_defined_swe.h"
#include "rvm/rvm_priorities.h"
#include "rvm/rvm_api.h"
#include "rvm/rvm_use_id_list.h"

#include <string.h>
#include "abb/abb.h"

/* Define a pointer to the PWR environment control block */
T_PWR_CTRL_BLOCK *pwr_ctrl = NULL;

/* Define a pointer to the PWR configuration block */
T_PWR_CFG_BLOCK  *pwr_cfg  = NULL;

// Function Prototypes
T_RVM_RETURN pwr_init(void);
T_RVM_RETURN pwr_start(void);
T_RVM_RETURN pwr_stop(T_RV_HDR *msg);
T_RVM_RETURN pwr_kill (void);

void ttr(unsigned trmask, char *format, ...);
void str(unsigned mask, char *string);


T_RVM_RETURN pwr_task_init (void);

/* Define global pointer to the error function */
static T_RVM_RETURN (*pwr_error_ft) (T_RVM_NAME    swe_name,
                                     T_RVM_RETURN  error_cause,
                                     T_RVM_ERROR_TYPE  error_type,
                                     T_RVM_STRING   error_msg);

// Handle timers
#include "lcc/lcc_handle_timers.c"

T_RVM_RETURN lcc_get_info (T_RVM_INFO_SWE  *infoSWE)
{
#if 1
   ttw(ttr(TTrEnv, "lcc_get_info(%d):" NL, 0));
#endif
   /* SWE info */

   /* TYPE3 : Blocking by using FFS */
   infoSWE->swe_type = RVM_SWE_TYPE_3; 
   infoSWE->type_info.type3.swe_use_id = LCC_USE_ID;
   infoSWE->type_info.type3.stack_size = LCC_STACK_SIZE;
   infoSWE->type_info.type3.priority   = RVM_LCC_TASK_PRIORITY;
   infoSWE->type_info.type3.version    = LCC_TASK_VERSION;

   memcpy( infoSWE->type_info.type3.swe_name, "LCC", sizeof("LCC") );

   /* Set the return path */
   infoSWE->type_info.type3.return_path.callback_func = NULL;
   infoSWE->type_info.type3.return_path.addr_id       = 0;

   /* memory bank info */
   infoSWE->type_info.type3.nb_mem_bank = 1;

   memcpy ((UINT8 *) infoSWE->type_info.type3.mem_bank[0].bank_name, "LCC_PRIM", 9);
   infoSWE->type_info.type3.mem_bank[0].initial_params.size          = LCC_MB_PRIM_SIZE;
   infoSWE->type_info.type3.mem_bank[0].initial_params.watermark     = LCC_MB_PRIM_WATERMARK;

   /* Linked SW entities : FIXME: LCC needs SPI & FFS */ 
   infoSWE->type_info.type3.nb_linked_swe = 0;

   /* generic functions */
   infoSWE->type_info.type3.set_info = pwr_set_info;
   infoSWE->type_info.type3.init     = pwr_init;
   infoSWE->type_info.type3.start    = pwr_start;
   infoSWE->type_info.type3.stop     = pwr_stop;
   infoSWE->type_info.type3.kill     = pwr_kill;

   return RV_OK;

}


T_RVM_RETURN pwr_set_info(T_RVF_ADDR_ID   addr_id,
                          T_RV_RETURN     return_path[],
                          T_RVF_MB_ID     mbId[],
                          T_RVM_RETURN  (*callBackFct) (T_RVM_NAME SWEntName,
                                                        T_RVM_RETURN errorCause,
                                                        T_RVM_ERROR_TYPE errorType,
                                                        T_RVM_STRING errorMsg))
{

    T_RVF_MB_STATUS   mb_status;

    ttw(ttr(TTrEnv, "pwr_set_info(%d)" NL, addr_id));
    ttw(ttr(TTrEnvLow, "  mbId[0] = %d" NL, mbId[0]));

    mb_status = rvf_get_buf(mbId[0],sizeof(T_PWR_CTRL_BLOCK),(void **) &pwr_ctrl);
    if (mb_status == RVF_RED) 
    {
        ttr(TTrFatal, "LCC FATAL: no memory: %d" NL, addr_id);
        return (RVM_MEMORY_ERR);
    }
    memset(&pwr_ctrl->state, 0xBA, sizeof(T_PWR_CTRL_BLOCK));

    mb_status = rvf_get_buf(mbId[0],sizeof(T_PWR_CFG_BLOCK),(void **) &pwr_cfg);
    if (mb_status == RVF_RED) 
    {
        ttr(TTrFatal, "LCC FATAL: no memory: %d" NL, addr_id);
        return (RVM_MEMORY_ERR);
    }
    memset(&pwr_cfg->cal.i2v , 0xBA, sizeof(T_PWR_CFG_BLOCK));

    /* store the pointer to the error function */
    pwr_error_ft = callBackFct ;

    /* Store the addr id */
    pwr_ctrl->addr_id = addr_id;


    /* Store the memory bank id */
    pwr_ctrl->prim_id = mbId[0];

    /* Battery & charging related initialisation */

    // State initialisation
    pwr_ctrl->state = CAL;

    // Create timers (Initialize...)
    pwr_create_timer(&pwr_ctrl->time_begin_T0);
    pwr_create_timer(&pwr_ctrl->time_begin_T1);
    pwr_create_timer(&pwr_ctrl->time_begin_T2);
    pwr_create_timer(&pwr_ctrl->time_begin_T3);
    pwr_create_timer(&pwr_ctrl->time_begin_T4);
    pwr_create_timer(&pwr_ctrl->time_begin_mod_cycle);
    pwr_create_timer(&pwr_ctrl->time_begin_mmi_rep);

    /* Flags initialisation and other state related init */

    // Calibration configuration file not read
    pwr_ctrl->flag_cal_cfg_read = 0;

    // Charger configuration file not read
    pwr_ctrl->flag_chg_cfg_read = 0;

    // Default configuration id used
    pwr_cfg->data.cfg_id = '1';

    // Number of battery identification measurements reported
    pwr_ctrl->count_bat_type = 0;
    pwr_ctrl->count_chg_type = 0;

    // Precharge PRE state - Applies only for 3.2V < Vbat < 3.6V
    pwr_ctrl->flag_prechg_started = 0;

    // Charger Interrupts are disabled from the start
    pwr_ctrl->flag_chg_int_disabled = 1;

    // First time in INI state
    pwr_ctrl->flag_ini_virgo = 0;

    // Default charger type is UNREGULATED
    // Meaning charger interrupts will not be enabled
    pwr_cfg->chg.type = UNREGULATED;

    // Unplug default is that charger is unplugged (0)
    memset(&pwr_ctrl->chg_unplug_vec , 0x0, CONSECUTIVE_CHG_UNPLUGS);
    pwr_ctrl->index = 0;


   // MMI hasn't registered
   pwr_ctrl->flag_mmi_registered = 0;

   // Init of plug/unplug state machine - from default the charger is not plugged
   pwr_ctrl->flag_chg_prev_plugged = 0;

    return RV_OK;
}


T_RVM_RETURN pwr_init(void)
{
   T_RVM_RETURN error;
   // Mask off the charger interrupts (plug/unplug) - in case it was a linear charger
   // Don't want to get killed in a vast number of interrupts...
   ABB_Write_Register_on_page(PAGE0, ITMASK, CHARGER_IT_MSK);
   pwr_ttr_init(0xff000000);
   pwr_ctrl->tmask = 0xff000000;
   ttw(ttr(TTrEnv, "pwr_init(%d)" NL, 0));
   return RV_OK;
}

T_RVM_RETURN pwr_start(void)
{
   T_RVM_RETURN error;
   ttw(ttr(TTrEnv, "pwr_start(%d)" NL, 0));
   pwr_task();
   ttw(ttr(TTrEnv, "pwr_start(%d)" NL, 0xFF));
   return RV_OK;
}

T_RVM_RETURN pwr_stop(T_RV_HDR *msg)
{
   ttw(ttr(TTrEnv, "pwr_stop(%d)" NL, 0));
   return RV_OK;
}

T_RVM_RETURN pwr_kill (void)
{
   T_RVM_RETURN error;

   ttw(ttr(TTrEnv, "pwr_kill(%d)" NL, 0));

   /* free all memory buffer previously allocated */
   rvf_free_buf ((void *) pwr_ctrl);
   rvf_free_buf ((void *) pwr_cfg);
   ttw(ttr(TTrEnv, "pwr_kill(%d)" NL, 0xFF));

   return RV_OK;
}

