/**
 * @file   kpd_env.c
 *
 * Coding of the Riviera Generic Functions 
 * (except handle message and handle timer).
 *
 * @author   Laurent Sollier (l-sollier@ti.com)
 * @version 0.1
 */

/*
 * History:
 *
 *   Date          Author       Modification
 *  ----------------------------------------
 *  10/10/2001     L Sollier    Create
 *
 *
 * (C) Copyright 2001 by Texas Instruments Incorporated, All Rights Reserved
 */



#include "kpd/kpd_env.h"
#include "kpd/kpd_i.h"

#include "rvm/rvm_priorities.h"
#include "rvm/rvm_use_id_list.h"

#include <string.h>


/* External declaration */
extern T_RV_RET kpd_core(void);


/**
 * Pointer on the structure gathering all the global variables
 * used by KPD instance.
 */
T_KPD_ENV_CTRL_BLK* kpd_env_ctrl_blk = NULL;


/**
 * @name Riviera generic functions implementation
 *
 */
/*@{*/

/**
 * function: kpd_get_info
 */
T_RVM_RETURN kpd_get_info(T_RVM_INFO_SWE* swe_info)
{
   /* SWE info */
   swe_info->swe_type = RVM_SWE_TYPE_4;
   swe_info->type_info.type4.swe_use_id = KPD_USE_ID;

   memcpy( swe_info->type_info.type4.swe_name, "KPD", sizeof("KPD") );

   swe_info->type_info.type4.stack_size = KPD_STACK_SIZE;
   swe_info->type_info.type4.priority   = RVM_KPD_TASK_PRIORITY;


   /* Set the return path */
   swe_info->type_info.type4.return_path.callback_func   = NULL;
   swe_info->type_info.type4.return_path.addr_id         = 0;


   /* memory bank info */
   swe_info->type_info.type4.nb_mem_bank = 1;

   memcpy ((UINT8 *)    swe_info->type_info.type4.mem_bank[0].bank_name, "KPD_PRIM", 9);
   swe_info->type_info.type4.mem_bank[0].initial_params.size          = KPD_MB_PRIM_SIZE;
   swe_info->type_info.type4.mem_bank[0].initial_params.watermark     = KPD_MB_PRIM_WATERMARK;

   /* linked SWE info */
   /* this SWE does not require any SWE to run */
   swe_info->type_info.type4.nb_linked_swe = 0;

   /* generic functions */
   swe_info->type_info.type4.set_info = kpd_set_info;
   swe_info->type_info.type4.init     = kpd_init;
   swe_info->type_info.type4.core     = kpd_core;
   swe_info->type_info.type4.stop     = kpd_stop;
   swe_info->type_info.type4.kill     = kpd_kill;

   return RVM_OK;

}


/**
 * function: kpd_set_info
 */
T_RVM_RETURN kpd_set_info (T_RVF_ADDR_ID   addr_id, 
                           T_RV_RETURN      return_path[], 
                           T_RVF_MB_ID      bk_id[],
                           T_RVM_RETURN   (*rvm_error_ft)(T_RVM_NAME       swe_name, 
                                                        T_RVM_RETURN       error_cause, 
                                                        T_RVM_ERROR_TYPE   error_type, 
                                                        T_RVM_STRING       error_msg))
{
   /* Create instance gathering all the variable used by EXPL instance */
   if (rvf_get_buf(bk_id[0], 
                  sizeof(T_KPD_ENV_CTRL_BLK),
                  (T_RVF_BUFFER**)&kpd_env_ctrl_blk) != RVF_GREEN)
   {
      /* The environemnt will cancel the EXPL instance creation. */
      return RVM_MEMORY_ERR;
   }

   /* Store the pointer to the error function */
   kpd_env_ctrl_blk->error_ft = rvm_error_ft ;
   /* Store the mem bank id. */
   kpd_env_ctrl_blk->prim_id = bk_id[0];
   /* Store the address id */
   kpd_env_ctrl_blk->addr_id = addr_id;
   kpd_env_ctrl_blk->swe_is_initialized = FALSE;


   /* return_path of linked SWE -> not used */

   return RVM_OK;
}


/**
 * function: kpd_init
 */
T_RVM_RETURN kpd_init(void)
{
   T_RV_RET ret = RV_OK;

   /* Initialise keypad driver */
   ret = kpd_initialize_keypad_driver();

   /* Note that the (kpd_env_ctrl_blk != 0) check is normally
      useless because if kpd_set_info fail, kpd_init will not
      be called */
   if ( (ret == RV_OK) && (kpd_env_ctrl_blk != 0))
   {
      kpd_env_ctrl_blk->swe_is_initialized = TRUE;
   }

   return ret;
}


/**
 * function: kpd_stop
 */
T_RVM_RETURN kpd_stop(void)
{
   /* NB: Other SWEs have not been killed yet, kpd can send messages to other SWEs. */

   return RVM_OK;
}


/**
 * function: kpd_kill
 */
T_RVM_RETURN kpd_kill (void)
{
   /*
    * DO NOT SEND MESSAGES
    */
   rvf_free_buf(kpd_env_ctrl_blk);

   kpd_kill_keypad_driver();

   return RVM_OK;
}


/*@}*/
