/******************************************************************************
 * Flash File System (ffs)
 * Idea, design and coding by Mads Meisner-Jensen, mmj@ti.com
 *
 * ffs test scaffold/framework
 *
 * $Id: tffs.h 1.7.1.15 Fri, 19 Dec 2003 12:00:13 +0100 tsj $
 *
 ******************************************************************************/

#include "ffs/ffs.h"


/******************************************************************************
 * Compile Defines
 ******************************************************************************/

// Maximum number of parameters for a test case
#define FFS_TESTCASE_PARAMS_MAX 2

// Maximum length of a test case name
#define FFS_TESTCASE_NAME_MAX 11


/******************************************************************************
 * Types used in test framework
 ******************************************************************************/

struct ffs_state_s {
    int inodes_used;   // taken from bstat entry for inodes block
    int inodes_lost;   // taken from bstat entry for inodes block
    int objects_free;  // taken from bstat entry for inodes block
    int objects_total; // accumulated number of valid objects

    int bytes_used;    // accumulated over all blocks
    int bytes_lost;    // accumulated over all blocks
    int bytes_free;    // accumulated over all blocks

    int blocks_free;   // accumulated over all blocks
};

struct ffs_params_s {
    // Actual parameters
    int filename_max;
    int pathdepth_max;
    int inodes_max;     
    int bytes_max;
    int numblocks;
    int atomsize;
    int blocks_free_min;

    // Pseudo parameters
    int block_size;  // size of each block
    int bytes_avail; // size of available data space
    int data_blocks; // number of blocks available for data storage
};

struct this_test_s {
    char name[FFS_TESTCASE_NAME_MAX];  // name of currently executing test case
    int  numcalls;  // total number of ffs function calls so far
};


/******************************************************************************
 * Types used in test cases
 ******************************************************************************/

typedef int (*pf_t)(int, int);

enum TEST_CASE_FLAGS {
    IT  = 0x01, // test case is runnable in target
    PC  = 0x02, // test case is runnable on PC
    RND = 0x04  // test case is randomly runnable
};

struct testcase_s {
    char *name;
    int flags;
    pf_t function;
    char *comment;
};

struct test_file_s {
    const char *name;
    char *data;
    int   size;
};

extern char *ffs_strerror(effs_t error);


/******************************************************************************
 * Globals
 ******************************************************************************/

extern struct ffs_params_s param;

extern struct dir_s dir;
extern struct stat_s stat;
extern struct xstat_s xstat;

extern int error;

extern int smallbuf_size;
extern char *smallbuf;
extern int bigbuf_size;
extern char *bigbuf;


/******************************************************************************
 * Functions
 ******************************************************************************/

void test_listall(void);
int  test_run(char *testname);
void test_init(int keepgoing);
void test_exit(void);
void test_begin(char *name, int *params);
void test_end(struct this_test_s *test, int n);
void test_error(struct this_test_s *test, int n);

int test_expect(int error, int xerror);
int test_expect_ok(int n);
int test_expect_equal(int n, int xn);
int test_expect_not_equal(int n, int xn);
int test_expect_greater_than(int n, int xn);
int test_expect_data(const void *data1, const void *data2, int size);
int test_expect_file(const char *name, const void *data, int size);
int test_expect_state(struct ffs_state_s *old, struct ffs_state_s *new);
int test_expect_objects(struct ffs_state_s *old, struct ffs_state_s *new);

int test_ffs_params_get(void);
int test_ffs_state_get(struct ffs_state_s *s);
void test_state_print(struct ffs_state_s *state);
void test_state_objects_print(struct ffs_state_s *old, struct ffs_state_s *new);
void test_state_bytes_print(struct ffs_state_s *old, struct ffs_state_s *new);
void test_ffs_state_copy(struct ffs_state_s *dst, struct ffs_state_s *src);
void test_statistics_print(void);

#define expect(error, xerror) if (test_expect(error, xerror)) return 1;
#define expect_ok(error) if (test_expect_ok(error)) return 1;
#define expect_eq(n, xnum) if (test_expect_equal(n, xnum)) return 1;
#define expect_ne(n, xnum) if (test_expect_not_equal(n, xnum)) return 1;
#define expect_gt(n, xnum) if (test_expect_greater_than(n, xnum)) return 1;

#if 0
// This does NOT work when <addr, size> is macro-expanded!
#define expect_file(name, addr, size) \
    if (test_expect_file(name, addr, size)) return 1;
#endif


/******************************************************************************
 * Test (blocking) versions of FFS API functions
 ******************************************************************************/

effs_t tffs_fcreate(const char *name, void *addr, int size);
effs_t tffs_fupdate(const char *name, void *addr, int size);
effs_t tffs_fwrite(const char *name, void *addr, int size);
effs_t tffs_file_write(const char *name, void *addr, int size, uint16 option);
int    tffs_fread(const char *name, void *addr, int size);
int    tffs_file_read(const char *name, void *addr, int size);
effs_t tffs_mkdir(const char *name);
int    tffs_opendir(const char *name, struct dir_s *dir);
int    tffs_readdir (struct dir_s *dir, char *name, int8 size);
effs_t tffs_symlink(const char *name, const char *actualpath);
int    tffs_readlink(const char *name, char *addr, int size);
int tffs_rename(const char *oldname, const char *newname);
effs_t tffs_stat(const char *name, struct stat_s *stat);
effs_t tffs_xstat(const char *name, struct xstat_s *stat);
effs_t tffs_fstat(fd_t fdi, struct stat_s *stat);
effs_t tffs_xfstat(fd_t fdi, struct xstat_s *stat);
effs_t tffs_linkstat(const char *name, struct stat_s *stat);
effs_t tffs_lstat(const char *name, struct stat_s *stat);
effs_t tffs_xlstat(const char *name, struct xstat_s *stat);
effs_t tffs_remove(const char *name);
effs_t tffs_fcontrol(const char *pathname, int8 action, uint16 param);
effs_t tffs_query(int8 query, void *p);
effs_t tffs_preformat(uint16 magic);
effs_t tffs_format(const char *name, uint16 magic);
effs_t tffs_initialize(void);
effs_t tffs_exit(void);
fd_t tffs_open(const char *pathname, ffs_options_t options);
effs_t tffs_close(fd_t fdi);
int tffs_write(fd_t fdi, void *addr, int size);
int tffs_read(fd_t fdi, void *addr, int size);
int tffs_seek(fd_t fdi, int offset, int whence);
effs_t tffs_truncate(const char *path, offset_t length); 
effs_t tffs_ftruncate(fd_t fdi, offset_t length); 
effs_t tffs_fdatasync(fd_t fdi); 

/******************************************************************************
 * Platform/OS dependent functions
 ******************************************************************************/

// Defined in task.c
#if (WITH_TFFS == 1)
extern void tffs_delay(int delay);
extern int tffs_wait(const char *name);
extern struct ffs_cnfpath_s *tffs_cnfpath_get(void);
extern UINT32 tffs_timer_begin(void);
extern UINT32 tffs_timer_end(UINT32 time_begin);
#else
#define tffs_delay(delay)
#endif




