/*
 * Flash File System (ffs)
 *
 * WCP PCM Compatibility Support
 */

#include <string.h>
#include "main/sys_types.h"
#include "memif/mem.h"
#include "ffs/ffs.h" 
#include "ffs/board/ffspcm.h"
#include "nucleus.h"
#include "csmi/csmi.h" 
#include "csmi/csmi_gsmctrl.h" 
#include "ffs/pcm.h" 


// Factory FFS area description
#pragma DATA_SECTION(FlashImage, ".ffs_img")
const unsigned char FlashImage[0x4000];


/*
 *
 * ffs_InitRFCap
 * 
 * Purpose  : Set rfcap for tri-band config
 *
 * Arguments: In : none
 *                 
 *            Out: none
 *
 * Returns  : none
 *
 */

#define SIZE_EF_RFCAP	16

static const UBYTE rfcap_TriBand[SIZE_EF_RFCAP] =   {0x00, 0x0F, 0x41, 0x10, 0x00, 0x00, 0x00, 0x00, \
                                                     0x40, 0x00, 0x00, 0xA5, 0x05, 0x00, 0xC0, 0x00};

static const UBYTE rfcap_DualExt[SIZE_EF_RFCAP] =   {0x00, 0x0B, 0x41, 0x00, 0x00, 0x00, 0x00, 0x00, \
                                                     0x40, 0x00, 0x00, 0xA5, 0x05, 0x00, 0xC0, 0x00};
                                                     
effs_t ffs_InitRFCap ()
{
    ffs_mkdir_nb("/gsm" ,0);
    ffs_mkdir_nb("/gsm/com" ,0);

    ffs_fwrite_nb("/gsm/com/rfcap", (void*)rfcap_TriBand, SIZE_EF_RFCAP, 0);
}

/*
 *
 * ffs_SetBand
 * 
 * Purpose  : Set std config according to the selected Band
 *
 * Arguments: In : Band number
 *                 
 *            Out: none
 *
 * Returns  : none
 *
 */

effs_t ffs_SetBand (unsigned char* band)
{
    // Check validity of std band (1, 3 and 6 are supported)
    switch(*band)
    {
        case 1:
        case 3:
        case 6:
          // fall through
          break;

        default:
          return EFFS_BADNAME;
    }

    // Write STD file and set global variable
    ffs_fwrite_nb("/pcm/STD", band, 1, 0);
}

/*
 *
 * ffs_GetBand
 * 
 * Purpose  : Get STD config as recorded in FFS
 *
 * Arguments: In : none
 *                 
 *            Out: none
 *
 * Returns  : STD
 *
 */

unsigned char ffs_GetBand ()
{
    UBYTE   ubStd = 0;
    effs_t  result;

    // get FFS value
    result = ffs_fread("/pcm/STD", &ubStd, 1);

    // default if an error occured : 900-1800
    if (result < EFFS_OK)
    {
        ubStd = 6;
    }

    return ubStd;
}

/*
 * ffs_restore
 *
 * Restores flash file system from MPU Persistent Storage
 * Restore Factory FFS in cae of file corruption
 *
 * Procedure described in PE023 document
 *
 */

void ffs_restore(void)
{
    effs_t res;

    // FFS restore procedure started
    // send a message through CSMI
    if( GC_RestoreFFSReq(FFS_BASE_ADDRESS, IMAGE_SIZE) == CSMI_ERROR )
    {
      // In case of MPU File corruption, 
      // restore working FFS from factory FFS
      memcpy((char *) FFS_BASE_ADDRESS, FlashImage, IMAGE_SIZE);

      // backup FFS on MPU side 
      // if we are running with the OS
      if(GC_RunningWithOs())
        ffs_backup();
    }
}

/*
 * ffs_backup
 *
 * Backup flash file system
 *
 * Procedure described in PE023 document
 *
 */

void ffs_backup(void)
{
    // FFS backup procedure started
    // send a message through CSMI
    GC_BackupFFSReq(FFS_BASE_ADDRESS, IMAGE_SIZE);
}
