/******************************************************************************
 * Flash File System (ffs)
 * Idea, design and coding by Mads Meisner-Jensen, mmj@ti.com
 *
 * ffs flash device definitions
 *
 * $Id: dev.c 1.34.1.25 Thu, 18 Dec 2003 10:50:52 +0100 tsj $
 *
 ******************************************************************************/

#include "ffs/ffs.h"
#include "ffs/board/drv.h" 
#include "config/board.cfg"
#include "config/rf.cfg"

/******************************************************************************

 Flash memories supported:

 The real flash device names are listed in the Device column. The names of
 compatible/similar devices are listed in the "aliases" column.
 Compatible/similar devices are e.g. combo devices using that particular
 flash device.

 The size column denotes the full device memory size in megabits (Mb). For
 dual and multi-bank devices, two or more numbers indicate the sizes of the
 respective banks, starting from the lowest address.

 ---------------+--------+-------------------------------------------------
 Device         |    Size| Aliases and Notes
 ---------------+--------+-----------------------------------------------------
 AMD            |        | 
 29DL161T       |15.5+0.5| 
 29DL162T       |14.0+2.0| 
 29DL163T       |12.0+4.0| 
 29DL164T       | 8.0+8.0| 
 29DL322T       |28.0+4.0| 
 29DL323T       |24.0+8.0| 
 ---------------+--------+-----------------------------------------------------
 Fujitsu        |        |
 29DL161T       |15.5+0.5| (Am29DL161)
 29DL162T       |14.0+2.0| (Am29DL162)
 29DL163T       |12.0+4.0| (Am29DL163)
 29DL164T       | 8.0+8.0| (Am29DL164)
 29DL321TD      |        | (Am29DL321), MB84VD22181, MB84VD22081, MB84VD22191
 29DL322TD      |28.0+4.0| (Am29DL322), MB84VD22182, MB84VD22082, MB84VD22192
 29DL323TD      |24.0+8.0| (Am29DL323), MB84VD22183, MB84VD22083, MB84VD22193,
                |        | BSample device
 ---------------+--------+-----------------------------------------------------
 Samsung        |        |
 K5A3240YT      |24+8    | 
 K5A3240YB      | 8+24   |
 K5A3340YT      |16+16   |
 K5A3340YB      |14+18   | (!?)
 ---------------+--------+-----------------------------------------------------
 SST            |        |
 SST36VF1601    |        |
 ---------------+--------+-----------------------------------------------------
 Toshiba        |        |
 ?              |        | TH50VSF2581AASB
 ---------------+--------+-----------------------------------------------------
 Intel          |        |
 28F160C3       |    16.0| Both Top and Bottom boot
 28F320C3       |    32.0| Both Top and Bottom boot
 28F640C3       |    64.0| Both Top and Bottom boot
 28F640W30      |    64.0| Both Top and Bottom boot, DSample device
 ---------------+--------+-----------------------------------------------------
 Test           |        |
                |        |
 ---------------+--------+-----------------------------------------------------

******************************************************************************/


/******************************************************************************
 * Flash Device Memory Maps
 ******************************************************************************/

// The memory map, flash_AAxBB[], used by each flash device definition in
// the flash_info[] array below can be either one of the pre-defined ones
// below or a custom-defined one. Each line in the memory map represents one
// physical device sector. The columns in the memory map has the following
// meaning:
//
// 1. Relative address offset of sector.
// 2. Base 2 logarithm of sector byte size, e.g. 64kB ~ 2^16.
//
// Note that does NOT support a flash definition with different sized blocks
// so all blocks contained in the device definition MUST be the same size!


// 128x64kb
static const struct block_info_s flash_128x64[] =
{
  { 0x00000, 16 },
  { 0x10000, 16 },
  { 0x20000, 16 },
  { 0x30000, 16 },
  { 0x40000, 16 },
  { 0x50000, 16 },
  { 0x60000, 16 },
  { 0x70000, 16 },
  { 0x80000, 16 },
  { 0x90000, 16 },
  { 0xa0000, 16 },
  { 0xb0000, 16 },
  { 0xc0000, 16 },
  { 0xd0000, 16 },
  { 0xe0000, 16 },
  { 0xf0000, 16 },
  { 0x100000, 16 },
  { 0x110000, 16 },
  { 0x120000, 16 },
  { 0x130000, 16 },
  { 0x140000, 16 },
  { 0x150000, 16 },
  { 0x160000, 16 },
  { 0x170000, 16 },
  { 0x180000, 16 },
  { 0x190000, 16 },
  { 0x1a0000, 16 },
  { 0x1b0000, 16 },
  { 0x1c0000, 16 },
  { 0x1d0000, 16 },
  { 0x1e0000, 16 },
  { 0x1f0000, 16 },
  { 0x200000, 16 },
  { 0x210000, 16 },
  { 0x220000, 16 },
  { 0x230000, 16 },
  { 0x240000, 16 },
  { 0x250000, 16 },
  { 0x260000, 16 },
  { 0x270000, 16 },
  { 0x280000, 16 },
  { 0x290000, 16 },
  { 0x2a0000, 16 },
  { 0x2b0000, 16 },
  { 0x2c0000, 16 },
  { 0x2d0000, 16 },
  { 0x2e0000, 16 },
  { 0x2f0000, 16 },
  { 0x300000, 16 },
  { 0x310000, 16 },
  { 0x320000, 16 },
  { 0x330000, 16 },
  { 0x340000, 16 },
  { 0x350000, 16 },
  { 0x360000, 16 },
  { 0x370000, 16 },
  { 0x380000, 16 },
  { 0x390000, 16 },
  { 0x3a0000, 16 },
  { 0x3b0000, 16 },
  { 0x3c0000, 16 },
  { 0x3d0000, 16 },
  { 0x3e0000, 16 },
  { 0x3f0000, 16 },
  { 0x400000, 16 },
  { 0x410000, 16 },
  { 0x420000, 16 },
  { 0x430000, 16 },
  { 0x440000, 16 },
  { 0x450000, 16 },
  { 0x460000, 16 },
  { 0x470000, 16 },
  { 0x480000, 16 },
  { 0x490000, 16 },
  { 0x4a0000, 16 },
  { 0x4b0000, 16 },
  { 0x4c0000, 16 },
  { 0x4d0000, 16 },
  { 0x4e0000, 16 },
  { 0x4f0000, 16 },
  { 0x500000, 16 },
  { 0x510000, 16 },
  { 0x520000, 16 },
  { 0x530000, 16 },
  { 0x540000, 16 },
  { 0x550000, 16 },
  { 0x560000, 16 },
  { 0x570000, 16 },
  { 0x580000, 16 },
  { 0x590000, 16 },
  { 0x5a0000, 16 },
  { 0x5b0000, 16 },
  { 0x5c0000, 16 },
  { 0x5d0000, 16 },
  { 0x5e0000, 16 },
  { 0x5f0000, 16 },
  { 0x600000, 16 },
  { 0x610000, 16 },
  { 0x620000, 16 },
  { 0x630000, 16 },
  { 0x640000, 16 },
  { 0x650000, 16 },
  { 0x660000, 16 },
  { 0x670000, 16 },
  { 0x680000, 16 },
  { 0x690000, 16 },
  { 0x6a0000, 16 },
  { 0x6b0000, 16 },
  { 0x6c0000, 16 },
  { 0x6d0000, 16 },
  { 0x6e0000, 16 },
  { 0x6f0000, 16 },
  { 0x700000, 16 },
  { 0x710000, 16 },
  { 0x720000, 16 },
  { 0x730000, 16 },
  { 0x740000, 16 },
  { 0x750000, 16 },
  { 0x760000, 16 },
  { 0x770000, 16 },
  { 0x780000, 16 },
  { 0x790000, 16 },
  { 0x7a0000, 16 },
  { 0x7b0000, 16 },
  { 0x7c0000, 16 },
  { 0x7d0000, 16 },
  { 0x7e0000, 16 },
  { 0x7f0000, 16 }
};

// 16x64kB
static const struct block_info_s flash_16x64[] =
{
    { 0x00000, 16 },
    { 0x10000, 16 },
    { 0x20000, 16 },
    { 0x30000, 16 },
    { 0x40000, 16 },
    { 0x50000, 16 },
    { 0x60000, 16 },
    { 0x70000, 16 },
    { 0x80000, 16 },
    { 0x90000, 16 },
    { 0xA0000, 16 },
    { 0xB0000, 16 },
    { 0xC0000, 16 },
    { 0xD0000, 16 },
    { 0xE0000, 16 },
    { 0xF0000, 16 }
};


// 8x8kB
static const struct block_info_s flash_8x8[] =
{
    { 0x0000, 13 },
    { 0x2000, 13 },
    { 0x4000, 13 },
    { 0x6000, 13 },
    { 0x8000, 13 },
    { 0xa000, 13 },
    { 0xc000, 13 },
    { 0xe000, 13 }
};


// 4x4kB
static const struct block_info_s flash_4x4[] =
{
    { 0x0000, 12 },
    { 0x1000, 12 },
    { 0x2000, 12 },
    { 0x3000, 12 }
};


/******************************************************************************
 * Flash Device Info Array
 ******************************************************************************/

// Each entry in the array below represents one flash device definition as
// FFS needs it. The columns have the following meaning:
//
// 1. Device memory map.
// 2. Absolute address of the first sector to be used by/for FFS.
// 3. Manufacturer code. Values are from FFS_MANUFACTURER enumeration
//    in drv.h
// 4. Device code. Available from device datasheet.
// 5. FFS device driver to use. Values are from FFS_DRIVER enumeration
//    in drv.h
// 6. Number of sectors to use, starting from the address given in
//    column 2.

/********** RAM configuration *************************************************
 * Note it is necessary to hardwire the configuration of the ram driver
 * in cfgffs.c. The ram driver is configured like and pseudo flash
 * device with only a few exceptions: Field 2 which is the absolute
 * address of the first sector MUST be set to zero, instead the address
 * is configured in cfgffs.c. Field 4 'Device code' is an arbitrary
 * value and not a real device code.
 ******************************************************************************/

/********** Multi-id configuration ********************************************
 * When 227Eh is output, it indicates that two additional codes, called
 * Extended Device Codes, will be required. The two additional codes will be
 * read and the lower byte of each code will be combined into one id, which
 * is used for auto-detection of the flash configuration. The device code
 * format for the multi-id devices is as follow: 0x227E, 0x22YY, 0x22ZZ ->
 * 0xYYZZ.
 ******************************************************************************/

const struct flash_info_s flash_info[] =
{
     //BOARD 34. Ram device 4x4kB. Note it is necessary to hardwire the conf
    //in cfgffs.c
    { &flash_4x4[0], 0, MANUFACT_RAM,  0x0404,
      FFS_DRIVER_RAM,  4},    

    // Ram device 8x8kB. Note it is necessary to hardwire the conf in cfgffs.c 
    { &flash_8x8[0], 0, MANUFACT_RAM,  0x080D,
      FFS_DRIVER_RAM,  8},    

    /********** AMD Devices ***********************************************/

	// AMD Am29DL640F. Ignoring the 8kB sectors
    // Multi-id device: 0x227E, 0x2221, 0x2200. Converted to 0x2100
    { &flash_16x64[0], (char *) 0x01800000, MANUFACT_AMD,     0x2100,
      FFS_DRIVER_AMD,  15 },

	// AMD Am29DL640G. Ignoring the 8kB sectors
    // Multi-id device: 0x227E, 0x2202, 0x2201. Converted to 0x0201
    { &flash_16x64[0], (char *) 0x700000, MANUFACT_AMD,     0x0201,
      FFS_DRIVER_AMD,  15 },

    // AMD Am29DL321DT does not exist as of 2000-0217

    // AMD Am29DL322DT. Ignoring the 8kB sectors 
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_AMD,     0x2255,
      FFS_DRIVER_AMD_SB, 7 },

    // AMD Am29DL323DT. Ignoring the 8kB sectors
    { &flash_16x64[0], (char *) 0x300000, MANUFACT_AMD,     0x2250,
      FFS_DRIVER_AMD, 15 },

    // AMD Am29DL161DT. Using 8x8kB sectors.
    { &flash_8x8[0],   (char *) 0x1F0000, MANUFACT_AMD,     0x2236,
      FFS_DRIVER_AMD,  8 },

    // AMD Am29DL162DT. Ignoring the 8kB sectors
    { &flash_16x64[0], (char *) 0x1C0000, MANUFACT_AMD,     0x222D,
      FFS_DRIVER_AMD,  3 },

    // AMD Am29DL163DT. Ignoring the 8kB sectors
    { &flash_16x64[0], (char *) 0x180000, MANUFACT_AMD,     0x2228,
      FFS_DRIVER_AMD,  7 },

    // AMD Am29DL164DT. Ignoring the 8kB sectors
    { &flash_16x64[0], (char *) 0x100000, MANUFACT_AMD,     0x2233,
      FFS_DRIVER_AMD, 15 },


    /********** Fujitsu Devices *******************************************/

    // Fujitsu MBM29DL321TD. Using 8x8kB sectors.
    { &flash_8x8[0],   (char *) 0x3f0000, MANUFACT_FUJITSU, 0x2259,
      FFS_DRIVER_AMD,  8 },

    // Fujitsu MBM29DL322TD. Ignoring all the 8kB sectors
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_FUJITSU, 0x2255,
      FFS_DRIVER_AMD,  7 },

    // Fujitsu MBM29DL323TD (BSample). Dual bank device definition.
    { &flash_16x64[0], (char *) 0x300000, MANUFACT_FUJITSU, 0x2250,
      FFS_DRIVER_AMD, 15 },

    // Fujitsu MBM29DL161DT (similar to Am29DL161T). 
    { &flash_8x8[0],   (char *) 0x1F0000, MANUFACT_FUJITSU, 0x2236,
      FFS_DRIVER_AMD,  8 },

    // Fujitsu MBM29DL162DT (similar to Am29DL162T).
    { &flash_16x64[0], (char *) 0x1C0000, MANUFACT_FUJITSU, 0x222D,
      FFS_DRIVER_AMD,  3 },

    // Fujitsu MBM29DL163DT (similar to Am29DL163T).
    { &flash_16x64[0], (char *) 0x180000, MANUFACT_FUJITSU, 0x2228,
      FFS_DRIVER_AMD,  7 },

    // Fujitsu MBM29DL164DT (similar to Am29DL164T).
    { &flash_16x64[0], (char *) 0x100000, MANUFACT_FUJITSU, 0x2233,
      FFS_DRIVER_AMD, 15 },

    /* This is the RITA flash configuration */
    // Fujitsu MBM29DL320FB
    // Multi-id device: 0x227E, 0x220A, 0x2200. Converted to 0x0A00
    { &flash_16x64[0], (char *) 0x320000, MANUFACT_FUJITSU, 0x0A00,
      FFS_DRIVER_AMD_SB, 13 },

    // Fujitsu MBM29DL320FT
    // Multi-id device: 0x227E, 0x220A, 0x2201. Converted to 0x0A01
    { &flash_16x64[0], (char *) 0x320000, MANUFACT_FUJITSU, 0x0A01,
      FFS_DRIVER_AMD_SB, 13 },

    // Fujitsu  MBM29DL640F, 
    // Multi-id device: 0x227E, 0x2221, 0x2200. Converted to 0x2100
    { &flash_16x64[0], (char *) 0x700000, MANUFACT_FUJITSU, 0x2100,
      FFS_DRIVER_AMD_SB, 15 },

    // Fujitsu  84VF5F5F4J2, 
    // Multi-id device: 0x227E, 0x2202, 0x2201. Converted to 0x0201
    { &flash_16x64[0], (char *) 0x700000, MANUFACT_FUJITSU, 0x0201,
      FFS_DRIVER_AMD_SB, 15 },

    // Fujitsu MB84VF5F5F4J2 stacked device. Using the 2nd sub device
    // The 8x8 are located both in top and bottom, thus only 126
    // blocks are used.
    { &flash_128x64[0], (char *) 0x1810000, MANUFACT_FUJITSU, 0xB496,
      FFS_DRIVER_AMD_SB, 126 },

    /********** Intel Devices *********************************************/

    // Intel 28F160C3-T 16Mb. Using top-most 8x8kB (parameter) sectors.
    { &flash_8x8[0],   (char *) 0x1F0000, MANUFACT_INTEL,   0x88C2,
      FFS_DRIVER_INTEL_SB,  8 },

    // Intel 28F160C3-B, 16Mb. Using top-most 6x64kB sectors
    { &flash_16x64[0], (char *) 0x1A0000, MANUFACT_INTEL,   0x88C3,
      FFS_DRIVER_INTEL_SB,  6 },

    // Intel 28F320C3-T, 32Mb. Using top-most 8x8kB (parameter) sectors.
    { &flash_8x8[0],   (char *) 0x3F0000, MANUFACT_INTEL,   0x88C4,
      FFS_DRIVER_INTEL_SB,  8 },

    // Intel 28F320C3-B, 32Mb. Using top-most 8x64kB sectors
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_INTEL,   0x88C5,
      FFS_DRIVER_INTEL_SB,  8 },

    // Intel 28F640C3-T, 64Mb. Using top-most 8x8kB (parameter) sectors.
    { &flash_8x8[0],   (char *) 0x7F0000, MANUFACT_INTEL,   0x88CC,
      FFS_DRIVER_INTEL_SB,  8 },

    // Intel 28F640C3-B, 64Mb. Using top-most 16x64kB sectors
    { &flash_16x64[0], (char *) 0x700000, MANUFACT_INTEL,   0x88CD,
      FFS_DRIVER_INTEL_SB, 15 },

    // Intel 28F640W30-T, 64Mb. Using top-most 8x8kB sectors
    { &flash_8x8[0], (char *) 0x7F0000, MANUFACT_INTEL,   0x8854,
      FFS_DRIVER_INTEL_SB, 8 },

    // Intel 28F640W30-B, 64Mb. (DSample). Using top-most 47x64kB sectors
//    { &flash_128x64[0], (char *) 0x500000, MANUFACT_INTEL,   0x8855,
//      FFS_DRIVER_INTEL_SB, 47 },

	// Intel 28F640W30-B, 64Mb. (DSample). Using top-most 15x64kB sectors
    { &flash_16x64[0], (char *) 0x700000, MANUFACT_INTEL,   0x8855,
      FFS_DRIVER_INTEL, 15 },

    // Intel 28F640W30-B, 64Mb. (DSample). Using top-most 15x64kB sectors
    { &flash_16x64[0], (char *) 0x700000, MANUFACT_INTEL,   0x88FF,
      FFS_DRIVER_INTEL_SB, 15 },

    /********** MXIC Devices *******************************************/

        // Intel 28F640W30-T, 64Mb. Using top-most 8x8kB sectors
    { &flash_8x8[0], (char *) 0x3F0000, MANUFACT_MXIC,   0x0024,
      FFS_DRIVER_INTEL_SB, 8 },


    /********** Samsung Devices *******************************************/

    // Samsung K5A3240YT Top boot, 24Mb + 8Mb
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_SAMSUNG, 0x22B7,
      FFS_DRIVER_AMD_SB,  7 },

    // Samsung K5A3340YT Top boot, 16Mb + 16Mb
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_SAMSUNG, 0x22BA,
      FFS_DRIVER_AMD_SB,  7 },

    // Samsung K5A3240YB Bottom boot, 8Mb + 24Mb
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_SAMSUNG, 0x223F,
      FFS_DRIVER_AMD_SB,  8 },

    // Samsung K5A3340YB Bottom boot, 14Mb + 18Mb (!?)
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_SAMSUNG, 0x223D,
      FFS_DRIVER_AMD_SB,  8 },
//zack add
   //Samsung K5A3240CG
   { &flash_16x64[0], (char *) 0x380000, MANUFACT_SAMSUNG, 0x22A0,
      FFS_DRIVER_AMD_SB,  7 }, 
   //Samsung K5L6331CAA-D270
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_SAMSUNG, 0x257E,
      FFS_DRIVER_AMD_SB,  15 },
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_SAMSUNG, 0x2506,
      FFS_DRIVER_AMD_SB,  15 },  
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_SAMSUNG, 0x2501,
      FFS_DRIVER_AMD_SB,  15 },  
//end zack

    /********** SST Devices ***********************************************/

    // SST SST36VF1601
    { &flash_16x64[0], (char *) 0x180000, MANUFACT_SST,     0x2761,
      FFS_DRIVER_SST,  8 },

    /********** Toshiba Devices *******************************************/

    // Toshiba TH50VSF2581AASB Bottom boot combo device
    { &flash_16x64[0], (char *) 0x380000, MANUFACT_TOSHIBA, 0x009C,
      FFS_DRIVER_AMD,  8 },


    /********** Special/Test Devices **************************************/

    // Intel 28F160C3-B, 16Mb. Single bank driver
    // Note that device ID is bogus!
    { &flash_16x64[0], (char *) 0x1A0000, MANUFACT_INTEL,   0x01C3,
      FFS_DRIVER_INTEL_SB,  6 },

    // Fujitsu MBM29DL323TD (BSample). Single bank device definition.
    // Note that device ID is bogus!
    { &flash_16x64[0], (char *) 0x300000, MANUFACT_FUJITSU, 0x0150,
      FFS_DRIVER_AMD_SB, 15 },

    // Fujitsu MBM29DL323TD (BSample). Pseudo single bank device definition.
    // Note that device ID is bogus!
    { &flash_16x64[0], (char *) 0x300000, MANUFACT_FUJITSU, 0x0250,
      FFS_DRIVER_AMD_PSEUDO_SB, 15 },

    // Fujitsu MBM29DL323TD (BSample). "True" single bank device definition
    // within first flash device bank!
    // Note that device ID is bogus!
    { &flash_16x64[0], (char *) 0x280000, MANUFACT_FUJITSU, 0x0350,
      FFS_DRIVER_AMD_SB,  8 },

    // AMD Am29DL323DT on EVA4 (base address = 0x200000)
    { &flash_16x64[0], (char *) 0x500000, MANUFACT_AMD,     0xF250,
      FFS_DRIVER_AMD, 15 },

    // PC Test/Simulation flash 128x64kb
    { &flash_128x64[0], (char *) 0x000000, MANUFACT_TEST,    0x0F12,
      FFS_DRIVER_TEST, 127 },

    // PC Test/Simulation flash 16x64kB
    { &flash_16x64[0], (char *) 0x000000, MANUFACT_TEST,    0x0F10,
      FFS_DRIVER_TEST, 15 },

    // PC Test/Simulation flash 8x8kB
    { &flash_8x8[0],   (char *) 0x000000, MANUFACT_TEST,    0x080D,
      FFS_DRIVER_TEST, 8 },

    // PC Test/Simulation flash 4x4kB
    { &flash_4x4[0],   (char *) 0x000000, MANUFACT_TEST,    0x0404,
      FFS_DRIVER_TEST, 4 },

    // terminator
    { 0, 0, 0, 0 }
};

