/*=============================================================================
 *     Copyright Texas Instruments, Inc., 2001.  All Rights Reserved.
 */



/**
 * The TaskController object manages multiple test subsystems within a
 * single task.
 * <pre>   
 * PrimHandler:
 *   When a subsystem wants to receive a certain primitive(s) it should
 *   implement a prim handler method, and call the addPrimHandler(s) method
 *   of the task controller.  The prim handler function takes a single arg,
 *   the primitive to handle.  For example:
 *   
 *       function MySubsystem()
 *       {
 *         // methods:
 *         function handlePrim( prim )
 *         {
 *           // this is called when a prim is recv'd
 *           state = state(prim);
 *         }
 *   
 *         // the active state machine... MySubsystem only has one active
 *         // synchronous process at a time, and so only needs one state
 *         // machine.  Initially no synchronous process is running, so
 *         // the state machine is null:
 *         var state = null;
 *   
 *         // register to receive the required primitives:
 *         taskController.addPrimHandler( FOO_BAR_REQ, handlePrim );
 *         taskController.addPrimHandlers( new Array( BAR_FOO_IND, BAR_FOOBAR_IND ), handlePrim );
 *         ...
 *         
 *   If a subsystem implements some synchronous method, it can call the run()
 *   method of the taskController, passing in an function that returns true
 *   if the current thread of execution is still blocked.
 *   
 *         function someSyncProcess()
 *         {
 *           state = function( prim )
 *           {
 *             if( prim instanceof ... )
 *               return null;
 *             else
 *               return state;
 *           };
 *           taskController.run( function() { return state != null; } );
 *         }
 *       }
 *   
 *   There can be multiple handlers for any primitive.  The task controller
 *   will call the prim handler method of each one.
 *   
 *   
 * EventHandler:
 *   An event is an object that contains an eventType field, which specifies the
 *   type of the event.  When a subsystem wants to receive certain events, it
 *   should implement an event handler method, and call addEventHandler(s) method 
 *   of the task controller to register to receive the specific event(s) it is 
 *   listening for.  The event handler function takes a single argument, the
 *   event to handle.  For example:
 *   
 *       function MySubsystem()
 *       {
 *         // methods:
 *         function handleEvent( event )
 *         {
 *           ...
 *         }
 *         ...
 *         
 *         taskController.addEventHandler( BETrafficHandoffInitiated, handleEvent );
 *         ...
 *       }
 *   
 * Timers:
 *   a function that takes no arguments, and is called when timer expires
 *   
 * Job:
 *   a function that takes no arguments, and is called later
 *   
 * Interface Summary:
 *   
 *   Blockable     ->   function()             returns true/false
 *   PrimHandler   ->   function(prim)         handle a primitive
 *   EventHandler  ->   function(event)        handle an event
 *   Timer         ->   function()             called when timer expires
 *   Job           ->   function()             called sometime... ???
 *   
 * Usage:
 *   Each toplevel script that uses subsystems that require a task controller
 *   should instantiate a single task controller called taskController.
 *   
 *   Each subsystem that requires a task controller should expect the toplevel
 *   script to have intantiated a task controller called taskController.
 * </pre>
 */
public function TaskController()
{
  /**
   * The routing context, which the task-controler is a layer above.  The task
   * controller registers handlers with the rctx for each primitive type handled.
   * The task-controller's handler enqueues the primitive, and the queue is read
   * from the {@link #run} method, so that the task-controller can call it's
   * clients handlers from the context of the thread calling run(), avoiding any
   * potential synchronization issues with badly written test scripts.
   */
  var esf  = services["esf routing"];
  var rctx = esf.getRoutingContext();

  
  var fxnTable          = new java.util.Hashtable();
  var primHandlerTable  = new java.util.Hashtable();
  var eventHandlerTable = new java.util.Hashtable();
  var timerManager      = services["esf routing"].getTimerManager();
  
  // timer list
  var timerListHead  = null;
  
  var jobQueue       = new pkg.util.Queue();
  const var maxJobQueueDepth = 2;
  
  var primQueue      = new pkg.util.Queue();
  
  var pausedTimer    = false;
  
  // used by ageTimers:
  var lastTime       = timerManager.getTime();
  
  
  /*
   * PrimHandler API, allows adding and removing of handlers primitive.
   */

  /**
   * Add a handler for the specified type.  The handler is a function that
   * takes a single argument, the primitive to handle.
   *   
   * @param type         the primitive type
   * @param handler      the handler object
   */
  public function addPrimHandler( type, handler )
  {
    var handlerList = primHandlerTable.get(type);
    
    if( handlerList == null )
    {
      handlerList = new java.util.LinkedList();
      primHandlerTable.put( type, handlerList );
      
      // need to use separate handler fxn for each type, per rctx API...
      var fxn = function(prim) { enqueue(prim); };
      fxnTable.put( type, fxn );
      
      rctx.addPrimHandler( fxn, rctx.type(type) );
    }
    
    handlerList.add(handler);
  }
  
  /**
   * Add a handler for an array of types.
   * 
   * @param types        the primitive types
   * @param handler      the handler object
   */
  public function addPrimHandlers( types, handler )
  {
    for( var i=0; i<types.length(); i++ )
      addPrimHandler( types[i], handler );
  }
  
  /**
   * Remove the specified handler for the specified type.  After this is
   * called, the task controller will no longer call the handler when a
   * primitive of the specified type is received.
   * 
   * @param type         the type to remove the handler for
   * @param handler      the handler to remove
   */
  public function removePrimHandler( type, handler )
  {
    var handlerList = primHandlerTable.get(type);
    
    if( handlerList != null )
    {
      handlerList.remove(handler);
      
      if( handlerList.size() == 0 )
      {
        var fxn = fxnTable.get(type);
        rctx.removePrimHandler(fxn);
        
        fxnTable.remove(type);
        primHandlerTable.remove(type);
      }
    }
  }

  /**
   * Remove the handler for an array of types
   * 
   * @param types        the primitive types
   * @param handler      the handler object
   */
  public function removePrimHandlers( types, handler )
  {
    for( var i=0; i<types.length(); i++ )
      removePrimHandler( types[i], handler );
  }

  
  
  /*
   * EventHandler API, allows adding and removing of event handlers, and
   * sending events.
   */
  
  /**
   * Add a handler for the specified type.  The handler is a function that
   * takes as it's single argument the event to handle.
   * 
   * @param type         the event type
   * @param handler      the handler object
   */
  public function addEventHandler( type, handler )
  {
    var handlerList = eventHandlerTable.get(type);
    
    if( handlerList == null )
    {
      handlerList = new java.util.LinkedList();
      eventHandlerTable.put( type, handlerList );
    }
    
    handlerList.add(handler);
  }

  /**
   * Add a handler for an array of types.
   * 
   * @param types        the event types
   * @param handler      the handler object
   */
  public function addEventHandlers( types, handler )
  {
    for( var i=0; i<types.length(); i++ )
      addEventHandler( types[i], handler );
  }
  
  /**
   * Remove the specified handler for the specified type.  After this is
   * called, the task controller will no longer call the handler when a
   * event of the specified type is sent.
   * 
   * @param type         the event type to remove the handler for
   * @param handler      the handler to remove
   */
  public function removeEventHandler( type, handler )
  {
    var handlerList = eventHandlerTable.get(type);
    
    if( handlerList != null )
      handlerList.remove(handler);
  }
  
  /**
   * Remove the handler for an array of types
   * 
   * @param types        the event types
   * @param handler      the handler object
   */
  public function removeEventHandlers( types, handler )
  {
    for( var i=0; i<types.length(); i++ )
      removeEventHandler( types[i], handler );
  }

  /**
   * Send an event.  The event is multicast to all subsystems registered to
   * receive the specific event type.
   * 
   * @param event        the event to send
   */
  public function sendEvent(event)
  {
    var handlerList = eventHandlerTable.get( event.eventType );
    
    if( handlerList != null )
    {
      var handlers = handlerList.toArray();
      
      for( var i=0; i<handlers.length(); i++ )
      {
        // add to job-queue:
        doJob( (new DispatchEventJob( event, handlers[i] )).doWork );
      }
    }
  }
  
  
  
  /*
   * Timer API,  allows setting timers.
   * 
   * Note, long term it might be better to manage timers by absolute
   * timeout, but as long as ageTimers() is called in the right places
   * it should be good-enough (not to mention easier) to manage the
   * timers this way.
   */
  
  /**
   * Set a timer.  When a timer expires, it is called.
   * <p>
   * It is safe to set a timer from an expired timer.
   * 
   * @param timer        a function
   * @param duration     number of ticks until timer expires
   */
  public function setTimer( timer, duration )
  {
    if( duration < 0 )
    {
      doJob( (new TimerListNode( 0, timer, null, null )).doWork );
      pkg.output.writeln("*** warning: timer set with negative duration: " + duration + " ***");
    }
    else
    {
      /* call ageTimers() first to ensure that the duration of the first
       * timer in the list is accurate... this is important because we are
       * maintaining timers by duration, rather than absolute timeout.  We
       * don't know how much time has elapsed since the last time ageTimers()
       * was called, so if we didn't call this we wouldn't know that the
       * duration of the head of the list is accurate.
       */
      ageTimers();
      
      var d = duration;
      
      /* search the timer list to figure out where to insert the timer... the
       * timer list is maintained in order, which the next timer to expire at
       * the head of the list.  The duration of the node at the head of the
       * list is the duration until that timer expires (which is decremented
       * by run(), if needed), and the duration of the remaining nodes is
       * the duration between the prev node expiring and the current node
       * expiring.
       */
      var prev = null;
      for( var node=timerListHead; 
           node!=null;
           prev=node, node=node.next )
      {
        // insert after all timers with same duration till expiry:
        if( (d - node.duration) < 0 )
        {
          var newNode = new TimerListNode( d, timer, prev, node );
          
          if( prev == null )
            timerListHead = newNode;
          
          node.duration -= d;
          
          return;
        }
        else
        {
          d -= node.duration;
        }
      }
      
      /* if we get here, we didn't find a place to insert, so append at the
       * end.
       */
      var newNode = new TimerListNode( d, timer, prev, null );
      
      if( prev == null )
        timerListHead = newNode;
    }
  }
  
  /**
   * Set a timeout to expire at the specified time (rather than duration).
   * 
   * @param timer        a function
   * @param timeout      time, in ticks, at which to expire the timer
   */
  public function setTimerAt( timer, timeout )
  {
    setTimer( timer, (timeout - timerManager.getTime()) );
  }
  
  
  
  /*
   * Run API, 
   */
  
  /**
   * If a thread runs for a long time without calling any TaskController
   * methods, timers may expire, but not be dispatched.  If no other
   * TaskController method is called for a while, yield() should be called
   * to check for expired timers and do other housekeeping...
   */
  public function yield()
  {
    ageTimers();
    flushJobQueue();
  }
  
  /**
   * Run until the blockable object indicates that run() can return.
   * 
   * @param blockable    an function that returns <code>true</code> if still 
   *                     blocked, or null to block forever
   */
  public function run(blockable)
  {
    if( blockable == null )
      blockable = function() { return true; };
    
    while( /*task.isRunning() &&*/ blockable() )
    {
      /* we don't know how long since this was called last, so we better
       * call it now to ensure the duration of the head of the timer list
       * is accurate.  This needs to be done at the beginning of the event
       * loop, to ensure that timerListHead.duration is accurate.
       */
      ageTimers();
      
      if( flushJobQueue() == 0 )
      {
        var duration = 0;
        
        if( timerListHead != null )
          duration = timerListHead.duration;
        
        var prim = dequeue(duration);
        
        if( (prim == null) && (duration == 0) )
        {
          pkg.output.writeln("TaskController: dequeue(0) => null");
        }
        else if( prim != null )
        {
          var handlerList = primHandlerTable.get( prim.getClass() );
          
          if( handlerList != null )
          {
            var handlers = handlerList.toArray();
            
            for( var i=0; i<handlers.length(); i++ )
            {
              // add to job-queue:
              doJob( (new DispatchPrimJob( prim, handlers[i] )).doWork );
            }
          }
        }
      }
    }
  }
  
  /**
   * Add a node to the job queue.  This method returns before job is called
   * 
   * @param job         a function
   */
  public function doJob(job)
  {
    jobQueue.enqueue(job);
    
    if( (jobQueue.getNumQueued() > maxJobQueueDepth) &&
        ! pausedTimer )
    {
      timerManager.pause();
      pausedTimer = true;
      //pkg.output.logInfoMsg("TaskController","pausing timer because jobQueue depth is " + jobQueue.getNumQueued());
    }
  }
  
  
  
  /*
   * Private API:
   */
  
  /**
   * Called from whatever thread context the primitive is received in.
   * 
   * @param prim        the rcvd prim
   */
  function enqueue(prim)
  {
    primQueue.enqueue(prim);
  }
  
  /**
   * Called from context of thread calling {@link #run}.
   * 
   * @param timeout     the timeout, in units of "ticks".  0 == wait forever
   * @return the rcvd prim, or <code>null</code> if timeout occurs before a
   *    primitive is received
   */
  function dequeue(timeout)
  {
    var isTimeout;
    
    if( timeout == 0 )
      isTimeout = function() { return false; };
    else
      isTimeout = (function() {
          var endTime = timerManager.getTime() + timeout;
          return function() {
            return endTime <= timerManager.getTime();
          };
        })();
    
    var prim = null;
    while( !isTimeout() && (prim == null) )
    {
      prim = primQueue.dequeue();
      if( prim == null )
        java.lang.Thread.sleep(100);
    }
    
    return prim;
  }
  
  /**
   * This is where timers get aged and expired.  It is called by various
   * TaskController methods to manage the active timers.
   * <p>
   * When a timer is expired, it is moved to the job queue.
   */
  function ageTimers()
  {
    var currentTime = timerManager.getTime();
    
    if( timerListHead != null )
    {
      timerListHead.duration -= (currentTime - lastTime);
      lastTime = currentTime;
      
      while( timerListHead.duration <= 0 )
      {
        var expiredTimer = timerListHead;
        
        if( timerListHead.next != null )
        {
          timerListHead.next.duration += timerListHead.duration;
          timerListHead = timerListHead.next;
          doJob( expiredTimer.doWork );
        }
        else
        {
          timerListHead = null;
          doJob( expiredTimer.doWork );
          break;
        }
      }
    }
    else
    {
      lastTime = currentTime;
    }
  }
  
  /**
   * Flush the job queue.  It is important that this TaskController not be
   * in an inconsistent state when the doWork() method of a node in the job
   * queue is called, because the doWork() method may indirectly cause
   * flushJobQueue() to be called.
   * 
   * @return number of jobs completed
   */
  function flushJobQueue()
  {
    var cnt = 0;
    var job = null;
    
    while( (job = jobQueue.dequeue()) != null )
    {
      job();
      cnt++;
    }
    
    if( pausedTimer )
    {
      timerManager.unpause();
      pausedTimer = false;
      
      //pkg.output.logInfoMsg("TaskController", "resuming timer");
    }
    
    return cnt;
  }
  
  
  /**
   * A node in the timerList, which is the list of active timers, sorted in 
   * order of expire.  A TimerListNode is also a Job, so when it expires, it
   * is added to the jobQueue.
   */
  function TimerListNode( public duration, timer, public prev, public next )
  {
    public function doWork()
    {
      timer();
    }
    
    if( this.prev != null )
    {
      // a little sanity checking:
      if( this.prev.next != this.next )
        throw new ProgrammingErrorException("prev.next != next");
      
      this.prev.next = this;
    }
  }
  
  
  /**
   * when a primitive is dispatched, it is converted into a job... the 
   * job queue is a list of things that need to be done, in this case 
   * primitives that need to be dispatched, but it can be anything.  The 
   * job queue is flushed before waiting for more primitives to ensure 
   * that all the handlers' handlePrim() methods get called before 
   * waiting for a new primitive.
   */
  function DispatchPrimJob( prim, handler )
  {
    public function doWork()
    {
      handler(prim);
    }
  }
  
  
  /**
   * When an event is sent, it is converted into a job (one per handler that
   * handles the event).  
   */
  function DispatchEventJob( event, handler )
  {
    public function doWork()
    {
      handler(event);
    }
  }
}



/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */

