/*=============================================================================
 *     Copyright Texas Instruments, Inc., 2001.  All Rights Reserved.
 */


/**
 * Base class for any test cases.
 *   
 * @param name           This is the name of the test case
 * @param args           zero or more additional args, for backwards 
 *    compatibility with base classes that try to use the old constructor
 */
public function TestCase( name, args... )
{
  /**
   * A local reference to the routing context, so we don't have to type in
   * the full name each time.
   */
  protected var rctx = pkg.util.rctx;
  
  /**
   * An array of all registered prim-handlers, used to track what handlers
   * have been registered, so that they can be unregistered when the test
   * is complete.
   */
  private var handlers = [];
  
  /**
   * Register a primitive handler, and track the handler so it can be removed
   * later.
   * 
   * @param handler the handler function
   * @param constraint specifies what the handler will handle
   * @see #removePrimHandlers
   */
  protected function addPrimHandler( handler, constraint )
  {
    handlers[ handlers.length() ] = handler;
    rctx.addPrimHandler( handler, constraint );
  }
  
  /**
   * Remove all previously registered primitive handlers.
   * 
   * @see #addPrimHandler
   */
  protected function removePrimHandlers()
  {
    for( var i=0; i<handlers.length(); i++ )
      rctx.removePrimHandler(handlers[i]);
    handlers = [];
  }
  
  // should get rid of this:
  protected var taskController  = new pkg.util.TaskController();
  
  private var infoLogger    = pkg.output.logInfoMsg;
  private var warningLogger = pkg.output.logWarning;
  private var errorLogger   = pkg.output.logError;
  
  /*=======================================================================*/
  /**
   * The top level script can call this method to start the test case.
   *
   */
  public function run()
  {
    runTest();
  }
  
  
  /*=======================================================================*/
  /**
   * Subclasses of test case should override this method to implement the
   * actual test code.
   *
   */
  public function runTest()
  {
    logError("You're test must implement the runTest() method");
  }
  
    
  /*=======================================================================*/
  /**
   * This method is used to log an informational message
   *
   */
  public function logInfo( msg )
  {
    infoLogger( name, msg );
  }
  
  
  /*=======================================================================*/
  /**
   * This method is used to log an Error message
   *
   */
  public function logError( msg )
  {
    errorLogger( name, msg );
  }
  
  
  /*=======================================================================*/
  /**
   * This method is used to log an Warning message
   *
   */
  public function logWarning( msg )
  {
    warningLogger( name, msg );
  }
  
}



/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
