/*=============================================================================
 *     Copyright 1996-2000 Texas Instruments Inc. All rights reserved.
 */



/*=============================================================================
 * File Description:
 *   This file contains the Timer class, which is an implementation of the
 *   timer event handler for test scripts.
 *   
 *   This could use taskController to set a timer for the next timeout, but
 *   instead it uses the timerManager directly... this is probably more
 *   efficient this way.
 */

var SendPrimTimer = ti.chimera.esf.SendPrimTimer;


/**
 * The <code>Timer</code> class simulates the timer event handler.
 */
public function Timer()
  extends pkg.util.Subsystem(0,"timer")
{
  var rctx = services["esf routing"].getRoutingContext();
  
  public function startReq( ehId, timeout, timerId, transId, repeating, duration )
  {
    timerManager.addTimer( new SendPrimTimer( rctx,
                                              timeout,
                                              timerId,
                                              (timerId>>8),
                                              transId,
                                              ( ( repeating == 0 ) ? -1 : duration ),
                                              new TIMER_TIMEOUT_IND( ehId,
                                                                     timeout,
                                                                     0,
                                                                     timerId,
                                                                     transId ) ) );
  }
  
  public function stopReq( timerId )
  {
    timerManager.stopTimerByTimerId( timerId );
  }
  
  public function stopFamilyReq( familyId )
  {
    while( timerManager.stopTimerByFamilyId(familyId) != null )
    {
    }
  }
  
  public function resetReq( timeout, timerId )
  {
    var t = timerManager.stopTimerByTimerId(timerId);
      
    t.setTimeout(timeout);
  
    timerManager.addTimer(t);
  }
  
  public function stopSpecificTransReq( timerId, transId )
  {
    var ts = [];  // timers that don't have matching transId need to be restarted:
    var t;
      
    while( (t = timerManager.stopTimerByTimerId(timerId)) != null )
    {
      if( t.getTransactionId() != transId )
      {
        ts[ts.length()] = t;
      }
    }
  
    for( var i=0; i<ts.length(); i++ )
    {
      timerManager.addTimer(ts[i]);
    }
  }
  
  // private methods:
  function handlePrim( prim )
  {
    if( prim instanceof TIMER_START_REQ  )
    {
      startReq( prim.getSourceEhId(),
                prim.getTimeout(),
                prim.getTimerId(),
                prim.getTransactionId(),
                prim.getRepeating(),
                prim.getDuration() );
    }
    else if( prim instanceof TIMER_STOP_REQ )
    {
      stopReq( prim.getTimerId() );
    }
    else if( prim instanceof TIMER_STOP_FAMILY_REQ )
    {
      stopFamilyReq( prim.getFamilyId() );
    }
    else if( prim instanceof TIMER_RESET_REQ )
    {
      resetReq( prim.getTimeout(), prim.getTimerId() );
    }
    else if( prim instanceof TIMER_STOP_SPECIFIC_TRANS_REQ )
    {
      stopSpecificTransReq( prim.getTimerId(), prim.getTransactionId() );
    }
  }
  
  
  taskController.addPrimHandlers( [ TIMER_START_REQ,
                                    TIMER_STOP_REQ,
                                    TIMER_STOP_FAMILY_REQ,
                                    TIMER_RESET_REQ,
                                    TIMER_STOP_SPECIFIC_TRANS_REQ ],
                                  handlePrim );
}




/*
 *   Local Variables:
 *   tab-width: 2
 *   indent-tabs-mode: nil
 *   mode: java
 *   c-indentation-style: java
 *   c-basic-offset: 2
 *   eval: (c-set-offset 'statement-cont '0)
 *   eval: (c-set-offset 'substatement-open '0)
 *   eval: (c-set-offset 'case-label '+)
 *   eval: (c-set-offset 'inclass '+)
 *   eval: (c-set-offset 'inline-open '0)
 *   End:
 */
