/**

  Utility functions used in the unit test for this entity
  as well as those used in unit tests for the subsystem
  for which this entity is a unit test capability.

 */
public var util = new (function() {

  public function IntegTestObstest(
        testName
      , logWriter
      , warningWriter
      , errorWriter
    )
    extends pkg.tf.TestCase( testName, logWriter, warningWriter, errorWriter)
  {

    // Increments every time an event is displayed
    var eventReport = 0;

    function createEventString( prefix )
    {
      synchronized (eventReport) {
        ++ eventReport;
        return ";" + prefix + "; Seq#;" + sequenceNumber + "; EV#;" + eventReport + "; EVENT=";
      }
    }

    // Follows sequence numbers of primitives
    var sequenceNumber = 0;

    var seqNumHandler = function( prim )
    {
      if (prim.getSequenceNumber() > -1)
      {
        sequenceNumber = prim.getSequenceNumber();
      }
    };

    // Insert sequence number info into info reports
    const var parentLogError = logError;
    public function logError( string )
    {
      parentLogError( createEventString( "TESTSCR" ) + string );
    }

    // Insert sequence number info into info reports
    const var parentLogWarning = logWarning;
    public function logWarning( string )
    {
      parentLogWarning( createEventString( "TESTSCR" ) + string );
    }

    // Insert sequence number info into info reports
    const var parentLogInfo = logInfo;
    public function logInfo( string )
    {
      parentLogInfo( createEventString( "TESTSCR" ) + string );
    }

    public function runTest()
    {
      /****************************************************************
       * Global variable initionalizations.
       ****************************************************************/
      const var agePkg     = pkg.esf.age.util;
      const var obsPkg     = pkg.esf.obs.util;
      const var obstestPkg = util;
      const var scriptEhId = 0x102;
      var rctx = services["esf routing"].getRoutingContext();

      /* initialize */
      var phase = testName + " Initialization:";

      // This data structure will accumulate entries that are decoded during
      //  a run of a test case deriving from this one.
      var entryList = java.util.Collections.synchronizedList( new java.util.LinkedList() );

      var aggregateHandler = new function() extends
        ti.chimera.service.RoutingService.RoutingContext.PrimHandler()
      {
        const var entryDecoder = pkg.esf.age.util.acquireEntryDecoder( logInfo, logWarning, logError );
        const var observationDecoder = new ti.observable.ObservationDecoder( );

        observationDecoder.setTestReporter( new function () extends ti.test.TestReporter() {
            public function report( string ) {
              pkg.output.logInfoMsg( createEventString( "AGGHAND" ) + string );
            }
          } ()
        );


        public function handlePrim( prim )
        {
          entryDecoder.addAggregateEntryInd( prim );

          logInfo( "DEBUG: Handling an aggregate with entries=" + entryDecoder.entryCount() );

          while (entryDecoder.entryCount() > 0)
          {
            const var entry = entryDecoder.get();
            const var obsEntry = observationDecoder.createCleanEntry( entry );

            if (obsEntry == null)
            {
              pkg.output.logWarning(" Not expecting an entry with id " + entry.getTag() );
            }
            else
            {
              entryList.add( obsEntry );
            }
          }
        }
      } ();


      var primDisplayHandler = function( prim )
      {
        pkg.output.logInfoMsg( createEventString( "PDISPLAYHAND" ) + prim.getName() + " with seqnum=" + prim.getSequenceNumber() );
      };


      rctx.addPrimHandler(
          seqNumHandler
        , null
        );
      rctx.addPrimHandler(
          aggregateHandler
        , rctx.and(
              rctx.type( AGE_AGGREGATE_ENTRY_IND )
            , rctx.destination( scriptEhId )
            )
        );
      rctx.addPrimHandler(
          primDisplayHandler
        , null
        );


      rctx.setConstraint( rctx.noHistory() );

      try
      {
        while (true)
        {
          logInfo( "_______Initialization______" );
          if (!obstestPkg.start( rctx, scriptEhId ) )
          {
            logError( "Error in starting obstest event handler " );
            break;
          }

          /* Insure that subsystem is quiescent before starting */
          if (!agePkg.integTestStart( rctx, scriptEhId, phase, 8, 500, 0 )) break;
          if (!obsPkg.reset( rctx, scriptEhId )) break;
          if (!agePkg.forceSendAll( rctx, scriptEhId )) break;

          /* Clear events created during the initialization stage */
          entryList.clear();

          /* call the subclass provided test func */
          logInfo( "_______Running_Test______" );
          runIntegTestObstest( rctx, scriptEhId, agePkg, obsPkg, obstestPkg, entryList );

          break;
        }
      }
      catch (e)
      {
        logError( "Exception seen." );
        logError( e );
      }

      /* cleanup */
      logInfo( "*******************CLEANUP*****************" );
      rctx.removePrimHandler( aggregateHandler   );
      rctx.removePrimHandler( seqNumHandler      );
      rctx.removePrimHandler( primDisplayHandler );

      agePkg.unitTestEnd( rctx, scriptEhId, phase );
      obsPkg.reset( rctx, scriptEhId );

      rctx = null;

    }

  }



  /**
   Transform operation that helps in the conversion of
   strings to arrays
   */
  public function createArrayFromString( string, length )
  {
    const var bos = new java.io.ByteArrayOutputStream();
    (new java.io.DataOutputStream( bos )).writeBytes( string );
    const var raw = bos.toByteArray();

    var result = [];
    for (var i = 0; i < length; ++i)
    {
      result[ i ] =
          (i < raw.length() )
        ? raw[ i ]
        : 0
        ;
    }

    return result;

  }



  /**
   Operation to insure that this event handler is in
   a started state.  Once started, this event handler
   is never stopped.
   */
  public function start( rctx, scriptEhId )
  {
    var req = new OBSTEST_START_REQ(
         scriptEhId
      );

    const var prim = rctx.sendAndWaitForPrim(
          req
        , rctx.type( OBSTEST_DONE_CNF )
      );

    return prim.getResult() != 0;
  }




  /**
   A data type that allows the script writer to create a list
   of operation specifications for this event handler which
   then can be issued en masse to the event handler for
   processing.
   */
  public function OpList ()
  {
    const var list = [];


    /**
      Append a set operation to the list of specified operations.

      @param name   the name of the observable for which this op applies
      @param value  an integer value that is used in setting the observable
                    (according to its particular usage).
     */
    public function addSet( name, value )
    {

      list[ list.length() ] =
          new ti.th.prim.Obstest_ObstestEh_Operation(
              ti.th.prim.Obstest_ObstestEh_Opcode.SET
            , createArrayFromString( name, 24 )
            , value
            );
      return this;
    }


    /**
      Append a trigger operation to the list of specified operations.  This
      means that the particular observable will be 'notified' some number
      of times.

      @param name      the name of the observable for which this op applies
      @param numTimes  the number of times the observable's notify op is
                       to be called
     */
    public function addTrigger( name, numTimes )
    {
      list[ list.length() ] =
          new ti.th.prim.Obstest_ObstestEh_Operation(
              ti.th.prim.Obstest_ObstestEh_Opcode.TRIGGER_NORMAL
            , createArrayFromString( name, 24 )
            , numTimes
            );
      return this;
    }


    /**
      Append a trigger operation to the list of specified operations.  This
      means that the particular observable will be 'notified' some number
      of times using the pointer form of notify

      @param name      the name of the observable for which this op applies
      @param numTimes  the number of times the observable's notify op is
                       to be called
     */
    public function addTriggerPtr( name, numTimes )
    {
      list[ list.length() ] =
          new ti.th.prim.Obstest_ObstestEh_Operation(
              ti.th.prim.Obstest_ObstestEh_Opcode.TRIGGER_PTR
            , createArrayFromString( name, 24 )
            , numTimes
            );
      return this;
    }


    /**
      Append a trigger operation to the list of specified operations.  This
      means that the particular observable will be 'notified' some number
      of times, and the processor cache will be cleared at each step of the
      way.  This is good for worst case analyses.

      @param name      the name of the observable for which this op applies
      @param numTimes  the number of times the observable's notify op is
                       to be called
     */
    public function addTriggerWithCacheClear( name, numTimes )
    {
      list[ list.length() ] =
          new ti.th.prim.Obstest_ObstestEh_Operation(
              ti.th.prim.Obstest_ObstestEh_Opcode.TRIGGER_PTR
            , createArrayFromString( name, 24 )
            , numTimes
            );
      return this;
    }


    /**
      Append a create operation request to the list of specified ops.  Note
      that this operation will fail if the target has reached the maximum
      number of observables that can be instantiated.
     */
    public function addCreate( )
    {
      list[ list.length() ] =
          new ti.th.prim.Obstest_ObstestEh_Operation(
              ti.th.prim.Obstest_ObstestEh_Opcode.CREATE
            , createArrayFromString( "xxx", 24 )
            , 0
            );
      return this;
    }


    /**
     Cause the target to perform the operation specification list.  If any
     operation in the list fails, then none of the following are run.  This
     operation returns a boolean indicating whether the operation was
     successful or not.
     */
    public function perform( rctx, scriptEhId )
    {
      var req = new OBSTEST_OPERATION_REQ(
           scriptEhId
        , this.getList()
        );

      const var prim = rctx.sendAndWaitForPrim(
            req
          , rctx.type( OBSTEST_DONE_CNF )
        );


      return prim.getResult() != 0;
    }

    public function get( i )
    {
      return list[ i ];
    }

    public function getList()
    {
      return list;
    }

  }


  /**
    This function creates a specification to set
    test observables.  This function takes as
    a parameter an array of values- element 'e' at index
    'n' in the array indicates that logical test observable
    'n' should be set to value 'e'.
   */
  public function createSetSpec( valueArr )
  {
    const var retval = new OpList();

    const var min =
        (testObservableCount < valueArr.length())
      ? testObservableCount
      : valueArr.length()
      ;

    for (var i = 0; i < min; ++i)
    {
      if (testObservableHasInstance[ i ])
      {
        retval.addSet(
              testObservableNames[ i ]
          ,   valueArr[ i ]
          );
      }
    }

    return retval;
  }


  /**
    This function creates a specification to trigger
    test observables.  This function takes as
    a parameter an array of values- each value is
    the logical test observable number to trigger
    next one time.
   */
  public function createTriggerSequence( triggerTimes, useNotifyPtr )
  {
    pkg.output.logInfoMsg( "createTriggerSequence:" + useNotifyPtr );
    const var retval = new OpList();

    for (var i = 0; i < triggerTimes.length(); ++i)
    {
      if (testObservableHasInstance[ i ])
      {
        (useNotifyPtr?retval.addTriggerPtr:retval.addTrigger)(
              testObservableNames[ triggerTimes[ i ] ]
          ,   1
          );
      }
    }

    return retval;
  }


  /**
    This function triggers a specified logical test
    observable a specified number of times
   */
  public function createTriggerNTimes( logicalId, times )
  {
    const var retval = new OpList();

    retval.addTrigger(
          testObservableNames[ logicalId ]
      ,   times
      );

    return retval;
  }


  // Set a particular observable value
  public function set(
      rctx
    , scriptEhId
    , logicalTestObservableNumber
    , value
    )
  {
    const var cmd = new OpList();
    cmd.addSet( testObservableNames[ logicalTestObservableNumber ], value );
    cmd.perform( rctx, scriptEhId );
  }


  public function verifyObservationValueSimple( value, primStream )
  {
    var undecoded = primStream.getStreamForRemaining();
    const var decoded = undecoded.getU4();

    const var primStreamString = "[" + primStream.getStreamForRemaining().getData().length() + "]";

    if (decoded != value)
    {
      pkg.output.logWarning( " Decoded value is " + decoded + " and expecting " + value +" : saw " + primStreamString );
    }

    return decoded == value;
  }


  public function verifyObservationValueStructured( value, primStream )
  {
    var undecoded = primStream.getStreamForRemaining();
    var decodedStruct = new ti.th.prim.Obstest_SampleStructuredObservation( undecoded );

    const var xpList = new pkg.util.expect.ExpectationList();
    xpList.add(
        /**/ ["result.getLeadingUint32()  ", "=="+value             ]
        ,    ["result.getAUint16()        ", "==(0xffff&"+value+")" ]
        ,    ["result.getAUint8()         ", "==(0x00ff&"+value+")" ]
        ,    ["result.getAUint8Array()[3] ", "==(0x00ff&"+value+")" ]
        ,    ["result.getAUint8Array()[0] ", "==(0x00ff&"+value+")" ]
        ,    ["result.getAUint8Array()[7] ", "==(0x00ff&"+value+")" ]
        ,    ["result.getSecondUint32()   ", "=="+value             ]
        ,    ["result.getTrailingUint8()  ", "==(0x00ff&"+value+")" ]
        );

    var list = new java.util.LinkedList();
    list.add( decodedStruct );

    var retval = xpList.evaluate( list, pkg.output.logError, pkg.output.logInfoMsg );

    pkg.output.logInfoMsg( " result of this function call (verifyStruct) is " + retval );

    return retval;
  }


  public function verifyObservationValueStructuredMacro( value, primStream )
  {
    var undecoded = primStream.getStreamForRemaining();
    var decodedStruct = new ti.th.prim.Obstest_TestStructure( undecoded );

    const var xpList = new pkg.util.expect.ExpectationList();
    xpList.add(
        /**/ ["result.getAUint32()        ", "=="+value             ]
        ,    ["result.getAUint8()         ", "==(0x00ff&"+value+")" ]
        ,    ["result.getAUint16()        ", "==(0xffff&"+value+")" ]
        ,    ["result.getAUint8Array()[3] ", "==(0x00ff&"+value+")" ]
        ,    ["result.getAUint8Array()[0] ", "==(0x00ff&"+value+")" ]
        ,    ["result.getAUint8Array()[7] ", "==(0x00ff&"+value+")" ]
        );

    var list = new java.util.LinkedList();
    list.add( decodedStruct );

    var retval = xpList.evaluate( list, pkg.output.logError, pkg.output.logInfoMsg );

    pkg.output.logInfoMsg( " result of this function call (verifyStruct) is " + retval );

    return retval;
  }



  /* This structure contains the list of test observables in the
     system.  There is one entry per test observable, and the index
     of each entry in the outer array is known as the 'logical test
     observable id' (as opposed to the actual observable id which
     can differ from system runtime to system runtime).

     Each entry in the outer array is a sub array.  The first element
     in the subarray is the textual name of a particular test observable.

     The second element is the logical id associated with the observable,
     or negative if there is no logical id instancing.

     The third element is true if there actually is an instance associated
     with the observable: some are totally compiled out at runtime.

     The fourth element in that subarray is the name of a function
     defined nearby that verifies whether a given observation from
     that particular observable corresponds to the value assigned
     during a 'set' operation for that test observable.  (he obstest
     event handler makes it easy for you to set each of its test observables
     given a single 32 bit value.  This function is simply the verification
     that the value was set the way you expect it to be.)

     This code must be maintained in sync with the code in the obstest
     event handler which implements the test observables.
   */
  /*private*/ const var testObservableDefs =
    // . observableName             . logicalId . hasInstance . verifyFunc
    [  [ "obstest_Simple"           , -1        , true        , verifyObservationValueSimple          ]
    ,  [ "obstest_Structured"       , -1        , true        , verifyObservationValueStructured      ]
    ,  [ "obstest_Array0"           , -1        , true        , verifyObservationValueSimple          ]
    ,  [ "obstest_Array1"           , -1        , true        , verifyObservationValueSimple          ]
    ,  [ "obstest_Array2"           , -1        , true        , verifyObservationValueSimple          ]
    ,  [ "obstest_Array3"           , -1        , true        , verifyObservationValueSimple          ]
    ,  [ "obstest_NotifySpprst"     , -1        , true        , verifyObservationValueSimple          ]
    ,  [ "obstest_StructuredMacro"  , -1        , true        , verifyObservationValueStructuredMacro ]
    ,  [ "obstest_CompiledOut"      , -1        , false       , verifyObservationValueSimple          ]
    ,  [ "obstest_Logical"          , 0         , true        , verifyObservationValueSimple          ]
    ,  [ "obstest_Logical"          , 1         , true        , verifyObservationValueSimple          ]
    ];


  /* A public definition of the number of test observables in the system */
  public const var testObservableCount = testObservableDefs.length();


  /* Retrieve an observable name given an entry number */
  public const var testObservableNames = new (function() {
    public function elementAt( value ) {return testObservableDefs[ value ][ 0 ];}
  }) ();


  /* Retrieve a logical id (or a negative value) given an entry number */
  public const var testObservableLogicalId = new (function() {
    public function elementAt( value ) {return testObservableDefs[ value ][ 1 ];}
  }) ();


  /* Retrieve a hasInstance flag given an entry number */
  public const var testObservableHasInstance = new (function() {
    public function elementAt( value ) {return testObservableDefs[ value ][ 2 ];}
  }) ();


  /* Retrieve a verification function given an entry number */
  public const var testObservableVerifyFuncs = new (function() {
    public function elementAt( value ) {return testObservableDefs[ value ][ 3 ];}
  }) ();


  /**
    Resolve the actual Observable ids for the test observables in the
    system by doing a lookup.  This function returns effectively an
    array indexed by logical test observable id where each entry
    is a true Observable id.
   */
  public function getTestObservableIds( rctx, scriptEhId )
  {
    var idLookupTemp = [];

    for (var i = 0; i < testObservableCount; ++i)
    {
      var value = pkg.esf.obs.util.lookupStreamIdByName( rctx, scriptEhId, testObservableNames[ i ] );
      var hasInstance = testObservableHasInstance[ i ];
      if( hasInstance && value <= 0 )
      {
        pkg.output.logError( "Expecting instance, but Observable " + testObservableNames[i] + " id reported as " + value );
      }

      if ( (!hasInstance) && (value > 0) )
      {
        pkg.output.logError( "Expecting NO instance, but Observable " + testObservableNames[i] + " id reported as " + value );
      }

      pkg.output.logInfoMsg( "Observable " + testObservableNames[i] + " id successfully reported as " + value );

      idLookupTemp[ i ] = value;
    }

    const var ids = idLookupTemp;

    return new (function() {
      public function elementAt( index )
      {
        if (index > idLookupTemp.length()) {
          pkg.output.logInfoMsg(
            " I can't resolve element " + index
          + " when there are only " + idLookupTemp.length()
          + " test observables! "
          );
        }
        return idLookupTemp[ index ];
      }
    }) ();

  }



}) ();
