/*=============================================================================
 *     Copyright Texas Instruments 2002. All Rights Reserved.
 */

/*=============================================================================
    Description:

    This is a multifunction test.  It verifies several things:
    + Push mode works only for appropriately configured observables
    + Endianness is correct for the machine on which the test is run
    + Decoding of the structured observation type works correctly

 */
public function IntegTestVerifyPushModeObservations( logWriter, warningWriter, errorWriter )
  extends pkg.esf.obstest.util.IntegTestObstest(
        "esf.obstest.IntegTestVerifyPushModeObservations"
      , logWriter
      , warningWriter
      , errorWriter
      )
{
  //==========================================================================
  /**
    @param rctx         the routing context to use for traffic
    @param scriptEhId   the event handler id to spoof for this test case
    @param agePkg       a package to use for age event handler convenience ops
    @param obsPkg       a package to use for obs event handler convenience ops
    @param obstestPkg   a package to use for obstest event handler convenience
    @param etList       a list into which events from the aggregate stream are
                        placed.
   */
  function runIntegTestObstest(
        rctx
      , scriptEhId
      , agePkg
      , obsPkg
      , obstestPkg
      , etList
    )
  {
    const var ids = obstestPkg.getTestObservableIds( rctx, scriptEhId );

    //  Create a controlled random number
    const var r = new (function()
    {
      const var rand = new java.util.Random( 4254 );

      public function callAsFunction( array )
      {
        return rand.nextInt( 0x1fffffff );
      }
    })();

    const var USE_NOTIFY     = false;
    const var USE_NOTIFY_PTR = true;

    // This structure controls the tests that are executed.  Each element
    //  of this structure specifies a particular test scenario.  Each test
    //  scenario is specified by three sub arrays: the first array identifies
    //  the logical test observables that are enabled for that particular test.
    //  Each element in the second array specifies a logical test observable
    //  that is to be triggered next in the sequence (duplicates allowed).
    //  The third array specifies the values that are sent to each of the
    //  logical test observables for this test, where each value is sent
    //  to the coindexed appropriate logical test observable.
    const var TEST_SCENARIO_LIST  =
      [ [ [ 0, 1, 2    ], [ 1  /* .  .  .  .  */ ], [ 0x12345678, 0x13243546, 0x14253647, 0x15263748, 0x16273849, 0x17283940, 0x94837261, 0x94837261, 0 ], USE_NOTIFY     ]
      , [ [ 0, 1, 2    ], [ 1, 1  /* .  .  .  */ ], [ 0x13243546, 0x14253647, 0x15263748, 0x16273849, 0x17283940, 0x94837261, 0x12345678, 0x12345678, 0 ], USE_NOTIFY     ]
      , [ [ 0, 1, 2    ], [ 3, 3  /* .  .  .  */ ], [ 0x14253641, 0x15263744, 0x16273844, 0x12345671, 0x13243547, 0x17283940, 0x94837261, 0x44837261, 0 ], USE_NOTIFY     ]
      , [ [ 7  /* */   ], [ 7  /* .  .  .  .  */ ], [ 0x14253642, 0x15263745, 0x16273843, 0x12345672, 0x13243544, 0x17283943, 0x94837264, 0x54837264, 0 ], USE_NOTIFY     ]
      , [ [ 0, 1, 7    ], [ 3, 7  /* .  .  .  */ ], [ 0x14253643, 0x15263746, 0x16273842, 0x12345673, 0x13243542, 0x17283944, 0x94837265, 0x64837265, 0 ], USE_NOTIFY     ]
      , [ [ 0, 2  /**/ ], [ 0, 2  /* .  .  .  */ ], [ 0x15263748, 0x16273849, 0x17283940, 0x12345678, 0x13243546, 0x14253647, 0x94837261, 0x74837261, 0 ], USE_NOTIFY     ]
      , [ [ 0, 2  /**/ ], [ 0, 2  /* .  .  .  */ ], [ r(),        r(),        r(),        r(),        r(),        r(),        r(),        r(),        0 ], USE_NOTIFY     ]
      , [ [ 2, 4  /**/ ], [ 0, 2, 4, 3, 2  /* */ ], [ r(),        r(),        r(),        r(),        r(),        r(),        r(),        r(),        0 ], USE_NOTIFY     ]
      , [ [ 4, 5  /**/ ], [ 5, 3, 4, 5  /* .  */ ], [ r(),        r(),        r(),        r(),        r(),        r(),        r(),        r(),        0 ], USE_NOTIFY     ]
      , [ [ 1, 4  /**/ ], [ 4, 4, 4, 1, 1, 5, 2  ], [ r(),        r(),        r(),        r(),        r(),        r(),        r(),        r(),        0 ], USE_NOTIFY     ]
      , [ [ 0, 1, 2    ], [ 1, 1  /* .  .  .  */ ], [ 0x14253641, 0x15263744, 0x16273844, 0x12345671, 0x13243547, 0x17283940, 0x94837261, 0x44837261, 0 ], USE_NOTIFY_PTR ]
      , [ [ 0, 1, 2    ], [ 3, 3  /* .  .  .  */ ], [ 0x14253642, 0x15263745, 0x16273843, 0x12345672, 0x13243544, 0x17283943, 0x94837264, 0x54837264, 0 ], USE_NOTIFY_PTR ]
      , [ [ 0, 1, 7    ], [ 3, 7  /* .  .  .  */ ], [ 0x14253643, 0x15263746, 0x16273842, 0x12345673, 0x13243542, 0x17283944, 0x94837265, 0x64837265, 0 ], USE_NOTIFY_PTR ]
      , [ [ 0, 2  /**/ ], [ 0, 2  /* .  .  .  */ ], [ 0x15263748, 0x16273849, 0x17283940, 0x12345678, 0x13243546, 0x14253647, 0x94837261, 0x74837261, 0 ], USE_NOTIFY_PTR ]
      , [ [ 0, 2  /**/ ], [ 0, 2  /* .  .  .  */ ], [ r(),        r(),        r(),        r(),        r(),        r(),        r(),        r(),        0 ], USE_NOTIFY_PTR ]
      , [ [ 2, 4  /**/ ], [ 0, 2, 4, 3, 2  /* */ ], [ r(),        r(),        r(),        r(),        r(),        r(),        r(),        r(),        0 ], USE_NOTIFY_PTR ]
      , [ [ 4, 5  /**/ ], [ 5, 3, 4, 5  /* .  */ ], [ r(),        r(),        r(),        r(),        r(),        r(),        r(),        r(),        0 ], USE_NOTIFY_PTR ]
      , [ [ 1, 4  /**/ ], [ 4, 4, 4, 1, 1, 5, 2  ], [ r(),        r(),        r(),        r(),        r(),        r(),        r(),        r(),        0 ], USE_NOTIFY_PTR ]
      ];

    // Clear statistics
//    agePkg.clearStats( rctx, scriptEhId );

    // For each test case in the list...
    pkg.util.forEach( TEST_SCENARIO_LIST, function( thisScenario )
    {
      const var ENABLED_OBSERVABLE_SET    = thisScenario[0];
      const var TRIGGERED_OBSERVABLE_LIST = thisScenario[1];
      const var OBSERVABLE_SET_VALUES     = thisScenario[2];
      const var WHICH_NOTIFY              = thisScenario[3];

      logInfo( "Starting Scenario '" + thisScenario );

      // Create a trigger expectation: given the above lists,
      //  some observables will be triggered but never heard
      //  since they are not enabled for push mode.
      const var TRIGGER_EXPECTATION;
      {
        var triggerExpect = [];

        pkg.util.forEach(
            TRIGGERED_OBSERVABLE_LIST
          , function( value )
        {
          if (pkg.util.isIn( value, ENABLED_OBSERVABLE_SET ))
          {
            triggerExpect[ triggerExpect.length() ] = value;
          }
        });
        TRIGGER_EXPECTATION = triggerExpect;
      }

      // Clear events
      if (etList.size() > 0)
      {
        logWarning( "Unexpected this is: etList.size()=" + etList.size() );
        for (var i = 0; i < etList.size(); ++i)
        {
          logWarning( "  Element " + i + " is " + etList.get( i ) );
        }
        etList.clear();
      }

      // since auto buffer refills are disabled, we'll be explicit
      agePkg.triggerBufferService( rctx, scriptEhId );

      // Query stats- just make sure they are in log
      agePkg.queryStats( rctx, scriptEhId );
      logInfo(" Check log for stats ");

      // Create the expectation evaluator
      const var xpList = new pkg.util.expect.ExpectationList();
      // Configure the observables that are active during this test.
      pkg.util.forEach(
          ENABLED_OBSERVABLE_SET
        , function( thisLogicalTestObservable )
      {
        const var obsIdUnderTest = ids[ thisLogicalTestObservable ];

        // Configure observable into push mode
        obsPkg.configureStreamExpecting(
            rctx
          , scriptEhId
          , obsIdUnderTest
          , 1, 0, 1, 0
          );

        //  Expect an initial report of the value of the observable, and
        //  ignore its value since we haven't configured it
        xpList.add(
                [ "true", "", "( IGNORED OBSERVATION VALUE )" ]
          ,     [ "result", " instanceof ti.observable.ObservationPartialDecodeEntry" ]
          ,     [ "result.getId()", "==" + obsIdUnderTest ]
          );
      });

      logInfo( "Now setting values to all test Observables: " + OBSERVABLE_SET_VALUES );

      // Set all of the test observables
      obstestPkg.createSetSpec( OBSERVABLE_SET_VALUES ).perform( rctx, scriptEhId );

      logInfo( "Now triggering values to some test Observables: " + TRIGGERED_OBSERVABLE_LIST );

      // Query stats- just make sure they are in log
      agePkg.queryStats( rctx, scriptEhId );
      logInfo(" Check log for stats ");

      // Trigger the test observables
      obstestPkg.createTriggerSequence( TRIGGERED_OBSERVABLE_LIST, WHICH_NOTIFY ).perform( rctx, scriptEhId );

      // Query stats- just make sure they are in log
      agePkg.queryStats( rctx, scriptEhId );
      logInfo(" Check log for stats ");

      logInfo( "Now expect specific observations: " + TRIGGER_EXPECTATION );

      // Expect a very specific observation sequence
      pkg.util.forEach(
          TRIGGER_EXPECTATION
        , function( thisLogicalTestObservable )
      {
        const var obsIdUnderTest = ids[ thisLogicalTestObservable ];
        const var value = OBSERVABLE_SET_VALUES[ thisLogicalTestObservable ];

        xpList.add(
                [ "result", " instanceof ti.observable.ObservationPartialDecodeEntry" ]
          ,     [ "result.getId()", "==" + obsIdUnderTest ]
          ,     [ "pkg.esf.obstest.util.testObservableVerifyFuncs[" + thisLogicalTestObservable + "]( " + value + ", result.getUndecoded() )" ]
          );
      });

      // Unconfigure the observables that are active during this test.
      pkg.util.forEach(
          ENABLED_OBSERVABLE_SET
        , function( thisLogicalTestObservable )
      {
        const var obsIdUnderTest = ids[ thisLogicalTestObservable ];

        // Configure observable out of push mode
        obsPkg.configureStreamExpecting(
            rctx
          , scriptEhId
          , obsIdUnderTest
          , 0, 0, 0, 0
          );
      });

        // Insure all written buffers sent...

        logInfo( " FORCE ALL BUFFERS TO BE SENT. " );
        obsPkg.getStatusStruct( rctx, scriptEhId );
        if (!agePkg.forceSendAll( rctx, scriptEhId )) return false;
        logInfo( " ALL BUFFERS MUST BE SENT, RIGHT??? " );
        
        logWarning( "NOTE: a delay was inserted here to avoid a chimera race condition."
                    + " This is not guaranteed to prevent errors after this." );
        java.lang.Thread.currentThread().sleep( 5000 );
        logInfo( "Fakie delay complete." );
        
        xpList.evaluate( etList, logError, logInfo );
        etList.clear();

        logInfo( " ABSOLUTE END OF TEST" );

    });
  }

}


