/*=============================================================================
    Copyright 2002 Texas Instruments Inc. All rights reserved.

    Author: Brad Hyslop (bhyslop@ti.com)

    This module contains the standard utilities used to configure
    the obs event handler from scripts.

 */

public var util = new (function() {


  /**
      Resets this event handler.  The event handler should have all
      state set to be completely quiescent.
   */
  public function reset(
          rctx
        , scriptEhId
      )
  {
    const var prim = rctx.sendAndWaitForPrim(
          new OBS_RESET_REQ(
            /* source ehId */          scriptEhId
          )
        , rctx.type( OBS_RESET_CNF )
        );
    return true;
  }


  /**
      Retrieves the current status from this event handler
   */
  public function getStatusStruct(
          rctx
        , scriptEhId
        )
  {
    const var prim = rctx.sendAndWaitForPrim(
                new OBS_QUERY_STATUS_REQ(
                  /* source ehId */  scriptEhId
                )
              , rctx.type(OBS_QUERY_STATUS_CNF)
              );
    return prim.getStatus();
  }


  /**
    Perform a query about the specified observable id
   */
  public function discoverObservable(
      rctx
    , scriptEhId
    , observableId
    )
  {
    var sendMe = new OBS_QUERY_OBSERVABLE_REQ( scriptEhId, observableId );

    const var prim = rctx.sendAndWaitForPrim(
              sendMe
            , rctx.type(OBS_QUERY_OBSERVABLE_CNF)
            );
    return prim;
  }


  /**
    Searches the current instances of observables in order
    to find and return an id for an observable with the
    designated id
   */
  public function lookupStreamIdByName(
        rctx
      , scriptEhId
      , searchName
    )
  {
    const var maxObs = getStatusStruct( rctx, scriptEhId ).getObservableMaxCurrent();

    for (var i = maxObs; i > 0; --i)
    {
      const var result = discoverObservable( rctx, scriptEhId, i ).getObservableName();

      // name without zero terminations in it
      var name = [];

      for (var j = 0; j < result.length(); ++j)
      {
        if (result[j] == 0) break;
        name[j] = result[j];
      }

      var lookup = new java.lang.String( name );

      if (lookup == searchName) return i;
    }

    return 0;
  }


  /**
      A set-and-verify operation that sets the particular
      observation stream generation for a particular id to
      a particular value and verifies that that value
      was set correctly.
   */
  public function configureStreamExpecting(
            rctx
          , scriptEhId
          , id
          , pushEnabled
          , pullEnabled
          , pushResult
          , pullResult
        )
  {
    const var existingState;
    var retval = true;

    existingState = rctx.sendAndWaitForPrim(
        new OBS_QUERY_OBSERVATION_STREAM_CONFIG_REQ(
          /* source ehId */          scriptEhId
             ,                       id
          )
      , rctx.type(OBS_OBSERVATION_STREAM_CONFIG_CNF)
      ).getConfig();

    const var configureResult;
    configureResult = rctx.sendAndWaitForPrim(
        new OBS_SET_OBSERVATION_STREAM_CONFIG_REQ(
          /* source ehId */          scriptEhId
        , new ti.th.prim.Obs_ObsEh_ObservationStreamConfigStruct(
            /* observableId */       id,
            /* pushEnabled */        pushEnabled,
            /* pullEnabled */        pullEnabled,
            [ ]
          )
        )
      , rctx.type(OBS_OBSERVATION_STREAM_CONFIG_CNF)
      ).getConfig();

    const var operationWasSuccessful =
          existingState.getPullEnabled() != configureResult.getPullEnabled()
      ||  existingState.getPushEnabled() != configureResult.getPushEnabled()
      ;

    if ( pushResult != configureResult.getPushEnabled())
    {
      pkg.output.logError( ""
          + "Observable " + id + ": "
          + "Expected push state "
          + pushResult
          + " but saw "
          + configureResult.getPushEnabled()
          );
    }

    if ( pullResult != configureResult.getPullEnabled())
    {
      pkg.output.logError( ""
          + "Observable " + id + ": "
          + "Expected pull state "
          + pullResult
          + " but saw "
          + configureResult.getPullEnabled()
          );
    }

    return retval;
  }


  /**
      Set the pull interval.
   */
  public function setPull(
        rctx
      , scriptEhId
      , interval
    )
  {

    const var evList = new java.util.LinkedList();
    evList.add(
           rctx.sendAndWaitForPrim(
              new OBS_SET_PULL_PERIOD_REQ(
              /* source ehId */          scriptEhId,
              /* pullPeriod */           interval
            )
          , rctx.type(OBS_PULL_PERIOD_CNF)
          )
       );

    const var xpList = new pkg.util.expect.ExpectationList();

    xpList.add( [ "result.getPullPeriod()", "==" + interval ] );

    return xpList.evaluate( evList, pkg.output.logError, pkg.output.logInfoMsg );
  }


  /**
    Trigger an explicit pull operation
   */
  public function triggerPull(
        rctx
      , scriptEhId
    )
  {
    rctx.sendAndWaitForPrim(
        new OBS_TRIGGER_PULL_REQ( scriptEhId )
      , rctx.type( OBS_TRIGGER_PULL_CNF )
    );

    return true;
  }

}) ();

